from typing import List, Optional, Any

from pyconfita.backend.backend import Backend, VType
from pyconfita.logging_interface import LoggingInterface


class Confita:
    logger: LoggingInterface
    backends: List[Backend]
    case_sensitive: bool

    def __init__(
        self,
        logger: LoggingInterface,
        backends: List[Backend],
        case_sensitive: bool = True,
        *args,
        **kwargs,
    ):
        """

        :param logger:
        :param backends: list of key-value backends. Order sets the
        evaluation order for values.
        :param case_sensitive: If False, enables multi-casing matching on key
        :param args:
        :param kwargs:
        """
        self.logger = logger
        self.backends = backends
        self.case_sensitive = case_sensitive
        if not case_sensitive:
            logger.zlog(
                **{
                    "level": "debug",
                    "message": {"message": f"case sensitivity disabled"},
                }
            )

    def get(self, key: str, v_type: VType = str, **kwargs) -> Optional[Any]:
        """
        Read the value at key in all the backends.
        Returns the last not None value found in order of the list of
        backends. Returns None if not found.

        :param key:
        :param v_type: type to convert the value into. Explicit conversion
        parameter. Defaults to `str`
        :param kwargs:
        :return:
        """
        _value = None

        for bk in self.backends:
            if self.case_sensitive:
                # Initial casing for key
                tmp_value = bk.get(key, v_type=v_type, **kwargs)
            else:
                # Try reading with casing variations on key
                tmp_value = (
                    bk.get(key, v_type=v_type, **kwargs)
                    or bk.get(key.upper(), v_type=v_type, **kwargs)
                    or bk.get(key.lower(), v_type=v_type, **kwargs)
                )
            if tmp_value is not None:  # Overrides if defined only
                _value = tmp_value
                self.logger.zlog(
                    **{
                        "level": "debug",
                        "message": {
                            "message": f"{bk.name} sets value = {tmp_value} (type = {v_type}) for key = {key}"
                        },
                    }
                )

        self.logger.zlog(
            **{
                "level": "debug",
                "message": {
                    "message": f"Final value read for {key} = {_value} (type = {v_type})"
                },
            }
        )
        return _value
