import os
import random

import openai

from vdom import pre
from IPython.core.magic import (
    register_cell_magic,
)

from IPython.display import display

"""
Initialize the openai API.
"""
openai.organization = os.getenv("OPENAI_ORGANIZATION")
openai.api_key = os.getenv("OPENAI_API_KEY")

if openai.api_key is None:
    raise Exception("Please set the OPENAI_API_KEY environment variable")

if openai.organization is None:
    raise Exception("Please set the OPENAI_ORGANIZATION environment variable")

"""
If the users puts #ignore or #keep at the top of their cells, they can control
what gets sent in a prompt to openai.
"""
# Cells we want to ignore
ignore_tokens = [
    "#ignore",
    "# ignore",
    "%%assist",
    "get_ipython",
    "%load_ext",
    "import genai",
    "%pip install",
]

# Cells we want to keep for sure, in case In[] is too long.
keep_tokens = ["#keep", "# keep"]


def ignored(inp):
    for ignore in ignore_tokens:
        if inp.startswith(ignore):
            return True
    return False


def estimate_tokens(cells: list[str]):
    """
    Estimate the number of tokens in a cell.
    """
    return len("".join(cells).split())


def truncate_prior_cells(cells: list[str], max_tokens: int = 500):
    """
    Truncate the priors to a reasonable length.
    """
    if max_tokens < 0:
        return []

    # Split into halves, with a preference for using #keep cells.
    first_half = cells[: len(cells) // 2]
    second_half = cells[len(cells) // 2 :]

    keepers = []

    # Only include the keeps from the first half
    for keep in keep_tokens:
        if first_half.startswith(keep):
            keepers.append(keep)

    if estimate_tokens(keepers) > max_tokens:
        # TODO: Trim down the keeps as an option
        raise Exception("Too many cells are marked as #keep")

    # Check if keeps + second_half is the right size
    if estimate_tokens(keepers + second_half) < max_tokens:
        return keepers + second_half

    # Split the second half now to see if we can fit more in.
    trimmed_second = truncate_prior_cells(
        second_half, max_tokens - estimate_tokens(keepers)
    )

    if estimate_tokens(keepers + trimmed_second) < max_tokens:
        return keepers + trimmed_second

    return keepers


def prior_code(inputs: list[str], max_length: int = 500):
    """
    Grab all of the inputs, up to a certain length, and return them as a string.
    """

    # Add lines in reverse order
    lines = []

    # Assume, naively at first that we can keep it all.
    for inp in inputs:
        if ignored(inp):
            continue
        lines.append(inp)

    if estimate_tokens(lines) > max_length:
        lines = truncate_prior_cells(lines, max_length)

    return "".join([inp.replace("# generated with %%assist", "") for inp in lines])


"""
Messages were generated using the text-davinci-002 model using this prompt:

Write some catch phrases for an AI that generates code cells for Jupyter
notebooks. The AI was trained on the work of other data scientists, analysts,
data engineers, programmers, and quirky artists. Crack some jokes, have some
fun. Be a quirky AI.

[
    "Phoning a friend 📲",
"""


def starting_message():
    return pre(
        random.choice(
            [
                "Phoning a friend 📲",
                "Reaching out to another data scientist 📊",
                "Just a little bit of data engineering 🔧",
                "Trying my best 💯",
                "Generating some code cells 💻",
                "Asking the internet 🌐",
                "Searching through my memory 💾",
                "What would a data analyst do? 🤔",
                "Querying my database 🗃️",
                "Running some tests 🏃‍",
                "One code cell, coming right up! 🚀",
                "I'm a machine, but I still enjoy helping you code. 😊",
            ]
        )
    )


def completion_made():
    return pre(
        random.choice(
            [
                "Enjoy your BRAND NEW CELL 🚙",
                "Just what you needed - more code cells! 🙌",
                "Here's to helping you code! 💻",
                "Ready, set, code! 🏁",
                "Coding, coding, coding... 🎵",
                "Just another code cell... 🙄",
                "Here's a code cell to help you with your analysis! 📊",
                "Need a code cell for your data engineering work? I got you covered! 🔥",
                "And now for something completely different - a code cell! 😜",
                "I got a little creative with this one - hope you like it! 🎨",
                "This one's for all the data nerds out there! 💙",
            ]
        )
    )


@register_cell_magic
def assist(line, cell):
    ip = get_ipython()

    previous_inputs = prior_code(ip.history_manager.input_hist_raw).strip()
    cell_text = "".join(cell).strip()

    prompt = f"""# Python\n{previous_inputs}\n{cell_text}""".strip()

    progress = display(starting_message(), display_id=True)

    # make a rough estimate
    prompt_token_count = len(prompt.split())

    completion = openai.Completion.create(
        model="text-davinci-002",
        prompt=prompt,
        max_tokens=4096 - prompt_token_count,
        temperature=0.5,
    )
    progress.update(completion_made())

    choice = completion.choices[0]
    ip.set_next_input(
        f"""# generated with %%assist\n{cell_text}{choice.text}""", replace=False
    )
