"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const path = require("path");
const asg = require("@aws-cdk/aws-autoscaling");
const aws_autoscaling_hooktargets_1 = require("@aws-cdk/aws-autoscaling-hooktargets");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const assets = require("@aws-cdk/aws-s3-assets");
const cdk = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
/**
 * @experimental
 */
class GitlabRunnerAutoscaling extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        const defaultProps = {
            instanceType: 't3.micro',
            tags: ['gitlab', 'awscdk', 'runner'],
            gitlabUrl: 'https://gitlab.com/',
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:alpine',
        };
        const runnerProps = { ...defaultProps, ...props };
        const asset = new assets.Asset(this, 'GitlabRunnerUserDataAsset', {
            path: path.join(__dirname, '../assets/userdata/amazon-cloudwatch-agent.json'),
        });
        const userData = ec2.UserData.forLinux();
        userData.addS3DownloadCommand({
            bucket: asset.bucket,
            bucketKey: asset.s3ObjectKey,
            localFile: '/opt/aws/amazon-cloudwatch-agent/etc/amazon-cloudwatch-agent.json',
        });
        userData.addCommands(...this.createUserData(runnerProps));
        this.instanceRole = (_a = runnerProps.instanceRole) !== null && _a !== void 0 ? _a : new iam.Role(this, 'GitlabRunnerInstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            description: 'For EC2 Instance (Gitlab Runner) Role',
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchAgentServerPolicy'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
            ],
        });
        this.vpc = (_b = runnerProps.vpc) !== null && _b !== void 0 ? _b : new ec2.Vpc(this, 'VPC');
        this.securityGroup = new ec2.SecurityGroup(this, 'GitlabRunnerSecurityGroup', {
            vpc: this.vpc,
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.instanceRole.roleName],
        });
        const lt = new ec2.CfnLaunchTemplate(this, 'GitlabRunnerLaunchTemplate', {
            launchTemplateData: {
                imageId: ec2.MachineImage.latestAmazonLinux({
                    generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                }).getImage(this).imageId,
                instanceType: runnerProps.instanceType,
                instanceMarketOptions: {
                    marketType: runnerProps.spotInstance ? 'spot' : undefined,
                    spotOptions: runnerProps.spotInstance ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
                blockDeviceMappings: [
                    {
                        deviceName: '/dev/xvda',
                        ebs: {
                            volumeSize: (_c = runnerProps.ebsSize) !== null && _c !== void 0 ? _c : 60,
                        },
                    },
                ],
                iamInstanceProfile: {
                    arn: instanceProfile.attrArn,
                },
                securityGroupIds: this.securityGroup.connections.securityGroups.map((m) => m.securityGroupId),
            },
        });
        this.autoscalingGroup = new asg.AutoScalingGroup(this, 'GitlabRunnerAutoscalingGroup', {
            instanceType: new ec2.InstanceType(runnerProps.instanceType),
            autoScalingGroupName: `Gitlab Runners (${runnerProps.instanceType})`,
            vpc: this.vpc,
            vpcSubnets: runnerProps.vpcSubnet,
            machineImage: ec2.MachineImage.latestAmazonLinux({
                generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            minCapacity: runnerProps.minCapacity,
            maxCapacity: runnerProps.maxCapacity,
            desiredCapacity: runnerProps.desiredCapacity,
        });
        const cfnAsg = this.autoscalingGroup.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        this.autoscalingGroup.node.tryRemoveChild('LaunchConfig');
        const unregisterPolicy = new iam.Policy(this, 'GitlabRunnerUnregisterPolicy', {
            statements: [
                new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    resources: ['*'],
                    actions: [
                        'ssm:SendCommand',
                        'autoscaling:DescribeAutoScalingGroups',
                        'logs:CreateLogGroup',
                        'logs:CreateLogStream',
                        'logs:PutLogEvents',
                    ],
                }),
            ],
        });
        const unregisterRole = new iam.Role(this, 'GitlabRunnerUnregisterRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            description: 'For Gitlab Runner Unregistering Function Role',
        });
        unregisterRole.attachInlinePolicy(unregisterPolicy);
        // Lambda function to unregiser runners on terminating instance
        const unregisterFunction = new lambda.Function(this, 'GitlabRunnerUnregisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.unregister',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHook', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_TERMINATING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(unregisterFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        // Lambda function to unregiser runners on destroying stack
        const unregisterCustomResource = new lambda.Function(this, 'GitlabRunnerUnregisterCustomResource', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        const unregisterProvider = new cr.Provider(this, 'GitlabRunnerUnregisterProvider', {
            onEventHandler: unregisterCustomResource,
        });
        const customResource = new cdk.CustomResource(this, 'GitlabRunnerCustomResource', {
            serviceToken: unregisterProvider.serviceToken,
            properties: {
                AutoScalingGroupNames: [this.autoscalingGroup.autoScalingGroupName],
            },
        });
        customResource.node.addDependency(unregisterProvider);
        new cdk.CfnOutput(this, 'GitlabRunnerAutoScalingGroupArn', {
            value: this.autoscalingGroup.autoScalingGroupArn,
        });
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @experimental
     */
    createUserData(props) {
        var _a;
        return [
            'yum update -y',
            'sleep 15 && yum install -y docker git amazon-cloudwatch-agent && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock',
            'systemctl restart docker && systemctl enable docker && systemctl start amazon-cloudwatch-agent && systemctl enable amazon-cloudwatch-agent',
            `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register ${props.gitlabRunnerImage} register --non-interactive --url ${props.gitlabUrl} --registration-token ${props.gitlabToken} \
      --docker-pull-policy if-not-present ${this.dockerVolumesList(props === null || props === void 0 ? void 0 : props.dockerVolumes)} \
      --executor docker --docker-image "alpine:latest" --description "A Runner on EC2 Instance (${props.instanceType})" \
      --tag-list "${(_a = props.tags) === null || _a === void 0 ? void 0 : _a.join(',')}" --docker-privileged`,
            `sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner ${props.gitlabRunnerImage}`,
        ];
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
//# sourceMappingURL=data:application/json;base64,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