"""
Module for functional methods of sequences. Inspired by methods available for F#
(https://docs.microsoft.com/en-us/dotnet/fsharp/language-reference/sequences)
A sequence is a logical series of elements all of one type. Sequences are particularly
useful when you have a large, ordered collection of data but do not necessarily expect
to use all of the elements. Individual sequence elements are computed only as required,
so a sequence can provide better performance than a list in situations in which not all
the elements are used.
"""
import builtins as __builtin__
import itertools as _itertools
import functools as _functools
import operator as _operator


class Pipe:
    """
    Overrides `>>` as an operator to chain Seq methods.
    """
    def __init__(self, function, *args, **kwargs):
        self.function = function
        self.args = args
        self.kwargs = kwargs
    def __rrshift__(self, other):
        return self.function(other, *self.args, **self.kwargs)
    def __call__(self):
        return self.function(*self.args, **self.kwargs)


def pipe(function):
    """
    decorator function to implement function chaining
    """
    @_functools.wraps(function)
    def wrapper(*args, **kwargs):
        return Pipe(function, *args, **kwargs)
    return wrapper


# sequence utility methods ---------------------------


fst = _operator.itemgetter(0)


snd = _operator.itemgetter(1)


def pack(function):
    """convert a multiple input function into a single input that will expand inputs when evoked"""
    return lambda p: function(*p)


def packed_function(function):
    """returns a function with a single input"""
    # needed in python 3.x since lambda functions are not automatically unpacked
    # as they were in python 2.7
    if function.__code__.co_argcount > 1:
        return pack(function)
    return function


class Seq:
    """
    Contains operations for working with values of type sequences (any generator, list, etc.)
    """

    # sequence initialization methods --------------------

    @staticmethod
    def unfold(function, initial):
        """
        generates a sequence from a computation function that takes a state and
        transforms it to produce each subsequent element in the sequence. The state
        is just a value that is used to compute each element, and can change as each
        element is computed. The second argument to Seq.unfold is the initial value
        that is used to start the sequence
        """
        _function = packed_function(function)
        state = initial
        while True:
            res = _function(state)
            if res is None:
                break
            value, state = res
            yield value

    @staticmethod
    def init(n, function):
        """
        Generates a new sequence which, when iterated, will return successive elements by
        calling the given function, up to the given count.
        """
        i = 0
        while i < n:
            yield function(i)
            i += 1

    @staticmethod
    def initInfinite(function):
        """
        Generates a new sequence which, when iterated, will return successive elements by
        calling the given function.
        """
        i = 0
        while True:
            yield function(i)
            i += 1


    # sequence methods ------------------------------------

    @staticmethod
    @pipe
    def sum(sequence):
        """
        Returns the sum of the elements in the sequence.
        """
        return __builtin__.sum(sequence)

    @staticmethod
    @pipe
    def sumBy(sequence, function):
        """
        Returns the sum of the results generated by applying the function to each element
        of the sequence.
        """
        return __builtin__.sum(__builtin__.map(function, sequence))


    @staticmethod
    @pipe
    def scan(sequence, function, state):
        """
        Like fold, but computes on-demand and returns the sequence of
        intermediary and final results.
        """
        yield state
        for x in sequence:
            state = function(state, x)
            yield state


    @staticmethod
    @pipe
    def scani(sequence, function, state):
        """
        Like fold, but computes on-demand and returns the sequence of enumerated
        intermediary and final results.
        """
        i = 0
        yield i, state
        for x in sequence:
            state = function(state, x)
            i += 1
            yield i, state

    @staticmethod
    @pipe
    def filter(sequence, function):
        """
        Returns a new collection containing only the elements of the collection
        for which the given predicate returns "true".
        """
        return __builtin__.filter(function, sequence)

    @staticmethod
    @pipe
    def takeWhile(sequence, function):
        """
        Returns a sequence that, when iterated, yields elements of the underlying sequence while the
        given predicate returns True, and then returns no further elements.
        """
        return _itertools.takewhile(function, sequence)

    @staticmethod
    @pipe
    def skipWhile(sequence, function):
        """
        Returns a sequence that, when iterated, skips elements of the underlying sequence
        while the given predicate returns True, and then yields the remaining elements of
        the sequence.
        """
        return _itertools.dropwhile(function, sequence)

    @staticmethod
    @pipe
    def skip(sequence, n):
        """
        Returns a sequence that skips N elements of the underlying sequence and then yields
        the remaining elements of the sequence.
        """
        _iterator = iter(sequence)
        for _ in range(n):
            next(_iterator)
        return _iterator

    @staticmethod
    @pipe
    def nth(sequence, n):
        """
        Return the n-th element of the given list. The first element (head of the list)
        is at position 0.
        """
        _iterator = iter(sequence)
        for _ in range(n):
            next(_iterator)
        return next(_iterator)

    @staticmethod
    @pipe
    def map(sequence, function):
        """
        Builds a new collection whose elements are the results of applying the given function
        to each of the elements of the collection. The given function will be applied as elements
        are demanded using the `next` method on enumerators retrieved from the object.
        """
        return __builtin__.map(function, sequence)

    @staticmethod
    @pipe
    def mapi(sequence, function):
        """
        Builds a new collection whose elements are the results of applying the given function
        to each of the elements of the collection. The integer index passed to the function
        indicates the index (from 0) of element being transformed.
        """
        _function=packed_function(function)
        return __builtin__.map(_function, enumerate(sequence))

    @staticmethod
    @pipe
    def toList(sequence):
        """
        Builds a list from the given collection.
        """
        return list(sequence)

    @staticmethod
    @pipe
    def toSet(sequence):
        """
        Builds a set from the given collection.
        """
        return set(sequence)

    @staticmethod
    @pipe
    def append(sequence1, sequence2):
        """
        Wraps the two given enumerations as a single concatenated enumeration.
        """
        return _itertools.chain(sequence2, sequence1)

    @staticmethod
    @pipe
    def take(sequence, n):
        """
        Returns the first N elements of the sequence.
        """
        return list(_itertools.islice(sequence, n))

    @staticmethod
    @pipe
    def max(sequence):
        """
        Returns the greatest of all elements of the sequence compared via builtin max operator.
        """
        return __builtin__.max(sequence)

    @staticmethod
    @pipe
    def min(sequence):
        """
        Returns the lowest of all elements of the sequence compared via builtin min operator.
        """
        return __builtin__.min(sequence)

    @staticmethod
    @pipe
    def reduce(sequence, function):
        """
        Applies a function to each element of the sequence, threading an accumulator argument
        through the computation. Begin by applying the function to the first two elements.
        Then feed this result into the function along with the third element and so on.
        Return the final result.
        """
        #_function = packed_function(function)
        return _functools.reduce(function, sequence)

    @staticmethod
    @pipe
    def window(sequence, n=2):
        """
        Returns a sliding window (of width n) over data from the iterable
        s -> (s0,s1,...s[n-1]), (s1,s2,...,sn), ...
        """
        _iterator = iter(sequence)
        result = tuple(_itertools.islice(_iterator, n))
        if len(result) == n:
            yield result
        for elem in _iterator:
            result = result[1:] + (elem,)
            yield result

    @staticmethod
    @pipe
    def product(sequence):
        """
        Returns the product of the elements in the sequence.
        """
        return _functools.reduce(_operator.mul, sequence)

    @staticmethod
    @pipe
    def head(sequence):
        """
        Returns the first element of the sequence.
        """
        return next(_itertools.islice(sequence, 1))

    @staticmethod
    @pipe
    def find(sequence, function):
        """
        Returns the first element for which the given function returns True.
        """
        return next(__builtin__.filter(function, sequence), None)

    @staticmethod
    @pipe
    def findIndex(sequence, function):
        """
        Returns the index of the first element for which the given function returns True.
        """
        return next(__builtin__.filter(lambda x: function(x[1]), enumerate(sequence)), None)[0]

    @staticmethod
    @pipe
    def rev(sequence):
        """
        Returns a new sequence with the elements in reverse order.
        """
        return reversed(list(sequence))

    @staticmethod
    @pipe
    def zip(sequence1, sequence2):
        """
        Combines the two sequences into a sequence of pairs. The two sequences need
        not have equal lengths: when one sequence is exhausted any remaining elements
        in the other sequence are ignored.
        """
        return __builtin__.zip(sequence1, sequence2)

    @staticmethod
    @pipe
    def flatten(sequence):
        """
        unnest each list item stored in the collection of collections
        """
        return _itertools.chain.from_iterable(
            __builtin__.map(lambda x: x, sequence))

    @staticmethod
    @pipe
    def length(sequence):
        """
        Returns the length of the sequence
        """
        count = 0
        for _ in sequence:
            count += 1
        return count

    @staticmethod
    @pipe
    def exists(sequence, function):
        """
        Tests if any element of the sequence satisfies the given predicate.
        """
        return __builtin__.any(function(x) for x in sequence)

    @staticmethod
    @pipe
    def collect(sequence, function):
        """
        Applies the given function to each element of the sequence and concatenates all the results.
        """
        for seq in __builtin__.map(function, sequence):
            for x in seq:
                yield x

    @staticmethod
    @pipe
    def distinct(sequence):
        """
        Returns a sequence that contains no duplicate entries according to generic hash
        and equality comparisons on the entries. If an element occurs multiple times in
        the sequence then the later occurrences are discarded.
        """
        seen = set()
        for x in sequence:
            if x in seen:
                continue
            seen.add(x)
            yield x

    @staticmethod
    @pipe
    def forall(sequence, function):
        """
        Tests if all elements of the sequence satisfy the given predicate.
        """
        for x in sequence:
            if not function(x):
                return False
        return True

    @staticmethod
    @pipe
    def maxBy(sequence, function):
        """
        Returns the greatest of all elements of the sequence, compared via `max`
        on the function result.
        """
        return __builtin__.max(sequence, key=function)

    @staticmethod
    @pipe
    def sort(sequence):
        """
        Yields a sequence ordered by keys.
        """
        return __builtin__.sorted(sequence)

    @staticmethod
    @pipe
    def sortBy(sequence, function):
        """
        Applies a key-generating function to each element of a sequence and yield
        a sequence ordered by keys.
        """
        return __builtin__.sorted(sequence, key=function)
