from typing import Any
from typing import Dict

from pydantic import BaseModel
from pydantic import Field


class BuiltinDataset(BaseModel):
    @classmethod
    def add_builtin_kwargs(cls, **kwargs: Any) -> Dict[str, Any]:
        """
        An interface to add builtin information to constructor kwargs. On
        duplicate keys, this method replace replaces existing data.

        :param kwargs: Any parameter to pass to the constructor of
            :class:`BuiltinDataset`
        :return: the given `kwargs` with additional data.

        .. doctest:: Test same in, same out

            >>> BuiltinDataset.add_builtin_kwargs()
            {}
            >>> BuiltinDataset.add_builtin_kwargs(k='v')
            {'k': 'v'}
        """
        return kwargs


class CountrySummary(BuiltinDataset):
    name: str = Field(title="Name", description="The country name")
    continent: str = Field(title="Continent", description="The continent name")
    cc: str = Field(
        title="CC",
        regex=r"^[a-zA-Z]{2}$",
        to_upper=True,
        description="The contintent code",
    )
    iso_a2: str = Field(
        regex=r"^[a-zA-Z]{2}$",
        to_upperc=True,
        title="ISO Alpha2",
        description="A two letter country code (ISO-3166)",
    )
    iso_a3: str = Field(
        regex=r"^[a-zA-Z]{3}$",
        to_upper=True,
        title="ISO Alpha3",
        description="A three letter country code (ISO-3166)",
    )
    number: str = Field(
        regex=r"^\d{3}$",
        title="Number",
        description="A three digit country code (ISO-3166)",
    )
    is_asean: bool = Field(
        description="True, if the country is an ASEAN member. False if not."
    )
    is_eu: bool = Field(
        description="True, if the country is a member of the EU. False if not."
    )
    is_nato: bool = Field(
        description="True, if the country is a NATO member. False if not."
    )
    is_eawu: bool = Field(
        description="True, if the country is a EAWU member. False if not."
    )
    is_g20: bool = Field(
        description="True, if the country is a G20 member. False if not. Since the EU "
        "is a G7 member, each EU country countrs as a G7 member."
    )
    is_g7: bool = Field(
        description="True, if the country is a G7 member. False if not. Since the EU "
        "is a G7 member, each EU country countrs as a G7 member."
    )

    @classmethod
    def add_builtin_kwargs(cls, **kwargs: Any) -> Dict[str, Any]:
        """
        Adds country info from `COUNTRY_INFO_MAP` to the curren `kwargs`. This
        method does not replaces existing keys.

        :param kwargs: Any parameter to pass to the constructor of
            :class:`CountrySummary`
        :return: The given `kwargs` with additional country info.

        .. doctest:: Test existing country

            >>> result = CountrySummary.add_builtin_kwargs(iso_a2="US")
            >>> assert result == COUNTRY_INFO_MAP["US"]

        .. doctest:: Test replace existing keys and return others

            >>> result = CountrySummary.add_builtin_kwargs(
            ...    iso_a2="ZZ",
            ...    is_eu=True,
            ...    foo="bla",
            ... )
            >>> assert result["is_eu"] == True
            >>> assert result["is_nato"] == False
            >>> assert result["foo"] == "bla"

        .. doctest:: Test return passed keys

            >>> result = CountrySummary.add_builtin_kwargs(iso_a2="ZZ", foo="bla")

        .. doctest:: Test missing iso_a2

            >>> result = CountrySummary.add_builtin_kwargs()
            >>> assert result == COUNTRY_INFO_MAP["ZZ"].dict()

        .. doctest:: Test invalid iso_a2

            >>> result = CountrySummary.add_builtin_kwargs(iso_a2="NJS")
            >>> expected = COUNTRY_INFO_MAP["ZZ"].dict()
            >>> expected["iso_a2"] = "NJS"
            >>> assert result == expected

        .. doctest:: Test iso_a2 not found

            >>> result = CountrySummary.add_builtin_kwargs(iso_a2="XX")
            >>> expected = COUNTRY_INFO_MAP["ZZ"].dict()
            >>> expected["iso_a2"] = "XX"
            >>> assert result == expected
        """
        default_iso_a2 = "ZZ"
        iso_a2 = kwargs.get("iso_a2") or default_iso_a2
        iso_a2 = iso_a2.upper()
        if iso_a2 not in COUNTRY_INFO_MAP:
            country_info = COUNTRY_INFO_MAP[default_iso_a2]
        else:
            country_info = COUNTRY_INFO_MAP[iso_a2]

        return {**country_info.dict(), **kwargs}


COUNTRY_INFO_MAP = {
    "AF": CountrySummary(
        name="Afghanistan",
        continent="Asia",
        cc="AS",
        iso_a2="AF",
        iso_a3="AFG",
        number="004",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AL": CountrySummary(
        name="Albania",
        continent="Europe",
        cc="EU",
        iso_a2="AL",
        iso_a3="ALB",
        number="008",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AQ": CountrySummary(
        name="Antarctica (the territory South of 60 deg S)",
        continent="Antarctica",
        cc="AN",
        iso_a2="AQ",
        iso_a3="ATA",
        number="010",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "DZ": CountrySummary(
        name="Algeria",
        continent="Africa",
        cc="AF",
        iso_a2="DZ",
        iso_a3="DZA",
        number="012",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AS": CountrySummary(
        name="American Samoa",
        continent="Oceania",
        cc="OC",
        iso_a2="AS",
        iso_a3="ASM",
        number="016",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AD": CountrySummary(
        name="Andorra",
        continent="Europe",
        cc="EU",
        iso_a2="AD",
        iso_a3="AND",
        number="020",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AO": CountrySummary(
        name="Angola",
        continent="Africa",
        cc="AF",
        iso_a2="AO",
        iso_a3="AGO",
        number="024",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AG": CountrySummary(
        name="Antigua and Barbuda",
        continent="North America",
        cc="NA",
        iso_a2="AG",
        iso_a3="ATG",
        number="028",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AZ": CountrySummary(
        name="Azerbaijan",
        continent="Asia",
        cc="AS",
        iso_a2="AZ",
        iso_a3="AZE",
        number="031",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AR": CountrySummary(
        name="Argentina",
        continent="South America",
        cc="SA",
        iso_a2="AR",
        iso_a3="ARG",
        number="032",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "AU": CountrySummary(
        name="Australia",
        continent="Oceania",
        cc="OC",
        iso_a2="AU",
        iso_a3="AUS",
        number="036",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "AT": CountrySummary(
        name="Austria",
        continent="Europe",
        cc="EU",
        iso_a2="AT",
        iso_a3="AUT",
        number="040",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "BS": CountrySummary(
        name="Bahamas",
        continent="North America",
        cc="NA",
        iso_a2="BS",
        iso_a3="BHS",
        number="044",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BH": CountrySummary(
        name="Bahrain",
        continent="Asia",
        cc="AS",
        iso_a2="BH",
        iso_a3="BHR",
        number="048",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BD": CountrySummary(
        name="Bangladesh",
        continent="Asia",
        cc="AS",
        iso_a2="BD",
        iso_a3="BGD",
        number="050",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AM": CountrySummary(
        name="Armenia",
        continent="Asia",
        cc="AS",
        iso_a2="AM",
        iso_a3="ARM",
        number="051",
        is_asean=False,
        is_nato=False,
        is_eawu=True,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BB": CountrySummary(
        name="Barbados",
        continent="North America",
        cc="NA",
        iso_a2="BB",
        iso_a3="BRB",
        number="052",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BE": CountrySummary(
        name="Belgium",
        continent="Europe",
        cc="EU",
        iso_a2="BE",
        iso_a3="BEL",
        number="056",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "BM": CountrySummary(
        name="Bermuda",
        continent="North America",
        cc="NA",
        iso_a2="BM",
        iso_a3="BMU",
        number="060",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BT": CountrySummary(
        name="Bhutan",
        continent="Asia",
        cc="AS",
        iso_a2="BT",
        iso_a3="BTN",
        number="064",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BO": CountrySummary(
        name="Bolivia",
        continent="South America",
        cc="SA",
        iso_a2="BO",
        iso_a3="BOL",
        number="068",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BA": CountrySummary(
        name="Bosnia and Herzegovina",
        continent="Europe",
        cc="EU",
        iso_a2="BA",
        iso_a3="BIH",
        number="070",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BW": CountrySummary(
        name="Botswana",
        continent="Africa",
        cc="AF",
        iso_a2="BW",
        iso_a3="BWA",
        number="072",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BV": CountrySummary(
        name="Bouvet Island (Bouvetoya)",
        continent="Antarctica",
        cc="AN",
        iso_a2="BV",
        iso_a3="BVT",
        number="074",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BR": CountrySummary(
        name="Brazil",
        continent="South America",
        cc="SA",
        iso_a2="BR",
        iso_a3="BRA",
        number="076",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "BZ": CountrySummary(
        name="Belize",
        continent="North America",
        cc="NA",
        iso_a2="BZ",
        iso_a3="BLZ",
        number="084",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "IO": CountrySummary(
        name="British Indian Ocean Territory (Chagos Archipelago)",
        continent="Asia",
        cc="AS",
        iso_a2="IO",
        iso_a3="IOT",
        number="086",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SB": CountrySummary(
        name="Solomon Islands",
        continent="Oceania",
        cc="OC",
        iso_a2="SB",
        iso_a3="SLB",
        number="090",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "VG": CountrySummary(
        name="British Virgin Islands",
        continent="North America",
        cc="NA",
        iso_a2="VG",
        iso_a3="VGB",
        number="092",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BN": CountrySummary(
        name="Brunei Darussalam",
        continent="Asia",
        cc="AS",
        iso_a2="BN",
        iso_a3="BRN",
        number="096",
        is_asean=True,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BG": CountrySummary(
        name="Bulgaria",
        continent="Europe",
        cc="EU",
        iso_a2="BG",
        iso_a3="BGR",
        number="100",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "MM": CountrySummary(
        name="Myanmar",
        continent="Asia",
        cc="AS",
        iso_a2="MM",
        iso_a3="MMR",
        number="104",
        is_asean=True,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BI": CountrySummary(
        name="Burundi",
        continent="Africa",
        cc="AF",
        iso_a2="BI",
        iso_a3="BDI",
        number="108",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BY": CountrySummary(
        name="Belarus",
        continent="Europe",
        cc="EU",
        iso_a2="BY",
        iso_a3="BLR",
        number="112",
        is_asean=False,
        is_nato=False,
        is_eawu=True,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "KH": CountrySummary(
        name="Cambodia",
        continent="Asia",
        cc="AS",
        iso_a2="KH",
        iso_a3="KHM",
        number="116",
        is_asean=True,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CM": CountrySummary(
        name="Cameroon",
        continent="Africa",
        cc="AF",
        iso_a2="CM",
        iso_a3="CMR",
        number="120",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CA": CountrySummary(
        name="Canada",
        continent="North America",
        cc="NA",
        iso_a2="CA",
        iso_a3="CAN",
        number="124",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=True,
    ),
    "CV": CountrySummary(
        name="Cape Verde",
        continent="Africa",
        cc="AF",
        iso_a2="CV",
        iso_a3="CPV",
        number="132",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "KY": CountrySummary(
        name="Cayman Islands",
        continent="North America",
        cc="NA",
        iso_a2="KY",
        iso_a3="CYM",
        number="136",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CF": CountrySummary(
        name="Central African Republic",
        continent="Africa",
        cc="AF",
        iso_a2="CF",
        iso_a3="CAF",
        number="140",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "LK": CountrySummary(
        name="Sri Lanka",
        continent="Asia",
        cc="AS",
        iso_a2="LK",
        iso_a3="LKA",
        number="144",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TD": CountrySummary(
        name="Chad",
        continent="Africa",
        cc="AF",
        iso_a2="TD",
        iso_a3="TCD",
        number="148",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CL": CountrySummary(
        name="Chile",
        continent="South America",
        cc="SA",
        iso_a2="CL",
        iso_a3="CHL",
        number="152",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CN": CountrySummary(
        name="China",
        continent="Asia",
        cc="AS",
        iso_a2="CN",
        iso_a3="CHN",
        number="156",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "TW": CountrySummary(
        name="Taiwan",
        continent="Asia",
        cc="AS",
        iso_a2="TW",
        iso_a3="TWN",
        number="158",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CX": CountrySummary(
        name="Christmas Island",
        continent="Asia",
        cc="AS",
        iso_a2="CX",
        iso_a3="CXR",
        number="162",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CC": CountrySummary(
        name="Cocos (Keeling) Islands",
        continent="Asia",
        cc="AS",
        iso_a2="CC",
        iso_a3="CCK",
        number="166",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CO": CountrySummary(
        name="Colombia",
        continent="South America",
        cc="SA",
        iso_a2="CO",
        iso_a3="COL",
        number="170",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "KM": CountrySummary(
        name="Comoros",
        continent="Africa",
        cc="AF",
        iso_a2="KM",
        iso_a3="COM",
        number="174",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "YT": CountrySummary(
        name="Mayotte",
        continent="Africa",
        cc="AF",
        iso_a2="YT",
        iso_a3="MYT",
        number="175",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CG": CountrySummary(
        name="Congo",
        continent="Africa",
        cc="AF",
        iso_a2="CG",
        iso_a3="COG",
        number="178",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CD": CountrySummary(
        name="Congo",
        continent="Africa",
        cc="AF",
        iso_a2="CD",
        iso_a3="COD",
        number="180",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CK": CountrySummary(
        name="Cook Islands",
        continent="Oceania",
        cc="OC",
        iso_a2="CK",
        iso_a3="COK",
        number="184",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CR": CountrySummary(
        name="Costa Rica",
        continent="North America",
        cc="NA",
        iso_a2="CR",
        iso_a3="CRI",
        number="188",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "HR": CountrySummary(
        name="Croatia",
        continent="Europe",
        cc="EU",
        iso_a2="HR",
        iso_a3="HRV",
        number="191",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "CU": CountrySummary(
        name="Cuba",
        continent="North America",
        cc="NA",
        iso_a2="CU",
        iso_a3="CUB",
        number="192",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CY": CountrySummary(
        name="Cyprus",
        continent="Asia",
        cc="AS",
        iso_a2="CY",
        iso_a3="CYP",
        number="196",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "CZ": CountrySummary(
        name="Czech Republic",
        continent="Europe",
        cc="EU",
        iso_a2="CZ",
        iso_a3="CZE",
        number="203",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "BJ": CountrySummary(
        name="Benin",
        continent="Africa",
        cc="AF",
        iso_a2="BJ",
        iso_a3="BEN",
        number="204",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "DK": CountrySummary(
        name="Denmark",
        continent="Europe",
        cc="EU",
        iso_a2="DK",
        iso_a3="DNK",
        number="208",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "DM": CountrySummary(
        name="Dominica",
        continent="North America",
        cc="NA",
        iso_a2="DM",
        iso_a3="DMA",
        number="212",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "DO": CountrySummary(
        name="Dominican Republic",
        continent="North America",
        cc="NA",
        iso_a2="DO",
        iso_a3="DOM",
        number="214",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "EC": CountrySummary(
        name="Ecuador",
        continent="South America",
        cc="SA",
        iso_a2="EC",
        iso_a3="ECU",
        number="218",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SV": CountrySummary(
        name="El Salvador",
        continent="North America",
        cc="NA",
        iso_a2="SV",
        iso_a3="SLV",
        number="222",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GQ": CountrySummary(
        name="Equatorial Guinea",
        continent="Africa",
        cc="AF",
        iso_a2="GQ",
        iso_a3="GNQ",
        number="226",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "ET": CountrySummary(
        name="Ethiopia",
        continent="Africa",
        cc="AF",
        iso_a2="ET",
        iso_a3="ETH",
        number="231",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "ER": CountrySummary(
        name="Eritrea",
        continent="Africa",
        cc="AF",
        iso_a2="ER",
        iso_a3="ERI",
        number="232",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "EE": CountrySummary(
        name="Estonia",
        continent="Europe",
        cc="EU",
        iso_a2="EE",
        iso_a3="EST",
        number="233",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "FO": CountrySummary(
        name="Faroe Islands",
        continent="Europe",
        cc="EU",
        iso_a2="FO",
        iso_a3="FRO",
        number="234",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "FK": CountrySummary(
        name="Falkland Islands (Malvinas)",
        continent="South America",
        cc="SA",
        iso_a2="FK",
        iso_a3="FLK",
        number="238",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GS": CountrySummary(
        name="South Georgia and the South Sandwich Islands",
        continent="Antarctica",
        cc="AN",
        iso_a2="GS",
        iso_a3="SGS",
        number="239",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "FJ": CountrySummary(
        name="Fiji",
        continent="Oceania",
        cc="OC",
        iso_a2="FJ",
        iso_a3="FJI",
        number="242",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "FI": CountrySummary(
        name="Finland",
        continent="Europe",
        cc="EU",
        iso_a2="FI",
        iso_a3="FIN",
        number="246",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "AX": CountrySummary(
        name="Åland Islands",
        continent="Europe",
        cc="EU",
        iso_a2="AX",
        iso_a3="ALA",
        number="248",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "FR": CountrySummary(
        name="France",
        continent="Europe",
        cc="EU",
        iso_a2="FR",
        iso_a3="FRA",
        number="250",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "GF": CountrySummary(
        name="French Guiana",
        continent="South America",
        cc="SA",
        iso_a2="GF",
        iso_a3="GUF",
        number="254",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PF": CountrySummary(
        name="French Polynesia",
        continent="Oceania",
        cc="OC",
        iso_a2="PF",
        iso_a3="PYF",
        number="258",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TF": CountrySummary(
        name="French Southern Territories",
        continent="Antarctica",
        cc="AN",
        iso_a2="TF",
        iso_a3="ATF",
        number="260",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "DJ": CountrySummary(
        name="Djibouti",
        continent="Africa",
        cc="AF",
        iso_a2="DJ",
        iso_a3="DJI",
        number="262",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GA": CountrySummary(
        name="Gabon",
        continent="Africa",
        cc="AF",
        iso_a2="GA",
        iso_a3="GAB",
        number="266",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GE": CountrySummary(
        name="Georgia",
        continent="Asia",
        cc="AS",
        iso_a2="GE",
        iso_a3="GEO",
        number="268",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GM": CountrySummary(
        name="Gambia",
        continent="Africa",
        cc="AF",
        iso_a2="GM",
        iso_a3="GMB",
        number="270",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PS": CountrySummary(
        name="Palestinian Territory",
        continent="Asia",
        cc="AS",
        iso_a2="PS",
        iso_a3="PSE",
        number="275",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "DE": CountrySummary(
        name="Germany",
        continent="Europe",
        cc="EU",
        iso_a2="DE",
        iso_a3="DEU",
        number="276",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "GH": CountrySummary(
        name="Ghana",
        continent="Africa",
        cc="AF",
        iso_a2="GH",
        iso_a3="GHA",
        number="288",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GI": CountrySummary(
        name="Gibraltar",
        continent="Europe",
        cc="EU",
        iso_a2="GI",
        iso_a3="GIB",
        number="292",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "KI": CountrySummary(
        name="Kiribati",
        continent="Oceania",
        cc="OC",
        iso_a2="KI",
        iso_a3="KIR",
        number="296",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GR": CountrySummary(
        name="Greece",
        continent="Europe",
        cc="EU",
        iso_a2="GR",
        iso_a3="GRC",
        number="300",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GL": CountrySummary(
        name="Greenland",
        continent="North America",
        cc="NA",
        iso_a2="GL",
        iso_a3="GRL",
        number="304",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GD": CountrySummary(
        name="Grenada",
        continent="North America",
        cc="NA",
        iso_a2="GD",
        iso_a3="GRD",
        number="308",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GP": CountrySummary(
        name="Guadeloupe",
        continent="North America",
        cc="NA",
        iso_a2="GP",
        iso_a3="GLP",
        number="312",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GU": CountrySummary(
        name="Guam",
        continent="Oceania",
        cc="OC",
        iso_a2="GU",
        iso_a3="GUM",
        number="316",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GT": CountrySummary(
        name="Guatemala",
        continent="North America",
        cc="NA",
        iso_a2="GT",
        iso_a3="GTM",
        number="320",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GN": CountrySummary(
        name="Guinea",
        continent="Africa",
        cc="AF",
        iso_a2="GN",
        iso_a3="GIN",
        number="324",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GY": CountrySummary(
        name="Guyana",
        continent="South America",
        cc="SA",
        iso_a2="GY",
        iso_a3="GUY",
        number="328",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "HT": CountrySummary(
        name="Haiti",
        continent="North America",
        cc="NA",
        iso_a2="HT",
        iso_a3="HTI",
        number="332",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "HM": CountrySummary(
        name="Heard Island and McDonald Islands",
        continent="Antarctica",
        cc="AN",
        iso_a2="HM",
        iso_a3="HMD",
        number="334",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "VA": CountrySummary(
        name="Holy See (Vatican City State)",
        continent="Europe",
        cc="EU",
        iso_a2="VA",
        iso_a3="VAT",
        number="336",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "HN": CountrySummary(
        name="Honduras",
        continent="North America",
        cc="NA",
        iso_a2="HN",
        iso_a3="HND",
        number="340",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "HK": CountrySummary(
        name="Hong Kong",
        continent="Asia",
        cc="AS",
        iso_a2="HK",
        iso_a3="HKG",
        number="344",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "HU": CountrySummary(
        name="Hungary",
        continent="Europe",
        cc="EU",
        iso_a2="HU",
        iso_a3="HUN",
        number="348",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "IS": CountrySummary(
        name="Iceland",
        continent="Europe",
        cc="EU",
        iso_a2="IS",
        iso_a3="ISL",
        number="352",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "IN": CountrySummary(
        name="India",
        continent="Asia",
        cc="AS",
        iso_a2="IN",
        iso_a3="IND",
        number="356",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "ID": CountrySummary(
        name="Indonesia",
        continent="Asia",
        cc="AS",
        iso_a2="ID",
        iso_a3="IDN",
        number="360",
        is_asean=True,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "IR": CountrySummary(
        name="Iran",
        continent="Asia",
        cc="AS",
        iso_a2="IR",
        iso_a3="IRN",
        number="364",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "IQ": CountrySummary(
        name="Iraq",
        continent="Asia",
        cc="AS",
        iso_a2="IQ",
        iso_a3="IRQ",
        number="368",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "IE": CountrySummary(
        name="Ireland",
        continent="Europe",
        cc="EU",
        iso_a2="IE",
        iso_a3="IRL",
        number="372",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "IL": CountrySummary(
        name="Israel",
        continent="Asia",
        cc="AS",
        iso_a2="IL",
        iso_a3="ISR",
        number="376",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "IT": CountrySummary(
        name="Italy",
        continent="Europe",
        cc="EU",
        iso_a2="IT",
        iso_a3="ITA",
        number="380",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "CI": CountrySummary(
        name="Cote d'Ivoire",
        continent="Africa",
        cc="AF",
        iso_a2="CI",
        iso_a3="CIV",
        number="384",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "JM": CountrySummary(
        name="Jamaica",
        continent="North America",
        cc="NA",
        iso_a2="JM",
        iso_a3="JAM",
        number="388",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "JP": CountrySummary(
        name="Japan",
        continent="Asia",
        cc="AS",
        iso_a2="JP",
        iso_a3="JPN",
        number="392",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=True,
    ),
    "KZ": CountrySummary(
        name="Kazakhstan",
        continent="Asia",
        cc="AS",
        iso_a2="KZ",
        iso_a3="KAZ",
        number="398",
        is_asean=False,
        is_nato=False,
        is_eawu=True,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "JO": CountrySummary(
        name="Jordan",
        continent="Asia",
        cc="AS",
        iso_a2="JO",
        iso_a3="JOR",
        number="400",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "KE": CountrySummary(
        name="Kenya",
        continent="Africa",
        cc="AF",
        iso_a2="KE",
        iso_a3="KEN",
        number="404",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "KP": CountrySummary(
        name="Korea",
        continent="Asia",
        cc="AS",
        iso_a2="KP",
        iso_a3="PRK",
        number="408",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "KR": CountrySummary(
        name="Korea",
        continent="Asia",
        cc="AS",
        iso_a2="KR",
        iso_a3="KOR",
        number="410",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "KW": CountrySummary(
        name="Kuwait",
        continent="Asia",
        cc="AS",
        iso_a2="KW",
        iso_a3="KWT",
        number="414",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "KG": CountrySummary(
        name="Kyrgyz Republic",
        continent="Asia",
        cc="AS",
        iso_a2="KG",
        iso_a3="KGZ",
        number="417",
        is_asean=False,
        is_nato=False,
        is_eawu=True,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "LA": CountrySummary(
        name="Lao People's Democratic Republic",
        continent="Asia",
        cc="AS",
        iso_a2="LA",
        iso_a3="LAO",
        number="418",
        is_asean=True,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "LB": CountrySummary(
        name="Lebanon",
        continent="Asia",
        cc="AS",
        iso_a2="LB",
        iso_a3="LBN",
        number="422",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "LS": CountrySummary(
        name="Lesotho",
        continent="Africa",
        cc="AF",
        iso_a2="LS",
        iso_a3="LSO",
        number="426",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "LV": CountrySummary(
        name="Latvia",
        continent="Europe",
        cc="EU",
        iso_a2="LV",
        iso_a3="LVA",
        number="428",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "LR": CountrySummary(
        name="Liberia",
        continent="Africa",
        cc="AF",
        iso_a2="LR",
        iso_a3="LBR",
        number="430",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "LY": CountrySummary(
        name="Libyan Arab Jamahiriya",
        continent="Africa",
        cc="AF",
        iso_a2="LY",
        iso_a3="LBY",
        number="434",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "LI": CountrySummary(
        name="Liechtenstein",
        continent="Europe",
        cc="EU",
        iso_a2="LI",
        iso_a3="LIE",
        number="438",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "LT": CountrySummary(
        name="Lithuania",
        continent="Europe",
        cc="EU",
        iso_a2="LT",
        iso_a3="LTU",
        number="440",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "LU": CountrySummary(
        name="Luxembourg",
        continent="Europe",
        cc="EU",
        iso_a2="LU",
        iso_a3="LUX",
        number="442",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "MO": CountrySummary(
        name="Macao",
        continent="Asia",
        cc="AS",
        iso_a2="MO",
        iso_a3="MAC",
        number="446",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MG": CountrySummary(
        name="Madagascar",
        continent="Africa",
        cc="AF",
        iso_a2="MG",
        iso_a3="MDG",
        number="450",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MW": CountrySummary(
        name="Malawi",
        continent="Africa",
        cc="AF",
        iso_a2="MW",
        iso_a3="MWI",
        number="454",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MY": CountrySummary(
        name="Malaysia",
        continent="Asia",
        cc="AS",
        iso_a2="MY",
        iso_a3="MYS",
        number="458",
        is_asean=True,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MV": CountrySummary(
        name="Maldives",
        continent="Asia",
        cc="AS",
        iso_a2="MV",
        iso_a3="MDV",
        number="462",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "ML": CountrySummary(
        name="Mali",
        continent="Africa",
        cc="AF",
        iso_a2="ML",
        iso_a3="MLI",
        number="466",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MT": CountrySummary(
        name="Malta",
        continent="Europe",
        cc="EU",
        iso_a2="MT",
        iso_a3="MLT",
        number="470",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "MQ": CountrySummary(
        name="Martinique",
        continent="North America",
        cc="NA",
        iso_a2="MQ",
        iso_a3="MTQ",
        number="474",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MR": CountrySummary(
        name="Mauritania",
        continent="Africa",
        cc="AF",
        iso_a2="MR",
        iso_a3="MRT",
        number="478",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MU": CountrySummary(
        name="Mauritius",
        continent="Africa",
        cc="AF",
        iso_a2="MU",
        iso_a3="MUS",
        number="480",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MX": CountrySummary(
        name="Mexico",
        continent="North America",
        cc="NA",
        iso_a2="MX",
        iso_a3="MEX",
        number="484",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "MC": CountrySummary(
        name="Monaco",
        continent="Europe",
        cc="EU",
        iso_a2="MC",
        iso_a3="MCO",
        number="492",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MN": CountrySummary(
        name="Mongolia",
        continent="Asia",
        cc="AS",
        iso_a2="MN",
        iso_a3="MNG",
        number="496",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MD": CountrySummary(
        name="Moldova",
        continent="Europe",
        cc="EU",
        iso_a2="MD",
        iso_a3="MDA",
        number="498",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "ME": CountrySummary(
        name="Montenegro",
        continent="Europe",
        cc="EU",
        iso_a2="ME",
        iso_a3="MNE",
        number="499",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MS": CountrySummary(
        name="Montserrat",
        continent="North America",
        cc="NA",
        iso_a2="MS",
        iso_a3="MSR",
        number="500",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MA": CountrySummary(
        name="Morocco",
        continent="Africa",
        cc="AF",
        iso_a2="MA",
        iso_a3="MAR",
        number="504",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MZ": CountrySummary(
        name="Mozambique",
        continent="Africa",
        cc="AF",
        iso_a2="MZ",
        iso_a3="MOZ",
        number="508",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "OM": CountrySummary(
        name="Oman",
        continent="Asia",
        cc="AS",
        iso_a2="OM",
        iso_a3="OMN",
        number="512",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NA": CountrySummary(
        name="Namibia",
        continent="Africa",
        cc="AF",
        iso_a2="NA",
        iso_a3="NAM",
        number="516",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NR": CountrySummary(
        name="Nauru",
        continent="Oceania",
        cc="OC",
        iso_a2="NR",
        iso_a3="NRU",
        number="520",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NP": CountrySummary(
        name="Nepal",
        continent="Asia",
        cc="AS",
        iso_a2="NP",
        iso_a3="NPL",
        number="524",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NL": CountrySummary(
        name="Netherlands",
        continent="Europe",
        cc="EU",
        iso_a2="NL",
        iso_a3="NLD",
        number="528",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "AN": CountrySummary(
        name="Netherlands Antilles",
        continent="North America",
        cc="NA",
        iso_a2="AN",
        iso_a3="ANT",
        number="530",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "CW": CountrySummary(
        name="Curaçao",
        continent="North America",
        cc="NA",
        iso_a2="CW",
        iso_a3="CUW",
        number="531",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AW": CountrySummary(
        name="Aruba",
        continent="North America",
        cc="NA",
        iso_a2="AW",
        iso_a3="ABW",
        number="533",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SX": CountrySummary(
        name="Sint Maarten (Netherlands)",
        continent="North America",
        cc="NA",
        iso_a2="SX",
        iso_a3="SXM",
        number="534",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BQ": CountrySummary(
        name="Bonaire",
        continent="North America",
        cc="NA",
        iso_a2="BQ",
        iso_a3="BES",
        number="535",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NC": CountrySummary(
        name="New Caledonia",
        continent="Oceania",
        cc="OC",
        iso_a2="NC",
        iso_a3="NCL",
        number="540",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "VU": CountrySummary(
        name="Vanuatu",
        continent="Oceania",
        cc="OC",
        iso_a2="VU",
        iso_a3="VUT",
        number="548",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NZ": CountrySummary(
        name="New Zealand",
        continent="Oceania",
        cc="OC",
        iso_a2="NZ",
        iso_a3="NZL",
        number="554",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NI": CountrySummary(
        name="Nicaragua",
        continent="North America",
        cc="NA",
        iso_a2="NI",
        iso_a3="NIC",
        number="558",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NE": CountrySummary(
        name="Niger",
        continent="Africa",
        cc="AF",
        iso_a2="NE",
        iso_a3="NER",
        number="562",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NG": CountrySummary(
        name="Nigeria",
        continent="Africa",
        cc="AF",
        iso_a2="NG",
        iso_a3="NGA",
        number="566",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NU": CountrySummary(
        name="Niue",
        continent="Oceania",
        cc="OC",
        iso_a2="NU",
        iso_a3="NIU",
        number="570",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NF": CountrySummary(
        name="Norfolk Island",
        continent="Oceania",
        cc="OC",
        iso_a2="NF",
        iso_a3="NFK",
        number="574",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "NO": CountrySummary(
        name="Norway",
        continent="Europe",
        cc="EU",
        iso_a2="NO",
        iso_a3="NOR",
        number="578",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MP": CountrySummary(
        name="Northern Mariana Islands",
        continent="Oceania",
        cc="OC",
        iso_a2="MP",
        iso_a3="MNP",
        number="580",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "UM": CountrySummary(
        name="United States Minor Outlying Islands",
        continent="North America",
        cc="NA",
        iso_a2="UM",
        iso_a3="UMI",
        number="581",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "FM": CountrySummary(
        name="Micronesia",
        continent="Oceania",
        cc="OC",
        iso_a2="FM",
        iso_a3="FSM",
        number="583",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MH": CountrySummary(
        name="Marshall Islands",
        continent="Oceania",
        cc="OC",
        iso_a2="MH",
        iso_a3="MHL",
        number="584",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PW": CountrySummary(
        name="Palau",
        continent="Oceania",
        cc="OC",
        iso_a2="PW",
        iso_a3="PLW",
        number="585",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PK": CountrySummary(
        name="Pakistan",
        continent="Asia",
        cc="AS",
        iso_a2="PK",
        iso_a3="PAK",
        number="586",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PA": CountrySummary(
        name="Panama",
        continent="North America",
        cc="NA",
        iso_a2="PA",
        iso_a3="PAN",
        number="591",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PG": CountrySummary(
        name="Papua New Guinea",
        continent="Oceania",
        cc="OC",
        iso_a2="PG",
        iso_a3="PNG",
        number="598",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PY": CountrySummary(
        name="Paraguay",
        continent="South America",
        cc="SA",
        iso_a2="PY",
        iso_a3="PRY",
        number="600",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PE": CountrySummary(
        name="Peru",
        continent="South America",
        cc="SA",
        iso_a2="PE",
        iso_a3="PER",
        number="604",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PH": CountrySummary(
        name="Philippines",
        continent="Asia",
        cc="AS",
        iso_a2="PH",
        iso_a3="PHL",
        number="608",
        is_asean=True,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PN": CountrySummary(
        name="Pitcairn Islands",
        continent="Oceania",
        cc="OC",
        iso_a2="PN",
        iso_a3="PCN",
        number="612",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PL": CountrySummary(
        name="Poland",
        continent="Europe",
        cc="EU",
        iso_a2="PL",
        iso_a3="POL",
        number="616",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "PT": CountrySummary(
        name="Portugal",
        continent="Europe",
        cc="EU",
        iso_a2="PT",
        iso_a3="PRT",
        number="620",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "GW": CountrySummary(
        name="Guinea-Bissau",
        continent="Africa",
        cc="AF",
        iso_a2="GW",
        iso_a3="GNB",
        number="624",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TL": CountrySummary(
        name="Timor-Leste",
        continent="Asia",
        cc="AS",
        iso_a2="TL",
        iso_a3="TLS",
        number="626",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PR": CountrySummary(
        name="Puerto Rico",
        continent="North America",
        cc="NA",
        iso_a2="PR",
        iso_a3="PRI",
        number="630",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "QA": CountrySummary(
        name="Qatar",
        continent="Asia",
        cc="AS",
        iso_a2="QA",
        iso_a3="QAT",
        number="634",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "RE": CountrySummary(
        name="Reunion",
        continent="Africa",
        cc="AF",
        iso_a2="RE",
        iso_a3="REU",
        number="638",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "RO": CountrySummary(
        name="Romania",
        continent="Europe",
        cc="EU",
        iso_a2="RO",
        iso_a3="ROU",
        number="642",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "RU": CountrySummary(
        name="Russian Federation",
        continent="Asia",
        cc="AS",
        iso_a2="RU",
        iso_a3="RUS",
        number="643",
        is_asean=False,
        is_nato=False,
        is_eawu=True,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "RW": CountrySummary(
        name="Rwanda",
        continent="Africa",
        cc="AF",
        iso_a2="RW",
        iso_a3="RWA",
        number="646",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BL": CountrySummary(
        name="Saint Barthelemy",
        continent="North America",
        cc="NA",
        iso_a2="BL",
        iso_a3="BLM",
        number="652",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SH": CountrySummary(
        name="Saint Helena",
        continent="Africa",
        cc="AF",
        iso_a2="SH",
        iso_a3="SHN",
        number="654",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "KN": CountrySummary(
        name="Saint Kitts and Nevis",
        continent="North America",
        cc="NA",
        iso_a2="KN",
        iso_a3="KNA",
        number="659",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AI": CountrySummary(
        name="Anguilla",
        continent="North America",
        cc="NA",
        iso_a2="AI",
        iso_a3="AIA",
        number="660",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "LC": CountrySummary(
        name="Saint Lucia",
        continent="North America",
        cc="NA",
        iso_a2="LC",
        iso_a3="LCA",
        number="662",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MF": CountrySummary(
        name="Saint Martin",
        continent="North America",
        cc="NA",
        iso_a2="MF",
        iso_a3="MAF",
        number="663",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "PM": CountrySummary(
        name="Saint Pierre and Miquelon",
        continent="North America",
        cc="NA",
        iso_a2="PM",
        iso_a3="SPM",
        number="666",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "VC": CountrySummary(
        name="Saint Vincent and the Grenadines",
        continent="North America",
        cc="NA",
        iso_a2="VC",
        iso_a3="VCT",
        number="670",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SM": CountrySummary(
        name="San Marino",
        continent="Europe",
        cc="EU",
        iso_a2="SM",
        iso_a3="SMR",
        number="674",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "ST": CountrySummary(
        name="Sao Tome and Principe",
        continent="Africa",
        cc="AF",
        iso_a2="ST",
        iso_a3="STP",
        number="678",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SA": CountrySummary(
        name="Saudi Arabia",
        continent="Asia",
        cc="AS",
        iso_a2="SA",
        iso_a3="SAU",
        number="682",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "SN": CountrySummary(
        name="Senegal",
        continent="Africa",
        cc="AF",
        iso_a2="SN",
        iso_a3="SEN",
        number="686",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "RS": CountrySummary(
        name="Serbia",
        continent="Europe",
        cc="EU",
        iso_a2="RS",
        iso_a3="SRB",
        number="688",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SC": CountrySummary(
        name="Seychelles",
        continent="Africa",
        cc="AF",
        iso_a2="SC",
        iso_a3="SYC",
        number="690",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SL": CountrySummary(
        name="Sierra Leone",
        continent="Africa",
        cc="AF",
        iso_a2="SL",
        iso_a3="SLE",
        number="694",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SG": CountrySummary(
        name="Singapore",
        continent="Asia",
        cc="AS",
        iso_a2="SG",
        iso_a3="SGP",
        number="702",
        is_asean=True,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SK": CountrySummary(
        name="Slovakia (Slovak Republic)",
        continent="Europe",
        cc="EU",
        iso_a2="SK",
        iso_a3="SVK",
        number="703",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "VN": CountrySummary(
        name="Vietnam",
        continent="Asia",
        cc="AS",
        iso_a2="VN",
        iso_a3="VNM",
        number="704",
        is_asean=True,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SI": CountrySummary(
        name="Slovenia",
        continent="Europe",
        cc="EU",
        iso_a2="SI",
        iso_a3="SVN",
        number="705",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "SO": CountrySummary(
        name="Somalia",
        continent="Africa",
        cc="AF",
        iso_a2="SO",
        iso_a3="SOM",
        number="706",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "ZA": CountrySummary(
        name="South Africa",
        continent="Africa",
        cc="AF",
        iso_a2="ZA",
        iso_a3="ZAF",
        number="710",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "ZW": CountrySummary(
        name="Zimbabwe",
        continent="Africa",
        cc="AF",
        iso_a2="ZW",
        iso_a3="ZWE",
        number="716",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "ES": CountrySummary(
        name="Spain",
        continent="Europe",
        cc="EU",
        iso_a2="ES",
        iso_a3="ESP",
        number="724",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "SS": CountrySummary(
        name="South Sudan",
        continent="Africa",
        cc="AF",
        iso_a2="SS",
        iso_a3="SSD",
        number="728",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "EH": CountrySummary(
        name="Western Sahara",
        continent="Africa",
        cc="AF",
        iso_a2="EH",
        iso_a3="ESH",
        number="732",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SD": CountrySummary(
        name="Sudan",
        continent="Africa",
        cc="AF",
        iso_a2="SD",
        iso_a3="SDN",
        number="736",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SR": CountrySummary(
        name="Suriname",
        continent="South America",
        cc="SA",
        iso_a2="SR",
        iso_a3="SUR",
        number="740",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SJ": CountrySummary(
        name="Svalbard & Jan Mayen Islands",
        continent="Europe",
        cc="EU",
        iso_a2="SJ",
        iso_a3="SJM",
        number="744",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SZ": CountrySummary(
        name="Swaziland",
        continent="Africa",
        cc="AF",
        iso_a2="SZ",
        iso_a3="SWZ",
        number="748",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SE": CountrySummary(
        name="Sweden",
        continent="Europe",
        cc="EU",
        iso_a2="SE",
        iso_a3="SWE",
        number="752",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=True,
        is_g20=True,
        is_g7=True,
    ),
    "CH": CountrySummary(
        name="Switzerland",
        continent="Europe",
        cc="EU",
        iso_a2="CH",
        iso_a3="CHE",
        number="756",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "SY": CountrySummary(
        name="Syrian Arab Republic",
        continent="Asia",
        cc="AS",
        iso_a2="SY",
        iso_a3="SYR",
        number="760",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TJ": CountrySummary(
        name="Tajikistan",
        continent="Asia",
        cc="AS",
        iso_a2="TJ",
        iso_a3="TJK",
        number="762",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TH": CountrySummary(
        name="Thailand",
        continent="Asia",
        cc="AS",
        iso_a2="TH",
        iso_a3="THA",
        number="764",
        is_asean=True,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TG": CountrySummary(
        name="Togo",
        continent="Africa",
        cc="AF",
        iso_a2="TG",
        iso_a3="TGO",
        number="768",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TK": CountrySummary(
        name="Tokelau",
        continent="Oceania",
        cc="OC",
        iso_a2="TK",
        iso_a3="TKL",
        number="772",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TO": CountrySummary(
        name="Tonga",
        continent="Oceania",
        cc="OC",
        iso_a2="TO",
        iso_a3="TON",
        number="776",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TT": CountrySummary(
        name="Trinidad and Tobago",
        continent="North America",
        cc="NA",
        iso_a2="TT",
        iso_a3="TTO",
        number="780",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "AE": CountrySummary(
        name="United Arab Emirates",
        continent="Asia",
        cc="AS",
        iso_a2="AE",
        iso_a3="ARE",
        number="784",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TN": CountrySummary(
        name="Tunisia",
        continent="Africa",
        cc="AF",
        iso_a2="TN",
        iso_a3="TUN",
        number="788",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TR": CountrySummary(
        name="Turkey",
        continent="Asia",
        cc="AS",
        iso_a2="TR",
        iso_a3="TUR",
        number="792",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "TM": CountrySummary(
        name="Turkmenistan",
        continent="Asia",
        cc="AS",
        iso_a2="TM",
        iso_a3="TKM",
        number="795",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TC": CountrySummary(
        name="Turks and Caicos Islands",
        continent="North America",
        cc="NA",
        iso_a2="TC",
        iso_a3="TCA",
        number="796",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TV": CountrySummary(
        name="Tuvalu",
        continent="Oceania",
        cc="OC",
        iso_a2="TV",
        iso_a3="TUV",
        number="798",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "UG": CountrySummary(
        name="Uganda",
        continent="Africa",
        cc="AF",
        iso_a2="UG",
        iso_a3="UGA",
        number="800",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "UA": CountrySummary(
        name="Ukraine",
        continent="Europe",
        cc="EU",
        iso_a2="UA",
        iso_a3="UKR",
        number="804",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "MK": CountrySummary(
        name="Macedonia",
        continent="Europe",
        cc="EU",
        iso_a2="MK",
        iso_a3="MKD",
        number="807",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "EG": CountrySummary(
        name="Egypt",
        continent="Africa",
        cc="AF",
        iso_a2="EG",
        iso_a3="EGY",
        number="818",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "GB": CountrySummary(
        name="United Kingdom",
        continent="Europe",
        cc="EU",
        iso_a2="GB",
        iso_a3="GBR",
        number="826",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=False,
    ),
    "GG": CountrySummary(
        name="Guernsey",
        continent="Europe",
        cc="EU",
        iso_a2="GG",
        iso_a3="GGY",
        number="831",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "JE": CountrySummary(
        name="Jersey",
        continent="Europe",
        cc="EU",
        iso_a2="JE",
        iso_a3="JEY",
        number="832",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "IM": CountrySummary(
        name="Isle of Man",
        continent="Europe",
        cc="EU",
        iso_a2="IM",
        iso_a3="IMN",
        number="833",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "TZ": CountrySummary(
        name="Tanzania",
        continent="Africa",
        cc="AF",
        iso_a2="TZ",
        iso_a3="TZA",
        number="834",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "US": CountrySummary(
        name="United States of America",
        continent="North America",
        cc="NA",
        iso_a2="US",
        iso_a3="USA",
        number="840",
        is_asean=False,
        is_nato=True,
        is_eawu=False,
        is_eu=False,
        is_g20=True,
        is_g7=True,
    ),
    "VI": CountrySummary(
        name="Virgin Islands",
        continent="North America",
        cc="NA",
        iso_a2="VI",
        iso_a3="VIR",
        number="850",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "BF": CountrySummary(
        name="Burkina Faso",
        continent="Africa",
        cc="AF",
        iso_a2="BF",
        iso_a3="BFA",
        number="854",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "UY": CountrySummary(
        name="Uruguay",
        continent="South America",
        cc="SA",
        iso_a2="UY",
        iso_a3="URY",
        number="858",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "UZ": CountrySummary(
        name="Uzbekistan",
        continent="Asia",
        cc="AS",
        iso_a2="UZ",
        iso_a3="UZB",
        number="860",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "VE": CountrySummary(
        name="Venezuela",
        continent="South America",
        cc="SA",
        iso_a2="VE",
        iso_a3="VEN",
        number="862",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "WF": CountrySummary(
        name="Wallis and Futuna",
        continent="Oceania",
        cc="OC",
        iso_a2="WF",
        iso_a3="WLF",
        number="876",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "WS": CountrySummary(
        name="Samoa",
        continent="Oceania",
        cc="OC",
        iso_a2="WS",
        iso_a3="WSM",
        number="882",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "YE": CountrySummary(
        name="Yemen",
        continent="Asia",
        cc="AS",
        iso_a2="YE",
        iso_a3="YEM",
        number="887",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "ZM": CountrySummary(
        name="Zambiaf",
        continent="Africa",
        cc="AF",
        iso_a2="ZM",
        iso_a3="ZMB",
        number="894",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
    "ZZ": CountrySummary(
        name="Unknown",
        continent="Unknown",
        cc="ZZ",
        iso_a2="ZZ",
        iso_a3="ZZZ",
        number="000",
        is_asean=False,
        is_nato=False,
        is_eawu=False,
        is_eu=False,
        is_g20=False,
        is_g7=False,
    ),
}
