"use strict";
const cdk = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'throws if maxBatchingWindow > 300 seconds'(test) {
        const stack = new cdk.Stack();
        const fn = new lib_1.Function(stack, 'fn', {
            handler: 'index.handler',
            code: lib_1.Code.fromInline('exports.handler = ${handler.toString()}'),
            runtime: lib_1.Runtime.NODEJS_10_X,
        });
        test.throws(() => new lib_1.EventSourceMapping(stack, 'test', {
            target: fn,
            eventSourceArn: '',
            maxBatchingWindow: cdk.Duration.seconds(301),
        }), /maxBatchingWindow cannot be over 300 seconds/);
        test.done();
    },
    'throws if maxRecordAge is below 60 seconds'(test) {
        const stack = new cdk.Stack();
        const fn = new lib_1.Function(stack, 'fn', {
            handler: 'index.handler',
            code: lib_1.Code.fromInline('exports.handler = ${handler.toString()}'),
            runtime: lib_1.Runtime.NODEJS_10_X,
        });
        test.throws(() => new lib_1.EventSourceMapping(stack, 'test', {
            target: fn,
            eventSourceArn: '',
            maxRecordAge: cdk.Duration.seconds(59),
        }), /maxRecordAge must be between 60 seconds and 7 days inclusive/);
        test.done();
    },
    'throws if maxRecordAge is over 7 days'(test) {
        const stack = new cdk.Stack();
        const fn = new lib_1.Function(stack, 'fn', {
            handler: 'index.handler',
            code: lib_1.Code.fromInline('exports.handler = ${handler.toString()}'),
            runtime: lib_1.Runtime.NODEJS_10_X,
        });
        test.throws(() => new lib_1.EventSourceMapping(stack, 'test', {
            target: fn,
            eventSourceArn: '',
            maxRecordAge: cdk.Duration.seconds(604801),
        }), /maxRecordAge must be between 60 seconds and 7 days inclusive/);
        test.done();
    },
    'throws if retryAttempts is negative'(test) {
        const stack = new cdk.Stack();
        const fn = new lib_1.Function(stack, 'fn', {
            handler: 'index.handler',
            code: lib_1.Code.fromInline('exports.handler = ${handler.toString()}'),
            runtime: lib_1.Runtime.NODEJS_10_X,
        });
        test.throws(() => new lib_1.EventSourceMapping(stack, 'test', {
            target: fn,
            eventSourceArn: '',
            retryAttempts: -1,
        }), /retryAttempts must be between 0 and 10000 inclusive, got -1/);
        test.done();
    },
    'throws if retryAttempts is over 10000'(test) {
        const stack = new cdk.Stack();
        const fn = new lib_1.Function(stack, 'fn', {
            handler: 'index.handler',
            code: lib_1.Code.fromInline('exports.handler = ${handler.toString()}'),
            runtime: lib_1.Runtime.NODEJS_10_X,
        });
        test.throws(() => new lib_1.EventSourceMapping(stack, 'test', {
            target: fn,
            eventSourceArn: '',
            retryAttempts: 10001,
        }), /retryAttempts must be between 0 and 10000 inclusive, got 10001/);
        test.done();
    },
    'accepts if retryAttempts is a token'(test) {
        const stack = new cdk.Stack();
        const fn = new lib_1.Function(stack, 'fn', {
            handler: 'index.handler',
            code: lib_1.Code.fromInline('exports.handler = ${handler.toString()}'),
            runtime: lib_1.Runtime.NODEJS_10_X,
        });
        new lib_1.EventSourceMapping(stack, 'test', {
            target: fn,
            eventSourceArn: '',
            retryAttempts: cdk.Lazy.numberValue({ produce: () => 100 }),
        });
        test.done();
    },
    'throws if parallelizationFactor is below 1'(test) {
        const stack = new cdk.Stack();
        const fn = new lib_1.Function(stack, 'fn', {
            handler: 'index.handler',
            code: lib_1.Code.fromInline('exports.handler = ${handler.toString()}'),
            runtime: lib_1.Runtime.NODEJS_10_X,
        });
        test.throws(() => new lib_1.EventSourceMapping(stack, 'test', {
            target: fn,
            eventSourceArn: '',
            parallelizationFactor: 0,
        }), /parallelizationFactor must be between 1 and 10 inclusive, got 0/);
        test.done();
    },
    'throws if parallelizationFactor is over 10'(test) {
        const stack = new cdk.Stack();
        const fn = new lib_1.Function(stack, 'fn', {
            handler: 'index.handler',
            code: lib_1.Code.fromInline('exports.handler = ${handler.toString()}'),
            runtime: lib_1.Runtime.NODEJS_10_X,
        });
        test.throws(() => new lib_1.EventSourceMapping(stack, 'test', {
            target: fn,
            eventSourceArn: '',
            parallelizationFactor: 11,
        }), /parallelizationFactor must be between 1 and 10 inclusive, got 11/);
        test.done();
    },
    'accepts if parallelizationFactor is a token'(test) {
        const stack = new cdk.Stack();
        const fn = new lib_1.Function(stack, 'fn', {
            handler: 'index.handler',
            code: lib_1.Code.fromInline('exports.handler = ${handler.toString()}'),
            runtime: lib_1.Runtime.NODEJS_10_X,
        });
        new lib_1.EventSourceMapping(stack, 'test', {
            target: fn,
            eventSourceArn: '',
            parallelizationFactor: cdk.Lazy.numberValue({ produce: () => 20 }),
        });
        test.done();
    },
    'import event source mapping'(test) {
        const stack = new cdk.Stack(undefined, undefined, { stackName: 'test-stack' });
        const imported = lib_1.EventSourceMapping.fromEventSourceMappingId(stack, 'imported', '14e0db71-5d35-4eb5-b481-8945cf9d10c2');
        test.equals(imported.eventSourceMappingId, '14e0db71-5d35-4eb5-b481-8945cf9d10c2');
        test.equals(imported.stack.stackName, 'test-stack');
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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