"use strict";
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const lambda = require("../lib");
function defineFunction(code, runtime = lambda.Runtime.NODEJS_10_X) {
    const stack = new cdk.Stack();
    return new lambda.Function(stack, 'Func', {
        handler: 'foom',
        code,
        runtime,
    });
}
function generateRandomString(bytes) {
    let s = '';
    for (let i = 0; i < bytes; ++i) {
        s += String.fromCharCode(Math.round(Math.random() * 256));
    }
    return s;
}
module.exports = {
    'lambda.Code.fromInline': {
        'fails if used with unsupported runtimes'(test) {
            test.throws(() => defineFunction(lambda.Code.fromInline('boom'), lambda.Runtime.GO_1_X), /Inline source not allowed for go1\.x/);
            test.throws(() => defineFunction(lambda.Code.fromInline('boom'), lambda.Runtime.JAVA_8), /Inline source not allowed for java8/);
            test.done();
        },
        'fails if larger than 4096 bytes'(test) {
            test.throws(() => defineFunction(lambda.Code.fromInline(generateRandomString(4097)), lambda.Runtime.NODEJS_10_X), /Lambda source is too large, must be <= 4096 but is 4097/);
            test.done();
        },
    },
    'lambda.Code.fromAsset': {
        'fails if a non-zip asset is used'(test) {
            // GIVEN
            const fileAsset = lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler', 'index.py'));
            // THEN
            test.throws(() => defineFunction(fileAsset), /Asset must be a \.zip file or a directory/);
            test.done();
        },
        'only one Asset object gets created even if multiple functions use the same AssetCode'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'MyStack');
            const directoryAsset = lambda.Code.fromAsset(path.join(__dirname, 'my-lambda-handler'));
            // WHEN
            new lambda.Function(stack, 'Func1', {
                handler: 'foom',
                runtime: lambda.Runtime.NODEJS_10_X,
                code: directoryAsset,
            });
            new lambda.Function(stack, 'Func2', {
                handler: 'foom',
                runtime: lambda.Runtime.NODEJS_10_X,
                code: directoryAsset,
            });
            // THEN
            const assembly = app.synth();
            const synthesized = assembly.stacks[0];
            // Func1 has an asset, Func2 does not
            test.deepEqual(synthesized.assets.length, 1);
            test.done();
        },
        'adds code asset metadata'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            stack.node.setContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT, true);
            const location = path.join(__dirname, 'my-lambda-handler');
            // WHEN
            new lambda.Function(stack, 'Func1', {
                code: lambda.Code.fromAsset(location),
                runtime: lambda.Runtime.NODEJS_10_X,
                handler: 'foom',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                Metadata: {
                    [cxapi.ASSET_RESOURCE_METADATA_PATH_KEY]: 'asset.9678c34eca93259d11f2d714177347afd66c50116e1e08996eff893d3ca81232',
                    [cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY]: 'Code',
                },
            }, assert_1.ResourcePart.CompleteDefinition));
            test.done();
        },
    },
    'lambda.Code.fromCfnParameters': {
        "automatically creates the Bucket and Key parameters when it's used in a Function"(test) {
            const stack = new cdk.Stack();
            const code = new lambda.CfnParametersCode();
            new lambda.Function(stack, 'Function', {
                code,
                runtime: lambda.Runtime.NODEJS_10_X,
                handler: 'index.handler',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Code: {
                    S3Bucket: {
                        Ref: 'FunctionLambdaSourceBucketNameParameter9E9E108F',
                    },
                    S3Key: {
                        Ref: 'FunctionLambdaSourceObjectKeyParameter1C7AED11',
                    },
                },
            }));
            test.equal(stack.resolve(code.bucketNameParam), 'FunctionLambdaSourceBucketNameParameter9E9E108F');
            test.equal(stack.resolve(code.objectKeyParam), 'FunctionLambdaSourceObjectKeyParameter1C7AED11');
            test.done();
        },
        'does not allow accessing the Parameter properties before being used in a Function'(test) {
            const code = new lambda.CfnParametersCode();
            test.throws(() => {
                test.notEqual(code.bucketNameParam, undefined);
            }, /bucketNameParam/);
            test.throws(() => {
                test.notEqual(code.objectKeyParam, undefined);
            }, /objectKeyParam/);
            test.done();
        },
        'allows passing custom Parameters when creating it'(test) {
            const stack = new cdk.Stack();
            const bucketNameParam = new cdk.CfnParameter(stack, 'BucketNameParam', {
                type: 'String',
            });
            const bucketKeyParam = new cdk.CfnParameter(stack, 'ObjectKeyParam', {
                type: 'String',
            });
            const code = lambda.Code.fromCfnParameters({
                bucketNameParam,
                objectKeyParam: bucketKeyParam,
            });
            test.equal(stack.resolve(code.bucketNameParam), 'BucketNameParam');
            test.equal(stack.resolve(code.objectKeyParam), 'ObjectKeyParam');
            new lambda.Function(stack, 'Function', {
                code,
                runtime: lambda.Runtime.NODEJS_10_X,
                handler: 'index.handler',
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
                Code: {
                    S3Bucket: {
                        Ref: 'BucketNameParam',
                    },
                    S3Key: {
                        Ref: 'ObjectKeyParam',
                    },
                },
            }));
            test.done();
        },
        'can assign parameters'(test) {
            // given
            const stack = new cdk.Stack();
            const code = new lambda.CfnParametersCode({
                bucketNameParam: new cdk.CfnParameter(stack, 'BucketNameParam', {
                    type: 'String',
                }),
                objectKeyParam: new cdk.CfnParameter(stack, 'ObjectKeyParam', {
                    type: 'String',
                }),
            });
            // when
            const overrides = stack.resolve(code.assign({
                bucketName: 'SomeBucketName',
                objectKey: 'SomeObjectKey',
            }));
            // then
            test.equal(overrides['BucketNameParam'], 'SomeBucketName');
            test.equal(overrides['ObjectKeyParam'], 'SomeObjectKey');
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5jb2RlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC5jb2RlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSw2QkFBNkI7QUFDN0IsNENBQXVGO0FBQ3ZGLHFDQUFxQztBQUNyQyx5Q0FBeUM7QUFFekMsaUNBQWlDO0FBMExqQyxTQUFTLGNBQWMsQ0FBQyxJQUFpQixFQUFFLFVBQTBCLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztJQUM3RixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixPQUFPLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1FBQ3hDLE9BQU8sRUFBRSxNQUFNO1FBQ2YsSUFBSTtRQUNKLE9BQU87S0FDUixDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQsU0FBUyxvQkFBb0IsQ0FBQyxLQUFhO0lBQ3pDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQztJQUNYLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxLQUFLLEVBQUUsRUFBRSxDQUFDLEVBQUU7UUFDOUIsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLEdBQUcsR0FBRyxDQUFDLENBQUMsQ0FBQztLQUMzRDtJQUNELE9BQU8sQ0FBQyxDQUFDO0FBQ1gsQ0FBQztBQXJNRCxpQkFBUztJQUNQLHdCQUF3QixFQUFFO1FBQ3hCLHlDQUF5QyxDQUFDLElBQVU7WUFDbEQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxzQ0FBc0MsQ0FBQyxDQUFDO1lBQ2pJLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUUscUNBQXFDLENBQUMsQ0FBQztZQUNoSSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBQ0QsaUNBQWlDLENBQUMsSUFBVTtZQUMxQyxJQUFJLENBQUMsTUFBTSxDQUNULEdBQUcsRUFBRSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLENBQUMsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLEVBQ3BHLHlEQUF5RCxDQUFDLENBQUM7WUFDN0QsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0lBQ0QsdUJBQXVCLEVBQUU7UUFDdkIsa0NBQWtDLENBQUMsSUFBVTtZQUMzQyxRQUFRO1lBQ1IsTUFBTSxTQUFTLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsbUJBQW1CLEVBQUUsVUFBVSxDQUFDLENBQUMsQ0FBQztZQUUvRixPQUFPO1lBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLEVBQUUsMkNBQTJDLENBQUMsQ0FBQztZQUMxRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsc0ZBQXNGLENBQUMsSUFBVTtZQUMvRixRQUFRO1lBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUM7WUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxTQUFTLENBQUMsQ0FBQztZQUM1QyxNQUFNLGNBQWMsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxtQkFBbUIsQ0FBQyxDQUFDLENBQUM7WUFFeEYsT0FBTztZQUNQLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsT0FBTyxFQUFFO2dCQUNsQyxPQUFPLEVBQUUsTUFBTTtnQkFDZixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO2dCQUNuQyxJQUFJLEVBQUUsY0FBYzthQUNyQixDQUFDLENBQUM7WUFFSCxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLE9BQU8sRUFBRTtnQkFDbEMsT0FBTyxFQUFFLE1BQU07Z0JBQ2YsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztnQkFDbkMsSUFBSSxFQUFFLGNBQWM7YUFDckIsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM3QixNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBRXZDLHFDQUFxQztZQUNyQyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQzdDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCwwQkFBMEIsQ0FBQyxJQUFVO1lBQ25DLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixLQUFLLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsdUNBQXVDLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFFM0UsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsbUJBQW1CLENBQUMsQ0FBQztZQUUzRCxPQUFPO1lBQ1AsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUU7Z0JBQ2xDLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUM7Z0JBQ3JDLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7Z0JBQ25DLE9BQU8sRUFBRSxNQUFNO2FBQ2hCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLEVBQUU7Z0JBQ3JELFFBQVEsRUFBRTtvQkFDUixDQUFDLEtBQUssQ0FBQyxnQ0FBZ0MsQ0FBQyxFQUFFLHdFQUF3RTtvQkFDbEgsQ0FBQyxLQUFLLENBQUMsb0NBQW9DLENBQUMsRUFBRSxNQUFNO2lCQUNyRDthQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7WUFDckMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0lBRUQsK0JBQStCLEVBQUU7UUFDL0Isa0ZBQWtGLENBQUMsSUFBVTtZQUMzRixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO1lBQzVDLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO2dCQUNyQyxJQUFJO2dCQUNKLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7Z0JBQ25DLE9BQU8sRUFBRSxlQUFlO2FBQ3pCLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsdUJBQXVCLEVBQUU7Z0JBQ3pELElBQUksRUFBRTtvQkFDSixRQUFRLEVBQUU7d0JBQ1IsR0FBRyxFQUFFLGlEQUFpRDtxQkFDdkQ7b0JBQ0QsS0FBSyxFQUFFO3dCQUNMLEdBQUcsRUFBRSxnREFBZ0Q7cUJBQ3REO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxFQUFFLGlEQUFpRCxDQUFDLENBQUM7WUFDbkcsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsRUFBRSxnREFBZ0QsQ0FBQyxDQUFDO1lBRWpHLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCxtRkFBbUYsQ0FBQyxJQUFVO1lBQzVGLE1BQU0sSUFBSSxHQUFHLElBQUksTUFBTSxDQUFDLGlCQUFpQixFQUFFLENBQUM7WUFFNUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ2YsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsZUFBZSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBQ2pELENBQUMsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1lBRXRCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRSxTQUFTLENBQUMsQ0FBQztZQUNoRCxDQUFDLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztZQUVyQixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsbURBQW1ELENBQUMsSUFBVTtZQUM1RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGVBQWUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO2dCQUNyRSxJQUFJLEVBQUUsUUFBUTthQUNmLENBQUMsQ0FBQztZQUNILE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7Z0JBQ25FLElBQUksRUFBRSxRQUFRO2FBQ2YsQ0FBQyxDQUFDO1lBRUgsTUFBTSxJQUFJLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQztnQkFDekMsZUFBZTtnQkFDZixjQUFjLEVBQUUsY0FBYzthQUMvQixDQUFDLENBQUM7WUFFSCxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxFQUFFLGlCQUFpQixDQUFDLENBQUM7WUFDbkUsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1lBRWpFLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO2dCQUNyQyxJQUFJO2dCQUNKLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7Z0JBQ25DLE9BQU8sRUFBRSxlQUFlO2FBQ3pCLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsdUJBQXVCLEVBQUU7Z0JBQ3pELElBQUksRUFBRTtvQkFDSixRQUFRLEVBQUU7d0JBQ1IsR0FBRyxFQUFFLGlCQUFpQjtxQkFDdkI7b0JBQ0QsS0FBSyxFQUFFO3dCQUNMLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsdUJBQXVCLENBQUMsSUFBVTtZQUNoQyxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLENBQUMsaUJBQWlCLENBQUM7Z0JBQ3hDLGVBQWUsRUFBRSxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO29CQUM5RCxJQUFJLEVBQUUsUUFBUTtpQkFDZixDQUFDO2dCQUNGLGNBQWMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFO29CQUM1RCxJQUFJLEVBQUUsUUFBUTtpQkFDZixDQUFDO2FBQ0gsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sU0FBUyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQztnQkFDMUMsVUFBVSxFQUFFLGdCQUFnQjtnQkFDNUIsU0FBUyxFQUFFLGVBQWU7YUFDM0IsQ0FBQyxDQUFDLENBQUM7WUFFSixPQUFPO1lBQ1AsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsaUJBQWlCLENBQUMsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1lBQzNELElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLGdCQUFnQixDQUFDLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFFekQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztLQUNGO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBleHBlY3QsIGhhdmVSZXNvdXJjZSwgaGF2ZVJlc291cmNlTGlrZSwgUmVzb3VyY2VQYXJ0IH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGN4YXBpIGZyb20gJ0Bhd3MtY2RrL2N4LWFwaSc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJy4uL2xpYic7XG5cbi8qIGVzbGludC1kaXNhYmxlIGRvdC1ub3RhdGlvbiAqL1xuXG5leHBvcnQgPSB7XG4gICdsYW1iZGEuQ29kZS5mcm9tSW5saW5lJzoge1xuICAgICdmYWlscyBpZiB1c2VkIHdpdGggdW5zdXBwb3J0ZWQgcnVudGltZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IGRlZmluZUZ1bmN0aW9uKGxhbWJkYS5Db2RlLmZyb21JbmxpbmUoJ2Jvb20nKSwgbGFtYmRhLlJ1bnRpbWUuR09fMV9YKSwgL0lubGluZSBzb3VyY2Ugbm90IGFsbG93ZWQgZm9yIGdvMVxcLngvKTtcbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IGRlZmluZUZ1bmN0aW9uKGxhbWJkYS5Db2RlLmZyb21JbmxpbmUoJ2Jvb20nKSwgbGFtYmRhLlJ1bnRpbWUuSkFWQV84KSwgL0lubGluZSBzb3VyY2Ugbm90IGFsbG93ZWQgZm9yIGphdmE4Lyk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdmYWlscyBpZiBsYXJnZXIgdGhhbiA0MDk2IGJ5dGVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICB0ZXN0LnRocm93cyhcbiAgICAgICAgKCkgPT4gZGVmaW5lRnVuY3Rpb24obGFtYmRhLkNvZGUuZnJvbUlubGluZShnZW5lcmF0ZVJhbmRvbVN0cmluZyg0MDk3KSksIGxhbWJkYS5SdW50aW1lLk5PREVKU18xMF9YKSxcbiAgICAgICAgL0xhbWJkYSBzb3VyY2UgaXMgdG9vIGxhcmdlLCBtdXN0IGJlIDw9IDQwOTYgYnV0IGlzIDQwOTcvKTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG4gICdsYW1iZGEuQ29kZS5mcm9tQXNzZXQnOiB7XG4gICAgJ2ZhaWxzIGlmIGEgbm9uLXppcCBhc3NldCBpcyB1c2VkJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgY29uc3QgZmlsZUFzc2V0ID0gbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdteS1sYW1iZGEtaGFuZGxlcicsICdpbmRleC5weScpKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4gZGVmaW5lRnVuY3Rpb24oZmlsZUFzc2V0KSwgL0Fzc2V0IG11c3QgYmUgYSBcXC56aXAgZmlsZSBvciBhIGRpcmVjdG9yeS8pO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdvbmx5IG9uZSBBc3NldCBvYmplY3QgZ2V0cyBjcmVhdGVkIGV2ZW4gaWYgbXVsdGlwbGUgZnVuY3Rpb25zIHVzZSB0aGUgc2FtZSBBc3NldENvZGUnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBhcHAgPSBuZXcgY2RrLkFwcCgpO1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKGFwcCwgJ015U3RhY2snKTtcbiAgICAgIGNvbnN0IGRpcmVjdG9yeUFzc2V0ID0gbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdteS1sYW1iZGEtaGFuZGxlcicpKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ0Z1bmMxJywge1xuICAgICAgICBoYW5kbGVyOiAnZm9vbScsXG4gICAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xMF9YLFxuICAgICAgICBjb2RlOiBkaXJlY3RvcnlBc3NldCxcbiAgICAgIH0pO1xuXG4gICAgICBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnRnVuYzInLCB7XG4gICAgICAgIGhhbmRsZXI6ICdmb29tJyxcbiAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1gsXG4gICAgICAgIGNvZGU6IGRpcmVjdG9yeUFzc2V0LFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGNvbnN0IGFzc2VtYmx5ID0gYXBwLnN5bnRoKCk7XG4gICAgICBjb25zdCBzeW50aGVzaXplZCA9IGFzc2VtYmx5LnN0YWNrc1swXTtcblxuICAgICAgLy8gRnVuYzEgaGFzIGFuIGFzc2V0LCBGdW5jMiBkb2VzIG5vdFxuICAgICAgdGVzdC5kZWVwRXF1YWwoc3ludGhlc2l6ZWQuYXNzZXRzLmxlbmd0aCwgMSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2FkZHMgY29kZSBhc3NldCBtZXRhZGF0YScodGVzdDogVGVzdCkge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgc3RhY2subm9kZS5zZXRDb250ZXh0KGN4YXBpLkFTU0VUX1JFU09VUkNFX01FVEFEQVRBX0VOQUJMRURfQ09OVEVYVCwgdHJ1ZSk7XG5cbiAgICAgIGNvbnN0IGxvY2F0aW9uID0gcGF0aC5qb2luKF9fZGlybmFtZSwgJ215LWxhbWJkYS1oYW5kbGVyJyk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdGdW5jMScsIHtcbiAgICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGxvY2F0aW9uKSxcbiAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1gsXG4gICAgICAgIGhhbmRsZXI6ICdmb29tJyxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywge1xuICAgICAgICBNZXRhZGF0YToge1xuICAgICAgICAgIFtjeGFwaS5BU1NFVF9SRVNPVVJDRV9NRVRBREFUQV9QQVRIX0tFWV06ICdhc3NldC45Njc4YzM0ZWNhOTMyNTlkMTFmMmQ3MTQxNzczNDdhZmQ2NmM1MDExNmUxZTA4OTk2ZWZmODkzZDNjYTgxMjMyJyxcbiAgICAgICAgICBbY3hhcGkuQVNTRVRfUkVTT1VSQ0VfTUVUQURBVEFfUFJPUEVSVFlfS0VZXTogJ0NvZGUnLFxuICAgICAgICB9LFxuICAgICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcbiAgfSxcblxuICAnbGFtYmRhLkNvZGUuZnJvbUNmblBhcmFtZXRlcnMnOiB7XG4gICAgXCJhdXRvbWF0aWNhbGx5IGNyZWF0ZXMgdGhlIEJ1Y2tldCBhbmQgS2V5IHBhcmFtZXRlcnMgd2hlbiBpdCdzIHVzZWQgaW4gYSBGdW5jdGlvblwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgY29kZSA9IG5ldyBsYW1iZGEuQ2ZuUGFyYW1ldGVyc0NvZGUoKTtcbiAgICAgIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdGdW5jdGlvbicsIHtcbiAgICAgICAgY29kZSxcbiAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1gsXG4gICAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIHtcbiAgICAgICAgQ29kZToge1xuICAgICAgICAgIFMzQnVja2V0OiB7XG4gICAgICAgICAgICBSZWY6ICdGdW5jdGlvbkxhbWJkYVNvdXJjZUJ1Y2tldE5hbWVQYXJhbWV0ZXI5RTlFMTA4RicsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBTM0tleToge1xuICAgICAgICAgICAgUmVmOiAnRnVuY3Rpb25MYW1iZGFTb3VyY2VPYmplY3RLZXlQYXJhbWV0ZXIxQzdBRUQxMScsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pKTtcblxuICAgICAgdGVzdC5lcXVhbChzdGFjay5yZXNvbHZlKGNvZGUuYnVja2V0TmFtZVBhcmFtKSwgJ0Z1bmN0aW9uTGFtYmRhU291cmNlQnVja2V0TmFtZVBhcmFtZXRlcjlFOUUxMDhGJyk7XG4gICAgICB0ZXN0LmVxdWFsKHN0YWNrLnJlc29sdmUoY29kZS5vYmplY3RLZXlQYXJhbSksICdGdW5jdGlvbkxhbWJkYVNvdXJjZU9iamVjdEtleVBhcmFtZXRlcjFDN0FFRDExJyk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnZG9lcyBub3QgYWxsb3cgYWNjZXNzaW5nIHRoZSBQYXJhbWV0ZXIgcHJvcGVydGllcyBiZWZvcmUgYmVpbmcgdXNlZCBpbiBhIEZ1bmN0aW9uJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBjb2RlID0gbmV3IGxhbWJkYS5DZm5QYXJhbWV0ZXJzQ29kZSgpO1xuXG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgIHRlc3Qubm90RXF1YWwoY29kZS5idWNrZXROYW1lUGFyYW0sIHVuZGVmaW5lZCk7XG4gICAgICB9LCAvYnVja2V0TmFtZVBhcmFtLyk7XG5cbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgICAgdGVzdC5ub3RFcXVhbChjb2RlLm9iamVjdEtleVBhcmFtLCB1bmRlZmluZWQpO1xuICAgICAgfSwgL29iamVjdEtleVBhcmFtLyk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnYWxsb3dzIHBhc3NpbmcgY3VzdG9tIFBhcmFtZXRlcnMgd2hlbiBjcmVhdGluZyBpdCcodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBidWNrZXROYW1lUGFyYW0gPSBuZXcgY2RrLkNmblBhcmFtZXRlcihzdGFjaywgJ0J1Y2tldE5hbWVQYXJhbScsIHtcbiAgICAgICAgdHlwZTogJ1N0cmluZycsXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IGJ1Y2tldEtleVBhcmFtID0gbmV3IGNkay5DZm5QYXJhbWV0ZXIoc3RhY2ssICdPYmplY3RLZXlQYXJhbScsIHtcbiAgICAgICAgdHlwZTogJ1N0cmluZycsXG4gICAgICB9KTtcblxuICAgICAgY29uc3QgY29kZSA9IGxhbWJkYS5Db2RlLmZyb21DZm5QYXJhbWV0ZXJzKHtcbiAgICAgICAgYnVja2V0TmFtZVBhcmFtLFxuICAgICAgICBvYmplY3RLZXlQYXJhbTogYnVja2V0S2V5UGFyYW0sXG4gICAgICB9KTtcblxuICAgICAgdGVzdC5lcXVhbChzdGFjay5yZXNvbHZlKGNvZGUuYnVja2V0TmFtZVBhcmFtKSwgJ0J1Y2tldE5hbWVQYXJhbScpO1xuICAgICAgdGVzdC5lcXVhbChzdGFjay5yZXNvbHZlKGNvZGUub2JqZWN0S2V5UGFyYW0pLCAnT2JqZWN0S2V5UGFyYW0nKTtcblxuICAgICAgbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ0Z1bmN0aW9uJywge1xuICAgICAgICBjb2RlLFxuICAgICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywge1xuICAgICAgICBDb2RlOiB7XG4gICAgICAgICAgUzNCdWNrZXQ6IHtcbiAgICAgICAgICAgIFJlZjogJ0J1Y2tldE5hbWVQYXJhbScsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBTM0tleToge1xuICAgICAgICAgICAgUmVmOiAnT2JqZWN0S2V5UGFyYW0nLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIGFzc2lnbiBwYXJhbWV0ZXJzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAvLyBnaXZlblxuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBjb2RlID0gbmV3IGxhbWJkYS5DZm5QYXJhbWV0ZXJzQ29kZSh7XG4gICAgICAgIGJ1Y2tldE5hbWVQYXJhbTogbmV3IGNkay5DZm5QYXJhbWV0ZXIoc3RhY2ssICdCdWNrZXROYW1lUGFyYW0nLCB7XG4gICAgICAgICAgdHlwZTogJ1N0cmluZycsXG4gICAgICAgIH0pLFxuICAgICAgICBvYmplY3RLZXlQYXJhbTogbmV3IGNkay5DZm5QYXJhbWV0ZXIoc3RhY2ssICdPYmplY3RLZXlQYXJhbScsIHtcbiAgICAgICAgICB0eXBlOiAnU3RyaW5nJyxcbiAgICAgICAgfSksXG4gICAgICB9KTtcblxuICAgICAgLy8gd2hlblxuICAgICAgY29uc3Qgb3ZlcnJpZGVzID0gc3RhY2sucmVzb2x2ZShjb2RlLmFzc2lnbih7XG4gICAgICAgIGJ1Y2tldE5hbWU6ICdTb21lQnVja2V0TmFtZScsXG4gICAgICAgIG9iamVjdEtleTogJ1NvbWVPYmplY3RLZXknLFxuICAgICAgfSkpO1xuXG4gICAgICAvLyB0aGVuXG4gICAgICB0ZXN0LmVxdWFsKG92ZXJyaWRlc1snQnVja2V0TmFtZVBhcmFtJ10sICdTb21lQnVja2V0TmFtZScpO1xuICAgICAgdGVzdC5lcXVhbChvdmVycmlkZXNbJ09iamVjdEtleVBhcmFtJ10sICdTb21lT2JqZWN0S2V5Jyk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG59O1xuXG5mdW5jdGlvbiBkZWZpbmVGdW5jdGlvbihjb2RlOiBsYW1iZGEuQ29kZSwgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUgPSBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCkge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgcmV0dXJuIG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdGdW5jJywge1xuICAgIGhhbmRsZXI6ICdmb29tJyxcbiAgICBjb2RlLFxuICAgIHJ1bnRpbWUsXG4gIH0pO1xufVxuXG5mdW5jdGlvbiBnZW5lcmF0ZVJhbmRvbVN0cmluZyhieXRlczogbnVtYmVyKSB7XG4gIGxldCBzID0gJyc7XG4gIGZvciAobGV0IGkgPSAwOyBpIDwgYnl0ZXM7ICsraSkge1xuICAgIHMgKz0gU3RyaW5nLmZyb21DaGFyQ29kZShNYXRoLnJvdW5kKE1hdGgucmFuZG9tKCkgKiAyNTYpKTtcbiAgfVxuICByZXR1cm4gcztcbn1cbiJdfQ==