import { CfnRoute } from './index';
import { Construct } from '@aws-cdk/core';
/**
 * Configuration for `HeaderMatch`.
 *
 * @stability stable
 */
export interface HeaderMatchConfig {
    /**
     * Route CFN configuration for the route header match.
     *
     * @stability stable
     */
    readonly headerMatch: CfnRoute.HttpRouteHeaderProperty;
}
/**
 * Used to generate header matching methods.
 *
 * @stability stable
 */
export declare abstract class HeaderMatch {
    /**
     * The value of the header with the given name in the request must match the specified value exactly.
     *
     * @param headerName the name of the header to match against.
     * @param headerValue The exact value to test against.
     * @stability stable
     */
    static valueIs(headerName: string, headerValue: string): HeaderMatch;
    /**
     * The value of the header with the given name in the request must not match the specified value exactly.
     *
     * @param headerName the name of the header to match against.
     * @param headerValue The exact value to test against.
     * @stability stable
     */
    static valueIsNot(headerName: string, headerValue: string): HeaderMatch;
    /**
     * The value of the header with the given name in the request must start with the specified characters.
     *
     * @param headerName the name of the header to match against.
     * @param prefix The prefix to test against.
     * @stability stable
     */
    static valueStartsWith(headerName: string, prefix: string): HeaderMatch;
    /**
     * The value of the header with the given name in the request must not start with the specified characters.
     *
     * @param headerName the name of the header to match against.
     * @param prefix The prefix to test against.
     * @stability stable
     */
    static valueDoesNotStartWith(headerName: string, prefix: string): HeaderMatch;
    /**
     * The value of the header with the given name in the request must end with the specified characters.
     *
     * @param headerName the name of the header to match against.
     * @param suffix The suffix to test against.
     * @stability stable
     */
    static valueEndsWith(headerName: string, suffix: string): HeaderMatch;
    /**
     * The value of the header with the given name in the request must not end with the specified characters.
     *
     * @param headerName the name of the header to match against.
     * @param suffix The suffix to test against.
     * @stability stable
     */
    static valueDoesNotEndWith(headerName: string, suffix: string): HeaderMatch;
    /**
     * The value of the header with the given name in the request must include the specified characters.
     *
     * @param headerName the name of the header to match against.
     * @param regex The regex to test against.
     * @stability stable
     */
    static valueMatchesRegex(headerName: string, regex: string): HeaderMatch;
    /**
     * The value of the header with the given name in the request must not include the specified characters.
     *
     * @param headerName the name of the header to match against.
     * @param regex The regex to test against.
     * @stability stable
     */
    static valueDoesNotMatchRegex(headerName: string, regex: string): HeaderMatch;
    /**
     * The value of the header with the given name in the request must be in a range of values.
     *
     * @param headerName the name of the header to match against.
     * @param start Match on values starting at and including this value.
     * @param end Match on values up to but not including this value.
     * @stability stable
     */
    static valuesIsInRange(headerName: string, start: number, end: number): HeaderMatch;
    /**
     * The value of the header with the given name in the request must not be in a range of values.
     *
     * @param headerName the name of the header to match against.
     * @param start Match on values starting at and including this value.
     * @param end Match on values up to but not including this value.
     * @stability stable
     */
    static valuesIsNotInRange(headerName: string, start: number, end: number): HeaderMatch;
    /**
     * Returns the header match configuration.
     *
     * @stability stable
     */
    abstract bind(scope: Construct): HeaderMatchConfig;
}
