"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
/**
 * (experimental) Property to specify if the compute environment uses On-Demand, SpotFleet, Fargate, or Fargate Spot compute resources.
 *
 * @experimental
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    ComputeResourceType["ON_DEMAND"] = "EC2";
    ComputeResourceType["SPOT"] = "SPOT";
    ComputeResourceType["FARGATE"] = "FARGATE";
    ComputeResourceType["FARGATE_SPOT"] = "FARGATE_SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * (experimental) Properties for how to prepare compute resources that are provisioned for a compute environment.
 *
 * @experimental
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * (experimental) Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 *
 * @experimental
 */
class ComputeEnvironment extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = { enabled: true, managed: true }) {
        var _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        jsiiDeprecationWarnings._aws_cdk_aws_batch_ComputeEnvironmentProps(props);
        const isFargate = ComputeResourceType.FARGATE === ((_b = props.computeResources) === null || _b === void 0 ? void 0 : _b.type)
            || ComputeResourceType.FARGATE_SPOT === ((_c = props.computeResources) === null || _c === void 0 ? void 0 : _c.type);
        ;
        this.validateProps(props, isFargate);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                placementGroup: props.computeResources.placementGroup,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole === null || spotFleetRole === void 0 ? void 0 : spotFleetRole.roleArn,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
                ...(!isFargate ? {
                    allocationStrategy: props.computeResources.allocationStrategy
                        || (props.computeResources.type === ComputeResourceType.SPOT
                            ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                            : AllocationStrategy.BEST_FIT),
                    instanceRole: props.computeResources.instanceRole
                        ? props.computeResources.instanceRole
                        : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                            roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                    assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                    managedPolicies: [
                                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                    ],
                                }).roleName],
                        }).attrArn,
                    instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                    minvCpus: props.computeResources.minvCpus || 0,
                } : {}),
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: (_e = (_d = props.serviceRole) === null || _d === void 0 ? void 0 : _d.roleArn) !== null && _e !== void 0 ? _e : new iam.Role(this, 'Resource-Service-Instance-Role', {
                managedPolicies: [
                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                ],
                assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
            }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * (experimental) Fetches an existing batch compute environment by its amazon resource name.
     *
     * @experimental
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.splitArn(computeEnvironmentArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        var _b;
        return (_b = props.managed) !== null && _b !== void 0 ? _b : true;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props, isFargate) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        if (props.computeResources) {
            if (isFargate) {
                // VALIDATE FOR FARGATE
                // Bid percentage cannot be set for Fargate evnvironments
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Bid percentage must not be set for Fargate compute environments');
                }
                // Allocation strategy cannot be set for Fargate evnvironments
                if (props.computeResources.allocationStrategy !== undefined) {
                    throw new Error('Allocation strategy must not be set for Fargate compute environments');
                }
                // Desired vCPUs cannot be set for Fargate evnvironments
                if (props.computeResources.desiredvCpus !== undefined) {
                    throw new Error('Desired vCPUs must not be set for Fargate compute environments');
                }
                // Image ID cannot be set for Fargate evnvironments
                if (props.computeResources.image !== undefined) {
                    throw new Error('Image must not be set for Fargate compute environments');
                }
                // Instance types cannot be set for Fargate evnvironments
                if (props.computeResources.instanceTypes !== undefined) {
                    throw new Error('Instance types must not be set for Fargate compute environments');
                }
                // EC2 key pair cannot be set for Fargate evnvironments
                if (props.computeResources.ec2KeyPair !== undefined) {
                    throw new Error('EC2 key pair must not be set for Fargate compute environments');
                }
                // Instance role cannot be set for Fargate evnvironments
                if (props.computeResources.instanceRole !== undefined) {
                    throw new Error('Instance role must not be set for Fargate compute environments');
                }
                // Launch template cannot be set for Fargate evnvironments
                if (props.computeResources.launchTemplate !== undefined) {
                    throw new Error('Launch template must not be set for Fargate compute environments');
                }
                // Min vCPUs cannot be set for Fargate evnvironments
                if (props.computeResources.minvCpus !== undefined) {
                    throw new Error('Min vCPUs must not be set for Fargate compute environments');
                }
                // Placement group cannot be set for Fargate evnvironments
                if (props.computeResources.placementGroup !== undefined) {
                    throw new Error('Placement group must not be set for Fargate compute environments');
                }
                // Spot fleet role cannot be set for Fargate evnvironments
                if (props.computeResources.spotFleetRole !== undefined) {
                    throw new Error('Spot fleet role must not be set for Fargate compute environments');
                }
            }
            else {
                // VALIDATE FOR ON_DEMAND AND SPOT
                if (props.computeResources.minvCpus) {
                    // minvCpus cannot be less than 0
                    if (props.computeResources.minvCpus < 0) {
                        throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                    }
                    // minvCpus cannot exceed max vCpus
                    if (props.computeResources.maxvCpus &&
                        props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                        throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                    }
                }
                // Setting a bid percentage is only allowed on SPOT resources +
                // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
                if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                    // VALIDATE FOR ON_DEMAND
                    // Bid percentage is not allowed
                    if (props.computeResources.bidPercentage !== undefined) {
                        throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                    }
                    // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                    if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                        throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                    }
                }
                else if (props.computeResources.type === ComputeResourceType.SPOT) {
                    // VALIDATE FOR SPOT
                    // Bid percentage must be from 0 - 100
                    if (props.computeResources.bidPercentage !== undefined &&
                        (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                        throw new Error('Bid percentage can only be a value between 0 and 100');
                    }
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        var _b;
        if (((_b = props.computeResources) === null || _b === void 0 ? void 0 : _b.allocationStrategy) && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.stack.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
_a = JSII_RTTI_SYMBOL_1;
ComputeEnvironment[_a] = { fqn: "@aws-cdk/aws-batch.ComputeEnvironment", version: "1.143.0" };
//# sourceMappingURL=data:application/json;base64,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