# A simple PV wrapper class.

import numpy
import weakref
import time

from . import cothread
from . import catools
from . import cadef

__all__ = ['PV', 'PV_array']


# This class implements an indirect call to target.method and holds on to target
# using a weak reference.  This helps to break the reference count otherwise
# generated by the camonitor callback and allows automatic closing of the object
# when it's dropped.
class _WeakMethod:
    def __init__(self, target, method):
        self.target = weakref.proxy(target)
        self.method = method

    def __call__(self, *args):
        getattr(self.target, self.method)(*args)


class PV(object):
    '''PV wrapper class.  Wraps access to a single PV as a persistent object
    with simple access methods.  Always contains the latest PV value.

    WARNING!  This API is a work in progress and may change in future releases
    in incompatible ways.'''

    def __init__(self, pv,
            on_update = None, initial_value = None, caput_wait = False,
            initial_timeout = (), **kargs):

        assert isinstance(pv, str), 'PV class only works for one PV at a time'

        self.name = pv
        self.__value_event = cothread.Event()
        self.__sync = cothread.Event(auto_reset = False)
        self.__value = initial_value
        self.caput_wait = caput_wait
        self.datatype = kargs.get('datatype', None)
        self.format = kargs.get('format', catools.FORMAT_RAW)

        self.__deadline_set = initial_timeout != ()
        assert initial_value is None or not self.__deadline_set, \
            'Cannot specify initial value and initial timeout'
        if self.__deadline_set:
            self.__deadline = cothread.AbsTimeout(initial_timeout)

        self.on_update = on_update
        self.__monitor = catools.camonitor(
            pv, _WeakMethod(self, '_on_update'), **kargs)

    def __del__(self):
        self.close()

    def close(self):
        self.__monitor.close()

    def _on_update(self, value):
        self.__value = value
        self.__value_event.Signal(value)
        self.__sync.Signal()
        if self.on_update:
            self.on_update(self)

    def sync(self, timeout = ()):
        '''Blocks until at least one update has been seen.'''
        if timeout == ():
            assert self.__deadline_set, 'Must specify sync timeout'
            timeout = self.__deadline
        catools.ca_timeout(self.__sync, timeout, self.name)

    def get(self):
        '''Returns current value.'''
        if self.__value is None and self.__deadline_set:
            catools.ca_timeout(self.__sync, self.__deadline, self.name)
        return self.__value

    def get_next(self, timeout = None, reset = False):
        '''Returns current value or blocks until next update.  Call .reset()
        first if more recent value required.'''
        if reset:
            self.reset()
        return catools.ca_timeout(self.__value_event, timeout, self.name)

    def reset(self):
        '''Ensures .get_next() will block until an update occurs.'''
        self.__value_event.Reset()

    def caput(self, value, **kargs):
        kargs.setdefault('wait', self.caput_wait)
        kargs.setdefault('datatype', self.datatype)
        return catools.caput(self.name, value, **kargs)

    def caget(self, **kargs):
        kargs.setdefault('datatype', self.datatype)
        kargs.setdefault('format', self.format)
        return catools.caget(self.name, **kargs)

    value = property(get, caput)


class PV_array(object):
    '''PV waveform wrapper class.  Wraps access to a list of PVs as a single
    waveform with simple access methods.  This class will only work if all of
    the PVs are of the same datatype and the same length.

    WARNING!  This API is a work in progress and may change in future releases
    in incompatible ways.'''

    def __init__(self, pvs,
            dtype = float, count = 1, on_update = None, caput_wait = False,
            **kargs):

        assert not isinstance(pvs, str), \
            'PV_array class only works for an array of PVs'

        self.names = pvs
        self.on_update = on_update
        self.dtype = dtype
        self.count = count
        self.caput_wait = caput_wait

        if count == 1:
            self.shape = len(pvs)
        else:
            self.shape = (len(pvs), count)
        self.__value = numpy.zeros(self.shape, dtype = dtype)
        self.seen = numpy.zeros(len(pvs), dtype = bool)
        self.__ok = numpy.zeros(len(pvs), dtype = bool)
        self.__timestamp = numpy.zeros(len(pvs), dtype = float)
        self.__severity = numpy.zeros(len(pvs), dtype = numpy.int16)
        self.__status   = numpy.zeros(len(pvs), dtype = numpy.int16)

        self.__monitors = catools.camonitor(
            pvs, _WeakMethod(self, '_on_update'),
            count = count, datatype = dtype,
            format = catools.FORMAT_TIME, notify_disconnect = True, **kargs)

    def __del__(self):
        self.close()

    def close(self):
        for monitor in self.__monitors:
            monitor.close()

    def _update_one(self, value, index):
        self.seen[index] = True
        self.__ok[index] = value.ok
        if value.ok:
            self.__value[index] = value
            self.__timestamp[index] = value.timestamp
            self.__severity[index] = value.severity
            self.__status[index] = value.status

    def _on_update(self, value, index):
        self._update_one(value, index)
        if self.on_update:
            self.on_update(self, index)

    def get(self):
        return +self.__value

    def caget(self, **kargs):
        dtype = kargs.pop('dtype', self.dtype)
        count = kargs.pop('count', self.count)
        return catools.caget(
            self.names, count = count, datatype = dtype,
            format=catools.FORMAT_TIME, **kargs)

    def sync(self, timeout = 5, throw = True):
        values = self.caget(timeout = timeout, throw = throw)
        for index, value in enumerate(values):
            if not self.seen[index]:
                self._update_one(value, index)

    def caput(self, value, **kargs):
        kargs.setdefault('wait', self.caput_wait)
        return catools.caput(self.names, value, **kargs)

    value = property(get, caput)
    ok        = property(lambda self: +self.__ok)
    timestamp = property(lambda self: +self.__timestamp)
    severity  = property(lambda self: +self.__severity)
    status    = property(lambda self: +self.__status)

    @property
    def all_ok(self):
        return numpy.all(self.__ok)
