"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'CodeCommit Repositories': {
        'add an SNS trigger to repository'(test) {
            const stack = new core_1.Stack();
            const props = {
                repositoryName: 'MyRepository',
            };
            const snsArn = 'arn:aws:sns:*:123456789012:my_topic';
            new lib_1.Repository(stack, 'MyRepository', props).notify(snsArn);
            assert_1.expect(stack).toMatch({
                Resources: {
                    MyRepository4C4BD5FC: {
                        Type: "AWS::CodeCommit::Repository",
                        Properties: {
                            RepositoryName: "MyRepository",
                            Triggers: [
                                {
                                    Events: [
                                        "all"
                                    ],
                                    DestinationArn: "arn:aws:sns:*:123456789012:my_topic",
                                    Name: "MyRepository/arn:aws:sns:*:123456789012:my_topic"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
        'fails when triggers have duplicate names'(test) {
            const stack = new core_1.Stack();
            const myRepository = new lib_1.Repository(stack, 'MyRepository', {
                repositoryName: 'MyRepository',
            }).notify('myTrigger');
            test.throws(() => myRepository.notify('myTrigger'));
            test.done();
        },
        'can be imported using a Repository ARN'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const repositoryArn = 'arn:aws:codecommit:us-east-1:585695036304:my-repo';
            // WHEN
            const repo = lib_1.Repository.fromRepositoryArn(stack, 'ImportedRepo', repositoryArn);
            // THEN
            test.deepEqual(stack.resolve(repo.repositoryArn), repositoryArn);
            test.deepEqual(stack.resolve(repo.repositoryName), 'my-repo');
            test.done();
        },
        'can be imported using just a Repository name (the ARN is deduced)'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            const repo = lib_1.Repository.fromRepositoryName(stack, 'ImportedRepo', 'my-repo');
            // THEN
            test.deepEqual(stack.resolve(repo.repositoryArn), {
                'Fn::Join': ['', [
                        'arn:',
                        { Ref: 'AWS::Partition' },
                        ':codecommit:',
                        { Ref: 'AWS::Region' },
                        ':',
                        { Ref: 'AWS::AccountId' },
                        ':my-repo'
                    ]],
            });
            test.deepEqual(stack.resolve(repo.repositoryName), 'my-repo');
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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