"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const api = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const core_1 = require("@aws-solutions-constructs/core");
// App setup
const app = new cdk.App();
const stack = new cdk.Stack(app, core_1.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-apigateway-iot with overriden params';
const apiGatewayProps = {
    restApiName: 'RestApi-Regional',
    description: 'Description for the Regional Rest Api',
    endpointConfiguration: { types: [api.EndpointType.REGIONAL] },
    apiKeySourceType: api.ApiKeySourceType.HEADER,
    defaultMethodOptions: {
        authorizationType: api.AuthorizationType.NONE,
    }
};
const policyJSON = {
    Version: "2012-10-17",
    Statement: [
        {
            Action: [
                "iot:UpdateThingShadow"
            ],
            Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
            Effect: "Allow"
        },
        {
            Action: [
                "iot:Publish"
            ],
            Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
            Effect: "Allow"
        }
    ]
};
const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
const iamRoleProps = {
    assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
    path: '/',
    inlinePolicies: { testPolicy: policyDocument }
};
// Create a policy that overrides the default policy that gets created with the construct
const apiGatewayExecutionRole = new iam.Role(stack, 'apigateway-iot-role', iamRoleProps);
// Api gateway setup
const props = {
    iotEndpoint: 'a1234567890123-ats',
    apiGatewayCreateApiKey: true,
    apiGatewayExecutionRole,
    apiGatewayProps
};
// Instantiate construct
new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot', props);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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