"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToIot = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("aws-cdk-lib/aws-apigateway");
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
/**
 * @summary The ApiGatewayIot class.
 */
class ApiGatewayToIot extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayIot class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToIotProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // IoT Core topic nesting. A topic in a publish or subscribe request can have no more than 7 forward slashes (/).
        // This excludes the first 3 slashes in the mandatory segments for Basic Ingest
        // Refer IoT Limits - https://docs.aws.amazon.com/general/latest/gr/iot-core.html#limits_iot
        this.topicNestingLevel = 7;
        defaults.CheckProps(props);
        // Assignment to local member variables to make these available to all member methods of the class.
        // (Split the string just in case user supplies fully qualified endpoint eg. ab123cdefghij4l-ats.iot.ap-south-1.amazonaws.com)
        this.iotEndpoint = props.iotEndpoint.trim().split('.')[0];
        // Mandatory params check
        if (!this.iotEndpoint || this.iotEndpoint.length < 0) {
            throw new Error('specify a valid iotEndpoint');
        }
        // Add additional params to the apiGatewayProps
        let extraApiGwProps = {
            binaryMediaTypes: ['application/octet-stream'],
            defaultMethodOptions: {
                apiKeyRequired: props.apiGatewayCreateApiKey
            }
        };
        // If apiGatewayProps are specified override the extra Api Gateway properties
        extraApiGwProps = defaults.consolidateProps(extraApiGwProps, props.apiGatewayProps);
        // Check whether an API Gateway execution role is specified?
        if (props.apiGatewayExecutionRole) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            // JSON that will be used for policy document
            const policyJSON = {
                Version: "2012-10-17",
                Statement: [
                    {
                        Action: [
                            "iot:UpdateThingShadow"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
                        Effect: "Allow"
                    },
                    {
                        Action: [
                            "iot:Publish"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
                        Effect: "Allow"
                    }
                ]
            };
            // Create a policy document
            const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
            // Props for IAM Role
            const iamRoleProps = {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                path: '/',
                inlinePolicies: { awsapigatewayiotpolicy: policyDocument }
            };
            // Create a policy that overrides the default policy that gets created with the construct
            this.apiGatewayRole = new iam.Role(this, 'apigateway-iot-role', iamRoleProps);
        }
        // Setup the API Gateway
        const globalRestApiResponse = defaults.GlobalRestApi(this, extraApiGwProps, props.logGroupProps);
        this.apiGateway = globalRestApiResponse.api;
        this.apiGatewayCloudWatchRole = globalRestApiResponse.role;
        this.apiGatewayLogGroup = globalRestApiResponse.logGroup;
        // Validate the Query Params
        const requestValidatorProps = {
            restApi: this.apiGateway,
            validateRequestBody: false,
            validateRequestParameters: true
        };
        this.requestValidator = new api.RequestValidator(this, `aws-apigateway-iot-req-val`, requestValidatorProps);
        // Create a resource for messages '/message'
        const msgResource = this.apiGateway.root.addResource('message');
        // Create resources from '/message/{topic-level-1}' through '/message/{topic-level-1}/..../{topic-level-7}'
        let topicPath = 'topics';
        let parentNode = msgResource;
        let integParams = {};
        let methodParams = {};
        for (let pathLevel = 1; pathLevel <= this.topicNestingLevel; pathLevel++) {
            const topicName = `topic-level-${pathLevel}`;
            const topicResource = parentNode.addResource(`{${topicName}}`);
            const integReqParam = JSON.parse(`{"integration.request.path.${topicName}": "method.request.path.${topicName}"}`);
            const methodReqParam = JSON.parse(`{"method.request.path.${topicName}": true}`);
            topicPath = `${topicPath}/{${topicName}}`;
            integParams = Object.assign(integParams, integReqParam);
            methodParams = Object.assign(methodParams, methodReqParam);
            this.addResourceMethod(topicResource, props, topicPath, integParams, methodParams);
            parentNode = topicResource;
        }
        // Create a resource for shadow updates '/shadow'
        const shadowResource = this.apiGateway.root.addResource('shadow');
        // Create resource '/shadow/{thingName}'
        const defaultShadowResource = shadowResource.addResource('{thingName}');
        const shadowReqParams = { 'integration.request.path.thingName': 'method.request.path.thingName' };
        const methodShadowReqParams = { 'method.request.path.thingName': true };
        this.addResourceMethod(defaultShadowResource, props, 'things/{thingName}/shadow', shadowReqParams, methodShadowReqParams);
        // Create resource '/shadow/{thingName}/{shadowName}'
        const namedShadowResource = defaultShadowResource.addResource('{shadowName}');
        const namedShadowReqParams = Object.assign({
            'integration.request.path.shadowName': 'method.request.path.shadowName'
        }, shadowReqParams);
        const methodNamedShadowReqParams = Object.assign({
            'method.request.path.shadowName': true
        }, methodShadowReqParams);
        this.addResourceMethod(namedShadowResource, props, 'things/{thingName}/shadow?name={shadowName}', namedShadowReqParams, methodNamedShadowReqParams);
    }
    /**
     * Adds a method to specified resource
     * @param resource API Gateway resource to which this method is added
     * @param resourcePath path of resource from root
     * @param integReqParams request paramters for the Integration method
     * @param methodReqParams request parameters at Method level
     */
    addResourceMethod(resource, props, resourcePath, integReqParams, methodReqParams) {
        const integResp = [
            {
                statusCode: "200",
                selectionPattern: "2\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "500",
                selectionPattern: "5\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "403",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            }
        ];
        // Method responses for the resource
        const methodResp = [
            {
                statusCode: "200"
            },
            {
                statusCode: "500"
            },
            {
                statusCode: "403"
            }
        ];
        // Override the default Integration Request Props
        const integrationReqProps = {
            subdomain: this.iotEndpoint,
            options: {
                requestParameters: integReqParams,
                integrationResponses: integResp,
                passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
            }
        };
        // Override the default Method Options
        const resourceMethodOptions = {
            requestParameters: methodReqParams,
            methodResponses: methodResp,
        };
        const resourceMethodParams = {
            service: 'iotdata',
            path: resourcePath,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: resource,
            requestTemplate: "$input.json('$')",
            requestValidator: this.requestValidator,
            awsIntegrationProps: integrationReqProps,
            methodOptions: resourceMethodOptions
        };
        const apiMethod = defaults.addProxyMethodToApiResource(resourceMethodParams);
        if (props.apiGatewayCreateApiKey === true) {
            // cfn Nag doesn't like having a HTTP Method with Authorization Set to None, supress the warning
            defaults.addCfnSuppressRules(apiMethod, [
                {
                    id: "W59",
                    reason: "When ApiKey is being created, we also set apikeyRequired to true, so techincally apiGateway still looks for apiKey even though user specified AuthorizationType to NONE",
                },
            ]);
        }
    }
}
exports.ApiGatewayToIot = ApiGatewayToIot;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToIot[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-iot.ApiGatewayToIot", version: "2.35.0" };
//# sourceMappingURL=data:application/json;base64,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