"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StringStep = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const choice_1 = require("../../domain/choice");
const data_type_1 = require("../../domain/data-type");
const operation_1 = require("../../domain/operation");
const string_variable_1 = require("../../interface/variables/string-variable");
const assert_aws_resource_step_1 = require("../../parent-steps/automation/assert-aws-resource-step");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const branch_step_1 = require("../../parent-steps/automation/branch-step");
const execute_script_step_1 = require("../../parent-steps/automation/execute-script-step");
const pause_step_1 = require("../../parent-steps/automation/pause-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const wait_for_resource_step_1 = require("../../parent-steps/automation/wait-for-resource-step");
const composite_step_1 = require("./composite-step");
// eslint-disable-next-line
const yaml = require('js-yaml');
/**
 * StringStep allows for including a step from an existing automation document verbatim.
 * This is useful in that it allows developers to integrate with existing document steps.
 * This step can be used just as you would use any other Step including simulation and deployment.
 */
class StringStep extends composite_step_1.CompositeAutomationStep {
    constructor(scope, id, props) {
        super(scope, id);
        const sharedProps = {};
        sharedProps.name = props.name;
        sharedProps.action = props.action;
        if (props.maxAttempts) {
            sharedProps.maxAttempts = props.maxAttempts;
        }
        if (props.timeoutSeconds) {
            sharedProps.timeoutSeconds = props.timeoutSeconds;
        }
        if (props.isEnd != undefined) {
            sharedProps.isEnd = props.isEnd;
        }
        if (props.isCritical != undefined) {
            sharedProps.isCritical = props.isCritical;
        }
        if (props.outputs != undefined) {
            sharedProps.outputs = this.toOutputs(props.outputs);
        }
        const { Service, Api, PropertySelector, DesiredValues, ...restParams } = props.inputs;
        switch (props.action) {
            case 'aws:executeAwsApi':
                this.automationStep = new aws_api_step_1.AwsApiStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    outputs: sharedProps.outputs,
                    ...sharedProps,
                });
                break;
            case 'aws:waitForAwsResourceProperty':
                this.automationStep = new wait_for_resource_step_1.WaitForResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:assertAwsResourceProperty':
                this.automationStep = new assert_aws_resource_step_1.AssertAwsResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:pause':
                this.automationStep = new pause_step_1.PauseStep(this, props.name, {
                    ...sharedProps,
                });
                break;
            case 'aws:sleep':
                this.automationStep = new sleep_step_1.SleepStep(this, props.name, {
                    sleepSeconds: this.parseDurationToSeconds(restParams.Duration),
                    ...sharedProps,
                });
                break;
            case 'aws:executeScript':
                this.automationStep = new execute_script_step_1.ExecuteScriptStep(this, props.name, {
                    language: execute_script_step_1.ExecuteScriptStep.getLanguage(restParams.Runtime),
                    inputs: Object.keys(restParams.InputPayload),
                    inlineCode: restParams.Script,
                    handlerName: restParams.Handler,
                    ...sharedProps,
                });
                break;
            case 'aws:branch':
                this.automationStep = new branch_step_1.BranchStep(this, props.name, {
                    choices: this.toChoices(restParams.Choices),
                    defaultStepName: restParams.Default,
                    ...sharedProps,
                });
                break;
            default:
                throw new Error('Action not implemented as StringStep ' + props.action);
        }
    }
    /**
       * Builds a step object from a yaml declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromYaml(scope, inputYaml) {
        return StringStep.fromObject(scope, yaml.load(inputYaml));
    }
    /**
       * Builds a step object from a json declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromJson(scope, json) {
        return StringStep.fromObject(scope, JSON.parse(json));
    }
    /**
       * Builds a step object from an object.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromObject(scope, props) {
        return new StringStep(scope, props.name, props);
    }
    addToDocument(doc) {
        doc.addStep(this.automationStep);
    }
    toOutputs(declaredOutputs) {
        return declaredOutputs.map(declaredOutput => this.toOutput(declaredOutput));
    }
    toOutput(declaredOutput) {
        return {
            name: declaredOutput.Name,
            selector: declaredOutput.Selector,
            outputType: data_type_1.DataType.fromDataType(declaredOutput.Type).dataTypeEnum,
        };
    }
    toChoices(declaredChoices) {
        return declaredChoices.map(declaredChoice => this.toChoice(declaredChoice));
    }
    toChoice(declaredChoice) {
        const operationEntry = Object.entries(declaredChoice)
            .filter(entry => operation_1.OperationEvaluator.STRING_TO_OPERATION[entry[0]] != undefined)[0];
        return new choice_1.Choice({
            jumpToStepName: declaredChoice.NextStep,
            variable: this.toVariable(declaredChoice.Variable),
            operation: operation_1.OperationEvaluator.fromOperationName(operationEntry[0]),
            constant: operationEntry[1],
        });
    }
    toVariable(variable) {
        if (variable.includes('{{')) {
            return new string_variable_1.StringVariable(variable.replace('{{', '').replace('}}', '').trim());
        }
        return new string_variable_1.HardCodedString(variable.trim());
    }
    parseDurationToSeconds(iso8601Duration) {
        // https://stackoverflow.com/a/29153059
        const iso8601DurationRegex = /(-)?P(?:([.,\d]+)Y)?(?:([.,\d]+)M)?(?:([.,\d]+)W)?(?:([.,\d]+)D)?T(?:([.,\d]+)H)?(?:([.,\d]+)M)?(?:([.,\d]+)S)?/;
        const matches = iso8601Duration.match(iso8601DurationRegex);
        if (matches == undefined) {
            throw new Error('Could not parse Duration' + iso8601Duration);
        }
        const periods = {
            years: matches[2] === undefined ? 0 : Number.parseInt(matches[2]),
            months: matches[3] === undefined ? 0 : Number.parseInt(matches[3]),
            weeks: matches[4] === undefined ? 0 : Number.parseInt(matches[4]),
            days: matches[5] === undefined ? 0 : Number.parseInt(matches[5]),
            hours: matches[6] === undefined ? 0 : Number.parseInt(matches[6]),
            minutes: matches[7] === undefined ? 0 : Number.parseInt(matches[7]),
            seconds: matches[8] === undefined ? 0 : Number.parseInt(matches[8]),
        };
        return periods.seconds +
            periods.minutes * 60 +
            periods.hours * 60 * 60 +
            periods.days * 60 * 60 * 24 +
            periods.weeks * 60 * 60 * 24 * 7 +
            periods.months * 60 * 60 * 24 * 30.5 +
            periods.years * 60 * 60 * 24 * 365;
    }
    ;
}
exports.StringStep = StringStep;
_a = JSII_RTTI_SYMBOL_1;
StringStep[_a] = { fqn: "@cdklabs/cdk-ssm-documents.StringStep", version: "0.0.13" };
//# sourceMappingURL=data:application/json;base64,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