"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.waitForAndAssertStackStatus = exports.waitForAndAssertImageAvailable = exports.waitForAndAssertInstanceStatus = exports.waitForAndAssertInstanceState = exports.waitForAndAssertResource = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const response_code_1 = require("../../../domain/response-code");
const stack_status_1 = require("../../../domain/stack-status");
const wait_for_and_assert_resource_1 = require("../../../patterns/automation/wait-for-and-assert-resource");
const simulate_pattern_1 = require("../../../utils/simulate-pattern");
/**
 * Wait for and assert on an EC2 resource value
 */
function waitForAndAssertResource(props, simulationProps) {
    const component = new wait_for_and_assert_resource_1.WaitForAndAssertResource(new aws_cdk_lib_1.Stack(), 'waitForAndAssertResource', props);
    const result = simulate_pattern_1.simulatePattern(component, simulationProps, {});
    if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
        throw new Error(`Wait and assert for resource failes for ${JSON.stringify(props)}: ${result.stackTrace}`);
    }
}
exports.waitForAndAssertResource = waitForAndAssertResource;
/**
 * Wait for and assert on an EC2 instance state
 */
function waitForAndAssertInstanceState(props, simulationProps) {
    const { waitForState, assertState, instanceIds } = props;
    waitForAndAssertResource({
        service: 'EC2',
        pascalCaseApi: 'DescribeInstanceStatus',
        apiParams: {
            InstanceIds: instanceIds,
            IncludeAllInstances: true,
        },
        selector: '$.InstanceStatuses..InstanceState.Name',
        waitForValues: waitForState,
        desiredValues: [assertState],
    }, simulationProps);
}
exports.waitForAndAssertInstanceState = waitForAndAssertInstanceState;
/**
 * Wait for and assert on an EC2 instance status
 */
function waitForAndAssertInstanceStatus(props, simulationProps) {
    const { waitForStatus, assertStatus, instanceIds } = props;
    waitForAndAssertResource({
        service: 'EC2',
        pascalCaseApi: 'DescribeInstanceStatus',
        apiParams: {
            InstanceIds: instanceIds,
            IncludeAllInstances: true,
        },
        selector: '$.InstanceStatuses..InstanceStatus.Status',
        waitForValues: waitForStatus,
        desiredValues: [assertStatus],
    }, simulationProps);
}
exports.waitForAndAssertInstanceStatus = waitForAndAssertInstanceStatus;
/**
 * Wait for and assert on an EC2 image for available state
 */
function waitForAndAssertImageAvailable(props, simulationProps) {
    const { imageId } = props;
    waitForAndAssertResource({
        service: 'EC2',
        pascalCaseApi: 'DescribeImages',
        selector: '$.Images..State',
        apiParams: {
            ImageIds: [imageId],
        },
        waitForValues: ['available', 'invalid', 'deregistered', 'transient', 'failed', 'error'],
        desiredValues: ['available'],
    }, simulationProps);
}
exports.waitForAndAssertImageAvailable = waitForAndAssertImageAvailable;
/**
 * Wait for and assert on a stack's status
 */
function waitForAndAssertStackStatus(props, simulationProps) {
    const { waitForStatus, assertStatus, stackName } = props;
    waitForAndAssertResource({
        service: 'CloudFormation',
        pascalCaseApi: 'DescribeStacks',
        apiParams: {
            StackName: stackName,
        },
        selector: '$.Stacks[0].StackStatus',
        waitForValues: waitForStatus.map(x => stack_status_1.StackStatus[x]),
        desiredValues: [stack_status_1.StackStatus[assertStatus]],
    }, simulationProps);
}
exports.waitForAndAssertStackStatus = waitForAndAssertStackStatus;
//# sourceMappingURL=data:application/json;base64,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