import { GenericVariable, HardCodedValueBase, IGenericVariable } from './variable';
/**
 * String type that can be null
 */
export declare type NullableString = string;
/**
 * A string variable
 */
export interface IStringVariable extends IGenericVariable {
    /**
       * Given the execution inputs, return the resolved value of this variable.
       * @param inputs are the execution inputs.
       */
    resolveToString(inputs: Record<string, any>): NullableString;
}
/**
 * A hard-coded string variable.
 * Used when not dependent on step inputs.
 */
export declare class HardCodedString extends HardCodedValueBase<string> implements IStringVariable {
    static of(val: string): HardCodedString;
    resolveToString(inputs: Record<string, any>): NullableString;
    protected assertType(value: any): void;
}
/**
 * A string variable reference.
 * Used to resolve the value from step inputs.
 */
export declare class StringVariable extends GenericVariable implements IStringVariable {
    static of(reference: string): StringVariable;
    resolveToString(inputs: Record<string, any>): NullableString;
    protected assertType(value: any): void;
}
/**
 * Replacement for strings using Java String format style "%s" replacements.
 * Example: new StringFormat("This %s a replacement: %s", [new HardCodedValue("is"), new ExportedVariable("myInput")]);
 * The supported variable strings are either:
 * 1. Implicit indices: "%s" where the first occurrence will match the first variable, the next will match the second...
 * 2. Explicit indices: Example: "%1$s"; where "%1$s" matches the first variable and "%1$s" matches the second.
 * Do not combine usage of implicit and explicit indices. Choose one per StringFormat instance.
 */
export declare class StringFormat implements IStringVariable {
    readonly format: string;
    readonly variables: IGenericVariable[];
    constructor(format: string, variables?: IGenericVariable[]);
    resolve(inputs: {
        [name: string]: any;
    }): any;
    print(): string | any;
    requiredInputs(): string[];
    toJSON(): any;
    resolveToString(inputs: Record<string, any>): NullableString;
}
export declare class DictFormat implements IStringVariable {
    readonly format: {
        [name: string]: any;
    };
    constructor(format: {
        [name: string]: any;
    });
    resolveToDict(inputs: {
        [name: string]: any;
    }): {
        [name: string]: any;
    };
    toJSON(): any;
    print(): string | any;
    requiredInputs(): string[];
    resolve(inputs: {
        [p: string]: any;
    }): any;
    resolveToString(inputs: Record<string, any>): NullableString;
}
/**
 * Assert that the given value is a NullableString.
 * @param value value to assert
 */
export declare function assertString(value: any): asserts value is NullableString;
/**
 * Whether the given value is a NullableString.
 * @param value value to check
 * @returns true if the value is a NullableString, otherwise false
 */
export declare function isString(value: any): value is NullableString;
