"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OperationEvaluator = exports.Operation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Operation to use for comparing a Choice's or Preconditions with provided value.
 * See https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-branch.html
 * or https://docs.aws.amazon.com/systems-manager/latest/userguide/document-schemas-features.html
 */
var Operation;
(function (Operation) {
    Operation[Operation["BOOLEAN_EQUALS"] = 0] = "BOOLEAN_EQUALS";
    Operation[Operation["CONTAINS"] = 1] = "CONTAINS";
    Operation[Operation["ENDS_WITH"] = 2] = "ENDS_WITH";
    Operation[Operation["STARTS_WITH"] = 3] = "STARTS_WITH";
    Operation[Operation["STRING_EQUALS"] = 4] = "STRING_EQUALS";
    Operation[Operation["EQUALS_IGNORE_CASE"] = 5] = "EQUALS_IGNORE_CASE";
    Operation[Operation["NUMERIC_EQUALS"] = 6] = "NUMERIC_EQUALS";
    Operation[Operation["NUMERIC_GREATER"] = 7] = "NUMERIC_GREATER";
    Operation[Operation["NUMERIC_GREATER_OR_EQUALS"] = 8] = "NUMERIC_GREATER_OR_EQUALS";
    Operation[Operation["NUMERIC_LESSER"] = 9] = "NUMERIC_LESSER";
    Operation[Operation["NUMERIC_LESSER_OR_EQUALS"] = 10] = "NUMERIC_LESSER_OR_EQUALS";
})(Operation = exports.Operation || (exports.Operation = {}));
class OperationEvaluator {
    constructor(operation) {
        this.operation = operation;
    }
    /**
       * Converts a string to OperationType
       * @param operationName an operation name to return its OperationType
       * @returns an OperationType
       * @throws if operationName is not supported
       */
    static fromOperationName(operationName) {
        const operation = OperationEvaluator.STRING_TO_OPERATION[operationName];
        if (operation != undefined) {
            return operation;
        }
        else {
            throw new Error(`Runtime ${operationName} not supported.`);
        }
    }
    /**
       * Evaluates this operation against the provided inputs.
       * @returns true if the evaluation is true. False otherwise.
       */
    evaluate(value1, value2) {
        switch (this.operation) {
            case Operation.BOOLEAN_EQUALS:
                return value1 == value2;
            case Operation.STRING_EQUALS:
                return value1 === value2;
            case Operation.EQUALS_IGNORE_CASE:
                return value1.toLowerCase() === value2.toLowerCase();
            case Operation.STARTS_WITH:
                return value1.startsWith(value2);
            case Operation.ENDS_WITH:
                return value1.endsWith(value2);
            case Operation.CONTAINS:
                return value1.includes(value2);
            case Operation.NUMERIC_EQUALS:
                return value1 == value2;
            case Operation.NUMERIC_GREATER:
                return value1 > value2;
            case Operation.NUMERIC_LESSER:
                return value1 < value2;
            case Operation.NUMERIC_GREATER_OR_EQUALS:
                return value1 >= value2;
            case Operation.NUMERIC_LESSER_OR_EQUALS:
                return value1 <= value2;
            default:
                throw new Error('No implementation for evaluate found for operation ' + this.operation);
        }
    }
    /**
       * @returns the string representation used by SSM for the operation.
       */
    toOperationName() {
        const operationEntry = Object.entries(OperationEvaluator.STRING_TO_OPERATION)
            .filter(entry => entry[1] == this.operation)[0];
        if (operationEntry) {
            return operationEntry[0];
        }
        else {
            throw new Error(`Operation ${this.operation} not supported.`);
        }
    }
}
exports.OperationEvaluator = OperationEvaluator;
_a = JSII_RTTI_SYMBOL_1;
OperationEvaluator[_a] = { fqn: "@cdklabs/cdk-ssm-documents.OperationEvaluator", version: "0.0.13" };
OperationEvaluator.STRING_TO_OPERATION = {
    BooleanEquals: Operation.BOOLEAN_EQUALS,
    StringEquals: Operation.STRING_EQUALS,
    EqualsIgnoreCase: Operation.EQUALS_IGNORE_CASE,
    StartsWith: Operation.STARTS_WITH,
    EndsWith: Operation.ENDS_WITH,
    Contains: Operation.CONTAINS,
    NumericEquals: Operation.NUMERIC_EQUALS,
    NumericGreater: Operation.NUMERIC_GREATER,
    NumericLesser: Operation.NUMERIC_LESSER,
    NumericGreaterOrEquals: Operation.NUMERIC_GREATER_OR_EQUALS,
    NumericLesserOrEquals: Operation.NUMERIC_LESSER_OR_EQUALS,
};
//# sourceMappingURL=data:application/json;base64,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