"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsmDocument = exports.DocumentFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const constructs_1 = require("constructs");
const data_type_1 = require("../domain/data-type");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const yaml = require('js-yaml');
var DocumentFormat;
(function (DocumentFormat) {
    DocumentFormat[DocumentFormat["JSON"] = 0] = "JSON";
    DocumentFormat[DocumentFormat["YAML"] = 1] = "YAML";
})(DocumentFormat = exports.DocumentFormat || (exports.DocumentFormat = {}));
class SsmDocument extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.documentName = props.documentName ?? id;
        this.description = props.description ?? this.documentName;
        this.header = props.header;
        this.assumeRole = props.assumeRole;
        this.docOutputs = props.docOutputs ?? [];
        this.docInputs = props.docInputs ?? [];
        if (this.assumeRole && !this.docInputs.map(i => i.name).includes(this.assumeRole)) {
            throw new Error('Assume rold specified but not provided in inputs: ' + this.assumeRole);
        }
        this.props = props;
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        new aws_ssm_1.CfnDocument(this, 'Resource', {
            ...this.props,
            ...{
                content: aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        const doc = this.buildSsmDocument();
                        return JSON.parse(JSON.stringify(doc));
                    },
                }),
                documentFormat: isYaml ? 'YAML' : 'JSON',
                documentType: this.documentType(),
                tags: [{ key: 'CdkGenerated', value: 'true' }, ...(this.props.tags || [])],
            },
        });
    }
    /**
     * The SSM Document inputs optionally specify a number of parameters including allowedValues, minItems etc.
     * This function builds an object containing the relevant (declared) input parameters.
     * The return object will be used to build the yaml/json representation of the document.
     */
    formatInputs() {
        const ssmInputs = {};
        this.docInputs.forEach(inp => {
            const nested = {
                type: new data_type_1.DataType(inp.inputType).toSsmString(),
            };
            if (inp.description != undefined) {
                nested.description = inp.description;
            }
            if (inp.defaultValue != undefined) {
                nested.default = inp.defaultValue;
            }
            if (inp.allowedValues != undefined) {
                nested.allowedValues = inp.allowedValues;
            }
            if (inp.allowedPattern != undefined) {
                nested.allowedPattern = inp.allowedPattern;
            }
            if (inp.minItems != undefined) {
                nested.minItems = inp.minItems;
            }
            if (inp.maxItems != undefined) {
                nested.maxItems = inp.maxItems;
            }
            ssmInputs[inp.name] = nested;
        });
        return ssmInputs;
    }
    /**
       * Synthesize before calling this function!
       * You can use this to Synthesize: cdk.SynthUtils.synthesize(stack);
       *
       * Converts the objects define in the SSM Document (including all of the steps) to an SSM document string.
       * The format is dependency on the documentFormat property provided to the class.
       * The yaml can be used as is and will behave (or at least should behave) as was simulated in the runSimulation().
       * @returns a string representation of this document as an SSM formatted yaml/json.
       */
    print() {
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        if (isYaml) {
            // Prepend the document with the header if defined
            return this.headerWithComments() + yaml.dump(this.buildSsmDocument());
        }
        else {
            // There is no way to provide the header comment in a json document
            return JSON.stringify(this.buildSsmDocument());
        }
    }
    /**
       * Prepend header line with # if it does not start with #.
       * If no header is provided, this method will return an emptry string.
       * Headers are only available in yaml format.
       */
    headerWithComments() {
        if (this.header == undefined) {
            return '';
        }
        return this.header.split(os_1.EOL).map(line => line.trim().startsWith('#') ? line : '# ' + line).join(os_1.EOL) +
            // "---" demarks the end of the header and the start of the document.
            '\n---\n';
    }
}
exports.SsmDocument = SsmDocument;
_a = JSII_RTTI_SYMBOL_1;
SsmDocument[_a] = { fqn: "@cdklabs/cdk-ssm-documents.SsmDocument", version: "0.0.13" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3NtLWRvY3VtZW50LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2RvY3VtZW50L3NzbS1kb2N1bWVudC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDJCQUF5QjtBQUN6Qiw2Q0FBd0Q7QUFDeEQsaURBQWtEO0FBQ2xELDJDQUF1QztBQUN2QyxtREFBK0M7QUFHL0MsaUVBQWlFO0FBQ2pFLE1BQU0sSUFBSSxHQUFHLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztBQUdoQyxJQUFZLGNBR1g7QUFIRCxXQUFZLGNBQWM7SUFDeEIsbURBQUksQ0FBQTtJQUNKLG1EQUFJLENBQUE7QUFDTixDQUFDLEVBSFcsY0FBYyxHQUFkLHNCQUFjLEtBQWQsc0JBQWMsUUFHekI7QUFrRUQsTUFBc0IsV0FBWSxTQUFRLHNCQUFTO0lBVWpELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7UUFDL0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDO1FBQzdDLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDO1FBQzFELElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQztRQUMzQixJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7UUFDbkMsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEVBQUUsQ0FBQztRQUN6QyxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUksRUFBRSxDQUFDO1FBQ3ZDLElBQUksSUFBSSxDQUFDLFVBQVUsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDakYsTUFBTSxJQUFJLEtBQUssQ0FBQyxvREFBb0QsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDekY7UUFDRCxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQztRQUNuQixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsSUFBSSxjQUFjLENBQUMsSUFBSSxDQUFDO1FBQ2hFLElBQUkscUJBQVcsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ2hDLEdBQUcsSUFBSSxDQUFDLEtBQUs7WUFDYixHQUFHO2dCQUNELE9BQU8sRUFBRSxrQkFBSSxDQUFDLEdBQUcsQ0FBQztvQkFDaEIsT0FBTyxFQUFFLEdBQUcsRUFBRTt3QkFDWixNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQzt3QkFDcEMsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztvQkFDekMsQ0FBQztpQkFDRixDQUFDO2dCQUNGLGNBQWMsRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsTUFBTTtnQkFDeEMsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZLEVBQUU7Z0JBQ2pDLElBQUksRUFBRSxDQUFDLEVBQUUsR0FBRyxFQUFFLGNBQWMsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxJQUFJLEVBQUUsQ0FBQyxDQUFDO2FBQzNFO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDTyxZQUFZO1FBQ3BCLE1BQU0sU0FBUyxHQUEwQixFQUFFLENBQUM7UUFDNUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDM0IsTUFBTSxNQUFNLEdBQTBCO2dCQUNwQyxJQUFJLEVBQUUsSUFBSSxvQkFBUSxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQyxXQUFXLEVBQUU7YUFDaEQsQ0FBQztZQUNGLElBQUksR0FBRyxDQUFDLFdBQVcsSUFBSSxTQUFTLEVBQUU7Z0JBQ2hDLE1BQU0sQ0FBQyxXQUFXLEdBQUcsR0FBRyxDQUFDLFdBQVcsQ0FBQzthQUN0QztZQUNELElBQUksR0FBRyxDQUFDLFlBQVksSUFBSSxTQUFTLEVBQUU7Z0JBQ2pDLE1BQU0sQ0FBQyxPQUFPLEdBQUcsR0FBRyxDQUFDLFlBQVksQ0FBQzthQUNuQztZQUNELElBQUksR0FBRyxDQUFDLGFBQWEsSUFBSSxTQUFTLEVBQUU7Z0JBQ2xDLE1BQU0sQ0FBQyxhQUFhLEdBQUcsR0FBRyxDQUFDLGFBQWEsQ0FBQzthQUMxQztZQUNELElBQUksR0FBRyxDQUFDLGNBQWMsSUFBSSxTQUFTLEVBQUU7Z0JBQ25DLE1BQU0sQ0FBQyxjQUFjLEdBQUcsR0FBRyxDQUFDLGNBQWMsQ0FBQzthQUM1QztZQUNELElBQUksR0FBRyxDQUFDLFFBQVEsSUFBSSxTQUFTLEVBQUU7Z0JBQzdCLE1BQU0sQ0FBQyxRQUFRLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQzthQUNoQztZQUNELElBQUksR0FBRyxDQUFDLFFBQVEsSUFBSSxTQUFTLEVBQUU7Z0JBQzdCLE1BQU0sQ0FBQyxRQUFRLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQzthQUNoQztZQUNELFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsTUFBTSxDQUFDO1FBQy9CLENBQUMsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQVNEOzs7Ozs7OztTQVFLO0lBQ0UsS0FBSztRQUNWLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxJQUFJLGNBQWMsQ0FBQyxJQUFJLENBQUM7UUFDaEUsSUFBSSxNQUFNLEVBQUU7WUFDVixrREFBa0Q7WUFDbEQsT0FBTyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLENBQUM7U0FDdkU7YUFBTTtZQUNMLG1FQUFtRTtZQUNuRSxPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFLENBQUMsQ0FBQztTQUNoRDtJQUNILENBQUM7SUFFRDs7OztTQUlLO0lBQ0csa0JBQWtCO1FBQ3hCLElBQUksSUFBSSxDQUFDLE1BQU0sSUFBSSxTQUFTLEVBQUU7WUFDNUIsT0FBTyxFQUFFLENBQUM7U0FDWDtRQUNELE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsUUFBRyxDQUFDLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLFFBQUcsQ0FBQztZQUM3RixxRUFBcUU7WUFDckUsU0FBUyxDQUFDO0lBQ3BCLENBQUM7O0FBaEhILGtDQWtIQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEVPTCB9IGZyb20gJ29zJztcbmltcG9ydCB7IENmblRhZywgSVJlc29sdmFibGUsIExhenkgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDZm5Eb2N1bWVudCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zc20nO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBEYXRhVHlwZSB9IGZyb20gJy4uL2RvbWFpbi9kYXRhLXR5cGUnO1xuaW1wb3J0IHsgRG9jdW1lbnRPdXRwdXQgfSBmcm9tICcuLi9kb21haW4vZG9jdW1lbnQtb3V0cHV0JztcbmltcG9ydCB7IElucHV0IH0gZnJvbSAnLi4vZG9tYWluL2lucHV0Jztcbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvbm8tcmVxdWlyZS1pbXBvcnRzXG5jb25zdCB5YW1sID0gcmVxdWlyZSgnanMteWFtbCcpO1xuXG5cbmV4cG9ydCBlbnVtIERvY3VtZW50Rm9ybWF0IHtcbiAgSlNPTixcbiAgWUFNTFxufVxuXG5leHBvcnQgaW50ZXJmYWNlIFNzbURvY3VtZW50UHJvcHMge1xuICAvKipcbiAgICAgKiAoT3B0aW9uYWwpIE5hbWUgb2YgdGhlIGRvY3VtZW50LiBXaWxsIGRlZmF1bHQgdG8gdGhlIGlkIHByb3ZpZGVkIGZvciB0aGUgQ0RLIG5vZGUuXG4gICAgICovXG4gIHJlYWRvbmx5IGRvY3VtZW50TmFtZT86IHN0cmluZztcbiAgLyoqXG4gICAgICogKE9wdGlvbmFsKSBEZXNjcmlwdGlvbiBvZiB0aGUgZG9jdW1lbnQuIERlZmF1bHRzIHRvIHRoZSBkb2N1bWVudCBuYW1lLlxuICAgICAqL1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcbiAgLyoqXG4gICAgICogKE9wdGlvbmFsKSBBIEhlYWRlci9jb21tZW50IHRvIGluY2x1ZGUgYXQgdGhlIHN0YXJ0IG9mIGEgWUFNTCBkb2N1bWVudC4gSlNPTiBkb2N1bWVudHMgZG8gbm90IHN1cHBvcnQgaGVhZGVycy5cbiAgICAgKi9cbiAgcmVhZG9ubHkgaGVhZGVyPzogc3RyaW5nO1xuICAvKipcbiAgICAgKiAoT3B0aW9uYWwpIEFzc3VtZSByb2xlIHRvIHVzZSBmb3IgdGhpcyBkb2N1bWVudC5cbiAgICAgKiBJZiBwcm92aWRlZCwgdGhpcyB2YWx1ZSBNVVNUIGJlIGluY2x1ZGVkIGFzIG9uZSBvZiB0aGUgZG9jdW1lbnRJbnB1dCBuYW1lcy5cbiAgICAgKi9cbiAgcmVhZG9ubHkgYXNzdW1lUm9sZT86IHN0cmluZztcbiAgLyoqXG4gICAgICogKE9wdGlvbmFsKSBPdXRwdXRzIHRvIGJlIGVtaXR0ZWQgZnJvbSB0aGUgZG9jdW1lbnQuXG4gICAgICogVGhlIG91dHB1dHMgYXJlIHBsYWNlZCBpbiBhIFN0cmluZ1NldCBjYWxsZWQgb3V0cHV0cyAoYXMgaXMgZG9uZSBpbiBTU00pLlxuICAgICAqIEBkZWZhdWx0IFtdXG4gICAgICovXG4gIHJlYWRvbmx5IGRvY091dHB1dHM/OiBEb2N1bWVudE91dHB1dFtdO1xuICAvKipcbiAgICAgKiAoT3B0aW9uYWwpIElucHV0cyByZXF1aXJlZCBieSB0aGUgZG9jdW1lbnQuXG4gICAgICovXG4gIHJlYWRvbmx5IGRvY0lucHV0cz86IElucHV0W107XG4gIC8qKlxuICAgICAqIChPcHRpb25hbCkgU3BlY2lmaWVzIHdoZXRoZXIgdGhpcyBkb2N1bWVudCBzaG91bGQgYmUgd3JpdHRlbiBhcyBZQU1MIG9yIEpTT04uXG4gICAgICogQGRlZmF1bHQgSlNPTlxuICAgICAqL1xuICByZWFkb25seSBkb2N1bWVudEZvcm1hdD86IERvY3VtZW50Rm9ybWF0O1xuXG4gIC8qKlxuICAgICAqIGBBV1M6OlNTTTo6RG9jdW1lbnQuUmVxdWlyZXNgLlxuICAgICAqXG4gICAgICogQGV4dGVybmFsXG4gICAgICogQGxpbmsgaHR0cDovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcmVzb3VyY2Utc3NtLWRvY3VtZW50Lmh0bWwjY2ZuLXNzbS1kb2N1bWVudC1yZXF1aXJlc1xuICAgICAqL1xuICByZWFkb25seSByZXF1aXJlcz86IEFycmF5PENmbkRvY3VtZW50LkRvY3VtZW50UmVxdWlyZXNQcm9wZXJ0eSB8IElSZXNvbHZhYmxlPiB8IElSZXNvbHZhYmxlO1xuICAvKipcbiAgICAgKiBgQVdTOjpTU006OkRvY3VtZW50LlRhZ3NgLlxuICAgICAqXG4gICAgICogQGV4dGVybmFsXG4gICAgICogQGxpbmsgaHR0cDovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcmVzb3VyY2Utc3NtLWRvY3VtZW50Lmh0bWwjY2ZuLXNzbS1kb2N1bWVudC10YWdzXG4gICAgICovXG4gIHJlYWRvbmx5IHRhZ3M/OiBDZm5UYWdbXTtcbiAgLyoqXG4gICAgICogYEFXUzo6U1NNOjpEb2N1bWVudC5UYXJnZXRUeXBlYC5cbiAgICAgKlxuICAgICAqIEBleHRlcm5hbFxuICAgICAqIEBsaW5rIGh0dHA6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLXNzbS1kb2N1bWVudC5odG1sI2Nmbi1zc20tZG9jdW1lbnQtdGFyZ2V0dHlwZVxuICAgICAqL1xuICByZWFkb25seSB0YXJnZXRUeXBlPzogc3RyaW5nO1xuICAvKipcbiAgICAgKiBgQVdTOjpTU006OkRvY3VtZW50LlZlcnNpb25OYW1lYC5cbiAgICAgKlxuICAgICAqIEBleHRlcm5hbFxuICAgICAqIEBsaW5rIGh0dHA6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLXNzbS1kb2N1bWVudC5odG1sI2Nmbi1zc20tZG9jdW1lbnQtdmVyc2lvbm5hbWVcbiAgICAgKi9cbiAgcmVhZG9ubHkgdmVyc2lvbk5hbWU/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBTc21Eb2N1bWVudCBleHRlbmRzIENvbnN0cnVjdCB7XG5cbiAgcmVhZG9ubHkgZG9jdW1lbnROYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGhlYWRlcj86IHN0cmluZztcbiAgcmVhZG9ubHkgYXNzdW1lUm9sZT86IHN0cmluZztcbiAgcmVhZG9ubHkgZG9jT3V0cHV0czogRG9jdW1lbnRPdXRwdXRbXTtcbiAgcmVhZG9ubHkgZG9jSW5wdXRzOiBJbnB1dFtdO1xuICByZWFkb25seSBwcm9wczogU3NtRG9jdW1lbnRQcm9wcztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogU3NtRG9jdW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgdGhpcy5kb2N1bWVudE5hbWUgPSBwcm9wcy5kb2N1bWVudE5hbWUgPz8gaWQ7XG4gICAgdGhpcy5kZXNjcmlwdGlvbiA9IHByb3BzLmRlc2NyaXB0aW9uID8/IHRoaXMuZG9jdW1lbnROYW1lO1xuICAgIHRoaXMuaGVhZGVyID0gcHJvcHMuaGVhZGVyO1xuICAgIHRoaXMuYXNzdW1lUm9sZSA9IHByb3BzLmFzc3VtZVJvbGU7XG4gICAgdGhpcy5kb2NPdXRwdXRzID0gcHJvcHMuZG9jT3V0cHV0cyA/PyBbXTtcbiAgICB0aGlzLmRvY0lucHV0cyA9IHByb3BzLmRvY0lucHV0cyA/PyBbXTtcbiAgICBpZiAodGhpcy5hc3N1bWVSb2xlICYmICF0aGlzLmRvY0lucHV0cy5tYXAoaSA9PiBpLm5hbWUpLmluY2x1ZGVzKHRoaXMuYXNzdW1lUm9sZSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXNzdW1lIHJvbGQgc3BlY2lmaWVkIGJ1dCBub3QgcHJvdmlkZWQgaW4gaW5wdXRzOiAnICsgdGhpcy5hc3N1bWVSb2xlKTtcbiAgICB9XG4gICAgdGhpcy5wcm9wcyA9IHByb3BzO1xuICAgIGNvbnN0IGlzWWFtbCA9IHRoaXMucHJvcHMuZG9jdW1lbnRGb3JtYXQgPT0gRG9jdW1lbnRGb3JtYXQuWUFNTDtcbiAgICBuZXcgQ2ZuRG9jdW1lbnQodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgLi4udGhpcy5wcm9wcyxcbiAgICAgIC4uLntcbiAgICAgICAgY29udGVudDogTGF6eS5hbnkoe1xuICAgICAgICAgIHByb2R1Y2U6ICgpID0+IHtcbiAgICAgICAgICAgIGNvbnN0IGRvYyA9IHRoaXMuYnVpbGRTc21Eb2N1bWVudCgpO1xuICAgICAgICAgICAgcmV0dXJuIEpTT04ucGFyc2UoSlNPTi5zdHJpbmdpZnkoZG9jKSk7XG4gICAgICAgICAgfSxcbiAgICAgICAgfSksXG4gICAgICAgIGRvY3VtZW50Rm9ybWF0OiBpc1lhbWwgPyAnWUFNTCcgOiAnSlNPTicsXG4gICAgICAgIGRvY3VtZW50VHlwZTogdGhpcy5kb2N1bWVudFR5cGUoKSxcbiAgICAgICAgdGFnczogW3sga2V5OiAnQ2RrR2VuZXJhdGVkJywgdmFsdWU6ICd0cnVlJyB9LCAuLi4odGhpcy5wcm9wcy50YWdzIHx8IFtdKV0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBTU00gRG9jdW1lbnQgaW5wdXRzIG9wdGlvbmFsbHkgc3BlY2lmeSBhIG51bWJlciBvZiBwYXJhbWV0ZXJzIGluY2x1ZGluZyBhbGxvd2VkVmFsdWVzLCBtaW5JdGVtcyBldGMuXG4gICAqIFRoaXMgZnVuY3Rpb24gYnVpbGRzIGFuIG9iamVjdCBjb250YWluaW5nIHRoZSByZWxldmFudCAoZGVjbGFyZWQpIGlucHV0IHBhcmFtZXRlcnMuXG4gICAqIFRoZSByZXR1cm4gb2JqZWN0IHdpbGwgYmUgdXNlZCB0byBidWlsZCB0aGUgeWFtbC9qc29uIHJlcHJlc2VudGF0aW9uIG9mIHRoZSBkb2N1bWVudC5cbiAgICovXG4gIHByb3RlY3RlZCBmb3JtYXRJbnB1dHMoKTogeyBbbmFtZTogc3RyaW5nXTogYW55IH0ge1xuICAgIGNvbnN0IHNzbUlucHV0czoge1tuYW1lOiBzdHJpbmddOiBhbnl9ID0ge307XG4gICAgdGhpcy5kb2NJbnB1dHMuZm9yRWFjaChpbnAgPT4ge1xuICAgICAgY29uc3QgbmVzdGVkOiB7W25hbWU6IHN0cmluZ106IGFueX0gPSB7XG4gICAgICAgIHR5cGU6IG5ldyBEYXRhVHlwZShpbnAuaW5wdXRUeXBlKS50b1NzbVN0cmluZygpLFxuICAgICAgfTtcbiAgICAgIGlmIChpbnAuZGVzY3JpcHRpb24gIT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIG5lc3RlZC5kZXNjcmlwdGlvbiA9IGlucC5kZXNjcmlwdGlvbjtcbiAgICAgIH1cbiAgICAgIGlmIChpbnAuZGVmYXVsdFZhbHVlICE9IHVuZGVmaW5lZCkge1xuICAgICAgICBuZXN0ZWQuZGVmYXVsdCA9IGlucC5kZWZhdWx0VmFsdWU7XG4gICAgICB9XG4gICAgICBpZiAoaW5wLmFsbG93ZWRWYWx1ZXMgIT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIG5lc3RlZC5hbGxvd2VkVmFsdWVzID0gaW5wLmFsbG93ZWRWYWx1ZXM7XG4gICAgICB9XG4gICAgICBpZiAoaW5wLmFsbG93ZWRQYXR0ZXJuICE9IHVuZGVmaW5lZCkge1xuICAgICAgICBuZXN0ZWQuYWxsb3dlZFBhdHRlcm4gPSBpbnAuYWxsb3dlZFBhdHRlcm47XG4gICAgICB9XG4gICAgICBpZiAoaW5wLm1pbkl0ZW1zICE9IHVuZGVmaW5lZCkge1xuICAgICAgICBuZXN0ZWQubWluSXRlbXMgPSBpbnAubWluSXRlbXM7XG4gICAgICB9XG4gICAgICBpZiAoaW5wLm1heEl0ZW1zICE9IHVuZGVmaW5lZCkge1xuICAgICAgICBuZXN0ZWQubWF4SXRlbXMgPSBpbnAubWF4SXRlbXM7XG4gICAgICB9XG4gICAgICBzc21JbnB1dHNbaW5wLm5hbWVdID0gbmVzdGVkO1xuICAgIH0pO1xuICAgIHJldHVybiBzc21JbnB1dHM7XG4gIH1cblxuICBwdWJsaWMgYWJzdHJhY3QgZG9jdW1lbnRUeXBlKCk6IHN0cmluZztcblxuICAvKipcbiAgICAgKiBEZWxlZ2F0ZXMgYnVpbGRpbmcgdGhlIFNTTSBEb2N1bWVudCB0byBiZSBjb252ZXJ0ZWQgdG8gYSB5YW1sL2pzb24gdG8gdGhlIHN1YmNsYXNzIChBdXRvbWF0aW9uIGV0YykuXG4gICAgICovXG4gIHByb3RlY3RlZCBhYnN0cmFjdCBidWlsZFNzbURvY3VtZW50KCk6IHsgW25hbWU6IHN0cmluZ106IGFueSB9O1xuXG4gIC8qKlxuICAgICAqIFN5bnRoZXNpemUgYmVmb3JlIGNhbGxpbmcgdGhpcyBmdW5jdGlvbiFcbiAgICAgKiBZb3UgY2FuIHVzZSB0aGlzIHRvIFN5bnRoZXNpemU6IGNkay5TeW50aFV0aWxzLnN5bnRoZXNpemUoc3RhY2spO1xuICAgICAqXG4gICAgICogQ29udmVydHMgdGhlIG9iamVjdHMgZGVmaW5lIGluIHRoZSBTU00gRG9jdW1lbnQgKGluY2x1ZGluZyBhbGwgb2YgdGhlIHN0ZXBzKSB0byBhbiBTU00gZG9jdW1lbnQgc3RyaW5nLlxuICAgICAqIFRoZSBmb3JtYXQgaXMgZGVwZW5kZW5jeSBvbiB0aGUgZG9jdW1lbnRGb3JtYXQgcHJvcGVydHkgcHJvdmlkZWQgdG8gdGhlIGNsYXNzLlxuICAgICAqIFRoZSB5YW1sIGNhbiBiZSB1c2VkIGFzIGlzIGFuZCB3aWxsIGJlaGF2ZSAob3IgYXQgbGVhc3Qgc2hvdWxkIGJlaGF2ZSkgYXMgd2FzIHNpbXVsYXRlZCBpbiB0aGUgcnVuU2ltdWxhdGlvbigpLlxuICAgICAqIEByZXR1cm5zIGEgc3RyaW5nIHJlcHJlc2VudGF0aW9uIG9mIHRoaXMgZG9jdW1lbnQgYXMgYW4gU1NNIGZvcm1hdHRlZCB5YW1sL2pzb24uXG4gICAgICovXG4gIHB1YmxpYyBwcmludCgpOiBzdHJpbmcge1xuICAgIGNvbnN0IGlzWWFtbCA9IHRoaXMucHJvcHMuZG9jdW1lbnRGb3JtYXQgPT0gRG9jdW1lbnRGb3JtYXQuWUFNTDtcbiAgICBpZiAoaXNZYW1sKSB7XG4gICAgICAvLyBQcmVwZW5kIHRoZSBkb2N1bWVudCB3aXRoIHRoZSBoZWFkZXIgaWYgZGVmaW5lZFxuICAgICAgcmV0dXJuIHRoaXMuaGVhZGVyV2l0aENvbW1lbnRzKCkgKyB5YW1sLmR1bXAodGhpcy5idWlsZFNzbURvY3VtZW50KCkpO1xuICAgIH0gZWxzZSB7XG4gICAgICAvLyBUaGVyZSBpcyBubyB3YXkgdG8gcHJvdmlkZSB0aGUgaGVhZGVyIGNvbW1lbnQgaW4gYSBqc29uIGRvY3VtZW50XG4gICAgICByZXR1cm4gSlNPTi5zdHJpbmdpZnkodGhpcy5idWlsZFNzbURvY3VtZW50KCkpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgICAqIFByZXBlbmQgaGVhZGVyIGxpbmUgd2l0aCAjIGlmIGl0IGRvZXMgbm90IHN0YXJ0IHdpdGggIy5cbiAgICAgKiBJZiBubyBoZWFkZXIgaXMgcHJvdmlkZWQsIHRoaXMgbWV0aG9kIHdpbGwgcmV0dXJuIGFuIGVtcHRyeSBzdHJpbmcuXG4gICAgICogSGVhZGVycyBhcmUgb25seSBhdmFpbGFibGUgaW4geWFtbCBmb3JtYXQuXG4gICAgICovXG4gIHByaXZhdGUgaGVhZGVyV2l0aENvbW1lbnRzKCk6IHN0cmluZyB7XG4gICAgaWYgKHRoaXMuaGVhZGVyID09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuICcnO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5oZWFkZXIuc3BsaXQoRU9MKS5tYXAobGluZSA9PiBsaW5lLnRyaW0oKS5zdGFydHNXaXRoKCcjJykgPyBsaW5lIDogJyMgJyArIGxpbmUpLmpvaW4oRU9MKSArXG4gICAgICAgICAgICAvLyBcIi0tLVwiIGRlbWFya3MgdGhlIGVuZCBvZiB0aGUgaGVhZGVyIGFuZCB0aGUgc3RhcnQgb2YgdGhlIGRvY3VtZW50LlxuICAgICAgICAgICAgJ1xcbi0tLVxcbic7XG4gIH1cblxufVxuIl19