from collections import OrderedDict

import numpy as np

from PyNumeca.reader.iecEntry import iecEntry
from PyNumeca.reader.iecGroup import iecGroup
from PyNumeca.reader.niBladeGeometryEntry import niBladeGeometryEntry
from PyNumeca.reader.zrCurveEntry import zrCurveEntry


class numecaParser(OrderedDict):
    def __init__(self, filename=""):
        super().__init__()
        self.fileName = filename
        self.stringData = ""
        self.parsing()

        if filename != "":
            self.load(filename)

    def load(self, filename):
        self.fileName = filename
        try:
            with open(self.fileName, 'r') as file:
                self.stringData = file.readlines()
        except Exception as e:
            self.error_notification(e)
        self.parsing()

    def extract_d3d_parameters_string(self):
        output_string = ""
        try:
            number_of_parameters = self['ROOT'] \
                ["DESIGN3D_COMPUTATION_0"] \
                ["DATABASE_GENERATION_MINAMO_0"] \
                ["SIMULATION_TASK_MANAGER_0"] \
                ["SIMULATION_TASKS_0"] \
                ["SIMULATION_TASK_PARAM_GEOM_MODEL_0"] \
                ["PARAMETRIC_LAYER_0"] \
                ["PARAMETERS_0"] \
                ["NUMBER_OF_PARAMETERS"].value

            output_string += number_of_parameters + "\n"

            for key, value in self['ROOT'] \
                    ["DESIGN3D_COMPUTATION_0"] \
                    ["DATABASE_GENERATION_MINAMO_0"] \
                    ["SIMULATION_TASK_MANAGER_0"] \
                    ["SIMULATION_TASKS_0"] \
                    ["SIMULATION_TASK_PARAM_GEOM_MODEL_0"] \
                    ["PARAMETRIC_LAYER_0"] \
                    ["PARAMETERS_0"].items():
                if "PARAMETER_" in key:
                    output_string += value["NAME"].value + "\n"

            return output_string

        except Exception as e:
            raise CustomError

    def extract_d3d_parameters_list(self):
        output_list = []
        try:
            for key, value in self['ROOT'] \
                    ["DESIGN3D_COMPUTATION_0"] \
                    ["DATABASE_GENERATION_MINAMO_0"] \
                    ["SIMULATION_TASK_MANAGER_0"] \
                    ["SIMULATION_TASKS_0"] \
                    ["SIMULATION_TASK_PARAM_GEOM_MODEL_0"] \
                    ["PARAMETRIC_LAYER_0"] \
                    ["PARAMETERS_0"].items():
                if "PARAMETER_" in key:
                    output_list.append(value["NAME"].value)

            return output_list
        except Exception as e:
            raise CustomError

    def create_d3d_default_parameter(self):
        stringlist = []
        stringlist.append("                     NI_BEGIN\tPARAMETER\n")
        stringlist.append("                        NAME                           DEFAULT\n")
        stringlist.append("                        PARAMETRIC_TYPE                DOUBLE\n")
        stringlist.append("                        LIMIT_MIN                      -1000000000\n")
        stringlist.append("                        LIMIT_MAX                      1000000000\n")
        stringlist.append("                        VALUE                          0.002\n")
        stringlist.append("                        VALUE_MIN                      0.001\n")
        stringlist.append("                        VALUE_MAX                      0.003\n")
        stringlist.append("                        VALUE_REF                      1\n")
        stringlist.append("                        NB_LEVELS                      2\n")
        stringlist.append("                        QUANTITY_TYPE                  VALUE\n")
        stringlist.append("                        UNCERTAIN                      FALSE\n")
        stringlist.append("                        REDUCTION                      FALSE\n")
        stringlist.append("                     NI_END\tPARAMETER\n")
        return self.deepTree(stringlist, 0)

    def create_d3d_parameter(self, name, type, value, min=None, max=None):
        new_parameter, _ = self.create_d3d_default_parameter()
        new_parameter["NAME"].value = name
        new_parameter["PARAMETRIC_TYPE"].value = type
        new_parameter["VALUE"].value = value
        if min is not None and max is not None:
            new_parameter["VALUE_MIN"].value = min
            new_parameter["VALUE_MAX"].value = max
        else:
            new_parameter["VALUE_MIN"].value = value
            new_parameter["VALUE_MAX"].value = value
        return new_parameter

    def add_d3d_parameter(self, *arg):
        parameters_layer = self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"] \
            ["SIMULATION_TASKS_0"] \
            ["SIMULATION_TASK_PARAM_GEOM_MODEL_0"] \
            ["PARAMETRIC_LAYER_0"] \
            ["PARAMETERS_0"]

        if len(arg) == 1 and isinstance(arg[0], iecGroup):
            new_parameter = arg[0]
        elif len(arg) == 3:
            new_parameter = self.create_d3d_parameter(arg[0], arg[1], arg[2])
        elif len(arg) == 5:
            new_parameter = self.create_d3d_parameter(arg[0], arg[1], arg[2], arg[3], arg[4])
        else:
            raise CustomError

        number_of_parameters = parameters_layer["NUMBER_OF_PARAMETERS"].value
        new_number_of_parameters = str(int(number_of_parameters) + 1)
        parameters_layer["NUMBER_OF_PARAMETERS"].value = new_number_of_parameters

        n = 0
        new_group_name = "PARAMETER" + "_" + str(n)
        while new_group_name in parameters_layer:
            n += 1
            new_group_name = "PARAMETER" + "_" + str(n)

        self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"] \
            ["SIMULATION_TASKS_0"] \
            ["SIMULATION_TASK_PARAM_GEOM_MODEL_0"] \
            ["PARAMETRIC_LAYER_0"] \
            ["PARAMETERS_0"] = self.insert_key_value(parameters_layer, new_group_name, "NI_END_PARAMETERS",
                                                     new_parameter)

    def remove_d3d_parameter(self, number=None):
        parameters_layer = self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"] \
            ["SIMULATION_TASKS_0"] \
            ["SIMULATION_TASK_PARAM_GEOM_MODEL_0"] \
            ["PARAMETRIC_LAYER_0"] \
            ["PARAMETERS_0"]
        number_of_parameters = parameters_layer["NUMBER_OF_PARAMETERS"].value

        if int(number_of_parameters) > 0 and number is None:
            new_number_of_parameters = str(int(number_of_parameters) - 1)
            parameters_layer["NUMBER_OF_PARAMETERS"].value = new_number_of_parameters

            n = 0
            last_group_name = "PARAMETER" + "_" + str(n)
            while last_group_name in parameters_layer:
                n += 1
                last_group_name = "PARAMETER" + "_" + str(n)
            n -= 1
            last_group_name = "PARAMETER" + "_" + str(n)

            self['ROOT'] \
                ["DESIGN3D_COMPUTATION_0"] \
                ["DATABASE_GENERATION_MINAMO_0"] \
                ["SIMULATION_TASK_MANAGER_0"] \
                ["SIMULATION_TASKS_0"] \
                ["SIMULATION_TASK_PARAM_GEOM_MODEL_0"] \
                ["PARAMETRIC_LAYER_0"] \
                ["PARAMETERS_0"].pop(last_group_name)

        elif int(number_of_parameters) > 0 and (isinstance(number, int) or isinstance(number, str)):

            new_number_of_parameters = str(int(number_of_parameters) - 1)
            parameters_layer["NUMBER_OF_PARAMETERS"].value = new_number_of_parameters

            if str(number).isdigit():
                last_group_name = "PARAMETER" + "_" + str(number)
            elif number in self.extract_d3d_parameters_list():
                last_group_name = ""
                for key, value in parameters_layer.items():
                    if "PARAMETER_" in key and number == value["NAME"].value:
                        last_group_name = key
            else:
                raise CustomError

            self['ROOT'] \
                ["DESIGN3D_COMPUTATION_0"] \
                ["DATABASE_GENERATION_MINAMO_0"] \
                ["SIMULATION_TASK_MANAGER_0"] \
                ["SIMULATION_TASKS_0"] \
                ["SIMULATION_TASK_PARAM_GEOM_MODEL_0"] \
                ["PARAMETRIC_LAYER_0"] \
                ["PARAMETERS_0"].pop(last_group_name)

        else:
            pass

    @staticmethod
    def insert_key_value(a_dict, key, pos_key, value):
        new_dict = iecGroup()
        for k, v in a_dict.items():
            if k == pos_key:
                new_dict[key] = value  # insert new key
            new_dict[k] = v
        return new_dict

    def parsing(self):
        if isinstance(self.stringData, list) and "GEOMETRY TURBO" in self.stringData[0]:
            self.stringData[0] = "GEOMETRY-TURBO\n"
        self['ROOT'], temp = self.deepTree(self.stringData, 0)
        self['ROOT'] = self.deep_reorder_zrcurve(self['ROOT'])
        # self['ROOT'] = self.deepReorderNiBladeGeometry(self['ROOT'])

    def deep_reorder_zrcurve(self, group):
        new_group = iecGroup()
        for key, value in group.items():
            if isinstance(value, iecGroup) and "zrcurve" in (list(value.items()))[0][0]:
                new_group[key] = zrCurveEntry(value)
            elif (isinstance(value, iecGroup) and any(
                    x in (list(value.items()))[0][0] for x in ["nibladegeometry", "NIBLADEGEOMETRY"])):
                new_group[key] = niBladeGeometryEntry(value)
                # print (key)
                # new_group[key] = self.deepReorderZRcurve(value)
            #    new_group[key] = niBladeGeometryEntry(value)
            elif isinstance(value, iecGroup):
                new_group[key] = self.deep_reorder_zrcurve(value)
            else:
                new_group[key] = value
        return new_group

    def deepTree(self, string_data, line_index):
        newcall = True
        group = iecGroup()
        while line_index < len(string_data):
            line_string = string_data[line_index]
            entry = iecEntry()
            entry.parseSimpleEntry(line_string)
            if newcall and "NI_BEGIN" in line_string:
                # key = entry.key + "_" + entry.value
                if isinstance(entry.value, list):
                    key = entry.key + "_" + entry.tag + "_" + entry.value[0]
                else:
                    key = entry.key + "_" + entry.value
                group[key] = entry
                newcall = False
                line_index += 1
            elif line_index == 0 and entry.key == "GEOMETRY-TURBO":
                string_data[0] = "GEOMETRY TURBO\n"
                group_name = "GEOMTURBO"
                group[group_name], line_index = self.deepTree(string_data, line_index)
            elif line_index == 0 and entry.value == "TURBO":
                key = entry.key + "_" + entry.value
                group[key] = entry
                newcall = False
                line_index += 1
            elif not newcall and "NI_BEGIN" in line_string:
                n = 0
                if isinstance(entry.value, list):
                    group_name = entry.tag + "_" + entry.value[0] + "_" + str(n)
                else:
                    group_name = entry.value + "_" + str(n)
                while group_name in group:
                    n += 1
                    if isinstance(entry.value, list):
                        group_name = entry.tag + "_" + entry.value[0] + "_" + str(n)
                    else:
                        group_name = entry.value + "_" + str(n)

                group[group_name], line_index = self.deepTree(string_data, line_index)
            elif "NI_END" in line_string:
                key = entry.key + "_" + entry.value
                group[key] = entry
                newcall = False
                line_index += 1
                return group, line_index
            elif line_string.lstrip(' ').startswith(('*', '#')):
                n = 0
                key = 'comment' + "_" + str(n)
                while key in group:
                    n += 1
                    key = 'comment_' + str(n)
                group[key] = entry
                newcall = False
                line_index += 1
            elif line_string == "\n":
                n = 0
                key = 'empty_line' + "_" + str(n)
                while key in group:
                    n += 1
                    key = 'empty_line_' + str(n)
                group[key] = entry
                newcall = False
                line_index += 1
            else:
                key = entry.key
                n = 0
                while key in group:
                    n += 1
                    key = entry.key + "_" + str(n)
                group[key] = entry
                newcall = False
                line_index += 1

        return group, line_index

    @staticmethod
    def error_notification(e):
        print(e)
        print("Reading Error Notification")

    def outputString(self):
        return self["ROOT"].outputString()

    def checkSetup(self):
        simulation_task_manager = self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"]

        simulation_task_manager["MODEL_EXTENSION"].value = "prt"
        simulation_task_manager["CFD_GEOM_EXTENSION"].value = "stp"

        for idx, group in enumerate(
                (simulation_task_manager["SIMULATION_TASKS_0"]["SIMULATION_TASK_PARAM_GEOM_MODEL_0"],
                 simulation_task_manager["SIMULATION_TASKS_0"]["SIMULATION_TASK_IGG_AUTOGRID_0"],
                 simulation_task_manager["SIMULATION_TASKS_0"]["SIMULATION_TASK_EURANUS_TURBO_0"],
                 simulation_task_manager["SIMULATION_TASKS_0"]["SIMULATION_TASK_CFVIEW_0"])):
            group["TASK_PROCESSING"].value = str(int(not (idx % 3)))  # Activate only n.0 and n.3
            group["CHAIN_TYPE"].value = "EXTERNAL"
            group["IMPORT_SCRIPTS"].value = "1"

    def setParamScript(self, script_name):
        self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"] \
            ["SIMULATION_TASKS_0"] \
            ["SIMULATION_TASK_PARAM_GEOM_MODEL_0"] \
            ["EXTERNAL_SCRIPT"].value = '"' + script_name + '"'

    def setMeshScript(self, script_name):
        self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"] \
            ["SIMULATION_TASKS_0"] \
            ["SIMULATION_TASK_IGG_AUTOGRID_0"] \
            ["EXTERNAL_SCRIPT"].value = '"' + script_name + '"'

    def setSolverScript(self, script_name):
        self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"] \
            ["SIMULATION_TASKS_0"] \
            ["SIMULATION_TASK_EURANUS_TURBO_0"] \
            ["EXTERNAL_SCRIPT"].value = '"' + script_name + '"'

    def setPostProScript(self, script_name):
        self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"] \
            ["SIMULATION_TASKS_0"] \
            ["SIMULATION_TASK_CFVIEW_0"] \
            ["EXTERNAL_SCRIPT"].value = '"' + script_name + '"'

    def createPostProDefaultMacro(self):
        stringlist = []
        stringlist.append("                  NI_BEGIN\tUSER_DEFINED_MACRO\n")
        stringlist.append("                     SOLUTION_FILE                                      3D\n")
        stringlist.append(
            "                     MACRO_FILE                                         myPath/myScript.py\n")
        stringlist.append("                     DATA_FILE                                          myScript.dat\n")
        stringlist.append("                     QUANTITY_NAME                                      myQuantity\n")
        stringlist.append("                     DIM                                                SCALAR\n")
        stringlist.append("                     ACTIVE                                             1\n")
        stringlist.append("                  NI_END\tUSER_DEFINED_MACRO\n")
        return self.deepTree(stringlist, 0)

    def createPostProMacro(self, script_name):
        new_post_pro_macro, _ = self.createPostProDefaultMacro()
        new_post_pro_macro["MACRO_FILE"].value = "macro_CFV/" + script_name
        new_post_pro_macro["DATA_FILE"].value = script_name.rsplit('.', 1)[0] + ".dat"
        new_post_pro_macro["QUANTITY_NAME"].value = script_name.rsplit('.', 1)[0]
        return new_post_pro_macro

    def addPostProMacro(self, script_name):
        postProMacroLayer = self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"] \
            ["SIMULATION_TASKS_0"] \
            ["SIMULATION_TASK_CFVIEW_0"] \
            ["USER_DEFINED_MACROS_EXTERNAL_0"]

        if isinstance(script_name, iecGroup):
            new_post_pro_macro = script_name
        elif isinstance(script_name, str):
            new_post_pro_macro = self.createPostProMacro(script_name)
        else:
            raise CustomError

        n = 0
        new_group_name = "USER_DEFINED_MACRO" + "_" + str(n)
        while new_group_name in postProMacroLayer:
            n += 1
            new_group_name = "USER_DEFINED_MACRO" + "_" + str(n)

        self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"] \
            ["SIMULATION_TASKS_0"] \
            ["SIMULATION_TASK_CFVIEW_0"] \
            ["USER_DEFINED_MACROS_EXTERNAL_0"] = \
            self.insert_key_value(postProMacroLayer, new_group_name, "NI_END_USER_DEFINED_MACROS_EXTERNAL",
                                  new_post_pro_macro)

    def removePostProMacro(self, number=None):
        post_pro_macro_layer = self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"] \
            ["SIMULATION_TASKS_0"] \
            ["SIMULATION_TASK_CFVIEW_0"] \
            ["USER_DEFINED_MACROS_EXTERNAL_0"]

        if number is None:
            n = 0
            last_group_name = "USER_DEFINED_MACRO" + "_" + str(n)
            while last_group_name in post_pro_macro_layer:
                n += 1
                last_group_name = "USER_DEFINED_MACRO" + "_" + str(n)
            n -= 1
            last_group_name = "USER_DEFINED_MACRO" + "_" + str(n)

        elif isinstance(number, int):
            last_group_name = "USER_DEFINED_MACRO" + "_" + str(number)

        else:
            raise CustomError

        self['ROOT'] \
            ["DESIGN3D_COMPUTATION_0"] \
            ["DATABASE_GENERATION_MINAMO_0"] \
            ["SIMULATION_TASK_MANAGER_0"] \
            ["SIMULATION_TASKS_0"] \
            ["SIMULATION_TASK_CFVIEW_0"] \
            ["USER_DEFINED_MACROS_EXTERNAL_0"].pop(last_group_name)

    def exportNpyArray(self, row_number=0, blade_number=0):
        ni_blade_geometry = self.retrieveNiBladeGeometry(row_number, blade_number)

        suction = []
        pressure = []
        for item in ni_blade_geometry.suctionList:
            section = np.vstack([item.X, item.Y, item.Z, np.zeros(item.numberOfPointsInt)]).transpose()
            suction.append(section)
        for item in ni_blade_geometry.pressureList:
            section = np.vstack([item.X, item.Y, item.Z, np.ones(item.numberOfPointsInt)]).transpose()
            section = np.flip(section, axis=0)
            pressure.append(section)
        return np.array([(np.asarray(suction), np.asarray(pressure))], dtype=float)

    def importNpyArray(self, array, row_number=0, blade_number=0):
        # array shape (2,nSections,nPoints,4)
        ni_blade_geometry = self.retrieveNiBladeGeometry(row_number, blade_number)
        n_sections_from_array = array.shape[1]
        ni_blade_geometry.setNumberOfSections(n_sections_from_array)
        for idx, suction in enumerate(ni_blade_geometry.suctionList):
            new_list = array[0][idx].transpose().tolist()
            suction.updateArrays(new_list[0], new_list[1], new_list[2])
        for idx, pressure in enumerate(ni_blade_geometry.pressureList):
            new_list = array[1][idx].transpose().tolist()
            pressure.updateArrays(new_list[0][::-1], new_list[1][::-1], new_list[2][::-1])

    def retrieveNiBladeGeometry(self, row_number=0, blade_number=0):
        row_occurence = -1
        blade_occurence = -1
        row_key = ""
        blade_key = ""

        for key in self["ROOT"]["GEOMTURBO"].keys():
            if "nirow" in key or "NIROW" in key:
                row_occurence += 1
                if row_occurence == row_number:
                    row_key = key

        for key in self["ROOT"]["GEOMTURBO"][row_key].keys():
            if "NIBlade" in key or "NIBLADE" in key:
                blade_occurence += 1
                if blade_occurence == blade_number:
                    blade_key = key

        if row_key == "":
            print("Row not defined")
        if blade_key == "":
            print("Blade not defined")

        if "NIROW" in row_key:
            ni_blade_geometry = self["ROOT"]["GEOMTURBO"][row_key][blade_key]["NIBLADEGEOMETRY_0"]
        elif "nirow" in row_key:
            ni_blade_geometry = self["ROOT"]["GEOMTURBO"][row_key][blade_key]["nibladegeometry_0"]
        else:
            print("ERROR")

        return ni_blade_geometry


class CustomError(Exception):
    pass
