"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegrationPattern = exports.TaskStateBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const chain_1 = require("../chain");
const state_1 = require("./state");
/**
 * Define a Task state in the state machine.
 *
 * Reaching a Task state causes some work to be executed, represented by the
 * Task's resource property. Task constructs represent a generic Amazon
 * States Language Task.
 *
 * For some resource types, more specific subclasses of Task may be available
 * which are more convenient to use.
 *
 * @stability stable
 */
class TaskStateBase extends state_1.State {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.endStates = [this];
        this.timeout = props.timeout;
        this.heartbeat = props.heartbeat;
    }
    /**
     * Add retry configuration for this state.
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     *
     * @stability stable
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state.
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     *
     * @stability stable
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state.
     *
     * @stability stable
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state.
     *
     * @stability stable
     */
    toStateJson() {
        return {
            ...this.renderNextEnd(),
            ...this.renderRetryCatch(),
            ...this.renderTaskBase(),
            ...this._renderTask(),
        };
    }
    /**
     * Return the given named metric for this Task.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metric(metricName, props) {
        var _b;
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: (_b = this.taskMetrics) === null || _b === void 0 ? void 0 : _b.metricDimensions,
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * The interval, in milliseconds, between the time the Task starts and the time it closes.
     *
     * @default - average over 5 minutes
     * @stability stable
     */
    metricRunTime(props) {
        var _b;
        return this.taskMetric((_b = this.taskMetrics) === null || _b === void 0 ? void 0 : _b.metricPrefixSingular, 'RunTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, for which the activity stays in the schedule state.
     *
     * @default - average over 5 minutes
     * @stability stable
     */
    metricScheduleTime(props) {
        var _b;
        return this.taskMetric((_b = this.taskMetrics) === null || _b === void 0 ? void 0 : _b.metricPrefixSingular, 'ScheduleTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, between the time the activity is scheduled and the time it closes.
     *
     * @default - average over 5 minutes
     * @stability stable
     */
    metricTime(props) {
        var _b;
        return this.taskMetric((_b = this.taskMetrics) === null || _b === void 0 ? void 0 : _b.metricPrefixSingular, 'Time', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the number of times this activity is scheduled.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricScheduled(props) {
        var _b;
        return this.taskMetric((_b = this.taskMetrics) === null || _b === void 0 ? void 0 : _b.metricPrefixPlural, 'Scheduled', props);
    }
    /**
     * Metric for the number of times this activity times out.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricTimedOut(props) {
        var _b;
        return this.taskMetric((_b = this.taskMetrics) === null || _b === void 0 ? void 0 : _b.metricPrefixPlural, 'TimedOut', props);
    }
    /**
     * Metric for the number of times this activity is started.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricStarted(props) {
        var _b;
        return this.taskMetric((_b = this.taskMetrics) === null || _b === void 0 ? void 0 : _b.metricPrefixPlural, 'Started', props);
    }
    /**
     * Metric for the number of times this activity succeeds.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricSucceeded(props) {
        var _b;
        return this.taskMetric((_b = this.taskMetrics) === null || _b === void 0 ? void 0 : _b.metricPrefixPlural, 'Succeeded', props);
    }
    /**
     * Metric for the number of times this activity fails.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricFailed(props) {
        var _b;
        return this.taskMetric((_b = this.taskMetrics) === null || _b === void 0 ? void 0 : _b.metricPrefixPlural, 'Failed', props);
    }
    /**
     * Metric for the number of times the heartbeat times out for this activity.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricHeartbeatTimedOut(props) {
        var _b;
        return this.taskMetric((_b = this.taskMetrics) === null || _b === void 0 ? void 0 : _b.metricPrefixPlural, 'HeartbeatTimedOut', props);
    }
    /**
     * Called whenever this state is bound to a graph.
     *
     * Can be overridden by subclasses.
     *
     * @stability stable
     */
    whenBoundToGraph(graph) {
        super.whenBoundToGraph(graph);
        for (const policyStatement of this.taskPolicies || []) {
            graph.registerPolicyStatement(policyStatement);
        }
    }
    taskMetric(prefix, suffix, props) {
        if (prefix === undefined) {
            throw new Error('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        }
        return this.metric(prefix + suffix, props);
    }
    renderTaskBase() {
        var _b, _c;
        return {
            Type: 'Task',
            Comment: this.comment,
            TimeoutSeconds: (_b = this.timeout) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            HeartbeatSeconds: (_c = this.heartbeat) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            InputPath: state_1.renderJsonPath(this.inputPath),
            OutputPath: state_1.renderJsonPath(this.outputPath),
            ResultPath: state_1.renderJsonPath(this.resultPath),
            ...this.renderResultSelector(),
        };
    }
}
exports.TaskStateBase = TaskStateBase;
_a = JSII_RTTI_SYMBOL_1;
TaskStateBase[_a] = { fqn: "@aws-cdk/aws-stepfunctions.TaskStateBase", version: "1.108.0" };
/**
 * AWS Step Functions integrates with services directly in the Amazon States Language.
 *
 * You can control these AWS services using service integration patterns:
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 * @stability stable
 */
var IntegrationPattern;
(function (IntegrationPattern) {
    IntegrationPattern["REQUEST_RESPONSE"] = "REQUEST_RESPONSE";
    IntegrationPattern["RUN_JOB"] = "RUN_JOB";
    IntegrationPattern["WAIT_FOR_TASK_TOKEN"] = "WAIT_FOR_TASK_TOKEN";
})(IntegrationPattern = exports.IntegrationPattern || (exports.IntegrationPattern = {}));
//# sourceMappingURL=data:application/json;base64,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