"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const iam = require("aws-cdk-lib/aws-iam");
const route53 = require("aws-cdk-lib/aws-route53");
const patterns = require("aws-cdk-lib/aws-route53-patterns");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const cr = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const origin_request_function_1 = require("./origin-request-function");
/**
 * A CloudFront static website hosted on S3
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.bucket = new s3.Bucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
        });
        const certificate = new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName: props.domainName,
            hostedZone: props.hostedZone,
            region: 'us-east-1',
        });
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: {
                origin: new origins.S3Origin(this.bucket),
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [
                    {
                        eventType: cloudfront.LambdaEdgeEventType.ORIGIN_REQUEST,
                        functionVersion: new origin_request_function_1.OriginRequestFunction(this, 'OriginRequest'),
                    },
                ],
                responseHeadersPolicy: props.responseHeadersPolicy ?? new cloudfront.ResponseHeadersPolicy(this, 'ResponseHeadersPolicy', {
                    securityHeadersBehavior: StaticWebsite.defaultSecurityHeadersBehavior,
                }),
            },
            defaultRootObject: 'index.html',
            domainNames: [props.domainName],
            certificate,
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2021,
            httpVersion: 'http2and3',
        });
        new route53.ARecord(this, 'ARecord', {
            recordName: props.domainName,
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
        });
        new route53.AaaaRecord(this, 'AaaaRecord', {
            recordName: props.domainName,
            zone: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(this.distribution)),
        });
        if (props.backendConfiguration) {
            // Save backend config to bucket, can be queried by the frontend
            new cr.AwsCustomResource(this, 'PutConfig', {
                policy: cr.AwsCustomResourcePolicy.fromStatements([
                    new iam.PolicyStatement({
                        actions: ['s3:PutObject'],
                        resources: [this.bucket.arnForObjects('config.json')],
                    }),
                ]),
                onUpdate: {
                    service: 'S3',
                    action: 'putObject',
                    parameters: {
                        Bucket: this.bucket.bucketName,
                        Key: 'config.json',
                        Body: aws_cdk_lib_1.Stack.of(this).toJsonString(props.backendConfiguration),
                        ContentType: 'application/json',
                        CacheControl: 'max-age=0, no-cache, no-store, must-revalidate',
                    },
                    physicalResourceId: cr.PhysicalResourceId.of('config'),
                },
            });
        }
        if (shouldAddRedirect(props)) {
            const httpsRedirect = new patterns.HttpsRedirect(this, 'HttpsRedirect', {
                targetDomain: props.domainName,
                zone: props.hostedZone,
                recordNames: props.redirects,
            });
            // Force minimum protocol version
            const redirectDistribution = httpsRedirect.node.tryFindChild('RedirectDistribution');
            const cfnDistribution = redirectDistribution.node.tryFindChild('CFDistribution');
            cfnDistribution.addPropertyOverride('DistributionConfig.ViewerCertificate.MinimumProtocolVersion', 'TLSv1.2_2021');
        }
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "cloudstructs.StaticWebsite", version: "0.6.10" };
/**
* Best practice security headers used as default
*/
StaticWebsite.defaultSecurityHeadersBehavior = {
    contentTypeOptions: {
        override: true,
    },
    frameOptions: {
        frameOption: cloudfront.HeadersFrameOption.DENY,
        override: true,
    },
    referrerPolicy: {
        referrerPolicy: cloudfront.HeadersReferrerPolicy.STRICT_ORIGIN_WHEN_CROSS_ORIGIN,
        override: true,
    },
    strictTransportSecurity: {
        accessControlMaxAge: aws_cdk_lib_1.Duration.seconds(31536000),
        includeSubdomains: true,
        preload: true,
        override: true,
    },
    xssProtection: {
        protection: true,
        modeBlock: true,
        override: true,
    },
};
function shouldAddRedirect(props) {
    if (props.redirects && props.redirects.length === 0) {
        return false;
    }
    if (!props.redirects && !aws_cdk_lib_1.Token.isUnresolved(props.domainName)
        && !aws_cdk_lib_1.Token.isUnresolved(props.hostedZone.zoneName)
        && props.domainName === props.hostedZone.zoneName) {
        return false;
    }
    return true;
}
//# sourceMappingURL=data:application/json;base64,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