from pymatgen.analysis.chemenv.coordination_environments.coordination_geometry_finder import symmetry_measure  # type: ignore
from itertools import permutations
import math
import numpy as np  # type: ignore
from typing import Dict, Union
from polyhedral_analysis.coordination_polyhedron import CoordinationPolyhedron


class RotationAnalyser(object):
    """Class for analysing rotational orientation of polyhedra.

    Attributes:
        reference_points (np.array): A Mx3xN numpy array of points around the
            origin ([0.0, 0.0, 0.0]) that define sets of centre-vertex
            vectors for specific reference orientations, e.g. for a single
            orientation of a tetrahedron, reference points will be a size 1x3x4
            array, e.g.::

                [[[ 1.0, -1.0,  1.0],
                  [-1.0, -1.0, -1.0],
                  [ 1.0,  1.0, -1.0],
                  [-1.0,  1.0,  1.0]]]

    """

    def __init__(self,
                 reference_points: np.ndarray) -> None:
        """Initialise a RotationAnalyser instance.

        Args:
            reference_points (np.array): Either a 3xN or Mx3xN numpy array
            of points around the origin ([0.0, 0.0, 0.0]) that define sets of 
            centre-vertex vectors, and are used to classify the rotational
            orientation of polyhedra.

            If a 3xN array is passed in, this will be converted to a 1x3xN array.

        Returns:
            None

        """
        if len(reference_points.shape) == 2:
            reference_points = np.array([reference_points])
        if len(reference_points.shape) == 1:
            raise ValueError(
                'Reference points must all contain the same number of coordinates')
        self.reference_points = reference_points

    def discrete_orientation(self,
                             points: np.ndarray) -> Dict[str, Union[int, float]]:
        """Find the discrete "closest orientation" for an input polyhedron of points.

        For example, a tetrahedon has 12 pure rotation symmetry operations. A distorted
        tetrahedron with vertices approximately aligned along the unordered vectors::

            [[ 1.0, -1.0,  1.0],
             [-1.0, -1.0, -1.0],
             [ 1.0,  1.0, -1.0],
             [-1.0,  1.0,  1.0]]

        can be in one of 12 orientations. This method compares the input points to
        all permutations of self.reference_points that can be generated by proper rotations, and returns
        an index for orientation that minimises the rotational distance between the reordered
        reference points and the input points.

        The algorithm is:
        1. Perform continuous symmetry measure analysis for all permutations of reference points.
        2. Collect operations that have the minimum continous symmetry measure value.
        These give the set of all propert and improper rotations.
        3. Find all proper rotations using det(M_rot)>0.
        4. Calculate the rotational distance.

        Args:
            points (np.array(float)): Nx3 numpy array of points describing the coordinates of the
                input polyhedron, centered around zero.

        Returns:
            (dict): Dictionary describing the orientation, with keys:
                - ``orientation_index`` (int): Index of this particular orientation.
                - ``reference_geometry_index`` (int): Index of the reference geometry
                    the closest discrete orientation is equivalent to.
                - ``rotational_distance`` (float): Angle of rotation from the relevant
                    reference orientation, in radians.
                - ``all_rotational_distances`` (np.array(float)): Array of angles of rotation
                    from each reference orientation.
                - ``symmetry_measure`` (float): The continuous symmetry measure (CSM)
                  for this polyhedron (J. Am. Chem. Soc., 114, 7843-7851 (1992)) 

        """
        points -= np.mean(points, axis=0, dtype=float)
        sm = []
        for rp in self.reference_points:
            sm.extend([symmetry_measure(points, p) for p in permutations(rp)])
        min_sm = min([s['symmetry_measure'] for s in sm])
        pure_rot_sm = [s for s in sm if math.isclose(
            s['symmetry_measure'], min_sm)]
        proper_rot_sm = [s for s in pure_rot_sm if np.linalg.det(
            s['rotation_matrix']) > 0]
        proper_rot_matrices = np.array(
            [s['rotation_matrix'] for s in proper_rot_sm])
        trace = np.trace(proper_rot_matrices, axis1=1, axis2=2)
        rot_distance = np.arccos((trace - 1.0) / 2.0)
        index = np.argmin(rot_distance)
        reference_geometry_index = index // int(
            len(proper_rot_sm) / len(self.reference_points))
        return {'orientation_index': index,
                'reference_geometry_index': reference_geometry_index,
                'rotational_distance': rot_distance[index],
                'symmetry_measure': proper_rot_sm[index]['symmetry_measure'],
                'all_rotational_distances': rot_distance}

    def polyhedron_orientation(self,
                               polyhedron: CoordinationPolyhedron) -> Dict[str, Union[int, float]]:
        points = polyhedron.abstract_geometry.points_wocs_csc()
        return self.discrete_orientation(points)
