# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/core.ipynb (unless otherwise specified).

__all__ = ['GroupedArray', 'transform_series', 'build_transform_name', 'TimeSeries', 'preprocessing_flow',
           'predictions_flow']

# Cell
import concurrent.futures
import copy
import inspect
import os
import warnings
from collections import OrderedDict
from itertools import chain
from typing import Any, Dict, List, Optional, Tuple, Union

import numpy as np
import pandas as pd
from fastcore.foundation import patch, tuplify
from numba import njit
from window_ops.shift import shift_array

# Internal Cell
date_features_dtypes = {
    'year': np.uint16,
    'month': np.uint8,
    'day': np.uint8,
    'hour': np.uint8,
    'minute': np.uint8,
    'second': np.uint8,
    'dayofyear': np.uint16,
    'day_of_year': np.uint16,
    'weekofyear': np.uint8,
    'week': np.uint8,
    'dayofweek': np.uint8,
    'day_of_week': np.uint8,
    'weekday': np.uint8,
    'quarter': np.uint8,
    'daysinmonth': np.uint8,
    'is_month_start': np.uint8,
    'is_month_end': np.uint8,
    'is_quarter_start': np.uint8,
    'is_quarter_end': np.uint8,
    'is_year_start': np.uint8,
    'is_year_end': np.uint8,
}


@njit
def _append_new(data, indptr, new):
    """Append each value of new to each group in data formed by indptr."""
    n_series = len(indptr) - 1
    new_data = np.empty(data.size + new.size, dtype=data.dtype)
    new_indptr = indptr.copy()
    new_indptr[1:] += np.arange(1, n_series + 1)
    for i in range(n_series):
        new_data[new_indptr[i] : new_indptr[i+1] - 1] = data[indptr[i] : indptr[i + 1]]
        new_data[new_indptr[i+1] - 1] = new[i]
    return new_data, new_indptr

# Cell
class GroupedArray:
    """Array made up of different groups. Can be thought of (and iterated) as a list of arrays.

    All the data is stored in a single 1d array `data`.
    The indices for the group boundaries are stored in another 1d array `indptr`."""

    def __init__(self, data: np.ndarray, indptr: np.ndarray):
        self.data = data
        self.indptr = indptr
        self.ngroups = len(indptr) - 1

    def __len__(self) -> int:
        return self.ngroups

    def __getitem__(self, idx: int) -> np.ndarray:
        return self.data[self.indptr[idx]:self.indptr[idx+1]]

    def take_from_groups(self, idx: Union[int, slice]) -> 'GroupedArray':
        """Takes `idx` from each group in the array."""
        ranges = [range(self.indptr[i], self.indptr[i+1])[idx] for i in range(self.ngroups)]
        items = [self.data[rng] for rng in ranges]
        sizes = np.array([item.size for item in items])
        data = np.hstack(items)
        indptr = np.append(0, sizes.cumsum())
        return GroupedArray(data, indptr)

    def append(self, new: np.ndarray) -> 'GroupedArray':
        """Appends each element of `new` to each existing group. Returns a copy."""
        if new.size != self.ngroups:
            raise ValueError(f'new must be of size {self.ngroups}')
        new_data, new_indptr = _append_new(self.data, self.indptr, new)
        return GroupedArray(new_data, new_indptr)

    def __repr__(self) -> str:
        return f'GroupedArray(ndata={self.data.size}, ngroups={self.ngroups})'

# Internal Cell
@njit
def _identity(x: np.ndarray) -> np.ndarray:
    """Do nothing to the input."""
    return x

# Cell
@njit(nogil=True)
def transform_series(data, indptr, updates_only, lag, func, *args) -> np.ndarray:
    """Shifts every group in `data` by `lag` and computes `func(shifted, *args)`.

    If `updates_only=True` only last value of the transformation for each group is returned,
    otherwise the full transformation is returned"""
    n_series = len(indptr) - 1
    if updates_only:
        out = np.empty_like(data[:n_series])
        for i in range(n_series):
            lagged = shift_array(data[indptr[i]:indptr[i+1]], lag)
            out[i] = func(lagged, *args)[-1]
    else:
        out = np.empty_like(data)
        for i in range(n_series):
            lagged = shift_array(data[indptr[i]:indptr[i+1]], lag)
            out[indptr[i]:indptr[i+1]] = func(lagged, *args)
    return out

# Cell
def build_transform_name(lag, tfm, *args) -> str:
    """Creates a name for a transformation based on `lag`, the name of the function and its arguments."""
    if lag == 0:
        return f'lag-{args[0]}'
    tfm_name = f'{tfm.__name__}_lag-{lag}'
    func_params = list(inspect.signature(tfm).parameters.items())[1:]  # remove input array argument
    changed_params = [f'{name}-{value}' for value, (name, param) in zip(args, func_params) if param.default != value]
    if changed_params:
        tfm_name += '_' + '_'.join(changed_params)
    return tfm_name

# Cell
class TimeSeries:
    """Utility class for storing and transforming time series data."""

    def __init__(self,
                 series_df: pd.DataFrame,
                 freq: str = 'D',
                 lags: List[int] = [],
                 lag_transforms: Dict[int, List[Tuple]] = {},
                 date_features: List[str] = [],
                 static_features: Optional[List[str]] = None,
                 num_threads: Optional[int] = None):
        if not series_df.index.is_monotonic_increasing:
            series_df = series_df.sort_index()
        data = series_df.y.values
        if data.dtype not in (np.float32, np.float64):
            data = data.astype(np.float32)
        sizes = series_df.groupby('unique_id').size().values
        cumsizes = sizes.cumsum()
        indptr = np.append(0, cumsizes)
        self.ga = GroupedArray(data, indptr)
        self.uids = series_df.index.unique(level='unique_id')
        self.last_dates = series_df.index.get_level_values('ds')[cumsizes - 1]
        self.freq = pd.tseries.frequencies.to_offset(freq)
        self.static_features = series_df.iloc[cumsizes - 1].reset_index('ds', drop=True).drop(columns='y')
        if static_features is not None:
            self.static_features = self.static_features[static_features]
        self.num_threads = num_threads or os.cpu_count()
        self.date_features = date_features

        self.transforms: Dict[str, Tuple[Any, ...]] = OrderedDict()
        for lag in lags:
            self.transforms[f'lag-{lag}'] = (lag, _identity)
        for lag in lag_transforms.keys():
            for tfm_args in lag_transforms[lag]:
                tfm, *args = tuplify(tfm_args)
                tfm_name = build_transform_name(lag, tfm, *args)
                self.transforms[tfm_name] = (lag, tfm, *args)

        self.y_pred: List[np.ndarray] = []
        self.curr_dates = self.last_dates
        self.test_dates: List[pd.DatetimeIndex] = []

    @property
    def n_series(self):
        return self.ga.ngroups

    @property
    def features(self):
        return list(self.transforms.keys()) + self.date_features

    def __repr__(self):
        return f'TimeSeries(n_series={self.n_series}, freq={self.freq}, transforms={self.transforms.keys()}, date_features={self.date_features})'

    def _apply_transforms(self, updates_only: bool = False):
        results = {}
        offset = 1 if updates_only else 0
        for tfm_name, (lag, tfm, *args) in self.transforms.items():
            results[tfm_name] =  transform_series(self.ga.data, self.ga.indptr, updates_only, lag - offset, tfm, *args)
        return results

    def _apply_multithreaded_transforms(self, updates_only: bool = False):
        future_to_result = {}
        results = {}
        offset = 1 if updates_only else 0
        with concurrent.futures.ThreadPoolExecutor(self.num_threads) as executor:
            for tfm_name, (lag, tfm, *args) in self.transforms.items():
                future = executor.submit(transform_series, self.ga.data, self.ga.indptr, updates_only, lag - offset, tfm, *args)
                future_to_result[future] = tfm_name
            for future in concurrent.futures.as_completed(future_to_result):
                tfm_name = future_to_result[future]
                results[tfm_name] = future.result()
        return results

# Cell
@patch
def compute_transforms(self: TimeSeries) -> Dict[str, np.ndarray]:
    """Compute the transformations defined in the constructor.

    If `num_threads > 1` these are computed using multithreading."""
    if self.num_threads == 1 or len(self.transforms) == 1:
        return self._apply_transforms()
    return self._apply_multithreaded_transforms()

# Cell
@patch
def update_y(self: TimeSeries, new: np.ndarray) -> None:
    """Appends the elements of `new` to every time serie.

    These values are used to update the transformations and are stored as predictions."""
    if len(self.y_pred) == 0:
        self.y_pred = []
    self.y_pred.append(new)
    new_arr = np.asarray(new)
    self.ga = self.ga.append(new_arr)

# Cell
@patch
def update_features(self: TimeSeries) -> pd.DataFrame:
    """Compute the current values of all the features using the latest values of the time series."""
    if self.curr_dates.equals(self.last_dates):
        self.curr_dates = self.last_dates.copy()
        self.test_dates = []
    self.curr_dates += self.freq
    self.test_dates.append(self.curr_dates)

    if self.num_threads == 1 or len(self.transforms) == 1:
        features = self._apply_transforms(updates_only=True)
    else:
        features = self._apply_multithreaded_transforms(updates_only=True)

    for feature in self.date_features:
        feat_vals = getattr(self.curr_dates, feature).values
        features[feature] = feat_vals.astype(date_features_dtypes[feature])

    features_df = pd.DataFrame(features, columns=self.features, index=self.uids)
    nulls_in_cols = features_df.isnull().any()
    if any(nulls_in_cols):
        warnings.warn(f'Found null values in {", ".join(nulls_in_cols[nulls_in_cols].index)}.')
    results_df = self.static_features.join(features_df)
    results_df['ds'] = self.curr_dates
    results_df = results_df.set_index('ds', append=True)
    return results_df

# Cell
@patch
def get_predictions(self: TimeSeries) -> pd.DataFrame:
    """Get all the predicted values with their corresponding ids and datestamps."""
    n_preds = len(self.y_pred)
    idx = pd.Index(chain.from_iterable([uid] * n_preds for uid in self.uids), name='unique_id', dtype=self.uids.dtype)
    df = pd.DataFrame({
        'ds': np.array(self.test_dates).ravel('F'),
        'y_pred': np.array(self.y_pred).ravel('F')},
        index=idx)
    return df

# Cell
def preprocessing_flow(df: pd.DataFrame,
                       freq: str = 'D',
                       lags: List[int] = [],
                       lag_transforms: Dict[int, List[Tuple]] = {},
                       date_features: List[str] = [],
                       static_features: Optional[List[str]] = None,
                       dropna: bool = True,
                       keep_last_n: Optional[int] = None,
                       num_threads: Optional[int] = os.cpu_count()) -> Tuple[TimeSeries, pd.DataFrame]:
    """Standard preprocessing flow.

    Returns a `TimeSeries` object for the forecasting step and a pandas DataFrame for training."""
    df = df.set_index('ds', append=True).sort_index()
    series = TimeSeries(df, freq, lags, lag_transforms, date_features,
                        static_features, num_threads)
    df = df.reset_index('ds')

    features = series.compute_transforms()  # type: ignore
    for k in series.transforms.keys():
        df[k] = features[k]

    if dropna:
        df.dropna(inplace=True)

    for feature in date_features:
        feat_vals = getattr(df.ds.dt, feature).values
        df[feature] = feat_vals.astype(date_features_dtypes[feature])

    if keep_last_n is not None:
        series.ga = series.ga.take_from_groups(slice(-keep_last_n, None))

    return series, df

# Cell
def predictions_flow(series: TimeSeries,
                     model,
                     horizon: int) -> pd.DataFrame:
    """Standard predictions flow.

    Uses `model` to compute the predictions for the next `horizon` steps for every serie in `series`."""
    series = copy.copy(series)
    # this avoids installing xgboost only to check if the model is instance of xgb.Booster
    model_is_xgb_booster = type(model).__module__ == 'xgboost.core' and type(model).__name__ == 'Booster'
    for _ in range(horizon):
        new_x = series.update_features()  # type: ignore
        if model_is_xgb_booster:
            import xgboost as xgb
            new_x = xgb.DMatrix(new_x)
        predictions = model.predict(new_x)
        series.update_y(predictions)  # type: ignore
    return series.get_predictions()  # type: ignore