"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyObjectCache = exports.transformJwkToKeyObject = exports.JwtRsaVerifier = exports.JwtRsaVerifierBase = exports.verifyJwtSync = exports.verifyJwt = exports.JwtSignatureAlgorithms = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const asn1_1 = require("./asn1");
const assert_1 = require("./assert");
const error_1 = require("./error");
const jwk_1 = require("./jwk");
const jwt_1 = require("./jwt");
/**
 * Enum to map supported JWT signature algorithms with OpenSSL message digest algorithm names
 */
var JwtSignatureAlgorithms;
(function (JwtSignatureAlgorithms) {
    JwtSignatureAlgorithms["RS256"] = "RSA-SHA256";
    JwtSignatureAlgorithms["RS384"] = "RSA-SHA384";
    JwtSignatureAlgorithms["RS512"] = "RSA-SHA512";
})(JwtSignatureAlgorithms = exports.JwtSignatureAlgorithms || (exports.JwtSignatureAlgorithms = {}));
/**
 * Verify a JWTs signature agains a JWK. This function throws an error if the JWT is not valid
 *
 * @param header The decoded and JSON parsed JWT header
 * @param headerB64 The JWT header in base64 encoded form
 * @param payload The decoded and JSON parsed JWT payload
 * @param payloadB64 The JWT payload in base64 encoded form
 * @param signatureB64 The JWT signature in base64 encoded form
 * @param jwk The JWK with which the JWT was signed
 * @param jwkToKeyObjectTransformer Function to transform the JWK into a Node.js native key object
 * @returns void
 */
function verifySignatureAgainstJwk(header, headerB64, payload, payloadB64, signatureB64, jwk, jwkToKeyObjectTransformer = exports.transformJwkToKeyObject) {
    // Check JWK use
    assert_1.assertStringEquals('JWK use', jwk.use, 'sig', error_1.JwkInvalidUseError);
    // Check JWK kty
    assert_1.assertStringEquals('JWK kty', jwk.kty, 'RSA', error_1.JwkInvalidKtyError);
    // Check that JWT signature algorithm matches JWK
    if (jwk.alg) {
        assert_1.assertStringEquals('JWT signature algorithm', header.alg, jwk.alg, error_1.JwtInvalidSignatureAlgorithmError);
    }
    // Check JWT signature algorithm is one of RS256, RS384, RS512
    assert_1.assertStringArrayContainsString('JWT signature algorithm', header.alg, ['RS256', 'RS384', 'RS512'], error_1.JwtInvalidSignatureAlgorithmError);
    // Convert JWK modulus and exponent into DER public key
    const publicKey = jwkToKeyObjectTransformer(jwk, payload.iss, header.kid);
    // Verify the JWT signature
    const valid = crypto_1.createVerify(JwtSignatureAlgorithms[header.alg])
        .update(`${headerB64}.${payloadB64}`)
        .verify(publicKey, signatureB64, 'base64');
    if (!valid) {
        throw new error_1.JwtInvalidSignatureError('Invalid signature');
    }
}
/**
 * Verify a JWT asynchronously (thus allowing for the JWKS to be fetched from the JWKS URI)
 *
 * @param jwt The JWT
 * @param jwksUri The JWKS URI, where the JWKS can be fetched from
 * @param options Verification options
 * @param jwkFetcher A function that can execute the fetch of the JWKS from the JWKS URI
 * @param jwkToKeyObjectTransformer A function that can transform a JWK into a Node.js native key object
 * @returns Promise that resolves to the payload of the JWT––if the JWT is valid, otherwise the promise rejects
 */
async function verifyJwt(jwt, jwksUri, options, jwkFetcher, jwkToKeyObjectTransformer) {
    return verifyDecomposedJwt(jwt_1.decomposeJwt(jwt), jwksUri, options, jwkFetcher, jwkToKeyObjectTransformer);
}
exports.verifyJwt = verifyJwt;
/**
 * Verify (asynchronously) a JWT that is already decomposed (by function `decomposeJwt`)
 *
 * @param decomposedJwt The decomposed JWT
 * @param jwksUri The JWKS URI, where the JWKS can be fetched from
 * @param options Verification options
 * @param jwkFetcher A function that can execute the fetch of the JWKS from the JWKS URI
 * @param jwkToKeyObjectTransformer A function that can transform a JWK into a Node.js native key object
 * @returns Promise that resolves to the payload of the JWT––if the JWT is valid, otherwise the promise rejects
 */
async function verifyDecomposedJwt(decomposedJwt, jwksUri, options, jwkFetcher = jwk_1.fetchJwk, jwkToKeyObjectTransformer) {
    const { header, headerB64, payload, payloadB64, signatureB64 } = decomposedJwt;
    const jwk = await jwkFetcher(jwksUri, decomposedJwt);
    verifySignatureAgainstJwk(header, headerB64, payload, payloadB64, signatureB64, jwk, jwkToKeyObjectTransformer);
    try {
        jwt_1.validateJwtFields(payload, options);
        if (options.customJwtCheck) {
            await options.customJwtCheck({ header, payload, jwk });
        }
    }
    catch (err) {
        if (options.includeRawJwtInErrors && err instanceof error_1.JwtInvalidClaimError) {
            throw err.withRawJwt(decomposedJwt);
        }
        throw err;
    }
    return payload;
}
/**
 * Verify a JWT synchronously, using a JWKS or JWK that has already been fetched
 *
 * @param jwt The JWT
 * @param jwkOrJwks The JWKS that includes the right JWK (indexed by kid). Alternatively, provide the right JWK directly
 * @param options Verification options
 * @param jwkToKeyObjectTransformer A function that can transform a JWK into a Node.js native key object
 * @returns The (JSON parsed) payload of the JWT––if the JWT is valid, otherwise an error is thrown
 */
function verifyJwtSync(jwt, jwkOrJwks, options, jwkToKeyObjectTransformer) {
    return verifyDecomposedJwtSync(jwt_1.decomposeJwt(jwt), jwkOrJwks, options, jwkToKeyObjectTransformer);
}
exports.verifyJwtSync = verifyJwtSync;
/**
 * Verify (synchronously) a JWT that is already decomposed (by function `decomposeJwt`)
 *
 * @param decomposedJwt The decomposed JWT
 * @param jwkOrJwks The JWKS that includes the right JWK (indexed by kid). Alternatively, provide the right JWK directly
 * @param options Verification options
 * @param jwkToKeyObjectTransformer A function that can transform a JWK into a Node.js native key object
 * @returns The (JSON parsed) payload of the JWT––if the JWT is valid, otherwise an error is thrown
 */
function verifyDecomposedJwtSync(decomposedJwt, jwkOrJwks, options, jwkToKeyObjectTransformer) {
    const { header, headerB64, payload, payloadB64, signatureB64 } = decomposedJwt;
    let jwk;
    if (jwk_1.isJwk(jwkOrJwks)) {
        jwk = jwkOrJwks;
    }
    else if (jwk_1.isJwks(jwkOrJwks)) {
        const locatedJwk = jwkOrJwks.keys.find((key) => key.kid === header.kid);
        if (!locatedJwk) {
            throw new error_1.KidNotFoundInJwksError(`JWK for kid ${header.kid} not found in the JWKS`);
        }
        jwk = locatedJwk;
    }
    else {
        throw new error_1.ParameterValidationError([
            `Expected a valid JWK or JWKS (parsed as JavaScript object), but received: ${jwkOrJwks}.`,
            "If you're passing a JWKS URI, use the async verify() method instead, it will download and parse the JWKS for you",
        ].join());
    }
    verifySignatureAgainstJwk(header, headerB64, payload, payloadB64, signatureB64, jwk, jwkToKeyObjectTransformer);
    try {
        jwt_1.validateJwtFields(payload, options);
        if (options.customJwtCheck) {
            const res = options.customJwtCheck({ header, payload, jwk });
            assert_1.assertIsNotPromise(res, () => new error_1.ParameterValidationError('Custom JWT checks must be synchronous but a promise was returned'));
        }
    }
    catch (err) {
        if (options.includeRawJwtInErrors && err instanceof error_1.JwtInvalidClaimError) {
            throw err.withRawJwt(decomposedJwt);
        }
        throw err;
    }
    return payload;
}
/**
 * Abstract class representing a verifier for JWTs signed with RSA (e.g. RS256, RS384, RS512)
 *
 * A class is used, because there is state:
 * - The JWKS is fetched (downloaded) from the JWKS URI and cached in memory
 * - Verification properties at verifier level, are used as default options for individual verify calls
 *
 * When instantiating this class, relevant type parameters should be provided, for your concrete case:
 * @param StillToProvide The verification options that you want callers of verify to provide on individual verify calls
 * @param SpecificVerifyProperties The verification options that you'll use
 * @param IssuerConfig The issuer config that you'll use (config options are used as default verification options)
 * @param MultiIssuer Verify multiple issuers (true) or just a single one (false)
 */
class JwtRsaVerifierBase {
    constructor(verifyProperties, jwksCache = new jwk_1.SimpleJwksCache()) {
        this.jwksCache = jwksCache;
        this.issuersConfig = new Map();
        this.publicKeyCache = new KeyObjectCache();
        if (Array.isArray(verifyProperties)) {
            if (!verifyProperties.length) {
                throw new error_1.ParameterValidationError('Provide at least one issuer configuration');
            }
            for (const prop of verifyProperties) {
                if (this.issuersConfig.has(prop.issuer)) {
                    throw new error_1.ParameterValidationError(`issuer ${prop.issuer} supplied multiple times`);
                }
                this.issuersConfig.set(prop.issuer, this.withJwksUri(prop));
            }
        }
        else {
            this.issuersConfig.set(verifyProperties.issuer, this.withJwksUri(verifyProperties));
        }
    }
    get expectedIssuers() {
        return Array.from(this.issuersConfig.keys());
    }
    getIssuerConfig(issuer) {
        if (!issuer) {
            if (this.issuersConfig.size !== 1) {
                throw new error_1.ParameterValidationError('issuer must be provided');
            }
            issuer = this.issuersConfig.keys().next().value;
        }
        const config = this.issuersConfig.get(issuer);
        if (!config) {
            throw new error_1.ParameterValidationError(`issuer not configured: ${issuer}`);
        }
        return config;
    }
    /**
     * This method loads a JWKS that you provide, into the JWKS cache, so that it is
     * available for JWT verification. Use this method to speed up the first JWT verification
     * (when the JWKS would otherwise have to be downloaded from the JWKS uri), or to provide the JWKS
     * in case the JwtVerifier does not have internet access to download the JWKS
     *
     * @param jwksThe JWKS
     * @param issuer The issuer for which you want to cache the JWKS
     *  Supply this field, if you instantiated the JwtVerifier with multiple issuers
     * @returns void
     */
    // public cacheJwks(
    //   ...[jwks, issuer]: MultiIssuer extends false
    //     ? [jwks: Jwks, issuer?: string]
    //     : [jwks: Jwks, issuer: string]
    // ): void {
    //   const issuerConfig = this.getIssuerConfig(issuer);
    //   this.jwksCache.addJwks(issuerConfig.jwksUri, jwks);
    //   this.publicKeyCache.clearCache(issuerConfig.issuer);
    // }
    /**
     * Hydrate the JWKS cache for (all of) the configured issuer(s).
     * This will fetch and cache the latest and greatest JWKS for concerned issuer(s).
     *
     * @param issuer The issuer to fetch the JWKS for
     * @returns void
     */
    async hydrate() {
        const jwksFetches = this.expectedIssuers
            .map((issuer) => this.getIssuerConfig(issuer).jwksUri)
            .map((jwksUri) => this.jwksCache.getJwks(jwksUri));
        await Promise.all(jwksFetches);
    }
    /**
     * Verify (synchronously) a JWT that is signed using RS256 / RS384 / RS512.
     *
     * @param jwt The JWT, as string
     * @param props Verification properties
     * @returns The payload of the JWT––if the JWT is valid, otherwise an error is thrown
     */
    verifySync(...[jwt, properties]) {
        const { decomposedJwt, jwksUri, verifyProperties } = this.getVerifyParameters(jwt, properties);
        return this.verifyDecomposedJwtSync(decomposedJwt, jwksUri, verifyProperties);
    }
    /**
     * Verify (synchronously) an already decomposed JWT, that is signed using RS256 / RS384 / RS512.
     *
     * @param decomposedJwt The decomposed Jwt
     * @param jwk The JWK to verify the JWTs signature with
     * @param verifyProperties The properties to use for verification
     * @returns The payload of the JWT––if the JWT is valid, otherwise an error is thrown
     */
    verifyDecomposedJwtSync(decomposedJwt, jwksUri, verifyProperties) {
        const jwk = this.jwksCache.getCachedJwk(jwksUri, decomposedJwt);
        return verifyDecomposedJwtSync(decomposedJwt, jwk, verifyProperties, this.publicKeyCache.transformJwkToKeyObject.bind(this.publicKeyCache));
    }
    /**
     * Verify (asynchronously) a JWT that is signed using RS256 / RS384 / RS512.
     * This call is asynchronous, and the JWKS will be fetched from the JWKS uri,
     * in case it is not yet available in the cache.
     *
     * @param jwt The JWT, as string
     * @param props Verification properties
     * @returns Promise that resolves to the payload of the JWT––if the JWT is valid, otherwise the promise rejects
     */
    async verify(...[jwt, properties]) {
        const { decomposedJwt, jwksUri, verifyProperties } = this.getVerifyParameters(jwt, properties);
        return this.verifyDecomposedJwt(decomposedJwt, jwksUri, verifyProperties);
    }
    /**
     * Verify (asynchronously) an already decomposed JWT, that is signed using RS256 / RS384 / RS512.
     *
     * @param decomposedJwt The decomposed Jwt
     * @param jwk The JWK to verify the JWTs signature with
     * @param verifyProperties The properties to use for verification
     * @returns The payload of the JWT––if the JWT is valid, otherwise an error is thrown
     */
    verifyDecomposedJwt(decomposedJwt, jwksUri, verifyProperties) {
        return verifyDecomposedJwt(decomposedJwt, jwksUri, verifyProperties, this.jwksCache.getJwk.bind(this.jwksCache), this.publicKeyCache.transformJwkToKeyObject.bind(this.publicKeyCache));
    }
    /**
     * Get the verification parameters to use, by merging the issuer configuration,
     * with the overriding properties that are now provided
     *
     * @param jwt: the JWT that is going to be verified
     * @param verifyProperties: the overriding properties, that override the issuer configuration
     * @returns The merged verification parameters
     */
    getVerifyParameters(jwt, verifyProperties) {
        const decomposedJwt = jwt_1.decomposeJwt(jwt);
        assert_1.assertStringArrayContainsString('Issuer', decomposedJwt.payload.iss, this.expectedIssuers, error_1.JwtInvalidIssuerError);
        const issuerConfig = this.getIssuerConfig(decomposedJwt.payload.iss);
        return {
            decomposedJwt,
            jwksUri: issuerConfig.jwksUri,
            verifyProperties: {
                ...issuerConfig,
                ...verifyProperties,
            },
        };
    }
    /**
     * Get issuer config with JWKS URI, by adding a default JWKS URI if needed
     *
     * @param config: the issuer config.
     * @returns The config with JWKS URI
     */
    withJwksUri(config) {
        if (config.jwksUri) {
            return config;
        }
        const issuerUri = new url_1.URL(config.issuer);
        return {
            jwksUri: new url_1.URL(path_1.join(issuerUri.pathname, '/.well-known/jwks.json'), config.issuer).href,
            ...config,
        };
    }
}
exports.JwtRsaVerifierBase = JwtRsaVerifierBase;
/**
 * Class representing a verifier for JWTs signed with RSA (e.g. RS256 / RS384 / RS512)
 */
class JwtRsaVerifier extends JwtRsaVerifierBase {
    // eslint-disable-next-line @typescript-eslint/explicit-module-boundary-types
    static create(verifyProperties, additionalProperties) {
        return new this(verifyProperties, additionalProperties === null || additionalProperties === void 0 ? void 0 : additionalProperties.jwksCache);
    }
}
exports.JwtRsaVerifier = JwtRsaVerifier;
/**
 * Transform the JWK into an RSA public key in Node.js native key object format
 *
 * @param jwk: the JWK
 * @returns the RSA public key in Node.js native key object format
 */
exports.transformJwkToKeyObject = (jwk) => crypto_1.createPublicKey({
    key: asn1_1.constructPublicKeyInDerFormat(Buffer.from(jwk.n, 'base64'), Buffer.from(jwk.e, 'base64')),
    format: 'der',
    type: 'spki',
});
/**
 * Class representing a cache of RSA public keys in Node.js native key object format
 *
 * Because it takes a bit of compute time to turn a JWK into Node.js native key object format,
 * we want to cache this computation.
 */
class KeyObjectCache {
    constructor(jwkToKeyObjectTransformer = exports.transformJwkToKeyObject) {
        this.jwkToKeyObjectTransformer = jwkToKeyObjectTransformer;
        this.publicKeys = new Map();
    }
    /**
     * Transform the JWK into an RSA public key in Node.js native key object format.
     * If the transformed JWK is already in the cache, it is returned from the cache instead.
     * The cache keys are: issuer, JWK kid (key id)
     *
     * @param jwk: the JWK
     * @param issuer: the issuer that uses the JWK for signing JWTs
     * @returns the RSA public key in Node.js native key object format
     */
    transformJwkToKeyObject(jwk, issuer) {
        var _a;
        if (!issuer) {
            return this.jwkToKeyObjectTransformer(jwk);
        }
        const cachedPublicKey = (_a = this.publicKeys.get(issuer)) === null || _a === void 0 ? void 0 : _a.get(jwk.kid);
        if (cachedPublicKey) {
            return cachedPublicKey;
        }
        const publicKey = this.jwkToKeyObjectTransformer(jwk);
        const cachedIssuer = this.publicKeys.get(issuer);
        if (cachedIssuer) {
            cachedIssuer.set(jwk.kid, publicKey);
        }
        else {
            this.publicKeys.set(issuer, new Map([[jwk.kid, publicKey]]));
        }
        return publicKey;
    }
    clearCache(issuer) {
        this.publicKeys.delete(issuer);
    }
}
exports.KeyObjectCache = KeyObjectCache;
//# sourceMappingURL=data:application/json;base64,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