"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateChangelog = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const logging = require("../logging");
const utils = require("../util");
/**
 * Prepends input changelog entry onto project-level changelog.
 *
 * Currently assumes a headerless changelog formatted according to
 * [conventional-changelog](https://github.com/conventional-changelog/conventional-changelog)
 * rules.
 *
 * @param cwd working directory (git repository)
 * @param options options
 */
async function updateChangelog(cwd, options) {
    const inputChangelog = path_1.join(cwd, options.inputChangelog);
    const outputChangelog = path_1.join(cwd, options.outputChangelog);
    const versionFile = path_1.join(cwd, options.versionFile);
    let version = (await utils.tryReadFile(versionFile)).trim();
    if (!version) {
        throw new Error(`Unable to determine version from ${versionFile}. Cannot proceed with changelog update. Did you run 'bump'?`);
    }
    const inputChangelogContent = await fs_extra_1.readFile(inputChangelog, "utf-8");
    const changelogVersionSearchPattern = `${version}`;
    if (!inputChangelogContent.includes(changelogVersionSearchPattern)) {
        throw new Error(`Supplied version ${version} was not found in input changelog. You may want to check it's content.`);
    }
    const outputChangelogContent = await fs_extra_1.readFile(outputChangelog, {
        encoding: "utf-8",
        flag: "a+",
    });
    if (outputChangelogContent.indexOf(changelogVersionSearchPattern) > -1) {
        logging.info(`Changelog already contains an entry for ${version}. Skipping changelog update.`);
        return;
    }
    const newChangelog = inputChangelogContent.trimEnd() +
        "\n\n" +
        outputChangelogContent.trimStart();
    await fs_extra_1.writeFile(outputChangelog, newChangelog);
    utils.exec(`git add ${outputChangelog} && git commit -m "chore(release): ${version}"`, { cwd });
}
exports.updateChangelog = updateChangelog;
//# sourceMappingURL=data:application/json;base64,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