"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaRuntime = exports.LambdaFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const case_1 = require("case");
const internal_1 = require("./internal");
const component_1 = require("../component");
const javascript_1 = require("../javascript");
const source_code_1 = require("../source-code");
/**
 * Generates a pre-bundled AWS Lambda function construct from handler code.
 *
 * To use this, create an AWS Lambda handler file under your source tree with
 * the `.lambda.ts` extension and add a `LambdaFunction` component to your
 * typescript project pointing to this entrypoint.
 *
 * This will add a task to your "compile" step which will use `esbuild` to
 * bundle the handler code into the build directory. It will also generate a
 * file `src/foo-function.ts` with a custom AWS construct called `FooFunction`
 * which extends `@aws-cdk/aws-lambda.Function` which is bound to the bundled
 * handle through an asset.
 *
 * @example
 *
 * new LambdaFunction(myProject, {
 *   srcdir: myProject.srcdir,
 *   entrypoint: 'src/foo.lambda.ts',
 * });
 */
class LambdaFunction extends component_1.Component {
    /**
     * Defines a pre-bundled AWS Lambda function construct from handler code.
     *
     * @param project The project to use
     * @param options Options
     */
    constructor(project, options) {
        super(project);
        const cdkDeps = options.cdkDeps;
        const bundler = javascript_1.Bundler.of(project);
        if (!bundler) {
            throw new Error("No bundler found. Please add a Bundler component to your project.");
        }
        const runtime = options.runtime ?? LambdaRuntime.NODEJS_14_X;
        // allow Lambda handler code to import dev-deps since they are only needed
        // during bundling
        const eslint = javascript_1.Eslint.of(project);
        eslint?.allowDevDeps(options.entrypoint);
        const entrypoint = options.entrypoint;
        if (!entrypoint.endsWith(internal_1.TYPESCRIPT_LAMBDA_EXT) &&
            !entrypoint.endsWith(internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT)) {
            throw new Error(`${entrypoint} must have a ${internal_1.TYPESCRIPT_LAMBDA_EXT} or ${internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT} extension`);
        }
        const basePath = path.posix.join(path.dirname(entrypoint), path.basename(entrypoint, options.edgeLambda ? internal_1.TYPESCRIPT_EDGE_LAMBDA_EXT : internal_1.TYPESCRIPT_LAMBDA_EXT));
        const constructFile = options.constructFile ?? `${basePath}-function.ts`;
        if (path.extname(constructFile) !== ".ts") {
            throw new Error(`Construct file name "${constructFile}" must have a .ts extension`);
        }
        // type names
        const constructName = options.constructName ?? case_1.pascal(path.basename(basePath)) + "Function";
        const propsType = `${constructName}Props`;
        const bundle = bundler.addBundle(entrypoint, {
            target: runtime.esbuildTarget,
            platform: runtime.esbuildPlatform,
            externals: runtime.defaultExternals,
            ...options.bundlingOptions,
            tsconfigPath: project?.tsconfigDev?.fileName,
        });
        // calculate the relative path between the directory containing the
        // generated construct source file to the directory containing the bundle
        // index.js by resolving them as absolute paths first.
        // e.g:
        //  - outfileAbs => `/project-outdir/assets/foo/bar/baz/foo-function/index.js`
        //  - constructAbs => `/project-outdir/src/foo/bar/baz/foo-function.ts`
        const outfileAbs = path.join(project.outdir, bundle.outfile);
        const constructAbs = path.join(project.outdir, constructFile);
        const relativeOutfile = path.relative(path.dirname(constructAbs), path.dirname(outfileAbs));
        const src = new source_code_1.SourceCode(project, constructFile);
        if (src.marker) {
            src.line(`// ${src.marker}`);
        }
        src.line("import * as path from 'path';");
        if (cdkDeps.cdkMajorVersion === 1) {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from '@aws-cdk/aws-cloudfront';");
                cdkDeps.addV1Dependencies("@aws-cdk/aws-cloudfront");
            }
            src.line("import * as lambda from '@aws-cdk/aws-lambda';");
            src.line("import { Construct } from '@aws-cdk/core';");
            cdkDeps.addV1Dependencies("@aws-cdk/aws-lambda");
            cdkDeps.addV1Dependencies("@aws-cdk/core");
        }
        else {
            if (options.edgeLambda) {
                src.line("import * as cloudfront from 'aws-cdk-lib/aws-cloudfront';");
            }
            src.line("import * as lambda from 'aws-cdk-lib/aws-lambda';");
            src.line("import { Construct } from 'constructs';");
        }
        src.line();
        src.line("/**");
        src.line(` * Props for ${constructName}`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export interface ${propsType} extends cloudfront.experimental.EdgeFunctionProps {`);
        }
        else {
            src.open(`export interface ${propsType} extends lambda.FunctionOptions {`);
        }
        src.close("}");
        src.line();
        src.line("/**");
        src.line(` * An AWS Lambda function which executes ${internal_1.convertToPosixPath(basePath)}.`);
        src.line(" */");
        if (options.edgeLambda) {
            src.open(`export class ${constructName} extends cloudfront.experimental.EdgeFunction {`);
        }
        else {
            src.open(`export class ${constructName} extends lambda.Function {`);
        }
        src.open(`constructor(scope: Construct, id: string, props?: ${propsType}) {`);
        src.open("super(scope, id, {");
        src.line(`description: '${internal_1.convertToPosixPath(entrypoint)}',`);
        src.line("...props,");
        src.line(`runtime: new lambda.Runtime('${runtime.functionRuntime}', lambda.RuntimeFamily.NODEJS),`);
        src.line("handler: 'index.handler',");
        src.line(`code: lambda.Code.fromAsset(path.join(__dirname, '${internal_1.convertToPosixPath(relativeOutfile)}')),`);
        src.close("});");
        if ((options.awsSdkConnectionReuse ?? true) && !options.edgeLambda) {
            src.line("this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });");
        }
        src.close("}");
        src.close("}");
        this.project.logger.verbose(`${basePath}: construct "${constructName}" generated under "${constructFile}"`);
        this.project.logger.verbose(`${basePath}: bundle task "${bundle.bundleTask.name}"`);
        if (bundle.watchTask) {
            this.project.logger.verbose(`${basePath}: bundle watch task "${bundle.watchTask.name}"`);
        }
    }
}
exports.LambdaFunction = LambdaFunction;
_a = JSII_RTTI_SYMBOL_1;
LambdaFunction[_a] = { fqn: "projen.awscdk.LambdaFunction", version: "0.67.87" };
/**
 * The runtime for the AWS Lambda function.
 */
class LambdaRuntime {
    constructor(
    /**
     * The Node.js runtime to use
     */
    functionRuntime, 
    /**
     * The esbuild setting to use.
     */
    esbuildTarget, 
    /**
     * Options for this runtime.
     */
    options) {
        this.functionRuntime = functionRuntime;
        this.esbuildTarget = esbuildTarget;
        this.esbuildPlatform = "node";
        this.defaultExternals = options?.defaultExternals ?? ["@aws-sdk/*"];
    }
}
exports.LambdaRuntime = LambdaRuntime;
_b = JSII_RTTI_SYMBOL_1;
LambdaRuntime[_b] = { fqn: "projen.awscdk.LambdaRuntime", version: "0.67.87" };
/**
 * Node.js 10.x
 * @deprecated NodeJS10 has been deprecated
 */
LambdaRuntime.NODEJS_10_X = new LambdaRuntime("nodejs10.x", "node10", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 12.x
 */
LambdaRuntime.NODEJS_12_X = new LambdaRuntime("nodejs12.x", "node12", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 14.x
 */
LambdaRuntime.NODEJS_14_X = new LambdaRuntime("nodejs14.x", "node14", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 16.x
 */
LambdaRuntime.NODEJS_16_X = new LambdaRuntime("nodejs16.x", "node16", { defaultExternals: ["aws-sdk"] });
/**
 * Node.js 18.x
 */
LambdaRuntime.NODEJS_18_X = new LambdaRuntime("nodejs18.x", "node18");
//# sourceMappingURL=data:application/json;base64,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