// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import React, { useMemo } from 'react';
import styles from './styles.css.js';
import { isSameMonth, isAfter, isBefore, isSameDay, addWeeks, addDays, isLastDayOfMonth, getDaysInMonth, isToday, } from 'date-fns';
import { getCalendarMonth } from 'mnth';
import { getDateLabel, renderDayName } from '../../../calendar/utils/intl';
import clsx from 'clsx';
import { formatDate } from '../../../internal/utils/date-time';
import useFocusVisible from '../../../internal/hooks/focus-visible/index.js';
import ScreenreaderOnly from '../../../internal/components/screenreader-only/index.js';
export function Grid({ baseDate, selectedStartDate, selectedEndDate, rangeStartDate, rangeEndDate, focusedDate, focusedDateRef, onSelectDate, onGridKeyDownHandler, onFocusedDateChange, isDateEnabled, locale, startOfWeek, todayAriaLabel, ariaLabelledby, className, }) {
    const baseDateTime = baseDate === null || baseDate === void 0 ? void 0 : baseDate.getTime();
    // `baseDateTime` is used as a more stable replacement for baseDate
    const weeks = useMemo(() => getCalendarMonth(baseDate, { firstDayOfWeek: startOfWeek }), 
    // eslint-disable-next-line react-hooks/exhaustive-deps
    [baseDateTime, startOfWeek]);
    const weekdays = weeks[0].map(date => date.getDay());
    const focusVisible = useFocusVisible();
    return (React.createElement("table", { role: "grid", "aria-labelledby": ariaLabelledby, className: clsx(styles.grid, className) },
        React.createElement("thead", null,
            React.createElement("tr", null, weekdays.map(dayIndex => (React.createElement("th", { key: dayIndex, scope: "col", className: clsx(styles['grid-cell'], styles['day-header']) },
                React.createElement("span", { "aria-hidden": "true" }, renderDayName(locale, dayIndex, 'short')),
                React.createElement(ScreenreaderOnly, null, renderDayName(locale, dayIndex, 'long'))))))),
        React.createElement("tbody", { onKeyDown: onGridKeyDownHandler }, weeks.map((week, weekIndex) => {
            return (React.createElement("tr", { key: weekIndex, className: styles.week }, week.map((date, dateIndex) => {
                const isStartDate = !!selectedStartDate && isSameDay(date, selectedStartDate);
                const isEndDate = !!selectedEndDate && isSameDay(date, selectedEndDate);
                const isSelected = isStartDate || isEndDate;
                const isRangeStartDate = !!rangeStartDate && isSameDay(date, rangeStartDate);
                const isRangeEndDate = !!rangeEndDate && isSameDay(date, rangeEndDate);
                const isFocused = !!focusedDate && isSameDay(date, focusedDate) && isSameMonth(date, baseDate);
                const dateIsInRange = isStartDate || isEndDate || isInRange(date, rangeStartDate, rangeEndDate);
                const inRangeStartWeek = rangeStartDate && isInRange(date, rangeStartDate, addDays(addWeeks(rangeStartDate, 1), -1));
                const inRangeEndWeek = rangeEndDate && isInRange(date, rangeEndDate, addDays(addWeeks(rangeEndDate, -1), 1));
                const onlyOneSelected = !!rangeStartDate && !!rangeEndDate
                    ? isSameDay(rangeStartDate, rangeEndDate)
                    : !selectedStartDate || !selectedEndDate;
                const isEnabled = !isDateEnabled || isDateEnabled(date);
                const isFocusable = isFocused && isEnabled;
                const baseClasses = {
                    [styles.day]: true,
                    [styles['grid-cell']]: true,
                    [styles['in-first-row']]: weekIndex === 0,
                    [styles['in-first-column']]: dateIndex === 0,
                };
                if (!isSameMonth(date, baseDate)) {
                    return (React.createElement("td", { key: `${weekIndex}:${dateIndex}`, ref: isFocused ? focusedDateRef : undefined, className: clsx(baseClasses, {
                            [styles['in-previous-month']]: isBefore(date, baseDate),
                            [styles['last-day-of-month']]: isLastDayOfMonth(date),
                            [styles['in-next-month']]: isAfter(date, baseDate),
                        }) }));
                }
                const handlers = {};
                if (isEnabled) {
                    handlers.onClick = () => onSelectDate(date);
                    handlers.onFocus = () => onFocusedDateChange(date);
                }
                // Can't be focused.
                let tabIndex = undefined;
                if (isFocusable && isEnabled) {
                    // Next focus target.
                    tabIndex = 0;
                }
                else if (isEnabled) {
                    // Can be focused programmatically.
                    tabIndex = -1;
                }
                // Screen-reader announcement for the focused day.
                let dayAnnouncement = getDateLabel(locale, date, 'short');
                if (isToday(date)) {
                    dayAnnouncement += '. ' + todayAriaLabel;
                }
                return (React.createElement("td", Object.assign({ ref: isFocused ? focusedDateRef : undefined, key: `${weekIndex}:${dateIndex}`, className: clsx(baseClasses, {
                        [styles['in-current-month']]: isSameMonth(date, baseDate),
                        [styles.enabled]: isEnabled,
                        [styles.selected]: isSelected,
                        [styles['start-date']]: isStartDate,
                        [styles['end-date']]: isEndDate,
                        [styles['range-start-date']]: isRangeStartDate,
                        [styles['range-end-date']]: isRangeEndDate,
                        [styles['no-range']]: isSelected && onlyOneSelected,
                        [styles['in-range']]: dateIsInRange,
                        [styles['in-range-border-top']]: !!inRangeStartWeek || date.getDate() <= 7,
                        [styles['in-range-border-bottom']]: !!inRangeEndWeek || date.getDate() > getDaysInMonth(date) - 7,
                        [styles['in-range-border-left']]: dateIndex === 0 || date.getDate() === 1 || isRangeStartDate,
                        [styles['in-range-border-right']]: dateIndex === week.length - 1 || isLastDayOfMonth(date) || isRangeEndDate,
                        [styles.today]: isToday(date),
                    }), "aria-selected": isEnabled ? isSelected || dateIsInRange : undefined, "aria-current": isToday(date) ? 'date' : undefined, "data-date": formatDate(date), "aria-disabled": !isEnabled, tabIndex: tabIndex }, handlers, focusVisible),
                    React.createElement("span", { className: styles['day-inner'], "aria-hidden": "true" }, date.getDate()),
                    React.createElement(ScreenreaderOnly, null, dayAnnouncement)));
            })));
        }))));
}
function isInRange(date, dateOne, dateTwo) {
    if (!dateOne || !dateTwo || isSameDay(dateOne, dateTwo)) {
        return false;
    }
    const inRange = (isAfter(date, dateOne) && isBefore(date, dateTwo)) || (isAfter(date, dateTwo) && isBefore(date, dateOne));
    return inRange || isSameDay(date, dateOne) || isSameDay(date, dateTwo);
}
//# sourceMappingURL=grid.js.map