# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['inetbox']

package_data = \
{'': ['*']}

install_requires = \
['bitstruct>=8.15.1,<9.0.0', 'miqro>=1.2.0', 'pyserial>=3.5,<4.0']

extras_require = \
{'truma_service': ['paho-mqtt>=1.6.1,<2.0.0']}

entry_points = \
{'console_scripts': ['truma_service = inetbox:truma_service.run']}

setup_kwargs = {
    'name': 'inetbox-py',
    'version': '0.2',
    'description': '',
    'long_description': '# inetbox.py\n\nThis is a software implementation of a Truma iNet box, a device for controlling mobile heater and AC units by Truma and Alde.\n\nThis software is not provided, endorsed, supported, or sponsored by Truma or Alde. It may or may not be working with their products. Please read the [license](./LICENSE) file, in particular:\n\nIN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING\nWILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES AND/OR CONVEYS\nTHE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES, INCLUDING ANY\nGENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE\nUSE OR INABILITY TO USE THE PROGRAM.\n\nThat said, it is working for me, and I hope it will work for you, too.\n\n## Hardware Requirements\n\nThis has been tested with a Truma Combi 4 and the CP Plus control panel (inet ready). I don\'t see why this wouldn\'t be working with the Combi 6 and E models as well.\n\nThe software runs on a Raspberry Pi, any newer model should do. This could also be ported to a Pi Pico Microcontroller, but I haven\'t done that yet.\n\nYou need a [LIN to UART Transceiver](https://amzn.to/3E1qITr) (Affiliate Link!) for connecting the Raspberry Pi to the LIN bus. On the transceiver module, the connections are as follows:\n\n * **GND** (any of the two) connects to a ground connection - e.g. on the power supply.\n * **12V** connects to a 12V power supply that also powers the Combi and CP Plus.\n * **TX** connects to pin 15 on the Raspberry Pi.\n * **RX** connects to pin 14 on the Raspberry Pi (14/15 might be the other way round, not sure).\n * **LIN** connects to an RJ12 connector (alias 6P6C, the one with 6 pins) going into any port on the Truma Combi heating, or using a [splitter module](https://amzn.to/3dL4bzT) (Affiliate Link!) into the existing connection between Combi and the control panel. Use standard RJ12 cables for the connection. The relevant pin is highlighted in this picture: \n\n![LIN Pin](./docs/pinout.jpg)\n\nThe other pins (**INH**, **SLP**, second **GND**) are not used.\n\n## Installation\n\nYou first need to **enable UART** on the Pi. For this, follow the steps listed under "Configure UART on Raspberry Pi" on [this website](https://www.electronicwings.com/raspberry-pi/raspberry-pi-uart-communication-using-python-and-c#header2) until the step to reboot the Pi.\n\nThen install the software:\n\n * If `pip3` is not installed, run `sudo apt install python3-pip` first.\n * Then run `pip3 install inetbox_py[truma_service]` if you want to install everything.\n * Alternatively, run `pip3 install inetbox_py` to just install the library in case you want to develop your own code using it. \n\n\n**After you have tested that the software works for you**, to install a system service using this software, run **as root**:\n\n```bash\npip3 install inetbox_py[truma_service]\ntruma_service --install\nsystemctl enable miqro_truma\nsystemctl start miqro_truma\n```\n\n## Using the MQTT Service\n\nIn the following, the MQTT service will be explained. You need an MQTT broker running (e.g. [Mosquitto](https://mosquitto.org/)) for this to work and you should be familiar with basic MQTT concepts.\n\n### Configuration\n\nDefine the MQTT broker by creating the file `/etc/miqro.yml` with the broker settings as follows (adapt as needed):\n\n```yaml\nbroker:\n  host: localhost\n  port: 1883\n  keepalive: 60\n  \nlog_level: INFO\n\nservices: {}\n\n```\nIf you need to configure MQTT authentication and/or TLS settings, follow [these](https://github.com/danielfett/miqro#authentication-and-tls) instructions.\n\n\nBy default, the application uses the serial port `/dev/serial0`. This should work fine on all Raspberry Pi versions, but in case you want to use a different port, you need to configure it in `/etc/miqro.yml` by adding a `services` section for the `truma` service like this (replacing the empty `services: {}` section above):\n\n\n```yaml\nservices:\n  truma:\n    serial_port: /dev/ttyS0\n```\n\nTo run the service:\n```\ntruma_service\n```\n\n\n### Initializing\n\nThis script plays the role of the inet box. You might need to initialize CP Plus again to make the fake inet box known to the system. This is an easy step that can safely be repeated (no settings are lost): After starting the software, go to the settings menu on the CP Plus and select "PR SET". The display will show "Init..." and after a few seconds, the initialization will be completed.\n\n### Does it work?\n\nIf the connection works, it can take up to a minute to receive the first status data. Everything is fine if you see the last line shown in the following in the service output:\n\n```\n2022-10-02 14:20:38,787  truma.main  WARNING    Service configuration for truma not found in \'services\' section of configuration file /etc/miqro.yml. Using empty configuration.\n2022-10-02 14:20:38,788  truma.main  INFO       started\n2022-10-02 14:20:38,790  truma.main  INFO       Using serial device /dev/serial0\n2022-10-02 14:20:38,792  truma.main  INFO       Loop stats:\n2022-10-02 14:20:38,793  truma.main  INFO        - Loop(_update_online_status) called 0 times, average duration 0.0s, load=0%\n2022-10-02 14:20:38,794  truma.main  INFO        - Loop(send_status) called 0 times, average duration 0.0s, load=0%\n2022-10-02 14:20:38,806  truma.main  INFO       MQTT connected, client=<paho.mqtt.client.Client object at 0xb5ee3090>, userdata=None, rc=0\n2022-10-02 14:20:38,806  truma.main  INFO       Subscribing to ...\n2022-10-02 14:20:38,807  truma.main  INFO         - service/truma/set/#\n2022-10-02 14:20:38,807  truma.main  INFO         - service/truma/enabled\n2022-10-02 14:21:27,234  inet-lin  INFO         Received status data from cp plus\n```\n\n### Debugging\n\nIf you want to enable debugging, you can set the environment variables `DEBUG_LIN=1`, `DEBUG_PROTOCOL=1`, and `DEBUG_APP=1`, to debug the LIN bus (byte level communication), the protocol layer (handing LIN bus specifics), and the application layer (handling the actual data), respectively.\n\nExample:\n\n`DEBUG_LIN=1 truma_service`\n\n### MQTT Topics\n\nWhen started, the service will connect to the LIN bus and publish any status updates acquired from there. When you send a command to modify a setting (e.g., to turn on the heating), the service will send the command to the LIN bus and publish the new status once the setting has been confirmed.\n\n#### MQTT topics for receiving status\n\n`service/truma/error` - some error messages are published here\n\n`service/truma/display_status/#` - frequent updates from CP Plus, similar to what is shown on the display. Note that not all values have been decoded yet.\n\n`service/truma/control_status/#` - less frequent updates, but includes values that can be modified. These are the values that would otherwise be available in the Truma inet app.\n\n#### Changing settings\n\nIn general, publish a message to `service/truma/set/<setting>` (where `<setting>` is one of the settings published in `service/truma/control_status/#`) with the value you want to set. After restarting the service, wait a minute or so until the first set of values has been published before changing settings.\n\nFor example:\n\n```bash\nmosquitto_pub -t \'service/truma/set/target_temp_water\' -m \'40\'\n```\nor\n\n```bash\nmosquitto_pub -t \'service/truma/set/target_temp_room\' -m \'10\'; mosquitto_pub -t \'service/truma/set/heating_mode\' -m \'eco\'\n```\n\nThere are some specifics for certain settings:\n\n * `target_temp_room` and `heating_mode` must both be enabled for the heating to work. It\'s best to set both together as in the example above.\n * `target_temp_room` can be set to 0 to turn off the heating, and 5-30 degrees otherwise.\n * `heating_mode` can be set to `off`, `eco` and `high` and defines the fan intensity for room heating.\n * `target_temp_water` must be set to one of `0` (off), `40` (equivalent to selecting \'eco\' on the display), `60` (\'high\'), or `200` (boost)\n * `energy_mix` and `el_power_level` should be set together.\n * `energy_mix` can be one of `none`/`gas`/`electricity`/`mix`\n * `el_power_level` can be set to `0`/`900`/`1800` when electric heating or mix is enabled\n\n## Reading Log Files\n\nThe tool [bin/read-logfile.py](bin/read-logfile.py) can be used to read log files created by, for example, a LIN bus logger. The file will be interpreted and the data will be printed to stdout.\n\nUsage: `python3 bin/read-logfile.py logfile.log`\n\nThe log file is expected to have the following format:\n\n```\n<timestamp> <databyte[0]> <databyte[1]> <databyte[..]> <x> <y>\n```\n\nwhere `<x>` and `<y>` are ignored. This is an example file:\n\n```\n00000,025   61                                                        9634    Checksum \n00000,076   E2                                                        9615    Checksum \n00000,126   3C  01  06  B8  40  03  00  00  FF      FC     classic    9615     \n00000,177   7D                                                        9615    Checksum \n00000,228   D8                                                        9615    Checksum \n00000,279   3C  7F  06  B2  00  17  46  00  1F      4B     classic    9615     \n00000,329   7D                                                        9615    Checksum \n\n```\n\nEmpty lines are ignored.\n\nIf your file looks different, use `--first` to define the first data byte position (in the format above, 1), and `--last` to define the end of the data bytes (in the format above, -2). The values are interpreted as python slice indices, so you can use negative values to count from the end. \n\n## Internals\n\nWhat does this software do internally?\n\nThe software tries to emulate the workings of the Truma iNet box. It plays the role of a \'slave\' device on the LIN bus and listens for messages from the CP Plus. Selected messages are answered appropriately as to make the CP Plus think that the iNet box is present and working, and to send commands to the CP Plus.\n\nTwo specific LIN messages are directly used to communicate with the CP Plus:\n * **PID 0x18** - the first bit defines whether a command for changing settings is ready at the iNet box. If the first bit is `1`, the CP Plus will send a message on the transport layer (see below) to request the settings update. See `InetboxLINProtocol.answer_to_d8_message` in [inetbox/inetbox.py](inetbox/inetbox.py) for details.\n\n * **PIDs 0x20-0x22** - contain frequent status updates closely resembling what is shown on the CP Plus display. It is unclear to me if the original iNet box observes these messages at all. Settings cannot be changed via these messaged. See `InetboxApp.parse_*` in [inetbox/inetbox.py](inetbox/inetbox.py) for the implementation.\n\nThe settings transfer from the CP Plus to the iNet box and vice versa (for modifying settings) is done via status buffers exchanged via the LIN transport layer. The relevant parts of the transport layer are implemented in this software. Please see the LIN specification for details on this layer. The low-level communication is based on the messages with PIDs 0x3C and 0x3D. \n\nA number of different status buffer types exist both for reading values from CP Plus as well as for writing values to CP Plus. The buffers can be identified by two bytes that follow a 10-byte preamble that is always the same. For example:\n\n * **ID 0x14, 0x33**: Coming from the CP Plus, this buffer contains most settings for heating (like target water temperature, current temperature, etc.).\n * **ID 0x0C, 0x32**: This status buffer is similar to the above, but is sent from the iNet box to the CP Plus for changing settings.\n\nSee `InetboxApp.STATUS_*` in [inetbox/inetbox.py](inetbox/inetbox.py) for details on the known buffer types.\n\n## Acknowledgements\n\nThis project is based on the work of the [WomoLIN project](https://github.com/muccc/WomoLIN), especially the initial protocol decoding and the inet box log files.\n',
    'author': 'Daniel Fett',
    'author_email': 'fett@danielfett.de',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
