"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrganizationAccount = exports.OrganizationAccountProvider = exports.OrganizationOU = exports.OrganizationOUProvider = exports.processOUObj = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
;
;
/**
 * @function processOUObj
 * Function to process an OrgObject and create the corresponding AWS resources
 *
 * @param {Construct} this The construct resources will be added to.
 * @param {custom_resources.Provider} ouProvider The custom resource provider for managing OUs
 * @param {custom_resources.Provider} accountProvider The custom resource provider for managing accounts
 * @param {OUObject} obj The OrgObject to process.
 * @param {string | OrganizationOU} parent The parent of the OrgObject. This is either a string, like for the org root, or an OrganizationOU object from the same stack.
 */
function processOUObj(ouProvider, accountProvider, obj, parent) {
    var _e, _f;
    const parentStr = parent instanceof OrganizationOU ? parent.resource.ref : parent;
    let id = (_e = obj.id) !== null && _e !== void 0 ? _e : obj.properties.name;
    const ou = new OrganizationOU(this, id, {
        provider: ouProvider,
        parent: parentStr,
        name: obj.properties.name,
        importOnDuplicate: obj.properties.importOnDuplicate,
        allowRecreateOnUpdate: obj.properties.allowRecreateOnUpdate,
    });
    (_f = obj.accounts) === null || _f === void 0 ? void 0 : _f.forEach((account) => {
        new OrganizationAccount(this, `${account.name}-${account.email.replace(/[^a-zA-Z ]/g, '')}`, {
            provider: accountProvider,
            parent: ou,
            name: account.name,
            email: account.email,
            importOnDuplicate: account.importOnDuplicate,
            allowMove: account.allowMove,
            disableDelete: account.disableDelete,
        });
    });
    obj.children.forEach(child => {
        processOUObj.call(this, ouProvider, accountProvider, child, ou);
    });
}
exports.processOUObj = processOUObj;
/**
 * The provider for OU custom resources.
 *
 * This creates a lambda function that handles custom resource requests for creating/updating/deleting OUs.
 *
 * @stability stable
 */
class OrganizationOUProvider extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let role;
        if (!props.role) {
            role = new aws_cdk_lib_1.aws_iam.Role(this, 'role', {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            let policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(this, 'policy', {
                statements: [
                    new aws_cdk_lib_1.aws_iam.PolicyStatement({
                        actions: [
                            'organizations:UpdateOrganizationalUnit',
                            'organizations:DeleteOrganizationalUnit',
                            'organizations:ListOrganizationalUnitsForParent',
                            'organizations:CreateOrganizationalUnit',
                        ],
                        resources: ['*'],
                    }),
                ],
            });
            role.addManagedPolicy(policy);
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        }
        else {
            role = props.role;
        }
        const handlersPath = path.join(__dirname, '../handlers');
        const onEvent = new aws_cdk_lib_1.aws_lambda.Function(this, 'handler', {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(handlersPath + '/ou'),
            handler: 'index.on_event',
            timeout: aws_cdk_lib_1.Duration.seconds(60),
            role,
        });
        this.provider = new aws_cdk_lib_1.custom_resources.Provider(this, 'provider', {
            onEventHandler: onEvent,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_DAY,
        });
    }
}
exports.OrganizationOUProvider = OrganizationOUProvider;
_a = JSII_RTTI_SYMBOL_1;
OrganizationOUProvider[_a] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOUProvider", version: "0.4.10" };
/**
 * The construct to create or update an Organization OU.
 *
 * This relies on the custom resource provider OrganizationOUProvider.
 *
 * @stability stable
 */
class OrganizationOU extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id);
        const importOnDuplicate = (_e = props.importOnDuplicate) !== null && _e !== void 0 ? _e : false;
        const allowRecreateOnUpdate = (_f = props.allowRecreateOnUpdate) !== null && _f !== void 0 ? _f : false;
        const parentStr = props.parent instanceof OrganizationOU ? props.parent.resource.ref : props.parent;
        this.resource = new aws_cdk_lib_1.CustomResource(this, 'ou', {
            serviceToken: props.provider.serviceToken,
            properties: {
                Parent: parentStr,
                Name: props.name,
                ImportOnDuplicate: importOnDuplicate,
                AllowRecreateOnUpdate: allowRecreateOnUpdate,
            },
        });
        this.resource.node.addDependency(props.provider);
        if (props.parent instanceof OrganizationOU) {
            this.resource.node.addDependency(props.parent);
        }
        ;
    }
}
exports.OrganizationOU = OrganizationOU;
_b = JSII_RTTI_SYMBOL_1;
OrganizationOU[_b] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationOU", version: "0.4.10" };
/**
 * The provider for account custom resources.
 *
 * This creates a lambda function that handles custom resource requests for creating/updating accounts.
 *
 * @stability stable
 */
class OrganizationAccountProvider extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let role;
        if (!props.role) {
            role = new aws_cdk_lib_1.aws_iam.Role(this, 'role', {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            let policy = new aws_cdk_lib_1.aws_iam.ManagedPolicy(this, 'policy', {
                statements: [
                    new aws_cdk_lib_1.aws_iam.PolicyStatement({
                        actions: [
                            'organizations:ListOrganizationalUnitsForParent',
                            'organizations:ListAccountsForParent',
                            'organizations:ListRoots',
                            'organizations:MoveAccount',
                            'organizations:DescribeAccount',
                            'organizations:DescribeCreateAccountStatus',
                            'organizations:CreateAccount',
                        ],
                        resources: ['*'],
                    }),
                ],
            });
            role.addManagedPolicy(policy);
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        }
        else {
            role = props.role;
        }
        const handlersPath = path.join(__dirname, '../handlers');
        const onEvent = new aws_cdk_lib_1.aws_lambda.Function(this, 'handler', {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_9,
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(handlersPath + '/account'),
            handler: 'index.on_event',
            timeout: aws_cdk_lib_1.Duration.seconds(300),
            role,
        });
        this.provider = new aws_cdk_lib_1.custom_resources.Provider(this, 'provider', {
            onEventHandler: onEvent,
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_DAY,
        });
    }
}
exports.OrganizationAccountProvider = OrganizationAccountProvider;
_c = JSII_RTTI_SYMBOL_1;
OrganizationAccountProvider[_c] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationAccountProvider", version: "0.4.10" };
/**
 * The construct to create or update an Organization account.
 *
 * This relies on the custom resource provider OrganizationAccountProvider.
 *
 * @stability stable
 */
class OrganizationAccount extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _e, _f, _g;
        super(scope, id);
        const importOnDuplicate = (_e = props.importOnDuplicate) !== null && _e !== void 0 ? _e : false;
        const allowMove = (_f = props.allowMove) !== null && _f !== void 0 ? _f : false;
        const disableDelete = (_g = props.disableDelete) !== null && _g !== void 0 ? _g : false;
        const parentStr = props.parent instanceof OrganizationOU ? props.parent.resource.ref : props.parent;
        this.resource = new aws_cdk_lib_1.CustomResource(this, 'ou', {
            serviceToken: props.provider.serviceToken,
            properties: {
                Parent: parentStr,
                Name: props.name,
                Email: props.email,
                ImportOnDuplicate: importOnDuplicate,
                AllowMove: allowMove,
                DisableDelete: disableDelete,
            },
        });
        this.resource.node.addDependency(props.provider);
        if (props.parent instanceof OrganizationOU) {
            this.resource.node.addDependency(props.parent);
        }
        ;
    }
}
exports.OrganizationAccount = OrganizationAccount;
_d = JSII_RTTI_SYMBOL_1;
OrganizationAccount[_d] = { fqn: "@renovosolutions/cdk-library-aws-organization.OrganizationAccount", version: "0.4.10" };
//# sourceMappingURL=data:application/json;base64,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