# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/05_database.ipynb (unless otherwise specified).

__all__ = ['Database']

# Cell
from os import getenv
from dotenv import load_dotenv
import psycopg2
import logging
import pandas as pd
import string

# Cell
class Database():

    # Global variables that will be populated when database_url is called by db_connect
    db_schema = None
    twitter_users_table = None
    tweets_table = None
    hashtags_table = None

    logger = None
    conn = None

    #export
    ENV = [
        "DATABASE_USER",
        "DATABASE_PASS",
        "DATABASE_HOST",
        "DATABASE_PORT",
        "DATABASE_NAME",
        "DATABASE_URL",
    ]


    def __init__(self):
        # TODO : handle connection string here
        # Logging
        self.logger = logging.getLogger("tweet-archiveur")
        logFormatter = logging.Formatter("%(asctime)s -  %(name)-12s %(levelname)-8s %(message)s")
        self.logger.setLevel(logging.DEBUG)
        # # File logger
        # fh = logging.FileHandler("tweet-archiveur.log")
        # fh.setLevel(logging.DEBUG)
        # fh.setFormatter(logFormatter)
        # logger.addHandler(fh)
        if not len(self.logger.handlers):
            # Console logger
            consoleHandler = logging.StreamHandler()
            consoleHandler.setFormatter(logFormatter)
            self.logger.addHandler(consoleHandler)
        self.logger.info(f'Loading database module...')

        if getenv('DATABASE_USER') is None:
            # Load .env only in Notebook, it will be populated at runtime by docker
            from pathlib import Path
            env_path = Path('..') / '.env'
            if env_path.is_file():
                load_dotenv(dotenv_path=env_path)
            else:
                self.logger.error(f"No {env_path} found !")
                exit(4)
        # Connect to database
        self.conn = psycopg2.connect(self.database_url())
        return


    def database_config(self):
        return tuple(getenv(env) for env in self.ENV)


    def database_url(self) -> str:
        #global db_schema, tweets_table, twitter_users_table, hashtags_table
        user, pswd, host, port, name, url = self.database_config()
        self.db_schema = getenv('DATABASE_USER')
        self.twitter_users_table = 'twitter_users'
        self.tweets_table = 'tweets'
        self.hashtags_table = 'tweets_hashtags'

        self.logger.debug(f"DEBUG : connect(user={user}, password=XXXX, host={host}, port={port}, database={name}, url={url})")
        if user is None and url is None:
            self.logger.error("Empty .env : no user or URL !")
            return None
        if url:
            return url
        else:
            return f"postgresql://{user}:{pswd}@{host}:{port}/{name}"

#     def db_connect(self):
#         return psycopg2.connect(self.database_url())


    def exec_query(self, sql):
        cur = self.conn.cursor()
        cur.execute(sql)
        self.conn.commit()
        cur.close()

    def create_tables_if_not_exist(self, force = False):
        self.exec_query(f'CREATE SCHEMA IF NOT EXISTS {self.db_schema}  AUTHORIZATION CURRENT_USER;')
        if force :
            self.logger.info("Cleaning database")
            # Drop table if exist
            self.exec_query(f'DROP TABLE IF EXISTS {self.db_schema}.{self.hashtags_table};')
            self.exec_query(f'DROP TABLE IF EXISTS {self.db_schema}.{self.tweets_table};')
            self.exec_query(f'DROP TABLE IF EXISTS {self.db_schema}.{self.twitter_users_table};')



        # Create table
        users = f'''
        CREATE TABLE IF NOT EXISTS {self.db_schema}.{self.twitter_users_table}
        (
            twitter_id bigint NOT NULL,
            name character varying(50) NOT NULL,
            twitter_followers integer,
            twitter_tweets integer,
            PRIMARY KEY (twitter_id)
        );
        '''
        self.exec_query(users)
        self.exec_query(f'ALTER TABLE {self.db_schema}.{self.twitter_users_table} OWNER to CURRENT_USER;')
        # Tweet of more than 500 char have been found
        tweets = f'''
        CREATE TABLE IF NOT EXISTS {self.db_schema}.{self.tweets_table}
        (
            tweet_id bigint NOT NULL,
            twitter_id bigint NOT NULL,
            datetime_utc timestamp without time zone,
            datetime_local timestamp without time zone,
            retweet integer,
            favorite integer,
            text character varying(600),
            PRIMARY KEY (tweet_id),
            CONSTRAINT fk_user
             FOREIGN KEY(twitter_id)
                 REFERENCES {self.twitter_users_table}(twitter_id)
        );'''
        self.exec_query(tweets)
        self.exec_query(f'ALTER TABLE {self.db_schema}.{self.tweets_table} OWNER to CURRENT_USER;')
        #if not is_table_exist(conn, hashtags_table):
        hashtags = f'''
        CREATE TABLE IF NOT EXISTS {self.db_schema}.{self.hashtags_table}
        (
            id SERIAL PRIMARY KEY,
            tweet_id bigint NOT NULL,
            twitter_id bigint NOT NULL,
            datetime_local timestamp without time zone,
            hashtag character varying(140),
            CONSTRAINT fk_user
             FOREIGN KEY(twitter_id)
                 REFERENCES {self.twitter_users_table}(twitter_id),
            CONSTRAINT fk_tweet
             FOREIGN KEY(tweet_id)
                 REFERENCES {self.tweets_table}(tweet_id)
        );'''
        self.exec_query(hashtags)
        self.exec_query(f'ALTER TABLE {self.db_schema}.{self.hashtags_table} OWNER TO CURRENT_USER;')


    # Bulk INSERT of values in a table
    def insert_pandas(self, table, df, fields, prevent_conflict = True, delete_where = None):
        """
        Using cursor.mogrify() to build the bulk insert query
        then cursor.execute() to execute the query
        Thanks to https://naysan.ca/2020/05/09/pandas-to-postgresql-using-psycopg2-bulk-insert-performance-benchmark/
        """
        cursor = self.conn.cursor()
        if delete_where is not None:
            query  = f'DELETE FROM {table} WHERE {list(delete_where.values())[0]} IN ({", ".join([str(i) for i in df.head(2).tweet_id.tolist()])})'
            cursor.execute(query)
        # Create a list of tupples from the dataframe values
        col = "'" + "', '".join(fields.keys()) + "'"
        df = eval("df[[" + col + "]]")
        #logger.debug(f"Bulk insert of {len(df)} lines of {len(df.columns)} columns.")
        tuples = [tuple(x) for x in df.to_numpy()]
        # Comma-separated dataframe columns
        cols = ','.join(fields.values())
        # SQL quert to execute

        param_type = param = "(" + ",".join(['%s' for i in range(len(df.columns))]) + ")"
        values = [cursor.mogrify(param_type, tup).decode('utf8') for tup in tuples]
        query  = "INSERT INTO %s(%s) VALUES " % (table, cols) + ",".join(values)
        # Get the primary key, we suppose it is the first one
        primary_key = list(fields.values())[0]
        # Get the list of other column, excluding the primary
        other_fields = list(fields.values())[1:]
        # Build the query to UPDATE if the line already exist
        if prevent_conflict:
            query += f' ON CONFLICT ({primary_key}) DO UPDATE SET '
            query += "(" + ", ".join(other_fields) + ")"
            excluded = ['EXCLUDED.' + col for col in other_fields]
            query += ' = (' + ", ".join(excluded) + ")"
        query += ';'
        try:
            cursor.execute(query, tuples)
            self.conn.commit()
        except (Exception, psycopg2.DatabaseError) as error:
            self.logger.error(f" {error} - Rollbacking {len(df)} lines !")
            self.conn.rollback()
            cursor.close()
            return 1
        cursor.close()

    def insert_twitter_users(self, df):
        if len(df) < 1:
            self.logger.debug('insert_twitter_users : Empty dataframe')
            return
        table = f'{self.db_schema}.{self.twitter_users_table}'
        fields = { # pandas : database
            'twitter_id' : 'twitter_id',
            'nom' : 'name',
            'twitter_followers' : 'twitter_followers',
            'twitter_tweets' : 'twitter_tweets'
        }
        self.insert_pandas(table, df, fields)


    def insert_tweets(self, df):
        if len(df) < 1:
            self.logger.debug('insert_tweets : Empty dataframe')
            return
        table = f'{self.db_schema}.{self.tweets_table}'
        df['text_new'] = df.text.apply(self.filter_str)
        fields = { # pandas : database
            'tweet_id' : 'tweet_id',
            'twitter_id' : 'twitter_id',
            'datetime_utc' : 'datetime_utc',
            'datetime_local' : 'datetime_local',
            'retweet' : 'retweet',
            'favorite' : 'favorite',
            'text_new' : 'text',
        }
        self.insert_pandas(table, df, fields)

    def insert_hashtags(self, df):
        if len(df) < 1:
            self.logger.debug('insert_hashtags : Empty dataframe')
            return
        table = f'{self.db_schema}.{self.hashtags_table}'
        fields = { # pandas : database
            'tweet_id' : 'tweet_id',
            'twitter_id' : 'twitter_id',
            'datetime_local' : 'datetime_local',
            'hashtag' : 'hashtag',
        }
        delete_where = { # pandas : database
            'tweet_id' : 'tweet_id',
        }
        self.insert_pandas(table, df, fields, prevent_conflict = False, delete_where = delete_where)


    def filter_str(self, s):
        s = s.replace('%', '%%')
        return s

    def __del__(self):
      # body of destructor
      self.conn.close()