"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cluster = exports.ClusterType = exports.NodeType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const parameter_group_1 = require("./parameter-group");
const aws_redshift_1 = require("aws-cdk-lib/aws-redshift");
const subnet_group_1 = require("./subnet-group");
/**
 * Possible Node Types to use in the cluster
 * used for defining {@link ClusterProps.nodeType}.
 */
var NodeType;
(function (NodeType) {
    /**
     * ds2.xlarge
     */
    NodeType["DS2_XLARGE"] = "ds2.xlarge";
    /**
     * ds2.8xlarge
     */
    NodeType["DS2_8XLARGE"] = "ds2.8xlarge";
    /**
     * dc1.large
     */
    NodeType["DC1_LARGE"] = "dc1.large";
    /**
     * dc1.8xlarge
     */
    NodeType["DC1_8XLARGE"] = "dc1.8xlarge";
    /**
     * dc2.large
     */
    NodeType["DC2_LARGE"] = "dc2.large";
    /**
     * dc2.8xlarge
     */
    NodeType["DC2_8XLARGE"] = "dc2.8xlarge";
    /**
     * ra3.xlplus
     */
    NodeType["RA3_XLPLUS"] = "ra3.xlplus";
    /**
     * ra3.4xlarge
     */
    NodeType["RA3_4XLARGE"] = "ra3.4xlarge";
    /**
     * ra3.16xlarge
     */
    NodeType["RA3_16XLARGE"] = "ra3.16xlarge";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * What cluster type to use.
 * Used by {@link ClusterProps.clusterType}
 */
var ClusterType;
(function (ClusterType) {
    /**
     * single-node cluster, the {@link ClusterProps.numberOfNodes} parameter is not required
     */
    ClusterType["SINGLE_NODE"] = "single-node";
    /**
     * multi-node cluster, set the amount of nodes using {@link ClusterProps.numberOfNodes} parameter
     */
    ClusterType["MULTI_NODE"] = "multi-node";
})(ClusterType = exports.ClusterType || (exports.ClusterType = {}));
/**
 * A new or imported clustered database.
 */
class ClusterBase extends aws_cdk_lib_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterName,
            targetType: secretsmanager.AttachmentTargetType.REDSHIFT_CLUSTER,
        };
    }
}
/**
 * Create a Redshift cluster a given number of nodes.
 *
 * @resource AWS::Redshift::Cluster
 */
class Cluster extends ClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_ClusterProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Cluster);
            }
            throw error;
        }
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets ?? {
            subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
        };
        this.parameterGroup = props.parameterGroup;
        const removalPolicy = props.removalPolicy ?? aws_cdk_lib_1.RemovalPolicy.RETAIN;
        const subnetGroup = props.subnetGroup ?? new subnet_group_1.ClusterSubnetGroup(this, 'Subnets', {
            description: `Subnets for ${id} Redshift cluster`,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            removalPolicy: removalPolicy,
        });
        const securityGroups = props.securityGroups ?? [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'Redshift security group',
                vpc: this.vpc,
            })];
        const securityGroupIds = securityGroups.map(sg => sg.securityGroupId);
        let secret;
        if (!props.masterUser.masterPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.masterUsername,
                encryptionKey: props.masterUser.encryptionKey,
            });
        }
        const clusterType = props.clusterType || ClusterType.MULTI_NODE;
        const nodeCount = this.validateNodeCount(clusterType, props.numberOfNodes);
        if (props.encrypted === false && props.encryptionKey !== undefined) {
            throw new Error('Cannot set property encryptionKey without enabling encryption!');
        }
        this.singleUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER;
        this.multiUserRotationApplication = secretsmanager.SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER;
        let loggingProperties;
        if (props.loggingProperties) {
            loggingProperties = {
                bucketName: props.loggingProperties.loggingBucket.bucketName,
                s3KeyPrefix: props.loggingProperties.loggingKeyPrefix,
            };
            props.loggingProperties.loggingBucket.addToResourcePolicy(new iam.PolicyStatement({
                actions: [
                    's3:GetBucketAcl',
                    's3:PutObject',
                ],
                resources: [
                    props.loggingProperties.loggingBucket.arnForObjects('*'),
                    props.loggingProperties.loggingBucket.bucketArn,
                ],
                principals: [
                    new iam.ServicePrincipal('redshift.amazonaws.com'),
                ],
            }));
        }
        this.cluster = new aws_redshift_1.CfnCluster(this, 'Resource', {
            // Basic
            allowVersionUpgrade: true,
            automatedSnapshotRetentionPeriod: 1,
            clusterType,
            clusterIdentifier: props.clusterName,
            clusterSubnetGroupName: subnetGroup.clusterSubnetGroupName,
            vpcSecurityGroupIds: securityGroupIds,
            port: props.port,
            clusterParameterGroupName: props.parameterGroup && props.parameterGroup.clusterParameterGroupName,
            // Admin (unsafeUnwrap here is safe)
            masterUsername: secret?.secretValueFromJson('username').unsafeUnwrap() ?? props.masterUser.masterUsername,
            masterUserPassword: secret?.secretValueFromJson('password').unsafeUnwrap()
                ?? props.masterUser.masterPassword?.unsafeUnwrap()
                ?? 'default',
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            nodeType: props.nodeType || NodeType.DC2_LARGE,
            numberOfNodes: nodeCount,
            loggingProperties,
            iamRoles: props?.roles?.map(role => role.roleArn),
            dbName: props.defaultDatabaseName || 'default_db',
            publiclyAccessible: props.publiclyAccessible || false,
            // Encryption
            kmsKeyId: props.encryptionKey?.keyId,
            encrypted: props.encrypted ?? true,
            classic: props.classicResizing,
            elasticIp: props.elasticIp,
        });
        this.cluster.applyRemovalPolicy(removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterName = this.cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = aws_cdk_lib_1.Token.asNumber(this.cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(this.cluster.attrEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups, defaultPort });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromClusterAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_ClusterAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromClusterAttributes);
            }
            throw error;
        }
        class Import extends ClusterBase {
            constructor() {
                super(...arguments);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: ec2.Port.tcp(attrs.clusterEndpointPort),
                });
                this.clusterName = attrs.clusterName;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.clusterEndpointPort);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_RotationMultiUserOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addRotationMultiUser);
            }
            throw error;
        }
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    validateNodeCount(clusterType, numberOfNodes) {
        if (clusterType === ClusterType.SINGLE_NODE) {
            // This property must not be set for single-node clusters; be generous and treat a value of 1 node as undefined.
            if (numberOfNodes !== undefined && numberOfNodes !== 1) {
                throw new Error('Number of nodes must be not be supplied or be 1 for cluster type single-node');
            }
            return undefined;
        }
        else {
            if (aws_cdk_lib_1.Token.isUnresolved(numberOfNodes)) {
                return numberOfNodes;
            }
            const nodeCount = numberOfNodes ?? 2;
            if (nodeCount < 2 || nodeCount > 100) {
                throw new Error('Number of nodes for cluster type multi-node must be at least 2 and no more than 100');
            }
            return nodeCount;
        }
    }
    /**
     * Adds a parameter to the Clusters' parameter group
     *
     * @param name the parameter name
     * @param value the parameter name
     */
    addToParameterGroup(name, value) {
        if (!this.parameterGroup) {
            const param = {};
            param[name] = value;
            this.parameterGroup = new parameter_group_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: this.cluster.clusterIdentifier ? `Parameter Group for the ${this.cluster.clusterIdentifier} Redshift cluster` : 'Cluster parameter group for family redshift-1.0',
                parameters: param,
            });
            this.cluster.clusterParameterGroupName = this.parameterGroup.clusterParameterGroupName;
        }
        else if (this.parameterGroup instanceof parameter_group_1.ClusterParameterGroup) {
            this.parameterGroup.addParameter(name, value);
        }
        else {
            throw new Error('Cannot add a parameter to an imported parameter group.');
        }
    }
}
exports.Cluster = Cluster;
_a = JSII_RTTI_SYMBOL_1;
Cluster[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Cluster", version: "2.42.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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