"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonPatch = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// copied from https://github.com/cdk8s-team/cdk8s-core/blob/6b317a7a6a2504e228bc56bf96fc98829f88c2be/src/json-patch.ts
// under Apache 2.0 license
const fast_json_patch_1 = require("fast-json-patch");
/**
 * Utility for applying RFC-6902 JSON-Patch to a document.
 *
 * Use the the `JsonPatch.apply(doc, ...ops)` function to apply a set of
 * operations to a JSON document and return the result.
 *
 * Operations can be created using the factory methods `JsonPatch.add()`,
 * `JsonPatch.remove()`, etc.
 *
 * @example
 *
 *const output = JsonPatch.apply(input,
 *  JsonPatch.replace('/world/hi/there', 'goodbye'),
 *  JsonPatch.add('/world/foo/', 'boom'),
 *  JsonPatch.remove('/hello'));
 *
 */
class JsonPatch {
    constructor(operation) {
        this.operation = operation;
    }
    /**
     * Applies a set of JSON-Patch (RFC-6902) operations to `document` and returns the result.
     * @param document The document to patch
     * @param ops The operations to apply
     * @returns The result document
     */
    static apply(document, ...ops) {
        const result = fast_json_patch_1.applyPatch(document, ops.map((o) => o._toJson()));
        return result.newDocument;
    }
    /**
     * Adds a value to an object or inserts it into an array. In the case of an
     * array, the value is inserted before the given index. The - character can be
     * used instead of an index to insert at the end of an array.
     *
     * @example JsonPatch.add('/biscuits/1', { "name": "Ginger Nut" })
     */
    static add(path, value) {
        return new JsonPatch({ op: 'add', path, value });
    }
    /**
     * Removes a value from an object or array.
     *
     * @example JsonPatch.remove('/biscuits')
     * @example JsonPatch.remove('/biscuits/0')
     */
    static remove(path) {
        return new JsonPatch({ op: 'remove', path });
    }
    /**
     * Replaces a value. Equivalent to a “remove” followed by an “add”.
     *
     * @example JsonPatch.replace('/biscuits/0/name', 'Chocolate Digestive')
     */
    static replace(path, value) {
        return new JsonPatch({ op: 'replace', path, value });
    }
    /**
     * Copies a value from one location to another within the JSON document. Both
     * from and path are JSON Pointers.
     *
     * @example JsonPatch.copy('/biscuits/0', '/best_biscuit')
     */
    static copy(from, path) {
        return new JsonPatch({ op: 'copy', from, path });
    }
    /**
     * Moves a value from one location to the other. Both from and path are JSON Pointers.
     *
     * @example JsonPatch.move('/biscuits', '/cookies')
     */
    static move(from, path) {
        return new JsonPatch({ op: 'move', from, path });
    }
    /**
     * Tests that the specified value is set in the document. If the test fails,
     * then the patch as a whole should not apply.
     *
     * @example JsonPatch.test('/best_biscuit/name', 'Choco Leibniz')
     */
    static test(path, value) {
        return new JsonPatch({ op: 'test', path, value });
    }
    /**
     * Returns the JSON representation of this JSON patch operation.
     *
     * @internal
     */
    _toJson() {
        return this.operation;
    }
}
exports.JsonPatch = JsonPatch;
_a = JSII_RTTI_SYMBOL_1;
JsonPatch[_a] = { fqn: "cdk-pipelines-github.JsonPatch", version: "0.2.204" };
//# sourceMappingURL=data:application/json;base64,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