import torch
import torch.nn as nn
import math
import torch.utils.model_zoo as model_zoo
import os
from oblect_place_config import opt

model_urls = {
    'resnet18': 'https://download.pytorch.org/models/resnet18-5c106cde.pth',
    'resnet34': 'https://download.pytorch.org/models/resnet34-333f7ec4.pth',
    'resnet50': 'https://download.pytorch.org/models/resnet50-19c8e357.pth',
    'resnet101': 'https://download.pytorch.org/models/resnet101-5d3b4d8f.pth',
    'resnet152': 'https://download.pytorch.org/models/resnet152-b121ed2d.pth',
}

def conv3x3(in_planes, out_planes, stride=1):
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=1, bias=False)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = nn.BatchNorm2d(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = nn.BatchNorm2d(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(inplanes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride,
                               padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, planes * 4, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(planes * 4)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out


class ResNet(nn.Module):

    def __init__(self, block, layers, num_classes=1000):
        self.inplanes = 64
        super(ResNet, self).__init__()
        self.conv1 = nn.Conv2d(3, 64, kernel_size=7, stride=2, padding=3,
                               bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2)
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2)
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2)
        self.avgpool = nn.AdaptiveAvgPool2d(1)
        self.fc = nn.Linear(512 * block.expansion, num_classes)

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                n = m.kernel_size[0] * m.kernel_size[1] * m.out_channels
                m.weight.data.normal_(0, math.sqrt(2. / n))
            elif isinstance(m, nn.BatchNorm2d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()

    def _make_layer(self, block, planes, blocks, stride=1):
        downsample = None
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                nn.Conv2d(self.inplanes, planes * block.expansion,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample))
        self.inplanes = planes * block.expansion
        for i in range(1, blocks):
            layers.append(block(self.inplanes, planes))

        return nn.Sequential(*layers)

    def forward(self, x):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = x.view(x.size(0), -1)
        x = self.fc(x)

        return x


def resnet(layers, pretrained=False, pretrained_weight=None, **kwargs):
    model = None
    if layers == 18:
        model = ResNet(BasicBlock, [2, 2, 2, 2], **kwargs)
    elif layers == 34:
        model = ResNet(BasicBlock, [3, 4, 6, 3], **kwargs)
    elif layers == 50:
        model = ResNet(Bottleneck, [3, 4, 6, 3], **kwargs)
    elif layers == 101:
        model = ResNet(Bottleneck, [3, 4, 23, 3], **kwargs)
    elif layers == 152:
        model = ResNet(Bottleneck, [3, 8, 36, 3], **kwargs)
    else:
        raise Exception('Unsupport resnet layers numbers: ', layers)
    if opt.without_mask:
        if pretrained:
            model.load_state_dict(torch.load(pretrained_weight))
            print('loaded pretrained resnet{} from {}'.format(layers, pretrained_weight))
    else:
        model.conv1 = nn.Conv2d(4, 64, kernel_size=7, stride=2, padding=3, bias=False)
        if pretrained:
            pretrained_state_dict = torch.load(pretrained_weight)
            conv1 = pretrained_state_dict['conv1.weight']
            new = torch.zeros(64, 1, 7, 7)
            for i, output_channel in enumerate(conv1):
                new[i] = 0.299 * output_channel[0] + 0.587 * output_channel[1] + 0.114 * output_channel[2]
            pretrained_state_dict['conv1.weight'] = torch.cat((conv1, new), dim=1)
            model.load_state_dict(pretrained_state_dict)
            print('loaded pretrained resnet{} from {}'.format(layers, pretrained_weight))
    return model

def resnet_for_depth(layers, pretrained=False, pretrained_weight=None, **kwargs):
    model = None
    if layers == 18:
        model = ResNet(BasicBlock, [2, 2, 2, 2], **kwargs)
    elif layers == 34:
        model = ResNet(BasicBlock, [3, 4, 6, 3], **kwargs)
    elif layers == 50:
        model = ResNet(Bottleneck, [3, 4, 6, 3], **kwargs)
    elif layers == 101:
        model = ResNet(Bottleneck, [3, 4, 23, 3], **kwargs)
    elif layers == 152:
        model = ResNet(Bottleneck, [3, 8, 36, 3], **kwargs)
    else:
        raise Exception('Unsupport resnet layers numbers: ', layers)

    model.conv1 = nn.Conv2d(2, 64, kernel_size=7, stride=2, padding=3, bias=False)
    if pretrained:
        pretrained_state_dict = torch.load(pretrained_weight)
        conv1 = pretrained_state_dict['conv1.weight']
        new = torch.zeros(64, 1, 7, 7) 
        for i, output_channel in enumerate(conv1): 
            new[i] = 0.299 * output_channel[0] + 0.587 * output_channel[1] + 0.114 * output_channel[2]
        pretrained_state_dict['conv1.weight'] = torch.cat((new, new), dim=1)
        model.load_state_dict(pretrained_state_dict)
        print('loaded pretrained resnet{} from {}'.format(layers, pretrained_weight))
    return model

if __name__ == '__main__':
    model = resnet(101, pretrained=False)
    input = torch.randn(2,4,256,256)
    print('Input shape {}, output shape {}'.format(input.shape, model(input).shape))
