/**
 \file 		splut.h
 \author	michael.zohner@ec-spride.de
 \copyright	ABY - A Framework for Efficient Mixed-protocol Secure Two-party Computation
			Copyright (C) 2019 Engineering Cryptographic Protocols Group, TU Darmstadt
			This program is free software: you can redistribute it and/or modify
            it under the terms of the GNU Lesser General Public License as published
            by the Free Software Foundation, either version 3 of the License, or
            (at your option) any later version.
            ABY is distributed in the hope that it will be useful,
            but WITHOUT ANY WARRANTY; without even the implied warranty of
            MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
            GNU Lesser General Public License for more details.
            You should have received a copy of the GNU Lesser General Public License
            along with this program. If not, see <http://www.gnu.org/licenses/>.
 \brief		Implementation of the SP-LUT protocol
 */

#ifndef __SPLUT_H__
#define __SPLUT_H__

#include "sharing.h"
#include <algorithm>
#include <vector>
#include "../circuit/abycircuit.h"
#include <ENCRYPTO_utils/cbitvector.h>
#include <ENCRYPTO_utils/timer.h>

class BooleanCircuit;
class XORMasking;

constexpr uint64_t aes_sbox_multi_seq_perm_out_ttable[16][32] =
 { { 0xc56f6bf27b777c63L , 0x76abd7fe2b670130L , 0x6fc5f26b777b637cL , 0xab76fed7672b3001L , 0x6bf2c56f7c637b77L , 0xd7fe76ab01302b67L , 0xf26b6fc5637c777bL , 0xfed7ab763001672bL , 0x7b777c63c56f6bf2L , 0x2b67013076abd7feL , 0x777b637c6fc5f26bL , 0x672b3001ab76fed7L , 0x7c637b776bf2c56fL , 0x1302b67d7fe76abL , 0x637c777bf26b6fc5L , 0x3001672bfed7ab76L , 0x76abd7fe2b670130L , 0xc56f6bf27b777c63L , 0xab76fed7672b3001L , 0x6fc5f26b777b637cL , 0xd7fe76ab01302b67L , 0x6bf2c56f7c637b77L , 0xfed7ab763001672bL , 0xf26b6fc5637c777bL , 0x2b67013076abd7feL , 0x7b777c63c56f6bf2L , 0x672b3001ab76fed7L , 0x777b637c6fc5f26bL , 0x1302b67d7fe76abL , 0x7c637b776bf2c56fL , 0x3001672bfed7ab76L , 0x637c777bf26b6fc5L},
 { 0xf04759fa7dc982caL , 0xc072a49cafa2d4adL , 0x47f0fa59c97dca82L , 0x72c09ca4a2afadd4L , 0x59faf04782ca7dc9L , 0xa49cc072d4adafa2L , 0xfa5947f0ca82c97dL , 0x9ca472c0add4a2afL , 0x7dc982caf04759faL , 0xafa2d4adc072a49cL , 0xc97dca8247f0fa59L , 0xa2afadd472c09ca4L , 0x82ca7dc959faf047L , 0xd4adafa2a49cc072L , 0xca82c97dfa5947f0L , 0xadd4a2af9ca472c0L , 0xc072a49cafa2d4adL , 0xf04759fa7dc982caL , 0x72c09ca4a2afadd4L , 0x47f0fa59c97dca82L , 0xa49cc072d4adafa2L , 0x59faf04782ca7dc9L , 0x9ca472c0add4a2afL , 0xfa5947f0ca82c97dL , 0xafa2d4adc072a49cL , 0x7dc982caf04759faL , 0xa2afadd472c09ca4L , 0xc97dca8247f0fa59L , 0xd4adafa2a49cc072L , 0x82ca7dc959faf047L , 0xadd4a2af9ca472c0L , 0xca82c97dfa5947f0L},
 { 0xccf73f362693fdb7L , 0x1531d871f1e5a534L , 0xf7cc363f9326b7fdL , 0x311571d8e5f134a5L , 0x3f36ccf7fdb72693L , 0xd8711531a534f1e5L , 0x363ff7ccb7fd9326L , 0x71d8311534a5e5f1L , 0x2693fdb7ccf73f36L , 0xf1e5a5341531d871L , 0x9326b7fdf7cc363fL , 0xe5f134a5311571d8L , 0xfdb726933f36ccf7L , 0xa534f1e5d8711531L , 0xb7fd9326363ff7ccL , 0x34a5e5f171d83115L , 0x1531d871f1e5a534L , 0xccf73f362693fdb7L , 0x311571d8e5f134a5L , 0xf7cc363f9326b7fdL , 0xd8711531a534f1e5L , 0x3f36ccf7fdb72693L , 0x71d8311534a5e5f1L , 0x363ff7ccb7fd9326L , 0xf1e5a5341531d871L , 0x2693fdb7ccf73f36L , 0xe5f134a5311571d8L , 0x9326b7fdf7cc363fL , 0xa534f1e5d8711531L , 0xfdb726933f36ccf7L , 0x34a5e5f171d83115L , 0xb7fd9326363ff7ccL},
 { 0x9a059618c323c704L , 0x75b227ebe2801207L , 0x59a189623c304c7L , 0xb275eb2780e20712L , 0x96189a05c704c323L , 0x27eb75b21207e280L , 0x1896059a04c723c3L , 0xeb27b275071280e2L , 0xc323c7049a059618L , 0xe280120775b227ebL , 0x23c304c7059a1896L , 0x80e20712b275eb27L , 0xc704c32396189a05L , 0x1207e28027eb75b2L , 0x4c723c31896059aL , 0x71280e2eb27b275L , 0x75b227ebe2801207L , 0x9a059618c323c704L , 0xb275eb2780e20712L , 0x59a189623c304c7L , 0x27eb75b21207e280L , 0x96189a05c704c323L , 0xeb27b275071280e2L , 0x1896059a04c723c3L , 0xe280120775b227ebL , 0xc323c7049a059618L , 0x80e20712b275eb27L , 0x23c304c7059a1896L , 0x1207e28027eb75b2L , 0xc704c32396189a05L , 0x71280e2eb27b275L , 0x4c723c31896059aL},
 { 0xa05a6e1b1a2c8309L , 0x842fe329b3d63b52L , 0x5aa01b6e2c1a0983L , 0x2f8429e3d6b3523bL , 0x6e1ba05a83091a2cL , 0xe329842f3b52b3d6L , 0x1b6e5aa009832c1aL , 0x29e32f84523bd6b3L , 0x1a2c8309a05a6e1bL , 0xb3d63b52842fe329L , 0x2c1a09835aa01b6eL , 0xd6b3523b2f8429e3L , 0x83091a2c6e1ba05aL , 0x3b52b3d6e329842fL , 0x9832c1a1b6e5aa0L , 0x523bd6b329e32f84L , 0x842fe329b3d63b52L , 0xa05a6e1b1a2c8309L , 0x2f8429e3d6b3523bL , 0x5aa01b6e2c1a0983L , 0xe329842f3b52b3d6L , 0x6e1ba05a83091a2cL , 0x29e32f84523bd6b3L , 0x1b6e5aa009832c1aL , 0xb3d63b52842fe329L , 0x1a2c8309a05a6e1bL , 0xd6b3523b2f8429e3L , 0x2c1a09835aa01b6eL , 0x3b52b3d6e329842fL , 0x83091a2c6e1ba05aL , 0x523bd6b329e32f84L , 0x9832c1a1b6e5aa0L},
 { 0x5bb1fc20ed00d153L , 0xcf584c4a39becb6aL , 0xb15b20fc00ed53d1L , 0x58cf4a4cbe396acbL , 0xfc205bb1d153ed00L , 0x4c4acf58cb6a39beL , 0x20fcb15b53d100edL , 0x4a4c58cf6acbbe39L , 0xed00d1535bb1fc20L , 0x39becb6acf584c4aL , 0xed53d1b15b20fcL , 0xbe396acb58cf4a4cL , 0xd153ed00fc205bb1L , 0xcb6a39be4c4acf58L , 0x53d100ed20fcb15bL , 0x6acbbe394a4c58cfL , 0xcf584c4a39becb6aL , 0x5bb1fc20ed00d153L , 0x58cf4a4cbe396acbL , 0xb15b20fc00ed53d1L , 0x4c4acf58cb6a39beL , 0xfc205bb1d153ed00L , 0x4a4c58cf6acbbe39L , 0x20fcb15b53d100edL , 0x39becb6acf584c4aL , 0xed00d1535bb1fc20L , 0xbe396acb58cf4a4cL , 0xed53d1b15b20fcL , 0xcb6a39be4c4acf58L , 0xd153ed00fc205bb1L , 0x6acbbe394a4c58cfL , 0x53d100ed20fcb15bL},
 { 0x85334d43fbaaefd0L , 0xa89f3c507f02f945L , 0x3385434daafbd0efL , 0x9fa8503c027f45f9L , 0x4d438533efd0fbaaL , 0x3c50a89ff9457f02L , 0x434d3385d0efaafbL , 0x503c9fa845f9027fL , 0xfbaaefd085334d43L , 0x7f02f945a89f3c50L , 0xaafbd0ef3385434dL , 0x27f45f99fa8503cL , 0xefd0fbaa4d438533L , 0xf9457f023c50a89fL , 0xd0efaafb434d3385L , 0x45f9027f503c9fa8L , 0xa89f3c507f02f945L , 0x85334d43fbaaefd0L , 0x9fa8503c027f45f9L , 0x3385434daafbd0efL , 0x3c50a89ff9457f02L , 0x4d438533efd0fbaaL , 0x503c9fa845f9027fL , 0x434d3385d0efaafbL , 0x7f02f945a89f3c50L , 0xfbaaefd085334d43L , 0x27f45f99fa8503cL , 0xaafbd0ef3385434dL , 0xf9457f023c50a89fL , 0xefd0fbaa4d438533L , 0x45f9027f503c9fa8L , 0xd0efaafb434d3385},
 { 0xf5389d928f40a351L , 0xd2f3ff1021dab6bcL , 0x38f5929d408f51a3L , 0xf3d210ffda21bcb6L , 0x9d92f538a3518f40L , 0xff10d2f3b6bc21daL , 0x929d38f551a3408fL , 0x10fff3d2bcb6da21L , 0x8f40a351f5389d92L , 0x21dab6bcd2f3ff10L , 0x408f51a338f5929dL , 0xda21bcb6f3d210ffL , 0xa3518f409d92f538L , 0xb6bc21daff10d2f3L , 0x51a3408f929d38f5L , 0xbcb6da2110fff3d2L , 0xd2f3ff1021dab6bcL , 0xf5389d928f40a351L , 0xf3d210ffda21bcb6L , 0x38f5929d408f51a3L , 0xff10d2f3b6bc21daL , 0x9d92f538a3518f40L , 0x10fff3d2bcb6da21L , 0x929d38f551a3408fL , 0x21dab6bcd2f3ff10L , 0x8f40a351f5389d92L , 0xda21bcb6f3d210ffL , 0x408f51a338f5929dL , 0xb6bc21daff10d2f3L , 0xa3518f409d92f538L , 0xbcb6da2110fff3d2L , 0x51a3408f929d38f5L},
 { 0x1744975fec130ccdL , 0x73195d643d7ea7c4L , 0x44175f9713eccd0cL , 0x1973645d7e3dc4a7L , 0x975f17440ccdec13L , 0x5d647319a7c43d7eL , 0x5f974417cd0c13ecL , 0x645d1973c4a77e3dL , 0xec130ccd1744975fL , 0x3d7ea7c473195d64L , 0x13eccd0c44175f97L , 0x7e3dc4a71973645dL , 0xccdec13975f1744L , 0xa7c43d7e5d647319L , 0xcd0c13ec5f974417L , 0xc4a77e3d645d1973L , 0x73195d643d7ea7c4L , 0x1744975fec130ccdL , 0x1973645d7e3dc4a7L , 0x44175f9713eccd0cL , 0x5d647319a7c43d7eL , 0x975f17440ccdec13L , 0x645d1973c4a77e3dL , 0x5f974417cd0c13ecL , 0x3d7ea7c473195d64L , 0xec130ccd1744975fL , 0x7e3dc4a71973645dL , 0x13eccd0c44175f97L , 0xa7c43d7e5d647319L , 0xccdec13975f1744L , 0xc4a77e3d645d1973L , 0xcd0c13ec5f974417L},
 { 0x88902a22dc4f8160L , 0xdb0b5ede14b8ee46L , 0x9088222a4fdc6081L , 0xbdbde5eb81446eeL , 0x2a2288908160dc4fL , 0x5ededb0bee4614b8L , 0x222a908860814fdcL , 0xde5e0bdb46eeb814L , 0xdc4f816088902a22L , 0x14b8ee46db0b5edeL , 0x4fdc60819088222aL , 0xb81446ee0bdbde5eL , 0x8160dc4f2a228890L , 0xee4614b85ededb0bL , 0x60814fdc222a9088L , 0x46eeb814de5e0bdbL , 0xdb0b5ede14b8ee46L , 0x88902a22dc4f8160L , 0xbdbde5eb81446eeL , 0x9088222a4fdc6081L , 0x5ededb0bee4614b8L , 0x2a2288908160dc4fL , 0xde5e0bdb46eeb814L , 0x222a908860814fdcL , 0x14b8ee46db0b5edeL , 0xdc4f816088902a22L , 0xb81446ee0bdbde5eL , 0x4fdc60819088222aL , 0xee4614b85ededb0bL , 0x8160dc4f2a228890L , 0x46eeb814de5e0bdbL , 0x60814fdc222a9088L},
 { 0x5c2406490a3a32e0L , 0x79e4959162acd3c2L , 0x245c49063a0ae032L , 0xe4799195ac62c2d3L , 0x6495c2432e00a3aL , 0x959179e4d3c262acL , 0x4906245ce0323a0aL , 0x9195e479c2d3ac62L , 0xa3a32e05c240649L , 0x62acd3c279e49591L , 0x3a0ae032245c4906L , 0xac62c2d3e4799195L , 0x32e00a3a06495c24L , 0xd3c262ac959179e4L , 0xe0323a0a4906245cL , 0xc2d3ac629195e479L , 0x79e4959162acd3c2L , 0x5c2406490a3a32e0L , 0xe4799195ac62c2d3L , 0x245c49063a0ae032L , 0x959179e4d3c262acL , 0x6495c2432e00a3aL , 0x9195e479c2d3ac62L , 0x4906245ce0323a0aL , 0x62acd3c279e49591L , 0xa3a32e05c240649L , 0xac62c2d3e4799195L , 0x3a0ae032245c4906L , 0xd3c262ac959179e4L , 0x32e00a3a06495c24L , 0xc2d3ac629195e479L , 0xe0323a0a4906245cL},
 { 0xa94ed58d6d37c8e7L , 0x8ae7a65eaf4566cL , 0x4ea98dd5376de7c8L , 0xae08657af4ea6c56L , 0xd58da94ec8e76d37L , 0x7a6508ae566ceaf4L , 0x8dd54ea9e7c8376dL , 0x657aae086c56f4eaL , 0x6d37c8e7a94ed58dL , 0xeaf4566c08ae7a65L , 0x376de7c84ea98dd5L , 0xf4ea6c56ae08657aL , 0xc8e76d37d58da94eL , 0x566ceaf47a6508aeL , 0xe7c8376d8dd54ea9L , 0x6c56f4ea657aae08L , 0x8ae7a65eaf4566cL , 0xa94ed58d6d37c8e7L , 0xae08657af4ea6c56L , 0x4ea98dd5376de7c8L , 0x7a6508ae566ceaf4L , 0xd58da94ec8e76d37L , 0x657aae086c56f4eaL , 0x8dd54ea9e7c8376dL , 0xeaf4566c08ae7a65L , 0x6d37c8e7a94ed58dL , 0xf4ea6c56ae08657aL , 0x376de7c84ea98dd5L , 0x566ceaf47a6508aeL , 0xc8e76d37d58da94eL , 0x6c56f4ea657aae08L , 0xe7c8376d8dd54ea9L},
 { 0xc6b4a61c2e2578baL , 0x8a8bbd4b1f74dde8L , 0xb4c61ca6252eba78L , 0x8b8a4bbd741fe8ddL , 0xa61cc6b478ba2e25L , 0xbd4b8a8bdde81f74L , 0x1ca6b4c6ba78252eL , 0x4bbd8b8ae8dd741fL , 0x2e2578bac6b4a61cL , 0x1f74dde88a8bbd4bL , 0x252eba78b4c61ca6L , 0x741fe8dd8b8a4bbdL , 0x78ba2e25a61cc6b4L , 0xdde81f74bd4b8a8bL , 0xba78252e1ca6b4c6L , 0xe8dd741f4bbd8b8aL , 0x8a8bbd4b1f74dde8L , 0xc6b4a61c2e2578baL , 0x8b8a4bbd741fe8ddL , 0xb4c61ca6252eba78L , 0xbd4b8a8bdde81f74L , 0xa61cc6b478ba2e25L , 0x4bbd8b8ae8dd741fL , 0x1ca6b4c6ba78252eL , 0x1f74dde88a8bbd4bL , 0x2e2578bac6b4a61cL , 0x741fe8dd8b8a4bbdL , 0x252eba78b4c61ca6L , 0xdde81f74bd4b8a8bL , 0x78ba2e25a61cc6b4L , 0xe8dd741f4bbd8b8aL , 0xba78252e1ca6b4c6L , },
 { 0xef6034866b53e70L , 0x9e1dc186b9573561L , 0xf60e4803b566703eL , 0x1d9e86c157b96135L , 0x3480ef63e7066b5L , 0xc1869e1d3561b957L , 0x4803f60e703eb566L , 0x86c11d9e613557b9L , 0x66b53e700ef60348L , 0xb95735619e1dc186L , 0xb566703ef60e4803L , 0x57b961351d9e86c1L , 0x3e7066b503480ef6L , 0x3561b957c1869e1dL , 0x703eb5664803f60eL , 0x613557b986c11d9eL , 0x9e1dc186b9573561L , 0xef6034866b53e70L , 0x1d9e86c157b96135L , 0xf60e4803b566703eL , 0xc1869e1d3561b957L , 0x3480ef63e7066b5L , 0x86c11d9e613557b9L , 0x4803f60e703eb566L , 0xb95735619e1dc186L , 0x66b53e700ef60348L , 0x57b961351d9e86c1L , 0xb566703ef60e4803L , 0x3561b957c1869e1dL , 0x3e7066b503480ef6L , 0x613557b986c11d9eL , 0x703eb5664803f60eL},
 { 0x948ed9691198f8e1L , 0xdf2855cee9871e9bL , 0x8e9469d99811e1f8L , 0x28dfce5587e99b1eL , 0xd969948ef8e11198L , 0x55cedf281e9be987L , 0x69d98e94e1f89811L , 0xce5528df9b1e87e9L , 0x1198f8e1948ed969L , 0xe9871e9bdf2855ceL , 0x9811e1f88e9469d9L , 0x87e99b1e28dfce55L , 0xf8e11198d969948eL , 0x1e9be98755cedf28L , 0xe1f8981169d98e94L , 0x9b1e87e9ce5528dfL , 0xdf2855cee9871e9bL , 0x948ed9691198f8e1L , 0x28dfce5587e99b1eL , 0x8e9469d99811e1f8L , 0x55cedf281e9be987L , 0xd969948ef8e11198L , 0xce5528df9b1e87e9L , 0x69d98e94e1f89811L , 0xe9871e9bdf2855ceL , 0x1198f8e1948ed969L , 0x87e99b1e28dfce55L , 0x9811e1f88e9469d9L , 0x1e9be98755cedf28L , 0xf8e11198d969948eL , 0x9b1e87e9ce5528dfL , 0xe1f8981169d98e94L},
 { 0x6842e6bf0d89a18cL , 0x16bb54b00f2d9941L , 0x4268bfe6890d8ca1L , 0xbb16b0542d0f4199L , 0xe6bf6842a18c0d89L , 0x54b016bb99410f2dL , 0xbfe642688ca1890dL , 0xb054bb1641992d0fL , 0xd89a18c6842e6bfL , 0xf2d994116bb54b0L , 0x890d8ca14268bfe6L , 0x2d0f4199bb16b054L , 0xa18c0d89e6bf6842L , 0x99410f2d54b016bbL , 0x8ca1890dbfe64268L , 0x41992d0fb054bb16L , 0x16bb54b00f2d9941L , 0x6842e6bf0d89a18cL , 0xbb16b0542d0f4199L , 0x4268bfe6890d8ca1L , 0x54b016bb99410f2dL , 0xe6bf6842a18c0d89L , 0xb054bb1641992d0fL , 0xbfe642688ca1890dL , 0xf2d994116bb54b0L , 0xd89a18c6842e6bfL , 0x2d0f4199bb16b054L , 0x890d8ca14268bfe6L , 0x99410f2d54b016bbL , 0xa18c0d89e6bf6842L , 0x41992d0fb054bb16L , 0x8ca1890dbfe64268L}
 };




//#define DEBUG_SPLUT
//#define BENCHBOOLTIME
/**
 BOOL SHARING without multiplication triples. Uses 1ooN OT instead.
 */
class SetupLUT: public Sharing {

public:
	/** Constructor of the class.*/
	SetupLUT(e_sharing context, e_role role, uint32_t sharebitlen, ABYCircuit* circuit, crypto* crypt, const std::string& circdir = ABY_CIRCUIT_DIR);

	/** Destructor of the class.*/
	virtual ~SetupLUT();

	//SUPER CLASS MEMBER FUNCTION
	void PrepareSetupPhase(ABYSetup* setup);
	void PerformSetupPhase(ABYSetup* setup);
	void FinishSetupPhase(ABYSetup* setup);
	void EvaluateLocalOperations(uint32_t level);
	void EvaluateInteractiveOperations(uint32_t level);

	void FinishCircuitLayer();

	void PrepareOnlinePhase();

	void PreComputationPhase() {
		return;
	}

	inline void InstantiateGate(GATE* gate);

	void GetDataToSend(std::vector<BYTE*>& sendbuf, std::vector<uint64_t>& bytesize);
	void GetBuffersToReceive(std::vector<BYTE*>& rcvbuf, std::vector<uint64_t>& rcvbytes);

	void EvaluateSIMDGate(uint32_t gateid);

	Circuit* GetCircuitBuildRoutine();

	uint32_t AssignInput(CBitVector& input);
	uint32_t GetOutput(CBitVector& out);

	uint32_t GetMaxCommunicationRounds();

	uint32_t GetNumNonLinearOperations() {
		return m_nTotalTTs;
	}
	;

	void Reset();
	/*vector<uint32_t> GetNumOTs() {
		return m_nTotalTTs;
	}
	;*/

	const char* sharing_type() {
		return "Setup LUT";
	}
	;

	void PrintPerformanceStatistics();
	//ENDS HERE
private:


	uint32_t m_nXORGates;
	uint64_t m_nTotalTTs;

	XORMasking *fMaskFct;
	std::vector<std::vector<uint32_t> > m_vTTGates;

	std::vector<uint32_t> m_vInputShareGates;
	std::vector<uint32_t> m_vOutputShareGates;

	uint32_t m_nInputShareSndSize;
	uint32_t m_nOutputShareSndSize;

	uint32_t m_nInputShareRcvSize;
	uint32_t m_nOutputShareRcvSize;

	uint32_t m_nNumANDSizes;

	//non_lin_vec_ctx* m_vANDs;
	//first dimension is for server / client, second dimension is for ins, third dimension is for outs
	std::vector<std::vector<std::vector<tt_lens_ctx> > > m_vNOTs;

	std::vector<std::vector<uint32_t> > m_vOutBitMapping;


	CBitVector m_vInputShareSndBuf;
	CBitVector m_vOutputShareSndBuf;

	//information on updated choice bits from the OT
	std::vector<std::vector<CBitVector*> > m_vChoiceUpdateSndBuf;
	std::vector<std::vector<uint32_t> > m_nChoiceUpdateSndCtr;
	std::vector<std::vector<CBitVector*> > m_vChoiceUpdateRcvBuf;
	std::vector<std::vector<uint32_t> > m_nChoiceUpdateRcvCtr;

	//information on updated masks from the OT
	std::vector<std::vector<CBitVector*> > m_vMaskUpdateSndBuf;
	std::vector<std::vector<uint32_t> > m_nMaskUpdateSndCtr;
	std::vector<std::vector<CBitVector*> > m_vMaskUpdateRcvBuf;
	std::vector<std::vector<uint32_t> > m_nMaskUpdateRcvCtr;


	std::vector<uint8_t*> m_vSndBufStash;
	std::vector<uint64_t> m_vSndBytesStash;

	CBitVector m_vInputShareRcvBuf;
	CBitVector m_vOutputShareRcvBuf;

	BooleanCircuit* m_cBoolCircuit;

	std::vector<std::vector<CBitVector**> > m_vPreCompOTX;

	std::vector<std::vector<CBitVector*> > m_vPreCompOTMasks;
	std::vector<std::vector<uint32_t> > m_vPreCompMaskIdx;

	std::vector<std::vector<CBitVector*> > m_vPreCompOTC;
	std::vector<std::vector<uint32_t> >m_vPreCompChoiceIdx;
	std::vector<std::vector<CBitVector*> > m_vPreCompOTR;


	std::vector<std::vector<CBitVector*> > m_vTableRnd;
	std::vector<std::vector<uint32_t> > m_nTableRndIdx;

	double t_snd, t_rcv;


	bool m_bPlaySender;

#ifdef BENCHBOOLTIME
	double m_nCombTime;
	double m_nSubsetTime;
	double m_nCombStructTime;
	double m_nSIMDTime;
	double m_nXORTime;
#endif
	/**
	 compute local operations on the given circuit layer
	 \param 	depth 	gate layer which should be evaluated
	 */
	//void Internal_EvaluateLocalOperations(uint32_t depth);
	/**
	 compute interactive operations on the given circuit layer
	 \param 	depth 	gate layer which should be evaluated
	 */
	//void Internal_EvaluateInteractiveOperations(uint32_t depth);
	/**
	 Share Values
	 \param 	gateid 	GateID
	 */
	inline void ShareValues(uint32_t gateid);
	/**
	 Reconstruct Values
	 \param 	gateid 	GateID
	 */
	inline void ReconstructValue(uint32_t gateid);

	/**
	 Method for assigning input shares.
	 */
	void AssignInputShares();
	/**
	 Method for assigning output shares.
	 */
	void AssignOutputShares();

	/**
	 Method for selective open of the given gate.
	 \param gateid 	Gate Identifier
	 */
	//inline void SelectiveOpen(uint32_t gateid);
	/**
	 Method for selective open vector of the given gate.
	 \param gateid 	Gate Identifier
	 */
	//inline void SelectiveOpenVec(uint32_t gateid);
	/**
	 Method for Evaluating MTs.
	 */
	//void EvaluateMTs();
	/**
	 Method for evaluating AND gate
	 */
	//void EvaluateANDGate();
	/**
	 Method for evaluating XOR gate for the inputted
	 gate object.
	 \param gateid		Gate identifier
	 */
	inline void EvaluateXORGate(uint32_t gateid);
	/**
	 Method for evaluating Truth Table gates for the inputted
	 gate object.
	 \param gateid		Gate identifier
	 */
	inline void EvaluateTTGate(uint32_t gateid);
	/**
	 Method for evaluating Inversion gate for the inputted
	 gate object.
	 \param gateid		Gate identifier
	 */
	inline void EvaluateINVGate(uint32_t gateid);
	/**
	 Method for evaluating Conversion gate for the inputted
	 gate object.
	 \param gateid		Gate identifier
	 */
	inline void EvaluateCONVGate(uint32_t gateid);
	/**
	 Method for evaluating Constant gate for the inputted
	 gate object.
	 \param gateid		Gate identifier
	 */
	inline void EvaluateConstantGate(uint32_t gateid);
	/**
	 Method for initialising MTs.
	 */
	//void InitializeMTs();

	/**
	 * Sender routine for evaluating truth-table gates
	 */
	inline void SenderEvaluateTTGates();
	/**
	 * Receiver routine for evaluating truth-table gates
	 */
	inline void ReceiverEvaluateTTGates();
	/**
	 Method for initialising.
	 */
	void Init();
	/**
	 Method for initiating a new layer.
	 */
	void InitNewLayer();

};

#define MAX_GATEVAL_BUFSIZE 256

constexpr uint64_t gateval_masks[8][8][32] = {//1
		{ {0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL, 0xffffffffffffffffL},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		},//2
		{ {0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L, 0x5555555555555555L},
		  {0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL, 0xAAAAAAAAAAAAAAAAL},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		},//3
		{ { 0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L },
		  { 0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L },
		  { 0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L,  0x4924924992492492L,  0x2492492449249249L,  0x9249249224924924L },
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		},//4
		{ { 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L, 0x1111111111111111L },
		  { 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L, 0x2222222222222222L },
		  { 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L, 0x4444444444444444L },
		  { 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L, 0x8888888888888888L },
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		  {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		},//5
		{ 	{ 0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L },
			{ 0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L },
			{ 0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L },
			{ 0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L },
			{ 0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L,  0x8421084221084210L,  0x0842108442108421L,  0x1084210884210842L,  0x2108421008421084L,  0x4210842110842108L },
			{0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
			{0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
			{0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		},//6
		{   { 0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L  },
			{ 0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L  },
			{ 0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L  },
			{ 0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L  },
			{ 0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L,  0x0410410441041041L,  0x1041041004104104L,  0x4104104110410410L  },
			{ 0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L,  0x0820820882082082L,  0x2082082008208208L,  0x8208208220820820L  },
			{ 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
			{ 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		},//7
		{   { 0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L  },
			{ 0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L  },
			{ 0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L  },
			{ 0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L  },
			{ 0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L  },
			{ 0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L  },
			{ 0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L,  0x4081020404081020L,  0x2040810202040810L,  0x1020408181020408L,  0x0810204040810204L,  0x0408102020408102L,  0x0204081010204081L,  0x8102040808102040L  },
		    {0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L, 0x0L},
		},//8
		{ { 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L, 0x0101010101010101L},
		  { 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L, 0x0202020202020202L},
		  { 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L, 0x0404040404040404L},
		  { 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L, 0x0808080808080808L},
		  { 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L, 0x1010101010101010L },
		  { 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L, 0x2020202020202020L },
		  { 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L, 0x4040404040404040L },
		  { 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L, 0x8080808080808080L }
		}
};



#endif /* __SPLUT_H__ */
