"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsEdgeToOrigin = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/* eslint-disable @typescript-eslint/indent */
const crypto = require("crypto");
const fs_1 = require("fs");
const os = require("os");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cf = require("aws-cdk-lib/aws-cloudfront");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodejs = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
/**
 * Create a new MicroApps Edge to Origin Function w/ `config.yml`
 */
class MicroAppsEdgeToOrigin extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props must be set');
        }
        const { addXForwardedHostHeader = true, assetNameRoot, assetNameSuffix, originRegion, signingMode = 'sign', removalPolicy, rootPathPrefix, replaceHostHeader = true, tableRulesArn, } = props;
        // Create the edge function config file from the construct options
        const edgeToOriginConfigYaml = MicroAppsEdgeToOrigin.generateEdgeToOriginConfig({
            originRegion: originRegion || aws_cdk_lib_1.Aws.REGION,
            addXForwardedHostHeader,
            replaceHostHeader,
            signingMode: signingMode === 'none' ? '' : signingMode,
            rootPathPrefix,
            ...(tableRulesArn
                ? {
                    tableName: tableRulesArn,
                }
                : {}),
        });
        //
        // Create the Edge to Origin Function
        //
        const edgeToOriginFuncProps = {
            functionName: assetNameRoot ? `${assetNameRoot}-edge-to-origin${assetNameSuffix}` : undefined,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_16_X,
            timeout: aws_cdk_lib_1.Duration.seconds(5),
            initialPolicy: [
                // This can't have a reference to the httpApi because it would mean
                // the parent stack (this stack) has to be created before the us-east-1
                // child stack for the Edge Lambda Function.
                // That's why we use a tag-based policy to allow the Edge Function
                // to invoke any API Gateway API that we apply a tag to
                // We allow the edge function to sign for all regions since
                // we may use custom closest region in the future.
                new iam.PolicyStatement({
                    actions: ['execute-api:Invoke'],
                    resources: [`arn:aws:execute-api:*:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*/*/*/*`],
                }),
                //
                // Grant permission to invoke tagged Function URLs
                //
                new iam.PolicyStatement({
                    actions: ['lambda:InvokeFunctionUrl'],
                    resources: [`arn:aws:lambda:*:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
                    conditions: {
                        StringEquals: { 'aws:ResourceTag/microapp-managed': 'true' },
                    },
                }),
            ],
            ...(removalPolicy ? { removalPolicy } : {}),
        };
        const rootDistPath = path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'dist');
        const rootDistExists = fs_1.existsSync(path.join(rootDistPath, 'index.js'));
        const localDistPath = path.join(__dirname, 'microapps-edge-to-origin');
        const localDistExists = fs_1.existsSync(path.join(localDistPath, 'index.js'));
        if (process.env.NODE_ENV === 'test' && rootDistExists) {
            // This is for tests run under jest - Prefer root dist bundle
            // This is also for anytime when the edge function has already been bundled
            this._edgeToOriginFunction = this.createEdgeFunction(rootDistPath, edgeToOriginConfigYaml, edgeToOriginFuncProps);
        }
        else if (localDistExists) {
            // Prefer local dist above root dist if both exist (when building for distribution)
            this._edgeToOriginFunction = this.createEdgeFunction(localDistPath, edgeToOriginConfigYaml, edgeToOriginFuncProps);
        }
        else if (rootDistExists) {
            // Use local dist if it exists (when deploying from CDK in this repo)
            this._edgeToOriginFunction = this.createEdgeFunction(rootDistPath, edgeToOriginConfigYaml, edgeToOriginFuncProps);
        }
        else {
            // This is used when bundling the app and building the CDK module
            // for distribution.
            fs_1.writeFileSync(path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'config.yml'), edgeToOriginConfigYaml);
            // Copy the appFrame.html to the place where the bundling will find it
            fs_1.copyFileSync(path.join(__dirname, '..', '..', 'microapps-router', 'appFrame.html'), path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'appFrame.html'));
            // This builds the function for distribution with the CDK Construct
            // and will be used during local builds and PR builds of microapps-core
            // if the microapps-edge-to-origin function is not already bundled.
            // This will fail to deploy in any region other than us-east-1
            this._edgeToOriginFunction = new lambdaNodejs.NodejsFunction(this, 'edge-to-apigwy-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-edge-to-origin', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                    commandHooks: {
                        beforeInstall: () => [],
                        beforeBundling: () => [],
                        afterBundling: (_inputDir, outputDir) => {
                            // 2022-10-02 - Note that this is ignoring the generated config
                            // file above and including the default template config file
                            return [
                                `${os.platform() === 'win32' ? 'copy' : 'cp'} ${path.join(__dirname, '..', '..', '..', 'configs', 'microapps-edge-to-origin', 'config.yml')} ${outputDir}`,
                                `${os.platform() === 'win32' ? 'copy' : 'cp'} ${path.join(__dirname, '..', '..', 'microapps-router', 'appFrame.html')} ${outputDir}`,
                            ];
                        },
                    },
                },
                ...edgeToOriginFuncProps,
            });
        }
        this._edgeToOriginLambdas = [
            {
                eventType: cf.LambdaEdgeEventType.ORIGIN_REQUEST,
                functionVersion: this._edgeToOriginFunction.currentVersion,
                includeBody: true,
            },
        ];
        // Grant access to the rules table
        if (tableRulesArn) {
            const tableRules = dynamodb.Table.fromTableName(this, 'tableRules', tableRulesArn);
            tableRules.grantReadData(this._edgeToOriginFunction);
        }
    }
    /**
     * Generate the yaml config for the edge lambda
     * @param props
     * @returns
     */
    static generateEdgeToOriginConfig(props) {
        return `originRegion: ${props.originRegion}
${props.signingMode === '' ? '' : `signingMode: ${props.signingMode}`}
addXForwardedHostHeader: ${props.addXForwardedHostHeader}
replaceHostHeader: ${props.replaceHostHeader}
${props.tableName ? `tableName: '${props.tableName}'` : ''}
${props.rootPathPrefix ? `rootPathPrefix: '${props.rootPathPrefix}'` : ''}`;
    }
    get edgeToOriginFunction() {
        return this._edgeToOriginFunction;
    }
    get edgeToOriginLambdas() {
        return this._edgeToOriginLambdas;
    }
    /**
     * Hash the stack name to make the EdgeFunction parameter name unique
     *
     * @param stack
     * @returns
     */
    hashStackName() {
        return crypto.createHash('sha1').update(aws_cdk_lib_1.Stack.of(this).stackName).digest('hex').substring(0, 8);
    }
    createEdgeFunction(distPath, edgeToOriginConfigYaml, edgeToOriginFuncProps) {
        var _b;
        fs_1.writeFileSync(path.join(distPath, 'config.yml'), edgeToOriginConfigYaml);
        fs_1.copyFileSync(path.join(__dirname, '..', '..', 'microapps-router', 'appFrame.html'), path.join(distPath, 'appFrame.html'));
        // The exclude varying per stack name is a kludge to get the asset bundled
        // with the stack-specifc config.yml file, otherwise they all get the file
        // generated for the first instance of the construct within any stack
        // in the app.
        const code = lambda.Code.fromAsset(distPath, { exclude: [`**/${aws_cdk_lib_1.Stack.of(this)}`] });
        const stackHash = (_b = this.hashStackName()) !== null && _b !== void 0 ? _b : '';
        // EdgeFunction has a bug where it will generate the same parameter
        // name across multiple stacks in the same region if the id param is constant
        const edge = new cf.experimental.EdgeFunction(this, `edge-to-apigwy-func-${stackHash}`, {
            stackId: `microapps-edge-to-origin-${stackHash}`,
            code,
            functionName: `microapps-edge-to-origin-${stackHash}`,
            handler: 'index.handler',
            ...edgeToOriginFuncProps,
        });
        aws_cdk_lib_1.Tags.of(edge).add('Name', aws_cdk_lib_1.Stack.of(this).stackName);
        return edge;
    }
}
exports.MicroAppsEdgeToOrigin = MicroAppsEdgeToOrigin;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsEdgeToOrigin[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsEdgeToOrigin", version: "0.3.0-alpha.3" };
//# sourceMappingURL=data:application/json;base64,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