/*! 
* DevExtreme Web
* Version: 14.1.7
* Build date: Sep 22, 2014
*
* Copyright (c) 2012 - 2014 Developer Express Inc. ALL RIGHTS RESERVED
* EULA: https://www.devexpress.com/Support/EULAs/DevExtreme.xml
*/

"use strict";

if (!window.DevExpress) {
    /*! Module core, file devexpress.js */
    (function($, global, undefined) {
        (function checkjQueryVersion(version) {
            version = version.split(".");
            if (version[0] < 1 || version[0] == 1 && version[1] < 10)
                throw Error("Your version of jQuery is too old. Please upgrade jQuery to 1.10.0 or later.");
        })($.fn.jquery);
        var Class = function() {
                var wrapOverridden = function(baseProto, methodName, method) {
                        return function() {
                                var prevCallBase = this.callBase;
                                this.callBase = baseProto[methodName];
                                try {
                                    return method.apply(this, arguments)
                                }
                                finally {
                                    this.callBase = prevCallBase
                                }
                            }
                    };
                var clonePrototype = function(obj) {
                        var func = function(){};
                        func.prototype = obj.prototype;
                        return new func
                    };
                var classImpl = function(){};
                var redefine = function(members) {
                        var that = this;
                        if (!members)
                            return that;
                        var memberNames = $.map(members, function(_, k) {
                                return k
                            });
                        $.each(["toString", "toLocaleString", "valueOf"], function() {
                            if (members[this])
                                memberNames.push(this)
                        });
                        $.each(memberNames, function() {
                            var overridden = $.isFunction(that.prototype[this]) && $.isFunction(members[this]);
                            that.prototype[this] = overridden ? wrapOverridden(that.parent.prototype, this, members[this]) : members[this]
                        });
                        return that
                    };
                var include = function() {
                        var classObj = this;
                        $.each(arguments, function() {
                            if (this.ctor)
                                classObj._includedCtors.push(this.ctor);
                            for (var name in this) {
                                if (name === "ctor")
                                    continue;
                                if (name in classObj.prototype)
                                    throw Error("Member name collision: " + name);
                                classObj.prototype[name] = this[name]
                            }
                        });
                        return classObj
                    };
                var subclassOf = function(parentClass) {
                        if (this.parent === parentClass)
                            return true;
                        if (!this.parent || !this.parent.subclassOf)
                            return false;
                        return this.parent.subclassOf(parentClass)
                    };
                classImpl.inherit = function(members) {
                    var inheritor = function() {
                            if (!this || this === global || typeof this.constructor !== "function")
                                throw Error("A class must be instantiated using the 'new' keyword");
                            var instance = this,
                                ctor = instance.ctor;
                            if (ctor)
                                ctor.apply(instance, arguments);
                            $.each(instance.constructor._includedCtors, function() {
                                this.call(instance)
                            })
                        };
                    inheritor.prototype = clonePrototype(this);
                    inheritor.inherit = this.inherit;
                    inheritor.redefine = redefine;
                    inheritor.include = include;
                    inheritor.subclassOf = subclassOf;
                    inheritor.parent = this;
                    inheritor._includedCtors = this._includedCtors ? this._includedCtors.slice(0) : [];
                    inheritor.prototype.constructor = inheritor;
                    inheritor.redefine(members);
                    return inheritor
                };
                return classImpl
            }();
        function createQueue(discardPendingTasks) {
            var _tasks = [],
                _busy = false;
            function exec() {
                while (_tasks.length) {
                    _busy = true;
                    var task = _tasks.shift(),
                        result = task();
                    if (result === undefined)
                        continue;
                    if (result.then) {
                        $.when(result).always(exec);
                        return
                    }
                    throw Error("Queued task returned unexpected result");
                }
                _busy = false
            }
            function add(task, removeTaskCallback) {
                if (!discardPendingTasks)
                    _tasks.push(task);
                else {
                    if (_tasks[0] && removeTaskCallback)
                        removeTaskCallback(_tasks[0]);
                    _tasks = [task]
                }
                if (!_busy)
                    exec()
            }
            function busy() {
                return _busy
            }
            return {
                    add: add,
                    busy: busy
                }
        }
        var parseUrl = function() {
                var a = document.createElement("a"),
                    props = ["protocol", "hostname", "port", "pathname", "search", "hash"];
                var normalizePath = function(value) {
                        if (value.charAt(0) !== "/")
                            value = "/" + value;
                        return value
                    };
                return function(url) {
                        a.href = url;
                        var result = {};
                        $.each(props, function() {
                            result[this] = a[this]
                        });
                        result.pathname = normalizePath(result.pathname);
                        return result
                    }
            }();
        global.DevExpress = global.DevExpress || {};
        var enqueueAsync = function(task) {
                var deferred = $.Deferred();
                setTimeout(function() {
                    deferred.resolve(task())
                }, 60);
                return deferred
            };
        var hideTopOverlayCallback = function() {
                var callbacks = [];
                return {
                        add: function(callback) {
                            var indexOfCallback = $.inArray(callback, callbacks);
                            if (indexOfCallback === -1)
                                callbacks.push(callback)
                        },
                        remove: function(callback) {
                            var indexOfCallback = $.inArray(callback, callbacks);
                            if (indexOfCallback !== -1)
                                callbacks.splice(indexOfCallback, 1)
                        },
                        fire: function() {
                            var callback = callbacks.pop(),
                                result = !!callback;
                            if (result)
                                callback();
                            return result
                        },
                        hasCallback: function() {
                            return callbacks.length > 0
                        },
                        reset: function() {
                            callbacks = []
                        }
                    }
            }();
        var overlayTargetContainer = function() {
                var defaultTargetContainer = "body";
                return function(targetContainer) {
                        if (arguments.length)
                            defaultTargetContainer = targetContainer;
                        return defaultTargetContainer
                    }
            }();
        $.extend(global.DevExpress, {
            VERSION: "14.1.7",
            abstract: function() {
                throw Error("Not implemented");
            },
            Class: Class,
            createQueue: createQueue,
            enqueue: createQueue().add,
            enqueueAsync: enqueueAsync,
            parseUrl: parseUrl,
            hideTopOverlayCallback: hideTopOverlayCallback,
            hardwareBackButton: $.Callbacks(),
            overlayTargetContainer: overlayTargetContainer,
            rtlEnabled: false
        })
    })(jQuery, this);
    /*! Module core, file inflector.js */
    (function($, DX, undefined) {
        var _normalize = function(text) {
                if (text === undefined || text === null)
                    return "";
                return String(text)
            };
        var _ucfirst = function(text) {
                return _normalize(text).charAt(0).toUpperCase() + text.substr(1)
            };
        var _chop = function(text) {
                return _normalize(text).replace(/([a-z\d])([A-Z])/g, "$1 $2").split(/[\s_-]+/)
            };
        var dasherize = function(text) {
                return $.map(_chop(text), function(p) {
                        return p.toLowerCase()
                    }).join("-")
            };
        var underscore = function(text) {
                return dasherize(text).replace(/-/g, "_")
            };
        var camelize = function(text, upperFirst) {
                return $.map(_chop(text), function(p, i) {
                        p = p.toLowerCase();
                        if (upperFirst || i > 0)
                            p = _ucfirst(p);
                        return p
                    }).join("")
            };
        var humanize = function(text) {
                return _ucfirst(dasherize(text).replace(/-/g, " "))
            };
        var titleize = function(text) {
                return $.map(_chop(text), function(p) {
                        return _ucfirst(p.toLowerCase())
                    }).join(" ")
            };
        DX.inflector = {
            dasherize: dasherize,
            camelize: camelize,
            humanize: humanize,
            titleize: titleize,
            underscore: underscore
        }
    })(jQuery, DevExpress);
    /*! Module core, file utils.common.js */
    (function($, DX, undefined) {
        var isDefined = function(object) {
                return object !== null && object !== undefined
            };
        var isString = function(object) {
                return $.type(object) === 'string'
            };
        var isNumber = function(object) {
                return typeof object === "number" && isFinite(object) || $.isNumeric(object)
            };
        var isObject = function(object) {
                return $.type(object) === 'object'
            };
        var isArray = function(object) {
                return $.type(object) === 'array'
            };
        var isDate = function(object) {
                return $.type(object) === 'date'
            };
        var isFunction = function(object) {
                return $.type(object) === 'function'
            };
        var isExponential = function(value) {
                return isNumber(value) && value.toString().indexOf('e') !== -1
            };
        var extendFromObject = function(target, source, overrideExistingValues) {
                target = target || {};
                for (var prop in source)
                    if (source.hasOwnProperty(prop)) {
                        var value = source[prop];
                        if (!(prop in target) || overrideExistingValues)
                            target[prop] = value
                    }
                return target
            };
        var clone = function() {
                function Clone(){}
                return function(obj) {
                        Clone.prototype = obj;
                        return new Clone
                    }
            }();
        var executeAsync = function(action, context) {
                var deferred = $.Deferred(),
                    normalizedContext = context || this;
                setTimeout(function() {
                    var result = action.call(normalizedContext);
                    if (result && result.done && $.isFunction(result.done))
                        result.done(function() {
                            deferred.resolveWith(normalizedContext)
                        });
                    else
                        deferred.resolveWith(normalizedContext)
                }, 0);
                return deferred.promise()
            };
        var stringFormat = function() {
                var s = arguments[0];
                for (var i = 0; i < arguments.length - 1; i++) {
                    var reg = new RegExp("\\{" + i + "\\}", "gm");
                    s = s.replace(reg, arguments[i + 1])
                }
                return s
            };
        var findBestMatches = function(targetFilter, items, mapFn) {
                var bestMatches = [],
                    maxMatchCount = 0;
                $.each(items, function(index, itemSrc) {
                    var matchCount = 0,
                        item = mapFn ? mapFn(itemSrc) : itemSrc;
                    $.each(targetFilter, function(paramName) {
                        var value = item[paramName];
                        if (value === undefined)
                            return;
                        if (value === targetFilter[paramName]) {
                            matchCount++;
                            return
                        }
                        matchCount = -1;
                        return false
                    });
                    if (matchCount < maxMatchCount)
                        return;
                    if (matchCount > maxMatchCount) {
                        bestMatches.length = 0;
                        maxMatchCount = matchCount
                    }
                    bestMatches.push(itemSrc)
                });
                return bestMatches
            };
        var preg_quote = function(str) {
                return (str + "").replace(/([\+\*\?\\\.\[\^\]\$\(\)\{\}\>\<\|\=\!\:])/g, "\\$1")
            };
        var replaceAll = function(text, searchToken, replacementToken) {
                return text.replace(new RegExp("(" + preg_quote(searchToken) + ")", "gi"), replacementToken)
            };
        var splitPair = function(raw) {
                switch (typeof raw) {
                    case"string":
                        return raw.split(/\s+/, 2);
                    case"object":
                        return [raw.x || raw.h, raw.y || raw.v];
                    case"number":
                        return [raw];
                    default:
                        return raw
                }
            };
        var stringPairToObject = function(raw) {
                var pair = splitPair(raw),
                    x = parseInt(pair && pair[0], 10),
                    y = parseInt(pair && pair[1], 10);
                if (!isFinite(x))
                    x = 0;
                if (!isFinite(y))
                    y = x;
                return {
                        x: x,
                        y: y
                    }
            };
        function icontains(elem, text) {
            var result = false;
            $.each($(elem).contents(), function(index, content) {
                if (content.nodeType === 3 && (content.textContent || content.nodeValue || "").toLowerCase().indexOf((text || "").toLowerCase()) > -1) {
                    result = true;
                    return false
                }
            });
            return result
        }
        $.expr[":"].dxicontains = $.expr.createPseudo(function(text) {
            return function(elem) {
                    return icontains(elem, text)
                }
        });
        function deepExtendArraySafe(target, changes) {
            var prevValue,
                newValue;
            for (var name in changes) {
                prevValue = target[name];
                newValue = changes[name];
                if (target === newValue)
                    continue;
                if ($.isPlainObject(newValue) && !(newValue instanceof $.Event))
                    target[name] = deepExtendArraySafe($.isPlainObject(prevValue) ? prevValue : {}, newValue);
                else if (newValue !== undefined)
                    target[name] = newValue
            }
            return target
        }
        function unwrapObservable(value) {
            if (DX.support.hasKo)
                return ko.utils.unwrapObservable(value);
            return value
        }
        DX.utils = {
            isDefined: isDefined,
            isString: isString,
            isNumber: isNumber,
            isObject: isObject,
            isArray: isArray,
            isDate: isDate,
            isFunction: isFunction,
            isExponential: isExponential,
            extendFromObject: extendFromObject,
            clone: clone,
            executeAsync: executeAsync,
            stringFormat: stringFormat,
            findBestMatches: findBestMatches,
            replaceAll: replaceAll,
            deepExtendArraySafe: deepExtendArraySafe,
            splitPair: splitPair,
            stringPairToObject: stringPairToObject,
            unwrapObservable: unwrapObservable
        }
    })(jQuery, DevExpress);
    /*! Module core, file utils.console.js */
    (function($, DX, undefined) {
        var logger = function() {
                var console = window.console;
                function info(text) {
                    if (!console || !$.isFunction(console.info))
                        return;
                    console.info(text)
                }
                function warn(text) {
                    if (!console || !$.isFunction(console.warn))
                        return;
                    console.warn(text)
                }
                function error(text) {
                    if (!console || !$.isFunction(console.error))
                        return;
                    console.error(text)
                }
                return {
                        info: info,
                        warn: warn,
                        error: error
                    }
            }();
        var debug = function() {
                function assert(condition, message) {
                    if (!condition)
                        throw new Error(message);
                }
                function assertParam(parameter, message) {
                    assert(parameter !== null && parameter !== undefined, message)
                }
                return {
                        assert: assert,
                        assertParam: assertParam
                    }
            }();
        $.extend(DX.utils, {
            logger: logger,
            debug: debug
        })
    })(jQuery, DevExpress);
    /*! Module core, file utils.math.js */
    (function($, DX, undefined) {
        var PI = Math.PI,
            LN10 = Math.LN10;
        var cos = Math.cos,
            sin = Math.sin,
            abs = Math.abs,
            log = Math.log,
            floor = Math.floor,
            ceil = Math.ceil,
            max = Math.max,
            min = Math.min,
            isNaN = window.isNaN,
            Number = window.Number,
            NaN = window.NaN;
        var isNumber = DX.utils.isNumber,
            isExponential = DX.utils.isExponential;
        var getPrecision = function(value) {
                var stringFraction,
                    stringValue = value.toString(),
                    pointIndex = stringValue.indexOf('.'),
                    startIndex,
                    precision;
                if (isExponential(value)) {
                    precision = getDecimalOrder(value);
                    if (precision < 0)
                        return Math.abs(precision);
                    else
                        return 0
                }
                if (pointIndex !== -1) {
                    startIndex = pointIndex + 1;
                    stringFraction = stringValue.substring(startIndex, startIndex + 20);
                    return stringFraction.length
                }
                return 0
            };
        var getLog = function(value, base) {
                if (!value)
                    return 0;
                return Math.log(value) / Math.log(base)
            };
        var raiseTo = function(power, base) {
                return Math.pow(base, power)
            };
        var sign = function(value) {
                if (value === 0)
                    return 0;
                return value / abs(value)
            };
        var normalizeAngle = function(angle) {
                return (angle % 360 + 360) % 360
            };
        var convertAngleToRendererSpace = function(angle) {
                return 90 - angle
            };
        var degreesToRadians = function(value) {
                return PI * value / 180
            };
        var getCosAndSin = function(angle) {
                var angleInRadians = degreesToRadians(angle);
                return {
                        cos: cos(angleInRadians),
                        sin: sin(angleInRadians)
                    }
            };
        var DECIMAL_ORDER_THRESHOLD = 1E-14;
        var getDecimalOrder = function(number) {
                var n = abs(number),
                    cn;
                if (!isNaN(n)) {
                    if (n > 0) {
                        n = log(n) / LN10;
                        cn = ceil(n);
                        return cn - n < DECIMAL_ORDER_THRESHOLD ? cn : floor(n)
                    }
                    return 0
                }
                return NaN
            };
        var getAppropriateFormat = function(start, end, count) {
                var order = max(getDecimalOrder(start), getDecimalOrder(end)),
                    precision = -getDecimalOrder(abs(end - start) / count),
                    format;
                if (!isNaN(order) && !isNaN(precision)) {
                    if (abs(order) <= 4) {
                        format = 'fixedPoint';
                        precision < 0 && (precision = 0);
                        precision > 4 && (precision = 4)
                    }
                    else {
                        format = 'exponential';
                        precision += order - 1;
                        precision > 3 && (precision = 3)
                    }
                    return {
                            format: format,
                            precision: precision
                        }
                }
                return null
            };
        var getFraction = function(value) {
                var valueString,
                    dotIndex;
                if (isNumber(value)) {
                    valueString = value.toString();
                    dotIndex = valueString.indexOf('.');
                    if (dotIndex >= 0)
                        if (isExponential(value))
                            return valueString.substr(dotIndex + 1, valueString.indexOf('e') - dotIndex - 1);
                        else {
                            valueString = value.toFixed(20);
                            return valueString.substr(dotIndex + 1, valueString.length - dotIndex + 1)
                        }
                }
                return ''
            };
        var getSignificantDigitPosition = function(value) {
                var fraction = getFraction(value),
                    i;
                if (fraction)
                    for (i = 0; i < fraction.length; i++)
                        if (fraction.charAt(i) !== '0')
                            return i + 1;
                return 0
            };
        var adjustValue = function(value) {
                var fraction = getFraction(value),
                    nextValue,
                    i;
                if (fraction)
                    for (i = 1; i <= fraction.length; i++) {
                        nextValue = roundValue(value, i);
                        if (nextValue !== 0 && fraction[i - 2] && fraction[i - 1] && fraction[i - 2] === fraction[i - 1])
                            return nextValue
                    }
                return value
            };
        var roundValue = function(value, precision) {
                if (precision > 20)
                    precision = 20;
                if (isNumber(value))
                    if (isExponential(value))
                        return Number(value.toExponential(precision));
                    else
                        return Number(value.toFixed(precision))
            };
        var applyPrecisionByMinDelta = function(min, delta, value) {
                var minPrecision = getPrecision(min),
                    deltaPrecision = getPrecision(delta);
                return roundValue(value, minPrecision < deltaPrecision ? deltaPrecision : minPrecision)
            };
        $.extend(DX.utils, {
            getLog: getLog,
            raiseTo: raiseTo,
            sign: sign,
            normalizeAngle: normalizeAngle,
            convertAngleToRendererSpace: convertAngleToRendererSpace,
            degreesToRadians: degreesToRadians,
            getCosAndSin: getCosAndSin,
            getDecimalOrder: getDecimalOrder,
            getAppropriateFormat: getAppropriateFormat,
            getFraction: getFraction,
            adjustValue: adjustValue,
            roundValue: roundValue,
            applyPrecisionByMinDelta: applyPrecisionByMinDelta,
            getSignificantDigitPosition: getSignificantDigitPosition
        });
        DX.utils.getPrecision = getPrecision
    })(jQuery, DevExpress);
    /*! Module core, file utils.date.js */
    (function($, DX, undefined) {
        var isObject = DX.utils.isObject,
            isString = DX.utils.isString,
            isDate = DX.utils.isDate;
        var dateUnitIntervals = ['millisecond', 'second', 'minute', 'hour', 'day', 'week', 'month', 'quarter', 'year'];
        var addSubValues = function(value1, value2, isSub) {
                return value1 + (isSub ? -1 : 1) * value2
            };
        var toMilliseconds = function(value) {
                switch (value) {
                    case'millisecond':
                        return 1;
                    case'second':
                        return toMilliseconds('millisecond') * 1000;
                    case'minute':
                        return toMilliseconds('second') * 60;
                    case'hour':
                        return toMilliseconds('minute') * 60;
                    case'day':
                        return toMilliseconds('hour') * 24;
                    case'week':
                        return toMilliseconds('day') * 7;
                    case'month':
                        return toMilliseconds('day') * 30;
                    case'quarter':
                        return toMilliseconds('month') * 3;
                    case'year':
                        return toMilliseconds('day') * 365;
                    default:
                        return 0
                }
            };
        function parseISO8601(isoString) {
            var result = new Date(0);
            var chunks = isoString.replace("Z", "").split("T"),
                date = String(chunks[0]).split("-"),
                time = String(chunks[1]).split(":");
            var year,
                month,
                day,
                hours,
                minutes,
                seconds,
                milliseconds;
            year = Number(date[0]);
            month = Number(date[1]) - 1;
            day = Number(date[2]);
            result.setUTCDate(day);
            result.setUTCMonth(month);
            result.setUTCFullYear(year);
            if (time.length) {
                hours = Number(time[0]);
                minutes = Number(time[1]);
                seconds = Number(String(time[2]).split(".")[0]);
                milliseconds = Number(String(time[2]).split(".")[1]) || 0;
                result.setUTCHours(hours);
                result.setUTCMinutes(minutes);
                result.setUTCSeconds(seconds);
                result.setUTCMilliseconds(milliseconds)
            }
            return result
        }
        function formatISO8601(date) {
            function pad(n) {
                if (n < 10)
                    return "0".concat(n);
                return String(n)
            }
            return [date.getFullYear(), "-", pad(date.getMonth() + 1), "-", pad(date.getDate()), "T", pad(date.getHours()), ":", pad(date.getMinutes()), ":", pad(date.getSeconds()), "Z"].join("")
        }
        var convertMillisecondsToDateUnits = function(value) {
                var i,
                    dateUnitCount,
                    dateUnitInterval,
                    dateUnitIntervals = ['millisecond', 'second', 'minute', 'hour', 'day', 'month', 'year'],
                    result = {};
                for (i = dateUnitIntervals.length - 1; i >= 0; i--) {
                    dateUnitInterval = dateUnitIntervals[i];
                    dateUnitCount = Math.floor(value / toMilliseconds(dateUnitInterval));
                    if (dateUnitCount > 0) {
                        result[dateUnitInterval + 's'] = dateUnitCount;
                        value -= convertDateUnitToMilliseconds(dateUnitInterval, dateUnitCount)
                    }
                }
                return result
            };
        var convertDateTickIntervalToMilliseconds = function(tickInterval) {
                var milliseconds = 0;
                if (isObject(tickInterval))
                    $.each(tickInterval, function(key, value) {
                        milliseconds += convertDateUnitToMilliseconds(key.substr(0, key.length - 1), value)
                    });
                if (isString(tickInterval))
                    milliseconds = convertDateUnitToMilliseconds(tickInterval, 1);
                return milliseconds
            };
        var convertDateUnitToMilliseconds = function(dateUnit, count) {
                return toMilliseconds(dateUnit) * count
            };
        var getDateUnitInterval = function(tickInterval) {
                var maxInterval = -1,
                    i;
                if (isString(tickInterval))
                    return tickInterval;
                if (isObject(tickInterval)) {
                    $.each(tickInterval, function(key, value) {
                        for (i = 0; i < dateUnitIntervals.length; i++)
                            if (value && (key === dateUnitIntervals[i] + 's' || key === dateUnitIntervals[i]) && maxInterval < i)
                                maxInterval = i
                    });
                    return dateUnitIntervals[maxInterval]
                }
                return ''
            };
        var correctDateWithUnitBeginning = function(date, dateInterval) {
                var dayMonth,
                    firstQuarterMonth,
                    dateUnitInterval = getDateUnitInterval(dateInterval);
                switch (dateUnitInterval) {
                    case'second':
                        date.setMilliseconds(0);
                        break;
                    case'minute':
                        date.setSeconds(0, 0);
                        break;
                    case'hour':
                        date.setMinutes(0, 0, 0);
                        break;
                    case'year':
                        date.setMonth(0);
                    case'month':
                        date.setDate(1);
                    case'day':
                        date.setHours(0, 0, 0, 0);
                        break;
                    case'week':
                        dayMonth = date.getDate();
                        if (date.getDay() !== 0)
                            dayMonth += 7 - date.getDay();
                        date.setDate(dayMonth);
                        date.setHours(0, 0, 0, 0);
                        break;
                    case'quarter':
                        firstQuarterMonth = DX.formatHelper.getFirstQuarterMonth(date.getMonth());
                        if (date.getMonth() !== firstQuarterMonth)
                            date.setMonth(firstQuarterMonth);
                        date.setDate(1);
                        date.setHours(0, 0, 0, 0);
                        break
                }
            };
        var getDatesDifferences = function(date1, date2) {
                var differences,
                    counter = 0;
                differences = {
                    year: date1.getFullYear() !== date2.getFullYear(),
                    month: date1.getMonth() !== date2.getMonth(),
                    day: date1.getDate() !== date2.getDate(),
                    hour: date1.getHours() !== date2.getHours(),
                    minute: date1.getMinutes() !== date2.getMinutes(),
                    second: date1.getSeconds() !== date2.getSeconds()
                };
                $.each(differences, function(key, value) {
                    if (value)
                        counter++
                });
                differences.count = counter;
                return differences
            };
        var addInterval = function(value, interval, isNegative) {
                var result = null,
                    intervalObject;
                if (isDate(value)) {
                    intervalObject = isString(interval) ? getDateIntervalByString(interval.toLowerCase()) : interval;
                    result = new Date(value.getTime());
                    if (intervalObject.years)
                        result.setFullYear(addSubValues(result.getFullYear(), intervalObject.years, isNegative));
                    if (intervalObject.quarters)
                        result.setMonth(addSubValues(result.getMonth(), 3 * intervalObject.quarters, isNegative));
                    if (intervalObject.months)
                        result.setMonth(addSubValues(result.getMonth(), intervalObject.months, isNegative));
                    if (intervalObject.weeks)
                        result.setDate(addSubValues(result.getDate(), 7 * intervalObject.weeks, isNegative));
                    if (intervalObject.days)
                        result.setDate(addSubValues(result.getDate(), intervalObject.days, isNegative));
                    if (intervalObject.hours)
                        result.setHours(addSubValues(result.getHours(), intervalObject.hours, isNegative));
                    if (intervalObject.minutes)
                        result.setMinutes(addSubValues(result.getMinutes(), intervalObject.minutes, isNegative));
                    if (intervalObject.seconds)
                        result.setSeconds(addSubValues(result.getSeconds(), intervalObject.seconds, isNegative));
                    if (intervalObject.milliseconds)
                        result.setMilliseconds(addSubValues(value.getMilliseconds(), intervalObject.milliseconds, isNegative))
                }
                else
                    result = addSubValues(value, interval, isNegative);
                return result
            };
        var getDateIntervalByString = function(intervalString) {
                var result = {};
                switch (intervalString) {
                    case'year':
                        result.years = 1;
                        break;
                    case'month':
                        result.months = 1;
                        break;
                    case'quarter':
                        result.months = 3;
                        break;
                    case'week':
                        result.days = 7;
                        break;
                    case'day':
                        result.days = 1;
                        break;
                    case'hour':
                        result.hours = 1;
                        break;
                    case'minute':
                        result.minutes = 1;
                        break;
                    case'second':
                        result.seconds = 1;
                        break;
                    case'millisecond':
                        result.milliseconds = 1;
                        break
                }
                return result
            };
        var sameMonthAndYear = function(date1, date2) {
                return date1 && date2 && date1.getFullYear() === date2.getFullYear() && date1.getMonth() === date2.getMonth()
            };
        var getFirstMonthDate = function(date) {
                return new Date(date.getFullYear(), date.getMonth(), 1)
            };
        var dateInRange = function(date, min, max) {
                return normalizeDate(date, min, max) === date
            };
        var normalizeDate = function(date, min, max) {
                var normalizedDate = date;
                if (date < min)
                    normalizedDate = min;
                else if (date > max)
                    normalizedDate = max;
                return normalizedDate
            };
        var getPower = function(value) {
                return value.toExponential().split("e")[1]
            };
        $.extend(DX.utils, {
            dateUnitIntervals: dateUnitIntervals,
            parseIso8601Date: parseISO8601,
            formatIso8601Date: formatISO8601,
            convertMillisecondsToDateUnits: convertMillisecondsToDateUnits,
            convertDateTickIntervalToMilliseconds: convertDateTickIntervalToMilliseconds,
            convertDateUnitToMilliseconds: convertDateUnitToMilliseconds,
            getDateUnitInterval: getDateUnitInterval,
            getDatesDifferences: getDatesDifferences,
            correctDateWithUnitBeginning: correctDateWithUnitBeginning,
            addInterval: addInterval,
            getDateIntervalByString: getDateIntervalByString,
            sameMonthAndYear: sameMonthAndYear,
            getFirstMonthDate: getFirstMonthDate,
            dateInRange: dateInRange,
            normalizeDate: normalizeDate,
            getPower: getPower
        })
    })(jQuery, DevExpress);
    /*! Module core, file utils.dom.js */
    (function($, DX, undefined) {
        var IOS_APP_BAR_HEIGHT = "20px";
        var timeRedrawOnResize = 100;
        var createResizeHandler = function(callback) {
                var $window = $(window),
                    timeout;
                var debug_callback = arguments[1];
                var handler = function() {
                        var width = $window.width(),
                            height = $window.height();
                        clearTimeout(timeout);
                        timeout = setTimeout(function() {
                            $window.width() === width && $window.height() === height && callback();
                            debug_callback && debug_callback()
                        }, timeRedrawOnResize)
                    };
                handler.stop = function() {
                    clearTimeout(timeout);
                    return this
                };
                return handler
            };
        var windowResizeCallbacks = function() {
                var prevSize,
                    callbacks = $.Callbacks(),
                    jqWindow = $(window),
                    resizeEventHandlerAttached = false,
                    originalCallbacksAdd = callbacks.add,
                    originalCallbacksRemove = callbacks.remove;
                var formatSize = function() {
                        return [jqWindow.width(), jqWindow.height()].join()
                    };
                var handleResize = function() {
                        var now = formatSize();
                        if (now === prevSize)
                            return;
                        prevSize = now;
                        setTimeout(callbacks.fire)
                    };
                prevSize = formatSize();
                callbacks.add = function() {
                    var result = originalCallbacksAdd.apply(callbacks, arguments);
                    if (!resizeEventHandlerAttached && callbacks.has()) {
                        jqWindow.on("resize", handleResize);
                        resizeEventHandlerAttached = true
                    }
                    return result
                };
                callbacks.remove = function() {
                    var result = originalCallbacksRemove.apply(callbacks, arguments);
                    if (!callbacks.has() && resizeEventHandlerAttached) {
                        jqWindow.off("resize", handleResize);
                        resizeEventHandlerAttached = false
                    }
                    return result
                };
                return callbacks
            }();
        var resetActiveElement = function() {
                var activeElement = document.activeElement;
                if (activeElement && activeElement !== document.body && activeElement.blur)
                    activeElement.blur()
            };
        var createMarkupFromString = function(str) {
                var tempElement = $("<div />");
                if (window.WinJS)
                    WinJS.Utilities.setInnerHTMLUnsafe(tempElement.get(0), str);
                else
                    tempElement.append(str);
                return tempElement.contents()
            };
        var initMobileViewport = function(options) {
                options = $.extend({}, options);
                var device = DX.devices.current();
                var realDevice = DX.devices.real();
                var allowZoom = options.allowZoom,
                    allowPan = options.allowPan,
                    allowSelection = "allowSelection" in options ? options.allowSelection : device.platform == "desktop";
                DX.overlayTargetContainer(".dx-viewport");
                var metaSelector = "meta[name=viewport]";
                if (!$(metaSelector).length)
                    $("<meta />").attr("name", "viewport").appendTo("head");
                var metaVerbs = ["width=device-width"],
                    msTouchVerbs = [];
                if (allowZoom)
                    msTouchVerbs.push("pinch-zoom");
                else
                    metaVerbs.push("initial-scale=1.0", "maximum-scale=1.0, user-scalable=no");
                if (allowPan)
                    msTouchVerbs.push("pan-x", "pan-y");
                if (!allowPan && !allowZoom)
                    $("html, body").css({
                        "-ms-content-zooming": "none",
                        "-ms-user-select": "none",
                        overflow: "hidden"
                    });
                else
                    $("html").css("-ms-overflow-style", "-ms-autohiding-scrollbar");
                if (!allowSelection) {
                    if (realDevice.ios)
                        $(document).on("selectstart", function() {
                            return false
                        });
                    $(".dx-viewport").css("user-select", "none")
                }
                $(metaSelector).attr("content", metaVerbs.join());
                $("html").css("-ms-touch-action", msTouchVerbs.join(" ") || "none");
                if (DX.support.touch)
                    $(document).off(".dxInitMobileViewport").on("touchmove.dxInitMobileViewport", function(e) {
                        var count = e.originalEvent.touches.length,
                            zoomDisabled = !allowZoom && count > 1,
                            panDisabled = !allowPan && count === 1 && !e.isScrollingEvent;
                        if (zoomDisabled || panDisabled)
                            e.preventDefault()
                    });
                realDevice = DX.devices.real();
                if (realDevice.ios) {
                    var isPhoneGap = document.location.protocol === "file:";
                    if (!isPhoneGap)
                        windowResizeCallbacks.add(function() {
                            var windowWidth = $(window).width();
                            $("body").width(windowWidth)
                        });
                    else if (realDevice.version[0] > 6) {
                        $(".dx-viewport").css("position", "relative");
                        $("body").css({"box-sizing": "border-box"});
                        $("body").css("padding-top", IOS_APP_BAR_HEIGHT);
                        if (realDevice.version[0] === 7 && realDevice.version[1] < 1) {
                            var setDeviceHeight = function() {
                                    var deviceHeight = "height=device-" + (Math.abs(window.orientation) === 90 ? "width" : "height");
                                    $(metaSelector).attr("content", metaVerbs.join() + "," + deviceHeight)
                                };
                            $(window).on("orientationchange", setDeviceHeight);
                            setDeviceHeight()
                        }
                    }
                }
            };
        var triggerVisibilityChangeEvent = function(event) {
                return function(element) {
                        $(element || "body").find(".dx-visibility-change-handler").each(function() {
                            $(this).triggerHandler(event)
                        })
                    }
            };
        $.extend(DX.utils, {
            createResizeHandler: createResizeHandler,
            windowResizeCallbacks: windowResizeCallbacks,
            resetActiveElement: resetActiveElement,
            createMarkupFromString: createMarkupFromString,
            triggerShownEvent: triggerVisibilityChangeEvent("dxshown"),
            triggerHidingEvent: triggerVisibilityChangeEvent("dxhiding"),
            initMobileViewport: initMobileViewport
        });
        DX.utils.__timeRedrawOnResize = timeRedrawOnResize
    })(jQuery, DevExpress);
    /*! Module core, file utils.graphics.js */
    (function($, DX, undefined) {
        var isFunction = DX.utils.isFunction,
            iDevice = /iphone|ipad/.test(navigator.userAgent.toLowerCase());
        var processSeriesTemplate = function(seriesTemplate, items) {
                var customizeSeries = isFunction(seriesTemplate.customizeSeries) ? seriesTemplate.customizeSeries : $.noop,
                    nameField = seriesTemplate.nameField || 'series',
                    generatedSeries = {},
                    seriesOrder = [],
                    series;
                for (var i = 0, length = items.length; i < length; i++) {
                    var data = items[i];
                    if (nameField in data) {
                        series = generatedSeries[data[nameField]];
                        if (!series) {
                            series = generatedSeries[data[nameField]] = {
                                name: data[nameField],
                                data: []
                            };
                            seriesOrder.push(series.name)
                        }
                        series.data.push(data)
                    }
                }
                return $.map(seriesOrder, function(orderedName) {
                        var group = generatedSeries[orderedName],
                            seriesOptions = customizeSeries.call(null, group.name);
                        return $.extend(group, seriesOptions)
                    })
            };
        var getNextDefsSvgId = function() {
                var numDefsSvgElements = 1;
                return function() {
                        return 'DevExpress_' + numDefsSvgElements++
                    }
            }();
        var getRootOffset = function(renderer) {
                var node,
                    result = {
                        left: 0,
                        top: 0
                    },
                    pointTransform,
                    root = renderer.getRoot();
                if (root) {
                    node = root.element;
                    if (node.getScreenCTM && !iDevice) {
                        var ctm = node.getScreenCTM();
                        if (ctm) {
                            pointTransform = node.createSVGPoint().matrixTransform(ctm);
                            result.left = pointTransform.x + (document.body.scrollLeft || document.documentElement.scrollLeft);
                            result.top = pointTransform.y + (document.body.scrollTop || document.documentElement.scrollTop)
                        }
                        else {
                            result.left = document.body.scrollLeft || document.documentElement.scrollLeft;
                            result.top = document.body.scrollTop || document.documentElement.scrollTop
                        }
                    }
                    else
                        result = $(node).offset()
                }
                return result
            };
        $.extend(DX.utils, {
            processSeriesTemplate: processSeriesTemplate,
            getNextDefsSvgId: getNextDefsSvgId,
            getRootOffset: getRootOffset
        })
    })(jQuery, DevExpress);
    /*! Module core, file utils.arrays.js */
    (function($, DX, undefined) {
        var wrapToArray = function(entity) {
                return $.isArray(entity) ? entity : [entity]
            };
        var removeDublicates = function(from, what) {
                if (!$.isArray(from) || from.length === 0)
                    return [];
                if (!$.isArray(what) || what.length === 0)
                    return from.slice();
                var result = [];
                $.each(from, function(_, value) {
                    var bIndex = $.inArray(value, what);
                    if (bIndex === -1)
                        result.push(value)
                });
                return result
            };
        $.extend(DX.utils, {
            wrapToArray: wrapToArray,
            removeDublicates: removeDublicates
        })
    })(jQuery, DevExpress);
    /*! Module core, file devices.js */
    (function($, DX, undefined) {
        var KNOWN_UA_TABLE = {
                iPhone: "iPhone",
                iPhone5: "iPhone 5",
                iPad: "iPad",
                iPadMini: "iPad Mini",
                androidPhone: "Android Mobile",
                androidTablet: "Android",
                win8: "MSAppHost",
                win8Phone: "Windows Phone 8",
                msSurface: "MSIE ARM Tablet PC",
                desktop: "desktop",
                tizen: "Tizen Mobile"
            };
        var DEFAULT_DEVICE = {
                deviceType: "",
                platform: "",
                version: [],
                phone: false,
                tablet: false,
                android: false,
                ios: false,
                win8: false,
                tizen: false,
                generic: false
            };
        var GENERIC_DEVICE = $.extend(DEFAULT_DEVICE, {
                platform: "generic",
                deviceType: "desktop",
                generic: true
            });
        var uaParsers = {
                win8: function(userAgent) {
                    var isPhone = /windows phone/i.test(userAgent),
                        isTablet = !isPhone && /arm(.*)trident/i.test(userAgent),
                        isDesktop = !isPhone && !isTablet && /msapphost/i.test(userAgent);
                    if (!(isPhone || isTablet || isDesktop))
                        return;
                    var matches = userAgent.match(/windows phone (\d+).(\d+)/i) || userAgent.match(/windows nt (\d+).(\d+)/i),
                        version = matches ? [parseInt(matches[1], 10), parseInt(matches[2], 10)] : [];
                    return {
                            deviceType: isPhone ? "phone" : isTablet ? "tablet" : "desktop",
                            platform: "win8",
                            version: version
                        }
                },
                ios: function(userAgent) {
                    if (!/ip(hone|od|ad)/i.test(userAgent))
                        return;
                    var isPhone = /ip(hone|od)/i.test(userAgent);
                    var matches = userAgent.match(/os (\d+)_(\d+)_?(\d+)?/i);
                    var version = matches ? [parseInt(matches[1], 10), parseInt(matches[2], 10), parseInt(matches[3] || 0, 10)] : [];
                    return {
                            deviceType: isPhone ? "phone" : "tablet",
                            platform: "ios",
                            version: version
                        }
                },
                android: function(userAgent) {
                    if (!/android|htc_|silk/i.test(userAgent))
                        return;
                    var isPhone = /mobile/i.test(userAgent);
                    var matches = userAgent.match(/android (\d+)\.(\d+)\.?(\d+)?/i);
                    var version = matches ? [parseInt(matches[1], 10), parseInt(matches[2], 10), parseInt(matches[3] || 0, 10)] : [];
                    return {
                            deviceType: isPhone ? "phone" : "tablet",
                            platform: "android",
                            version: version
                        }
                },
                tizen: function(userAgent) {
                    if (!/tizen/i.test(userAgent))
                        return;
                    var isPhone = /mobile/i.test(userAgent);
                    var matches = userAgent.match(/tizen (\d+)\.(\d+)/i);
                    var version = matches ? [parseInt(matches[1], 10), parseInt(matches[2], 10)] : [];
                    return {
                            deviceType: isPhone ? "phone" : "tablet",
                            platform: "tizen",
                            version: version
                        }
                }
            };
        DX.Devices = DX.Class.inherit({
            ctor: function(options) {
                this._window = options && options.window || window;
                this._realDevice = this._getDevice();
                this._currentDevice = undefined;
                this._currentOrientation = undefined;
                this.orientationChanged = $.Callbacks();
                this._recalculateOrientation();
                DX.utils.windowResizeCallbacks.add($.proxy(this._recalculateOrientation, this))
            },
            current: function(deviceOrName) {
                if (deviceOrName) {
                    this._currentDevice = this._getDevice(deviceOrName);
                    DX.ui.themes.init({_autoInit: true})
                }
                else {
                    if (!this._currentDevice) {
                        deviceOrName = undefined;
                        try {
                            deviceOrName = this._getDeviceOrNameFromWindowScope()
                        }
                        catch(e) {
                            deviceOrName = this._getDeviceNameFromSessionStorage()
                        }
                        finally {
                            if (!deviceOrName)
                                deviceOrName = this._getDeviceNameFromSessionStorage()
                        }
                        this._currentDevice = this._getDevice(deviceOrName)
                    }
                    return this._currentDevice
                }
            },
            real: function() {
                var forceDevice = arguments[0];
                if ($.isPlainObject(forceDevice)) {
                    $.extend(this._realDevice, forceDevice);
                    return
                }
                return $.extend({}, this._realDevice)
            },
            orientation: function() {
                return this._currentOrientation
            },
            isRippleEmulator: function() {
                return !!this._window.tinyHippos
            },
            attachCssClasses: function(element, device) {
                var realDevice = this._realDevice,
                    $element = $(element);
                device = device || this.current();
                if (device.deviceType)
                    $element.addClass("dx-device-" + device.deviceType);
                $element.addClass("dx-device-" + realDevice.platform);
                if (realDevice.version && realDevice.version.length)
                    $element.addClass("dx-device-" + realDevice.platform + "-" + realDevice.version[0]);
                if (DX.devices.isSimulator())
                    $element.addClass("dx-simulator");
                if (DX.rtlEnabled)
                    $element.addClass("dx-rtl")
            },
            isSimulator: function() {
                try {
                    return this._isSimulator || this._window.top !== this._window.self && this._window.top["dx-force-device"] || this.isRippleEmulator()
                }
                catch(e) {
                    return false
                }
            },
            forceSimulator: function() {
                this._isSimulator = true
            },
            _getDevice: function(deviceName) {
                if (deviceName === "genericPhone")
                    deviceName = {
                        deviceType: "phone",
                        platform: "generic",
                        generic: true
                    };
                if ($.isPlainObject(deviceName))
                    return this._fromConfig(deviceName);
                else {
                    var ua;
                    if (deviceName) {
                        ua = KNOWN_UA_TABLE[deviceName];
                        if (!ua)
                            throw Error("Unknown device");
                    }
                    else
                        ua = navigator.userAgent;
                    return this._fromUA(ua)
                }
            },
            _getDeviceOrNameFromWindowScope: function() {
                var result;
                if (this._window.top["dx-force-device-object"] || this._window.top["dx-force-device"])
                    result = this._window.top["dx-force-device-object"] || this._window.top["dx-force-device"];
                return result
            },
            _getDeviceNameFromSessionStorage: function() {
                return this._window.sessionStorage && (sessionStorage.getItem("dx-force-device") || sessionStorage.getItem("dx-simulator-device"))
            },
            _fromConfig: function(config) {
                var shortcuts = {
                        phone: config.deviceType === "phone",
                        tablet: config.deviceType === "tablet",
                        android: config.platform === "android",
                        ios: config.platform === "ios",
                        win8: config.platform === "win8",
                        tizen: config.platform === "tizen",
                        generic: config.platform === "generic"
                    };
                return $.extend({}, DEFAULT_DEVICE, this._currentDevice, shortcuts, config)
            },
            _fromUA: function(ua) {
                var config;
                $.each(uaParsers, function(platform, parser) {
                    config = parser(ua);
                    return !config
                });
                if (config)
                    return this._fromConfig(config);
                return GENERIC_DEVICE
            },
            _changeOrientation: function() {
                var $window = $(this._window),
                    orientation = $window.height() > $window.width() ? "portrait" : "landscape";
                if (this._currentOrientation === orientation)
                    return;
                this._currentOrientation = orientation;
                this.orientationChanged.fire({orientation: orientation})
            },
            _recalculateOrientation: function() {
                var windowWidth = $(this._window).width();
                if (this._currentWidth === windowWidth)
                    return;
                this._currentWidth = windowWidth;
                this._changeOrientation()
            }
        });
        DX.devices = new DX.Devices
    })(jQuery, DevExpress);
    /*! Module core, file browser.js */
    (function($, DX, global, undefined) {
        var webkitRegExp = /(webkit)[ \/]([\w.]+)/,
            operaRegExp = /(opera)(?:.*version)?[ \/]([\w.]+)/,
            ieRegExp = /(msie) (\d{1,2}\.\d)/,
            ie11RegExp = /(trident).*rv:(\d{1,2}\.\d)/,
            mozillaRegExp = /(mozilla)(?:.*? rv:([\w.]+))?/;
        var ua = navigator.userAgent.toLowerCase();
        var browser = function() {
                var result = {},
                    matches = webkitRegExp.exec(ua) || operaRegExp.exec(ua) || ieRegExp.exec(ua) || ie11RegExp.exec(ua) || ua.indexOf("compatible") < 0 && mozillaRegExp.exec(ua) || [],
                    browserName = matches[1],
                    browserVersion = matches[2];
                if (browserName === "trident")
                    browserName = "msie";
                if (browserName) {
                    result[browserName] = true;
                    result.version = browserVersion
                }
                return result
            }();
        DX.browser = browser
    })(jQuery, DevExpress, this);
    /*! Module core, file support.js */
    (function($, DX, window) {
        var cssPrefixes = ["", "Webkit", "Moz", "O", "ms"],
            styles = document.createElement("dx").style;
        var transitionEndEventNames = {
                WebkitTransition: 'webkitTransitionEnd',
                MozTransition: 'transitionend',
                OTransition: 'oTransitionEnd',
                msTransition: 'MsTransitionEnd',
                transition: 'transitionend'
            };
        var styleProp = function(prop) {
                prop = DX.inflector.camelize(prop, true);
                for (var i = 0, cssPrefixesCount = cssPrefixes.length; i < cssPrefixesCount; i++) {
                    var specific = cssPrefixes[i] + prop;
                    if (specific in styles)
                        return specific
                }
            };
        var supportProp = function(prop) {
                return !!styleProp(prop)
            };
        var isNativeScrollingSupported = function(device) {
                var realDevice = DX.devices.real(),
                    realPlatform = realDevice.platform,
                    realVersion = realDevice.version,
                    isObsoleteAndroid = realVersion && realVersion[0] < 4 && realPlatform === "android",
                    isNativeScrollDevice = !isObsoleteAndroid && $.inArray(realPlatform, ["ios", "android", "win8"]) > -1;
                return isNativeScrollDevice
            };
        DX.support = {
            touch: "ontouchstart" in window,
            pointer: window.navigator.pointerEnabled,
            transform3d: supportProp("transform"),
            transition: supportProp("transition"),
            transitionEndEventName: transitionEndEventNames[styleProp("transition")],
            animation: supportProp("animation"),
            nativeScrolling: isNativeScrollingSupported(),
            winJS: "WinJS" in window,
            styleProp: styleProp,
            supportProp: supportProp,
            hasKo: !!window.ko,
            hasNg: !window.ko && !!window.angular,
            inputType: function(type) {
                if (type === "text")
                    return true;
                var input = document.createElement("input");
                try {
                    input.setAttribute("type", type);
                    input.value = "wrongValue";
                    return !input.value
                }
                catch(e) {
                    return false
                }
            }
        }
    })(jQuery, DevExpress, this);
    /*! Module core, file position.js */
    (function($, DX, undefined) {
        var horzRe = /left|right/,
            vertRe = /top|bottom/,
            collisionRe = /fit|flip|none/;
        var normalizeAlign = function(raw) {
                var result = {
                        h: "center",
                        v: "center"
                    };
                var pair = DX.utils.splitPair(raw);
                if (pair)
                    $.each(pair, function() {
                        var w = String(this).toLowerCase();
                        if (horzRe.test(w))
                            result.h = w;
                        else if (vertRe.test(w))
                            result.v = w
                    });
                return result
            };
        var normalizeOffset = function(raw) {
                var values = DX.utils.stringPairToObject(raw);
                return {
                        h: values.x,
                        v: values.y
                    }
            };
        var normalizeCollision = function(raw) {
                var pair = DX.utils.splitPair(raw),
                    h = String(pair && pair[0]).toLowerCase(),
                    v = String(pair && pair[1]).toLowerCase();
                if (!collisionRe.test(h))
                    h = "none";
                if (!collisionRe.test(v))
                    v = h;
                return {
                        h: h,
                        v: v
                    }
            };
        var getAlignFactor = function(align) {
                switch (align) {
                    case"center":
                        return 0.5;
                    case"right":
                    case"bottom":
                        return 1;
                    default:
                        return 0
                }
            };
        var inverseAlign = function(align) {
                switch (align) {
                    case"left":
                        return "right";
                    case"right":
                        return "left";
                    case"top":
                        return "bottom";
                    case"bottom":
                        return "top";
                    default:
                        return align
                }
            };
        var calculateOversize = function(data, bounds) {
                var oversize = 0;
                if (data.myLocation < bounds.min)
                    oversize += bounds.min - data.myLocation;
                if (data.myLocation > bounds.max)
                    oversize += data.myLocation - bounds.max;
                return oversize
            };
        var initMyLocation = function(data) {
                data.myLocation = data.atLocation + getAlignFactor(data.atAlign) * data.atSize - getAlignFactor(data.myAlign) * data.mySize + data.offset
            };
        var decolliders = {
                fit: function(data, bounds) {
                    var result = false;
                    if (data.myLocation > bounds.max) {
                        data.myLocation = bounds.max;
                        result = true
                    }
                    if (data.myLocation < bounds.min) {
                        data.myLocation = bounds.min;
                        result = true
                    }
                    return result
                },
                flip: function(data, bounds) {
                    if (data.myAlign === "center" && data.atAlign === "center")
                        return false;
                    if (data.myLocation < bounds.min || data.myLocation > bounds.max) {
                        var inverseData = $.extend({}, data, {
                                myAlign: inverseAlign(data.myAlign),
                                atAlign: inverseAlign(data.atAlign),
                                offset: -data.offset
                            });
                        initMyLocation(inverseData);
                        inverseData.oversize = calculateOversize(inverseData, bounds);
                        if (inverseData.myLocation >= bounds.min && inverseData.myLocation <= bounds.max || inverseData.myLocation > data.myLocation || inverseData.oversize < data.oversize) {
                            data.myLocation = inverseData.myLocation;
                            data.oversize = inverseData.oversize;
                            return true
                        }
                    }
                    return false
                }
            };
        var scrollbarWidth;
        var defaultPositionResult = {
                h: {
                    location: 0,
                    flip: false,
                    fit: false,
                    oversize: 0
                },
                v: {
                    location: 0,
                    flip: false,
                    fit: false,
                    oversize: 0
                }
            };
        var calculatePosition = function(what, options) {
                var $what = $(what),
                    currentOffset = $what.offset(),
                    result = $.extend(true, {}, defaultPositionResult, {
                        h: {location: currentOffset.left},
                        v: {location: currentOffset.top}
                    });
                if (!options)
                    return result;
                var my = normalizeAlign(options.my),
                    at = normalizeAlign(options.at),
                    of = options.of || window,
                    offset = normalizeOffset(options.offset),
                    collision = normalizeCollision(options.collision),
                    boundaryOffset = normalizeOffset(options.boundaryOffset);
                var h = {
                        mySize: $what.outerWidth(),
                        myAlign: my.h,
                        atAlign: at.h,
                        offset: offset.h,
                        collision: collision.h,
                        boundaryOffset: boundaryOffset.h
                    };
                var v = {
                        mySize: $what.outerHeight(),
                        myAlign: my.v,
                        atAlign: at.v,
                        offset: offset.v,
                        collision: collision.v,
                        boundaryOffset: boundaryOffset.v
                    };
                if (of.preventDefault) {
                    h.atLocation = of.pageX;
                    v.atLocation = of.pageY;
                    h.atSize = 0;
                    v.atSize = 0
                }
                else {
                    of = $(of);
                    if ($.isWindow(of[0])) {
                        h.atLocation = of.scrollLeft();
                        v.atLocation = of.scrollTop();
                        h.atSize = of.width();
                        v.atSize = of.height()
                    }
                    else if (of[0].nodeType === 9) {
                        h.atLocation = 0;
                        v.atLocation = 0;
                        h.atSize = of.width();
                        v.atSize = of.height()
                    }
                    else {
                        var o = of.offset();
                        h.atLocation = o.left;
                        v.atLocation = o.top;
                        h.atSize = of.outerWidth();
                        v.atSize = of.outerHeight()
                    }
                }
                initMyLocation(h);
                initMyLocation(v);
                var bounds = function() {
                        var win = $(window),
                            windowWidth = win.width(),
                            windowHeight = win.height(),
                            left = win.scrollLeft(),
                            top = win.scrollTop(),
                            hScrollbar = document.width > document.documentElement.clientWidth,
                            vScrollbar = document.height > document.documentElement.clientHeight,
                            hZoomLevel = DX.support.touch ? document.documentElement.clientWidth / (vScrollbar ? windowWidth - scrollbarWidth : windowWidth) : 1,
                            vZoomLevel = DX.support.touch ? document.documentElement.clientHeight / (hScrollbar ? windowHeight - scrollbarWidth : windowHeight) : 1;
                        if (scrollbarWidth === undefined)
                            scrollbarWidth = calculateScrollbarWidth();
                        return {
                                h: {
                                    min: left + h.boundaryOffset,
                                    max: left + windowWidth / hZoomLevel - h.mySize - h.boundaryOffset
                                },
                                v: {
                                    min: top + v.boundaryOffset,
                                    max: top + windowHeight / vZoomLevel - v.mySize - v.boundaryOffset
                                }
                            }
                    }();
                h.oversize = calculateOversize(h, bounds.h);
                v.oversize = calculateOversize(v, bounds.v);
                if (decolliders[h.collision])
                    result.h[h.collision] = decolliders[h.collision](h, bounds.h);
                if (decolliders[v.collision])
                    result.v[v.collision] = decolliders[v.collision](v, bounds.v);
                $.extend(true, result, {
                    h: {
                        location: Math.round(h.myLocation),
                        oversize: Math.round(h.oversize)
                    },
                    v: {
                        location: Math.round(v.myLocation),
                        oversize: Math.round(v.oversize)
                    }
                });
                return result
            };
        var position = function(what, options) {
                var $what = $(what);
                if (!options)
                    return $what.offset();
                DX.translator.resetPosition($what);
                var offset = $what.offset(),
                    targetPosition = options.h && options.v ? options : calculatePosition($what, options);
                DX.translator.move($what, {
                    left: Math.round(targetPosition.h.location - offset.left),
                    top: Math.round(targetPosition.v.location - offset.top)
                });
                return targetPosition
            };
        $.extend(DX, {
            calculatePosition: calculatePosition,
            position: position,
            inverseAlign: inverseAlign
        });
        var calculateScrollbarWidth = function() {
                var $scrollDiv = $("<div>").css({
                        width: 100,
                        height: 100,
                        overflow: "scroll",
                        position: "absolute",
                        top: -9999
                    }).appendTo($("body")),
                    result = $scrollDiv.get(0).offsetWidth - $scrollDiv.get(0).clientWidth;
                $scrollDiv.remove();
                return result
            }
    })(jQuery, DevExpress);
    /*! Module core, file action.js */
    (function($, DX, undefined) {
        var actionExecutors = {};
        var registerExecutor = function(name, executor) {
                if ($.isPlainObject(name)) {
                    $.each(name, registerExecutor);
                    return
                }
                actionExecutors[name] = executor
            };
        var unregisterExecutor = function(name) {
                var args = $.makeArray(arguments);
                $.each(args, function() {
                    delete actionExecutors[this]
                })
            };
        registerExecutor({
            func: {execute: function(e) {
                    if ($.isFunction(e.action)) {
                        e.result = e.action.apply(e.context, e.args);
                        e.handled = true
                    }
                }},
            url: {execute: function(e) {
                    if (typeof e.action === "string" && e.action.charAt(0) !== "#")
                        document.location = e.action
                }},
            hash: {execute: function(e) {
                    if (typeof e.action === "string" && e.action.charAt(0) === "#")
                        document.location.hash = e.action
                }}
        });
        var Action = DX.Class.inherit({
                ctor: function(action, config) {
                    config = config || {};
                    this._action = action || $.noop;
                    this._context = config.context || window;
                    this._beforeExecute = config.beforeExecute || $.noop;
                    this._afterExecute = config.afterExecute || $.noop;
                    this._component = config.component;
                    this._excludeValidators = config.excludeValidators
                },
                execute: function() {
                    var e = {
                            action: this._action,
                            args: Array.prototype.slice.call(arguments),
                            context: this._context,
                            component: this._component,
                            cancel: false,
                            handled: false
                        };
                    if (!this._validateAction(e))
                        return;
                    this._beforeExecute.call(this._context, e);
                    if (e.cancel)
                        return;
                    var result = this._executeAction(e);
                    this._afterExecute.call(this._context, e);
                    return result
                },
                _validateAction: function(e) {
                    var excludeValidators = this._excludeValidators;
                    $.each(actionExecutors, function(name, executor) {
                        if (excludeValidators && $.inArray(name, excludeValidators) > -1)
                            return;
                        if (executor.validate)
                            executor.validate(e);
                        if (e.cancel)
                            return false
                    });
                    return !e.cancel
                },
                _executeAction: function(e) {
                    var result;
                    $.each(actionExecutors, function(index, executor) {
                        if (executor.execute)
                            executor.execute(e);
                        if (e.handled) {
                            result = e.result;
                            return false
                        }
                    });
                    return result
                }
            });
        $.extend(DX, {
            registerActionExecutor: registerExecutor,
            unregisterActionExecutor: unregisterExecutor,
            Action: Action
        });
        DX.__internals = {actionExecutors: actionExecutors}
    })(jQuery, DevExpress);
    /*! Module core, file translator.js */
    (function($, DX, undefined) {
        var support = DX.support,
            TRANSLATOR_DATA_KEY = "dxTranslator",
            TRANSFORM_MATRIX_REGEX = /matrix(3d)?\((.+?)\)/,
            TRANSLATE_REGEX = /translate(?:3d)?\((.+?)\)/;
        var locate = function($element) {
                var result,
                    position,
                    finalPosition;
                if (support.transform3d) {
                    var translate = getTranslate($element);
                    result = {
                        left: translate.x,
                        top: translate.y
                    }
                }
                else {
                    var originalTop = $element.css("top"),
                        originalLeft = $element.css("left");
                    position = $element.position();
                    $element.css({
                        transform: "none",
                        top: 0,
                        left: 0
                    });
                    clearCache($element);
                    finalPosition = $element.position();
                    result = {
                        left: position.left - finalPosition.left || parseInt(originalLeft) || 0,
                        top: position.top - finalPosition.top || parseInt(originalTop) || 0
                    };
                    $element.css({
                        top: originalTop,
                        left: originalLeft
                    })
                }
                return result
            };
        var move = function($element, position) {
                if (!support.transform3d) {
                    $element.css(position);
                    return
                }
                var translate = getTranslate($element),
                    left = position.left,
                    top = position.top;
                if (left !== undefined)
                    translate.x = left || 0;
                if (top !== undefined)
                    translate.y = top || 0;
                $element.css({transform: getTranslateCss(translate)});
                if (isPersentValue(left) || isPersentValue(top))
                    clearCache($element)
            };
        var isPersentValue = function(value) {
                return $.type(value) === "string" && value[value.length - 1] === "%"
            };
        var getTranslate = function($element) {
                var result = $element.data(TRANSLATOR_DATA_KEY);
                if (!result) {
                    var transformValue = $element.css("transform") || getTranslateCss({
                            x: 0,
                            y: 0
                        }),
                        matrix = transformValue.match(TRANSFORM_MATRIX_REGEX),
                        is3D = matrix && matrix[1];
                    if (matrix) {
                        matrix = matrix[2].split(",");
                        if (is3D === "3d")
                            matrix = matrix.slice(12, 15);
                        else {
                            matrix.push(0);
                            matrix = matrix.slice(4, 7)
                        }
                    }
                    else
                        matrix = [0, 0, 0];
                    result = {
                        x: parseFloat(matrix[0]),
                        y: parseFloat(matrix[1]),
                        z: parseFloat(matrix[2])
                    };
                    cacheTranslate($element, result)
                }
                return result
            };
        var cacheTranslate = function($element, translate) {
                $element.data(TRANSLATOR_DATA_KEY, translate)
            };
        var clearCache = function($element) {
                $element.removeData(TRANSLATOR_DATA_KEY)
            };
        var resetPosition = function($element) {
                $element.css({
                    left: 0,
                    top: 0,
                    transform: "none"
                });
                clearCache($element)
            };
        var parseTranslate = function(translateString) {
                var result = translateString.match(TRANSLATE_REGEX);
                if (!result || !result[1])
                    return;
                result = result[1].split(",");
                result = {
                    x: parseFloat(result[0]),
                    y: parseFloat(result[1]),
                    z: parseFloat(result[2])
                };
                return result
            };
        var getTranslateCss = function(translate) {
                translate.x = translate.x || 0;
                translate.y = translate.y || 0;
                var xValueString = isPersentValue(translate.x) ? translate.x : translate.x + "px";
                var yValueString = isPersentValue(translate.y) ? translate.y : translate.y + "px";
                return "translate(" + xValueString + ", " + yValueString + ")"
            };
        DX.translator = {
            move: move,
            locate: locate,
            clearCache: clearCache,
            parseTranslate: parseTranslate,
            getTranslate: getTranslate,
            getTranslateCss: getTranslateCss,
            resetPosition: resetPosition
        }
    })(jQuery, DevExpress);
    /*! Module core, file animationFrame.js */
    (function($, DX, undefined) {
        var FRAME_ANIMATION_STEP_TIME = 1000 / 60,
            requestAnimationFrame = function(callback, element) {
                return this.setTimeout(callback, FRAME_ANIMATION_STEP_TIME)
            },
            cancelAnimationFrame = function(requestID) {
                return this.clearTimeout(requestID)
            },
            nativeRequestAnimationFrame = window.requestAnimationFrame || window.webkitRequestAnimationFrame || window.mozRequestAnimationFrame || window.oRequestAnimationFrame || window.msRequestAnimationFrame,
            nativeCancelAnimationFrame = window.cancelAnimationFrame || window.webkitCancelAnimationFrame || window.mozCancelAnimationFrame || window.oCancelAnimationFrame || window.msCancelAnimationFrame;
        if (nativeRequestAnimationFrame && nativeCancelAnimationFrame) {
            requestAnimationFrame = nativeRequestAnimationFrame;
            cancelAnimationFrame = nativeCancelAnimationFrame
        }
        if (nativeRequestAnimationFrame && !nativeCancelAnimationFrame) {
            var cancelledRequests = {};
            requestAnimationFrame = function(callback) {
                var requestId = nativeRequestAnimationFrame.call(window, function() {
                        try {
                            if (requestId in cancelledRequests)
                                return;
                            callback.apply(this, arguments)
                        }
                        finally {
                            delete cancelledRequests[requestId]
                        }
                    });
                return requestId
            };
            cancelAnimationFrame = function(requestId) {
                cancelledRequests[requestId] = true
            }
        }
        requestAnimationFrame = $.proxy(requestAnimationFrame, window);
        cancelAnimationFrame = $.proxy(cancelAnimationFrame, window);
        $.extend(DX, {
            requestAnimationFrame: requestAnimationFrame,
            cancelAnimationFrame: cancelAnimationFrame
        })
    })(jQuery, DevExpress);
    /*! Module core, file animator.js */
    (function($, DX, undefined) {
        DX.Animator = DX.Class.inherit({
            ctor: function() {
                this._finished = true;
                this._stopped = false
            },
            start: function() {
                this._stopped = false;
                this._finished = false;
                this._stepCore()
            },
            stop: function() {
                this._stopped = true
            },
            _stepCore: function() {
                if (this._isStopped()) {
                    this._stop();
                    return
                }
                if (this._isFinished()) {
                    this._finished = true;
                    this._complete();
                    return
                }
                this._step();
                DX.requestAnimationFrame.call(window, $.proxy(this._stepCore, this))
            },
            _step: DX.abstract,
            _isFinished: $.noop,
            _stop: $.noop,
            _complete: $.noop,
            _isStopped: function() {
                return this._stopped
            },
            inProgress: function() {
                return !(this._stopped || this._finished)
            }
        })
    })(jQuery, DevExpress);
    /*! Module core, file fx.js */
    (function($, DX, undefined) {
        var translator = DX.translator,
            support = DX.support,
            transitionEndEventName = support.transitionEndEventName + ".dxFX";
        var CSS_TRANSITION_EASING_REGEX = /cubic-bezier\((\d+(?:\.\d+)?)\s*,\s*(\d+(?:\.\d+)?)\s*,\s*(\d+(?:\.\d+)?)\s*,\s*(\d+(?:\.\d+)?)\)/,
            RELATIVE_VALUE_REGEX = /^([+-])=(.*)/i,
            ANIM_DATA_KEY = "dxAnimData",
            ANIM_QUEUE_KEY = "dxAnimQueue",
            TRANSFORM_PROP = "transform";
        var TransitionAnimationStrategy = {
                animate: function($element, config) {
                    var that = this,
                        deferred = $.Deferred();
                    config.transitionAnimation = {finish: function() {
                            that._cleanup($element, config);
                            deferred.resolveWith($element, [config, $element])
                        }};
                    this._startAnimation($element, config);
                    this._completeAnimationCallback($element, config).done(function() {
                        config.transitionAnimation.finish()
                    });
                    if (!config.duration)
                        config.transitionAnimation.finish();
                    return deferred.promise()
                },
                _completeAnimationCallback: function($element, config) {
                    var startTime = $.now() + config.delay,
                        deferred = $.Deferred(),
                        transitionEndFired = $.Deferred(),
                        simulatedTransitionEndFired = $.Deferred();
                    $element.one(transitionEndEventName, function(e) {
                        if ($.now() - startTime >= config.duration)
                            transitionEndFired.reject()
                    });
                    config.transitionAnimation.simulatedEndEventTimer = setTimeout(function() {
                        simulatedTransitionEndFired.reject()
                    }, config.duration + config.delay);
                    $.when(transitionEndFired, simulatedTransitionEndFired).fail($.proxy(function() {
                        deferred.resolve()
                    }, this));
                    return deferred.promise()
                },
                _startAnimation: function($element, config) {
                    $element.css("transform");
                    $element.css({
                        transitionProperty: "all",
                        transitionDelay: config.delay + "ms",
                        transitionDuration: config.duration + "ms",
                        transitionTimingFunction: config.easing
                    });
                    setProps($element, config.to)
                },
                _cleanup: function($element, config) {
                    $element.css("transition", "none").off(transitionEndEventName);
                    clearTimeout(config.transitionAnimation.simulatedEndEventTimer)
                },
                stop: function($element, config, jumpToEnd) {
                    if (!config)
                        return;
                    if (jumpToEnd)
                        config.transitionAnimation.finish();
                    else {
                        $.each(config.to, function(key) {
                            $element.css(key, $element.css(key))
                        });
                        this._cleanup($element, config)
                    }
                }
            };
        var FrameAnimationStrategy = {
                animate: function($element, config) {
                    var deferred = $.Deferred(),
                        that = this;
                    if (!config)
                        return deferred.reject().promise();
                    $.each(config.to, function(prop) {
                        if (config.from[prop] === undefined)
                            config.from[prop] = that._normalizeValue($element.css(prop))
                    });
                    if (config.to[TRANSFORM_PROP]) {
                        config.from[TRANSFORM_PROP] = that._parseTransform(config.from[TRANSFORM_PROP]);
                        config.to[TRANSFORM_PROP] = that._parseTransform(config.to[TRANSFORM_PROP])
                    }
                    config.frameAnimation = {
                        to: config.to,
                        from: config.from,
                        currentValue: config.from,
                        easing: convertTransitionTimingFuncToJQueryEasing(config.easing),
                        duration: config.duration,
                        startTime: (new Date).valueOf(),
                        finish: function() {
                            this.currentValue = this.to;
                            this.draw();
                            deferred.resolve()
                        },
                        draw: function() {
                            var currentValue = $.extend({}, this.currentValue);
                            if (currentValue[TRANSFORM_PROP])
                                currentValue[TRANSFORM_PROP] = $.map(currentValue[TRANSFORM_PROP], function(value, prop) {
                                    if (prop === "translate")
                                        return translator.getTranslateCss(value);
                                    else if (prop === "scale")
                                        return "scale(" + value + ")";
                                    else if (prop.substr(0, prop.length - 1) === "rotate")
                                        return prop + "(" + value + "deg)"
                                }).join(" ");
                            $element.css(currentValue)
                        }
                    };
                    if (config.delay) {
                        config.frameAnimation.startTime += config.delay;
                        config.frameAnimation.delayTimeout = setTimeout(function() {
                            that._animationStep($element, config)
                        }, config.delay)
                    }
                    else
                        that._animationStep($element, config);
                    return deferred.promise()
                },
                _parseTransform: function(transformString) {
                    var result = {};
                    $.each(transformString.match(/(\w|\d)+\([^\)]*\)\s*/g), function(i, part) {
                        var translateData = translator.parseTranslate(part),
                            scaleData = part.match(/scale\((.+?)\)/),
                            rotateData = part.match(/(rotate.)\((.+)deg\)/);
                        if (translateData)
                            result.translate = translateData;
                        if (scaleData && scaleData[1])
                            result.scale = parseFloat(scaleData[1]);
                        if (rotateData && rotateData[1])
                            result[rotateData[1]] = parseFloat(rotateData[2])
                    });
                    return result
                },
                stop: function($element, config, jumpToEnd) {
                    var frameAnimation = config && config.frameAnimation;
                    if (!frameAnimation)
                        return;
                    clearTimeout(frameAnimation.delayTimeout);
                    if (jumpToEnd)
                        frameAnimation.finish();
                    delete config.frameAnimation
                },
                _animationStep: function($element, config) {
                    var frameAnimation = config && config.frameAnimation;
                    if (!frameAnimation)
                        return;
                    var now = (new Date).valueOf();
                    if (now >= frameAnimation.startTime + frameAnimation.duration) {
                        frameAnimation.finish();
                        return
                    }
                    frameAnimation.currentValue = this._calcStepValue(frameAnimation, now - frameAnimation.startTime);
                    frameAnimation.draw();
                    DX.requestAnimationFrame($.proxy(function() {
                        this._animationStep($element, config)
                    }, this))
                },
                _calcStepValue: function(frameAnimation, currentDuration) {
                    var calcValueRecursively = function(from, to) {
                            var result = $.isArray(to) ? [] : {};
                            var calcEasedValue = function(propName) {
                                    var x = currentDuration / frameAnimation.duration,
                                        t = currentDuration,
                                        b = 1 * from[propName],
                                        c = to[propName] - from[propName],
                                        d = frameAnimation.duration;
                                    return $.easing[frameAnimation.easing](x, t, b, c, d)
                                };
                            $.each(to, function(propName, endPropValue) {
                                if (typeof endPropValue === "string" && parseFloat(endPropValue, 10) === false)
                                    return true;
                                result[propName] = typeof endPropValue === "object" ? calcValueRecursively(from[propName], endPropValue) : calcEasedValue(propName)
                            });
                            return result
                        };
                    return calcValueRecursively(frameAnimation.from, frameAnimation.to)
                },
                _normalizeValue: function(value) {
                    var numericValue = parseFloat(value, 10);
                    if (numericValue === false)
                        return value;
                    return numericValue
                }
            };
        var animationStrategies = {
                transition: support.transition ? TransitionAnimationStrategy : FrameAnimationStrategy,
                frame: FrameAnimationStrategy
            };
        var getAnimationStrategy = function(config) {
                return animationStrategies[config && config.strategy || "transition"]
            };
        var TransitionTimingFuncMap = {
                linear: "cubic-bezier(0, 0, 1, 1)",
                ease: "cubic-bezier(0.25, 0.1, 0.25, 1)",
                "ease-in": "cubic-bezier(0.42, 0, 1, 1)",
                "ease-out": "cubic-bezier(0, 0, 0.58, 1)",
                "ease-in-out": "cubic-bezier(0.42, 0, 0.58, 1)"
            };
        var convertTransitionTimingFuncToJQueryEasing = function(cssTransitionEasing) {
                cssTransitionEasing = TransitionTimingFuncMap[cssTransitionEasing] || cssTransitionEasing;
                var bezCoeffs = cssTransitionEasing.match(CSS_TRANSITION_EASING_REGEX);
                if (!bezCoeffs)
                    return "linear";
                bezCoeffs = bezCoeffs.slice(1, 5);
                $.each(bezCoeffs, function(index, value) {
                    bezCoeffs[index] = parseFloat(value)
                });
                var easingName = "cubicbezier_" + bezCoeffs.join("_").replace(/\./g, "p");
                if (!$.isFunction($.easing[easingName])) {
                    var polynomBezier = function(x1, y1, x2, y2) {
                            var Cx = 3 * x1,
                                Bx = 3 * (x2 - x1) - Cx,
                                Ax = 1 - Cx - Bx,
                                Cy = 3 * y1,
                                By = 3 * (y2 - y1) - Cy,
                                Ay = 1 - Cy - By;
                            var bezierX = function(t) {
                                    return t * (Cx + t * (Bx + t * Ax))
                                };
                            var bezierY = function(t) {
                                    return t * (Cy + t * (By + t * Ay))
                                };
                            var findXfor = function(t) {
                                    var x = t,
                                        i = 0,
                                        z;
                                    while (i < 14) {
                                        z = bezierX(x) - t;
                                        if (Math.abs(z) < 1e-3)
                                            break;
                                        x = x - z / derivativeX(x);
                                        i++
                                    }
                                    return x
                                };
                            var derivativeX = function(t) {
                                    return Cx + t * (2 * Bx + t * 3 * Ax)
                                };
                            return function(t) {
                                    return bezierY(findXfor(t))
                                }
                        };
                    $.easing[easingName] = function(x, t, b, c, d) {
                        return c * polynomBezier(bezCoeffs[0], bezCoeffs[1], bezCoeffs[2], bezCoeffs[3])(t / d) + b
                    }
                }
                return easingName
            };
        var baseConfigValidator = function(config, animationType) {
                $.each(["from", "to"], function() {
                    if (!$.isPlainObject(config[this]))
                        throw Error("Animation with the '" + animationType + "' type requires '" + this + "' configuration as an plain object.");
                })
            };
        var CustomAnimationConfigurator = {setup: function($element, config){}};
        var SlideAnimationConfigurator = {
                validateConfig: function(config) {
                    baseConfigValidator(config, "slide")
                },
                setup: function($element, config) {
                    var location = translator.locate($element);
                    this._setUpConfig(location, config.from);
                    this._setUpConfig(location, config.to);
                    translator.clearCache($element);
                    if (!support.transform3d && $element.css("position") === "static")
                        $element.css("position", "relative")
                },
                _setUpConfig: function(location, config) {
                    config.left = "left" in config ? config.left : "+=0";
                    config.top = "top" in config ? config.top : "+=0";
                    this._initNewPosition(location, config)
                },
                _initNewPosition: function(location, config) {
                    var position = {
                            left: config.left,
                            top: config.top
                        };
                    delete config.left;
                    delete config.top;
                    var relativeValue = this._getRelativeValue(position.left);
                    if (relativeValue !== undefined)
                        position.left = relativeValue + location.left;
                    else
                        config.left = 0;
                    relativeValue = this._getRelativeValue(position.top);
                    if (relativeValue !== undefined)
                        position.top = relativeValue + location.top;
                    else
                        config.top = 0;
                    var translate = {
                            x: 0,
                            y: 0
                        };
                    if (support.transform3d)
                        translate = {
                            x: position.left,
                            y: position.top
                        };
                    else {
                        config.left = position.left;
                        config.top = position.top
                    }
                    config[TRANSFORM_PROP] = translator.getTranslateCss(translate)
                },
                _getRelativeValue: function(value) {
                    var relativeValue;
                    if (typeof value === "string" && (relativeValue = RELATIVE_VALUE_REGEX.exec(value)))
                        return parseInt(relativeValue[1] + "1") * relativeValue[2]
                }
            };
        var FadeAnimationConfigurator = {setup: function($element, config) {
                    var from = config.from,
                        fromOpacity = $.isPlainObject(from) ? $element.css("opacity") : String(from),
                        toOpacity = String(config.to);
                    config.from = {opacity: fromOpacity};
                    config.to = {opacity: toOpacity}
                }};
        var PopAnimationConfigurator = {
                validateConfig: function(config) {
                    baseConfigValidator(config, "pop")
                },
                setup: function($element, config) {
                    var from = config.from,
                        to = config.to,
                        fromOpacity = "opacity" in from ? from.opacity : $element.css("opacity"),
                        toOpacity = "opacity" in to ? to.opacity : 1,
                        fromScale = "scale" in from ? from.scale : 0,
                        toScale = "scale" in to ? to.scale : 1;
                    config.from = {opacity: fromOpacity};
                    var translate = translator.getTranslate($element);
                    config.from[TRANSFORM_PROP] = this._getCssTransform(translate, fromScale);
                    config.to = {opacity: toOpacity};
                    config.to[TRANSFORM_PROP] = this._getCssTransform(translate, toScale)
                },
                _getCssTransform: function(translate, scale) {
                    return translator.getTranslateCss(translate) + "scale(" + scale + ")"
                }
            };
        var animationConfigurators = {
                custom: CustomAnimationConfigurator,
                slide: SlideAnimationConfigurator,
                fade: FadeAnimationConfigurator,
                pop: PopAnimationConfigurator
            };
        var getAnimationConfigurator = function(type) {
                var result = animationConfigurators[type];
                if (!result)
                    throw Error("Unknown animation type \"" + type + "\"");
                return result
            };
        var defaultConfig = {
                type: "custom",
                from: {},
                to: {},
                duration: 400,
                start: $.noop,
                complete: $.noop,
                easing: "ease",
                delay: 0
            };
        var animate = function(element, config) {
                var $element = $(element);
                config = $.extend(true, {}, defaultConfig, config);
                if (!$element.length)
                    return $.Deferred().resolve().promise();
                var configurator = getAnimationConfigurator(config.type);
                if ($.isFunction(configurator.validateConfig))
                    configurator.validateConfig(config);
                return pushInAnimationQueue($element, config)
            };
        var pushInAnimationQueue = function($element, config) {
                config.deferred = config.deferred || $.Deferred();
                var queueData = getAnimQueueData($element);
                writeAnimQueueData($element, queueData);
                queueData.push(config);
                if (!isAnimating($element))
                    shiftFromAnimationQueue($element, queueData);
                return config.deferred.promise()
            };
        var getAnimQueueData = function($element) {
                return $element.data(ANIM_QUEUE_KEY) || []
            };
        var writeAnimQueueData = function($element, queueData) {
                $element.data(ANIM_QUEUE_KEY, queueData)
            };
        var destroyAnimQueueData = function($element) {
                $element.removeData(ANIM_QUEUE_KEY)
            };
        var isAnimating = function($element) {
                return !!$element.data(ANIM_DATA_KEY)
            };
        var shiftFromAnimationQueue = function($element, queueData) {
                var queueData = getAnimQueueData($element);
                if (!queueData.length)
                    return;
                var config = queueData.shift();
                if (queueData.length === 0)
                    destroyAnimQueueData($element);
                executeAnimation($element, config).done(function() {
                    shiftFromAnimationQueue($element)
                })
            };
        var executeAnimation = function($element, config) {
                setupPosition($element, config.from);
                setupPosition($element, config.to);
                var configurator = getAnimationConfigurator(config.type);
                configurator.setup($element, config);
                $element.data(ANIM_DATA_KEY, config);
                if (DX.fx.off)
                    config.duration = 0;
                setProps($element, config.from);
                config.start.apply(this, [$element, config]);
                return getAnimationStrategy(config).animate($element, config).done(function() {
                        $element.removeData(ANIM_DATA_KEY);
                        config.complete.apply(this, [$element, config]);
                        config.deferred.resolveWith(this, [$element, config])
                    })
            };
        var setupPosition = function($element, config) {
                if (!config.position)
                    return;
                var position = DX.calculatePosition($element, config.position),
                    offset = $element.offset(),
                    currentPosition = $element.position();
                $.extend(config, {
                    left: position.h.location - offset.left + currentPosition.left,
                    top: position.v.location - offset.top + currentPosition.top
                });
                delete config.position
            };
        var setProps = function($element, props) {
                $.each(props, function(key, value) {
                    $element.css(key, value)
                })
            };
        var stop = function(element, jumpToEnd) {
                var $element = $(element),
                    queueData = getAnimQueueData($element);
                $.each(queueData, function(_, config) {
                    config.duration = 0
                });
                var config = $element.data(ANIM_DATA_KEY);
                getAnimationStrategy(config).stop($element, config, jumpToEnd);
                $element.removeData(ANIM_DATA_KEY);
                destroyAnimQueueData($element)
            };
        DX.fx = {
            off: false,
            animationTypes: animationConfigurators,
            animate: animate,
            isAnimating: isAnimating,
            stop: stop
        };
        DX.fx.__internals = {convertTransitionTimingFuncToJQueryEasing: convertTransitionTimingFuncToJQueryEasing}
    })(jQuery, DevExpress);
    /*! Module core, file endpointSelector.js */
    (function($, DX, undefined) {
        var location = window.location,
            DXPROXY_HOST = "dxproxy.devexpress.com:8000",
            WIN_JS = location.protocol === "ms-appx:",
            IS_DXPROXY = location.host === DXPROXY_HOST,
            IS_LOCAL = isLocalHostName(location.hostname);
        function isLocalHostName(url) {
            return /^(localhost$|127\.)/i.test(url)
        }
        var extractProxyAppId = function() {
                return location.pathname.split("/")[1]
            };
        var formatProxyUrl = function(localUrl) {
                var urlData = DX.parseUrl(localUrl);
                if (!isLocalHostName(urlData.hostname))
                    return localUrl;
                return "http://" + DXPROXY_HOST + "/" + extractProxyAppId() + "_" + urlData.port + urlData.pathname + urlData.search
            };
        var EndpointSelector = DX.EndpointSelector = function(config) {
                this.config = config
            };
        EndpointSelector.prototype = {urlFor: function(key) {
                var bag = this.config[key];
                if (!bag)
                    throw Error("Unknown endpoint key");
                if (IS_DXPROXY)
                    return formatProxyUrl(bag.local);
                if (bag.production)
                    if (WIN_JS && !Debug.debuggerEnabled || !WIN_JS && !IS_LOCAL)
                        return bag.production;
                return bag.local
            }}
    })(jQuery, DevExpress);
    /*! Module core, file formatHelper.js */
    (function($, DX, undefined) {
        var utils = DX.utils;
        DX.NumericFormat = {
            currency: 'C',
            fixedpoint: 'N',
            exponential: '',
            percent: 'P',
            decimal: 'D'
        };
        DX.LargeNumberFormatPostfixes = {
            1: 'K',
            2: 'M',
            3: 'B',
            4: 'T'
        };
        var MAX_LARGE_NUMBER_POWER = 4,
            DECIMAL_BASE = 10;
        DX.LargeNumberFormatPowers = {
            largenumber: 'auto',
            thousands: 1,
            millions: 2,
            billions: 3,
            trillions: 4
        };
        DX.DateTimeFormat = {
            longdate: 'D',
            longtime: 'T',
            monthandday: 'M',
            monthandyear: 'Y',
            quarterandyear: 'qq',
            shortdate: 'd',
            shorttime: 't',
            millisecond: 'fff',
            second: 'T',
            minute: 't',
            hour: 't',
            day: 'dd',
            week: 'dd',
            month: 'MMMM',
            quarter: 'qq',
            year: 'yyyy',
            longdatelongtime: 'D',
            shortdateshorttime: 'd'
        };
        DX.formatHelper = {
            defaultQuarterFormat: 'Q{0}',
            romanDigits: ['I', 'II', 'III', 'IV'],
            _addFormatSeparator: function(format1, format2) {
                var separator = ' ';
                if (format2)
                    return format1 + separator + format2;
                return format1
            },
            _getDateTimeFormatPattern: function(dateTimeFormat) {
                return Globalize.findClosestCulture().calendar.patterns[DX.DateTimeFormat[dateTimeFormat.toLowerCase()]]
            },
            _isDateFormatContains: function(format) {
                var result = false;
                $.each(DX.DateTimeFormat, function(key, value) {
                    result = key === format.toLowerCase();
                    return !result
                });
                return result
            },
            getQuarter: function(month) {
                return Math.floor(month / 3)
            },
            getFirstQuarterMonth: function(month) {
                return this.getQuarter(month) * 3
            },
            _getQuarterString: function(date, format) {
                var quarter = this.getQuarter(date.getMonth());
                switch (format) {
                    case'q':
                        return this.romanDigits[quarter];
                    case'qq':
                        return utils.stringFormat(this.defaultQuarterFormat, this.romanDigits[quarter]);
                    case'Q':
                        return (quarter + 1).toString();
                    case'QQ':
                        return utils.stringFormat(this.defaultQuarterFormat, (quarter + 1).toString())
                }
                return ''
            },
            _formatCustomString: function(value, format) {
                var regExp = /qq|q|QQ|Q/g,
                    quarterFormat,
                    result = '',
                    index = 0;
                regExp.lastIndex = 0;
                while (index < format.length) {
                    quarterFormat = regExp.exec(format);
                    if (!quarterFormat || quarterFormat.index > index)
                        result += Globalize.format(value, format.substring(index, quarterFormat ? quarterFormat.index : format.length));
                    if (quarterFormat) {
                        result += this._getQuarterString(value, quarterFormat[0]);
                        index = quarterFormat.index + quarterFormat[0].length
                    }
                    else
                        index = format.length
                }
                return result
            },
            _parseNumberFormatString: function(format) {
                var formatList,
                    formatObject = {};
                if (!format || typeof format !== 'string')
                    return;
                formatList = format.toLowerCase().split(' ');
                $.each(formatList, function(index, value) {
                    if (value in DX.NumericFormat)
                        formatObject.formatType = value;
                    else if (value in DX.LargeNumberFormatPowers)
                        formatObject.power = DX.LargeNumberFormatPowers[value]
                });
                if (formatObject.power && !formatObject.formatType)
                    formatObject.formatType = 'fixedpoint';
                if (formatObject.formatType)
                    return formatObject
            },
            _calculateNumberPower: function(value, base, minPower, maxPower) {
                var number = Math.abs(value);
                var power = 0;
                if (number > 1)
                    while (number && number >= base && (maxPower === undefined || power < maxPower)) {
                        power++;
                        number = number / base
                    }
                else if (number > 0 && number < 1)
                    while (number < 1 && (minPower === undefined || power > minPower)) {
                        power--;
                        number = number * base
                    }
                return power
            },
            _getNumberByPower: function(number, power, base) {
                var result = number;
                while (power > 0) {
                    result = result / base;
                    power--
                }
                while (power < 0) {
                    result = result * base;
                    power++
                }
                return result
            },
            _formatNumber: function(value, formatObject, precision) {
                var powerPostfix;
                if (formatObject.power === 'auto')
                    formatObject.power = this._calculateNumberPower(value, 1000, 0, MAX_LARGE_NUMBER_POWER);
                if (formatObject.power)
                    value = this._getNumberByPower(value, formatObject.power, 1000);
                powerPostfix = DX.LargeNumberFormatPostfixes[formatObject.power] || '';
                return this._formatNumberCore(value, formatObject.formatType, precision) + powerPostfix
            },
            _formatNumberExponential: function(value, precision) {
                var power = this._calculateNumberPower(value, DECIMAL_BASE),
                    number = this._getNumberByPower(value, power, DECIMAL_BASE),
                    powString;
                precision = precision === undefined ? 1 : precision;
                if (number.toFixed(precision || 0) >= DECIMAL_BASE) {
                    power++;
                    number = number / DECIMAL_BASE
                }
                powString = (power >= 0 ? '+' : '') + power.toString();
                return this._formatNumberCore(number, 'fixedpoint', precision) + 'E' + powString
            },
            _formatNumberCore: function(value, format, precision) {
                if (format === 'exponential')
                    return this._formatNumberExponential(value, precision);
                else
                    return Globalize.format(value, DX.NumericFormat[format] + (utils.isNumber(precision) ? precision : 0))
            },
            _formatDate: function(date, format, formatString) {
                var resultFormat = DX.DateTimeFormat[format.toLowerCase()];
                format = format.toLowerCase();
                if (format === 'quarterandyear')
                    resultFormat = this._getQuarterString(date, resultFormat) + ' yyyy';
                if (format === 'quarter')
                    return this._getQuarterString(date, resultFormat);
                if (format === 'longdatelongtime')
                    return this._formatDate(date, 'longdate') + ' ' + this._formatDate(date, 'longtime');
                if (format === 'shortdateshorttime')
                    return this._formatDate(date, 'shortDate') + ' ' + this._formatDate(date, 'shortTime');
                return Globalize.format(date, resultFormat)
            },
            format: function(value, format, precision) {
                if (format && format.format)
                    if (format.dateType)
                        return this._formatDateEx(value, format);
                    else if (utils.isNumber(value) && isFinite(value))
                        return this._formatNumberEx(value, format);
                return this._format(value, format, precision)
            },
            _format: function(value, format, precision) {
                var numberFormatObject;
                if (!utils.isString(format) || format === '' || !utils.isNumber(value) && !utils.isDate(value))
                    return utils.isDefined(value) ? value.toString() : '';
                numberFormatObject = this._parseNumberFormatString(format);
                if (utils.isNumber(value) && numberFormatObject)
                    return this._formatNumber(value, numberFormatObject, precision);
                if (utils.isDate(value) && this._isDateFormatContains(format))
                    return this._formatDate(value, format);
                if (!numberFormatObject && !this._isDateFormatContains(format))
                    return this._formatCustomString(value, format)
            },
            _formatNumberEx: function(value, formatInfo) {
                var that = this,
                    numericFormatType = DX.NumericFormat[formatInfo.format.toLowerCase()],
                    numberFormat = Globalize.culture().numberFormat,
                    currencyFormat = formatInfo.currencyCulture && Globalize.cultures[formatInfo.currencyCulture] ? Globalize.cultures[formatInfo.currencyCulture].numberFormat.currency : numberFormat.currency,
                    percentFormat = numberFormat.percent,
                    formatSettings = that._getUnitFormatSettings(value, formatInfo),
                    unit = formatSettings.unit,
                    precision = formatSettings.precision,
                    showTrailingZeros = formatSettings.showTrailingZeros,
                    includeGroupSeparator = formatSettings.includeGroupSeparator,
                    groupSymbol = numberFormat[","],
                    floatingSymbol = numberFormat["."],
                    number,
                    isNegative,
                    pattern,
                    currentFormat,
                    regexParts = /n|\$|-|%/g,
                    result = "";
                value = that._applyUnitToValue(value, unit);
                number = Math.abs(value);
                isNegative = value < 0;
                switch (numericFormatType) {
                    case DX.NumericFormat.decimal:
                        pattern = "n";
                        number = Math[isNegative ? "ceil" : "floor"](number);
                        if (precision > 0) {
                            var str = "" + number;
                            for (var i = str.length; i < precision; i += 1)
                                str = "0" + str;
                            number = str
                        }
                        if (isNegative)
                            number = "-" + number;
                        break;
                    case DX.NumericFormat.fixedpoint:
                        currentFormat = numberFormat;
                    case DX.NumericFormat.currency:
                        currentFormat = currentFormat || currencyFormat;
                    case DX.NumericFormat.percent:
                        currentFormat = currentFormat || percentFormat;
                        pattern = isNegative ? currentFormat.pattern[0] : currentFormat.pattern[1] || "n";
                        number = Globalize.format(number * (numericFormatType === DX.NumericFormat.percent ? 100 : 1), "N" + precision);
                        if (!showTrailingZeros)
                            number = that._excludeTrailingZeros(number, floatingSymbol);
                        if (!includeGroupSeparator)
                            number = number.replace(new RegExp('\\' + groupSymbol, 'g'), '');
                        break;
                    case DX.NumericFormat.exponential:
                        return that._formatNumberExponential(value, precision);
                    default:
                        throw"Illegal numeric format: '" + numericFormatType + "'";
                }
                for (; ; ) {
                    var lastIndex = regexParts.lastIndex,
                        matches = regexParts.exec(pattern);
                    result += pattern.slice(lastIndex, matches ? matches.index : pattern.length);
                    if (matches)
                        switch (matches[0]) {
                            case"-":
                                if (/[1-9]/.test(number))
                                    result += numberFormat["-"];
                                break;
                            case"$":
                                result += currencyFormat.symbol;
                                break;
                            case"%":
                                result += percentFormat.symbol;
                                break;
                            case"n":
                                result += number + unit;
                                break
                        }
                    else
                        break
                }
                return (formatInfo.plus && value > 0 ? "+" : '') + result
            },
            _excludeTrailingZeros: function(strValue, floatingSymbol) {
                var floatingIndex = strValue.indexOf(floatingSymbol),
                    stopIndex,
                    i;
                if (floatingIndex < 0)
                    return strValue;
                stopIndex = strValue.length;
                for (i = stopIndex - 1; i >= floatingIndex && (strValue[i] === '0' || i === floatingIndex); i--)
                    stopIndex--;
                return strValue.substring(0, stopIndex)
            },
            _getUnitFormatSettings: function(value, formatInfo) {
                var unit = formatInfo.unit || '',
                    precision = formatInfo.precision || 0,
                    includeGroupSeparator = formatInfo.includeGroupSeparator || false,
                    showTrailingZeros = formatInfo.showTrailingZeros === undefined ? true : formatInfo.showTrailingZeros,
                    significantDigits = formatInfo.significantDigits || 1,
                    absValue;
                if (unit.toLowerCase() === 'auto') {
                    showTrailingZeros = false;
                    absValue = Math.abs(value);
                    if (significantDigits < 1)
                        significantDigits = 1;
                    if (absValue >= 1000000000) {
                        unit = 'B';
                        absValue /= 1000000000
                    }
                    else if (absValue >= 1000000) {
                        unit = 'M';
                        absValue /= 1000000
                    }
                    else if (absValue >= 1000) {
                        unit = 'K';
                        absValue /= 1000
                    }
                    else
                        unit = '';
                    if (absValue == 0)
                        precision = 0;
                    else if (absValue < 1) {
                        precision = significantDigits;
                        var smallValue = Math.pow(10, -significantDigits);
                        while (absValue < smallValue) {
                            smallValue /= 10;
                            precision++
                        }
                    }
                    else if (absValue >= 100)
                        precision = significantDigits - 3;
                    else if (absValue >= 10)
                        precision = significantDigits - 2;
                    else
                        precision = significantDigits - 1
                }
                if (precision < 0)
                    precision = 0;
                return {
                        unit: unit,
                        precision: precision,
                        showTrailingZeros: showTrailingZeros,
                        includeGroupSeparator: includeGroupSeparator
                    }
            },
            _applyUnitToValue: function(value, unit) {
                if (unit == 'B')
                    return value.toFixed(1) / 1000000000;
                if (unit == 'M')
                    return value / 1000000;
                if (unit == 'K')
                    return value / 1000;
                return value
            },
            _formatDateEx: function(value, formatInfo) {
                var that = this,
                    format = formatInfo.format,
                    dateType = formatInfo.dateType,
                    calendar = Globalize.culture().calendars.standard,
                    time = undefined,
                    index,
                    dateStr;
                format = format.toLowerCase();
                if (dateType !== 'num' || format === 'dayofweek')
                    switch (format) {
                        case'monthyear':
                            return that._formatDate(value, 'monthandyear');
                        case'quarteryear':
                            return that._getQuarterString(value, 'QQ') + ' ' + value.getFullYear();
                        case'daymonthyear':
                            return that._formatDate(value, dateType + 'Date');
                        case'datehour':
                            time = new Date(value.getTime());
                            time.setMinutes(0);
                            dateStr = dateType === 'timeOnly' ? '' : that._formatDate(value, dateType + 'Date');
                            return dateType === 'timeOnly' ? that._formatDate(time, 'shorttime') : dateStr + ' ' + that._formatDate(time, 'shorttime');
                        case'datehourminute':
                            dateStr = dateType === 'timeOnly' ? '' : that._formatDate(value, dateType + 'Date');
                            return dateType === 'timeOnly' ? that._formatDate(value, 'shorttime') : dateStr + ' ' + that._formatDate(value, 'shorttime');
                        case'datehourminutesecond':
                            dateStr = dateType === 'timeOnly' ? '' : that._formatDate(value, dateType + 'Date');
                            return dateType === 'timeOnly' ? that._formatDate(value, 'longtime') : dateStr + ' ' + that._formatDate(value, 'longtime');
                        case'year':
                            dateStr = value.toString();
                            return dateType === 'abbr' ? dateStr.slice(2, 4) : dateStr;
                        case'quarter':
                            return utils.stringFormat(that.defaultQuarterFormat, value.toString());
                        case'month':
                            index = value - 1;
                            return dateType === 'abbr' ? calendar.months.namesAbbr[index] : calendar.months.names[index];
                        case'hour':
                            if (dateType === 'long') {
                                time = new Date;
                                time.setHours(value);
                                time.setMinutes(0);
                                return that._formatDate(time, 'shorttime')
                            }
                            else
                                return value.toString();
                        case'dayofweek':
                            index = utils.isString(value) ? $.inArray(value, ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday']) : value;
                            if (dateType !== 'num')
                                return dateType === 'abbr' ? calendar.days.namesAbbr[index] : calendar.days.names[index];
                            else
                                return ((index - calendar.firstDay + 1 + 7) % 8).toString();
                        default:
                            return value.toString()
                    }
                else
                    return value.toString()
            },
            getTimeFormat: function(showSecond) {
                if (showSecond)
                    return this._getDateTimeFormatPattern('longtime');
                return this._getDateTimeFormatPattern('shorttime')
            },
            getDateFormatByDifferences: function(dateDifferences) {
                var resultFormat = '';
                if (dateDifferences.millisecond)
                    resultFormat = DX.DateTimeFormat.millisecond;
                if (dateDifferences.hour || dateDifferences.minute || dateDifferences.second)
                    resultFormat = this._addFormatSeparator(this.getTimeFormat(dateDifferences.second), resultFormat);
                if (dateDifferences.year && dateDifferences.month && dateDifferences.day)
                    return this._addFormatSeparator(this._getDateTimeFormatPattern('shortdate'), resultFormat);
                if (dateDifferences.year && dateDifferences.month)
                    return DX.DateTimeFormat['monthandyear'];
                if (dateDifferences.year)
                    return DX.DateTimeFormat['year'];
                if (dateDifferences.month && dateDifferences.day)
                    return this._addFormatSeparator(this._getDateTimeFormatPattern('monthandday'), resultFormat);
                if (dateDifferences.month)
                    return DX.DateTimeFormat['month'];
                if (dateDifferences.day)
                    return this._addFormatSeparator('dddd, dd', resultFormat);
                return resultFormat
            },
            getDateFormatByTicks: function(ticks) {
                var resultFormat,
                    maxDif,
                    currentDif,
                    i,
                    dateUnitInterval;
                if (ticks.length > 1) {
                    maxDif = utils.getDatesDifferences(ticks[0], ticks[1]);
                    for (i = 1; i < ticks.length - 1; i++) {
                        currentDif = utils.getDatesDifferences(ticks[i], ticks[i + 1]);
                        if (maxDif.count < currentDif.count)
                            maxDif = currentDif
                    }
                }
                else
                    maxDif = {
                        year: true,
                        month: true,
                        day: true,
                        hour: ticks[0].getHours() > 0,
                        minute: ticks[0].getMinutes() > 0,
                        second: ticks[0].getSeconds() > 0
                    };
                resultFormat = this.getDateFormatByDifferences(maxDif);
                return resultFormat
            },
            getDateFormatByTickInterval: function(startValue, endValue, tickInterval) {
                var resultFormat,
                    dateDifferences,
                    dateUnitInterval,
                    dateDifferencesConverter = {
                        quarter: 'month',
                        week: 'day'
                    },
                    correctDateDifferences = function(dateDifferences, tickInterval, value) {
                        switch (tickInterval) {
                            case'year':
                                dateDifferences.month = value;
                            case'quarter':
                            case'month':
                                dateDifferences.day = value;
                            case'week':
                            case'day':
                                dateDifferences.hour = value;
                            case'hour':
                                dateDifferences.minute = value;
                            case'minute':
                                dateDifferences.second = value;
                            case'second':
                                dateDifferences.millisecond = value
                        }
                    },
                    correctDifferencesByMaxDate = function(differences, minDate, maxDate) {
                        if (!maxDate.getMilliseconds() && maxDate.getSeconds()) {
                            if (maxDate.getSeconds() - minDate.getSeconds() === 1) {
                                differences.millisecond = true;
                                differences.second = false
                            }
                        }
                        else if (!maxDate.getSeconds() && maxDate.getMinutes()) {
                            if (maxDate.getMinutes() - minDate.getMinutes() === 1) {
                                differences.second = true;
                                differences.minute = false
                            }
                        }
                        else if (!maxDate.getMinutes() && maxDate.getHours()) {
                            if (maxDate.getHours() - minDate.getHours() === 1) {
                                differences.minute = true;
                                differences.hour = false
                            }
                        }
                        else if (!maxDate.getHours() && maxDate.getDate() > 1) {
                            if (maxDate.getDate() - minDate.getDate() === 1) {
                                differences.hour = true;
                                differences.day = false
                            }
                        }
                        else if (maxDate.getDate() === 1 && maxDate.getMonth()) {
                            if (maxDate.getMonth() - minDate.getMonth() === 1) {
                                differences.day = true;
                                differences.month = false
                            }
                        }
                        else if (!maxDate.getMonth() && maxDate.getFullYear())
                            if (maxDate.getFullYear() - minDate.getFullYear() === 1) {
                                differences.month = true;
                                differences.year = false
                            }
                    };
                tickInterval = utils.isString(tickInterval) ? tickInterval.toLowerCase() : tickInterval;
                dateDifferences = utils.getDatesDifferences(startValue, endValue);
                if (startValue !== endValue)
                    correctDifferencesByMaxDate(dateDifferences, startValue > endValue ? endValue : startValue, startValue > endValue ? startValue : endValue);
                dateUnitInterval = utils.getDateUnitInterval(dateDifferences);
                correctDateDifferences(dateDifferences, dateUnitInterval, true);
                dateUnitInterval = utils.getDateUnitInterval(tickInterval || 'second');
                correctDateDifferences(dateDifferences, dateUnitInterval, false);
                dateDifferences[dateDifferencesConverter[dateUnitInterval] || dateUnitInterval] = true;
                resultFormat = this.getDateFormatByDifferences(dateDifferences);
                return resultFormat
            }
        }
    })(jQuery, DevExpress);
    /*! Module core, file color.js */
    (function(DX, undefined) {
        var standardColorNames = {
                aliceblue: 'f0f8ff',
                antiquewhite: 'faebd7',
                aqua: '00ffff',
                aquamarine: '7fffd4',
                azure: 'f0ffff',
                beige: 'f5f5dc',
                bisque: 'ffe4c4',
                black: '000000',
                blanchedalmond: 'ffebcd',
                blue: '0000ff',
                blueviolet: '8a2be2',
                brown: 'a52a2a',
                burlywood: 'deb887',
                cadetblue: '5f9ea0',
                chartreuse: '7fff00',
                chocolate: 'd2691e',
                coral: 'ff7f50',
                cornflowerblue: '6495ed',
                cornsilk: 'fff8dc',
                crimson: 'dc143c',
                cyan: '00ffff',
                darkblue: '00008b',
                darkcyan: '008b8b',
                darkgoldenrod: 'b8860b',
                darkgray: 'a9a9a9',
                darkgreen: '006400',
                darkkhaki: 'bdb76b',
                darkmagenta: '8b008b',
                darkolivegreen: '556b2f',
                darkorange: 'ff8c00',
                darkorchid: '9932cc',
                darkred: '8b0000',
                darksalmon: 'e9967a',
                darkseagreen: '8fbc8f',
                darkslateblue: '483d8b',
                darkslategray: '2f4f4f',
                darkturquoise: '00ced1',
                darkviolet: '9400d3',
                deeppink: 'ff1493',
                deepskyblue: '00bfff',
                dimgray: '696969',
                dodgerblue: '1e90ff',
                feldspar: 'd19275',
                firebrick: 'b22222',
                floralwhite: 'fffaf0',
                forestgreen: '228b22',
                fuchsia: 'ff00ff',
                gainsboro: 'dcdcdc',
                ghostwhite: 'f8f8ff',
                gold: 'ffd700',
                goldenrod: 'daa520',
                gray: '808080',
                green: '008000',
                greenyellow: 'adff2f',
                honeydew: 'f0fff0',
                hotpink: 'ff69b4',
                indianred: 'cd5c5c',
                indigo: '4b0082',
                ivory: 'fffff0',
                khaki: 'f0e68c',
                lavender: 'e6e6fa',
                lavenderblush: 'fff0f5',
                lawngreen: '7cfc00',
                lemonchiffon: 'fffacd',
                lightblue: 'add8e6',
                lightcoral: 'f08080',
                lightcyan: 'e0ffff',
                lightgoldenrodyellow: 'fafad2',
                lightgrey: 'd3d3d3',
                lightgreen: '90ee90',
                lightpink: 'ffb6c1',
                lightsalmon: 'ffa07a',
                lightseagreen: '20b2aa',
                lightskyblue: '87cefa',
                lightslateblue: '8470ff',
                lightslategray: '778899',
                lightsteelblue: 'b0c4de',
                lightyellow: 'ffffe0',
                lime: '00ff00',
                limegreen: '32cd32',
                linen: 'faf0e6',
                magenta: 'ff00ff',
                maroon: '800000',
                mediumaquamarine: '66cdaa',
                mediumblue: '0000cd',
                mediumorchid: 'ba55d3',
                mediumpurple: '9370d8',
                mediumseagreen: '3cb371',
                mediumslateblue: '7b68ee',
                mediumspringgreen: '00fa9a',
                mediumturquoise: '48d1cc',
                mediumvioletred: 'c71585',
                midnightblue: '191970',
                mintcream: 'f5fffa',
                mistyrose: 'ffe4e1',
                moccasin: 'ffe4b5',
                navajowhite: 'ffdead',
                navy: '000080',
                oldlace: 'fdf5e6',
                olive: '808000',
                olivedrab: '6b8e23',
                orange: 'ffa500',
                orangered: 'ff4500',
                orchid: 'da70d6',
                palegoldenrod: 'eee8aa',
                palegreen: '98fb98',
                paleturquoise: 'afeeee',
                palevioletred: 'd87093',
                papayawhip: 'ffefd5',
                peachpuff: 'ffdab9',
                peru: 'cd853f',
                pink: 'ffc0cb',
                plum: 'dda0dd',
                powderblue: 'b0e0e6',
                purple: '800080',
                red: 'ff0000',
                rosybrown: 'bc8f8f',
                royalblue: '4169e1',
                saddlebrown: '8b4513',
                salmon: 'fa8072',
                sandybrown: 'f4a460',
                seagreen: '2e8b57',
                seashell: 'fff5ee',
                sienna: 'a0522d',
                silver: 'c0c0c0',
                skyblue: '87ceeb',
                slateblue: '6a5acd',
                slategray: '708090',
                snow: 'fffafa',
                springgreen: '00ff7f',
                steelblue: '4682b4',
                tan: 'd2b48c',
                teal: '008080',
                thistle: 'd8bfd8',
                tomato: 'ff6347',
                turquoise: '40e0d0',
                violet: 'ee82ee',
                violetred: 'd02090',
                wheat: 'f5deb3',
                white: 'ffffff',
                whitesmoke: 'f5f5f5',
                yellow: 'ffff00',
                yellowgreen: '9acd32'
            };
        var standardColorTypes = [{
                    re: /^rgb\((\d{1,3}),\s*(\d{1,3}),\s*(\d{1,3})\)$/,
                    process: function(colorString) {
                        return [parseInt(colorString[1], 10), parseInt(colorString[2], 10), parseInt(colorString[3], 10)]
                    }
                }, {
                    re: /^rgba\((\d{1,3}),\s*(\d{1,3}),\s*(\d{1,3}),\s*(\d*\.*\d+)\)$/,
                    process: function(colorString) {
                        return [parseInt(colorString[1], 10), parseInt(colorString[2], 10), parseInt(colorString[3], 10), parseFloat(colorString[4])]
                    }
                }, {
                    re: /^#(\w{2})(\w{2})(\w{2})$/,
                    process: function(colorString) {
                        return [parseInt(colorString[1], 16), parseInt(colorString[2], 16), parseInt(colorString[3], 16)]
                    }
                }, {
                    re: /^#(\w{1})(\w{1})(\w{1})$/,
                    process: function(colorString) {
                        return [parseInt(colorString[1] + colorString[1], 16), parseInt(colorString[2] + colorString[2], 16), parseInt(colorString[3] + colorString[3], 16)]
                    }
                }, {
                    re: /^hsv\((\d{1,3}),\s*(\d{1,3}),\s*(\d{1,3})\)$/,
                    process: function(colorString) {
                        var h = parseInt(colorString[1], 10),
                            s = parseInt(colorString[2], 10),
                            v = parseInt(colorString[3], 10),
                            rgb = hsvToRgb(h, s, v);
                        return [rgb[0], rgb[1], rgb[2], 1, [h, s, v]]
                    }
                }, {
                    re: /^hsl\((\d{1,3}),\s*(\d{1,3}),\s*(\d{1,3})\)$/,
                    process: function(colorString) {
                        var h = parseInt(colorString[1], 10),
                            s = parseInt(colorString[2], 10),
                            l = parseInt(colorString[3], 10),
                            rgb = hslToRgb(h, s, l);
                        return [rgb[0], rgb[1], rgb[2], 1, null, [h, s, l]]
                    }
                }];
        function Color(value) {
            this.baseColor = value;
            var color;
            if (value) {
                color = String(value).toLowerCase().replace(/ /g, '');
                color = standardColorNames[color] ? '#' + standardColorNames[color] : color;
                color = parseColor(color)
            }
            if (!color)
                this.colorIsInvalid = true;
            color = color || {};
            this.r = normalize(color[0]);
            this.g = normalize(color[1]);
            this.b = normalize(color[2]);
            this.a = normalize(color[3], 1, 1);
            if (color[4])
                this.hsv = {
                    h: color[4][0],
                    s: color[4][1],
                    v: color[4][2]
                };
            else
                this.hsv = toHsvFromRgb(this.r, this.g, this.b);
            if (color[5])
                this.hsl = {
                    h: color[5][0],
                    s: color[5][1],
                    l: color[5][2]
                };
            else
                this.hsl = toHslFromRgb(this.r, this.g, this.b)
        }
        function parseColor(color) {
            if (color === "transparent")
                return [0, 0, 0, 0];
            var result,
                i = 0,
                ii = standardColorTypes.length,
                str;
            for (; i < ii; ++i) {
                str = standardColorTypes[i].re.exec(color);
                if (str)
                    return standardColorTypes[i].process(str)
            }
            return null
        }
        function normalize(colorComponent, def, max) {
            def = def || 0;
            max = max || 255;
            return colorComponent < 0 || isNaN(colorComponent) ? def : colorComponent > max ? max : colorComponent
        }
        function toHexFromRgb(r, g, b) {
            return '#' + (0X01000000 | r << 16 | g << 8 | b).toString(16).slice(1)
        }
        function toHsvFromRgb(r, g, b) {
            var max = Math.max(r, g, b),
                min = Math.min(r, g, b),
                delta = max - min,
                H,
                S,
                V;
            V = max;
            S = max === 0 ? 0 : 1 - min / max;
            if (max === min)
                H = 0;
            else
                switch (max) {
                    case r:
                        H = 60 * ((g - b) / delta);
                        if (g < b)
                            H = H + 360;
                        break;
                    case g:
                        H = 60 * ((b - r) / delta) + 120;
                        break;
                    case b:
                        H = 60 * ((r - g) / delta) + 240;
                        break
                }
            S *= 100;
            V *= 100 / 255;
            return {
                    h: Math.round(H),
                    s: Math.round(S),
                    v: Math.round(V)
                }
        }
        function hsvToRgb(h, s, v) {
            var Vdec,
                Vinc,
                Vmin,
                Hi,
                a,
                r,
                g,
                b;
            Hi = Math.floor(h / 60);
            Vmin = (100 - s) * v / 100;
            a = (v - Vmin) * (h % 60 / 60);
            Vinc = Vmin + a;
            Vdec = v - a;
            switch (Hi) {
                case 0:
                    r = v;
                    g = Vinc;
                    b = Vmin;
                    break;
                case 1:
                    r = Vdec;
                    g = v;
                    b = Vmin;
                    break;
                case 2:
                    r = Vmin;
                    g = v;
                    b = Vinc;
                    break;
                case 3:
                    r = Vmin;
                    g = Vdec;
                    b = v;
                    break;
                case 4:
                    r = Vinc;
                    g = Vmin;
                    b = v;
                    break;
                case 5:
                    r = v;
                    g = Vmin;
                    b = Vdec;
                    break
            }
            return [Math.round(r * 2.55), Math.round(g * 2.55), Math.round(b * 2.55)]
        }
        function calculateHue(r, g, b, delta) {
            var max = Math.max(r, g, b);
            switch (max) {
                case r:
                    return (g - b) / delta + (g < b ? 6 : 0);
                case g:
                    return (b - r) / delta + 2;
                case b:
                    return (r - g) / delta + 4
            }
        }
        function toHslFromRgb(r, g, b) {
            r = convertTo01Bounds(r, 255);
            g = convertTo01Bounds(g, 255);
            b = convertTo01Bounds(b, 255);
            var max = Math.max(r, g, b),
                min = Math.min(r, g, b),
                maxMinSumm = max + min,
                h,
                s,
                l = maxMinSumm / 2;
            if (max === min)
                h = s = 0;
            else {
                var delta = max - min;
                if (l > 0.5)
                    s = delta / (2 - maxMinSumm);
                else
                    s = delta / maxMinSumm;
                h = calculateHue(r, g, b, delta);
                h /= 6
            }
            return {
                    h: _round(h * 360),
                    s: _round(s * 100),
                    l: _round(l * 100)
                }
        }
        function makeTc(colorPart, h) {
            var Tc = h;
            if (colorPart === "r")
                Tc = h + 1 / 3;
            if (colorPart === "b")
                Tc = h - 1 / 3;
            return Tc
        }
        function modifyTc(Tc) {
            if (Tc < 0)
                Tc += 1;
            if (Tc > 1)
                Tc -= 1;
            return Tc
        }
        function hueToRgb(p, q, Tc) {
            Tc = modifyTc(Tc);
            if (Tc < 1 / 6)
                return p + (q - p) * 6 * Tc;
            if (Tc < 1 / 2)
                return q;
            if (Tc < 2 / 3)
                return p + (q - p) * (2 / 3 - Tc) * 6;
            return p
        }
        function hslToRgb(h, s, l) {
            var r,
                g,
                b,
                h = convertTo01Bounds(h, 360),
                s = convertTo01Bounds(s, 100),
                l = convertTo01Bounds(l, 100);
            if (s === 0)
                r = g = b = l;
            else {
                var q = l < 0.5 ? l * (1 + s) : l + s - l * s,
                    p = 2 * l - q;
                r = hueToRgb(p, q, makeTc("r", h));
                g = hueToRgb(p, q, makeTc("g", h));
                b = hueToRgb(p, q, makeTc("b", h))
            }
            return [_round(r * 255), _round(g * 255), _round(b * 255)]
        }
        function convertTo01Bounds(n, max) {
            n = Math.min(max, Math.max(0, parseFloat(n)));
            if (Math.abs(n - max) < 0.000001)
                return 1;
            return n % max / parseFloat(max)
        }
        function isIntegerBtwMinAndMax(number, min, max) {
            min = min || 0;
            max = max || 255;
            if (number % 1 !== 0 || number < min || number > max || typeof number !== 'number' || isNaN(number))
                return false;
            return true
        }
        var _round = Math.round;
        Color.prototype = {
            constructor: Color,
            highlight: function(step) {
                step = step || 10;
                return this.alter(step).toHex()
            },
            darken: function(step) {
                step = step || 10;
                return this.alter(-step).toHex()
            },
            alter: function(step) {
                var result = new Color;
                result.r = normalize(this.r + step);
                result.g = normalize(this.g + step);
                result.b = normalize(this.b + step);
                return result
            },
            blend: function(blendColor, opacity) {
                var other = blendColor instanceof Color ? blendColor : new Color(blendColor),
                    result = new Color;
                result.r = normalize(_round(this.r * (1 - opacity) + other.r * opacity));
                result.g = normalize(_round(this.g * (1 - opacity) + other.g * opacity));
                result.b = normalize(_round(this.b * (1 - opacity) + other.b * opacity));
                return result
            },
            toHex: function() {
                return toHexFromRgb(this.r, this.g, this.b)
            },
            getPureColor: function() {
                var rgb = hsvToRgb(this.hsv.h, 100, 100);
                return new Color("rgb(" + rgb.join(",") + ")")
            },
            isValidHex: function(hex) {
                return /(^#[0-9A-F]{6}$)|(^#[0-9A-F]{3}$)/i.test(hex)
            },
            isValidRGB: function(r, g, b) {
                if (!isIntegerBtwMinAndMax(r) || !isIntegerBtwMinAndMax(g) || !isIntegerBtwMinAndMax(b))
                    return false;
                return true
            },
            isValidAlpha: function(a) {
                if (isNaN(a) || a < 0 || a > 1 || typeof a !== 'number')
                    return false;
                return true
            },
            colorIsInvalid: false
        };
        DX.Color = Color
    })(DevExpress);
    /*! Module core, file localization.js */
    (function($, DX, undefined) {
        Globalize.localize = function(key, cultureSelector) {
            var neutral = (cultureSelector || this.cultureSelector || "").substring(0, 2);
            return this.findClosestCulture(cultureSelector).messages[key] || this.findClosestCulture(neutral).messages[key] || this.cultures["default"].messages[key]
        };
        var localization = function() {
                var newMessages = {};
                return {
                        setup: function(localizablePrefix) {
                            this.localizeString = function(text) {
                                var regex = new RegExp("(^|[^a-zA-Z_0-9" + localizablePrefix + "-]+)(" + localizablePrefix + "{1,2})([a-zA-Z_0-9-]+)", "g"),
                                    escapeString = localizablePrefix + localizablePrefix;
                                return text.replace(regex, function(str, prefix, escape, localizationKey) {
                                        var result = prefix + localizablePrefix + localizationKey;
                                        if (escape !== escapeString)
                                            if (Globalize.cultures["default"].messages[localizationKey])
                                                result = prefix + Globalize.localize(localizationKey);
                                            else
                                                newMessages[localizationKey] = DX.inflector.humanize(localizationKey);
                                        return result
                                    })
                            }
                        },
                        localizeNode: function(node) {
                            var that = this;
                            $(node).each(function(index, nodeItem) {
                                if (!nodeItem.nodeType)
                                    return;
                                if (nodeItem.nodeType === 3)
                                    nodeItem.nodeValue = that.localizeString(nodeItem.nodeValue);
                                else {
                                    $.each(nodeItem.attributes || [], function(index, attr) {
                                        if (typeof attr.value === "string") {
                                            var localizedValue = that.localizeString(attr.value);
                                            if (attr.value !== localizedValue)
                                                attr.value = localizedValue
                                        }
                                    });
                                    $(nodeItem).contents().each(function(index, node) {
                                        that.localizeNode(node)
                                    })
                                }
                            })
                        },
                        getDictionary: function(onlyNew) {
                            if (onlyNew)
                                return newMessages;
                            return $.extend({}, newMessages, Globalize.cultures["default"].messages)
                        }
                    }
            }();
        localization.setup("@");
        DX.localization = localization
    })(jQuery, DevExpress);
    /*! Module core, file localization.en.js */
    Globalize.addCultureInfo("default", {messages: {
            Yes: "Yes",
            No: "No",
            Cancel: "Cancel",
            Clear: "Clear",
            Done: "Done",
            Loading: "Loading...",
            Select: "Select...",
            Search: "Search",
            Back: "Back",
            OK: "OK",
            "dxLookup-searchPlaceholder": "Minimum character number: {0}",
            "dxCollectionContainerWidget-noDataText": "No data to display",
            "dxList-pullingDownText": "Pull down to refresh...",
            "dxList-pulledDownText": "Release to refresh...",
            "dxList-refreshingText": "Refreshing...",
            "dxList-pageLoadingText": "Loading...",
            "dxList-nextButtonText": "More",
            "dxListEditDecorator-delete": "Delete",
            "dxListEditDecorator-more": "More",
            "dxScrollView-pullingDownText": "Pull down to refresh...",
            "dxScrollView-pulledDownText": "Release to refresh...",
            "dxScrollView-refreshingText": "Refreshing...",
            "dxScrollView-reachBottomText": "Loading...",
            "dxSwitch-onText": "ON",
            "dxSwitch-offText": "OFF",
            "dxDateBox-simulatedDataPickerTitleTime": "Select time",
            "dxDateBox-simulatedDataPickerTitleDate": "Select date",
            "dxDateBox-simulatedDataPickerTitleDateTime": "Select date and time",
            "dxDataGrid-columnChooserTitle": "Column Chooser",
            "dxDataGrid-columnChooserEmptyText": "Drag a column here to hide it",
            "dxDataGrid-groupContinuesMessage": "Continues on the next page",
            "dxDataGrid-groupContinuedMessage": "Continued from the previous page",
            "dxDataGrid-editingEditRow": "Edit",
            "dxDataGrid-editingSaveRowChanges": "Save",
            "dxDataGrid-editingCancelRowChanges": "Cancel",
            "dxDataGrid-editingDeleteRow": "Delete",
            "dxDataGrid-editingUndeleteRow": "Undelete",
            "dxDataGrid-editingConfirmDeleteMessage": "Are you sure you want to delete this record?",
            "dxDataGrid-editingConfirmDeleteTitle": "",
            "dxDataGrid-groupPanelEmptyText": "Drag a column header here to group by that column",
            "dxDataGrid-noDataText": "No data",
            "dxDataGrid-searchPanelPlaceholder": "Search...",
            "dxDataGrid-filterRowShowAllText": "(All)",
            "dxDataGrid-filterRowResetOperationText": "Reset",
            "dxDataGrid-filterRowOperationEquals": "Equals",
            "dxDataGrid-filterRowOperationNotEquals": "Does not equal",
            "dxDataGrid-filterRowOperationLess": "Less than",
            "dxDataGrid-filterRowOperationLessOrEquals": "Less than or equal to",
            "dxDataGrid-filterRowOperationGreater": "Greater than",
            "dxDataGrid-filterRowOperationGreaterOrEquals": "Greater than or equal to",
            "dxDataGrid-filterRowOperationStartsWith": "Starts with",
            "dxDataGrid-filterRowOperationContains": "Contains",
            "dxDataGrid-filterRowOperationNotContains": "Does not contain",
            "dxDataGrid-filterRowOperationEndsWith": "Ends with",
            "dxDataGrid-trueText": "true",
            "dxDataGrid-falseText": "false",
            "dxDataGrid-sortingAscendingText": "Sort Ascending",
            "dxDataGrid-sortingDescendingText": "Sort Descending",
            "dxDataGrid-sortingClearText": "Clear Sorting"
        }});
    /*! Module core, file data.js */
    (function($, DX, undefined) {
        var bracketsToDots = function(expr) {
                return expr.replace(/\[/g, ".").replace(/\]/g, "")
            };
        var unwrapObservable = DX.utils.unwrapObservable;
        var isObservable = function(value) {
                return DX.support.hasKo && ko.isObservable(value)
            };
        var readPropValue = function(obj, propName) {
                if (propName === "this")
                    return obj;
                return obj[propName]
            };
        var assignPropValue = function(obj, propName, value) {
                if (propName === "this")
                    throw Error("Cannot assign to self");
                var propValue = obj[propName];
                if (isObservable(propValue))
                    propValue(value);
                else
                    obj[propName] = value
            };
        var compileGetter = function(expr) {
                if (arguments.length > 1)
                    expr = $.makeArray(arguments);
                if (!expr || expr === "this")
                    return function(obj) {
                            return obj
                        };
                if ($.isFunction(expr))
                    return expr;
                if ($.isArray(expr))
                    return combineGetters(expr);
                expr = bracketsToDots(expr);
                var path = expr.split(".");
                return function(obj, options) {
                        options = options || {};
                        var current = unwrapObservable(obj);
                        $.each(path, function() {
                            if (!current)
                                return false;
                            var next = unwrapObservable(current[this]);
                            if ($.isFunction(next) && !options.functionsAsIs)
                                next = next.call(current);
                            current = next
                        });
                        return current
                    }
            };
        var combineGetters = function(getters) {
                var compiledGetters = {};
                $.each(getters, function() {
                    compiledGetters[this] = compileGetter(this)
                });
                return function(obj, options) {
                        var result = {};
                        $.each(compiledGetters, function(name) {
                            var value = this(obj, options),
                                current,
                                path,
                                last,
                                i;
                            if (value === undefined)
                                return;
                            current = result;
                            path = name.split(".");
                            last = path.length - 1;
                            for (i = 0; i < last; i++)
                                current = current[path[i]] = {};
                            current[path[i]] = value
                        });
                        return result
                    }
            };
        var compileSetter = function(expr) {
                expr = expr || "this";
                expr = bracketsToDots(expr);
                var pos = expr.lastIndexOf("."),
                    targetGetter = compileGetter(expr.substr(0, pos)),
                    targetPropName = expr.substr(1 + pos);
                return function(obj, value, options) {
                        options = options || {};
                        var target = targetGetter(obj, {functionsAsIs: options.functionsAsIs}),
                            prevTargetValue = readPropValue(target, targetPropName);
                        if (!options.functionsAsIs && $.isFunction(prevTargetValue) && !isObservable(prevTargetValue))
                            target[targetPropName](value);
                        else {
                            prevTargetValue = unwrapObservable(prevTargetValue);
                            if (options.merge && $.isPlainObject(value) && (prevTargetValue === undefined || $.isPlainObject(prevTargetValue)) && !(value instanceof $.Event)) {
                                if (!prevTargetValue)
                                    assignPropValue(target, targetPropName, {});
                                DX.utils.deepExtendArraySafe(unwrapObservable(readPropValue(target, targetPropName)), value)
                            }
                            else
                                assignPropValue(target, targetPropName, value)
                        }
                    }
            };
        var normalizeBinaryCriterion = function(crit) {
                return [crit[0], crit.length < 3 ? "=" : String(crit[1]).toLowerCase(), crit.length < 2 ? true : crit[crit.length - 1]]
            };
        var normalizeSortingInfo = function(info) {
                if (!$.isArray(info))
                    info = [info];
                return $.map(info, function(i) {
                        return {
                                selector: $.isFunction(i) || typeof i === "string" ? i : i.getter || i.field || i.selector,
                                desc: !!(i.desc || String(i.dir).charAt(0).toLowerCase() === "d")
                            }
                    })
            };
        var Guid = DX.Class.inherit({
                ctor: function(value) {
                    if (value)
                        value = String(value);
                    this._value = this._normalize(value || this._generate())
                },
                _normalize: function(value) {
                    value = value.replace(/[^a-f0-9]/ig, "").toLowerCase();
                    while (value.length < 32)
                        value += "0";
                    return [value.substr(0, 8), value.substr(8, 4), value.substr(12, 4), value.substr(16, 4), value.substr(20, 12)].join("-")
                },
                _generate: function() {
                    var value = "";
                    for (var i = 0; i < 32; i++)
                        value += Math.round(Math.random() * 15).toString(16);
                    return value
                },
                toString: function() {
                    return this._value
                },
                valueOf: function() {
                    return this._value
                },
                toJSON: function() {
                    return this._value
                }
            });
        var toComparable = function(value, caseSensitive) {
                if (value instanceof Date)
                    return value.getTime();
                if (value instanceof Guid)
                    return value.valueOf();
                if (!caseSensitive && typeof value === "string")
                    return value.toLowerCase();
                return value
            };
        var keysEqual = function(keyExpr, key1, key2) {
                if ($.isArray(keyExpr)) {
                    var names = $.map(key1, function(v, k) {
                            return k
                        }),
                        name;
                    for (var i = 0; i < names.length; i++) {
                        name = names[i];
                        if (toComparable(key1[name], true) != toComparable(key2[name], true))
                            return false
                    }
                    return true
                }
                return toComparable(key1, true) == toComparable(key2, true)
            };
        var BASE64_CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
        var base64_encode = function(input) {
                if (!$.isArray(input))
                    input = stringToByteArray(String(input));
                var result = "";
                for (var i = 0; i < input.length; i += 3) {
                    var octet1 = input[i],
                        octet2 = input[i + 1],
                        octet3 = input[i + 2];
                    result += $.map([octet1 >> 2, (octet1 & 3) << 4 | octet2 >> 4, isNaN(octet2) ? 64 : (octet2 & 15) << 2 | octet3 >> 6, isNaN(octet3) ? 64 : octet3 & 63], function(item) {
                        return BASE64_CHARS.charAt(item)
                    }).join("")
                }
                return result
            };
        var stringToByteArray = function(str) {
                var bytes = [],
                    code,
                    i;
                for (i = 0; i < str.length; i++) {
                    code = str.charCodeAt(i);
                    if (code < 128)
                        bytes.push(code);
                    else if (code < 2048)
                        bytes.push(192 + (code >> 6), 128 + (code & 63));
                    else if (code < 65536)
                        bytes.push(224 + (code >> 12), 128 + (code >> 6 & 63), 128 + (code & 63));
                    else if (code < 2097152)
                        bytes.push(240 + (code >> 18), 128 + (code >> 12 & 63), 128 + (code >> 6 & 63), 128 + (code & 63))
                }
                return bytes
            };
        var errorMessageFromXhr = function() {
                var textStatusMessages = {
                        timeout: "Network connection timeout",
                        error: "Unspecified network error",
                        parsererror: "Unexpected server response"
                    };
                var textStatusDetails = {
                        timeout: "possible causes: the remote host is not accessible, overloaded or is not included into the domain white-list when being run in the native container",
                        error: "if the remote host is located on another domain, make sure it properly supports cross-origin resource sharing (CORS), or use the JSONP approach instead",
                        parsererror: "the remote host did not respond with valid JSON data"
                    };
                var explainTextStatus = function(textStatus) {
                        var result = textStatusMessages[textStatus];
                        if (!result)
                            return textStatus;
                        result += " (" + textStatusDetails[textStatus] + ")";
                        return result
                    };
                return function(xhr, textStatus) {
                        if (xhr.status < 400)
                            return explainTextStatus(textStatus);
                        return xhr.statusText
                    }
            }();
        var data = DX.data = {
                utils: {
                    compileGetter: compileGetter,
                    compileSetter: compileSetter,
                    normalizeBinaryCriterion: normalizeBinaryCriterion,
                    normalizeSortingInfo: normalizeSortingInfo,
                    toComparable: toComparable,
                    keysEqual: keysEqual,
                    errorMessageFromXhr: errorMessageFromXhr
                },
                Guid: Guid,
                base64_encode: base64_encode,
                queryImpl: {},
                queryAdapters: {},
                query: function() {
                    var impl = $.isArray(arguments[0]) ? "array" : "remote";
                    return data.queryImpl[impl].apply(this, arguments)
                },
                errorHandler: null,
                _handleError: function(error) {
                    if (window.console)
                        console.warn("[DevExpress.data]: " + error);
                    if (data.errorHandler)
                        data.errorHandler(error)
                }
            }
    })(jQuery, DevExpress);
    /*! Module core, file data.query.array.js */
    (function($, DX, undefined) {
        var Class = DX.Class,
            data = DX.data,
            queryImpl = data.queryImpl,
            compileGetter = data.utils.compileGetter,
            toComparable = data.utils.toComparable;
        var Iterator = Class.inherit({
                toArray: function() {
                    var result = [];
                    this.reset();
                    while (this.next())
                        result.push(this.current());
                    return result
                },
                countable: function() {
                    return false
                }
            });
        var ArrayIterator = Iterator.inherit({
                ctor: function(array) {
                    this.array = array;
                    this.index = -1
                },
                next: function() {
                    if (this.index + 1 < this.array.length) {
                        this.index++;
                        return true
                    }
                    return false
                },
                current: function() {
                    return this.array[this.index]
                },
                reset: function() {
                    this.index = -1
                },
                toArray: function() {
                    return this.array.slice(0)
                },
                countable: function() {
                    return true
                },
                count: function() {
                    return this.array.length
                }
            });
        var WrappedIterator = Iterator.inherit({
                ctor: function(iter) {
                    this.iter = iter
                },
                next: function() {
                    return this.iter.next()
                },
                current: function() {
                    return this.iter.current()
                },
                reset: function() {
                    return this.iter.reset()
                }
            });
        var MapIterator = WrappedIterator.inherit({
                ctor: function(iter, mapper) {
                    this.callBase(iter);
                    this.index = -1;
                    this.mapper = mapper
                },
                current: function() {
                    return this.mapper(this.callBase(), this.index)
                },
                next: function() {
                    var hasNext = this.callBase();
                    if (hasNext)
                        this.index++;
                    return hasNext
                }
            });
        var SortIterator = Iterator.inherit({
                ctor: function(iter, getter, desc) {
                    if (!(iter instanceof MapIterator))
                        iter = new MapIterator(iter, this._wrap);
                    this.iter = iter;
                    this.rules = [{
                            getter: getter,
                            desc: desc
                        }]
                },
                thenBy: function(getter, desc) {
                    var result = new SortIterator(this.sortedIter || this.iter, getter, desc);
                    if (!this.sortedIter)
                        result.rules = this.rules.concat(result.rules);
                    return result
                },
                next: function() {
                    this._ensureSorted();
                    return this.sortedIter.next()
                },
                current: function() {
                    this._ensureSorted();
                    return this.sortedIter.current()
                },
                reset: function() {
                    delete this.sortedIter
                },
                countable: function() {
                    return this.sortedIter || this.iter.countable()
                },
                count: function() {
                    if (this.sortedIter)
                        return this.sortedIter.count();
                    return this.iter.count()
                },
                _ensureSorted: function() {
                    if (this.sortedIter)
                        return;
                    $.each(this.rules, function() {
                        this.getter = compileGetter(this.getter)
                    });
                    this.sortedIter = new MapIterator(new ArrayIterator(this.iter.toArray().sort($.proxy(this._compare, this))), this._unwrap)
                },
                _wrap: function(record, index) {
                    return {
                            index: index,
                            value: record
                        }
                },
                _unwrap: function(wrappedItem) {
                    return wrappedItem.value
                },
                _compare: function(x, y) {
                    var xIndex = x.index,
                        yIndex = y.index;
                    x = x.value;
                    y = y.value;
                    if (x === y)
                        return xIndex - yIndex;
                    for (var i = 0, rulesCount = this.rules.length; i < rulesCount; i++) {
                        var rule = this.rules[i],
                            xValue = toComparable(rule.getter(x)),
                            yValue = toComparable(rule.getter(y)),
                            factor = rule.desc ? -1 : 1;
                        if (xValue < yValue)
                            return -factor;
                        if (xValue > yValue)
                            return factor;
                        if (xValue !== yValue)
                            return !xValue ? -factor : factor
                    }
                    return xIndex - yIndex
                }
            });
        var compileCriteria = function() {
                var compileGroup = function(crit) {
                        var operands = [],
                            bag = ["return function(d) { return "],
                            index = 0,
                            pushAnd = false;
                        $.each(crit, function() {
                            if ($.isArray(this) || $.isFunction(this)) {
                                if (pushAnd)
                                    bag.push(" && ");
                                operands.push(compileCriteria(this));
                                bag.push("op[", index, "](d)");
                                index++;
                                pushAnd = true
                            }
                            else {
                                bag.push(/and|&/i.test(this) ? " && " : " || ");
                                pushAnd = false
                            }
                        });
                        bag.push(" }");
                        return new Function("op", bag.join(""))(operands)
                    };
                var toString = function(value) {
                        return DX.utils.isDefined(value) ? value.toString() : ''
                    };
                var compileBinary = function(crit) {
                        crit = data.utils.normalizeBinaryCriterion(crit);
                        var getter = compileGetter(crit[0]),
                            op = crit[1],
                            value = crit[2];
                        value = toComparable(value);
                        switch (op.toLowerCase()) {
                            case"=":
                                return compileEquals(getter, value);
                            case"<>":
                                return compileEquals(getter, value, true);
                            case">":
                                return function(obj) {
                                        return toComparable(getter(obj)) > value
                                    };
                            case"<":
                                return function(obj) {
                                        return toComparable(getter(obj)) < value
                                    };
                            case">=":
                                return function(obj) {
                                        return toComparable(getter(obj)) >= value
                                    };
                            case"<=":
                                return function(obj) {
                                        return toComparable(getter(obj)) <= value
                                    };
                            case"startswith":
                                return function(obj) {
                                        return toComparable(toString(getter(obj))).indexOf(value) === 0
                                    };
                            case"endswith":
                                return function(obj) {
                                        var getterValue = toComparable(toString(getter(obj)));
                                        return getterValue.lastIndexOf(value) === getterValue.length - toString(value).length
                                    };
                            case"contains":
                                return function(obj) {
                                        return toComparable(toString(getter(obj))).indexOf(value) > -1
                                    };
                            case"notcontains":
                                return function(obj) {
                                        return toComparable(toString(getter(obj))).indexOf(value) === -1
                                    }
                        }
                        throw Error("Unknown filter operation: " + op);
                    };
                function compileEquals(getter, value, negate) {
                    return function(obj) {
                            obj = toComparable(getter(obj));
                            var result = useStrictComparison(value) ? obj === value : obj == value;
                            if (negate)
                                result = !result;
                            return result
                        }
                }
                function useStrictComparison(value) {
                    return value === "" || value === 0 || value === null || value === false || value === undefined
                }
                return function(crit) {
                        if ($.isFunction(crit))
                            return crit;
                        if ($.isArray(crit[0]))
                            return compileGroup(crit);
                        return compileBinary(crit)
                    }
            }();
        var FilterIterator = WrappedIterator.inherit({
                ctor: function(iter, criteria) {
                    this.callBase(iter);
                    this.criteria = compileCriteria(criteria)
                },
                next: function() {
                    while (this.iter.next())
                        if (this.criteria(this.current()))
                            return true;
                    return false
                }
            });
        var GroupIterator = Iterator.inherit({
                ctor: function(iter, getter) {
                    this.iter = iter;
                    this.getter = getter
                },
                next: function() {
                    this._ensureGrouped();
                    return this.groupedIter.next()
                },
                current: function() {
                    this._ensureGrouped();
                    return this.groupedIter.current()
                },
                reset: function() {
                    delete this.groupedIter
                },
                countable: function() {
                    return !!this.groupedIter
                },
                count: function() {
                    return this.groupedIter.count()
                },
                _ensureGrouped: function() {
                    if (this.groupedIter)
                        return;
                    var hash = {},
                        keys = [],
                        iter = this.iter,
                        getter = compileGetter(this.getter);
                    iter.reset();
                    while (iter.next()) {
                        var current = iter.current(),
                            key = getter(current);
                        if (key in hash)
                            hash[key].push(current);
                        else {
                            hash[key] = [current];
                            keys.push(key)
                        }
                    }
                    this.groupedIter = new ArrayIterator($.map(keys, function(key) {
                        return {
                                key: key,
                                items: hash[key]
                            }
                    }))
                }
            });
        var SelectIterator = WrappedIterator.inherit({
                ctor: function(iter, getter) {
                    this.callBase(iter);
                    this.getter = compileGetter(getter)
                },
                current: function() {
                    return this.getter(this.callBase())
                },
                countable: function() {
                    return this.iter.countable()
                },
                count: function() {
                    return this.iter.count()
                }
            });
        var SliceIterator = WrappedIterator.inherit({
                ctor: function(iter, skip, take) {
                    this.callBase(iter);
                    this.skip = Math.max(0, skip);
                    this.take = Math.max(0, take);
                    this.pos = 0
                },
                next: function() {
                    if (this.pos >= this.skip + this.take)
                        return false;
                    while (this.pos < this.skip && this.iter.next())
                        this.pos++;
                    this.pos++;
                    return this.iter.next()
                },
                reset: function() {
                    this.callBase();
                    this.pos = 0
                },
                countable: function() {
                    return this.iter.countable()
                },
                count: function() {
                    return Math.min(this.iter.count() - this.skip, this.take)
                }
            });
        queryImpl.array = function(iter, queryOptions) {
            queryOptions = queryOptions || {};
            if (!(iter instanceof Iterator))
                iter = new ArrayIterator(iter);
            var handleError = function(error) {
                    var handler = queryOptions.errorHandler;
                    if (handler)
                        handler(error);
                    data._handleError(error)
                };
            var aggregate = function(seed, step, finalize) {
                    var d = $.Deferred().fail(handleError);
                    try {
                        iter.reset();
                        if (arguments.length < 2) {
                            step = arguments[0];
                            seed = iter.next() ? iter.current() : undefined
                        }
                        var accumulator = seed;
                        while (iter.next())
                            accumulator = step(accumulator, iter.current());
                        d.resolve(finalize ? finalize(accumulator) : accumulator)
                    }
                    catch(x) {
                        d.reject(x)
                    }
                    return d.promise()
                };
            var select = function(getter) {
                    if (!$.isFunction(getter) && !$.isArray(getter))
                        getter = $.makeArray(arguments);
                    return chainQuery(new SelectIterator(iter, getter))
                };
            var selectProp = function(name) {
                    return select(compileGetter(name))
                };
            var chainQuery = function(iter) {
                    return queryImpl.array(iter, queryOptions)
                };
            return {
                    toArray: function() {
                        return iter.toArray()
                    },
                    enumerate: function() {
                        var d = $.Deferred().fail(handleError);
                        try {
                            d.resolve(iter.toArray())
                        }
                        catch(x) {
                            d.reject(x)
                        }
                        return d.promise()
                    },
                    sortBy: function(getter, desc) {
                        return chainQuery(new SortIterator(iter, getter, desc))
                    },
                    thenBy: function(getter, desc) {
                        if (iter instanceof SortIterator)
                            return chainQuery(iter.thenBy(getter, desc));
                        throw Error();
                    },
                    filter: function(criteria) {
                        if (!$.isArray(criteria))
                            criteria = $.makeArray(arguments);
                        return chainQuery(new FilterIterator(iter, criteria))
                    },
                    slice: function(skip, take) {
                        if (take === undefined)
                            take = Number.MAX_VALUE;
                        return chainQuery(new SliceIterator(iter, skip, take))
                    },
                    select: select,
                    groupBy: function(getter) {
                        return chainQuery(new GroupIterator(iter, getter))
                    },
                    aggregate: aggregate,
                    count: function() {
                        if (iter.countable()) {
                            var d = $.Deferred().fail(handleError);
                            try {
                                d.resolve(iter.count())
                            }
                            catch(x) {
                                d.reject(x)
                            }
                            return d.promise()
                        }
                        return aggregate(0, function(count) {
                                return 1 + count
                            })
                    },
                    sum: function(getter) {
                        if (getter)
                            return selectProp(getter).sum();
                        return aggregate(0, function(sum, item) {
                                return sum + item
                            })
                    },
                    min: function(getter) {
                        if (getter)
                            return selectProp(getter).min();
                        return aggregate(function(min, item) {
                                return item < min ? item : min
                            })
                    },
                    max: function(getter) {
                        if (getter)
                            return selectProp(getter).max();
                        return aggregate(function(max, item) {
                                return item > max ? item : max
                            })
                    },
                    avg: function(getter) {
                        if (getter)
                            return selectProp(getter).avg();
                        var count = 0;
                        return aggregate(0, function(sum, item) {
                                count++;
                                return sum + item
                            }, function(sum) {
                                return count ? sum / count : undefined
                            })
                    }
                }
        }
    })(jQuery, DevExpress);
    /*! Module core, file data.query.remote.js */
    (function($, DX, undefined) {
        var data = DX.data,
            queryImpl = data.queryImpl;
        queryImpl.remote = function(url, queryOptions, tasks) {
            tasks = tasks || [];
            queryOptions = queryOptions || {};
            var createTask = function(name, args) {
                    return {
                            name: name,
                            args: args
                        }
                };
            var exec = function(executorTask) {
                    var d = $.Deferred(),
                        adapterFactory,
                        adapter,
                        taskQueue,
                        currentTask;
                    var rejectWithNotify = function(error) {
                            var handler = queryOptions.errorHandler;
                            if (handler)
                                handler(error);
                            data._handleError(error);
                            d.reject(error)
                        };
                    try {
                        adapterFactory = queryOptions.adapter || "odata";
                        if (!$.isFunction(adapterFactory))
                            adapterFactory = data.queryAdapters[adapterFactory];
                        adapter = adapterFactory(queryOptions);
                        taskQueue = [].concat(tasks).concat(executorTask);
                        while (taskQueue.length) {
                            currentTask = taskQueue[0];
                            if (String(currentTask.name) !== "enumerate")
                                if (!adapter[currentTask.name] || adapter[currentTask.name].apply(adapter, currentTask.args) === false)
                                    break;
                            taskQueue.shift()
                        }
                        adapter.exec(url).done(function(result, extra) {
                            if (!taskQueue.length)
                                d.resolve(result, extra);
                            else {
                                var clientChain = queryImpl.array(result, {errorHandler: queryOptions.errorHandler});
                                $.each(taskQueue, function() {
                                    clientChain = clientChain[this.name].apply(clientChain, this.args)
                                });
                                clientChain.done($.proxy(d.resolve, d)).fail($.proxy(d.reject, d))
                            }
                        }).fail(rejectWithNotify)
                    }
                    catch(x) {
                        rejectWithNotify(x)
                    }
                    return d.promise()
                };
            var query = {};
            $.each(["sortBy", "thenBy", "filter", "slice", "select", "groupBy"], function() {
                var name = this;
                query[name] = function() {
                    return queryImpl.remote(url, queryOptions, tasks.concat(createTask(name, arguments)))
                }
            });
            $.each(["count", "min", "max", "sum", "avg", "aggregate", "enumerate"], function() {
                var name = this;
                query[name] = function() {
                    return exec.call(this, createTask(name, arguments))
                }
            });
            return query
        }
    })(jQuery, DevExpress);
    /*! Module core, file data.odata.js */
    (function($, DX, undefined) {
        var data = DX.data,
            utils = DX.utils,
            Guid = data.Guid;
        var VERBOSE_DATE_REGEX = /^\/Date\((-?\d+)((\+|-)?(\d+)?)\)\/$/;
        var ISO8601_DATE_REGEX = /^(\d{4})(?:-?W(\d+)(?:-?(\d+)D?)?|(?:-(\d+))?-(\d+))(?:[T ](\d+):(\d+)(?::(\d+)(?:\.(\d+))?)?)?(?:Z(-?\d*))?$/;
        var TIMEZONE_OFFSET = (new Date).getTimezoneOffset() * 60 * 1000;
        var JSON_VERBOSE_MIME_TYPE = "application/json;odata=verbose";
        function stringify(object) {
            return JSON.stringify(object, replacer);
            function replacer(key, value) {
                if (this[key] instanceof Date)
                    return utils.formatIso8601Date(this[key]);
                return value
            }
        }
        var ajaxOptionsForRequest = function(request, requestOptions) {
                request = $.extend({
                    method: "get",
                    url: "",
                    params: {},
                    payload: null,
                    headers: {}
                }, request);
                requestOptions = requestOptions || {};
                var beforeSend = requestOptions.beforeSend;
                if (beforeSend)
                    beforeSend(request);
                var method = (request.method || "get").toLowerCase(),
                    isGet = method === "get",
                    useJsonp = isGet && requestOptions.jsonp,
                    params = $.extend({}, request.params),
                    ajaxData = isGet ? params : stringify(request.payload),
                    qs = !isGet && $.param(params),
                    url = request.url,
                    contentType = !isGet && JSON_VERBOSE_MIME_TYPE;
                if (qs)
                    url += (url.indexOf("?") > -1 ? "&" : "?") + qs;
                if (useJsonp)
                    ajaxData["$format"] = "json";
                return {
                        url: url,
                        data: ajaxData,
                        dataType: useJsonp ? "jsonp" : "json",
                        jsonp: useJsonp && "$callback",
                        type: method,
                        timeout: 30000,
                        headers: request.headers,
                        contentType: contentType,
                        accepts: {json: [JSON_VERBOSE_MIME_TYPE, "text/plain"].join()},
                        xhrFields: {withCredentials: requestOptions.withCredentials}
                    }
            };
        var sendRequest = function(request, requestOptions) {
                var d = $.Deferred();
                $.ajax(ajaxOptionsForRequest(request, requestOptions)).always(function(obj, textStatus) {
                    var tuplet = interpretJsonFormat(obj, textStatus),
                        error = tuplet.error,
                        data = tuplet.data,
                        nextUrl = tuplet.nextUrl,
                        extra;
                    if (error)
                        d.reject(error);
                    else if (requestOptions.countOnly)
                        d.resolve(tuplet.count);
                    else if (nextUrl)
                        sendRequest({url: nextUrl}, requestOptions).fail($.proxy(d.reject, d)).done(function(nextData) {
                            d.resolve(data.concat(nextData))
                        });
                    else {
                        if (isFinite(tuplet.count))
                            extra = {totalCount: tuplet.count};
                        d.resolve(data, extra)
                    }
                });
                return d.promise()
            };
        var formatDotNetError = function(errorObj) {
                var message,
                    currentError = errorObj;
                if ("message" in errorObj)
                    if (errorObj.message.value)
                        message = errorObj.message.value;
                    else
                        message = errorObj.message;
                while (currentError = currentError.innererror || currentError.internalexception) {
                    message = currentError.message;
                    if (currentError.internalexception && message.indexOf("inner exception") === -1)
                        break
                }
                return message
            };
        var errorFromResponse = function(obj, textStatus) {
                if (textStatus === "nocontent")
                    return null;
                var httpStatus = 200,
                    message = "Unknown error",
                    response = obj;
                if (textStatus !== "success") {
                    httpStatus = obj.status;
                    message = data.utils.errorMessageFromXhr(obj, textStatus);
                    try {
                        response = $.parseJSON(obj.responseText)
                    }
                    catch(x) {}
                }
                var errorObj = response && (response.error || response["@odata.error"]);
                if (errorObj) {
                    message = formatDotNetError(errorObj) || message;
                    if (httpStatus === 200)
                        httpStatus = 500;
                    if (errorObj.code)
                        httpStatus = Number(errorObj.code);
                    return $.extend(Error(message), {
                            httpStatus: httpStatus,
                            errorDetails: errorObj
                        })
                }
                else if (httpStatus !== 200)
                    return $.extend(Error(message), {httpStatus: httpStatus})
            };
        var interpretJsonFormat = function(obj, textStatus) {
                var error = errorFromResponse(obj, textStatus),
                    value;
                if (error)
                    return {error: error};
                if (!$.isPlainObject(obj))
                    return {data: obj};
                if ("d" in obj && (utils.isArray(obj.d) || utils.isObject(obj.d)))
                    value = interpretVerboseJsonFormat(obj, textStatus);
                else
                    value = interpretLightJsonFormat(obj, textStatus);
                transformDates(value);
                return value
            };
        var interpretVerboseJsonFormat = function(obj, textStatus) {
                var data = obj.d;
                if (!data)
                    return {error: Error("Malformed or unsupported JSON response received")};
                data = data.results || data;
                return {
                        data: data,
                        nextUrl: obj.d.__next,
                        count: parseInt(obj.d.__count, 10)
                    }
            };
        var interpretLightJsonFormat = function(obj, textStatus) {
                var data = obj.value || obj;
                if (!data)
                    return {error: Error("Malformed or unsupported JSON response received")};
                return {
                        data: data,
                        nextUrl: obj["@odata.nextLink"],
                        count: parseInt(obj["@odata.count"], 10)
                    }
            };
        var EdmLiteral = DX.Class.inherit({
                ctor: function(value) {
                    this._value = value
                },
                valueOf: function() {
                    return this._value
                }
            });
        var transformDates = function(obj) {
                $.each(obj, function(key, value) {
                    if (value !== null && typeof value === "object")
                        transformDates(value);
                    else if (typeof value === "string")
                        if (value.match(VERBOSE_DATE_REGEX))
                            obj[key] = new Date(TIMEZONE_OFFSET + Number(RegExp.$1) + RegExp.$2 * 60000);
                        else if (ISO8601_DATE_REGEX.test(value))
                            obj[key] = new Date(TIMEZONE_OFFSET + utils.parseIso8601Date(obj[key]).valueOf())
                })
            };
        var serializeDate = function() {
                var pad = function(part) {
                        part = String(part);
                        if (part.length < 2)
                            part = "0" + part;
                        return part
                    };
                return function(date) {
                        var result = ["datetime'", date.getFullYear(), "-", pad(date.getMonth() + 1), "-", pad(date.getDate())];
                        if (date.getHours() || date.getMinutes() || date.getSeconds() || date.getMilliseconds()) {
                            result.push("T", pad(date.getHours()), ":", pad(date.getMinutes()), ":", pad(date.getSeconds()));
                            if (date.getMilliseconds())
                                result.push(".", date.getMilliseconds())
                        }
                        result.push("'");
                        return result.join("")
                    }
            }();
        var serializeString = function(value) {
                return "'" + value.replace(/</g, "%26lt").replace(/'/g, "''") + "'"
            };
        var serializePropName = function(propName) {
                if (propName instanceof EdmLiteral)
                    return propName.valueOf();
                return propName.replace(/\./g, "/")
            };
        var serializeValueV4 = function(value) {
                if (value instanceof Date)
                    return utils.formatIso8601Date(value);
                return serializeValueV2(value)
            };
        var serializeValueV2 = function(value) {
                if (value instanceof Date)
                    return serializeDate(value);
                if (value instanceof Guid)
                    return "guid'" + value + "'";
                if (value instanceof EdmLiteral)
                    return value.valueOf();
                if (typeof value === "string")
                    return serializeString(value);
                return String(value)
            };
        var DEFAULT_PROTOCOL_VERSION = 2;
        var serializeValue = function(value, protocolVersion) {
                protocolVersion = protocolVersion || DEFAULT_PROTOCOL_VERSION;
                switch (protocolVersion) {
                    case 2:
                    case 3:
                        return serializeValueV2(value);
                    case 4:
                        return serializeValueV4(value);
                    default:
                        throw new Error("Unknown OData protocol version");
                }
            };
        var serializeKey = function(key, protocolVersion) {
                if ($.isPlainObject(key)) {
                    var parts = [];
                    $.each(key, function(k, v) {
                        parts.push(serializePropName(k) + "=" + serializeValue(v, protocolVersion))
                    });
                    return parts.join()
                }
                return serializeValue(key)
            };
        var keyConverters = {
                String: function(value) {
                    return value + ""
                },
                Int32: function(value) {
                    return Math.floor(value)
                },
                Int64: function(value) {
                    if (value instanceof EdmLiteral)
                        return value;
                    return new EdmLiteral(value + "L")
                },
                Guid: function(value) {
                    if (value instanceof Guid)
                        return value;
                    return new Guid(value)
                },
                Boolean: function(value) {
                    return !!value
                },
                Single: function(value) {
                    if (value instanceof EdmLiteral)
                        return value;
                    return new EdmLiteral(value + "f")
                },
                Decimal: function(value) {
                    if (value instanceof EdmLiteral)
                        return value;
                    return new EdmLiteral(value + "m")
                }
            };
        var compileCriteria = function() {
                var createBinaryOperationFormatter = function(op) {
                        return function(prop, val, bag) {
                                bag.push(prop, " ", op, " ", val)
                            }
                    };
                var createStringFuncFormatter = function(op, reverse) {
                        return function(prop, val, bag) {
                                if (reverse)
                                    bag.push(op, "(", val, ",", prop, ")");
                                else
                                    bag.push(op, "(", prop, ",", val, ")")
                            }
                    };
                var formatters = {
                        "=": createBinaryOperationFormatter("eq"),
                        "<>": createBinaryOperationFormatter("ne"),
                        ">": createBinaryOperationFormatter("gt"),
                        ">=": createBinaryOperationFormatter("ge"),
                        "<": createBinaryOperationFormatter("lt"),
                        "<=": createBinaryOperationFormatter("le"),
                        startswith: createStringFuncFormatter("startswith"),
                        endswith: createStringFuncFormatter("endswith")
                    };
                var formattersV2 = $.extend({}, formatters, {
                        contains: createStringFuncFormatter("substringof", true),
                        notcontains: createStringFuncFormatter("not substringof", true)
                    });
                var formattersV4 = $.extend({}, formatters, {
                        contains: createStringFuncFormatter("contains"),
                        notcontains: createStringFuncFormatter("not contains")
                    });
                var compileBinary = function(criteria, bag, protocolVersion) {
                        criteria = data.utils.normalizeBinaryCriterion(criteria);
                        var op = criteria[1],
                            formatters = protocolVersion === 4 ? formattersV4 : formattersV2,
                            formatter = formatters[op.toLowerCase()];
                        if (!formatter)
                            throw Error("Unknown filter operation: " + op);
                        formatter(serializePropName(criteria[0]), serializeValue(criteria[2], protocolVersion), bag)
                    };
                var compileGroup = function(criteria, bag, protocolVersion) {
                        var pushAnd = false;
                        $.each(criteria, function() {
                            if ($.isArray(this)) {
                                if (pushAnd)
                                    bag.push(" and ");
                                bag.push("(");
                                compileCore(this, bag, protocolVersion);
                                bag.push(")");
                                pushAnd = true
                            }
                            else {
                                bag.push(/and|&/i.test(this) ? " and " : " or ");
                                pushAnd = false
                            }
                        })
                    };
                var compileCore = function(criteria, bag, protocolVersion) {
                        if ($.isArray(criteria[0]))
                            compileGroup(criteria, bag, protocolVersion);
                        else
                            compileBinary(criteria, bag, protocolVersion)
                    };
                return function(criteria, protocolVersion) {
                        var bag = [];
                        compileCore(criteria, bag, protocolVersion);
                        return bag.join("")
                    }
            }();
        var createODataQueryAdapter = function(queryOptions) {
                var sorting = [],
                    criteria = [],
                    select,
                    skip,
                    take,
                    countQuery;
                var hasSlice = function() {
                        return skip || take !== undefined
                    };
                var sortCore = function(getter, desc, reset) {
                        if (hasSlice() || typeof getter !== "string")
                            return false;
                        if (reset)
                            sorting = [];
                        var rule = serializePropName(getter);
                        if (desc)
                            rule += " desc";
                        sorting.push(rule)
                    };
                var generateExpand = function() {
                        var hash = {};
                        if (queryOptions.expand)
                            $.each($.makeArray(queryOptions.expand), function() {
                                hash[serializePropName(this)] = 1
                            });
                        if (select)
                            $.each(select, function() {
                                var path = this.split(".");
                                if (path.length < 2)
                                    return;
                                path.pop();
                                hash[serializePropName(path.join("."))] = 1
                            });
                        return $.map(hash, function(k, v) {
                                return v
                            }).join() || undefined
                    };
                var requestData = function() {
                        var result = {};
                        if (!countQuery) {
                            if (sorting.length)
                                result["$orderby"] = sorting.join(",");
                            if (skip)
                                result["$skip"] = skip;
                            if (take !== undefined)
                                result["$top"] = take;
                            if (select)
                                result["$select"] = serializePropName(select.join());
                            result["$expand"] = generateExpand()
                        }
                        if (criteria.length)
                            result["$filter"] = compileCriteria(criteria.length < 2 ? criteria[0] : criteria, queryOptions.version);
                        if (countQuery)
                            result["$top"] = 0;
                        if (queryOptions.requireTotalCount || countQuery)
                            if (queryOptions.version !== 4)
                                result["$inlinecount"] = "allpages";
                            else
                                result["$count"] = "true";
                        return result
                    };
                return {
                        exec: function(url) {
                            return sendRequest({
                                    url: url,
                                    params: $.extend(requestData(), queryOptions && queryOptions.params)
                                }, {
                                    beforeSend: queryOptions.beforeSend,
                                    jsonp: queryOptions.jsonp,
                                    withCredentials: queryOptions.withCredentials,
                                    countOnly: countQuery
                                })
                        },
                        sortBy: function(getter, desc) {
                            return sortCore(getter, desc, true)
                        },
                        thenBy: function(getter, desc) {
                            return sortCore(getter, desc, false)
                        },
                        slice: function(skipCount, takeCount) {
                            if (hasSlice())
                                return false;
                            skip = skipCount;
                            take = takeCount
                        },
                        filter: function(criterion) {
                            if (hasSlice() || $.isFunction(criterion))
                                return false;
                            if (!$.isArray(criterion))
                                criterion = $.makeArray(arguments);
                            if (criteria.length)
                                criteria.push("and");
                            criteria.push(criterion)
                        },
                        select: function(expr) {
                            if (select || $.isFunction(expr))
                                return false;
                            if (!$.isArray(expr))
                                expr = $.makeArray(arguments);
                            select = expr
                        },
                        count: function() {
                            countQuery = true
                        }
                    }
            };
        $.extend(true, data, {
            EdmLiteral: EdmLiteral,
            utils: {odata: {
                    sendRequest: sendRequest,
                    serializePropName: serializePropName,
                    serializeValue: serializeValue,
                    serializeKey: serializeKey,
                    keyConverters: keyConverters
                }},
            queryAdapters: {odata: createODataQueryAdapter}
        });
        data.OData__internals = {interpretJsonFormat: interpretJsonFormat}
    })(jQuery, DevExpress);
    /*! Module core, file data.store.abstract.js */
    (function($, DX, undefined) {
        var Class = DX.Class,
            abstract = DX.abstract,
            data = DX.data,
            normalizeSortingInfo = data.utils.normalizeSortingInfo;
        var STORE_CALLBACK_NAMES = ["loading", "loaded", "modifying", "modified", "inserting", "inserted", "updating", "updated", "removing", "removed"];
        function multiLevelGroup(query, groupInfo) {
            query = query.groupBy(groupInfo[0].selector);
            if (groupInfo.length > 1)
                query = query.select(function(g) {
                    return $.extend({}, g, {items: multiLevelGroup(data.query(g.items), groupInfo.slice(1)).toArray()})
                });
            return query
        }
        data.utils.multiLevelGroup = multiLevelGroup;
        function arrangeSortingInfo(groupInfo, sortInfo) {
            var filteredGroup = [];
            $.each(groupInfo, function(_, group) {
                var collision = $.grep(sortInfo, function(sort) {
                        return group.selector == sort.selector
                    });
                if (collision.length < 1)
                    filteredGroup.push(group)
            });
            return filteredGroup.concat(sortInfo)
        }
        data.Store = Class.inherit({
            ctor: function(options) {
                var that = this;
                options = options || {};
                $.each(STORE_CALLBACK_NAMES, function() {
                    var callbacks = that[this] = $.Callbacks();
                    if (this in options)
                        callbacks.add(options[this])
                });
                this._key = options.key;
                this._errorHandler = options.errorHandler;
                this._useDefaultSearch = true
            },
            _customLoadOptions: function() {
                return null
            },
            key: function() {
                return this._key
            },
            keyOf: function(obj) {
                if (!this._keyGetter)
                    this._keyGetter = data.utils.compileGetter(this.key());
                return this._keyGetter(obj)
            },
            _requireKey: function() {
                if (!this.key())
                    throw Error("Key expression is required for this operation");
            },
            load: function(options) {
                var that = this;
                options = options || {};
                this.loading.fire(options);
                return this._loadImpl(options).done(function(result, extra) {
                        that.loaded.fire(result, extra)
                    })
            },
            _loadImpl: function(options) {
                var filter = options.filter,
                    sort = options.sort,
                    select = options.select,
                    group = options.group,
                    skip = options.skip,
                    take = options.take,
                    q = this.createQuery(options);
                if (filter)
                    q = q.filter(filter);
                if (group)
                    group = normalizeSortingInfo(group);
                if (sort || group) {
                    sort = normalizeSortingInfo(sort || []);
                    if (group)
                        sort = arrangeSortingInfo(group, sort);
                    $.each(sort, function(index) {
                        q = q[index ? "thenBy" : "sortBy"](this.selector, this.desc)
                    })
                }
                if (select)
                    q = q.select(select);
                if (group)
                    q = multiLevelGroup(q, group);
                if (take || skip)
                    q = q.slice(skip || 0, take);
                return q.enumerate()
            },
            createQuery: abstract,
            totalCount: function(options) {
                return this._addFailHandlers(this._totalCountImpl(options))
            },
            _totalCountImpl: function(options) {
                options = options || {};
                var q = this.createQuery(),
                    group = options.group,
                    filter = options.filter;
                if (filter)
                    q = q.filter(filter);
                if (group) {
                    group = normalizeSortingInfo(group);
                    q = multiLevelGroup(q, group)
                }
                return q.count()
            },
            byKey: function(key, extraOptions) {
                return this._addFailHandlers(this._byKeyImpl(key, extraOptions))
            },
            _byKeyImpl: abstract,
            insert: function(values) {
                var that = this;
                that.modifying.fire();
                that.inserting.fire(values);
                return that._addFailHandlers(that._insertImpl(values).done(function(callbackValues, callbackKey) {
                        that.inserted.fire(callbackValues, callbackKey);
                        that.modified.fire()
                    }))
            },
            _insertImpl: abstract,
            update: function(key, values) {
                var that = this;
                that.modifying.fire();
                that.updating.fire(key, values);
                return that._addFailHandlers(that._updateImpl(key, values).done(function(callbackKey, callbackValues) {
                        that.updated.fire(callbackKey, callbackValues);
                        that.modified.fire()
                    }))
            },
            _updateImpl: abstract,
            remove: function(key) {
                var that = this;
                that.modifying.fire();
                that.removing.fire(key);
                return that._addFailHandlers(that._removeImpl(key).done(function(callbackKey) {
                        that.removed.fire(callbackKey);
                        that.modified.fire()
                    }))
            },
            _removeImpl: abstract,
            _addFailHandlers: function(deferred) {
                return deferred.fail(this._errorHandler, data._handleError)
            }
        })
    })(jQuery, DevExpress);
    /*! Module core, file data.store.array.js */
    (function($, DX, undefined) {
        var data = DX.data,
            Guid = data.Guid;
        var trivialPromise = function(_) {
                var d = $.Deferred();
                return d.resolve.apply(d, arguments).promise()
            };
        var rejectedPromise = function(_) {
                var d = $.Deferred();
                return d.reject.apply(d, arguments).promise()
            };
        data.ArrayStore = data.Store.inherit({
            ctor: function(options) {
                if ($.isArray(options))
                    options = {data: options};
                else
                    options = options || {};
                this.callBase(options);
                var initialArray = options.data;
                if (initialArray && !$.isArray(initialArray))
                    throw Error("Invalid 'data' option value");
                this._array = initialArray || []
            },
            createQuery: function() {
                return data.query(this._array, {errorHandler: this._errorHandler})
            },
            _byKeyImpl: function(key) {
                return trivialPromise(this._array[this._indexByKey(key)])
            },
            _insertImpl: function(values) {
                var keyExpr = this.key(),
                    keyValue,
                    obj = {};
                $.extend(obj, values);
                if (keyExpr) {
                    keyValue = this.keyOf(obj);
                    if (keyValue === undefined || typeof keyValue === "object" && $.isEmptyObject(keyValue)) {
                        if ($.isArray(keyExpr))
                            throw Error("Compound keys cannot be auto-generated");
                        keyValue = obj[keyExpr] = String(new Guid)
                    }
                    else if (this._array[this._indexByKey(keyValue)] !== undefined)
                        return rejectedPromise(Error("Attempt to insert an item with the duplicate key"))
                }
                else
                    keyValue = obj;
                this._array.push(obj);
                return trivialPromise(values, keyValue)
            },
            _updateImpl: function(key, values) {
                var target;
                if (this.key()) {
                    var index = this._indexByKey(key);
                    if (index < 0)
                        return rejectedPromise(Error("Data item not found"));
                    target = this._array[index]
                }
                else
                    target = key;
                DX.utils.deepExtendArraySafe(target, values);
                return trivialPromise(key, values)
            },
            _removeImpl: function(key) {
                var index = this._indexByKey(key);
                if (index > -1)
                    this._array.splice(index, 1);
                return trivialPromise(key)
            },
            _indexByKey: function(key) {
                for (var i = 0, arrayLength = this._array.length; i < arrayLength; i++)
                    if (data.utils.keysEqual(this.key(), this.keyOf(this._array[i]), key))
                        return i;
                return -1
            },
            clear: function() {
                this._array = []
            }
        })
    })(jQuery, DevExpress);
    /*! Module core, file data.store.local.js */
    (function($, DX, undefined) {
        var Class = DX.Class,
            abstract = DX.abstract,
            data = DX.data;
        var LocalStoreBackend = Class.inherit({
                ctor: function(store, storeOptions) {
                    this._store = store;
                    this._dirty = false;
                    var immediate = this._immediate = storeOptions.immediate;
                    var flushInterval = Math.max(100, storeOptions.flushInterval || 10 * 1000);
                    if (!immediate) {
                        var saveProxy = $.proxy(this.save, this);
                        setInterval(saveProxy, flushInterval);
                        $(window).on("beforeunload", saveProxy);
                        if (window.cordova)
                            document.addEventListener("pause", saveProxy, false)
                    }
                },
                notifyChanged: function() {
                    this._dirty = true;
                    if (this._immediate)
                        this.save()
                },
                load: function() {
                    this._store._array = this._loadImpl();
                    this._dirty = false
                },
                save: function() {
                    if (!this._dirty)
                        return;
                    this._saveImpl(this._store._array);
                    this._dirty = false
                },
                _loadImpl: abstract,
                _saveImpl: abstract
            });
        var DomLocalStoreBackend = LocalStoreBackend.inherit({
                ctor: function(store, storeOptions) {
                    this.callBase(store, storeOptions);
                    var name = storeOptions.name;
                    if (!name)
                        throw Error("Name is required");
                    this._key = "dx-data-localStore-" + name
                },
                _loadImpl: function() {
                    var raw = localStorage.getItem(this._key);
                    if (raw)
                        return JSON.parse(raw);
                    return []
                },
                _saveImpl: function(array) {
                    if (!array.length)
                        localStorage.removeItem(this._key);
                    else
                        localStorage.setItem(this._key, JSON.stringify(array))
                }
            });
        var localStoreBackends = {dom: DomLocalStoreBackend};
        data.LocalStore = data.ArrayStore.inherit({
            ctor: function(options) {
                if (typeof options === "string")
                    options = {name: options};
                else
                    options = options || {};
                this.callBase(options);
                this._backend = new localStoreBackends[options.backend || "dom"](this, options);
                this._backend.load()
            },
            clear: function() {
                this.callBase();
                this._backend.notifyChanged()
            },
            _insertImpl: function(values) {
                var b = this._backend;
                return this.callBase(values).done($.proxy(b.notifyChanged, b))
            },
            _updateImpl: function(key, values) {
                var b = this._backend;
                return this.callBase(key, values).done($.proxy(b.notifyChanged, b))
            },
            _removeImpl: function(key) {
                var b = this._backend;
                return this.callBase(key).done($.proxy(b.notifyChanged, b))
            }
        })
    })(jQuery, DevExpress);
    /*! Module core, file data.store.odata.js */
    (function($, DX, undefined) {
        var Class = DX.Class,
            data = DX.data,
            odataUtils = data.utils.odata;
        var escapeServiceOperationParams = function(params, version) {
                if (!params)
                    return params;
                var result = {};
                $.each(params, function(k, v) {
                    result[k] = odataUtils.serializeValue(v, version)
                });
                return result
            };
        var convertSimpleKey = function(keyType, keyValue) {
                var converter = odataUtils.keyConverters[keyType];
                if (!converter)
                    throw Error("Unknown key type: " + keyType);
                return converter(keyValue)
            };
        var SharedMethods = {
                _extractServiceOptions: function(options) {
                    options = options || {};
                    this._url = String(options.url).replace(/\/+$/, "");
                    this._beforeSend = options.beforeSend;
                    this._jsonp = options.jsonp;
                    this._version = options.version;
                    this._withCredentials = options.withCredentials
                },
                _sendRequest: function(url, method, params, payload) {
                    return odataUtils.sendRequest({
                            url: url,
                            method: method,
                            params: params || {},
                            payload: payload
                        }, {
                            beforeSend: this._beforeSend,
                            jsonp: this._jsonp,
                            withCredentials: this._withCredentials
                        })
                },
                version: function() {
                    return this._version
                }
            };
        var ODataStore = data.Store.inherit({
                ctor: function(options) {
                    this.callBase(options);
                    this._extractServiceOptions(options);
                    this._keyType = options.keyType
                },
                _customLoadOptions: function() {
                    return ["expand", "customQueryParams"]
                },
                _byKeyImpl: function(key, extraOptions) {
                    var params = {};
                    if (extraOptions)
                        if (extraOptions.expand)
                            params["$expand"] = $.map($.makeArray(extraOptions.expand), odataUtils.serializePropName).join();
                    return this._sendRequest(this._byKeyUrl(key), "GET", params)
                },
                createQuery: function(loadOptions) {
                    loadOptions = loadOptions || {};
                    return data.query(this._url, {
                            beforeSend: this._beforeSend,
                            errorHandler: this._errorHandler,
                            jsonp: this._jsonp,
                            version: this._version,
                            withCredentials: this._withCredentials,
                            params: escapeServiceOperationParams(loadOptions.customQueryParams, this._version),
                            expand: loadOptions.expand,
                            requireTotalCount: loadOptions.requireTotalCount
                        })
                },
                _insertImpl: function(values) {
                    this._requireKey();
                    var that = this,
                        d = $.Deferred();
                    $.when(this._sendRequest(this._url, "POST", null, values)).done(function(serverResponse) {
                        d.resolve(values, that.keyOf(serverResponse))
                    }).fail($.proxy(d.reject, d));
                    return d.promise()
                },
                _updateImpl: function(key, values) {
                    var d = $.Deferred();
                    $.when(this._sendRequest(this._byKeyUrl(key), "MERGE", null, values)).done(function() {
                        d.resolve(key, values)
                    }).fail($.proxy(d.reject, d));
                    return d.promise()
                },
                _removeImpl: function(key) {
                    var d = $.Deferred();
                    $.when(this._sendRequest(this._byKeyUrl(key), "DELETE")).done(function() {
                        d.resolve(key)
                    }).fail($.proxy(d.reject, d));
                    return d.promise()
                },
                _byKeyUrl: function(key) {
                    var keyType = this._keyType;
                    if ($.isPlainObject(keyType))
                        $.each(keyType, function(subKeyName, subKeyType) {
                            key[subKeyName] = convertSimpleKey(subKeyType, key[subKeyName])
                        });
                    else if (keyType)
                        key = convertSimpleKey(keyType, key);
                    return this._url + "(" + encodeURIComponent(odataUtils.serializeKey(key, this._version)) + ")"
                }
            }).include(SharedMethods);
        var ODataContext = Class.inherit({
                ctor: function(options) {
                    var that = this;
                    that._extractServiceOptions(options);
                    that._errorHandler = options.errorHandler;
                    $.each(options.entities || [], function(entityAlias, entityOptions) {
                        that[entityAlias] = new ODataStore($.extend({}, options, {url: that._url + "/" + encodeURIComponent(entityOptions.name || entityAlias)}, entityOptions))
                    })
                },
                get: function(operationName, params) {
                    return this.invoke(operationName, params, "GET")
                },
                invoke: function(operationName, params, httpMethod) {
                    httpMethod = httpMethod || "POST";
                    var d = $.Deferred();
                    $.when(this._sendRequest(this._url + "/" + encodeURIComponent(operationName), httpMethod, escapeServiceOperationParams(params, this._version))).done(function(r) {
                        if (r && operationName in r)
                            r = r[operationName];
                        d.resolve(r)
                    }).fail([this._errorHandler, data._handleError, $.proxy(d.reject, d)]);
                    return d.promise()
                },
                objectLink: function(entityAlias, key) {
                    var store = this[entityAlias];
                    if (!store)
                        throw Error("Unknown entity name or alias: " + entityAlias);
                    return {__metadata: {uri: store._byKeyUrl(key)}}
                }
            }).include(SharedMethods);
        $.extend(data, {
            ODataStore: ODataStore,
            ODataContext: ODataContext
        })
    })(jQuery, DevExpress);
    /*! Module core, file data.store.rest.js */
    (function($, DX, undefined) {
        var data = DX.data;
        function createAjaxFailureHandler(deferred) {
            return function(xhr, textStatus) {
                    if (!xhr || !xhr.getResponseHeader)
                        deferred.reject.apply(deferred, arguments);
                    else
                        deferred.reject(Error(data.utils.errorMessageFromXhr(xhr, textStatus)))
                }
        }
        function operationCustomizerPropName(operationName) {
            return "_customize" + DX.inflector.camelize(operationName, true)
        }
        function pathPropName(operationName) {
            return "_" + operationName + "Path"
        }
        data.RestStore = data.Store.inherit({
            ctor: function(options) {
                DX.utils.logger.warn("RestStore is deprecated, use CustomStore instead");
                var that = this;
                that.callBase(options);
                options = options || {};
                that._url = String(options.url).replace(/\/+$/, "");
                that._jsonp = options.jsonp;
                that._withCredentials = options.withCredentials;
                $.each(["Load", "Insert", "Update", "Remove", "ByKey", "Operation"], function() {
                    var value = options["customize" + this];
                    if (value)
                        that[operationCustomizerPropName(this)] = value
                });
                $.each(["load", "insert", "update", "remove", "byKey"], function() {
                    var value = options[this + "Path"];
                    if (value)
                        that[pathPropName(this)] = value
                })
            },
            _loadImpl: function(options) {
                var d = $.Deferred(),
                    ajaxOptions = {
                        url: this._formatUrlNoKey("load"),
                        type: "GET"
                    };
                $.when(this._createAjax(ajaxOptions, "load", options)).done($.proxy(d.resolve, d)).fail(createAjaxFailureHandler(d));
                return this._addFailHandlers(d.promise())
            },
            createQuery: function() {
                throw Error("Not supported");
            },
            _insertImpl: function(values) {
                var d = $.Deferred(),
                    that = this,
                    ajaxOptions = {
                        url: this._formatUrlNoKey("insert"),
                        type: "POST",
                        contentType: "application/json",
                        data: JSON.stringify(values)
                    };
                $.when(this._createAjax(ajaxOptions, "insert")).done(function(serverResponse) {
                    d.resolve(values, that.key() && that.keyOf(serverResponse))
                }).fail(createAjaxFailureHandler(d));
                return d.promise()
            },
            _updateImpl: function(key, values) {
                var d = $.Deferred(),
                    ajaxOptions = {
                        url: this._formatUrlWithKey("update", key),
                        type: "PUT",
                        contentType: "application/json",
                        data: JSON.stringify(values)
                    };
                $.when(this._createAjax(ajaxOptions, "update")).done(function() {
                    d.resolve(key, values)
                }).fail(createAjaxFailureHandler(d));
                return d.promise()
            },
            _removeImpl: function(key) {
                var d = $.Deferred(),
                    ajaxOptions = {
                        url: this._formatUrlWithKey("remove", key),
                        type: "DELETE"
                    };
                $.when(this._createAjax(ajaxOptions, "remove")).done(function() {
                    d.resolve(key)
                }).fail(createAjaxFailureHandler(d));
                return d.promise()
            },
            _byKeyImpl: function(key) {
                var d = $.Deferred(),
                    ajaxOptions = {
                        url: this._formatUrlWithKey("byKey", key),
                        type: "GET"
                    };
                $.when(this._createAjax(ajaxOptions, "byKey")).done(function(data) {
                    d.resolve(data)
                }).fail(createAjaxFailureHandler(d));
                return d.promise()
            },
            _createAjax: function(ajaxOptions, operationName, extra) {
                var customizationFunc,
                    customizationResult;
                function isDeferred(obj) {
                    return "done" in obj && "fail" in obj
                }
                if (this._jsonp && ajaxOptions.type === "GET")
                    ajaxOptions.dataType = "jsonp";
                else
                    $.extend(true, ajaxOptions, {xhrFields: {withCredentials: this._withCredentials}});
                customizationFunc = this[operationCustomizerPropName("operation")];
                if (customizationFunc) {
                    customizationResult = customizationFunc(ajaxOptions, operationName, extra);
                    if (customizationResult) {
                        if (isDeferred(customizationResult))
                            return customizationResult;
                        ajaxOptions = customizationResult
                    }
                }
                customizationFunc = this[operationCustomizerPropName(operationName)];
                if (customizationFunc) {
                    customizationResult = customizationFunc(ajaxOptions, extra);
                    if (customizationResult) {
                        if (isDeferred(customizationResult))
                            return customizationResult;
                        ajaxOptions = customizationResult
                    }
                }
                return $.ajax(ajaxOptions)
            },
            _formatUrlNoKey: function(operationName) {
                var url = this._url,
                    path = this[pathPropName(operationName)];
                if (!path)
                    return url;
                if ($.isFunction(path))
                    return path(url);
                return url + "/" + path
            },
            _formatUrlWithKey: function(operationName, key) {
                var url = this._url,
                    path = this[pathPropName(operationName)];
                if (!path)
                    return url + "/" + encodeURIComponent(key);
                if ($.isFunction(path))
                    return path(url, key);
                return url + "/" + path + "/" + encodeURIComponent(key)
            }
        })
    })(jQuery, DevExpress);
    /*! Module core, file data.store.custom.js */
    (function($, DX, undefined) {
        var data = DX.data;
        var ERROR_QUERY_NOT_SUPPORTED = "CustomStore does not support creating queries",
            ERROR_MISSING_USER_FUNC = "Required option is not specified or is not a function: ",
            ERROR_INVALID_RETURN = "Invalid return value: ";
        var TOTAL_COUNT = "totalCount",
            LOAD = "load",
            BY_KEY = "byKey",
            INSERT = "insert",
            UPDATE = "update",
            REMOVE = "remove";
        function isPromise(obj) {
            return obj && $.isFunction(obj.done) && $.isFunction(obj.fail) && $.isFunction(obj.promise)
        }
        function trivialPromise(value) {
            return $.Deferred().resolve(value).promise()
        }
        function ensureRequiredFuncOption(name, obj) {
            if (!$.isFunction(obj))
                throw Error(ERROR_MISSING_USER_FUNC + name);
        }
        function throwInvalidUserFuncResult(name) {
            throw Error(ERROR_INVALID_RETURN + name);
        }
        function createUserFuncFailureHandler(pendingDeferred) {
            function errorMessageFromXhr(promiseArguments) {
                var xhr = promiseArguments[0],
                    textStatus = promiseArguments[1];
                if (!xhr || !xhr.getResponseHeader)
                    return null;
                return data.utils.errorMessageFromXhr(xhr, textStatus)
            }
            return function(arg) {
                    var error;
                    if (arg instanceof Error)
                        error = arg;
                    else
                        error = Error(errorMessageFromXhr(arguments) || arg && String(arg) || "Unknown error");
                    pendingDeferred.reject(error)
                }
        }
        data.CustomStore = data.Store.inherit({
            ctor: function(options) {
                options = options || {};
                this.callBase(options);
                this._useDefaultSearch = false;
                this._loadFunc = options[LOAD];
                this._totalCountFunc = options[TOTAL_COUNT];
                this._byKeyFunc = options[BY_KEY] || options.lookup;
                this._insertFunc = options[INSERT];
                this._updateFunc = options[UPDATE];
                this._removeFunc = options[REMOVE]
            },
            createQuery: function() {
                throw Error(ERROR_QUERY_NOT_SUPPORTED);
            },
            _totalCountImpl: function(options) {
                var userFunc = this._totalCountFunc,
                    userResult,
                    d = $.Deferred();
                ensureRequiredFuncOption(TOTAL_COUNT, userFunc);
                userResult = userFunc(options);
                if (!isPromise(userResult)) {
                    userResult = Number(userResult);
                    if (!isFinite(userResult))
                        throwInvalidUserFuncResult(TOTAL_COUNT);
                    userResult = trivialPromise(userResult)
                }
                userResult.done(function(count) {
                    d.resolve(Number(count))
                }).fail(createUserFuncFailureHandler(d));
                return d.promise()
            },
            _loadImpl: function(options) {
                var userFunc = this._loadFunc,
                    userResult,
                    d = $.Deferred();
                ensureRequiredFuncOption(LOAD, userFunc);
                userResult = userFunc(options);
                if ($.isArray(userResult))
                    userResult = trivialPromise(userResult);
                else if (userResult === null || userResult === undefined)
                    userResult = trivialPromise([]);
                else if (!isPromise(userResult))
                    throwInvalidUserFuncResult(LOAD);
                userResult.done(function(data, extra) {
                    d.resolve(data, extra)
                }).fail(createUserFuncFailureHandler(d));
                return this._addFailHandlers(d.promise())
            },
            _byKeyImpl: function(key) {
                var userFunc = this._byKeyFunc,
                    userResult,
                    d = $.Deferred();
                ensureRequiredFuncOption(BY_KEY, userFunc);
                userResult = userFunc(key);
                if (!isPromise(userResult))
                    userResult = trivialPromise(userResult);
                userResult.done(function(obj) {
                    d.resolve(obj)
                }).fail(createUserFuncFailureHandler(d));
                return d.promise()
            },
            _insertImpl: function(values) {
                var userFunc = this._insertFunc,
                    userResult,
                    d = $.Deferred();
                ensureRequiredFuncOption(INSERT, userFunc);
                userResult = userFunc(values);
                if (!isPromise(userResult))
                    userResult = trivialPromise(userResult);
                userResult.done(function(newKey) {
                    d.resolve(values, newKey)
                }).fail(createUserFuncFailureHandler(d));
                return d.promise()
            },
            _updateImpl: function(key, values) {
                var userFunc = this._updateFunc,
                    userResult,
                    d = $.Deferred();
                ensureRequiredFuncOption(UPDATE, userFunc);
                userResult = userFunc(key, values);
                if (!isPromise(userResult))
                    userResult = trivialPromise();
                userResult.done(function() {
                    d.resolve(key, values)
                }).fail(createUserFuncFailureHandler(d));
                return d.promise()
            },
            _removeImpl: function(key) {
                var userFunc = this._removeFunc,
                    userResult,
                    d = $.Deferred();
                ensureRequiredFuncOption(REMOVE, userFunc);
                userResult = userFunc(key);
                if (!isPromise(userResult))
                    userResult = trivialPromise();
                userResult.done(function() {
                    d.resolve(key)
                }).fail(createUserFuncFailureHandler(d));
                return d.promise()
            }
        });
        data.CustomStore_internals = {ERRORS: {
                QUERY_NOT_SUPPORTED: ERROR_QUERY_NOT_SUPPORTED,
                MISSING_USER_FUNC: ERROR_MISSING_USER_FUNC,
                INVALID_RETURN: ERROR_INVALID_RETURN
            }}
    })(jQuery, DevExpress);
    /*! Module core, file data.dataSource.js */
    (function($, DX, undefined) {
        var data = DX.data,
            CustomStore = data.CustomStore,
            Class = DX.Class;
        var storeTypeRegistry = {
                jaydata: "JayDataStore",
                breeze: "BreezeStore",
                odata: "ODataStore",
                local: "LocalStore",
                array: "ArrayStore"
            };
        function normalizeDataSourceOptions(options) {
            var store;
            function createCustomStoreFromLoadFunc() {
                var storeConfig = {};
                $.each(["key", "load", "byKey", "lookup", "totalCount", "insert", "update", "remove"], function() {
                    storeConfig[this] = options[this];
                    delete options[this]
                });
                return new CustomStore(storeConfig)
            }
            function createStoreFromConfig(storeConfig) {
                var storeCtor = data[storeTypeRegistry[storeConfig.type]];
                delete storeConfig.type;
                return new storeCtor(storeConfig)
            }
            function createCustomStoreFromUrl(url) {
                return new CustomStore({load: function() {
                            return $.getJSON(url)
                        }})
            }
            if (typeof options === "string")
                options = createCustomStoreFromUrl(options);
            if (options === undefined)
                options = [];
            if ($.isArray(options) || options instanceof data.Store)
                options = {store: options};
            else
                options = $.extend({}, options);
            if (options.store === undefined)
                options.store = [];
            store = options.store;
            if ("load" in options)
                store = createCustomStoreFromLoadFunc();
            else if ($.isArray(store))
                store = new data.ArrayStore(store);
            else if ($.isPlainObject(store))
                store = createStoreFromConfig($.extend({}, store));
            options.store = store;
            return options
        }
        function normalizeStoreLoadOptionAccessorArguments(originalArguments) {
            switch (originalArguments.length) {
                case 0:
                    return undefined;
                case 1:
                    return originalArguments[0]
            }
            return $.makeArray(originalArguments)
        }
        function generateStoreLoadOptionAccessor(optionName) {
            return function() {
                    var args = normalizeStoreLoadOptionAccessorArguments(arguments);
                    if (args !== undefined)
                        this._storeLoadOptions[optionName] = args;
                    return this._storeLoadOptions[optionName]
                }
        }
        function addOldUserDataSourceBackwardCompatibilityOptions(dataSource, storeLoadOptions) {
            storeLoadOptions.refresh = !dataSource._paginate || dataSource._pageIndex === 0;
            if (storeLoadOptions.searchValue !== null)
                storeLoadOptions.searchString = storeLoadOptions.searchValue
        }
        var DataSource = Class.inherit({
                ctor: function(options) {
                    options = normalizeDataSourceOptions(options);
                    this._store = options.store;
                    this._storeLoadOptions = this._extractLoadOptions(options);
                    this._mapFunc = options.map;
                    this._postProcessFunc = options.postProcess;
                    this._pageIndex = 0;
                    this._pageSize = options.pageSize !== undefined ? options.pageSize : 20;
                    this._items = [];
                    this._totalCount = -1;
                    this._isLoaded = false;
                    this._loadingCount = 0;
                    this._preferSync = options._preferSync;
                    this._loadQueue = this._createLoadQueue();
                    this._searchValue = "searchValue" in options ? options.searchValue : null;
                    this._searchOperation = options.searchOperation || "contains";
                    this._searchExpr = options.searchExpr;
                    this._paginate = options.paginate;
                    if (this._paginate === undefined)
                        this._paginate = !this.group();
                    this._isLastPage = !this._paginate;
                    this._userData = {};
                    this.changed = $.Callbacks();
                    this.loadError = $.Callbacks();
                    this.loadingChanged = $.Callbacks()
                },
                dispose: function() {
                    this.changed.empty();
                    this.loadError.empty();
                    this.loadingChanged.empty();
                    delete this._store;
                    this._disposed = true
                },
                _extractLoadOptions: function(options) {
                    var result = {},
                        names = ["sort", "filter", "select", "group", "requireTotalCount"],
                        customNames = this._store._customLoadOptions();
                    if (customNames)
                        names = names.concat(customNames);
                    $.each(names, function() {
                        result[this] = options[this]
                    });
                    return result
                },
                loadOptions: function() {
                    return this._storeLoadOptions
                },
                items: function() {
                    return this._items
                },
                pageIndex: function(newIndex) {
                    if (newIndex !== undefined) {
                        this._pageIndex = newIndex;
                        this._isLastPage = !this._paginate
                    }
                    return this._pageIndex
                },
                paginate: function(value) {
                    if (arguments.length < 1)
                        return this._paginate;
                    value = !!value;
                    if (this._paginate !== value) {
                        this._paginate = value;
                        this.pageIndex(0)
                    }
                },
                isLastPage: function() {
                    return this._isLastPage
                },
                sort: generateStoreLoadOptionAccessor("sort"),
                filter: function() {
                    var newFilter = normalizeStoreLoadOptionAccessorArguments(arguments);
                    if (newFilter !== undefined) {
                        this._storeLoadOptions.filter = newFilter;
                        this.pageIndex(0)
                    }
                    return this._storeLoadOptions.filter
                },
                group: generateStoreLoadOptionAccessor("group"),
                select: generateStoreLoadOptionAccessor("select"),
                searchValue: function(value) {
                    if (value !== undefined) {
                        this.pageIndex(0);
                        this._searchValue = value
                    }
                    return this._searchValue
                },
                searchOperation: function(op) {
                    if (op !== undefined) {
                        this.pageIndex(0);
                        this._searchOperation = op
                    }
                    return this._searchOperation
                },
                searchExpr: function(expr) {
                    var argc = arguments.length;
                    if (argc) {
                        if (argc > 1)
                            expr = $.makeArray(arguments);
                        this.pageIndex(0);
                        this._searchExpr = expr
                    }
                    return this._searchExpr
                },
                store: function() {
                    return this._store
                },
                key: function() {
                    return this._store && this._store.key()
                },
                totalCount: function() {
                    return this._totalCount
                },
                isLoaded: function() {
                    return this._isLoaded
                },
                isLoading: function() {
                    return this._loadingCount > 0
                },
                _createLoadQueue: function() {
                    return DX.createQueue()
                },
                _changeLoadingCount: function(increment) {
                    var oldLoading = this.isLoading(),
                        newLoading;
                    this._loadingCount += increment;
                    newLoading = this.isLoading();
                    if (oldLoading ^ newLoading)
                        this.loadingChanged.fire(newLoading)
                },
                _scheduleLoadCallbacks: function(deferred) {
                    var thisSource = this;
                    thisSource._changeLoadingCount(1);
                    deferred.always(function() {
                        thisSource._changeLoadingCount(-1)
                    })
                },
                _scheduleChangedCallbacks: function(deferred) {
                    var that = this;
                    deferred.done(function() {
                        that.changed.fire()
                    })
                },
                loadSingle: function(propName, propValue) {
                    var that = this;
                    var d = $.Deferred().fail(this.loadError.fire),
                        key = this.key(),
                        store = this._store,
                        loadOptions = this._createStoreLoadOptions();
                    function handleSuccess(data) {
                        d.resolve(that._transformLoadedData(data)[0])
                    }
                    if (arguments.length < 2) {
                        propValue = propName;
                        propName = key
                    }
                    delete loadOptions.skip;
                    delete loadOptions.group;
                    delete loadOptions.refresh;
                    delete loadOptions.pageIndex;
                    delete loadOptions.searchString;
                    if (propName === key || store instanceof data.CustomStore)
                        store.byKey(propValue, loadOptions).done(handleSuccess).fail(d.reject);
                    else {
                        loadOptions.take = 1;
                        loadOptions._preferSync = true;
                        loadOptions.filter = loadOptions.filter ? [loadOptions.filter, [propName, propValue]] : [propName, propValue];
                        store.load(loadOptions).done(handleSuccess).fail(d.reject)
                    }
                    return d.promise()
                },
                load: function() {
                    var thisSource = this,
                        d = $.Deferred(),
                        errorCallback = this.loadError,
                        storeLoadOptions;
                    this._scheduleLoadCallbacks(d);
                    this._scheduleChangedCallbacks(d);
                    storeLoadOptions = this._createStoreLoadOptions();
                    function loadTask() {
                        if (thisSource._disposed)
                            return undefined;
                        return thisSource._loadFromStore(storeLoadOptions, d)
                    }
                    this._loadQueue.add(function() {
                        loadTask();
                        return d.promise()
                    }, function() {
                        thisSource._changeLoadingCount(-1)
                    });
                    return d.promise().fail($.proxy(errorCallback.fire, errorCallback))
                },
                _addSearchOptions: function(storeLoadOptions) {
                    if (this._disposed)
                        return;
                    if (this.store()._useDefaultSearch)
                        this._addSearchFilter(storeLoadOptions);
                    else {
                        storeLoadOptions.searchValue = this._searchValue;
                        storeLoadOptions.searchExpr = this._searchExpr
                    }
                },
                _createStoreLoadOptions: function() {
                    var result = $.extend({}, this._storeLoadOptions);
                    this._addSearchOptions(result);
                    if (this._paginate) {
                        result.pageIndex = this._pageIndex;
                        if (this._pageSize) {
                            result.skip = this._pageIndex * this._pageSize;
                            result.take = this._pageSize
                        }
                    }
                    result.userData = this._userData;
                    addOldUserDataSourceBackwardCompatibilityOptions(this, result);
                    return result
                },
                _addSearchFilter: function(storeLoadOptions) {
                    var value = this._searchValue,
                        op = this._searchOperation,
                        selector = this._searchExpr,
                        searchFilter = [];
                    if (!value)
                        return;
                    if (!selector)
                        selector = "this";
                    if (!$.isArray(selector))
                        selector = [selector];
                    $.each(selector, function(i, item) {
                        if (searchFilter.length)
                            searchFilter.push("or");
                        searchFilter.push([item, op, value])
                    });
                    if (storeLoadOptions.filter)
                        storeLoadOptions.filter = [searchFilter, storeLoadOptions.filter];
                    else
                        storeLoadOptions.filter = searchFilter
                },
                _loadFromStore: function(storeLoadOptions, pendingDeferred) {
                    var thisSource = this;
                    function handleSuccess(data, extra) {
                        function processResult() {
                            thisSource._processStoreLoadResult(data, extra, storeLoadOptions, pendingDeferred)
                        }
                        if (thisSource._preferSync)
                            processResult();
                        else
                            DX.utils.executeAsync(processResult)
                    }
                    return this.store().load(storeLoadOptions).done(handleSuccess).fail($.proxy(pendingDeferred.reject, pendingDeferred))
                },
                _processStoreLoadResult: function(data, extra, storeLoadOptions, pendingDeferred) {
                    var thisSource = this;
                    function resolvePendingDeferred() {
                        thisSource._isLoaded = true;
                        thisSource._totalCount = isFinite(extra.totalCount) ? extra.totalCount : -1;
                        return pendingDeferred.resolve(data, extra)
                    }
                    function proceedLoadingTotalCount() {
                        thisSource.store().totalCount(storeLoadOptions).done(function(count) {
                            extra.totalCount = count;
                            resolvePendingDeferred()
                        }).fail(function(){})
                    }
                    if (thisSource._disposed)
                        return;
                    data = thisSource._transformLoadedData(data);
                    if (!$.isPlainObject(extra))
                        extra = {};
                    thisSource._items = data;
                    if (!data.length || !thisSource._paginate || thisSource._pageSize && data.length < thisSource._pageSize)
                        thisSource._isLastPage = true;
                    if (storeLoadOptions.requireTotalCount && !isFinite(extra.totalCount))
                        proceedLoadingTotalCount();
                    else
                        resolvePendingDeferred()
                },
                _transformLoadedData: function(data) {
                    var result = $.makeArray(data);
                    if (this._mapFunc)
                        result = $.map(result, this._mapFunc);
                    if (this._postProcessFunc)
                        result = this._postProcessFunc(result);
                    return result
                }
            });
        data.Store.redefine({toDataSource: function(options) {
                DX.utils.logger.warn("toDataSource() method is deprecated, use 'new DevExpress.data.DataSource(...)' instead");
                return new DataSource($.extend({store: this}, options))
            }});
        $.extend(true, data, {
            DataSource: DataSource,
            createDataSource: function(options) {
                DX.utils.logger.warn("createDataSource() method is deprecated, use 'new DevExpress.data.DataSource(...)' instead");
                return new DataSource(options)
            },
            utils: {
                storeTypeRegistry: storeTypeRegistry,
                normalizeDataSourceOptions: normalizeDataSourceOptions
            }
        })
    })(jQuery, DevExpress);
    /*! Module core, file ko.js */
    (function($, DX, undefined) {
        if (!DX.support.hasKo)
            return;
        var ko = window.ko;
        (function checkKnockoutVersion(version) {
            version = version.split(".");
            if (version[0] < 2 || version[0] == 2 && version[1] < 3)
                throw Error("Your version of KnockoutJS is too old. Please upgrade KnockoutJS to 2.3.0 or later.");
        })(ko.version)
    })(jQuery, DevExpress);
    /*! Module core, file ng.js */
    (function($, DX, undefined) {
        if (!DX.support.hasNg)
            return;
        DX.ng = {module: window.angular.module("dx", ["ngSanitize"])}
    })(jQuery, DevExpress);
    /*! Module core, file component.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            utils = DX.utils,
            dataUtils = DX.data.utils;
        var Component = DX.Class.inherit({
                NAME: "Component",
                _deprecatedOptions: {},
                _optionAliases: {},
                _setDefaultOptions: function(){},
                _defaultOptionsRules: function() {
                    return []
                },
                _setOptionsByDevice: function() {
                    var rules = this._defaultOptionsRules(),
                        currentDevice = DX.devices.current(),
                        result = {};
                    if (this._customRules)
                        rules = rules.concat(this._customRules);
                    var deviceMatch = function(device, filter) {
                            filter = $.makeArray(filter);
                            return filter.length === 1 && $.isEmptyObject(filter[0]) || utils.findBestMatches(device, filter).length > 0
                        };
                    $.each(rules, function(index, rule) {
                        var deviceFilter = rule.device || {},
                            match;
                        if ($.isFunction(deviceFilter))
                            match = deviceFilter(currentDevice);
                        else
                            match = deviceMatch(currentDevice, deviceFilter);
                        if (match)
                            $.extend(result, rule.options)
                    });
                    this.option(result)
                },
                _optionsByReference: function() {
                    return {}
                },
                ctor: function(options) {
                    if (!this.NAME)
                        throw Error("NAME is not specified");
                    this._options = {};
                    this._updateLockCount = 0;
                    this.optionChanged = $.Callbacks();
                    this.beginUpdate();
                    try {
                        this._suppressDeprecatedWarnings();
                        this._setDefaultOptions();
                        this._setOptionsByDevice();
                        this._resumeDeprecatedWarnings();
                        this._initialOptions = $.extend({}, this.option());
                        this._initOptions(options || {})
                    }
                    finally {
                        this.endUpdate()
                    }
                },
                _initOptions: function(options) {
                    this.option(options)
                },
                _optionValuesEqual: function(name, oldValue, newValue) {
                    oldValue = dataUtils.toComparable(oldValue, true);
                    newValue = dataUtils.toComparable(newValue, true);
                    if (oldValue && newValue && oldValue.jquery && newValue.jquery)
                        return newValue.is(oldValue);
                    if (oldValue === null || typeof oldValue !== "object")
                        return oldValue === newValue;
                    return false
                },
                _init: $.noop,
                _optionChanged: $.noop,
                instance: function() {
                    return this
                },
                beginUpdate: function() {
                    this._updateLockCount++
                },
                endUpdate: function() {
                    this._updateLockCount--;
                    if (!this._updateLockCount)
                        if (!this._initializing && !this._initialized) {
                            this._initializing = true;
                            try {
                                this._init()
                            }
                            finally {
                                this._initializing = false;
                                this._initialized = true
                            }
                        }
                },
                _logWarningIfDeprecated: function(option) {
                    var info = this._deprecatedOptions[option];
                    if (info && !this._deprecatedOptionsSuppressed)
                        this._logDeprecatedWarning(option, info)
                },
                _logDeprecatedWarningCount: 0,
                _logDeprecatedWarning: function(option, info) {
                    utils.logger.warn("'" + option + "' option is deprecated since " + info.since + ". " + info.message);
                    ++this._logDeprecatedWarningCount
                },
                _suppressDeprecatedWarnings: function() {
                    this._deprecatedOptionsSuppressed = true
                },
                _resumeDeprecatedWarnings: function() {
                    this._deprecatedOptionsSuppressed = false
                },
                _getOptionAliases: function(option) {
                    return $.map(this._optionAliases, function(aliasedOption, alias) {
                            return option === aliasedOption ? alias : undefined
                        })
                },
                _notifyOptionChanged: function(option, value, previousValue) {
                    var that = this,
                        optionWithAliases;
                    if (this._initialized) {
                        optionWithAliases = this._getOptionAliases(option);
                        optionWithAliases.push(option);
                        $.each(optionWithAliases, function(index, name) {
                            var topLevelName = name.split(/[.\[]/)[0];
                            that.optionChanged.fireWith(that, [topLevelName, value, previousValue]);
                            that._optionChanged(topLevelName, value, previousValue)
                        })
                    }
                },
                initialOption: function(optionName) {
                    var options = this._initialOptions;
                    return options[optionName]
                },
                option: function(options) {
                    var that = this,
                        name = options,
                        value = arguments[1];
                    if (arguments.length < 2 && $.type(name) !== "object") {
                        if (name) {
                            this._logWarningIfDeprecated(name);
                            if (this._optionAliases[name])
                                name = this._optionAliases[name]
                        }
                        $.each(this._optionAliases, function(alias, option) {
                            that._options[alias] = dataUtils.compileGetter(option)(that._options, {functionsAsIs: true})
                        });
                        return dataUtils.compileGetter(name)(that._options, {functionsAsIs: true})
                    }
                    if (typeof name === "string") {
                        options = {};
                        options[name] = value
                    }
                    that.beginUpdate();
                    try {
                        $.each(options, function(name, value) {
                            that._logWarningIfDeprecated(name);
                            if (that._optionAliases[name])
                                name = that._optionAliases[name];
                            var prevValue = dataUtils.compileGetter(name)(that._options, {functionsAsIs: true});
                            if (that._optionValuesEqual(name, prevValue, value))
                                return;
                            dataUtils.compileSetter(name)(that._options, value, {
                                functionsAsIs: true,
                                merge: !that._optionsByReference()[name]
                            });
                            that._notifyOptionChanged(name, value, prevValue)
                        })
                    }
                    finally {
                        that.endUpdate()
                    }
                }
            });
        $.extend(DX, {Component: Component})
    })(jQuery, DevExpress);
    /*! Module core, file DOMComponent.js */
    (function($, DX, undefined) {
        var windowResizeCallbacks = DX.utils.windowResizeCallbacks;
        var RTL_DIRECTION_CLASS = "dx-rtl",
            COMPONENT_NAMES_DATA_KEY = "dxComponents",
            VISIBILITY_CHANGE_CLASS = "dx-visibility-change-handler",
            VISIBILITY_CHANGE_EVENTNAMESPACE = "dxVisibilityChange";
        var DOMComponent = DX.Component.inherit({
                NAME: "DOMComponent",
                NAMESPACE: DX,
                _setDefaultOptions: function() {
                    this.callBase();
                    this.option({rtlEnabled: DX.rtlEnabled})
                },
                ctor: function(element, options) {
                    this._$element = $(element);
                    this._element().data(this.NAME, this);
                    this._attachInstanceToElement(this._$element);
                    this.disposing = $.Callbacks();
                    this.callBase(options)
                },
                _attachInstanceToElement: $.noop,
                _visibilityChanged: DX.abstract,
                _dimensionChanged: DX.abstract,
                _init: function() {
                    this.callBase();
                    this._attachWindowResizeCallback()
                },
                _attachWindowResizeCallback: function() {
                    if (this._isDimensionChangeSupported()) {
                        var windowResizeCallBack = this._windowResizeCallBack = $.proxy(this._dimensionChanged, this);
                        windowResizeCallbacks.add(windowResizeCallBack)
                    }
                },
                _isDimensionChangeSupported: function() {
                    return this._dimensionChanged !== DX.abstract
                },
                _render: function() {
                    this._toggleRTLDirection(this.option("rtlEnabled"));
                    this._renderVisiblityChange()
                },
                _renderVisiblityChange: function() {
                    if (!this._isVisibilityChangeSupported())
                        return;
                    this._element().addClass(VISIBILITY_CHANGE_CLASS);
                    this._attachVisiblityChangeHandlers()
                },
                _attachVisiblityChangeHandlers: function() {
                    var that = this;
                    that._element().off("." + VISIBILITY_CHANGE_EVENTNAMESPACE).on("dxhiding." + VISIBILITY_CHANGE_EVENTNAMESPACE, function() {
                        that._visibilityChanged(false)
                    }).on("dxshown." + VISIBILITY_CHANGE_EVENTNAMESPACE, function() {
                        that._visibilityChanged(true)
                    })
                },
                _isVisibilityChangeSupported: function() {
                    return this._visibilityChanged !== DX.abstract
                },
                _clean: $.noop,
                _modelByElement: $.noop,
                _invalidate: function() {
                    if (!this._updateLockCount)
                        throw Error("Invalidate called outside update transaction");
                    this._requireRefresh = true
                },
                _refresh: function() {
                    this._clean();
                    this._render()
                },
                _dispose: function() {
                    this._clean();
                    this._detachWindowResizeCallback();
                    this.optionChanged.empty();
                    this.disposing.fireWith(this).empty()
                },
                _detachWindowResizeCallback: function() {
                    if (this._isDimensionChangeSupported())
                        windowResizeCallbacks.remove(this._windowResizeCallBack)
                },
                _toggleRTLDirection: function(rtl) {
                    this._element().toggleClass(RTL_DIRECTION_CLASS, rtl)
                },
                _createAction: function(actionSource, config) {
                    var that = this;
                    config = $.extend({}, config);
                    var element = config.element || that._element(),
                        model = that._modelByElement(element);
                    config.context = model || that;
                    config.component = that;
                    var action = new DX.Action(actionSource, config);
                    return function(e) {
                            if (!arguments.length)
                                e = {};
                            if (e instanceof $.Event)
                                throw Error("Action must be executed with jQuery.Event like action({ jQueryEvent: event })");
                            if (!$.isPlainObject(e))
                                e = {actionValue: e};
                            return action.execute.call(action, $.extend(e, {
                                    component: that,
                                    element: element,
                                    model: model
                                }))
                        }
                },
                _createActionByOption: function(optionName, config) {
                    if (typeof optionName !== "string")
                        throw Error("Option name type is unexpected");
                    this._suppressDeprecatedWarnings();
                    var action = this._createAction(this.option(optionName), config);
                    this._resumeDeprecatedWarnings();
                    return action
                },
                _optionChanged: function(name, value, prevValue) {
                    if (name === "rtlEnabled")
                        this._invalidate()
                },
                _element: function() {
                    return this._$element
                },
                endUpdate: function() {
                    var requireRender = !this._initializing && !this._initialized;
                    this.callBase.apply(this, arguments);
                    if (!this._updateLockCount)
                        if (requireRender)
                            this._render();
                        else if (this._requireRefresh) {
                            this._requireRefresh = false;
                            this._refresh()
                        }
                }
            });
        var registerComponent = function(name, componentClass) {
                componentClass.redefine({_attachInstanceToElement: function($element) {
                        $element.data(name, this);
                        if (!$element.data(COMPONENT_NAMES_DATA_KEY))
                            $element.data(COMPONENT_NAMES_DATA_KEY, []);
                        $element.data(COMPONENT_NAMES_DATA_KEY).push(name)
                    }});
                componentClass.prototype.NAMESPACE[name] = componentClass;
                componentClass.prototype.NAME = name;
                componentClass.defaultOptions = function(rule) {
                    componentClass.prototype._customRules = componentClass.prototype._customRules || [];
                    componentClass.prototype._customRules.push(rule)
                };
                $.fn[name] = function(options) {
                    var isMemberInvoke = typeof options === "string",
                        result;
                    if (isMemberInvoke) {
                        var memberName = options,
                            memberArgs = $.makeArray(arguments).slice(1);
                        this.each(function() {
                            var instance = $(this).data(name);
                            if (!instance)
                                throw Error(DX.utils.stringFormat("Component {0} has not been initialized on this element", name));
                            var member = instance[memberName],
                                memberValue = member.apply(instance, memberArgs);
                            if (result === undefined)
                                result = memberValue
                        })
                    }
                    else {
                        this.each(function() {
                            var instance = $(this).data(name);
                            if (instance)
                                instance.option(options);
                            else
                                new componentClass(this, options)
                        });
                        result = this
                    }
                    return result
                }
            };
        var getComponents = function(element) {
                element = $(element);
                var names = element.data(COMPONENT_NAMES_DATA_KEY);
                if (!names)
                    return [];
                return $.map(names, function(name) {
                        return element.data(name)
                    })
            };
        var disposeComponents = function() {
                $.each(getComponents(this), function() {
                    this._dispose()
                })
            };
        var originalCleanData = $.cleanData;
        $.cleanData = function(element) {
            $.each(element, disposeComponents);
            return originalCleanData.apply(this, arguments)
        };
        registerComponent("DOMComponent", DOMComponent);
        DX.registerComponent = registerComponent
    })(jQuery, DevExpress);
    /*! Module core, file social.js */
    DevExpress.social = {};
    /*! Module core, file facebook.js */
    (function($, DX, undefined) {
        function notifyDeprecated() {
            DX.utils.logger.warn("DevExpress.social API is deprecated. Use official Facebook library instead")
        }
        var social = DX.social;
        var location = window.location,
            navigator = window.navigator,
            encodeURIComponent = window.encodeURIComponent,
            decodeURIComponent = window.decodeURIComponent,
            iosStandaloneMode = navigator.standalone,
            cordovaMode = false;
        if (window.cordova)
            $(document).on("deviceready", function() {
                cordovaMode = true
            });
        var ACCESS_TOKEN_KEY = "dx-facebook-access-token",
            IOS_STANDALONE_STEP1_KEY = "dx-facebook-step1",
            IOS_STANDALONE_STEP2_KEY = "dx-facebook-step2";
        var accessToken = null,
            expires = null,
            connectionChanged = $.Callbacks();
        var pendingLoginRedirectUrl;
        var isConnected = function() {
                return !!accessToken
            };
        var getAccessTokenObject = function() {
                return {
                        accessToken: accessToken,
                        expiresIn: accessToken ? expires : 0
                    }
            };
        var FB = social.Facebook = {
                loginRedirectUrl: "FacebookLoginCallback.html",
                connectionChanged: connectionChanged,
                isConnected: isConnected,
                getAccessTokenObject: getAccessTokenObject,
                jsonp: false
            };
        var login = function(appId, options) {
                notifyDeprecated();
                options = options || {};
                if (cordovaMode)
                    pendingLoginRedirectUrl = "https://www.facebook.com/connect/login_success.html";
                else
                    pendingLoginRedirectUrl = formatLoginRedirectUrl();
                var scope = (options.permissions || []).join(),
                    url = "https://www.facebook.com/dialog/oauth?display=popup&client_id=" + appId + "&redirect_uri=" + encodeURIComponent(pendingLoginRedirectUrl) + "&scope=" + encodeURIComponent(scope) + "&response_type=token";
                if (iosStandaloneMode)
                    putData(IOS_STANDALONE_STEP1_KEY, location.href);
                if (cordovaMode)
                    startLogin_cordova(url);
                else
                    startLogin_browser(url)
            };
        var formatLoginRedirectUrl = function() {
                var pathSegments = location.pathname.split(/\//g);
                pathSegments.pop();
                pathSegments.push(FB.loginRedirectUrl);
                return location.protocol + "//" + location.host + pathSegments.join("/")
            };
        var startLogin_browser = function(loginUrl) {
                var width = 512,
                    height = 320,
                    left = (screen.width - width) / 2,
                    top = (screen.height - height) / 2;
                window.open(loginUrl, null, "width=" + width + ",height=" + height + ",toolbar=0,scrollbars=0,status=0,resizable=0,menuBar=0,left=" + left + ",top=" + top)
            };
        var startLogin_cordova = function(loginUrl) {
                var ref = window.open(loginUrl, "_blank");
                ref.addEventListener('exit', function(event) {
                    pendingLoginRedirectUrl = null
                });
                ref.addEventListener('loadstop', function(event) {
                    var url = unescape(event.url);
                    if (url.indexOf(pendingLoginRedirectUrl) === 0) {
                        ref.close();
                        _processLoginRedirectUrl(url)
                    }
                })
            };
        var handleLoginRedirect = function() {
                var opener = window.opener;
                if (iosStandaloneMode) {
                    putData(IOS_STANDALONE_STEP2_KEY, location.href);
                    location.href = getData(IOS_STANDALONE_STEP1_KEY)
                }
                else if (opener && opener.DevExpress) {
                    opener.DevExpress.social.Facebook._processLoginRedirectUrl(location.href);
                    window.close()
                }
            };
        var _processLoginRedirectUrl = function(url) {
                var params = parseUrlFragment(url);
                expires = params.expires_in;
                changeToken(params.access_token);
                pendingLoginRedirectUrl = null
            };
        var parseUrlFragment = function(url) {
                var hash = url.split("#")[1];
                if (!hash)
                    return {};
                var pairs = hash.split(/&/g),
                    result = {};
                $.each(pairs, function(i) {
                    var splitPair = this.split("=");
                    result[splitPair[0]] = decodeURIComponent(splitPair[1])
                });
                return result
            };
        var logout = function() {
                notifyDeprecated();
                changeToken(null)
            };
        var changeToken = function(value) {
                if (value === accessToken)
                    return;
                accessToken = value;
                putData(ACCESS_TOKEN_KEY, value);
                connectionChanged.fire(!!value)
            };
        var api = function(resource, method, params) {
                notifyDeprecated();
                if (!isConnected())
                    throw Error("Not connected");
                if (typeof method !== "string") {
                    params = method;
                    method = undefined
                }
                method = (method || "get").toLowerCase();
                var d = $.Deferred();
                var args = arguments;
                $.ajax({
                    url: "https://graph.facebook.com/" + resource,
                    type: method,
                    data: $.extend({access_token: accessToken}, params),
                    dataType: FB.jsonp && method === "get" ? "jsonp" : "json"
                }).done(function(response) {
                    response = response || simulateErrorResponse();
                    if (response.error)
                        d.reject(response.error);
                    else
                        d.resolve(response)
                }).fail(function(xhr) {
                    var response;
                    try {
                        response = $.parseJSON(xhr.responseText);
                        var tries = args[3] || 0;
                        if (tries++ < 3 && response.error.code == 190 && response.error.error_subcode == 466) {
                            setTimeout(function() {
                                api(resource, method, params, tries).done(function(result) {
                                    d.resolve(result)
                                }).fail(function(error) {
                                    d.reject(error)
                                })
                            }, 500);
                            return
                        }
                    }
                    catch(x) {
                        response = simulateErrorResponse()
                    }
                    d.reject(response.error)
                });
                return d.promise()
            };
        var simulateErrorResponse = function() {
                return {error: {message: "Unknown error"}}
            };
        var ensureStorageBackend = function() {
                if (!hasStorageBackend())
                    throw Error("HTML5 sessionStorage or jQuery.cookie plugin is required");
            };
        var hasStorageBackend = function() {
                return !!($.cookie || window.sessionStorage)
            };
        var putData = function(key, data) {
                ensureStorageBackend();
                data = JSON.stringify(data);
                if (window.sessionStorage)
                    if (data === null)
                        sess.removeItem(key);
                    else
                        sessionStorage.setItem(key, data);
                else
                    $.cookie(key, data)
            };
        var getData = function(key) {
                ensureStorageBackend();
                try {
                    return JSON.parse(window.sessionStorage ? sessionStorage.getItem(key) : $.cookie(key))
                }
                catch(x) {
                    return null
                }
            };
        if (hasStorageBackend())
            accessToken = getData(ACCESS_TOKEN_KEY);
        if (iosStandaloneMode) {
            var url = getData(IOS_STANDALONE_STEP2_KEY);
            if (url) {
                _processLoginRedirectUrl(url);
                putData(IOS_STANDALONE_STEP1_KEY, null);
                putData(IOS_STANDALONE_STEP2_KEY, null)
            }
        }
        $.extend(FB, {
            login: login,
            logout: logout,
            handleLoginRedirect: handleLoginRedirect,
            _processLoginRedirectUrl: _processLoginRedirectUrl,
            api: api
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.js */
    (function($, DX, undefined) {
        var ui = DX.ui = {};
        var TemplateProvider = DX.Class.inherit({
                getTemplateClass: function() {
                    return Template
                },
                supportDefaultTemplate: function() {
                    return false
                },
                getDefaultTemplate: function() {
                    return null
                }
            });
        var Template = DX.Class.inherit({
                ctor: function(element, owner) {
                    this._template = this._element = $(element).detach();
                    this._owner = owner
                },
                render: function(container) {
                    var renderedTemplate = this._template.clone();
                    container.append(renderedTemplate);
                    return renderedTemplate
                },
                dispose: function() {
                    this._owner = null
                },
                owner: function() {
                    return this._owner
                }
            });
        var GESTURE_LOCK_KEY = "dxGestureLock";
        DX.registerActionExecutor({
            designMode: {validate: function(e) {
                    if (DX.designMode)
                        e.cancel = true
                }},
            gesture: {validate: function(e) {
                    if (!e.args.length)
                        return;
                    var args = e.args[0],
                        jQueryEvent = args.jQueryEvent;
                    if (!jQueryEvent)
                        return;
                    var element = $(jQueryEvent.target);
                    while (element && element.length) {
                        if (element.data(GESTURE_LOCK_KEY)) {
                            e.cancel = true;
                            break
                        }
                        element = element.parent()
                    }
                }},
            disabled: {validate: function(e) {
                    if (!e.args.length)
                        return;
                    var args = e.args[0],
                        element = args.itemElement || args.element;
                    if (element && element.is(".dx-state-disabled, .dx-state-disabled *"))
                        e.cancel = true
                }}
        });
        $.extend(ui, {
            TemplateProvider: TemplateProvider,
            Template: Template,
            initViewport: function() {
                DX.utils.logger.warn("DevExpress.ui.initViewport is deprecated. Use DX.utils.initMobileViewport instead");
                DX.utils.initMobileViewport()
            }
        });
        ui.__internals = ui.__internals || {};
        $.extend(ui.__internals, {Template: Template})
    })(jQuery, DevExpress);
    /*! Module core, file ko.components.js */
    (function($, DX, undefined) {
        if (!DX.support.hasKo)
            return;
        var ko = window.ko,
            ui = DX.ui,
            LOCKS_DATA_KEY = "dxKoLocks",
            CREATED_WITH_KO_DATA_KEY = "dxKoCreation";
        var Locks = function() {
                var info = {};
                var currentCount = function(lockName) {
                        return info[lockName] || 0
                    };
                return {
                        obtain: function(lockName) {
                            info[lockName] = currentCount(lockName) + 1
                        },
                        release: function(lockName) {
                            var count = currentCount(lockName);
                            if (count < 1)
                                throw Error("Not locked");
                            if (count === 1)
                                delete info[lockName];
                            else
                                info[lockName] = count - 1
                        },
                        locked: function(lockName) {
                            return currentCount(lockName) > 0
                        }
                    }
            };
        var registerComponentKoBinding = function(componentName) {
                ko.bindingHandlers[componentName] = {init: function(domNode, valueAccessor) {
                        var element = $(domNode),
                            ctorOptions = {},
                            optionNameToModelMap = {};
                        var applyModelValueToOption = function(optionName, modelValue) {
                                var component = element.data(componentName),
                                    locks = element.data(LOCKS_DATA_KEY),
                                    optionValue = ko.utils.unwrapObservable(modelValue);
                                if (ko.isWriteableObservable(modelValue))
                                    optionNameToModelMap[optionName] = modelValue;
                                if (component) {
                                    if (locks.locked(optionName))
                                        return;
                                    locks.obtain(optionName);
                                    try {
                                        component.option(optionName, optionValue)
                                    }
                                    finally {
                                        locks.release(optionName)
                                    }
                                }
                                else
                                    ctorOptions[optionName] = optionValue
                            };
                        var handleOptionChanged = function(optionName, optionValue) {
                                if (!(optionName in optionNameToModelMap))
                                    return;
                                var element = this._$element,
                                    locks = element.data(LOCKS_DATA_KEY);
                                if (locks.locked(optionName))
                                    return;
                                locks.obtain(optionName);
                                try {
                                    optionNameToModelMap[optionName](optionValue)
                                }
                                finally {
                                    locks.release(optionName)
                                }
                            };
                        var createComponent = function() {
                                element.data(CREATED_WITH_KO_DATA_KEY, true);
                                element[componentName](ctorOptions);
                                ctorOptions = null;
                                element.data(LOCKS_DATA_KEY, new Locks);
                                element.data(componentName).optionChanged.add(handleOptionChanged)
                            };
                        ko.computed(function() {
                            var component = element.data(componentName);
                            if (component)
                                component.beginUpdate();
                            $.each(ko.unwrap(valueAccessor()), function(modelName, modelValueExpr) {
                                ko.computed(function() {
                                    applyModelValueToOption(modelName, modelValueExpr)
                                }, null, {disposeWhenNodeIsRemoved: domNode})
                            });
                            if (component)
                                component.endUpdate();
                            else
                                createComponent()
                        }, null, {disposeWhenNodeIsRemoved: domNode});
                        return {controlsDescendantBindings: ui[componentName] && ui[componentName].subclassOf(ui.Widget)}
                    }}
            };
        var KoComponent = DX.DOMComponent.inherit({_modelByElement: function(element) {
                    if (element.length)
                        return ko.dataFor(element.get(0))
                }});
        var originalRegisterComponent = DX.registerComponent;
        var registerKoComponent = function(name, componentClass) {
                originalRegisterComponent(name, componentClass);
                registerComponentKoBinding(name)
            };
        ko.bindingHandlers.dxAction = {update: function(element, valueAccessor, allBindingsAccessor, viewModel) {
                var $element = $(element);
                var unwrappedValue = ko.utils.unwrapObservable(valueAccessor()),
                    actionSource = unwrappedValue,
                    actionOptions = {context: element};
                if (unwrappedValue.execute) {
                    actionSource = unwrappedValue.execute;
                    $.extend(actionOptions, unwrappedValue)
                }
                var action = new DX.Action(actionSource, actionOptions);
                $element.off(".dxActionBinding").on("dxclick.dxActionBinding", function(e) {
                    action.execute({
                        element: $element,
                        model: viewModel,
                        evaluate: function(expression) {
                            var context = viewModel;
                            if (expression.length > 0 && expression[0] === "$")
                                context = ko.contextFor(element);
                            var getter = DX.data.utils.compileGetter(expression);
                            return getter(context)
                        },
                        jQueryEvent: e
                    });
                    if (!actionOptions.bubbling)
                        e.stopPropagation()
                })
            }};
        var cleanKoData = function(element, andSelf) {
                var cleanNode = function() {
                        ko.cleanNode(this)
                    };
                if (andSelf)
                    element.each(cleanNode);
                else
                    element.find("*").each(cleanNode)
            };
        var originalEmpty = $.fn.empty;
        $.fn.empty = function() {
            cleanKoData(this, false);
            return originalEmpty.apply(this, arguments)
        };
        var originalRemove = $.fn.remove;
        $.fn.remove = function(selector, keepData) {
            if (!keepData) {
                var subject = this;
                if (selector)
                    subject = subject.filter(selector);
                cleanKoData(subject, true)
            }
            return originalRemove.call(this, selector, keepData)
        };
        var originalHtml = $.fn.html;
        $.fn.html = function(value) {
            if (typeof value === "string")
                cleanKoData(this, false);
            return originalHtml.apply(this, arguments)
        };
        DX.registerComponent = registerKoComponent;
        registerKoComponent("DOMComponent", KoComponent)
    })(jQuery, DevExpress);
    /*! Module core, file ng.components.js */
    (function($, DX, undefined) {
        if (!DX.support.hasNg)
            return;
        var ui = DX.ui,
            compileSetter = DX.data.utils.compileSetter,
            compileGetter = DX.data.utils.compileGetter;
        var CREATED_WITH_NG_DATA_KEY = "dxNgCreation",
            TEMPLATES_DATA_KEY = "dxTemplates",
            COMPILER_DATA_KEY = "dxNgCompiler",
            DEFAULT_COMPILER_DATA_KEY = "dxDefaultCompilerGetter",
            ANONYMOUS_TEMPLATE_NAME = "template";
        var phoneJsModule = DX.ng.module;
        var ComponentBuilder = DX.Class.inherit({
                ctor: function(options) {
                    this._componentName = options.componentName;
                    this._$element = options.$element;
                    this._$templates = options.$templates;
                    this._scope = options.scope;
                    this._compile = options.compile;
                    this._ngOptions = options.ngOptions;
                    this._componentDisposing = $.Callbacks();
                    this._$element.data(CREATED_WITH_NG_DATA_KEY, true);
                    if (options.ngOptions.data)
                        this._initDataScope(options.ngOptions.data)
                },
                initDefaultCompilerGetter: function() {
                    var that = this;
                    that._$element.data(DEFAULT_COMPILER_DATA_KEY, function($template) {
                        return that._compilerByTemplate($template)
                    })
                },
                initTemplateCompilers: function() {
                    var that = this;
                    if (this._$templates)
                        this._$templates.each(function(i, template) {
                            $(template).data(COMPILER_DATA_KEY, that._compilerByTemplate(template))
                        })
                },
                initComponentWithBindings: function() {
                    this._initComponent(this._scope);
                    this._initComponentBindings()
                },
                _initDataScope: function(data) {
                    if (typeof data === "string") {
                        var dataStr = data,
                            rootScope = this._scope;
                        data = rootScope.$eval(data);
                        this._scope = rootScope.$new();
                        this._synchronizeDataScopes(rootScope, this._scope, data, dataStr)
                    }
                    $.extend(this._scope, data)
                },
                _synchronizeDataScopes: function(parentScope, childScope, data, parentPrefix) {
                    var that = this;
                    $.each(data, function(fieldPath) {
                        that._synchronizeScopeField({
                            parentScope: parentScope,
                            childScope: childScope,
                            fieldPath: fieldPath,
                            parentPrefix: parentPrefix
                        })
                    })
                },
                _initComponent: function(scope) {
                    this._component = this._$element[this._componentName](this._evalOptions(scope)).data(this._componentName)
                },
                _initComponentBindings: function() {
                    var that = this,
                        optionDependencies = {};
                    if (that._ngOptions.bindingOptions)
                        $.each(that._ngOptions.bindingOptions, function(optionPath, valuePath) {
                            var separatorIndex = optionPath.search(/\[|\./),
                                optionForSubscribe = separatorIndex > -1 ? optionPath.substring(0, separatorIndex) : optionPath,
                                watchMethod = $.isArray(that._scope.$eval(valuePath)) ? "$watchCollection" : "$watch";
                            if (!optionDependencies[optionForSubscribe])
                                optionDependencies[optionForSubscribe] = {};
                            optionDependencies[optionForSubscribe][optionPath] = valuePath;
                            var clearWatcher = that._scope[watchMethod](valuePath, function(newValue, oldValue) {
                                    if (newValue !== oldValue)
                                        that._component.option(optionPath, newValue)
                                }, true);
                            that._component.disposing.add(function() {
                                clearWatcher();
                                that._componentDisposing.fire()
                            })
                        });
                    that._component.optionChanged.add(function(optionName, optionValue) {
                        if (that._scope.$root.$$phase === "$digest" || !optionDependencies || !optionDependencies[optionName])
                            return;
                        safeApply(function(scope) {
                            $.each(optionDependencies[optionName], function(optionPath, valuePath) {
                                var setter = compileSetter(valuePath),
                                    getter = compileGetter(optionPath);
                                var tmpData = {};
                                tmpData[optionName] = optionValue;
                                setter(scope, getter(tmpData))
                            })
                        }, that._scope)
                    })
                },
                _compilerByTemplate: function(template) {
                    var that = this,
                        scopeItemsPath = this._getScopeItemsPath();
                    return function(data, index) {
                            var $resultMarkup = $(template).clone(),
                                templateScope;
                            if (data !== undefined) {
                                var dataIsScope = data.$id,
                                    templateScope = dataIsScope ? data : that._createScopeWithData(data);
                                $resultMarkup.on("$destroy", function() {
                                    var destroyAlreadyCalled = !templateScope.$parent;
                                    if (destroyAlreadyCalled)
                                        return;
                                    templateScope.$destroy()
                                })
                            }
                            else
                                templateScope = that._scope;
                            if (scopeItemsPath)
                                that._synchronizeScopes(templateScope, scopeItemsPath, index);
                            safeApply(that._compile($resultMarkup), templateScope);
                            return $resultMarkup
                        }
                },
                _getScopeItemsPath: function() {
                    if (ui[this._componentName].subclassOf(ui.CollectionContainerWidget) && this._ngOptions.bindingOptions)
                        return this._ngOptions.bindingOptions.items
                },
                _createScopeWithData: function(data) {
                    var newScope = this._scope.$new(true);
                    if (typeof data === "object")
                        $.extend(newScope, data);
                    else
                        newScope.scopeValue = data;
                    return newScope
                },
                _synchronizeScopes: function(itemScope, parentPrefix, itemIndex) {
                    var that = this,
                        item = compileGetter(parentPrefix + "[" + itemIndex + "]")(this._scope);
                    if (!$.isPlainObject(item))
                        item = {scopeValue: item};
                    $.each(item, function(itemPath) {
                        that._synchronizeScopeField({
                            parentScope: that._scope,
                            childScope: itemScope,
                            fieldPath: itemPath,
                            parentPrefix: parentPrefix,
                            itemIndex: itemIndex
                        })
                    })
                },
                _synchronizeScopeField: function(args) {
                    var parentScope = args.parentScope,
                        childScope = args.childScope,
                        fieldPath = args.fieldPath,
                        parentPrefix = args.parentPrefix,
                        itemIndex = args.itemIndex;
                    var innerPathSuffix = fieldPath === "scopeValue" ? "" : "." + fieldPath,
                        collectionField = itemIndex !== undefined,
                        optionOuterBag = [parentPrefix],
                        optionOuterPath;
                    if (collectionField)
                        optionOuterBag.push("[", itemIndex, "]");
                    optionOuterBag.push(innerPathSuffix);
                    optionOuterPath = optionOuterBag.join("");
                    var clearParentWatcher = parentScope.$watch(optionOuterPath, function(newValue, oldValue) {
                            if (newValue !== oldValue)
                                compileSetter(fieldPath)(childScope, newValue)
                        });
                    var clearItemWatcher = childScope.$watch(fieldPath, function(newValue, oldValue) {
                            if (newValue !== oldValue) {
                                if (collectionField && !compileGetter(parentPrefix)(parentScope)[itemIndex]) {
                                    clearItemWatcher();
                                    return
                                }
                                compileSetter(optionOuterPath)(parentScope, newValue)
                            }
                        });
                    this._componentDisposing.add([clearParentWatcher, clearItemWatcher])
                },
                _evalOptions: function(scope) {
                    var result = $.extend({}, this._ngOptions);
                    delete result.data;
                    delete result.bindingOptions;
                    if (this._ngOptions.bindingOptions)
                        $.each(this._ngOptions.bindingOptions, function(key, value) {
                            result[key] = scope.$eval(value)
                        });
                    return result
                }
            });
        var safeApply = function(func, scope) {
                if (scope.$root.$$phase)
                    func(scope);
                else
                    scope.$apply(function() {
                        func(scope)
                    })
            };
        var extractTemplates = function($element, componentName) {
                if ($element.data(TEMPLATES_DATA_KEY))
                    return $element.data(TEMPLATES_DATA_KEY);
                var $templates;
                if (ui[componentName] && ui[componentName].subclassOf(ui.Widget) && $.trim($element.html())) {
                    var isAnonymousTemplate = !$element.children().first().attr("data-options");
                    if (isAnonymousTemplate)
                        $templates = $("<div/>").attr("data-options", "dxTemplate: { name: '" + ANONYMOUS_TEMPLATE_NAME + "' }").append($element.contents());
                    else
                        $templates = $element.children().detach();
                    $element.data(TEMPLATES_DATA_KEY, $templates)
                }
                return $templates
            };
        var NgComponent = DX.DOMComponent.inherit({
                _modelByElement: function(element) {
                    if (element.length)
                        return element.scope()
                },
                _createActionByOption: function() {
                    var action = this.callBase.apply(this, arguments);
                    var component = this,
                        wrappedAction = function() {
                            var that = this,
                                scope = component._modelByElement(component._element()),
                                args = arguments;
                            if (!scope || scope.$root.$$phase)
                                return action.apply(that, args);
                            return scope.$apply(function() {
                                    return action.apply(that, args)
                                })
                        };
                    return wrappedAction
                }
            });
        var originalRegisterComponent = DX.registerComponent;
        var registerNgComponent = function(componentName, componentClass) {
                originalRegisterComponent(componentName, componentClass);
                phoneJsModule.directive(componentName, ["$compile", function(compile) {
                        return {
                                restrict: "A",
                                compile: function($element) {
                                    var $templates = extractTemplates($element, componentName);
                                    return function(scope, $element, attrs) {
                                            var componentBuilder = new ComponentBuilder({
                                                    componentName: componentName,
                                                    compile: compile,
                                                    $element: $element,
                                                    scope: scope,
                                                    ngOptions: attrs[componentName] ? scope.$eval(attrs[componentName]) : {},
                                                    $templates: $templates
                                                });
                                            componentBuilder.initTemplateCompilers();
                                            componentBuilder.initDefaultCompilerGetter();
                                            componentBuilder.initComponentWithBindings()
                                        }
                                }
                            }
                    }])
            };
        DX.registerComponent = registerNgComponent;
        registerNgComponent("DOMComponent", NgComponent)
    })(jQuery, DevExpress);
    /*! Module core, file ko.templates.js */
    (function($, DX, undefined) {
        if (!DX.support.hasKo)
            return;
        var ko = window.ko,
            ui = DX.ui,
            CREATED_WITH_KO_DATA_KEY = "dxKoCreation";
        var KoTemplate = ui.Template.inherit({
                ctor: function(element) {
                    this.callBase.apply(this, arguments);
                    this._template = $("<div>").append(element);
                    this._registerKoTemplate()
                },
                _cleanTemplateElement: function() {
                    this._element.each(function() {
                        ko.cleanNode(this)
                    })
                },
                _registerKoTemplate: function() {
                    var template = this._template.get(0);
                    new ko.templateSources.anonymousTemplate(template)['nodes'](template)
                },
                render: function(container, data) {
                    data = data !== undefined ? data : ko.dataFor(container.get(0)) || {};
                    var containerBindingContext = ko.contextFor(container[0]);
                    var bindingContext = containerBindingContext ? containerBindingContext.createChildContext(data) : data;
                    var renderBag = $("<div />").appendTo(container);
                    ko.renderTemplate(this._template.get(0), bindingContext, null, renderBag.get(0));
                    var result = renderBag.contents();
                    container.append(result);
                    renderBag.remove();
                    return result
                },
                dispose: function() {
                    this.callBase();
                    this._template.remove()
                }
            });
        var KoTemplateProvider = ui.TemplateProvider.inherit({
                getTemplateClass: function(widget) {
                    return this._createdWithKo(widget) ? KoTemplate : this.callBase(widget)
                },
                supportDefaultTemplate: function(widget) {
                    return this._createdWithKo(widget) ? true : this.callBase(widget)
                },
                getDefaultTemplate: function(widget) {
                    if (this._createdWithKo(widget))
                        return defaultKoTemplate(widget.NAME)
                },
                _createdWithKo: function(widget) {
                    return !!widget._element().data(CREATED_WITH_KO_DATA_KEY)
                }
            });
        var defaultKoTemplate = function() {
                var cache = {};
                return function(widgetName) {
                        if (!DEFAULT_ITEM_TEMPLATE_GENERATORS[widgetName])
                            widgetName = "base";
                        if (!cache[widgetName]) {
                            var html = DEFAULT_ITEM_TEMPLATE_GENERATORS[widgetName](),
                                markup = DX.utils.createMarkupFromString(html);
                            cache[widgetName] = new KoTemplate(markup)
                        }
                        return cache[widgetName]
                    }
            }();
        var createElementWithBindAttr = function(tagName, bindings, closeTag, additionalProperties) {
                closeTag = closeTag === undefined ? true : closeTag;
                var bindAttr = $.map(bindings, function(value, key) {
                        return key + ":" + value
                    }).join(",");
                additionalProperties = additionalProperties || "";
                return "<" + tagName + " data-bind=\"" + bindAttr + "\" " + additionalProperties + ">" + (closeTag ? "</" + tagName + ">" : "")
            };
        var defaultKoTemplateBasicBindings = {css: "{ 'dx-state-disabled': $data.disabled, 'dx-state-invisible': !($data.visible === undefined || ko.unwrap($data.visible)) }"};
        var DEFAULT_ITEM_TEMPLATE_GENERATORS = {base: function() {
                    var template = [createElementWithBindAttr("div", defaultKoTemplateBasicBindings, false)],
                        htmlBinding = createElementWithBindAttr("div", {html: "html"}),
                        textBinding = createElementWithBindAttr("div", {text: "text"}),
                        primitiveBinding = createElementWithBindAttr("div", {text: "String($data)"});
                    template.push("<!-- ko if: $data.html && !$data.text -->", htmlBinding, "<!-- /ko -->", "<!-- ko if: !$data.html && $data.text -->", textBinding, "<!-- /ko -->", "<!-- ko ifnot: $.isPlainObject($data) -->", primitiveBinding, "<!-- /ko -->", "</div>");
                    return template.join("")
                }};
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxPivotTabs = function() {
            var template = DEFAULT_ITEM_TEMPLATE_GENERATORS.base(),
                titleBinding = createElementWithBindAttr("span", {text: "title"});
            var divInnerStart = template.indexOf(">") + 1,
                divInnerFinish = template.length - 6;
            template = [template.substring(0, divInnerStart), titleBinding, template.substring(divInnerFinish, template.length)];
            return template.join("")
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxPanorama = function() {
            var template = DEFAULT_ITEM_TEMPLATE_GENERATORS.base(),
                headerBinding = createElementWithBindAttr("div", {text: "header"}, true, 'class="dx-panorama-item-header"');
            var divInnerStart = template.indexOf(">") + 1;
            template = [template.substring(0, divInnerStart), "<!-- ko if: $data.header -->", headerBinding, "<!-- /ko -->", template.substring(divInnerStart, template.length)];
            return template.join("")
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxList = function() {
            var template = DEFAULT_ITEM_TEMPLATE_GENERATORS.base(),
                keyBinding = createElementWithBindAttr("div", {text: "key"});
            template = [template.substring(0, template.length - 6), "<!-- ko if: $data.key -->" + keyBinding + "<!-- /ko -->", "</div>"];
            return template.join("")
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxToolbar = function() {
            var template = DEFAULT_ITEM_TEMPLATE_GENERATORS.base();
            template = [template.substring(0, template.length - 6), "<!-- ko if: $data.widget -->"];
            $.each(["button", "tabs", "dropDownMenu"], function() {
                var bindingName = DX.inflector.camelize(["dx", "-", this].join("")),
                    bindingObj = {};
                bindingObj[bindingName] = "$data.options";
                template.push("<!-- ko if: $data.widget === '", this, "' -->", createElementWithBindAttr("div", bindingObj), "<!-- /ko -->")
            });
            template.push("<!-- /ko -->");
            return template.join("")
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxGallery = function() {
            var template = DEFAULT_ITEM_TEMPLATE_GENERATORS.base(),
                primitiveBinding = createElementWithBindAttr("div", {text: "String($data)"}),
                imgBinding = createElementWithBindAttr("img", {attr: "{ src: String($data) }"}, false);
            template = template.replace(primitiveBinding, imgBinding);
            return template
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxTabs = function() {
            var template = DEFAULT_ITEM_TEMPLATE_GENERATORS.base(),
                baseTextBinding = createElementWithBindAttr("div", {text: "text"}),
                iconBinding = createElementWithBindAttr("span", {
                    attr: "{ 'class': 'dx-icon-' + $data.icon }",
                    css: "{ 'dx-icon': true }"
                }),
                iconSrcBinding = createElementWithBindAttr("img", {
                    attr: "{ src: $data.iconSrc }",
                    css: "{ 'dx-icon': true }"
                }, false),
                textBinding = "<!-- ko if: $data.icon -->" + iconBinding + "<!-- /ko -->" + "<!-- ko if: !$data.icon && $data.iconSrc -->" + iconSrcBinding + "<!-- /ko -->" + "<span class=\"dx-tab-text\" data-bind=\"text: $data.text\"></span>";
            template = template.replace("<!-- ko if: !$data.html && $data.text -->", "<!-- ko if: !$data.html && ($data.text || $data.icon || $data.iconSrc) -->").replace(baseTextBinding, textBinding);
            return template
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxActionSheet = function() {
            return createElementWithBindAttr("div", {dxButton: "{ text: $data.text, clickAction: $data.clickAction, type: $data.type, disabled: !!ko.unwrap($data.disabled) }"})
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxNavBar = DEFAULT_ITEM_TEMPLATE_GENERATORS.dxTabs;
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxMenu = function() {
            var template = [createElementWithBindAttr("div", defaultKoTemplateBasicBindings, false)],
                iconBinding = createElementWithBindAttr("span", {
                    attr: "{ 'class': 'dx-icon-' + $data.icon }",
                    css: "{ 'dx-icon': true }"
                }),
                iconSrcBinding = createElementWithBindAttr("img", {
                    attr: "{ src: $data.iconSrc }",
                    css: "{ 'dx-icon': true }"
                }),
                textBinding = createElementWithBindAttr("span", {
                    text: "text",
                    css: "{ 'dx-menu-item-text': true }"
                }),
                primitiveBinding = createElementWithBindAttr("span", {
                    text: "String($data)",
                    css: "{ 'dx-menu-item-text': true }"
                }),
                popout = '<span class="dx-menu-item-popout-container"><div class="dx-menu-item-popout"></div></span>';
            template.push('<div class="dx-menu-item-content">', '<!-- ko if: $data.icon -->', iconBinding, '<!-- /ko -->', '<!-- ko if: !$data.icon && $data.iconSrc -->', iconSrcBinding, '<!-- /ko -->', '<!-- ko if: $.isPlainObject($data) -->', textBinding, '<!-- /ko -->', '<!-- ko ifnot: $.isPlainObject($data) -->', primitiveBinding, '<!-- /ko -->', '<!-- ko if: $data.items -->', popout, '<!-- /ko -->', '</div>', '</div>');
            return template.join("")
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxContextMenu = DEFAULT_ITEM_TEMPLATE_GENERATORS.dxMenu;
        $.extend(ui, {
            TemplateProvider: KoTemplateProvider,
            Template: KoTemplate,
            defaultTemplate: defaultKoTemplate
        });
        ui.__internals = ui.__internals || {};
        $.extend(ui.__internals, {KoTemplate: KoTemplate})
    })(jQuery, DevExpress);
    /*! Module core, file ng.templates.js */
    (function($, DX, undefined) {
        if (!DX.support.hasNg)
            return;
        var ui = DX.ui;
        var CREATED_WITH_NG_DATA_KEY = "dxNgCreation",
            COMPILER_DATA_KEY = "dxNgCompiler",
            DEFAULT_COMPILER_DATA_KEY = "dxDefaultCompilerGetter";
        var NgTemplate = ui.Template.inherit({
                ctor: function() {
                    this.callBase.apply(this, arguments);
                    this._compiler = this._template.data(COMPILER_DATA_KEY)
                },
                render: function(container, data, index) {
                    var compiler = this._compiler,
                        result = $.isFunction(compiler) ? compiler(data, index) : compiler;
                    container.append(result);
                    return result
                },
                setCompiler: function(compilerGetter) {
                    this._compiler = compilerGetter(this._element)
                }
            });
        var NgTemplateProvider = ui.TemplateProvider.inherit({
                getTemplateClass: function(widget) {
                    if (this._createdWithNg(widget))
                        return NgTemplate;
                    return this.callBase(widget)
                },
                supportDefaultTemplate: function(widget) {
                    return this._createdWithNg(widget) ? true : this.callBase(widget)
                },
                getDefaultTemplate: function(widget) {
                    if (this._createdWithNg(widget)) {
                        var compilerGetter = widget._element().data(DEFAULT_COMPILER_DATA_KEY),
                            template = defaultNgTemplate(widget.NAME);
                        template.setCompiler(compilerGetter);
                        return template
                    }
                },
                _createdWithNg: function(widget) {
                    return !!widget._element().data(CREATED_WITH_NG_DATA_KEY)
                }
            });
        var defaultNgTemplate = function() {
                var cache = {};
                return function(widgetName) {
                        if (!DEFAULT_ITEM_TEMPLATE_GENERATORS[widgetName])
                            widgetName = "base";
                        if (!cache[widgetName])
                            cache[widgetName] = DEFAULT_ITEM_TEMPLATE_GENERATORS[widgetName]();
                        return new NgTemplate(cache[widgetName])
                    }
            }();
        var baseElements = {
                container: function() {
                    return $("<div>").attr("ng-class", "{ 'dx-state-invisible': !visible && visible != undefined, 'dx-state-disabled': !!disabled }")
                },
                html: function() {
                    return $("<div>").attr("ng-if", "html").attr("ng-bind-html", "html")
                },
                text: function() {
                    return $("<div>").attr("ng-if", "text").attr("ng-bind", "text")
                },
                primitive: function() {
                    return $("<div>").attr("ng-if", "scopeValue").attr("ng-bind-html", "'' + scopeValue")
                }
            };
        var DEFAULT_ITEM_TEMPLATE_GENERATORS = {base: function() {
                    return baseElements.container().append(baseElements.html()).append(baseElements.text()).append(baseElements.primitive())
                }};
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxList = function() {
            return DEFAULT_ITEM_TEMPLATE_GENERATORS.base().append($("<div>").attr("ng-if", "key").attr("ng-bind", "key"))
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxToolbar = function() {
            var template = DEFAULT_ITEM_TEMPLATE_GENERATORS.base();
            $.each(["button", "tabs", "dropDownMenu"], function(i, widgetName) {
                var bindingName = "dx-" + DX.inflector.dasherize(this);
                $("<div>").attr("ng-if", "widget === '" + widgetName + "'").attr(bindingName, "options").appendTo(template)
            });
            return template
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxGallery = function() {
            return baseElements.container().append(baseElements.html()).append(baseElements.text()).append($("<img>").attr("ng-if", "scopeValue").attr("ng-src", "{{'' + scopeValue}}"))
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxTabs = function() {
            var container = baseElements.container();
            var text = $("<span>").addClass("dx-tab-text").attr("ng-bind", "text").attr("ng-if", "text"),
                icon = $("<span>").attr("ng-if", "icon").addClass("dx-icon").attr("ng-class", "'dx-icon-' + icon"),
                iconSrc = $("<img>").attr("ng-if", "iconSrc").addClass("dx-icon").attr("ng-src", "{{iconSrc}}");
            return container.append(baseElements.html()).append(icon).append(iconSrc).append(text).append(baseElements.primitive())
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxMenu = function() {
            var container = baseElements.container();
            var content = $("<div>").addClass("dx-menu-item-content"),
                text = $("<span>").attr("ng-if", "text").addClass("dx-menu-item-text").attr("ng-bind", "text"),
                icon = $("<span>").attr("ng-if", "icon").addClass("dx-icon").attr("ng-class", "'dx-icon-' + icon"),
                iconSrc = $("<img>").attr("ng-if", "iconSrc").addClass("dx-icon").attr("ng-src", "{{iconSrc}}"),
                popout = $("<span>").addClass("dx-menu-item-popout-container").attr("ng-if", "items").append($("<div>").addClass("dx-menu-item-popout"));
            content.append(baseElements.html()).append(icon).append(iconSrc).append(text).append(popout).append(baseElements.primitive()).appendTo(container);
            return container
        },
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxActionSheet = function() {
            return $("<div>").attr("dx-button", "{ bindingOptions: { text: 'text', clickAction: 'clickAction', type: 'type', disabled: 'disabled' } }")
        };
        DEFAULT_ITEM_TEMPLATE_GENERATORS.dxNavBar = DEFAULT_ITEM_TEMPLATE_GENERATORS.dxTabs;
        $.extend(ui, {
            Template: NgTemplate,
            TemplateProvider: NgTemplateProvider
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.themes.js */
    (function($, DX, undefined) {
        var DX_LINK_SELECTOR = "link[rel=dx-theme]",
            THEME_ATTR = "data-theme",
            ACTIVE_ATTR = "data-active";
        var context,
            $activeThemeLink,
            knownThemes,
            currentThemeName,
            pendingThemeName;
        var THEME_MARKER_PREFIX = "dx.";
        function readThemeMarker() {
            var element = $("<div></div>", context).addClass("dx-theme-marker").appendTo(context.documentElement),
                result;
            try {
                result = element.css("font-family");
                if (!result)
                    return null;
                result = result.replace(/["']/g, "");
                if (result.substr(0, THEME_MARKER_PREFIX.length) !== THEME_MARKER_PREFIX)
                    return null;
                return result.substr(THEME_MARKER_PREFIX.length)
            }
            finally {
                element.remove()
            }
        }
        function waitForThemeLoad(themeName, callback) {
            var timerId,
                waitStartTime;
            pendingThemeName = themeName;
            function handleLoaded() {
                pendingThemeName = null;
                callback()
            }
            if (isPendingThemeLoaded())
                handleLoaded();
            else {
                waitStartTime = $.now();
                timerId = setInterval(function() {
                    var isLoaded = isPendingThemeLoaded(),
                        isTimeout = !isLoaded && $.now() - waitStartTime > 15 * 1000;
                    if (isTimeout)
                        DX.utils.logger.warn("Theme loading timed out: " + pendingThemeName);
                    if (isLoaded || isTimeout) {
                        clearInterval(timerId);
                        handleLoaded()
                    }
                }, 10)
            }
        }
        function isPendingThemeLoaded() {
            return !pendingThemeName || readThemeMarker() === pendingThemeName
        }
        function processMarkup() {
            var $allThemeLinks = $(DX_LINK_SELECTOR, context);
            if (!$allThemeLinks.length)
                return;
            knownThemes = {};
            $activeThemeLink = $(DX.utils.createMarkupFromString("<link rel=stylesheet>"), context);
            $allThemeLinks.each(function() {
                var link = $(this, context),
                    fullThemeName = link.attr(THEME_ATTR),
                    url = link.attr("href"),
                    isActive = link.attr(ACTIVE_ATTR) === "true";
                knownThemes[fullThemeName] = {
                    url: url,
                    isActive: isActive
                }
            });
            $allThemeLinks.last().after($activeThemeLink);
            $allThemeLinks.remove()
        }
        function resolveFullThemeName(desiredThemeName) {
            var desiredThemeParts = desiredThemeName.split("."),
                result = null;
            if (knownThemes)
                $.each(knownThemes, function(knownThemeName, themeData) {
                    var knownThemeParts = knownThemeName.split(".");
                    if (knownThemeParts[0] !== desiredThemeParts[0])
                        return;
                    if (desiredThemeParts[1] && desiredThemeParts[1] !== knownThemeParts[1])
                        return;
                    if (!result || themeData.isActive)
                        result = knownThemeName;
                    if (themeData.isActive)
                        return false
                });
            return result
        }
        function initContext(newContext) {
            try {
                if (newContext !== context)
                    knownThemes = null
            }
            catch(x) {
                knownThemes = null
            }
            context = newContext
        }
        function init(options) {
            options = options || {};
            initContext(options.context || document);
            processMarkup();
            currentThemeName = undefined;
            current(options)
        }
        function current(options) {
            if (!arguments.length)
                return currentThemeName || readThemeMarker();
            options = options || {};
            if (typeof options === "string")
                options = {theme: options};
            var isAutoInit = options._autoInit,
                loadCallback = options.loadCallback,
                currentThemeData;
            currentThemeName = options.theme || currentThemeName;
            if (isAutoInit && !currentThemeName)
                currentThemeName = themeNameFromDevice(DX.devices.current());
            currentThemeName = resolveFullThemeName(currentThemeName);
            if (currentThemeName)
                currentThemeData = knownThemes[currentThemeName];
            if (currentThemeData) {
                $activeThemeLink.removeAttr("href");
                $activeThemeLink.attr("href", knownThemes[currentThemeName].url);
                if (loadCallback)
                    waitForThemeLoad(currentThemeName, loadCallback);
                else if (pendingThemeName)
                    pendingThemeName = currentThemeName
            }
            else if (isAutoInit) {
                if (loadCallback)
                    loadCallback()
            }
            else
                throw Error("Unknown theme: " + currentThemeName);
        }
        function themeNameFromDevice(device) {
            var themeName = device.platform,
                majorVersion = device.version && device.version[0];
            if (themeName === "ios" && (!majorVersion || majorVersion > 6))
                themeName += "7";
            return themeName
        }
        function getCssClasses(themeName) {
            themeName = themeName || current();
            var result = [],
                themeNameParts = themeName && themeName.split(".");
            if (themeNameParts) {
                result.push("dx-theme-" + themeNameParts[0], "dx-theme-" + themeNameParts[0] + "-typography");
                if (themeNameParts.length > 1)
                    result.push("dx-color-scheme-" + themeNameParts[1])
            }
            return result
        }
        function attachCssClasses(element, themeName) {
            $(element).addClass(getCssClasses(themeName).join(" "))
        }
        function detachCssClasses(element, themeName) {
            $(element).removeClass(getCssClasses(themeName).join(" "))
        }
        $.holdReady(true);
        init({
            _autoInit: true,
            loadCallback: function() {
                $.holdReady(false)
            }
        });
        $(function() {
            if ($(DX_LINK_SELECTOR, context).length)
                throw Error("LINK[rel=dx-theme] tags must go before DevExpress included scripts");
        });
        DX.ui.themes = {
            init: init,
            current: current,
            attachCssClasses: attachCssClasses,
            detachCssClasses: detachCssClasses
        };
        DX.ui.themes.__internals = {
            themeNameFromDevice: themeNameFromDevice,
            waitForThemeLoad: waitForThemeLoad,
            resetTheme: function() {
                $activeThemeLink.attr("href", "about:blank");
                currentThemeName = null;
                pendingThemeName = null
            }
        }
    })(jQuery, DevExpress);
    /*! Module core, file ui.events.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            eventNS = $.event,
            specialNS = eventNS.special,
            EVENT_SOURCES_REGEX = {
                mouse: /(mouse|wheel|click)/i,
                wheel: /wheel/i,
                touch: /^touch/i,
                keyboard: /^key/i,
                pointer: /pointer/i
            };
        var eventSource = function(e) {
                var result = "other";
                $.each(EVENT_SOURCES_REGEX, function(key) {
                    if (this.test(e.type)) {
                        result = key;
                        return false
                    }
                });
                return result
            };
        var isPointerEvent = function(e) {
                return eventSource(e) === "pointer"
            };
        var isMouseEvent = function(e) {
                return eventSource(e) === "mouse" || isPointerEvent(e) && e.pointerType === "mouse"
            };
        var isTouchEvent = function(e) {
                return eventSource(e) === "touch" || isPointerEvent(e) && e.pointerType === "touch"
            };
        var isKeyboardEvent = function(e) {
                return eventSource(e) === "keyboard"
            };
        var isMouseWheelEvent = function(e) {
                return EVENT_SOURCES_REGEX["wheel"].test(e.type)
            };
        var addNamespace = function(eventNames, namespace) {
                if (!namespace)
                    throw Error("Namespace is not defined");
                if (typeof eventNames === "string")
                    return addNamespace(eventNames.split(/\s+/g), namespace);
                $.each(eventNames, function(index, eventName) {
                    eventNames[index] = eventName + "." + namespace
                });
                return eventNames.join(" ")
            };
        var eventData = function(e) {
                if (isPointerEvent(e) && isTouchEvent(e)) {
                    var touch = (e.originalEvent.originalEvent || e.originalEvent).changedTouches[0];
                    return {
                            x: touch.pageX,
                            y: touch.pageY,
                            time: e.timeStamp
                        }
                }
                if (isMouseEvent(e))
                    return {
                            x: e.pageX,
                            y: e.pageY,
                            time: e.timeStamp
                        };
                if (isTouchEvent(e)) {
                    var touch = (e.changedTouches || e.originalEvent.changedTouches)[0];
                    return {
                            x: touch.pageX,
                            y: touch.pageY,
                            time: e.timeStamp
                        }
                }
            };
        var eventDelta = function(from, to) {
                return {
                        x: to.x - from.x,
                        y: to.y - from.y,
                        time: to.time - from.time || 1
                    }
            };
        var hasTouches = function(e) {
                if (isTouchEvent(e))
                    return (e.originalEvent.touches || []).length;
                return 0
            };
        var needSkipEvent = function(e) {
                var $target = $(e.target),
                    touchInInput = $target.is("input, textarea, select");
                if (isMouseWheelEvent(e) && touchInInput)
                    return false;
                if (isMouseEvent(e))
                    return touchInInput || e.which > 1;
                if (isTouchEvent(e))
                    return touchInInput && $target.is(":focus") || (e.originalEvent.changedTouches || e.originalEvent.originalEvent.changedTouches).length !== 1
            };
        var createEvent = function(sourceEvent, props) {
                var event = $.Event(sourceEvent),
                    originalEvent = event.originalEvent,
                    propNames = $.event.props.slice();
                if (isMouseEvent(sourceEvent) || isTouchEvent(sourceEvent))
                    $.merge(propNames, $.event.mouseHooks.props);
                if (isKeyboardEvent(sourceEvent))
                    $.merge(propNames, $.event.keyHooks.props);
                if (originalEvent)
                    $.each(propNames, function() {
                        event[this] = originalEvent[this]
                    });
                if (props)
                    $.extend(event, props);
                return event
            };
        var fireEvent = function(props) {
                var event = createEvent(props.originalEvent, props);
                $.event.trigger(event, null, props.target || event.target);
                return event
            };
        var handleGestureEvent = function(e, type) {
                var gestureEvent = $(e.target).data("dxGestureEvent");
                if (!gestureEvent || gestureEvent === type) {
                    $(e.target).data("dxGestureEvent", type);
                    return true
                }
                return false
            };
        var registerEvent = function(eventName, eventObject) {
                var strategy = {};
                if ("noBubble" in eventObject)
                    strategy.noBubble = eventObject.noBubble;
                if ("bindType" in eventObject)
                    strategy.bindType = eventObject.bindType;
                if ("delegateType" in eventObject)
                    strategy.delegateType = eventObject.delegateType;
                $.each(["setup", "teardown", "add", "remove", "trigger", "handle", "_default", "dispose"], function(_, methodName) {
                    if (!eventObject[methodName])
                        return;
                    strategy[methodName] = function() {
                        var args = $.makeArray(arguments);
                        args.unshift(this);
                        return eventObject[methodName].apply(eventObject, args)
                    }
                });
                specialNS[eventName] = strategy
            };
        ui.events = {
            eventSource: eventSource,
            isPointerEvent: isPointerEvent,
            isMouseEvent: isMouseEvent,
            isTouchEvent: isTouchEvent,
            isKeyboardEvent: isKeyboardEvent,
            addNamespace: addNamespace,
            hasTouches: hasTouches,
            eventData: eventData,
            eventDelta: eventDelta,
            needSkipEvent: needSkipEvent,
            createEvent: createEvent,
            fireEvent: fireEvent,
            handleGestureEvent: handleGestureEvent,
            registerEvent: registerEvent
        }
    })(jQuery, DevExpress);
    /*! Module core, file ko.events.js */
    (function($, DX, undefined) {
        if (!DX.support.hasKo)
            return;
        var ko = window.ko,
            events = DX.ui.events;
        var originalRegisterEvent = events.registerEvent;
        var registerKoEvent = function(eventName, eventObject) {
                originalRegisterEvent(eventName, eventObject);
                var koBindingEventName = events.addNamespace(eventName, eventName + "Binding");
                ko.bindingHandlers[eventName] = {update: function(element, valueAccessor, allBindingsAccessor, viewModel) {
                        var $element = $(element),
                            unwrappedValue = ko.utils.unwrapObservable(valueAccessor()),
                            eventSource = unwrappedValue.execute ? unwrappedValue.execute : unwrappedValue;
                        $element.off(koBindingEventName).on(koBindingEventName, $.isPlainObject(unwrappedValue) ? unwrappedValue : {}, function(e) {
                            eventSource(viewModel, e)
                        })
                    }}
            };
        $.extend(events, {registerEvent: registerKoEvent})
    })(jQuery, DevExpress);
    /*! Module core, file ng.events.js */
    (function($, DX, undefined) {
        if (!DX.support.hasNg)
            return;
        var events = DX.ui.events;
        var originalRegisterEvent = events.registerEvent;
        var registerNgEvent = function(eventName, eventObject) {
                originalRegisterEvent(eventName, eventObject);
                var ngEventName = eventName.slice(0, 2) + eventName.charAt(2).toUpperCase() + eventName.slice(3);
                DX.ng.module.directive(ngEventName, ['$parse', function($parse) {
                        return function(scope, element, attr) {
                                var attrValue = $.trim(attr[ngEventName]),
                                    handler,
                                    eventOptions = {};
                                if (attrValue.charAt(0) === "{") {
                                    eventOptions = scope.$eval(attrValue);
                                    handler = $parse(eventOptions.execute)
                                }
                                else
                                    handler = $parse(attr[ngEventName]);
                                element.on(eventName, eventOptions, function(e) {
                                    scope.$apply(function() {
                                        handler(scope, {$event: e})
                                    })
                                })
                            }
                    }])
            };
        $.extend(events, {registerEvent: registerNgEvent})
    })(jQuery, DevExpress);
    /*! Module core, file ui.hierarchicalKeyDownProcessor.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            events = ui.events;
        ui.HierarchicalKeyDownProcessor = DX.Class.inherit({
            _keydown: events.addNamespace("keydown", "HierarchicalKeyDownProcessor"),
            codes: {
                "9": "tab",
                "13": "enter",
                "27": "escape",
                "33": "pageUp",
                "34": "pageDown",
                "37": "leftArrow",
                "38": "upArrow",
                "39": "rightArrow",
                "40": "downArrow",
                "32": "space",
                "70": "F",
                "65": "A"
            },
            ctor: function(options) {
                var _this = this;
                options = options || {};
                if (options.element)
                    this._element = $(options.element);
                this._handler = options.handler;
                this._context = options.context;
                this._childProcessors = [];
                if (this._element)
                    this._element.on(this._keydown, function(e) {
                        _this.process(e)
                    })
            },
            dispose: function() {
                if (this._element)
                    this._element.off(this._keydown);
                this._element = undefined;
                this._handler = undefined;
                this._context = undefined;
                this._childProcessors = undefined
            },
            attachChildProcessor: function() {
                var childProcessor = new ui.HierarchicalKeyDownProcessor;
                this._childProcessors.push(childProcessor);
                return childProcessor
            },
            reinitialize: function(childHandler, childContext) {
                this._context = childContext;
                this._handler = childHandler;
                return this
            },
            process: function(e) {
                var args = {
                        key: this.codes[e.which],
                        ctrl: e.ctrlKey,
                        shift: e.shiftKey,
                        originalEvent: e
                    };
                if (this.codes[e.which] && this._handler && this._handler.call(this._context, args))
                    $.each(this._childProcessors, function(index, childProcessor) {
                        childProcessor.process(e)
                    })
            }
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.dialog.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            utils = DX.utils;
        var DEFAULT_BUTTON = {
                text: "OK",
                clickAction: function() {
                    return true
                }
            };
        var DX_DIALOG_CLASSNAME = "dx-dialog",
            DX_DIALOG_WRAPPER_CLASSNAME = DX_DIALOG_CLASSNAME + "-wrapper",
            DX_DIALOG_ROOT_CLASSNAME = DX_DIALOG_CLASSNAME + "-root",
            DX_DIALOG_CONTENT_CLASSNAME = DX_DIALOG_CLASSNAME + "-content",
            DX_DIALOG_MESSAGE_CLASSNAME = DX_DIALOG_CLASSNAME + "-message",
            DX_DIALOG_BUTTONS_CLASSNAME = DX_DIALOG_CLASSNAME + "-buttons",
            DX_DIALOG_BUTTON_CLASSNAME = DX_DIALOG_CLASSNAME + "-button";
        var FakeDialogComponent = DX.Component.inherit({
                NAME: "dxDialog",
                ctor: function(element, options) {
                    this.callBase(options)
                },
                _defaultOptionsRules: function() {
                    return this.callBase().slice(0).concat([{
                                device: [{platform: "ios"}, {platform: "ios7"}],
                                options: {width: 276}
                            }, {
                                device: {platform: "android"},
                                options: {
                                    lWidth: "60%",
                                    pWidth: "80%"
                                }
                            }, {
                                device: {
                                    platform: "win8",
                                    phone: false
                                },
                                options: {width: function() {
                                        return $(window).width()
                                    }}
                            }, {
                                device: {
                                    platform: "win8",
                                    phone: true
                                },
                                options: {position: {
                                        my: "top center",
                                        at: "top center",
                                        of: window,
                                        offset: "0 0"
                                    }}
                            }])
                }
            });
        var dialog = function(options) {
                var that = this,
                    result;
                if (!ui.dxPopup)
                    throw new Error("DevExpress.ui.dxPopup required");
                var deferred = $.Deferred();
                var defaultOptions = (new FakeDialogComponent).option();
                options = $.extend(defaultOptions, options);
                var $holder = $(DX.overlayTargetContainer());
                var $element = $("<div>").addClass(DX_DIALOG_CLASSNAME).appendTo($holder);
                var $message = $("<div>").addClass(DX_DIALOG_MESSAGE_CLASSNAME).html(String(options.message));
                var $buttons = $("<div>").addClass(DX_DIALOG_BUTTONS_CLASSNAME);
                var popupInstance = $element.dxPopup({
                        title: options.title || that.title,
                        height: "auto",
                        width: function() {
                            var isPortrait = $(window).height() > $(window).width(),
                                key = (isPortrait ? "p" : "l") + "Width",
                                widthOption = options.hasOwnProperty(key) ? options[key] : options["width"];
                            return $.isFunction(widthOption) ? widthOption() : widthOption
                        },
                        contentReadyAction: function() {
                            popupInstance.content().addClass(DX_DIALOG_CONTENT_CLASSNAME).append($message).append($buttons)
                        },
                        animation: {
                            show: {
                                type: "pop",
                                duration: 400
                            },
                            hide: {
                                type: "pop",
                                duration: 400,
                                to: {
                                    opacity: 0,
                                    scale: 0
                                },
                                from: {
                                    opacity: 1,
                                    scale: 1
                                }
                            }
                        },
                        rtlEnabled: DX.rtlEnabled
                    }).data("dxPopup");
                popupInstance._wrapper().addClass(DX_DIALOG_WRAPPER_CLASSNAME);
                if (options.position)
                    popupInstance.option("position", options.position);
                $.each(options.buttons || [DEFAULT_BUTTON], function() {
                    var button = $("<div>").addClass(DX_DIALOG_BUTTON_CLASSNAME).appendTo($buttons);
                    var action = new DX.Action(this.clickAction, {context: popupInstance});
                    button.dxButton($.extend(this, {clickAction: function() {
                            result = action.execute(arguments);
                            hide()
                        }}))
                });
                popupInstance._wrapper().addClass(DX_DIALOG_ROOT_CLASSNAME);
                function show() {
                    popupInstance.show();
                    utils.resetActiveElement();
                    return deferred.promise()
                }
                function hide(value) {
                    popupInstance.hide().done(function() {
                        popupInstance._element().remove()
                    });
                    deferred.resolve(result || value)
                }
                return {
                        show: show,
                        hide: hide
                    }
            };
        var alert = function(message, title) {
                var dialogInstance,
                    options = $.isPlainObject(message) ? message : {
                        title: title,
                        message: message
                    };
                dialogInstance = ui.dialog.custom(options);
                return dialogInstance.show()
            };
        var confirm = function(message, title) {
                var dialogInstance,
                    options = $.isPlainObject(message) ? message : {
                        title: title,
                        message: message,
                        buttons: [{
                                text: Globalize.localize("Yes"),
                                clickAction: function() {
                                    return true
                                }
                            }, {
                                text: Globalize.localize("No"),
                                clickAction: function() {
                                    return false
                                }
                            }]
                    };
                dialogInstance = ui.dialog.custom(options);
                return dialogInstance.show()
            };
        var notify = function(message, type, displayTime) {
                var options = $.isPlainObject(message) ? message : {message: message};
                if (!ui.dxToast) {
                    alert(options.message);
                    return
                }
                var userHiddenAction = options.hiddenAction;
                $.extend(options, {
                    type: type,
                    displayTime: displayTime,
                    hiddenAction: function(args) {
                        args.element.remove();
                        new DX.Action(userHiddenAction, {context: args.model}).execute(arguments)
                    }
                });
                $("<div>").appendTo(DX.overlayTargetContainer()).dxToast(options).dxToast("instance").show()
            };
        $.extend(ui, {
            notify: notify,
            dialog: {
                custom: dialog,
                alert: alert,
                confirm: confirm,
                FakeDialogComponent: FakeDialogComponent
            }
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.dataHelper.js */
    (function($, DX, undefined) {
        var data = DX.data;
        var DATA_SOURCE_OPTIONS_METHOD = "_dataSourceOptions",
            DATA_SOURCE_CHANGED_METHOD = "_handleDataSourceChanged",
            DATA_SOURCE_LOAD_ERROR_METHOD = "_handleDataSourceLoadError",
            DATA_SOURCE_LOADING_CHANGED_METHOD = "_handleDataSourceLoadingChanged";
        DX.ui.DataHelperMixin = {
            ctor: function() {
                this.disposing.add(function() {
                    this._disposeDataSource()
                })
            },
            _refreshDataSource: function() {
                this._initDataSource();
                this._loadDataSource()
            },
            _initDataSource: function() {
                var dataSourceOptions = this.option("dataSource"),
                    widgetDataSourceOptions,
                    dataSourceType;
                this._disposeDataSource();
                if (dataSourceOptions) {
                    if (dataSourceOptions instanceof data.DataSource) {
                        this._isSharedDataSource = true;
                        this._dataSource = dataSourceOptions
                    }
                    else {
                        widgetDataSourceOptions = DATA_SOURCE_OPTIONS_METHOD in this ? this[DATA_SOURCE_OPTIONS_METHOD]() : {};
                        dataSourceType = this._dataSourceType ? this._dataSourceType() : data.DataSource;
                        this._dataSource = new dataSourceType($.extend(true, {}, widgetDataSourceOptions, data.utils.normalizeDataSourceOptions(dataSourceOptions)))
                    }
                    this._addDataSourceHandlers()
                }
            },
            _addDataSourceHandlers: function() {
                if (DATA_SOURCE_CHANGED_METHOD in this)
                    this._addDataSourceChangeHandler();
                if (DATA_SOURCE_LOAD_ERROR_METHOD in this)
                    this._addDataSourceLoadErrorHandler();
                if (DATA_SOURCE_LOADING_CHANGED_METHOD in this)
                    this._addDataSourceLoadingChangedHandler()
            },
            _addDataSourceChangeHandler: function() {
                var that = this,
                    dataSource = this._dataSource;
                this._dataSourceChangedHandler = function() {
                    that[DATA_SOURCE_CHANGED_METHOD](dataSource.items())
                };
                dataSource.changed.add(this._dataSourceChangedHandler)
            },
            _addDataSourceLoadErrorHandler: function() {
                this._dataSourceLoadErrorHandler = $.proxy(this[DATA_SOURCE_LOAD_ERROR_METHOD], this);
                this._dataSource.loadError.add(this._dataSourceLoadErrorHandler)
            },
            _addDataSourceLoadingChangedHandler: function() {
                this._dataSourceLoadingChangedHandler = $.proxy(this[DATA_SOURCE_LOADING_CHANGED_METHOD], this);
                this._dataSource.loadingChanged.add(this._dataSourceLoadingChangedHandler)
            },
            _loadDataSource: function() {
                if (this._dataSource) {
                    var dataSource = this._dataSource;
                    if (dataSource.isLoaded())
                        this._dataSourceChangedHandler();
                    else
                        dataSource.load()
                }
            },
            _loadSingle: function(key, value) {
                key = key === "this" ? this._dataSource.key() || "this" : key;
                return this._dataSource.loadSingle(key, value)
            },
            _disposeDataSource: function() {
                if (this._dataSource) {
                    if (this._isSharedDataSource) {
                        delete this._isSharedDataSource;
                        this._dataSource.changed.remove(this._dataSourceChangedHandler);
                        this._dataSource.loadError.remove(this._dataSourceLoadErrorHandler);
                        this._dataSource.loadingChanged.remove(this._dataSourceLoadingChangedHandler)
                    }
                    else
                        this._dataSource.dispose();
                    delete this._dataSource;
                    delete this._dataSourceChangedHandler;
                    delete this._dataSourceLoadErrorHandler;
                    delete this._dataSourceLoadingChangedHandler
                }
            }
        }
    })(jQuery, DevExpress);
    /*! Module core, file ui.events.mspointer.js */
    (function($, DX, undefined) {
        var POINTER_TYPE_MAP = {
                2: "touch",
                3: "pen",
                4: "mouse"
            };
        var pointerEventHook = {
                filter: function(event, originalEvent) {
                    var pointerType = originalEvent.pointerType;
                    if ($.isNumeric(pointerType))
                        event.pointerType = POINTER_TYPE_MAP[pointerType];
                    return event
                },
                props: $.event.mouseHooks.props.concat(["pointerId", "originalTarget", "namespace", "width", "height", "pressure", "result", "tiltX", "charCode", "tiltY", "detail", "isPrimary", "prevValue"])
            };
        $.each(["MSPointerDown", "MSPointerMove", "MSPointerUp", "MSPointerCancel", "MSPointerOver", "MSPointerOut", "MSPointerEnter", "MSPointerLeave", "pointerdown", "pointermove", "pointerup", "pointercancel", "pointerover", "pointerout", "pointerenter", "pointerleave"], function() {
            $.event.fixHooks[this] = pointerEventHook
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.events.touch.js */
    (function($, DX, undefined) {
        var touchEventHook = {
                filter: function(event, originalEvent) {
                    if (originalEvent.changedTouches.length)
                        $.each(["pageX", "pageY", "screenX", "screenY", "clientX", "clientY"], function() {
                            event[this] = originalEvent.changedTouches[0][this]
                        });
                    return event
                },
                props: $.event.mouseHooks.props.concat(["touches", "changedTouches", "targetTouches", "detail", "result", "namespace", "originalTarget", "charCode", "prevValue"])
            };
        $.each(["touchstart", "touchmove", "touchend", "touchcancel"], function() {
            $.event.fixHooks[this] = touchEventHook
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.events.pointer.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            support = DX.support,
            device = $.proxy(DX.devices.real, DX.devices),
            events = ui.events;
        var POINTER_EVENTS_NAMESPACE = "dxPointerEvents",
            MouseStrategyEventMap = {
                dxpointerdown: "mousedown",
                dxpointermove: "mousemove",
                dxpointerup: "mouseup",
                dxpointercancel: ""
            },
            TouchStrategyEventMap = {
                dxpointerdown: "touchstart",
                dxpointermove: "touchmove",
                dxpointerup: "touchend",
                dxpointercancel: "touchcancel"
            },
            PointerStrategyEventMap = {
                dxpointerdown: "pointerdown",
                dxpointermove: "pointermove",
                dxpointerup: "pointerup",
                dxpointercancel: "pointercancel"
            },
            MouseAndTouchStrategyEventMap = {
                dxpointerdown: "touchstart mousedown",
                dxpointermove: "touchmove mousemove",
                dxpointerup: "touchend mouseup",
                dxpointercancel: "touchcancel"
            };
        var eventMap = function() {
                if (support.touch && !(device().tablet || device().phone))
                    return MouseAndTouchStrategyEventMap;
                if (support.touch)
                    return TouchStrategyEventMap;
                return MouseStrategyEventMap
            }();
        var skipTouchWithSameIdentifier = function(pointerEvent) {
                return device().platform === "ios" && (pointerEvent === "dxpointerdown" || pointerEvent === "dxpointerup")
            };
        var SingleEventStrategy = DX.Class.inherit({
                ctor: function(eventName, originalEvents) {
                    this._eventName = eventName;
                    this._eventNamespace = [POINTER_EVENTS_NAMESPACE, ".", this._eventName].join("");
                    this._originalEvents = originalEvents;
                    this._pointerId = 0;
                    this._handlerCount = 0
                },
                _handler: function(e) {
                    if (this._eventName === "dxpointerdown")
                        $(e.target).data("dxGestureEvent", null);
                    if (events.isTouchEvent(e) && skipTouchWithSameIdentifier(this._eventName)) {
                        var touch = e.changedTouches[0];
                        if (this._pointerId === touch.identifier && this._pointerId !== 0)
                            return;
                        this._pointerId = touch.identifier
                    }
                    return events.fireEvent({
                            type: this._eventName,
                            pointerType: events.eventSource(e),
                            originalEvent: e
                        })
                },
                setup: function() {
                    if (this._handlerCount > 0)
                        return;
                    $(document).on(events.addNamespace(this._originalEvents, this._eventNamespace), $.proxy(this._handler, this))
                },
                add: function() {
                    this._handlerCount++
                },
                remove: function() {
                    this._handlerCount--
                },
                teardown: function() {
                    if (this._handlerCount)
                        return;
                    $(document).off("." + this._eventNamespace)
                },
                dispose: function() {
                    $(document).off("." + this._eventNamespace)
                }
            });
        var MultiEventStrategy = SingleEventStrategy.inherit({
                EVENT_LOCK_TIMEOUT: 100,
                _handler: function(e) {
                    if (events.isTouchEvent(e))
                        this._skipNextEvents = true;
                    if (events.isMouseEvent(e) && this._mouseLocked)
                        return;
                    if (events.isMouseEvent(e) && this._skipNextEvents) {
                        this._skipNextEvents = false;
                        this._mouseLocked = true;
                        clearTimeout(this._unlockMouseTimer);
                        this._unlockMouseTimer = setTimeout($.proxy(function() {
                            this._mouseLocked = false
                        }, this), this.EVENT_LOCK_TIMEOUT);
                        return
                    }
                    return this.callBase(e)
                },
                dispose: function() {
                    this.callBase();
                    this._skipNextEvents = false;
                    this._mouseLocked = false;
                    clearTimeout(this._unlockMouseTimer)
                }
            });
        var getStrategy = function() {
                return eventMap === MouseAndTouchStrategyEventMap ? MultiEventStrategy : SingleEventStrategy
            };
        $.each(eventMap, function(pointerEvent, originalEvents) {
            var Strategy = getStrategy();
            events.registerEvent(pointerEvent, new Strategy(pointerEvent, originalEvents))
        });
        DX.ui.events.__internals = DX.ui.events.__internals || {};
        $.extend(DX.ui.events.__internals, {
            SingleEventStrategy: SingleEventStrategy,
            MultiEventStrategy: MultiEventStrategy,
            MouseStrategyEventMap: MouseStrategyEventMap,
            TouchStrategyEventMap: TouchStrategyEventMap,
            PointerStrategyEventMap: PointerStrategyEventMap,
            MouseAndTouchStrategyEventMap: MouseAndTouchStrategyEventMap,
            eventMap: eventMap
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.events.click.js */
    (function($, DX, wnd, undefined) {
        var ua = navigator.userAgent,
            screen = wnd.screen,
            ui = DX.ui,
            utils = DX.utils,
            events = ui.events,
            support = DX.support,
            device = $.proxy(DX.devices.real, DX.devices),
            EVENTS_NAME_SPACE = "dxSpecialEvents",
            CLICK_NAME_SPACE = "dxClick" + EVENTS_NAME_SPACE,
            CLICK_EVENT_NAME = "dxclick",
            SCROLLABLE_PARENT_DATA_KEY = "dxClickScrollableParent",
            SCROLLABLE_PARENT_SCROLL_OFFSET_DATA_KEY = "dxClickScrollableParentOffset",
            preferNativeClick = function() {
                var iPhone4SAndElder = device().deviceType === "phone" && screen.height <= 480,
                    iPad2AndElder = device().deviceType === "tablet" && wnd.devicePixelRatio < 2,
                    IOS7AndNewer = device().platform === "ios" && device().version[0] > 6;
                return IOS7AndNewer && (iPhone4SAndElder || iPad2AndElder)
            }(),
            useNativeClick = function() {
                if (!support.touch)
                    return true;
                var chromeInfo = ua.match(/Chrome\/([0-9]+)/) || [],
                    chrome = !!chromeInfo[0],
                    chromeVersion = ~~chromeInfo[1],
                    android = device().platform === "android";
                if (chrome)
                    if (android) {
                        if ($("meta[name=viewport][content*='width=device-width']").length)
                            return false;
                        if (chromeVersion > 31 && wnd.innerWidth <= screen.width)
                            return true;
                        if ($("meta[name=viewport][content*='user-scalable=no']").length)
                            return true
                    }
                    else
                        return true;
                return false
            }();
        var SimulatedStrategy = DX.Class.inherit({
                TOUCH_BOUNDARY: 10,
                ctor: function() {
                    this._startX = 0;
                    this._startY = 0;
                    this._handlerCount = 0;
                    this._target = null
                },
                _touchWasMoved: function(e) {
                    var boundary = this.TOUCH_BOUNDARY;
                    return Math.abs(e.pageX - this._startX) > boundary || Math.abs(e.pageY - this._startY) > boundary
                },
                _getClosestScrollable: function($element) {
                    var $scrollParent = $();
                    if ($element.data(SCROLLABLE_PARENT_DATA_KEY))
                        $scrollParent = $element.data(SCROLLABLE_PARENT_DATA_KEY);
                    else {
                        var $current = $element;
                        while ($current.length) {
                            if ($current[0].scrollHeight - $current[0].offsetHeight > 1) {
                                $scrollParent = $current;
                                $element.data(SCROLLABLE_PARENT_DATA_KEY, $scrollParent);
                                break
                            }
                            $current = $current.parent()
                        }
                    }
                    return $scrollParent
                },
                _saveClosestScrollableOffset: function($element) {
                    var $scrollable = this._getClosestScrollable($element);
                    if ($scrollable.length)
                        $element.data(SCROLLABLE_PARENT_SCROLL_OFFSET_DATA_KEY, $scrollable.scrollTop())
                },
                _closestScrollableWasMoved: function($element) {
                    var $scrollable = $element.data(SCROLLABLE_PARENT_DATA_KEY);
                    return $scrollable && $scrollable.scrollTop() !== $element.data(SCROLLABLE_PARENT_SCROLL_OFFSET_DATA_KEY)
                },
                _hasClosestScrollable: function($element) {
                    var $scrollable = this._getClosestScrollable($element);
                    if (!$scrollable.length)
                        return false;
                    if ($scrollable.is("body"))
                        return false;
                    if ($scrollable === window)
                        return false;
                    if ($scrollable.css("overflow") === "hidden")
                        return false;
                    return true
                },
                _handleStart: function(e) {
                    this._reset();
                    if (events.isMouseEvent(e) && e.which !== 1)
                        return;
                    this._saveClosestScrollableOffset($(e.target));
                    this._target = e.target;
                    this._startX = e.pageX;
                    this._startY = e.pageY
                },
                _handleEnd: function(e) {
                    var $target = $(e.target);
                    if (!$target.is(this._target) || this._touchWasMoved(e))
                        return;
                    if (this._nativeClickShouldBeUsed($target) || this._closestScrollableWasMoved($target))
                        return;
                    var targetIsInput = $target.is("input, textarea");
                    if (!targetIsInput && !e.dxPreventBlur)
                        utils.resetActiveElement();
                    this._fireClickEvent(e)
                },
                _handleCancel: function(e) {
                    this._reset()
                },
                _reset: function() {
                    this._target = null
                },
                _handleClick: function(e) {
                    var $target = $(e.target);
                    if ($target.is(this._target)) {
                        if (this._nativeClickShouldBeUsed($target))
                            this._fireClickEvent(e)
                    }
                    else if ($target.is("input, textarea"))
                        utils.resetActiveElement()
                },
                _nativeClickShouldBeUsed: function($target) {
                    return preferNativeClick && this._hasClosestScrollable($target)
                },
                _fireClickEvent: function(e) {
                    if (events.handleGestureEvent(e, CLICK_EVENT_NAME))
                        events.fireEvent({
                            type: CLICK_EVENT_NAME,
                            originalEvent: e
                        })
                },
                _makeElementClickable: function($element) {
                    if (!$element.attr("onclick"))
                        $element.attr("onclick", "void(0)")
                },
                setup: function(element) {
                    var $element = $(element);
                    this._makeElementClickable($element);
                    if (this._handlerCount > 0)
                        return;
                    var $doc = $(document).on(events.addNamespace("dxpointerdown", CLICK_NAME_SPACE), $.proxy(this._handleStart, this)).on(events.addNamespace("dxpointerup", CLICK_NAME_SPACE), $.proxy(this._handleEnd, this)).on(events.addNamespace("dxpointercancel", CLICK_NAME_SPACE), $.proxy(this._handleCancel, this)).on(events.addNamespace("click", CLICK_NAME_SPACE), $.proxy(this._handleClick, this))
                },
                add: function() {
                    this._handlerCount++
                },
                remove: function() {
                    this._handlerCount--
                },
                teardown: function(element) {
                    if (this._handlerCount)
                        return;
                    $(element).off("." + CLICK_NAME_SPACE);
                    this.dispose()
                },
                dispose: function() {
                    $(document).off("." + CLICK_NAME_SPACE)
                }
            });
        var NativeStrategy = DX.Class.inherit({
                bindType: "click",
                delegateType: "click",
                handle: function(element, event) {
                    event.type = "dxclick";
                    if (events.handleGestureEvent(event, CLICK_EVENT_NAME))
                        return event.handleObj.handler.call(element, event)
                }
            });
        events.registerEvent(CLICK_EVENT_NAME, new(useNativeClick ? NativeStrategy : SimulatedStrategy));
        DX.ui.events.__internals = DX.ui.events.__internals || {};
        $.extend(DX.ui.events.__internals, {
            NativeClickStrategy: NativeStrategy,
            SimulatedClickStrategy: SimulatedStrategy,
            preferNativeClickAccessor: function(value) {
                if (!arguments.length)
                    return preferNativeClick;
                preferNativeClick = value
            }
        })
    })(jQuery, DevExpress, window);
    /*! Module core, file ui.events.hold.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            events = ui.events,
            jqSpecialEvent = $.event.special,
            EVENTS_NAME_SPACE = "dxSpecialEvents",
            HOLD_NAME_SPACE = "dxHold",
            HOLD_EVENT_NAME = "dxhold",
            HOLD_TIMER_DATA_KEY = EVENTS_NAME_SPACE + "HoldTimer";
        var Hold = DX.Class.inherit({
                HOLD_TIMEOUT: 750,
                TOUCH_BOUNDARY: 5,
                _startX: 0,
                _startY: 0,
                _touchWasMoved: function(e) {
                    var boundary = this.TOUCH_BOUNDARY;
                    return Math.abs(e.pageX - this._startX) > boundary || Math.abs(e.pageY - this._startY) > boundary
                },
                setup: function(element, data) {
                    var $target,
                        holdInited = false;
                    var handleStart = function(e) {
                            if (holdInited)
                                return;
                            $target = $(e.target);
                            holdInited = true;
                            if ($target.data(HOLD_TIMER_DATA_KEY))
                                return;
                            this._startX = e.pageX;
                            this._startY = e.pageY;
                            var holdTimeout = data && "timeout" in data ? data.timeout : this.HOLD_TIMEOUT;
                            var holdTimer = setTimeout(function() {
                                    $target.removeData(HOLD_TIMER_DATA_KEY);
                                    if (events.handleGestureEvent(e, HOLD_EVENT_NAME))
                                        events.fireEvent({
                                            type: HOLD_EVENT_NAME,
                                            originalEvent: e
                                        })
                                }, holdTimeout);
                            $target.data(HOLD_TIMER_DATA_KEY, holdTimer)
                        };
                    var handleMove = function(e) {
                            if (!this._touchWasMoved(e))
                                return;
                            handleEnd()
                        };
                    var handleEnd = function() {
                            if ($target) {
                                clearTimeout($target.data(HOLD_TIMER_DATA_KEY));
                                $target.removeData(HOLD_TIMER_DATA_KEY);
                                $target = null;
                                holdInited = false
                            }
                        };
                    $(element).on(events.addNamespace("dxpointerdown", HOLD_NAME_SPACE), $.proxy(handleStart, this)).on(events.addNamespace("dxpointermove", HOLD_NAME_SPACE), $.proxy(handleMove, this)).on(events.addNamespace("dxpointerup", HOLD_NAME_SPACE), $.proxy(handleEnd, this))
                },
                teardown: function(element) {
                    element = $(element);
                    clearTimeout(element.data(HOLD_TIMER_DATA_KEY));
                    element.removeData(HOLD_TIMER_DATA_KEY).off("." + HOLD_NAME_SPACE)
                }
            });
        events.registerEvent(HOLD_EVENT_NAME, new Hold)
    })(jQuery, DevExpress);
    /*! Module core, file ui.events.hover.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            events = ui.events;
        var HOVER_EVENTS_NAMESPACE = "dxSpecialEvents",
            HOVER_START = "dxhoverstart",
            HOVER_END = "dxhoverend",
            MOUSE_ENTER = "mouseenter",
            MOUSE_LEAVE = "mouseleave",
            TOUCH_START = "dxpointerdown";
        var HoverStart = DX.Class.inherit({
                ctor: function() {
                    this._wasTouch = false;
                    this._selector = undefined
                },
                setup: function(element, data) {
                    var $element = $(element);
                    this._selector = data ? data.selector : undefined;
                    if (!!$element.data("dxHoverStart"))
                        return;
                    $element.data("dxHoverStart", true).on(events.addNamespace(MOUSE_ENTER, HOVER_EVENTS_NAMESPACE), this._selector, $.proxy(this._handleEnter, this)).on(events.addNamespace(TOUCH_START, HOVER_EVENTS_NAMESPACE), $.proxy(this._handleTouchStart, this))
                },
                _handleEnter: function(e) {
                    if (!this._wasTouch)
                        events.fireEvent({
                            type: HOVER_START,
                            originalEvent: e
                        })
                },
                _handleTouchStart: function(e) {
                    this._wasTouch = events.isTouchEvent(e)
                },
                teardown: function(element) {
                    $(element).data("dxHoverStart", false).off(events.addNamespace(MOUSE_ENTER, HOVER_EVENTS_NAMESPACE), this._selector).off(events.addNamespace(TOUCH_START, HOVER_EVENTS_NAMESPACE))
                }
            });
        var HoverEnd = DX.Class.inherit({
                ctor: function() {
                    this._wasTouch = false;
                    this._selector = undefined
                },
                setup: function(element, data) {
                    var $element = $(element);
                    this._selector = data ? data.selector : undefined;
                    if (!!$element.data("dxHoverEnd"))
                        return;
                    $element.data("dxHoverEnd", true).on(events.addNamespace(MOUSE_LEAVE, HOVER_EVENTS_NAMESPACE), this._selector, $.proxy(this._handleLeave, this)).on(events.addNamespace(TOUCH_START, HOVER_EVENTS_NAMESPACE), $.proxy(this._handleTouchStart, this))
                },
                _handleLeave: function(e) {
                    if (!this._wasTouch)
                        events.fireEvent({
                            type: HOVER_END,
                            originalEvent: e
                        })
                },
                _handleTouchStart: function(e) {
                    this._wasTouch = events.isTouchEvent(e)
                },
                teardown: function(element) {
                    $(element).data("dxHoverEnd", false).off(events.addNamespace(MOUSE_LEAVE, HOVER_EVENTS_NAMESPACE), this._selector).off(events.addNamespace(TOUCH_START, HOVER_EVENTS_NAMESPACE))
                }
            });
        events.registerEvent(HOVER_START, new HoverStart);
        events.registerEvent(HOVER_END, new HoverEnd)
    })(jQuery, DevExpress);
    /*! Module core, file ui.events.wheel.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            events = ui.events;
        var EVENT_NAME = "dxmousewheel",
            EVENT_NAMESPACE = "dxWheel";
        var WHEEL_DISTANCE = 10;
        $.event.fixHooks["wheel"] = $.event.mouseHooks;
        var wheelEvent = document.onmousewheel !== undefined ? "mousewheel" : "wheel";
        var wheel = {
                setup: function(element, data) {
                    var $element = $(element);
                    $element.on(events.addNamespace(wheelEvent, EVENT_NAMESPACE), $.proxy(wheel._handleWheel, wheel))
                },
                teardown: function(element) {
                    var $element = $(element);
                    $element.off("." + EVENT_NAMESPACE)
                },
                _handleWheel: function(e) {
                    $(e.target).data("dxGestureEvent", null);
                    var delta = this._getWheelDelta(e.originalEvent);
                    events.fireEvent({
                        type: EVENT_NAME,
                        originalEvent: e,
                        delta: delta
                    });
                    e.stopPropagation()
                },
                _getWheelDelta: function(event) {
                    return event.wheelDelta / 60 || -event.deltaY
                }
            };
        events.registerEvent(EVENT_NAME, wheel)
    })(jQuery, DevExpress);
    /*! Module core, file ui.gestureEmitter.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            events = ui.events;
        var GestureEmitter = DX.Class.inherit({
                ctor: function(element) {
                    this._$element = $(element);
                    this._cancelCallback = $.Callbacks()
                },
                getElement: function() {
                    return this._$element
                },
                getDirection: function() {
                    return this.direction
                },
                validate: function(e) {
                    return e.type !== "dxmousewheel"
                },
                configurate: function(data) {
                    $.extend(this, data)
                },
                addCancelCallback: function(callback) {
                    this._cancelCallback.add(callback)
                },
                removeCancelCallback: function() {
                    this._cancelCallback.empty()
                },
                init: $.noop,
                start: $.noop,
                move: $.noop,
                end: $.noop,
                cancel: $.noop,
                wheel: $.noop,
                _fireEvent: function(eventName, event, params) {
                    var eventData = {
                            type: eventName,
                            originalEvent: event,
                            target: this.getElement().get(0)
                        };
                    event = events.fireEvent($.extend(eventData, params));
                    if (event.cancel)
                        this._cancel(event);
                    return event
                },
                _cancel: function(e) {
                    this._cancelCallback.fire()
                }
            });
        var gestureEmitters = [];
        $.extend(ui, {
            GestureEmitter: GestureEmitter,
            gestureEmitters: gestureEmitters
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.scrollEmitter.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            events = ui.events;
        var SCROLL_INIT_EVENT = "dxscrollinit",
            SCROLL_START_EVENT = "dxscrollstart",
            SCROLL_MOVE_EVENT = "dxscroll",
            SCROLL_END_EVENT = "dxscrollend",
            SCROLL_STOP_EVENT = "dxscrollstop",
            SCROLL_CANCEL_EVENT = "dxscrollcancel",
            SCROLL_WHEEL_EVENT = "dxscrollwheel",
            INERTIA_TIMEOUT = 100,
            VELOCITY_CALC_TIMEOUT = 200,
            FRAME_DURATION = Math.round(1000 / 60);
        var ScrollEmitter = ui.GestureEmitter.inherit({
                ctor: function(element) {
                    this.callBase(element);
                    this.direction = "vertical"
                },
                init: function(e) {
                    this._savedEventData = this._prevEventData = events.eventData(e);
                    this._fireEvent(SCROLL_INIT_EVENT, e)
                },
                start: function(e) {
                    this._fireEvent(SCROLL_START_EVENT, e, {delta: events.eventDelta(this._prevEventData, events.eventData(e))})
                },
                move: function(e) {
                    var currentEventData = events.eventData(e);
                    this._fireEvent(SCROLL_MOVE_EVENT, e, {delta: events.eventDelta(this._prevEventData, currentEventData)});
                    var eventDelta = events.eventDelta(this._savedEventData, currentEventData);
                    if (eventDelta.time > VELOCITY_CALC_TIMEOUT)
                        this._savedEventData = this._prevEventData;
                    this._prevEventData = currentEventData
                },
                end: function(e) {
                    var endEventDelta = events.eventDelta(this._prevEventData, events.eventData(e));
                    var velocity = {
                            x: 0,
                            y: 0
                        };
                    if (endEventDelta.time < INERTIA_TIMEOUT) {
                        var deltaEventData = events.eventDelta(this._savedEventData, this._prevEventData);
                        velocity = {
                            x: deltaEventData.x * FRAME_DURATION / deltaEventData.time,
                            y: deltaEventData.y * FRAME_DURATION / deltaEventData.time
                        }
                    }
                    this._fireEvent(SCROLL_END_EVENT, e, {velocity: velocity})
                },
                stop: function(e) {
                    this._fireEvent(SCROLL_STOP_EVENT, e)
                },
                cancel: function(e) {
                    this._fireEvent(SCROLL_CANCEL_EVENT, e)
                },
                wheel: function(e) {
                    this._fireEvent(SCROLL_WHEEL_EVENT, e, {delta: e.delta})
                }
            });
        ui.gestureEmitters.push({
            emitter: ScrollEmitter,
            events: [SCROLL_INIT_EVENT, SCROLL_START_EVENT, SCROLL_MOVE_EVENT, SCROLL_END_EVENT, SCROLL_STOP_EVENT, SCROLL_CANCEL_EVENT, SCROLL_WHEEL_EVENT]
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.swipeEmitter.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            utils = DX.utils,
            events = ui.events,
            SWIPE_START_EVENT = "dxswipestart",
            SWIPE_EVENT = "dxswipe",
            SWIPE_END_EVENT = "dxswipeend";
        var HorizontalStrategy = {
                defaultItemSizeFunc: function() {
                    return this.getElement().width()
                },
                getBounds: function() {
                    return [this._maxLeftOffset, this._maxRightOffset]
                },
                calcOffsetRatio: function(e) {
                    var endEventData = events.eventData(e);
                    return (endEventData.x - (this._startEventData && this._startEventData.x || 0)) / this._itemSizeFunc().call(this, e)
                },
                isFastSwipe: function(e) {
                    var endEventData = events.eventData(e);
                    return this.FAST_SWIPE_SPEED_LIMIT * Math.abs(endEventData.x - this._tickData.x) >= endEventData.time - this._tickData.time
                }
            };
        var VerticalStrategy = {
                defaultItemSizeFunc: function() {
                    return this.getElement().height()
                },
                getBounds: function() {
                    return [this._maxTopOffset, this._maxBottomOffset]
                },
                calcOffsetRatio: function(e) {
                    var endEventData = events.eventData(e);
                    return (endEventData.y - (this._startEventData && this._startEventData.y || 0)) / this._itemSizeFunc().call(this, e)
                },
                isFastSwipe: function(e) {
                    var endEventData = events.eventData(e);
                    return this.FAST_SWIPE_SPEED_LIMIT * Math.abs(endEventData.y - this._tickData.y) >= endEventData.time - this._tickData.time
                }
            };
        var STRATEGIES = {
                horizontal: HorizontalStrategy,
                vertical: VerticalStrategy
            };
        var SwipeEmitter = ui.GestureEmitter.inherit({
                TICK_INTERVAL: 300,
                FAST_SWIPE_SPEED_LIMIT: 5,
                ctor: function(element) {
                    this.callBase(element);
                    this.direction = "horizontal";
                    this.elastic = true
                },
                _getStrategy: function() {
                    return STRATEGIES[this.direction]
                },
                _defaultItemSizeFunc: function() {
                    return this._getStrategy().defaultItemSizeFunc.call(this)
                },
                _itemSizeFunc: function() {
                    return this.itemSizeFunc || this._defaultItemSizeFunc
                },
                init: function(e) {
                    this._startEventData = events.eventData(e);
                    this._tickData = {time: 0}
                },
                start: function(e) {
                    e = this._fireEvent(SWIPE_START_EVENT, e);
                    if (!e.cancel) {
                        this._maxLeftOffset = e.maxLeftOffset;
                        this._maxRightOffset = e.maxRightOffset;
                        this._maxTopOffset = e.maxTopOffset;
                        this._maxBottomOffset = e.maxBottomOffset
                    }
                },
                move: function(e) {
                    var strategy = this._getStrategy(),
                        moveEventData = events.eventData(e),
                        offset = strategy.calcOffsetRatio.call(this, e);
                    offset = this._fitOffset(offset, this.elastic);
                    if (moveEventData.time - this._tickData.time > this.TICK_INTERVAL)
                        this._tickData = moveEventData;
                    this._fireEvent(SWIPE_EVENT, e, {offset: offset});
                    e.preventDefault()
                },
                end: function(e) {
                    var strategy = this._getStrategy(),
                        offsetRatio = strategy.calcOffsetRatio.call(this, e),
                        isFast = strategy.isFastSwipe.call(this, e),
                        startOffset = offsetRatio,
                        targetOffset = this._calcTargetOffset(offsetRatio, isFast);
                    startOffset = this._fitOffset(startOffset, this.elastic);
                    targetOffset = this._fitOffset(targetOffset, false);
                    this._fireEvent(SWIPE_END_EVENT, e, {
                        offset: startOffset,
                        targetOffset: targetOffset
                    })
                },
                _fitOffset: function(offset, elastic) {
                    var strategy = this._getStrategy(),
                        bounds = strategy.getBounds.call(this);
                    if (offset < -bounds[0])
                        return elastic ? (-2 * bounds[0] + offset) / 3 : -bounds[0];
                    if (offset > bounds[1])
                        return elastic ? (2 * bounds[1] + offset) / 3 : bounds[1];
                    return offset
                },
                _calcTargetOffset: function(offsetRatio, isFast) {
                    var result;
                    if (isFast) {
                        result = Math.ceil(Math.abs(offsetRatio));
                        if (offsetRatio < 0)
                            result = -result
                    }
                    else
                        result = Math.round(offsetRatio);
                    return result
                }
            });
        ui.gestureEmitters.push({
            emitter: SwipeEmitter,
            events: [SWIPE_START_EVENT, SWIPE_EVENT, SWIPE_END_EVENT]
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.dragEmitter.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            utils = DX.utils,
            events = ui.events,
            wrapToArray = utils.wrapToArray;
        var DRAG_START_EVENT = "dxdragstart",
            DRAG_EVENT = "dxdrag",
            DRAG_END_EVENT = "dxdragend",
            DRAG_ENTER_EVENT = "dxdragenter",
            DRAG_LEAVE_EVENT = "dxdragleave",
            DROP_EVENT = "dxdrop";
        var knownDropTargets = [],
            knownDropTargetConfigs = [];
        var dropTargetRegistration = {
                setup: function(element, data) {
                    var knownDropTarget = $.inArray(element, knownDropTargets) !== -1;
                    if (!knownDropTarget) {
                        knownDropTargets.push(element);
                        knownDropTargetConfigs.push(data || {})
                    }
                },
                teardown: function(element, data) {
                    var elementEvents = $._data(element, "events"),
                        handlersCount = 0;
                    $.each([DRAG_ENTER_EVENT, DRAG_LEAVE_EVENT, DROP_EVENT], function(_, eventName) {
                        var eventHandlers = elementEvents[eventName];
                        if (eventHandlers)
                            handlersCount += eventHandlers.length
                    });
                    if (!handlersCount) {
                        var index = $.inArray(element, knownDropTargets);
                        knownDropTargets.splice(index, 1);
                        knownDropTargetConfigs.splice(index, 1)
                    }
                }
            };
        events.registerEvent(DRAG_ENTER_EVENT, dropTargetRegistration);
        events.registerEvent(DRAG_LEAVE_EVENT, dropTargetRegistration);
        events.registerEvent(DROP_EVENT, dropTargetRegistration);
        var getItemConfig = function($element) {
                var dropTargetIndex = $.inArray($element.get(0), knownDropTargets);
                return knownDropTargetConfigs[dropTargetIndex]
            };
        var getItemPosition = function($element) {
                var dropTargetConfig = getItemConfig($element);
                if (dropTargetConfig.itemPositionFunc)
                    return dropTargetConfig.itemPositionFunc();
                else
                    return $element.offset()
            };
        var getItemSize = function($element) {
                var dropTargetConfig = getItemConfig($element);
                if (dropTargetConfig.itemSizeFunc)
                    return dropTargetConfig.itemSizeFunc();
                else
                    return {
                            width: $element.width(),
                            height: $element.height()
                        }
            };
        var DragEmitter = ui.GestureEmitter.inherit({
                ctor: function(element) {
                    this.callBase(element);
                    this.direction = "both"
                },
                init: function(e) {
                    var eventData = events.eventData(e);
                    this._startEventData = eventData
                },
                start: function(e) {
                    e = this._fireEvent(DRAG_START_EVENT, e);
                    this._maxLeftOffset = e.maxLeftOffset;
                    this._maxRightOffset = e.maxRightOffset;
                    this._maxTopOffset = e.maxTopOffset;
                    this._maxBottomOffset = e.maxBottomOffset;
                    var dropTargets = wrapToArray(e.targetElements || knownDropTargets);
                    this._$dropTargetElements = $.map(dropTargets, function(element) {
                        return $(element)
                    })
                },
                move: function(e) {
                    var eventData = events.eventData(e),
                        dragOffset = this._calculateOffset(eventData);
                    this._fireEvent(DRAG_EVENT, e, {offset: dragOffset});
                    this._processDropTargets(e, dragOffset);
                    e.preventDefault()
                },
                _calculateOffset: function(eventData) {
                    return {
                            x: this._calculateXOffset(eventData),
                            y: this._calculateYOffset(eventData)
                        }
                },
                _calculateXOffset: function(eventData) {
                    if (this.direction !== "vertical") {
                        var offset = eventData.x - this._startEventData.x;
                        return this._fitOffset(offset, this._maxLeftOffset, this._maxRightOffset)
                    }
                    return 0
                },
                _calculateYOffset: function(eventData) {
                    if (this.direction !== "horizontal") {
                        var offset = eventData.y - this._startEventData.y;
                        return this._fitOffset(offset, this._maxTopOffset, this._maxBottomOffset)
                    }
                    return 0
                },
                _fitOffset: function(offset, minOffset, maxOffset) {
                    if (minOffset != null)
                        offset = Math.max(offset, -minOffset);
                    if (maxOffset != null)
                        offset = Math.min(offset, maxOffset);
                    return offset
                },
                _processDropTargets: function(e, dragOffset) {
                    var target = this._findDropTarget(e),
                        sameTarget = target === this._$currentDropTarget;
                    if (!sameTarget) {
                        this._fireDropTargetEvent(e, DRAG_LEAVE_EVENT);
                        this._$currentDropTarget = target;
                        this._fireDropTargetEvent(e, DRAG_ENTER_EVENT)
                    }
                },
                _fireDropTargetEvent: function(event, eventName) {
                    if (!this._$currentDropTarget)
                        return;
                    var eventData = {
                            type: eventName,
                            originalEvent: event,
                            draggingElement: this._$element.get(0),
                            target: this._$currentDropTarget.get(0)
                        };
                    events.fireEvent(eventData)
                },
                _findDropTarget: function(e) {
                    var that = this,
                        $result;
                    $.each(this._$dropTargetElements, function(_, $target) {
                        if (that._checkDropTarget($target, e)) {
                            $result = $target;
                            return false
                        }
                    });
                    return $result
                },
                _checkDropTarget: function($target, e) {
                    var isDraggingElement = $target.get(0) === this._$element.get(0);
                    if (isDraggingElement)
                        return false;
                    var targetPosition = getItemPosition($target);
                    if (e.pageX < targetPosition.left)
                        return false;
                    if (e.pageY < targetPosition.top)
                        return false;
                    var targetSize = getItemSize($target);
                    if (e.pageX > targetPosition.left + targetSize.width)
                        return false;
                    if (e.pageY > targetPosition.top + targetSize.height)
                        return false;
                    return $target
                },
                end: function(e) {
                    var eventData = events.eventData(e);
                    this._fireEvent(DRAG_END_EVENT, e, {offset: this._calculateOffset(eventData)});
                    this._fireDropTargetEvent(e, DROP_EVENT);
                    delete this._$currentDropTarget
                }
            });
        ui.gestureEmitters.push({
            emitter: DragEmitter,
            events: [DRAG_START_EVENT, DRAG_EVENT, DRAG_END_EVENT]
        });
        DX.ui.events.__internals = DX.ui.events.__internals || {};
        $.extend(DX.ui.events.__internals, {dropTargets: knownDropTargets})
    })(jQuery, DevExpress);
    /*! Module core, file ui.events.gesture.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            events = ui.events,
            utils = DX.utils;
        var abs = Math.abs;
        var GESTURE_EVENT = "dxGesture",
            GESTURE_EVENT_DATA = "dxGestureEmitter",
            GESTURE_LOCK_KEY = "dxGestureLock",
            GESTURE_UNLOCK_TIMEOUT = 400,
            TOUCH_BOUNDARY = 10,
            HORIZONTAL = "horizontal",
            VERTICAL = "vertical",
            BOTH = "both";
        var GestureEventManager = DX.Class.inherit({
                SLEEP: 0,
                INITED: 1,
                STARTED: 2,
                ctor: function() {
                    this._attachHandlers();
                    this.reset()
                },
                _attachHandlers: function() {
                    $(document).on(events.addNamespace("dxpointerdown", GESTURE_EVENT), $.proxy(this._handlePointerDown, this)).on(events.addNamespace("dxpointermove", GESTURE_EVENT), $.proxy(this._handlePointerMove, this)).on(events.addNamespace("dxpointerup dxpointercancel", GESTURE_EVENT), $.proxy(this._handlePointerUp, this)).on(events.addNamespace("dxmousewheel", GESTURE_EVENT), $.proxy(this._handleMouseWheel, this))
                },
                _eachEmitter: function(callback) {
                    $.each(this._activeEmitters || {}, function(direction, emitter) {
                        return callback(emitter, direction)
                    })
                },
                _noEmitters: function() {
                    return $.isEmptyObject(this._activeEmitters)
                },
                reset: function() {
                    this._eachEmitter(function(emitter) {
                        emitter.removeCancelCallback()
                    });
                    this._forgetGesture();
                    this._stage = this.SLEEP;
                    this._activeEmitters = {}
                },
                _handlePointerDown: function(e) {
                    if (events.needSkipEvent(e) || events.hasTouches(e) > 1)
                        return;
                    this.reset();
                    this._activeEmitters = this._closestEmitter(e);
                    if (this._noEmitters())
                        return;
                    this._startEvent = e;
                    this._startEventData = events.eventData(e);
                    this._stage = this.INITED;
                    this._applyToActive("init", e)
                },
                _applyToActive: function(method) {
                    var args = $.makeArray(arguments).slice(1);
                    this._eachEmitter(function(emitter) {
                        if (method in emitter)
                            emitter[method].apply(emitter, args)
                    })
                },
                _closestEmitter: function(e) {
                    var result = {},
                        foundForAllDirections = false,
                        $element = $(e.target);
                    while ($element.length && !foundForAllDirections) {
                        var emitter = $element.data(GESTURE_EVENT_DATA);
                        if (emitter && emitter.validate(e)) {
                            var direction = emitter.getDirection(e);
                            if (direction) {
                                emitter.addCancelCallback($.proxy(this._handleCancel, this, emitter, e));
                                result[direction] = result[direction] || emitter
                            }
                        }
                        foundForAllDirections = result[HORIZONTAL] && result[VERTICAL] || result[BOTH];
                        $element = $element.parent()
                    }
                    return result
                },
                _handleCancel: function(canceledEmitter, e) {
                    var canceledDirection;
                    this._eachEmitter(function(emitter, direction) {
                        if (emitter === canceledEmitter) {
                            canceledDirection = direction;
                            emitter.removeCancelCallback()
                        }
                    });
                    this._forgetGesture([canceledEmitter]);
                    this._cancelEmitter(canceledEmitter, e);
                    if (this._noEmitters())
                        this.reset()
                },
                _handlePointerMove: function(e) {
                    if (this._stage === this.INITED && this._directionDetected(e))
                        this._handleStart(e);
                    if (this._stage === this.STARTED)
                        this._handleMove(e)
                },
                _directionDetected: function(e) {
                    var delta = events.eventDelta(this._startEventData, events.eventData(e));
                    return delta.x || delta.y
                },
                _handleStart: function(e) {
                    this._filterEmitters(e);
                    if (this._noEmitters())
                        return;
                    this._resetActiveElement();
                    this._applyToActive("start", this._startEvent);
                    this._stage = this.STARTED
                },
                _resetActiveElement: function() {
                    if (DX.devices.real().platform !== "ios")
                        return;
                    this._eachEmitter(function(emitter) {
                        if ($(":focus", emitter.getElement()).length)
                            utils.resetActiveElement()
                    })
                },
                _filterEmitters: function(e) {
                    this._filterByDirection(e);
                    if (this._emitersAmount() > 1)
                        this._takeFirstEmitter()
                },
                _emitersAmount: function() {
                    var result = 0;
                    this._eachEmitter(function() {
                        result++
                    });
                    return result
                },
                _filterByDirection: function(e) {
                    var delta = events.eventDelta(this._startEventData, events.eventData(e)),
                        horizontalMove = abs(delta.y) < abs(delta.x),
                        verticalMove = abs(delta.y) > abs(delta.x),
                        horizontalEmmiter = this._activeEmitters[HORIZONTAL],
                        verticalEmmiter = this._activeEmitters[VERTICAL],
                        bothEmitter = this._activeEmitters[BOTH],
                        existsHorizontalEmitter = horizontalEmmiter || bothEmitter,
                        existsVerticalEmitter = verticalEmmiter || bothEmitter;
                    if (horizontalMove && existsHorizontalEmitter)
                        this._cancelEmitter(verticalEmmiter, e);
                    else if (verticalMove && existsVerticalEmitter)
                        this._cancelEmitter(horizontalEmmiter, e)
                },
                _cancelEmitter: function(canceledEmmiter, e) {
                    if (!canceledEmmiter)
                        return;
                    canceledEmmiter.cancel(e);
                    delete this._activeEmitters[canceledEmmiter.getDirection(e)]
                },
                _takeFirstEmitter: function() {
                    var activeEmitters = {};
                    this._eachEmitter(function(emitter, direction) {
                        activeEmitters[direction] = emitter;
                        return false
                    });
                    this._activeEmitters = activeEmitters
                },
                _prepareGesture: function() {
                    this._gestureLocked = true;
                    clearTimeout(this._gestureEndTimer);
                    this._eachEmitter(function(emitter) {
                        emitter.getElement().data(GESTURE_LOCK_KEY, true)
                    });
                    ui.feedback.reset()
                },
                _handleMove: function(e) {
                    if (!this._gestureLocked) {
                        var delta = events.eventDelta(this._startEventData, events.eventData(e));
                        if (abs(delta.x) > TOUCH_BOUNDARY || abs(delta.y) > TOUCH_BOUNDARY) {
                            events.handleGestureEvent(e, GESTURE_EVENT);
                            this._prepareGesture()
                        }
                    }
                    this._applyToActive("move", e)
                },
                _forgetGesture: function(activeEmitters) {
                    activeEmitters = activeEmitters || this._activeEmitters;
                    if (this._noEmitters())
                        return;
                    this._gestureLocked = false;
                    this._gestureEndTimer = setTimeout(function() {
                        $.each(activeEmitters, function(_, emitter) {
                            emitter.getElement().data(GESTURE_LOCK_KEY, false)
                        })
                    }, GESTURE_UNLOCK_TIMEOUT)
                },
                _handlePointerUp: function(e) {
                    if (!DX.devices.isRippleEmulator() && events.hasTouches(e))
                        return;
                    if (this._stage === this.STARTED)
                        this._applyToActive("end", e);
                    else if (this._stage === this.INITED)
                        this._applyToActive("stop", e);
                    this.reset()
                },
                _handleMouseWheel: function(e) {
                    this._handlePointerDown(e);
                    if (this._stage !== this.INITED)
                        return;
                    this._takeFirstEmitter();
                    this._eachEmitter(function(emitter, direction) {
                        var prop = direction !== "horizontal" ? "pageY" : "pageX";
                        e[prop] += e.delta
                    });
                    this._handlePointerMove(e);
                    this._handlePointerUp(e)
                },
                isActive: function(element) {
                    var result = false;
                    this._eachEmitter(function(emitter) {
                        result = result || emitter.getElement().is(element)
                    });
                    return result
                }
            });
        var gestureEventManager = new GestureEventManager;
        var registerEmitter = function(emitterConfig) {
                var emitterClass = emitterConfig.emitter;
                $.each(emitterConfig.events, function(_, eventName) {
                    events.registerEvent(eventName, {
                        noBubble: true,
                        setup: function(element, data) {
                            var emitter = $(element).data(GESTURE_EVENT_DATA) || new emitterClass(element);
                            emitter.configurate(data);
                            $(element).data(GESTURE_EVENT_DATA, emitter)
                        },
                        teardown: function(element) {
                            if (gestureEventManager.isActive(element))
                                gestureEventManager.reset();
                            $(element).removeData(GESTURE_EVENT_DATA)
                        }
                    })
                })
            };
        $.each(ui.gestureEmitters, function(_, emitterConfig) {
            registerEmitter(emitterConfig)
        });
        events.__internals = events.__internals || {};
        $.extend(events.__internals, {registerEmitter: registerEmitter})
    })(jQuery, DevExpress);
    /*! Module core, file ui.widget.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            UI_FEEDBACK = "UIFeedback",
            UI_FEEDBACK_CLASS = "dx-feedback",
            ACTIVE_STATE_CLASS = "dx-state-active",
            DISABLED_STATE_CLASS = "dx-state-disabled",
            INVISIBLE_STATE_CLASS = "dx-state-invisible",
            HOVER_STATE_CLASS = "dx-state-hover",
            FEEDBACK_SHOW_TIMEOUT = 30,
            FEEDBACK_HIDE_TIMEOUT = 400,
            HOVER_START = "dxhoverstart",
            HOVER_END = "dxhoverend",
            ANONYMOUS_TEMPLATE_NAME = "template",
            TEMPLATE_SELECTOR = "[data-options*='dxTemplate']",
            TEMPLATES_DATA_KEY = "dxTemplates";
        var getTemplateOptions = function(element) {
                var options = $(element).data("options");
                if ($.trim(options).charAt(0) !== "{")
                    options = "{" + options + "}";
                return new Function("return " + options)().dxTemplate
            };
        var activeElement,
            events = ui.events;
        ui.feedback = {reset: function() {
                handleEnd(true)
            }};
        ui.Widget = DX.DOMComponent.inherit({
            NAME: "Widget",
            NAMESPACE: ui,
            _setDefaultOptions: function() {
                this.callBase();
                this.option({
                    disabled: false,
                    visible: true,
                    activeStateEnabled: true,
                    width: undefined,
                    height: undefined,
                    contentReadyAction: null,
                    hoverStateEnabled: false
                })
            },
            _init: function() {
                this.callBase();
                this._feedbackShowTimeout = FEEDBACK_SHOW_TIMEOUT;
                this._feedbackHideTimeout = FEEDBACK_HIDE_TIMEOUT;
                if (this._templatesSupported()) {
                    this._initTemplates();
                    this._initContentReadyAction()
                }
            },
            _templatesSupported: function() {
                return this._renderContentImpl !== DX.abstract
            },
            _initTemplates: function() {
                var that = this,
                    templates = {},
                    dataTemplateElements = this._element().data(TEMPLATES_DATA_KEY),
                    templateElements = dataTemplateElements ? dataTemplateElements : this._element().contents().filter(TEMPLATE_SELECTOR);
                this._templateProvider = new ui.TemplateProvider;
                this._templateClass = this._templateProvider.getTemplateClass(this);
                if (templateElements.length) {
                    var templatesMap = {};
                    templateElements.each(function() {
                        var templateOptions = getTemplateOptions(this);
                        if (!templateOptions)
                            return;
                        if (!templateOptions.name)
                            throw Error("Template name was not specified");
                        templatesMap[templateOptions.name] = templatesMap[templateOptions.name] || [];
                        templatesMap[templateOptions.name].push(this)
                    });
                    $.each(templatesMap, function(templateName, value) {
                        var deviceTemplate = that._findTemplateByDevice(value);
                        if (deviceTemplate)
                            templates[templateName] = that._createTemplate(deviceTemplate)
                    })
                }
                else
                    templates[ANONYMOUS_TEMPLATE_NAME] = that._createTemplate(that._element().contents());
                this.option("_templates", templates)
            },
            _getTemplateByOption: function(optionName) {
                return this._getTemplate(this.option(optionName))
            },
            _getTemplate: function(templateName) {
                var result = this._acquireTemplate.apply(this, arguments);
                if (!result && this._templateProvider.supportDefaultTemplate(this)) {
                    result = this._templateProvider.getDefaultTemplate(this);
                    if (!result)
                        throw Error(DX.utils.stringFormat("Template \"{0}\" was not found and no default template specified!", templateName));
                }
                return result
            },
            _acquireTemplate: function(templateSource) {
                if (templateSource == null)
                    return templateSource;
                if (templateSource instanceof this._templateClass)
                    return templateSource;
                if (templateSource.nodeType || templateSource.jquery) {
                    templateSource = $(templateSource);
                    if (templateSource.is("script"))
                        templateSource = templateSource.html();
                    return this._createTemplate(templateSource)
                }
                if (typeof templateSource === "string")
                    return this.option("_templates")[templateSource];
                if ($.isFunction(templateSource)) {
                    var args = $.makeArray(arguments).slice(1);
                    return this._acquireTemplate(templateSource.apply(this, args))
                }
                return this._acquireTemplate(templateSource.toString())
            },
            _createTemplate: function(element) {
                return new this._templateClass(element, this)
            },
            _findTemplateByDevice: function(templates) {
                var suitableTemplate = DX.utils.findBestMatches(DX.devices.current(), templates, function(template) {
                        return getTemplateOptions(template)
                    })[0];
                $.each(templates, function(index, template) {
                    if (template !== suitableTemplate)
                        $(template).remove()
                });
                return suitableTemplate
            },
            _cleanTemplates: function() {
                var that = this;
                $.each(this.option("_templates"), function(templateName, template) {
                    if (that === template.owner())
                        template.dispose()
                })
            },
            _initContentReadyAction: function() {
                this._contentReadyAction = this._createActionByOption("contentReadyAction", {excludeValidators: ["gesture", "designMode", "disabled"]})
            },
            _render: function() {
                this.callBase();
                this._element().addClass("dx-widget");
                this._toggleDisabledState(this.option("disabled"));
                this._toggleVisibility(this.option("visible"));
                this._refreshFeedback();
                this._renderDimensions();
                if (this._templatesSupported())
                    this._renderContent();
                this._attachHoverEvents()
            },
            _renderContent: function() {
                this._renderContentImpl();
                this._fireContentReadyAction()
            },
            _renderContentImpl: DX.abstract,
            _fireContentReadyAction: function() {
                this._contentReadyAction({excludeValidators: ["disabled", "gesture"]})
            },
            _dispose: function() {
                if (this._templatesSupported())
                    this._cleanTemplates();
                this._contentReadyAction = null;
                this._clearTimers();
                if (activeElement && activeElement.closest(this._element()).length)
                    activeElement = null;
                this.callBase()
            },
            _clean: function() {
                this.callBase();
                this._element().empty()
            },
            _clearTimers: function() {
                clearTimeout(this._feedbackHideTimer);
                clearTimeout(this._feedbackShowTimer)
            },
            _toggleVisibility: function(visible) {
                this._element().toggleClass(INVISIBLE_STATE_CLASS, !visible)
            },
            _attachHoverEvents: function() {
                var that = this,
                    hoverableSelector = that._activeStateUnit,
                    nameStart = events.addNamespace(HOVER_START, UI_FEEDBACK),
                    nameEnd = events.addNamespace(HOVER_END, UI_FEEDBACK);
                that._element().off(nameStart, hoverableSelector).off(nameEnd, hoverableSelector);
                if (this.option("hoverStateEnabled")) {
                    var startAction = new DX.Action(function(args) {
                            var $target = args.element;
                            that._refreshHoveredElement($target)
                        });
                    that._element().on(nameStart, hoverableSelector, {selector: hoverableSelector}, function(e) {
                        startAction.execute({element: $(e.target)})
                    }).on(nameEnd, hoverableSelector, {selector: hoverableSelector}, function(e) {
                        e.stopImmediatePropagation();
                        that._forgetHoveredElement()
                    })
                }
                else
                    this._toggleHoverClass(false)
            },
            _refreshHoveredElement: function(hoveredElement) {
                var selector = this._activeStateUnit || this._element();
                this._forgetHoveredElement();
                this._hoveredElement = hoveredElement.closest(selector);
                this._toggleHoverClass(true)
            },
            _forgetHoveredElement: function() {
                this._toggleHoverClass(false);
                delete this._hoveredElement
            },
            _toggleHoverClass: function(value) {
                if (this._hoveredElement)
                    this._hoveredElement.toggleClass(HOVER_STATE_CLASS, value && this.option("hoverStateEnabled"))
            },
            _renderDimensions: function() {
                var width = this.option("width"),
                    height = this.option("height");
                this._setDimension(width, "width");
                this._setDimension(height, "height")
            },
            _setDimension: function(dimensionSize, dimension) {
                var $element = this._element();
                dimensionSize = $.isFunction(dimensionSize) ? dimensionSize() : dimensionSize;
                if ($.isNumeric(dimensionSize))
                    dimensionSize = dimensionSize + "px";
                if (!dimensionSize && dimension === "width")
                    dimensionSize = this._calculateWidth();
                $element.css(dimension, dimensionSize)
            },
            _calculateWidth: function() {
                var $element = this._element(),
                    explicitWidth = $element[0].style.width,
                    calculatedWidth;
                if (explicitWidth[explicitWidth.length - 1] === "%" && !this.option("width"))
                    return explicitWidth;
                else
                    calculatedWidth = explicitWidth && explicitWidth !== "auto" && explicitWidth !== "inherit" ? $element.outerWidth() : this.option("width");
                return calculatedWidth
            },
            _refreshFeedback: function() {
                if (this._feedbackDisabled()) {
                    this._feedbackOff(true);
                    this._element().removeClass(UI_FEEDBACK_CLASS)
                }
                else
                    this._element().addClass(UI_FEEDBACK_CLASS)
            },
            _feedbackDisabled: function() {
                return !this.option("activeStateEnabled") || this.option("disabled")
            },
            _feedbackOn: function(element, immediate) {
                if (this._feedbackDisabled())
                    return;
                this._clearTimers();
                if (immediate)
                    this._feedbackShow(element);
                else
                    this._feedbackShowTimer = window.setTimeout($.proxy(this._feedbackShow, this, element), this._feedbackShowTimeout);
                this._saveActiveElement()
            },
            _feedbackShow: function(element) {
                var activeStateElement = this._element();
                if (this._activeStateUnit)
                    activeStateElement = $(element).closest(this._activeStateUnit);
                if (!activeStateElement.hasClass(DISABLED_STATE_CLASS)) {
                    activeStateElement.addClass(ACTIVE_STATE_CLASS);
                    this._toggleHoverClass(false)
                }
            },
            _saveActiveElement: function() {
                activeElement = this._element()
            },
            _feedbackOff: function(immediate) {
                this._clearTimers();
                if (immediate)
                    this._feedbackHide();
                else
                    this._feedbackHideTimer = window.setTimeout($.proxy(this._feedbackHide, this), this._feedbackHideTimeout)
            },
            _feedbackHide: function() {
                var activeStateElement = this._element();
                if (this._activeStateUnit)
                    activeStateElement = activeStateElement.find(this._activeStateUnit);
                activeStateElement.removeClass(ACTIVE_STATE_CLASS);
                this._toggleHoverClass(!this.option("disabled"));
                this._clearActiveElement()
            },
            _clearActiveElement: function() {
                var rootDomElement = this._element().get(0),
                    activeDomElement = activeElement && activeElement.get(0);
                if (activeDomElement && (activeDomElement === rootDomElement || $.contains(rootDomElement, activeDomElement)))
                    activeElement = null
            },
            _toggleDisabledState: function(value) {
                this._element().toggleClass(DISABLED_STATE_CLASS, Boolean(value));
                this._toggleHoverClass(!value)
            },
            _optionChanged: function(name, value) {
                switch (name) {
                    case"disabled":
                        this._toggleDisabledState(value);
                        this._refreshFeedback();
                        break;
                    case"activeStateEnabled":
                        this._refreshFeedback();
                        break;
                    case"hoverStateEnabled":
                        this._attachHoverEvents();
                        break;
                    case"visible":
                        this._toggleVisibility(value);
                        break;
                    case"width":
                    case"height":
                        this._renderDimensions();
                        break;
                    case"contentReadyAction":
                        this._initContentReadyAction();
                        break;
                    case"_templates":
                        this._refresh();
                        break;
                    default:
                        this.callBase.apply(this, arguments)
                }
            },
            repaint: function() {
                this._refresh()
            }
        });
        var handleStart = function(args, immediate) {
                var e = args.jQueryEvent,
                    $target = args.element,
                    widget;
                if (events.needSkipEvent(e))
                    return;
                if (activeElement) {
                    widget = getWidget(activeElement);
                    if (widget)
                        widget._feedbackOff(true)
                }
                var closestFeedbackElement = $target.closest("." + UI_FEEDBACK_CLASS);
                if (closestFeedbackElement.length) {
                    widget = getWidget(closestFeedbackElement);
                    if (!widget)
                        return;
                    widget._feedbackOn($target, immediate);
                    if (immediate)
                        widget._feedbackOff()
                }
            };
        var handleEnd = function(immediate) {
                if (!activeElement)
                    return;
                var widget = getWidget(activeElement);
                if (widget)
                    widget._feedbackOff(immediate)
            };
        var getWidget = function(widgetElement) {
                var result;
                $.each(widgetElement.data("dxComponents") || [], function(index, componentName) {
                    if (ui[componentName] && ui[componentName].subclassOf(ui.Widget)) {
                        result = widgetElement.data(componentName);
                        return false
                    }
                });
                return result
            };
        $(function() {
            var startAction = new DX.Action(handleStart);
            $(document).on(events.addNamespace("dxpointerdown", UI_FEEDBACK), function(e) {
                startAction.execute({
                    jQueryEvent: e,
                    element: $(e.target)
                })
            }).on(events.addNamespace("dxpointerup dxpointercancel", UI_FEEDBACK), function(e) {
                var activeElementClicked = activeElement && $(e.target).closest("." + UI_FEEDBACK_CLASS).get(0) === activeElement.get(0);
                if (activeElementClicked)
                    startAction.execute({
                        jQueryEvent: e,
                        element: $(e.target)
                    }, true);
                handleEnd()
            })
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.editor.js */
    (function($, DX, undefined) {
        var ui = DX.ui;
        DX.registerComponent("dxEditor", ui.Widget.inherit({
            _setDefaultOptions: function() {
                this.callBase();
                this.option({
                    value: undefined,
                    valueChangeAction: undefined
                })
            },
            _recreateValueChangeAction: function() {
                this._valueChangeAction = this._createActionByOption("valueChangeAction")
            },
            _suppressValueChangeAction: function() {
                this._valueChangeActionSuppressed = true
            },
            _resumeValueChangeAction: function() {
                this._valueChangeActionSuppressed = false
            },
            _render: function() {
                this._recreateValueChangeAction();
                this.callBase()
            },
            _raiseValueChangeAction: function(value, previousValue, extraArguments) {
                var args = {
                        value: value,
                        previousValue: previousValue,
                        jQueryEvent: this._valueChangeEventInstance
                    };
                if (extraArguments)
                    args = $.extend(args, extraArguments);
                this._valueChangeAction(args)
            },
            _optionChanged: function(name, value, previousValue) {
                switch (name) {
                    case"valueChangeAction":
                        this._recreateValueChangeAction();
                        break;
                    case"value":
                        if (!this._valueChangeActionSuppressed)
                            this._raiseValueChangeAction(value, previousValue);
                        this._valueChangeEventInstance = undefined;
                        break;
                    default:
                        this.callBase.apply(this, arguments)
                }
            }
        }))
    })(jQuery, DevExpress);
    /*! Module core, file ui.template.js */
    (function($, DX, undefined) {
        var isString = DX.utils.isString;
        var currentTemplateEngine;
        var templateEngines = [];
        var BaseTemplate = DevExpress.Class.inherit({
                _compile: function(html, element) {
                    return element
                },
                _render: function(template, data) {
                    return template
                },
                ctor: function(element) {
                    this._element = $(element);
                    if (this._element.length === 1) {
                        if (this._element[0].nodeName.toLowerCase() !== "script")
                            this._element = $("<div />").append(this._element);
                        this._template = this._compile(this._element.html() || "", this._element)
                    }
                },
                render: function(container, data) {
                    var result;
                    if (this._template) {
                        result = this._render(this._template, data);
                        if (isString(result))
                            result = $.parseHTML(result);
                        result = $(result);
                        if (container)
                            container.append(result);
                        return result
                    }
                },
                owner: $.noop,
                dispose: $.noop
            });
        var createTemplateEngine = function(options) {
                if (options && options.compile && options.render)
                    return BaseTemplate.inherit({
                            allowRenderToDetachedContainer: options.allowRenderToDetachedContainer !== false,
                            _compile: options.compile,
                            _render: options.render
                        });
                else
                    throw Error("Template Engine must contains compile and render methods");
            };
        if (window.ko) {
            var koCustomTemplateEngine = function(){};
            koCustomTemplateEngine.prototype = ko.utils.extend(new ko.templateEngine, {
                renderTemplateSource: function(templateSource, bindingContext, options) {
                    var precompiledTemplate = templateSource["data"]("precompiledTemplate");
                    if (!precompiledTemplate) {
                        precompiledTemplate = new currentTemplateEngine(templateSource.domElement);
                        templateSource["data"]("precompiledTemplate", precompiledTemplate)
                    }
                    return precompiledTemplate.render(null, bindingContext.$data)
                },
                allowTemplateRewriting: false
            })
        }
        DevExpress.ui.setTemplateEngine = function(templateEngine) {
            if (isString(templateEngine)) {
                currentTemplateEngine = templateEngines && templateEngines[templateEngine];
                if (!currentTemplateEngine && templateEngine !== "default")
                    throw Error(DX.utils.stringFormat("Template Engine \"{0}\" is not supported", templateEngine));
            }
            else
                currentTemplateEngine = createTemplateEngine(templateEngine) || currentTemplateEngine;
            if (window.ko)
                ko.setTemplateEngine(currentTemplateEngine ? new koCustomTemplateEngine : new ko.nativeTemplateEngine)
        };
        DevExpress.ui.TemplateProvider = DevExpress.ui.TemplateProvider.inherit({getTemplateClass: function() {
                if (currentTemplateEngine)
                    return currentTemplateEngine;
                return this.callBase.apply(this, arguments)
            }});
        var registerTemplateEngine = function(name, templateOptions) {
                templateEngines[name] = createTemplateEngine(templateOptions)
            };
        registerTemplateEngine("jquery-tmpl", {
            compile: function(html, element) {
                return element
            },
            render: function(template, data) {
                return template.tmpl(data)
            }
        });
        registerTemplateEngine("jsrender", {
            compile: function(html) {
                return $.templates(html)
            },
            render: function(template, data) {
                return template.render(data)
            }
        });
        registerTemplateEngine("mustache", {
            compile: function(html) {
                return Mustache.compile(html)
            },
            render: function(template, data) {
                return template(data)
            }
        });
        registerTemplateEngine("hogan", {
            compile: function(html) {
                return Hogan.compile(html)
            },
            render: function(template, data) {
                return template.render(data)
            }
        });
        registerTemplateEngine("underscore", {
            compile: function(html) {
                return _.template(html)
            },
            render: function(template, data) {
                return template(data)
            }
        });
        registerTemplateEngine("handlebars", {
            compile: function(html) {
                return Handlebars.compile(html)
            },
            render: function(template, data) {
                return template(data)
            }
        });
        registerTemplateEngine("doT", {
            compile: function(html) {
                return doT.template(html)
            },
            render: function(template, data) {
                return template(data)
            }
        })
    })(jQuery, DevExpress);
    /*! Module core, file ui.collectionContainerWidget.js */
    (function($, DX, undefined) {
        var ui = DX.ui,
            events = ui.events;
        var CollectionContainerWidget = ui.Widget.inherit({
                _setDefaultOptions: function() {
                    this.callBase();
                    this.option({
                        items: [],
                        itemTemplate: "item",
                        itemRender: null,
                        itemRenderedAction: null,
                        itemClickAction: null,
                        itemHoldAction: null,
                        itemHoldTimeout: 750,
                        noDataText: Globalize.localize("dxCollectionContainerWidget-noDataText"),
                        dataSource: null,
                        selectedIndex: -1,
                        itemSelectAction: null
                    })
                },
                _init: function() {
                    this.callBase();
                    this._cleanRenderedItems();
                    this._refreshDataSource()
                },
                _dataSourceOptions: function() {
                    var options = {
                            paginate: false,
                            _preferSync: false
                        };
                    if ($.isArray(this.option("dataSource")))
                        options._preferSync = true;
                    return options
                },
                _cleanRenderedItems: function() {
                    this._renderedItemsCount = 0
                },
                _fireSelectItemEvent: function(index, prevIndex) {
                    if (this._selectionEnabled()) {
                        this._updateSelectedIndex(index, prevIndex);
                        this._handleItemEvent(this._selectedItemElement(index), "itemSelectAction", {
                            selectedIndex: index,
                            previousIndex: prevIndex
                        }, {excludeValidators: ["gesture", "disabled"]})
                    }
                },
                _optionChanged: function(name, value, prevValue) {
                    switch (name) {
                        case"items":
                            if (this._selectionEnabled()) {
                                var itemsCount = value && value.length || 0,
                                    maxIndex = Math.max(itemsCount - 1, 0);
                                if (maxIndex < this.option("selectedIndex"))
                                    this.option("selectedIndex", 0)
                            }
                            this._cleanRenderedItems();
                            this._invalidate();
                            break;
                        case"dataSource":
                            this._refreshDataSource();
                            if (!this._dataSource)
                                this.option("items", []);
                            this._renderEmptyMessage();
                            break;
                        case"noDataText":
                            this._renderEmptyMessage();
                            break;
                        case"itemTemplate":
                            this._invalidate();
                            break;
                        case"itemRender":
                            this._itemRender = null;
                            this._invalidate();
                            break;
                        case"itemRenderedAction":
                            this._createItemRenderAction();
                            break;
                        case"itemClickAction":
                            break;
                        case"itemHoldAction":
                        case"itemHoldTimeout":
                            this._attachHoldEvent();
                            break;
                        case"selectedIndex":
                            this._fireSelectItemEvent(value, prevValue);
                            break;
                        case"itemSelectAction":
                            break;
                        default:
                            this.callBase(name, value, prevValue)
                    }
                },
                _expectNextPageLoading: function() {
                    this._startIndexForAppendedItems = 0
                },
                _expectLastItemLoading: function() {
                    this._startIndexForAppendedItems = -1
                },
                _forgetNextPageLoading: function() {
                    this._startIndexForAppendedItems = null
                },
                _handleDataSourceChanged: function(newItems) {
                    var items = this.option("items");
                    if (this._initialized && items && this._shouldAppendItems()) {
                        this._renderedItemsCount = items.length;
                        if (!this._dataSource.isLastPage() || this._startIndexForAppendedItems !== -1)
                            this.option().items = items.concat(newItems.slice(this._startIndexForAppendedItems));
                        this._renderContent();
                        this._forgetNextPageLoading()
                    }
                    else
                        this.option("items", newItems)
                },
                _handleDataSourceLoadError: function() {
                    this._forgetNextPageLoading()
                },
                _shouldAppendItems: function() {
                    return this._startIndexForAppendedItems != null && this._allowDinamicItemsAppend()
                },
                _allowDinamicItemsAppend: function() {
                    return false
                },
                _clean: function() {
                    this._itemContainer().empty()
                },
                _refresh: function() {
                    this._cleanRenderedItems();
                    this.callBase.apply(this, arguments)
                },
                _itemContainer: function() {
                    return this._element()
                },
                _itemClass: DX.abstract,
                _itemSelector: function() {
                    return "." + this._itemClass()
                },
                _itemDataKey: DX.abstract,
                _itemElements: function() {
                    return this._itemContainer().find(this._itemSelector())
                },
                _render: function() {
                    this.callBase();
                    this._attachClickEvent();
                    this._attachHoldEvent();
                    if (this._selectionEnabled()) {
                        this._renderSelectedIndex(this.option("selectedIndex"));
                        this._attachSelectedEvent()
                    }
                },
                _selectionEnabled: function() {
                    return this._renderSelectedIndex !== DX.abstract
                },
                _selectionByClickEnabled: function() {
                    return true
                },
                _renderSelectedIndex: DX.abstract,
                _attachSelectedEvent: function() {
                    if (!this._selectionByClickEnabled())
                        return;
                    var itemSelector = this._itemSelector(),
                        itemSelectHandler = this._createAction($.proxy(function(e) {
                            this._handleItemSelect(e.jQueryEvent)
                        }, this)),
                        eventName = events.addNamespace("dxclick", this.NAME);
                    this._element().off(eventName, itemSelector).on(eventName, itemSelector, $.proxy(function(e) {
                        var $itemElement = $(e.target).closest(itemSelector);
                        itemSelectHandler({
                            itemElement: $itemElement,
                            jQueryEvent: e
                        });
                        this._handleItemClick(e)
                    }, this))
                },
                _handleItemSelect: function(e) {
                    if (events.needSkipEvent(e))
                        return;
                    var items = this.option("items"),
                        selectedItem = $(e.currentTarget).data(this._itemDataKey()),
                        selectedItemIndex = $.inArray(selectedItem, items);
                    this.option("selectedIndex", selectedItemIndex)
                },
                _updateSelectedIndex: function() {
                    this._renderSelectedIndex.apply(this, arguments)
                },
                _selectedItemElement: function(index) {
                    return this._itemElements().eq(index)
                },
                _attachClickEvent: function() {
                    if (this._selectionEnabled() && this._selectionByClickEnabled())
                        return;
                    var itemSelector = this._itemSelector(),
                        eventName = events.addNamespace("dxclick", this.NAME);
                    this._itemContainer().off(eventName, itemSelector).on(eventName, itemSelector, $.proxy(this._handleItemClick, this))
                },
                _handleItemClick: function(e) {
                    this._handleItemJQueryEvent(e, "itemClickAction")
                },
                _attachHoldEvent: function() {
                    var $itemContainer = this._itemContainer(),
                        itemSelector = this._itemSelector(),
                        eventName = events.addNamespace("dxhold", this.NAME);
                    $itemContainer.off(eventName, itemSelector);
                    if (this._shouldAttachHoldEvent())
                        $itemContainer.on(eventName, itemSelector, {timeout: this.option("itemHoldTimeout")}, $.proxy(this._handleItemHold, this))
                },
                _shouldAttachHoldEvent: function() {
                    return this.option("itemHoldAction")
                },
                _handleItemHold: function(e) {
                    this._handleItemJQueryEvent(e, "itemHoldAction")
                },
                _renderContentImpl: function() {
                    var items = this.option("items") || [];
                    if (this._renderedItemsCount)
                        this._renderItems(items.slice(this._renderedItemsCount));
                    else
                        this._renderItems(items)
                },
                _renderItems: function(items) {
                    if (items.length)
                        $.each(items, $.proxy(this._renderItem, this));
                    this._renderEmptyMessage()
                },
                _renderItem: function(index, itemData, container) {
                    container = container || this._itemContainer();
                    var itemRenderer = this._getItemRenderer(),
                        itemTemplateName = this._getItemTemplateName(itemData),
                        itemTemplate = this._getTemplate(itemTemplateName, index, itemData),
                        itemElement,
                        renderArgs = {
                            index: index,
                            item: itemData,
                            container: container
                        };
                    if (itemRenderer)
                        itemElement = this._createItemByRenderer(itemRenderer, renderArgs);
                    else if (itemTemplate)
                        itemElement = this._createItemByTemplate(itemTemplate, renderArgs);
                    else
                        itemElement = this._createItemByRenderer(this._itemRenderDefault, renderArgs);
                    itemElement.addClass(this._itemClass()).data(this._itemDataKey(), itemData);
                    var postprocessRenderArgs = {
                            itemElement: itemElement,
                            itemData: itemData,
                            itemIndex: index
                        };
                    this._postprocessRenderItem(postprocessRenderArgs);
                    this._getItemRenderAction()({
                        itemElement: itemElement,
                        itemData: itemData
                    });
                    return itemElement
                },
                _createItemRenderAction: function() {
                    return this._itemRenderAction = this._createActionByOption("itemRenderedAction", {
                            element: this._element(),
                            excludeValidators: ["gesture", "designMode", "disabled"]
                        })
                },
                _getItemRenderAction: function() {
                    return this._itemRenderAction || this._createItemRenderAction()
                },
                _getItemRenderer: function() {
                    this._itemRender = this._itemRender || this.option("itemRender");
                    return this._itemRender
                },
                _createItemByRenderer: function(itemRenderer, renderArgs) {
                    var itemElement = $("<div />").appendTo(renderArgs.container);
                    var rendererResult = itemRenderer.call(this, renderArgs.item, renderArgs.index, itemElement);
                    if (rendererResult != null && itemElement[0] !== rendererResult[0])
                        itemElement.append(rendererResult);
                    return itemElement
                },
                _getItemTemplateName: function(itemData) {
                    return itemData && itemData.template || this.option("itemTemplate")
                },
                _createItemByTemplate: function(itemTemplate, renderArgs) {
                    return itemTemplate.render(renderArgs.container, renderArgs.item, renderArgs.index, "ignoreTarget")
                },
                _itemRenderDefault: function(item, index, itemElement) {
                    if ($.isPlainObject(item)) {
                        if (item.visible !== undefined && !item.visible)
                            itemElement.hide();
                        if (item.disabled)
                            itemElement.addClass("dx-state-disabled");
                        if (item.text)
                            itemElement.text(item.text);
                        if (item.html)
                            itemElement.html(item.html)
                    }
                    else
                        itemElement.html(String(item))
                },
                _postprocessRenderItem: $.noop,
                _renderEmptyMessage: function() {
                    if (!this._selectionEnabled()) {
                        var noDataText = this.option("noDataText"),
                            items = this.option("items"),
                            dataSourceLoading = this._dataSource && this._dataSource.isLoading(),
                            hideNoData = !noDataText || items && items.length || dataSourceLoading;
                        if (hideNoData && this._$nodata) {
                            this._$nodata.remove();
                            this._$nodata = null
                        }
                        if (!hideNoData) {
                            this._$nodata = this._$nodata || $("<div />").addClass("dx-empty-message");
                            this._$nodata.appendTo(this._itemContainer()).text(noDataText)
                        }
                    }
                },
                _handleItemJQueryEvent: function(jQueryEvent, handlerOptionName, actionArgs, actionConfig) {
                    this._handleItemEvent(jQueryEvent.target, handlerOptionName, $.extend(actionArgs, {jQueryEvent: jQueryEvent}), actionConfig)
                },
                _handleItemEvent: function(initiator, handlerOptionName, actionArgs, actionConfig) {
                    var action = this._createActionByOption(handlerOptionName, actionConfig);
                    return this._handleItemEventImpl(initiator, action, actionArgs)
                },
                _handleItemEventByHandler: function(initiator, handler, actionArgs, actionConfig) {
                    var action = this._createAction(handler, actionConfig);
                    return this._handleItemEventImpl(initiator, action, actionArgs)
                },
                _handleItemEventImpl: function(initiator, action, actionArgs) {
                    var $itemElement = this._closestItemElement($(initiator));
                    actionArgs = $.extend({
                        itemElement: $itemElement,
                        itemData: this._getItemData($itemElement)
                    }, actionArgs);
                    return action(actionArgs)
                },
                _closestItemElement: function($element) {
                    return $($element).closest(this._itemSelector())
                },
                _getItemData: function($itemElement) {
                    return $itemElement.data(this._itemDataKey())
                },
                itemElements: function() {
                    return this._itemElements()
                },
                itemsContainer: function() {
                    return this._itemContainer()
                }
            }).include(ui.DataHelperMixin);
        ui.CollectionContainerWidget = CollectionContainerWidget
    })(jQuery, DevExpress);
    /*! Module core, file ui.tooltip.js */
    (function($, DX, undefined) {
        var $tooltip = null;
        var createTooltip = function(options) {
                options = $.extend({position: "top"}, options);
                var content = options.content;
                delete options.content;
                return $("<div />").html(content).appendTo(DX.overlayTargetContainer()).dxTooltip(options)
            };
        var removeTooltip = function() {
                if (!$tooltip)
                    return;
                $tooltip.remove();
                $tooltip = null
            };
        var tooltip = {
                show: function(options) {
                    removeTooltip();
                    $tooltip = createTooltip(options);
                    return $tooltip.dxTooltip("show")
                },
                hide: function() {
                    if (!$tooltip)
                        return $.when();
                    return $tooltip.dxTooltip("hide").done(removeTooltip).promise()
                }
            };
        DX.ui.tooltip = tooltip
    })(jQuery, DevExpress)
}
if (!DevExpress.MOD_VIZ_CORE) {
    if (!window.DevExpress)
        throw Error('Required module is not referenced: core');
    /*! Module viz-core, file namespaces.js */
    (function(DevExpress) {
        DevExpress.viz = {}
    })(DevExpress);
    /*! Module viz-core, file namespaces.js */
    (function(DevExpress) {
        DevExpress.viz.core = {}
    })(DevExpress);
    /*! Module viz-core, file errorsWarnings.js */
    (function(DevExpress) {
        DevExpress.viz.core.errorsWarnings = {
            E2001: "Invalid data source",
            E2002: "Axis type and data type are incompatible",
            E2003: "\"{0}\" data source field contains data of unsupported type",
            E2004: "\"{0}\" data source field is inconsistent",
            E2101: "Unknown series type was specified: {0}",
            E2102: "Ambiguity occurred between two value axes with the same name",
            E2103: "\"{0}\" option must be a function",
            E2104: "Invalid logarithm base",
            E2105: "Invalid value of a \"{0}\"",
            E2106: "Invalid visible range",
            E2201: "Invalid type of data source field",
            E2202: "Invalid scale {0} value",
            E2203: "The \"{0}\" field of the \"selectedRange\" configuration object is not valid",
            W2001: "{0} cannot be drawn because its container is invisible",
            W2002: "The {0} data field is absent",
            W2003: "Tick interval is too small",
            W2101: "\"{0}\" pane does not exist; \"{1}\" pane is used instead",
            W2102: "Value axis with the \"{0}\" name was created automatically",
            W2103: "Chart title was hidden due to container size",
            W2104: "Legend was hidden due to container size",
            W2105: "Title of \"{0}\" axis was hidden due to container size",
            W2106: "Labels of \"{0}\" axis were hidden due to container size",
            W2301: "Invalid value range"
        }
    })(DevExpress);
    /*! Module viz-core, file tickProvider.js */
    (function($, DX, undefined) {
        var utils = DX.utils,
            math = Math,
            mathCeil = math.ceil,
            mathFloor = math.floor,
            mathAbs = math.abs,
            mathRound = math.round,
            core = DX.viz.core;
        var TICKS_COUNT_LIMIT = 2000;
        core.outOfScreen = {
            x: -1000,
            y: -1000
        };
        core.tickIntervalCalculator = {
            _defaultNumberMultipliers: [1, 2, 3, 5],
            _defaultGridSpacingFactor: 30,
            _getCommonTickInterval: function(deltaCoef, numberMultipliers) {
                var factor,
                    result = 0,
                    hasResult = false,
                    i;
                for (factor = 1; !hasResult; factor *= 10)
                    for (i = 0; i < numberMultipliers.length; i++) {
                        result = numberMultipliers[i] * factor;
                        if (deltaCoef <= result) {
                            hasResult = true;
                            break
                        }
                    }
                return result
            },
            _getNumericTickInterval: function(deltaCoef, numberMultipliers) {
                var factor,
                    result = 0,
                    newResult,
                    hasResult = false,
                    i;
                if (deltaCoef > 1.0)
                    result = this._getCommonTickInterval(deltaCoef, numberMultipliers);
                else if (deltaCoef > 0) {
                    result = 1;
                    for (factor = 0.1; !hasResult; factor /= 10)
                        for (i = numberMultipliers.length - 1; i >= 0; i--) {
                            newResult = numberMultipliers[i] * factor;
                            if (deltaCoef > newResult) {
                                hasResult = true;
                                break
                            }
                            result = newResult
                        }
                }
                return utils.adjustValue(result)
            },
            _getLogarithmicTickInterval: function(deltaCoef, numberMultipliers) {
                var result = 0;
                if (deltaCoef !== 0)
                    result = this._getCommonTickInterval(deltaCoef, numberMultipliers);
                return utils.adjustValue(result)
            },
            _getDateTimeTickInterval: function(deltaCoef, numberMultipliers) {
                var dateTimeMultipliers = {
                        millisecond: [1, 2, 5, 10, 25, 100, 250, 300, 500],
                        second: [1, 2, 3, 5, 10, 15, 20, 30],
                        minute: [1, 2, 3, 5, 10, 15, 20, 30],
                        hour: [1, 2, 3, 4, 6, 8, 12],
                        day: [1, 2, 3, 5, 7, 10, 14],
                        month: [1, 2, 3, 6]
                    },
                    result = {},
                    factor,
                    key,
                    specificMultipliers,
                    yearsCount,
                    i;
                if (deltaCoef > 0 && deltaCoef < 1.0)
                    return {milliseconds: 1};
                else if (deltaCoef === 0)
                    return 0;
                for (key in dateTimeMultipliers)
                    if (dateTimeMultipliers.hasOwnProperty(key)) {
                        specificMultipliers = dateTimeMultipliers[key];
                        for (i = 0; i < specificMultipliers.length; i++)
                            if (deltaCoef <= utils.convertDateUnitToMilliseconds(key, specificMultipliers[i])) {
                                result[key + 's'] = specificMultipliers[i];
                                return result
                            }
                    }
                for (factor = 1; ; factor *= 10)
                    for (i = 0; i < numberMultipliers.length; i++) {
                        yearsCount = factor * numberMultipliers[i];
                        if (deltaCoef <= utils.convertDateUnitToMilliseconds('year', yearsCount))
                            return {years: yearsCount}
                    }
                return null
            },
            getTickInterval: function(options) {
                var that = this,
                    gridSpacingFactor = options.gridSpacingFactor || that._defaultGridSpacingFactor,
                    numberMultipliers = options.numberMultipliers || that._defaultNumberMultipliers,
                    businessDelta = options.businessDelta,
                    screenDelta = options.screenDelta,
                    deltaCoef = screenDelta > 0 && gridSpacingFactor < screenDelta ? gridSpacingFactor * businessDelta / screenDelta : 0;
                that._testNumberMultipliers = numberMultipliers;
                if (options.axisType === 'logarithmic')
                    return that._getLogarithmicTickInterval(deltaCoef, numberMultipliers);
                else
                    switch (options.dataType) {
                        case'numeric':
                            return that._getNumericTickInterval(deltaCoef, numberMultipliers);
                            break;
                        case'datetime':
                            return that._getDateTimeTickInterval(deltaCoef, numberMultipliers);
                            break
                    }
                return null
            }
        };
        core.minorTickIntervalCalculator = {
            _defaultNumberMultipliers: [2, 4, 5, 8, 10],
            _defaultGridSpacingFactor: 15,
            _getDateTimeTickInterval: function(businessDelta, deltaCoef, numberMultipliers) {
                var result,
                    i;
                for (i = numberMultipliers.length - 1; i >= 0; i--) {
                    this.testResultNumberMultiplier = numberMultipliers[i];
                    result = mathFloor(businessDelta / numberMultipliers[i]);
                    if (deltaCoef <= result)
                        return utils.convertMillisecondsToDateUnits(result)
                }
                return 0
            },
            _getCommonTickInterval: function(businessDelta, deltaCoef, numberMultipliers) {
                var result,
                    i;
                for (i = numberMultipliers.length - 1; i >= 0; i--) {
                    this.testResultNumberMultiplier = numberMultipliers[i];
                    result = businessDelta / numberMultipliers[i];
                    if (deltaCoef <= result)
                        return utils.adjustValue(result)
                }
                return 0
            },
            getTickInterval: function(options) {
                var that = this,
                    gridSpacingFactor = !utils.isDefined(options.gridSpacingFactor) ? that._defaultGridSpacingFactor : options.gridSpacingFactor,
                    numberMultipliers = options.numberMultipliers || that._defaultNumberMultipliers,
                    businessDelta = options.businessDelta,
                    deltaCoef = gridSpacingFactor * businessDelta / options.screenDelta;
                if (options.axisType === 'logarithmic')
                    return that._getCommonTickInterval(businessDelta, deltaCoef, numberMultipliers);
                else
                    switch (options.dataType) {
                        case'numeric':
                            return that._getCommonTickInterval(businessDelta, deltaCoef, numberMultipliers);
                        case'datetime':
                            return that._getDateTimeTickInterval(businessDelta, deltaCoef, numberMultipliers)
                    }
                return 0
            }
        };
        core.tickProvider = {
            _appendFakeSVGElement: function(value, text, options) {
                var textOptions = $.extend({}, options.textOptions, {rotate: 0});
                return options.renderer.createText(text, core.outOfScreen.x + (options.translator.translate(value) || 0), core.outOfScreen.y, textOptions).append()
            },
            _getDistanceByAngle: function(elementHeight, rotationAngle) {
                return elementHeight / mathAbs(math.sin(rotationAngle * (math.PI / 180)))
            },
            _areDisplayValuesValid: function(value1, value2, options) {
                var that = this,
                    getText = that._getTextFunc(options),
                    rotationAngle = options.textOptions && utils.isNumber(options.textOptions.rotate) ? options.textOptions.rotate : 0,
                    svgElement1 = that._appendFakeSVGElement(value1, getText(value1), options),
                    svgElement2 = that._appendFakeSVGElement(value2, getText(value2), options),
                    bBox1 = svgElement1.getBBox(),
                    bBox2 = svgElement2.getBBox(),
                    result,
                    translator = options.translator;
                if (rotationAngle !== 0)
                    result = that._getDistanceByAngle(bBox1.height, rotationAngle) <= mathAbs(bBox2.x - bBox1.x);
                else if (options.isHorizontal)
                    result = !translator.getBusinessRange().invert ? bBox1.x + bBox1.width < bBox2.x : bBox2.x + bBox2.width < bBox1.x;
                else
                    result = mathAbs(translator.translate(value1) - translator.translate(value2)) > bBox1.height;
                svgElement1.remove();
                svgElement2.remove();
                return result
            },
            _removeInvalidDatesWithUnitBegining: function(dates, options) {
                if (dates.length <= 1 || !options.setTicksAtUnitBeginning || !utils.isDate(options.min))
                    return;
                if (!this._areDisplayValuesValid(dates[0], dates[1], options))
                    dates.splice(1, 1)
            },
            _getValueSize: function(values, options) {
                var that = this,
                    value,
                    rotationAngle = options.textOptions ? options.textOptions.rotate : 0,
                    svgElement,
                    bBox,
                    result,
                    getText = that._getTextFunc(options),
                    i;
                if (values.length === 0)
                    return 0;
                options.isRotate = utils.isNumber(rotationAngle) && rotationAngle !== 0;
                if (options.isRotate || !options.isHorizontal)
                    value = getText(values[0]);
                else {
                    value = [];
                    for (i = 0; i < values.length; i++)
                        value.push(getText(values[i]));
                    value = value.join('\n')
                }
                svgElement = that._appendFakeSVGElement(value, value, options);
                bBox = svgElement.getBBox();
                if (options.isRotate)
                    result = that._getDistanceByAngle(bBox.height, rotationAngle);
                else
                    result = options.isHorizontal ? bBox.width : bBox.height;
                svgElement.remove();
                return mathCeil(result)
            },
            _adjustNumericTickValue: function(value, interval, min) {
                return utils.isExponential(value) ? utils.adjustValue(value) : utils.applyPrecisionByMinDelta(min, interval, value)
            },
            _generateStartTick: function(tickInterval, options) {
                var that = this,
                    milliseconds = 0,
                    boundedRule = options.min - options.max < 0,
                    startTick = options.min,
                    isDate = utils.isDate(options.min),
                    currentTickInterval = isDate ? utils.convertDateTickIntervalToMilliseconds(tickInterval) : tickInterval,
                    nextTick;
                if (options.axisType === 'logarithmic')
                    startTick = utils.raiseTo(mathFloor(utils.adjustValue(utils.getLog(options.min, options.base)) / currentTickInterval * currentTickInterval), options.base);
                else {
                    startTick = mathFloor(options.min / currentTickInterval) * currentTickInterval;
                    startTick = isDate ? new Date(startTick) : that._adjustNumericTickValue(startTick, currentTickInterval, options.min)
                }
                while (boundedRule === startTick - options.min < 0 && startTick !== options.min) {
                    nextTick = that._nextTick(startTick, tickInterval, options);
                    if (nextTick !== startTick)
                        startTick = nextTick;
                    else
                        return nextTick
                }
                return startTick
            },
            _nextTick: function(tick, tickInterval, options) {
                var nextTick,
                    lgPower,
                    that = this;
                if (options.axisType === 'logarithmic') {
                    lgPower = utils.addInterval(utils.adjustValue(utils.getLog(tick, options.base)), tickInterval, options.min > options.max);
                    nextTick = utils.raiseTo(lgPower, options.base);
                    nextTick = that._adjustNumericTickValue(nextTick, tickInterval, math.min(options.min, options.max))
                }
                else {
                    nextTick = utils.addInterval(tick, tickInterval, options.min > options.max);
                    if (options.dataType === 'numeric')
                        nextTick = that._adjustNumericTickValue(nextTick, tickInterval, options.min);
                    if (options.dataType === 'datetime' && options.setTicksAtUnitBeginning)
                        utils.correctDateWithUnitBeginning(nextTick, tickInterval)
                }
                return nextTick
            },
            _addMinorTicks: function(majorTick1, majorTick2, ticksInfo, options, isReverse) {
                var that = this,
                    i,
                    dataType = options.dataType,
                    businessDelta,
                    minorTicks = [],
                    interval = 0,
                    minorTickIntervalsCount = options.minorTickCount + 1,
                    intervalsCount,
                    tickInterval;
                options.min = majorTick1;
                options.max = majorTick2;
                if (!utils.isDefined(options.tickInterval)) {
                    options.businessDelta = businessDelta = mathAbs(options.max - options.min);
                    options.screenDelta = businessDelta * options.deltaCoef;
                    if (utils.isDefined(options.minorTickCount)) {
                        if (!ticksInfo.majorTicks.autoArrangementStep || ticksInfo.majorTicks.autoArrangementStep <= 1) {
                            intervalsCount = options.minorTickCount + 1;
                            interval = dataType === 'datetime' ? utils.convertDateTickIntervalToMilliseconds(ticksInfo.majorTickInterval) : ticksInfo.majorTickInterval;
                            minorTickIntervalsCount = mathRound(businessDelta / interval * intervalsCount) || 1
                        }
                        tickInterval = dataType === 'datetime' ? utils.convertMillisecondsToDateUnits(businessDelta / minorTickIntervalsCount) : businessDelta / minorTickIntervalsCount;
                        if ($.isNumeric(tickInterval))
                            tickInterval = utils.adjustValue(tickInterval)
                    }
                    else if (utils.isDate(majorTick1))
                        tickInterval = core.minorTickIntervalCalculator.getTickInterval(options)
                }
                options = $.extend(true, {}, options, {tickInterval: tickInterval});
                minorTicks = that.getTicks(options);
                if (isReverse)
                    minorTicks.reverse();
                if (minorTicks.length > 0)
                    if (mathCeil(mathAbs(majorTick2 - minorTicks[minorTicks.length - 1]) * options.deltaCoef) < 2)
                        minorTicks.pop();
                for (i = 0; i < minorTicks.length; i++) {
                    ticksInfo.minorTicks.push(minorTicks[i]);
                    ticksInfo.fullTicks.push(minorTicks[i])
                }
            },
            _addLeftBoudedTicks: function(ticksInfo, min, minorTicksOptions) {
                if (utils.isDefined(min) && ticksInfo.majorTicks[0].valueOf() !== min.valueOf()) {
                    minorTicksOptions.addMinMax.max = true;
                    this._addMinorTicks(ticksInfo.majorTicks[0], min, ticksInfo, minorTicksOptions, true);
                    minorTicksOptions.addMinMax.max = false;
                    if (minorTicksOptions.showCustomBoundaryTicks) {
                        if (ticksInfo.minorTicks.length > 0 && ticksInfo.minorTicks[0].valueOf() === min.valueOf())
                            ticksInfo.minorTicks.shift(min);
                        ticksInfo.customBoundaryTicks.push(min);
                        ticksInfo.fullTicks.unshift(min)
                    }
                }
            },
            _addRightBoudedTicks: function(ticksInfo, max, minorTicksOptions) {
                var lastMajorTick = ticksInfo.majorTicks[ticksInfo.majorTicks.length - 1];
                ticksInfo.fullTicks.push(lastMajorTick);
                if (utils.isDefined(max) && lastMajorTick.valueOf() !== max.valueOf()) {
                    minorTicksOptions.addMinMax.min = false;
                    minorTicksOptions.addMinMax.max = true;
                    this._addMinorTicks(lastMajorTick, max, ticksInfo, minorTicksOptions);
                    if (minorTicksOptions.showCustomBoundaryTicks) {
                        if (ticksInfo.minorTicks.length > 0 && ticksInfo.minorTicks[ticksInfo.minorTicks.length - 1].valueOf() === max.valueOf())
                            ticksInfo.minorTicks.pop(max);
                        ticksInfo.customBoundaryTicks.push(max);
                        ticksInfo.fullTicks.push(max)
                    }
                }
            },
            _correctBoundedTicks: function(min, max, ticks, addMinMax) {
                addMinMax = $.extend({}, {
                    min: true,
                    max: true
                }, addMinMax);
                if (ticks.length > 0) {
                    if (!addMinMax.min && ticks[0].valueOf() === min.valueOf())
                        ticks.shift();
                    if (!addMinMax.max || ticks[ticks.length - 1].valueOf() !== max.valueOf())
                        ticks.pop()
                }
            },
            _initializeMinorTicksOptions: function(min, max, screenDelta, ticksInfo, options) {
                var that = this,
                    businessDelta,
                    hasMinorsCount = utils.isDefined(options.minorTickCount);
                $.extend(true, options, {
                    addMinMax: {
                        min: false,
                        max: false
                    },
                    deltaCoef: that._getDeltaCoef(screenDelta, max, min, options.axisType, options.base)
                }, options);
                options.numberMultipliers = hasMinorsCount ? [options.minorTickCount + 1] : options.numberMultipliers;
                options.gridSpacingFactor = hasMinorsCount ? 0 : options.gridSpacingFactor;
                if (!hasMinorsCount && ticksInfo.majorTicks.length > 1) {
                    options.businessDelta = businessDelta = that._getBusinessDelta(options.axisType, ticksInfo.majorTicks[0], ticksInfo.majorTicks[1], options.base);
                    if (that.needTickIntervalCalculation(businessDelta, ticksInfo.minorTickInterval, options.incidentOccured)) {
                        options.screenDelta = businessDelta * options.deltaCoef;
                        ticksInfo.minorTickInterval = core.minorTickIntervalCalculator.getTickInterval(options);
                        if (utils.isNumber(min))
                            options.tickInterval = ticksInfo.minorTickInterval;
                        else
                            options.tickInterval = undefined
                    }
                }
            },
            _getDataType: function(value) {
                return utils.isDate(value) ? 'datetime' : 'numeric'
            },
            _getBusinessDelta: function(type, min, max, base) {
                return type === 'logarithmic' ? mathRound(mathAbs(utils.getLog(min, base) - utils.getLog(max, base))) : mathAbs(min - max)
            },
            _getDeltaCoef: function(screenDelta, max, min, type, base) {
                return screenDelta / this._getBusinessDelta(type, min, max, base)
            },
            _initializeMajorTicksOptions: function(min, max, screenDelta, ticksInfo, options) {
                var businessDelta;
                options.screenDelta = screenDelta;
                $.extend(true, options, {
                    min: min,
                    max: max,
                    screenDelta: screenDelta,
                    isHorizontal: true
                });
                if (utils.isDefined(min) && utils.isDefined(max)) {
                    options.businessDelta = businessDelta = this._getBusinessDelta(options.axisType, min, max, options.base);
                    if (this.needTickIntervalCalculation(businessDelta, ticksInfo.majorTickInterval, options.incidentOccured)) {
                        options.isStartTickGenerated = true;
                        ticksInfo.majorTickInterval = core.tickIntervalCalculator.getTickInterval(options);
                        options.tickInterval = ticksInfo.majorTickInterval
                    }
                }
            },
            _getTextFunc: function(options) {
                return options.getText || function(value) {
                        return value.toString()
                    }
            },
            _generateTicks: function(options) {
                var that = this,
                    ticks = [],
                    tick,
                    boundedRule = options.max - options.min > 0,
                    leftBound,
                    rightBound,
                    tickInterval,
                    isStartTickGenerated = options.isStartTickGenerated,
                    businessDelta,
                    useTicksAutoArrangement = options.useTicksAutoArrangement;
                options.dataType = options.dataType || that._getDataType(options.min);
                options.businessDelta = businessDelta = that._getBusinessDelta(options.axisType, options.min, options.max, options.base);
                if (!utils.isDefined(options.min) || !utils.isDefined(options.max) || isNaN(options.min) || isNaN(options.max)) {
                    ticks = options.isHorizontal ? ['canvas_position_left', 'canvas_position_center', 'canvas_position_right'] : ['canvas_position_bottom', 'canvas_position_middle', 'canvas_position_top'];
                    useTicksAutoArrangement = false;
                    ticks.hideLabels = true
                }
                else {
                    tickInterval = $.isNumeric(options.min) && $.isNumeric(options.max) && !$.isNumeric(options.tickInterval) ? undefined : options.tickInterval;
                    if (this.needTickIntervalCalculation(businessDelta, tickInterval, options.incidentOccured)) {
                        isStartTickGenerated = utils.isDefined(isStartTickGenerated) ? isStartTickGenerated : true;
                        tickInterval = core.tickIntervalCalculator.getTickInterval(options)
                    }
                    ticks.tickInterval = tickInterval;
                    that.isTestStartTickGenerated = isStartTickGenerated;
                    that.isTestTickInterval = tickInterval;
                    that.testGridSpacingFactor = options.gridSpacingFactor;
                    if (tickInterval && tickInterval.valueOf() !== 0 && options.min.valueOf() !== options.max.valueOf()) {
                        tick = isStartTickGenerated ? that._generateStartTick(tickInterval, options) : options.min;
                        do {
                            ticks.push(tick);
                            tick = that._nextTick(tick, tickInterval, options);
                            if (ticks[ticks.length - 1].valueOf() === tick.valueOf())
                                break;
                            leftBound = tick - options.min > 0;
                            rightBound = options.max - tick > 0
                        } while (boundedRule === leftBound && boundedRule === rightBound);
                        ticks.push(tick);
                        that._correctBoundedTicks(options.min, options.max, ticks, options.addMinMax)
                    }
                    if (options.min.valueOf() === options.max.valueOf()) {
                        tick = options.min;
                        ticks.push(tick)
                    }
                }
                return ticks
            },
            _getAutoArrangementStep: function(ticks, options) {
                var that = this,
                    requiredValuesCount,
                    addedSpacing = options.isHorizontal ? options.textSpacing : 0;
                if (options.getCustomAutoArrangementStep)
                    return options.getCustomAutoArrangementStep(ticks, options);
                if (options.maxDisplayValueSize > 0) {
                    requiredValuesCount = mathFloor((options.screenDelta + options.textSpacing) / (options.maxDisplayValueSize + addedSpacing));
                    return mathCeil((options.ticksCount || ticks.length) / requiredValuesCount)
                }
                return 1
            },
            _getAutoArrangementTicks: function(ticks, options, step) {
                var that = this,
                    resultTicks = ticks,
                    i;
                if (step > 1) {
                    resultTicks = [];
                    for (i = 0; i < ticks.length; i += step)
                        resultTicks.push(ticks[i]);
                    resultTicks.tickInterval = ticks.tickInterval * step
                }
                return resultTicks
            },
            isOverlappedTicks: function(ticks, options) {
                options.maxDisplayValueSize = this._getValueSize(ticks, options);
                return this._getAutoArrangementStep(ticks, options) > 1
            },
            getCorrectedTicks: function(ticks, ticksOptions) {
                var step = Math.ceil(core.tickIntervalCalculator.getTickInterval({
                        screenDelta: ticksOptions.screenDelta * 4,
                        businessDelta: ticks.length,
                        gridSpacingFactor: ticksOptions.gridSpacingFactor,
                        numberMultipliers: ticksOptions.numberMultipliers,
                        dataType: "numeric"
                    })) || ticks.length;
                return this._getAutoArrangementTicks(ticks, ticksOptions, step)
            },
            needTickIntervalCalculation: function(businessDelta, tickInterval, incidentOccured) {
                var date;
                if (utils.isDefined(tickInterval)) {
                    if (!utils.isNumber(tickInterval)) {
                        date = new Date;
                        tickInterval = utils.addInterval(date, tickInterval) - date;
                        if (!tickInterval)
                            return true
                    }
                    if (utils.isNumber(tickInterval))
                        if (tickInterval > 0 && businessDelta / tickInterval > TICKS_COUNT_LIMIT) {
                            if (incidentOccured)
                                incidentOccured('W2003')
                        }
                        else
                            return false
                }
                return true
            },
            getTickIntervals: function(min, max, screenDelta, majorTicksOptions, minorTicksOptions, options) {
                var that = this,
                    i,
                    options = options || {},
                    ticksInfo = {
                        majorTickInterval: majorTicksOptions.tickInterval,
                        minorTickInterval: minorTicksOptions.tickInterval,
                        majorTicks: []
                    };
                majorTicksOptions.base = minorTicksOptions.base = options.base;
                majorTicksOptions.axisType = minorTicksOptions.axisType = options.axisType;
                majorTicksOptions.dataType = minorTicksOptions.dataType = options.dataType;
                that._initializeMajorTicksOptions(min, max, screenDelta, ticksInfo, majorTicksOptions);
                if (utils.isDefined(min) && utils.isDefined(max)) {
                    ticksInfo.majorTicks.push(min);
                    ticksInfo.majorTicks.push(that._nextTick(min, ticksInfo.majorTickInterval, {
                        min: min,
                        max: max,
                        setTicksAtUnitBeginning: majorTicksOptions.setTicksAtUnitBeginning,
                        dataType: options.dataType,
                        axisType: options.axisType,
                        base: options.base
                    }));
                    that._initializeMinorTicksOptions(min, max, screenDelta, ticksInfo, minorTicksOptions)
                }
                return ticksInfo
            },
            getFullTicks: function(min, max, screenDelta, majorTicksOptions, minorTicksOptions, options) {
                var that = this,
                    i,
                    options = options || {},
                    ticksInfo = {
                        customBoundaryTicks: [],
                        fullTicks: [],
                        majorTickInterval: majorTicksOptions.tickInterval,
                        majorTicks: [],
                        minorTickInterval: minorTicksOptions.tickInterval,
                        minorTicks: []
                    };
                majorTicksOptions.base = minorTicksOptions.base = options.base;
                majorTicksOptions.axisType = minorTicksOptions.axisType = options.axisType;
                majorTicksOptions.dataType = minorTicksOptions.dataType = options.dataType || that._getDataType(min);
                that._initializeMajorTicksOptions(min, max, screenDelta, ticksInfo, majorTicksOptions);
                ticksInfo.majorTicks = that.getTicks(majorTicksOptions);
                if (utils.isDefined(min) && utils.isDefined(max) && ticksInfo.majorTicks.length > 0) {
                    if (ticksInfo.majorTicks.autoArrangementStep && ticksInfo.majorTicks.autoArrangementStep > 1 && !utils.isDefined(minorTicksOptions.tickInterval) && !utils.isDefined(minorTicksOptions.minorTickCount))
                        minorTicksOptions.tickInterval = ticksInfo.minorTickInterval = majorTicksOptions.tickInterval;
                    that._initializeMinorTicksOptions(min, max, screenDelta, ticksInfo, minorTicksOptions);
                    that._addLeftBoudedTicks(ticksInfo, min, minorTicksOptions);
                    for (i = 0; i < ticksInfo.majorTicks.length - 1; i++) {
                        ticksInfo.fullTicks.push(ticksInfo.majorTicks[i]);
                        that._addMinorTicks(ticksInfo.majorTicks[i], ticksInfo.majorTicks[i + 1], ticksInfo, minorTicksOptions)
                    }
                    that._addRightBoudedTicks(ticksInfo, max, minorTicksOptions)
                }
                return ticksInfo
            },
            getTicks: function(options) {
                var that = this,
                    step,
                    maxDisplayValue,
                    ticks = options.customTicks ? options.customTicks : that._generateTicks(options);
                if (options.useTicksAutoArrangement) {
                    options.maxDisplayValueSize = that._getValueSize(ticks, options);
                    step = that._getAutoArrangementStep(ticks, options);
                    if (step > 1) {
                        if (utils.isDefined(options.tickInterval) || utils.isDefined(options.customTicks)) {
                            utils.isDefined(options.customTicks) && (ticks.tickInterval = options.tickInterval);
                            ticks = that._getAutoArrangementTicks(ticks, options, step)
                        }
                        else
                            ticks = that._generateTicks($.extend({}, options, {gridSpacingFactor: options.maxDisplayValueSize}));
                        ticks.autoArrangementStep = step
                    }
                    that._removeInvalidDatesWithUnitBegining(ticks, options)
                }
                return ticks
            }
        }
    })(jQuery, DevExpress);
    /*! Module viz-core, file numericTranslator.js */
    (function($, DX, undefined) {
        var utils = DX.utils,
            isDefined = utils.isDefined,
            getPower = utils.getPower,
            round = Math.round;
        DX.viz.core.numericTranslatorFunctions = {
            translate: function(bp) {
                var that = this,
                    canvasOptions = that._canvasOptions,
                    doubleError = canvasOptions.rangeDoubleError,
                    specialValue = that.translateSpecialCase(bp);
                if (isDefined(specialValue))
                    return specialValue;
                if (isNaN(bp) || bp.valueOf() + doubleError < canvasOptions.rangeMin || bp.valueOf() - doubleError > canvasOptions.rangeMax)
                    return null;
                return round(that._calculateProjection((bp - canvasOptions.rangeMinVisible) * canvasOptions.ratioOfCanvasRange))
            },
            untranslate: function(pos) {
                var canvasOptions = this._canvasOptions,
                    startPoint = canvasOptions.startPoint;
                if (pos < startPoint || pos > canvasOptions.endPoint)
                    return null;
                return this._calculateUnProjection((pos - startPoint) / canvasOptions.ratioOfCanvasRange)
            },
            getInterval: function() {
                return round(this._canvasOptions.ratioOfCanvasRange * (this._businessRange.interval || Math.abs(this._canvasOptions.rangeMax - this._canvasOptions.rangeMin)))
            }
        }
    })(jQuery, DevExpress);
    /*! Module viz-core, file datetimeTranslator.js */
    (function($, DX, undefined) {
        var core = DX.viz.core,
            numericTranslator = core.numericTranslatorFunctions;
        core.datetimeTranslatorFunctions = {
            translate: numericTranslator.translate,
            untranslate: function(pos) {
                var result = numericTranslator.untranslate.call(this, pos);
                return result === null ? result : new Date(result)
            },
            getInterval: numericTranslator.getInterval
        }
    })(jQuery, DevExpress);
    /*! Module viz-core, file categoryTranslator.js */
    (function($, DX, undefined) {
        var isDefined = DX.utils.isDefined,
            round = Math.round;
        function doubleRound(value) {
            return round(round(value * 10) / 10)
        }
        DX.viz.core.categoryTranslatorFunctions = {
            translate: function(category) {
                var that = this,
                    canvasOptions = that._canvasOptions,
                    categoryRecord = that._categoriesToPoints[category],
                    stickDelta,
                    specialValue = that.translateSpecialCase(category);
                if (isDefined(specialValue))
                    return specialValue;
                if (!categoryRecord)
                    return 0;
                stickDelta = that._businessRange.stick ? categoryRecord.index : categoryRecord.index + 0.5;
                return round(canvasOptions.startPoint + canvasOptions.interval * stickDelta)
            },
            untranslate: function(pos) {
                var that = this,
                    canvasOptions = that._canvasOptions,
                    startPoint = canvasOptions.startPoint,
                    categoriesLength = that._categories.length,
                    result = 0;
                if (pos < startPoint || pos > canvasOptions.endPoint)
                    return null;
                result = doubleRound((pos - startPoint) / canvasOptions.interval + (that._businessRange.stick ? 0.5 : 0) - 0.5);
                if (categoriesLength === result)
                    result--;
                if (canvasOptions.invert)
                    result = categoriesLength - result - 1;
                return that._categories[result]
            },
            getInterval: function() {
                return this._canvasOptions.interval
            }
        }
    })(jQuery, DevExpress);
    /*! Module viz-core, file logarithmicTranslator.js */
    (function($, DX, undefined) {
        var core = DX.viz.core,
            numericTranslator = core.numericTranslatorFunctions,
            utils = DX.utils,
            raiseTo = utils.raiseTo,
            getLog = utils.getLog;
        core.logarithmicTranslatorFunctions = {
            translate: function(bp) {
                var that = this,
                    specialValue = that.translateSpecialCase(bp);
                if (utils.isDefined(specialValue))
                    return specialValue;
                return numericTranslator.translate.call(that, getLog(bp, that._businessRange.base))
            },
            untranslate: function(pos) {
                var result = numericTranslator.untranslate.call(this, pos);
                return result === null ? result : raiseTo(result, this._businessRange.base)
            },
            getInterval: numericTranslator.getInterval
        }
    })(jQuery, DevExpress);
    /*! Module viz-core, file translator1D.js */
    (function(DX, undefined) {
        var _Number = Number;
        function Translator1D() {
            this.setDomain(arguments[0], arguments[1]).setCodomain(arguments[2], arguments[3])
        }
        Translator1D.prototype = {
            constructor: Translator1D,
            setDomain: function(domain1, domain2) {
                var that = this;
                that._domain1 = _Number(domain1);
                that._domain2 = _Number(domain2);
                that._domainDelta = that._domain2 - that._domain1;
                return that
            },
            setCodomain: function(codomain1, codomain2) {
                var that = this;
                that._codomain1 = _Number(codomain1);
                that._codomain2 = _Number(codomain2);
                that._codomainDelta = that._codomain2 - that._codomain1;
                return that
            },
            getDomain: function() {
                return [this._domain1, this._domain2]
            },
            getCodomain: function() {
                return [this._codomain1, this._codomain2]
            },
            getDomainStart: function() {
                return this._domain1
            },
            getDomainEnd: function() {
                return this._domain2
            },
            getCodomainStart: function() {
                return this._codomain1
            },
            getCodomainEnd: function() {
                return this._codomain2
            },
            getDomainRange: function() {
                return this._domainDelta
            },
            getCodomainRange: function() {
                return this._codomainDelta
            },
            translate: function(value) {
                var ratio = (_Number(value) - this._domain1) / this._domainDelta;
                return 0 <= ratio && ratio <= 1 ? this._codomain1 + ratio * this._codomainDelta : NaN
            },
            adjust: function(value) {
                var ratio = (_Number(value) - this._domain1) / this._domainDelta,
                    result = NaN;
                if (ratio < 0)
                    result = this._domain1;
                else if (ratio > 1)
                    result = this._domain2;
                else if (0 <= ratio && ratio <= 1)
                    result = _Number(value);
                return result
            }
        };
        DX.viz.core.Translator1D = Translator1D
    })(DevExpress);
    /*! Module viz-core, file translator2D.js */
    (function($, DX, undefined) {
        var core = DX.viz.core,
            utils = DX.utils,
            getLog = utils.getLog,
            getPower = utils.getPower,
            raiseTo = utils.raiseTo,
            isDefined = utils.isDefined,
            CANVAS_PROP = ["width", "height", "left", "top", "bottom", "right"],
            NUMBER_EQUALITY_CORRECTION = 1,
            DATETIME_EQUALITY_CORRECTION = 60000;
        var validateCanvas = function(canvas) {
                $.each(CANVAS_PROP, function(_, prop) {
                    canvas[prop] = parseInt(canvas[prop]) || 0
                });
                return canvas
            };
        var raiseToFlooredLog = function(value, base, correction) {
                return raiseTo(Math.floor(getLog(value, base)) + (correction || 0), base)
            };
        var makeCategoriesToPoints = function(categories, invert) {
                var categoriesToPoints = {},
                    category,
                    length = categories.length,
                    i;
                for (i = 0; i < length; i++) {
                    category = categories[i];
                    categoriesToPoints[category] = {
                        name: category,
                        index: invert ? length - 1 - i : i
                    }
                }
                return categoriesToPoints
            };
        var validateBusinessRange = function(businessRange) {
                function validate(valueSelector, baseValueSeletor) {
                    if (!isDefined(businessRange[valueSelector]) && isDefined(businessRange[baseValueSeletor]))
                        businessRange[valueSelector] = businessRange[baseValueSeletor]
                }
                validate("minVisible", "min");
                validate("maxVisible", "max");
                return businessRange
            };
        core.Translator2D = function(businessRange, canvas, options) {
            var that = this;
            that._options = $.extend(that._options || {}, options);
            that._canvas = validateCanvas(canvas);
            that.updateBusinessRange(businessRange)
        };
        $.extend(core.Translator2D.prototype, {
            reinit: function() {
                var that = this,
                    range = that._businessRange,
                    categories = range.categories || [],
                    script = {},
                    canvasOptions = that._prepareCanvasOptions(),
                    interval,
                    correctedCategoriesCount;
                switch (range.axisType) {
                    case"logarithmic":
                        script = core.logarithmicTranslatorFunctions;
                        break;
                    case"discrete":
                        script = core.categoryTranslatorFunctions;
                        that._categories = categories;
                        correctedCategoriesCount = categories.length - (range.stick ? 1 : 0);
                        if (correctedCategoriesCount > 0)
                            interval = canvasOptions.canvasLength / correctedCategoriesCount;
                        else
                            interval = canvasOptions.canvasLength;
                        canvasOptions.interval = interval;
                        that._categoriesToPoints = makeCategoriesToPoints(categories, canvasOptions.invert);
                        break;
                    default:
                        if (range.dataType === "datetime")
                            script = core.datetimeTranslatorFunctions;
                        else
                            script = core.numericTranslatorFunctions
                }
                that.translate = script.translate;
                that.untranslate = script.untranslate;
                that.getInterval = script.getInterval
            },
            _getCanvasBounds: function(range) {
                var min = range.min,
                    max = range.max,
                    minVisible = range.minVisible,
                    maxVisible = range.maxVisible,
                    newMin,
                    newMax,
                    base = range.base,
                    isDateTime = utils.isDate(max) || utils.isDate(min),
                    correction = isDateTime ? DATETIME_EQUALITY_CORRECTION : NUMBER_EQUALITY_CORRECTION;
                if (isDefined(min) && isDefined(max) && min.valueOf() === max.valueOf()) {
                    newMin = min.valueOf() - correction;
                    newMax = max.valueOf() + correction;
                    if (isDateTime) {
                        min = new Date(newMin);
                        max = new Date(newMax)
                    }
                    else {
                        min = min !== 0 ? newMin : 0;
                        max = newMax
                    }
                }
                if (isDefined(minVisible) && isDefined(maxVisible) && minVisible.valueOf() === maxVisible.valueOf()) {
                    newMin = minVisible.valueOf() - correction;
                    newMax = maxVisible.valueOf() + correction;
                    if (isDateTime) {
                        minVisible = newMin < min.valueOf() ? min : new Date(newMin);
                        maxVisible = newMax > max.valueOf() ? max : new Date(newMax)
                    }
                    else {
                        if (minVisible !== 0)
                            minVisible = newMin < min ? min : newMin;
                        maxVisible = newMax > max ? max : newMax
                    }
                }
                if (range.axisType === 'logarithmic') {
                    maxVisible = getLog(maxVisible, base);
                    minVisible = getLog(minVisible, base);
                    min = getLog(min, base);
                    max = getLog(max, base)
                }
                return {
                        base: base,
                        rangeMin: min,
                        rangeMax: max,
                        rangeMinVisible: minVisible,
                        rangeMaxVisible: maxVisible
                    }
            },
            _prepareCanvasOptions: function() {
                var that = this,
                    rangeMin,
                    rangeMax,
                    rangeMinVisible,
                    rangeMaxVisible,
                    businessRange = that._businessRange,
                    canvasOptions = that._canvasOptions = that._getCanvasBounds(businessRange),
                    length;
                if (that._options.direction === "horizontal") {
                    canvasOptions.startPoint = that._canvas.left;
                    length = that._canvas.width;
                    canvasOptions.endPoint = that._canvas.width - that._canvas.right;
                    canvasOptions.invert = businessRange.invert
                }
                else {
                    canvasOptions.startPoint = that._canvas.top;
                    length = that._canvas.height;
                    canvasOptions.endPoint = that._canvas.height - that._canvas.bottom;
                    canvasOptions.invert = !businessRange.invert
                }
                that.canvasLength = canvasOptions.canvasLength = canvasOptions.endPoint - canvasOptions.startPoint;
                canvasOptions.rangeDoubleError = Math.pow(10, getPower(canvasOptions.rangeMax - canvasOptions.rangeMin) - getPower(length) - 2);
                canvasOptions.ratioOfCanvasRange = canvasOptions.canvasLength / (canvasOptions.rangeMaxVisible - canvasOptions.rangeMinVisible);
                return canvasOptions
            },
            updateBusinessRange: function(businessRange) {
                this._businessRange = validateBusinessRange(businessRange);
                this.reinit()
            },
            getBusinessRange: function() {
                return this._businessRange
            },
            getCanvasVisibleArea: function() {
                return {
                        min: this._canvasOptions.startPoint,
                        max: this._canvasOptions.endPoint
                    }
            },
            translateSpecialCase: function(value) {
                var that = this,
                    canvasOptions = that._canvasOptions,
                    startPoint = canvasOptions.startPoint,
                    endPoint = canvasOptions.endPoint,
                    range = that._businessRange,
                    minVisible = range.minVisible,
                    maxVisible = range.maxVisible,
                    invert,
                    result = null;
                switch (value) {
                    case"canvas_position_default":
                        if (minVisible <= 0 && maxVisible >= 0)
                            result = that.translate(0);
                        else {
                            invert = range.invert ^ (minVisible <= 0 && maxVisible <= 0);
                            if (that._options.direction === "horizontal")
                                result = invert ? endPoint : startPoint;
                            else
                                result = invert ? startPoint : endPoint
                        }
                        break;
                    case"canvas_position_left":
                    case"canvas_position_top":
                        result = startPoint;
                        break;
                    case"canvas_position_center":
                    case"canvas_position_middle":
                        result = startPoint + canvasOptions.canvasLength / 2;
                        break;
                    case"canvas_position_right":
                    case"canvas_position_bottom":
                        result = endPoint;
                        break;
                    case"canvas_position_start":
                        result = range.invert ? endPoint : startPoint;
                        break;
                    case"canvas_position_end":
                        result = range.invert ? startPoint : endPoint;
                        break
                }
                return result
            },
            _calculateProjection: function(distance) {
                var canvasOptions = this._canvasOptions;
                return canvasOptions.invert ? canvasOptions.endPoint - distance : canvasOptions.startPoint + distance
            },
            _calculateUnProjection: function(distance) {
                var canvasOptions = this._canvasOptions;
                return canvasOptions.invert ? canvasOptions.rangeMaxVisible.valueOf() - distance : canvasOptions.rangeMinVisible.valueOf() + distance
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file rectangle.js */
    (function(DX, undefined) {
        var isFinite = window.isFinite;
        DX.viz.core.Rectangle = DX.Class.inherit({
            ctor: function(options) {
                var that = this;
                options = options || {};
                that.left = Number(options.left) || 0;
                that.right = Number(options.right) || 0;
                that.top = Number(options.top) || 0;
                that.bottom = Number(options.bottom) || 0
            },
            width: function() {
                return this.right - this.left
            },
            height: function() {
                return this.bottom - this.top
            },
            horizontalMiddle: function() {
                return (this.left + this.right) / 2
            },
            verticalMiddle: function() {
                return (this.top + this.bottom) / 2
            },
            raw: function() {
                var that = this;
                return {
                        left: that.left,
                        top: that.top,
                        right: that.right,
                        bottom: that.bottom
                    }
            },
            clone: function() {
                return new this.constructor(this.raw())
            },
            move: function(dx, dy) {
                var result = this.clone();
                if (isFinite(dx) && isFinite(dy)) {
                    result.left += Number(dx);
                    result.right += Number(dx);
                    result.top += Number(dy);
                    result.bottom += Number(dy)
                }
                return result
            },
            inflate: function(dx, dy) {
                var result = this.clone();
                if (isFinite(dx) && isFinite(dy)) {
                    result.left -= Number(dx);
                    result.right += Number(dx);
                    result.top -= Number(dy);
                    result.bottom += Number(dy)
                }
                return result
            },
            scale: function(factor) {
                var that = this;
                if (factor > 0)
                    return that.inflate(that.width() * (factor - 1) / 2, that.height() * (factor - 1) / 2);
                return that.clone()
            }
        })
    })(DevExpress);
    /*! Module viz-core, file themes.js */
    (function($, DX, undefined) {
        var viz = DX.viz,
            core = viz.core;
        var currentThemeId = 0;
        var findThemeId = function(themeName) {
                var themeId,
                    themes = viz.themes;
                for (themeId = 0; themeId < themes.length; themeId++)
                    if (themes[themeId].name === themeName)
                        return themeId;
                return -1
            };
        core.findTheme = function(themeName) {
            var themeId = findThemeId(themeName),
                themes = viz.themes;
            if (themeId < 0)
                themeId = currentThemeId;
            return themes[themeId]
        };
        core.currentTheme = function(themeName, colorScheme, version) {
            var themeId = -1,
                themes = viz.themes;
            if (themeName === undefined)
                return themes[currentThemeId].name;
            else {
                if (version && colorScheme)
                    themeId = findThemeId(themeName + ':' + version + '-' + colorScheme);
                if (themeId < 0 && version)
                    themeId = findThemeId(themeName + ':' + version);
                if (colorScheme && themeId < 0)
                    themeId = findThemeId(themeName + '-' + colorScheme);
                if (themeId < 0)
                    themeId = findThemeId(themeName);
                currentThemeId = themeId >= 0 ? themeId : 0
            }
        };
        core.registerTheme = function(theme, basedOnThemeName) {
            var baseTheme,
                extendedTheme,
                themes = viz.themes;
            if (!theme || !theme.name || !core.findTheme(theme.name))
                return;
            baseTheme = core.findTheme(basedOnThemeName);
            if (baseTheme) {
                extendedTheme = $.extend(true, {}, baseTheme, theme);
                themes.push(extendedTheme)
            }
            else
                themes.push(theme)
        }
    })(jQuery, DevExpress);
    /*! Module viz-core, file palette.js */
    (function(DX, $, undefined) {
        var _String = window.String,
            _Color = DX.Color,
            _isArray = DX.utils.isArray,
            _isString = DX.utils.isString,
            _extend = $.extend;
        var palettes = {
                'default': {
                    simpleSet: ['#5f8b95', '#ba4d51', '#af8a53', '#955f71', '#859666', '#7e688c'],
                    indicatingSet: ['#a3b97c', '#e1b676', '#ec7f83'],
                    gradientSet: ['#5f8b95', '#ba4d51']
                },
                'harmony light': {
                    simpleSet: ['#fcb65e', '#679ec5', '#ad79ce', '#7abd5c', '#e18e92', '#b6d623', '#b7abea', '#85dbd5'],
                    indicatingSet: ['#b6d623', '#fcb65e', '#e18e92'],
                    gradientSet: ['#7abd5c', '#fcb65e']
                },
                'soft pastel': {
                    simpleSet: ['#60a69f', '#78b6d9', '#6682bb', '#a37182', '#eeba69', '#90ba58', '#456c68', '#7565a4'],
                    indicatingSet: ['#90ba58', '#eeba69', '#a37182'],
                    gradientSet: ['#78b6d9', '#eeba69']
                },
                pastel: {
                    simpleSet: ['#bb7862', '#70b3a1', '#bb626a', '#057d85', '#ab394b', '#dac599', '#153459', '#b1d2c6'],
                    indicatingSet: ['#70b3a1', '#dac599', '#bb626a'],
                    gradientSet: ['#bb7862', '#70b3a1']
                },
                bright: {
                    simpleSet: ['#70c92f', '#f8ca00', '#bd1550', '#e97f02', '#9d419c', '#7e4452', '#9ab57e', '#36a3a6'],
                    indicatingSet: ['#70c92f', '#f8ca00', '#bd1550'],
                    gradientSet: ['#e97f02', '#f8ca00']
                },
                soft: {
                    simpleSet: ['#cbc87b', '#9ab57e', '#e55253', '#7e4452', '#e8c267', '#565077', '#6babac', '#ad6082'],
                    indicatingSet: ['#9ab57e', '#e8c267', '#e55253'],
                    gradientSet: ['#9ab57e', '#e8c267']
                },
                ocean: {
                    simpleSet: ['#75c099', '#acc371', '#378a8a', '#5fa26a', '#064970', '#38c5d2', '#00a7c6', '#6f84bb'],
                    indicatingSet: ['#c8e394', '#7bc59d', '#397c8b'],
                    gradientSet: ['#acc371', '#38c5d2']
                },
                vintage: {
                    simpleSet: ['#dea484', '#efc59c', '#cb715e', '#eb9692', '#a85c4c', '#f2c0b5', '#c96374', '#dd956c'],
                    indicatingSet: ['#ffe5c6', '#f4bb9d', '#e57660'],
                    gradientSet: ['#efc59c', '#cb715e']
                },
                violet: {
                    simpleSet: ['#d1a1d1', '#eeacc5', '#7b5685', '#7e7cad', '#a13d73', '#5b41ab', '#e287e2', '#689cc1'],
                    indicatingSet: ['#d8e2f6', '#d0b2da', '#d56a8a'],
                    gradientSet: ['#eeacc5', '#7b5685']
                }
            };
        var currentPaletteName = 'default';
        function currentPalette(name) {
            if (name === undefined)
                return currentPaletteName;
            else {
                name = String(name).toLowerCase();
                currentPaletteName = name in palettes ? name : 'default'
            }
        }
        function getPalette(palette, parameters) {
            var result;
            if (_isArray(palette))
                result = palette;
            else {
                parameters = parameters || {};
                var type = parameters.type || 'simpleSet';
                if (_isString(palette)) {
                    var name = palette.toLowerCase(),
                        baseContainer = palettes[name],
                        themedContainer = parameters.theme && palettes[name + '_' + _String(parameters.theme).toLowerCase()];
                    result = themedContainer && themedContainer[type] || baseContainer && baseContainer[type]
                }
                if (!result)
                    result = palettes[currentPaletteName][type]
            }
            return result ? result.slice(0) : null
        }
        function registerPalette(name, palette, theme) {
            var item = {};
            if (_isArray(palette))
                item.simpleSet = palette.slice(0);
            else if (palette) {
                item.simpleSet = _isArray(palette.simpleSet) ? palette.simpleSet.slice(0) : undefined;
                item.indicatingSet = _isArray(palette.indicatingSet) ? palette.indicatingSet.slice(0) : undefined;
                item.gradientSet = _isArray(palette.gradientSet) ? palette.gradientSet.slice(0) : undefined
            }
            if (item.simpleSet || item.indicatingSet || item.gradientSet) {
                var paletteName = _String(name).toLowerCase();
                if (theme)
                    paletteName = paletteName + '_' + _String(theme).toLowerCase();
                _extend(palettes[paletteName] = palettes[paletteName] || {}, item)
            }
        }
        function RingBuf(buf) {
            var ind = 0;
            this.next = function() {
                var res = buf[ind++];
                if (ind == buf.length)
                    this.reset();
                return res
            };
            this.reset = function() {
                ind = 0
            }
        }
        function Palette(palette, parameters) {
            parameters = parameters || {};
            this._originalPalette = getPalette(palette, parameters);
            var stepHighlight = parameters ? parameters.stepHighlight || 0 : 0;
            this._paletteSteps = new RingBuf([0, stepHighlight, -stepHighlight]);
            this._resetPalette()
        }
        _extend(Palette.prototype, {
            dispose: function() {
                this._originalPalette = this._palette = this._paletteSteps = null;
                return this
            },
            getNextColor: function() {
                var that = this;
                if (that._currentColor >= that._palette.length)
                    that._resetPalette();
                return that._palette[that._currentColor++]
            },
            _resetPalette: function() {
                var that = this;
                that._currentColor = 0;
                var step = that._paletteSteps.next(),
                    originalPalette = that._originalPalette;
                if (step) {
                    var palette = that._palette = [],
                        i = 0,
                        ii = originalPalette.length;
                    for (; i < ii; ++i)
                        palette[i] = getNewColor(originalPalette[i], step)
                }
                else
                    that._palette = originalPalette.slice(0)
            },
            reset: function() {
                this._paletteSteps.reset();
                this._resetPalette();
                return this
            }
        });
        function getNewColor(currentColor, step) {
            var newColor = new _Color(currentColor).alter(step),
                lightness = getLightness(newColor);
            if (lightness > 200 || lightness < 55)
                newColor = new _Color(currentColor).alter(-step / 2);
            return newColor.toHex()
        }
        function getLightness(color) {
            return color.r * 0.3 + color.g * 0.59 + color.b * 0.11
        }
        function GradientPalette(source, size) {
            var palette = getPalette(source, {type: 'gradientSet'});
            palette = size > 0 ? createGradientColors(palette[0], palette[1], size) : [];
            this.getColor = function(index) {
                return palette[index] || null
            };
            this._DEBUG_source = source;
            this._DEBUG_size = size
        }
        function createGradientColors(start, end, count) {
            var startColor = new _Color(start),
                endColor = new _Color(end);
            if (count === 1)
                return [startColor.blend(endColor, 0.5).toHex()];
            else {
                var list = [],
                    step = 1 / (count - 1),
                    i,
                    ii = count;
                list.push(0);
                for (i = 1; i < ii - 1; ++i)
                    list.push(step * i);
                list.push(1);
                for (i = 0; i < ii; ++i)
                    list[i] = startColor.blend(endColor, list[i]).toHex();
                return list
            }
        }
        _extend(DX.viz.core, {
            registerPalette: registerPalette,
            getPalette: getPalette,
            Palette: Palette,
            GradientPalette: GradientPalette,
            currentPalette: currentPalette
        });
        DX.viz.core._DEBUG_palettes = palettes
    })(DevExpress, jQuery);
    /*! Module viz-core, file baseThemeManager.js */
    (function(DX, $, undefined) {
        var _isString = DX.utils.isString,
            _findTheme = DX.viz.core.findTheme,
            _extend = $.extend,
            _each = $.each;
        DX.viz.core.BaseThemeManager = DX.Class.inherit({
            dispose: function() {
                this._theme = this._font = null;
                return this
            },
            setTheme: function(theme) {
                theme = theme || {};
                var that = this,
                    themeObj = _findTheme(_isString(theme) ? theme : theme.name);
                that._themeName = themeObj.name;
                that._font = _extend({}, themeObj.font, theme.font);
                that._themeSection && _each(that._themeSection.split('.'), function(_, path) {
                    themeObj = _extend(true, {}, themeObj[path], that._IE8 ? themeObj[path + 'IE8'] : {})
                });
                that._theme = _extend(true, {}, themeObj, _isString(theme) ? {} : theme);
                that._initializeTheme();
                return that
            },
            theme: function() {
                return this._theme
            },
            themeName: function() {
                return this._themeName
            },
            _initializeTheme: function(){},
            _initializeFont: function(font) {
                _extend(font, this._font, _extend({}, font))
            }
        })
    })(DevExpress, jQuery);
    /*! Module viz-core, file textCloud.js */
    (function(DX, undefined) {
        var min = Math.min;
        DX.viz.core.TextCloud = DX.Class.inherit(function() {
            var DEFAULT_OPTIONS = {
                    horMargin: 8,
                    verMargin: 4,
                    tailLength: 10
                };
            var COEFFICIENTS_MAP = {};
            COEFFICIENTS_MAP['right-bottom'] = COEFFICIENTS_MAP['rb'] = [0, -1, -1, 0, 0, 1, 1, 0];
            COEFFICIENTS_MAP['bottom-right'] = COEFFICIENTS_MAP['br'] = [-1, 0, 0, -1, 1, 0, 0, 1];
            COEFFICIENTS_MAP['left-bottom'] = COEFFICIENTS_MAP['lb'] = [0, -1, 1, 0, 0, 1, -1, 0];
            COEFFICIENTS_MAP['bottom-left'] = COEFFICIENTS_MAP['bl'] = [1, 0, 0, -1, -1, 0, 0, 1];
            COEFFICIENTS_MAP['left-top'] = COEFFICIENTS_MAP['lt'] = [0, 1, 1, 0, 0, -1, -1, 0];
            COEFFICIENTS_MAP['top-left'] = COEFFICIENTS_MAP['tl'] = [1, 0, 0, 1, -1, 0, 0, -1];
            COEFFICIENTS_MAP['right-top'] = COEFFICIENTS_MAP['rt'] = [0, 1, -1, 0, 0, -1, 1, 0];
            COEFFICIENTS_MAP['top-right'] = COEFFICIENTS_MAP['tr'] = [-1, 0, 0, 1, 1, 0, 0, -1];
            return {
                    setup: function(options) {
                        var that = this,
                            ops = $.extend({}, DEFAULT_OPTIONS, options),
                            x = ops.x,
                            y = ops.y,
                            type = COEFFICIENTS_MAP[ops.type],
                            cloudWidth = ops.textWidth + 2 * ops.horMargin,
                            cloudHeight = ops.textHeight + 2 * ops.verMargin,
                            tailWidth = ops.tailLength,
                            tailHeight = tailWidth,
                            cx = x,
                            cy = y;
                        if (type[0] & 1)
                            tailHeight = min(tailHeight, cloudHeight / 3);
                        else
                            tailWidth = min(tailWidth, cloudWidth / 3);
                        that._points = [x, y, x += type[0] * (cloudWidth + tailWidth), y += type[1] * (cloudHeight + tailHeight), x += type[2] * cloudWidth, y += type[3] * cloudHeight, x += type[4] * cloudWidth, y += type[5] * cloudHeight, x += type[6] * (cloudWidth - tailWidth), y += type[7] * (cloudHeight - tailHeight)];
                        that._cx = cx + type[0] * tailWidth + (type[0] + type[2]) * cloudWidth / 2;
                        that._cy = cy + type[1] * tailHeight + (type[1] + type[3]) * cloudHeight / 2;
                        that._cloudWidth = cloudWidth;
                        that._cloudHeight = cloudHeight;
                        that._tailLength = ops.tailLength;
                        return that
                    },
                    points: function() {
                        return this._points.slice(0)
                    },
                    cx: function() {
                        return this._cx
                    },
                    cy: function() {
                        return this._cy
                    },
                    width: function() {
                        return this._cloudWidth
                    },
                    height: function() {
                        return this._cloudHeight
                    },
                    tailLength: function() {
                        return this._tailLength
                    }
                }
        }())
    })(DevExpress);
    /*! Module viz-core, file parseUtils.js */
    (function($, DX) {
        var viz = DX.viz,
            core = viz.core,
            Class = DX.Class,
            isDefined = DX.utils.isDefined;
        var parseUtils = Class.inherit({
                ctor: function(options) {
                    options = options || {};
                    this._incidentOccured = $.isFunction(options.incidentOccured) ? options.incidentOccured : $.noop
                },
                correctValueType: function(type) {
                    return type === 'numeric' || type === 'datetime' || type === 'string' ? type : ''
                },
                _parsers: {
                    string: function(val) {
                        return isDefined(val) ? '' + val : val
                    },
                    numeric: function(val) {
                        if (!isDefined(val))
                            return val;
                        var parsedVal = Number(val);
                        if (isNaN(parsedVal))
                            parsedVal = undefined;
                        return parsedVal
                    },
                    datetime: function(val) {
                        if (!isDefined(val))
                            return val;
                        var parsedVal,
                            numVal = Number(val);
                        if (!isNaN(numVal))
                            parsedVal = new Date(numVal);
                        else
                            parsedVal = new Date(val);
                        if (isNaN(Number(parsedVal)))
                            parsedVal = undefined;
                        return parsedVal
                    }
                },
                getParser: function(valueType, entity) {
                    var that = this,
                        parser,
                        message = 'valueType is unknown.';
                    if (entity)
                        message = 'The type specified as the "valueType" field of the ' + entity + ' configuration object is unknown.';
                    valueType = that.correctValueType(valueType);
                    parser = that._parsers[valueType];
                    if (!parser)
                        this._incidentOccured.call(null, message);
                    return parser || $.noop
                }
            });
        core.ParseUtils = parseUtils
    })(jQuery, DevExpress);
    /*! Module viz-core, file utils.js */
    (function($, DX) {
        var core = DX.viz.core,
            math = Math;
        core.utils = {decreaseGaps: function(object, keys, decrease) {
                var arrayGaps,
                    eachDecrease,
                    middleValue;
                do {
                    arrayGaps = $.map(keys, function(key) {
                        return object[key] ? object[key] : null
                    });
                    middleValue = math.ceil(decrease / arrayGaps.length);
                    arrayGaps.push(middleValue);
                    eachDecrease = math.min.apply(null, arrayGaps);
                    $.each(keys, function(_, key) {
                        if (object[key]) {
                            object[key] -= eachDecrease;
                            decrease -= eachDecrease
                        }
                    })
                } while (decrease > 0 && arrayGaps.length > 1);
                return decrease
            }}
    })(jQuery, DevExpress);
    /*! Module viz-core, file loadIndicator.js */
    (function($, DX) {
        var viz = DX.viz,
            core = viz.core,
            ANIMATION_SETTINGS = {
                easing: 'linear',
                duration: 150
            },
            INVISIBLE_POINT = {
                x: -10000,
                y: -10000
            };
        var applySettings = function(element, settings, animate, complete) {
                var prevAnimation = element.animation;
                if (prevAnimation) {
                    prevAnimation.options.complete = null;
                    prevAnimation.stop()
                }
                if (animate)
                    element.animate(settings, {complete: complete});
                else {
                    element.applySettings(settings);
                    complete && complete()
                }
            };
        core.LoadIndicator = DX.Class.inherit({
            ctor: function(options, widgetContainer) {
                var that = this;
                that._$widgetContainer = $(widgetContainer);
                that._$container = $('<div>', {css: {
                        position: 'relative',
                        height: 0,
                        padding: 0,
                        margin: 0,
                        border: 0
                    }}).appendTo(that._$widgetContainer);
                that._updateContainer();
                that.applyOptions(options);
                that._endLoadingCompleteHandler = function() {
                    that._endLoad = false;
                    that._externalComplete && that._externalComplete();
                    that._externalComplete = null;
                    that._onCompleteAction && that[that._onCompleteAction]();
                    that._onCompleteAction = null
                };
                that._$container.hide()
            },
            _updateRenderer: function(width, height, top) {
                if (this._renderer)
                    this._renderer.recreateCanvas(width, height);
                else if (this._$container.get(0)) {
                    this._renderer = new viz.renderers.Renderer({
                        width: width,
                        height: height,
                        animation: ANIMATION_SETTINGS
                    });
                    this._renderer.draw(this._$container[0])
                }
                this._renderer && this._renderer.getRoot().applySettings({style: {
                        position: 'absolute',
                        top: top,
                        left: 0
                    }});
                return this._renderer
            },
            applyOptions: function(options, width, height) {
                var pane = this._pane;
                if (pane && options) {
                    pane.rect.applySettings({fill: options.backgroundColor});
                    pane.text.applySettings({
                        font: options.font,
                        text: options.text
                    })
                }
                if (this.isShown && (width || height))
                    this._updateContainer(width, height)
            },
            _draw: function() {
                var pane,
                    renderer = this._renderer;
                if (renderer) {
                    pane = this._pane = {};
                    pane.rect = renderer.createRect(0, 0, 0, 0, 0, {opacity: 0}).append();
                    pane.text = renderer.createText('', 0, 0, {
                        align: 'center',
                        translateX: INVISIBLE_POINT.x,
                        translateY: INVISIBLE_POINT.y
                    }).append()
                }
            },
            _updateContainer: function(width, height) {
                var that = this,
                    $widgetContainer = that._$widgetContainer,
                    canvasTop;
                width = width || $widgetContainer.width();
                height = height || $widgetContainer.height();
                if ($widgetContainer.get(0))
                    canvasTop = $widgetContainer.offset().top - that._$container.offset().top;
                else
                    canvasTop = -height;
                that._updateRenderer(width, height, canvasTop);
                if (!that._pane)
                    that._draw();
                else {
                    that._pane.rect.applySettings({
                        width: width,
                        height: height
                    });
                    that._pane.text.move(width / 2, height / 2)
                }
            },
            dispose: function() {
                this._$widgetContainer = null;
                this._$container.remove().detach();
                this._$container = null;
                this._renderer.dispose();
                this._renderer = null;
                this._pane = null
            },
            toForeground: function() {
                this._$container.appendTo(this._$widgetContainer)
            },
            show: function(width, height) {
                if (this._endLoad) {
                    this._onCompleteAction = 'show';
                    return
                }
                this._$container.show();
                this._updateContainer(width, height);
                applySettings(this._pane.rect, {opacity: 0.85}, true);
                this.isShown = true
            },
            endLoading: function(complete, disableAnimation) {
                this._externalComplete = complete;
                if (this._endLoad)
                    return;
                if (this.isShown) {
                    this._endLoad = true;
                    applySettings(this._pane.rect, {opacity: 1}, !disableAnimation, this._endLoadingCompleteHandler)
                }
                else
                    complete && complete()
            },
            hide: function() {
                var that = this;
                if (this._endLoad) {
                    this._onCompleteAction = 'hide';
                    return
                }
                if (this.isShown) {
                    this._pane.text.move(INVISIBLE_POINT.x, INVISIBLE_POINT.y);
                    applySettings(that._pane.rect, {opacity: 0}, true, function() {
                        that._$container.hide()
                    });
                    this.isShown = false
                }
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file tooltip.js */
    (function($, DX, undefined) {
        var ARROW_WIDTH = 20,
            MAX_SHADOW_SIZE = 10,
            formatHelper = DX.formatHelper,
            X_INTERVAL = 15,
            _max = Math.max,
            _round = Math.round,
            _isFunction = DX.utils.isFunction,
            _isDefined = DX.utils.isDefined,
            _extend = $.extend,
            FORMAT_PRECISION = {
                argument: ['argumentFormat', 'argumentPrecision'],
                percent: ['percent', 'percentPrecision'],
                value: ['format', 'precision']
            },
            VISIBLE = {visibility: 'visible'},
            HIDDEN = {visibility: 'hidden'},
            LEFT = 'left',
            RIGHT = 'right';
        DX.viz.core.Tooltip = DX.Class.inherit({
            ctor: function(options, group, renderer) {
                this._state = {};
                this._options = {};
                this._renderer = renderer;
                this._group = group;
                this._cloud = renderer.createArea();
                this._textGroup = renderer.createGroup();
                if (!$.isEmptyObject(options))
                    this.update(options);
                this._createTextContent()
            },
            dispose: function() {
                this._shadow.dispose();
                this._shadow = null;
                this._cloud = null;
                this._text = null;
                this._group = null;
                this._options = null;
                this._renderer = null;
                this._tooltipTextArray = null;
                this._textGroup = null;
                return this
            },
            update: function(options) {
                options = options || {};
                var that = this,
                    group = that._group,
                    shadowOptions = options.shadow || {},
                    shadow = that._shadow = that._shadow || that._renderer.createFilter('shadow').applySettings({
                        width: '200%',
                        height: '200%',
                        color: shadowOptions.color,
                        opacity: shadowOptions.opacity,
                        dx: shadowOptions.offsetX,
                        dy: shadowOptions.offsetY,
                        blur: shadowOptions.blur,
                        x: '-50%',
                        y: '-50%'
                    }),
                    borderSettings = options.border,
                    shapeSettings = _extend({
                        opacity: options.opacity,
                        filter: shadow.append().ref
                    }, borderSettings && borderSettings.visible ? {
                        strokeWidth: borderSettings.width,
                        stroke: borderSettings.color,
                        strokeOpacity: borderSettings.opacity,
                        dashStyle: borderSettings.dashStyle
                    } : {
                        strokeWidth: null,
                        stroke: null
                    }),
                    textSettings = _extend({}, {
                        align: 'center',
                        font: options.font
                    }, options.text);
                that._options = options;
                that.setSize(options.canvasWidth, options.canvasHeight);
                that._customizeTooltip = _isFunction(options.customizeTooltip) ? options.customizeTooltip : null;
                if (!that._customizeTooltip && _isFunction(options.customizeText))
                    that._customizeTooltip = function() {
                        return {text: options.customizeText.apply(this, arguments)}
                    };
                that._cloud.applySettings(shapeSettings).append(group);
                that._text && options.font && that._text.applySettings({font: {size: options.font.size}});
                that._textGroup.applySettings(textSettings).append(group);
                that.hide();
                return that
            },
            formatValue: function(value, specialFormat) {
                var formatObj = FORMAT_PRECISION[specialFormat || 'value'],
                    format = formatObj[0] in this._options ? this._options[formatObj[0]] : specialFormat;
                return formatHelper.format(value, format, this._options[formatObj[1]] || 0)
            },
            prepare: function(formatObject, params, defaultTextValueField) {
                this._state = this._state || {};
                defaultTextValueField = defaultTextValueField || 'valueText';
                var defaultText = formatObject[defaultTextValueField] || '';
                _extend(this._state, params);
                if (this._customizeTooltip) {
                    var customize = this._customizeTooltip.call(formatObject, formatObject);
                    customize = $.isPlainObject(customize) ? customize : {};
                    if ('text' in customize)
                        this._state.text = _isDefined(customize.text) ? String(customize.text) : '';
                    else {
                        if ($.isArray(defaultText)) {
                            this._options._justify = true;
                            this._createTextContent();
                            defaultText = defaultText.join('<br/>')
                        }
                        this._state.text = defaultText
                    }
                    this._state.color = customize.color || this._options.color
                }
                else {
                    this._state.text = defaultText;
                    this._state.color = this._options.color
                }
                if (this._options._justify)
                    this._state.text = this._state.text.split('<br/>');
                if (this._state.visibility == VISIBLE && !!this._state.text)
                    this.show();
                return !!this._state.text
            },
            enabled: function() {
                return !!this._options.enabled
            },
            formatColorTooltip: function(that) {
                return that._customizeTooltip && that._customizeTooltip.call(this, this)
            },
            _getData: function() {
                var that = this,
                    x = that._state.x,
                    y = that._state.y,
                    xt = x,
                    yt = y,
                    align = 'center',
                    points = [],
                    bbox = that._state.textBBox,
                    options = that._options,
                    paddingLeftRight = options.paddingLeftRight,
                    paddingTopBottom = options.paddingTopBottom,
                    arrowLength = options.arrowLength > 0 ? options.arrowLength : 0,
                    horPosition = options.cloudHorizontalPosition,
                    verPosition = options.cloudVerticalPosition,
                    isHorPositionDefined = horPosition !== undefined && horPosition !== null,
                    isVerPositionDefined = verPosition !== undefined && verPosition !== null,
                    cloudWidth = bbox.width + paddingLeftRight * 2,
                    cloudHeight = bbox.height + paddingTopBottom * 2,
                    updatedText;
                updatedText = that._checkWidthText(cloudWidth, cloudHeight);
                if (updatedText) {
                    that._state.textBBox = bbox = updatedText.bbox;
                    cloudWidth = updatedText.cloudWidth;
                    cloudHeight = updatedText.cloudHeight;
                    paddingLeftRight = updatedText.paddingLeftRight;
                    paddingTopBottom = updatedText.paddingTopBottom
                }
                if (isHorPositionDefined ? horPosition === RIGHT : cloudWidth / 2 > x) {
                    points = that._setArrowLeft(cloudWidth, cloudHeight, arrowLength, x, y);
                    align = LEFT;
                    xt += paddingLeftRight
                }
                else if (isHorPositionDefined ? horPosition === LEFT : x + cloudWidth / 2 > that._canvasWidth) {
                    points = that._setArrowRight(cloudWidth, cloudHeight, arrowLength, x, y);
                    align = RIGHT;
                    xt -= paddingLeftRight
                }
                else
                    points = that._setArrowCenter(cloudWidth, cloudHeight, arrowLength, x, y);
                if (isVerPositionDefined ? verPosition === 'top' : cloudHeight + arrowLength < y) {
                    yt -= arrowLength + cloudHeight / 2 - bbox.height / 2 + that._state.offset;
                    that.tooltipInverted = false
                }
                else {
                    yt += arrowLength + cloudHeight / 2 + bbox.height / 2 + that._state.offset;
                    that.tooltipInverted = true
                }
                yt = that._correctYTextContent(yt);
                return {
                        points: points,
                        text: {
                            x: xt,
                            y: yt,
                            align: align
                        }
                    }
            },
            _updateTextContent: function() {
                if (this._options._justify) {
                    this._textGroup.clear();
                    this._calculateTextContent();
                    this._locateTextContent(0, 0, 'center')
                }
                else
                    this._text.updateText(this._state.text);
                this._state.textBBox = this._textGroup.getBBox()
            },
            _correctYTextContent: function(y) {
                var bbox;
                if (this._options._justify) {
                    this._locateTextContent(0, y, 'center');
                    bbox = this._textGroup.getBBox()
                }
                else {
                    this._text.applySettings({y: y});
                    bbox = this._text.getBBox()
                }
                return y - (bbox.y + bbox.height - y)
            },
            _adjustTextContent: function(data) {
                if (this._options._justify)
                    this._locateTextContent(data.text.x, data.text.y, data.text.align);
                else
                    this._text.applySettings({
                        x: data.text.x,
                        y: data.text.y,
                        align: data.text.align
                    })
            },
            _updateTooltip: function() {
                var that = this,
                    box,
                    data,
                    scale;
                data = that._getData();
                that._cloud.applySettings({
                    points: data.points,
                    fill: that._state.color,
                    'class': that._state.className
                });
                that._adjustTextContent(data);
                box = that._group.getBBox();
                if (box.y + box.height > that._canvasHeight) {
                    scale = (that._canvasHeight - box.y) / box.height;
                    that._group.applySettings({
                        scale: scale,
                        translateX: that._state.x * (1 - scale),
                        translateY: that._state.y * (1 - scale)
                    })
                }
                else
                    that._group.applySettings({
                        scale: 1,
                        translateX: 0,
                        translateY: 0
                    })
            },
            _createTextContent: function() {
                var that = this,
                    options = that._options,
                    fontSize;
                that._textGroup.clear();
                that._text = null;
                if (!options._justify) {
                    fontSize = options.font && options.font.size;
                    that._text = that._renderer.createText(undefined, 0, 0, {font: {size: fontSize}}).append(that._textGroup)
                }
            },
            _getTextContentParams: function() {
                var that = this,
                    i,
                    text,
                    textBBox,
                    textArray = that._state.text,
                    textArrayLength = textArray.length,
                    textParams = {
                        width: [],
                        height: []
                    };
                that._tooltipTextArray = [];
                for (i = 0; i < textArrayLength; i++) {
                    text = that._renderer.createText(textArray[i], 0, 0, {}).append(this._textGroup);
                    that._tooltipTextArray.push(text);
                    textBBox = text.getBBox();
                    textParams.width.push(textBBox.width)
                }
                that._lineHeight = -2 * textBBox.y - textBBox.height;
                return textParams
            },
            _locateTextContent: function(x, y, alignment) {
                var that = this,
                    tooltipTextArray = that._tooltipTextArray,
                    textWidth = that._textContentWidth,
                    lineSpacing = that._options.lineSpacing,
                    yDelta = (lineSpacing > 0 ? lineSpacing : 0) + that._lineHeight,
                    leftXCoord,
                    rightXCoord,
                    i,
                    rtl = that._options._rtl;
                if (alignment === LEFT)
                    leftXCoord = x;
                else if (alignment === RIGHT)
                    leftXCoord = x - textWidth;
                else
                    leftXCoord = _round(x - textWidth / 2);
                rightXCoord = leftXCoord + textWidth;
                for (i = tooltipTextArray.length - 1; i >= 0; i -= 2) {
                    tooltipTextArray[i].applySettings({
                        x: !rtl ? rightXCoord : leftXCoord,
                        y: y,
                        align: !rtl ? RIGHT : LEFT
                    });
                    if (tooltipTextArray[i - 1])
                        tooltipTextArray[i - 1].applySettings({
                            x: !rtl ? leftXCoord : rightXCoord,
                            y: y,
                            align: !rtl ? LEFT : RIGHT
                        });
                    y -= yDelta
                }
            },
            _calculateTextContent: function() {
                var that = this,
                    textArray = that._state.text,
                    textArrayLength = textArray.length,
                    textParams,
                    width,
                    stringWidthArray = [],
                    i;
                textParams = that._getTextContentParams();
                for (i = 0; i < textArrayLength; i += 2) {
                    if (textParams.width[i + 1])
                        width = textParams.width[i] + X_INTERVAL + textParams.width[i + 1];
                    else
                        width = textParams.width[i];
                    stringWidthArray.push(width)
                }
                that._textContentWidth = _max.apply(null, stringWidthArray)
            },
            setSize: function(width, height) {
                this._canvasWidth = _isDefined(width) ? width : this._canvasWidth;
                this._canvasHeight = _isDefined(height) ? height : this._canvasHeight;
                return this
            },
            getBBox: function() {
                var that = this,
                    options = that._options,
                    paddingLeftRight = options.paddingLeftRight || 0,
                    paddingTopBottom = options.paddingTopBottom || 0,
                    borderWidth = options.border.visible && options.border.width || 0,
                    tooltipBBox = that._textGroup.getBBox();
                return tooltipBBox.isEmpty ? tooltipBBox : {
                        x: tooltipBBox.x - paddingLeftRight - borderWidth / 2 - MAX_SHADOW_SIZE,
                        y: tooltipBBox.y - paddingTopBottom - borderWidth / 2 - MAX_SHADOW_SIZE,
                        height: tooltipBBox.height + 2 * paddingTopBottom + borderWidth + MAX_SHADOW_SIZE * 2,
                        width: tooltipBBox.width + 2 * paddingLeftRight + borderWidth + MAX_SHADOW_SIZE * 2,
                        isEmpty: false
                    }
            },
            show: function() {
                this._state.visibility = VISIBLE;
                this._updateTextContent();
                this.move(this._state.x, this._state.y, this._state.offset);
                this._cloud.applySettings(VISIBLE);
                this._textGroup.applySettings(VISIBLE);
                return this
            },
            hide: function() {
                this._state.visibility = HIDDEN;
                this._cloud.applySettings(HIDDEN);
                this._textGroup.applySettings(HIDDEN);
                return this
            },
            move: function(x, y, offset) {
                this._state.x = _isDefined(x) ? x : this._state.x || 0;
                this._state.y = _isDefined(y) ? y : this._state.y || 0;
                this._state.offset = _isDefined(offset) ? offset : this._state.offset || 0;
                this._updateTooltip();
                return this
            },
            _setArrowCenter: function(cloudWidth, cloudHeight, arrowLength, x, y) {
                var that = this,
                    position = that._options.cloudVerticalPosition,
                    isPosDefined = position !== undefined && position !== null,
                    verticalInvert = !(isPosDefined ? position === 'top' : cloudHeight + arrowLength < y),
                    x0 = x,
                    y0 = verticalInvert ? y + that._state.offset : y - that._state.offset,
                    x1 = x0 + ARROW_WIDTH / 2,
                    y1 = verticalInvert ? y0 + arrowLength : y0 - arrowLength,
                    x2 = x1 + cloudWidth / 2 - ARROW_WIDTH / 2,
                    y2 = y1,
                    x3 = x2,
                    y3 = verticalInvert ? y2 + cloudHeight : y2 - cloudHeight,
                    x4 = x3 - cloudWidth,
                    y4 = y3,
                    x5 = x4,
                    y5 = verticalInvert ? y4 - cloudHeight : y4 + cloudHeight,
                    x6 = x5 + cloudWidth / 2 - ARROW_WIDTH / 2,
                    y6 = y5;
                return [x0, y0, x1, y1, x2, y2, x3, y3, x4, y4, x5, y5, x6, y6]
            },
            _setArrowLeft: function(cloudWidth, cloudHeight, arrowLength, x, y) {
                var that = this,
                    position = that._options.cloudVerticalPosition,
                    isPosDefined = position !== undefined && position !== null,
                    verticalInvert = !(isPosDefined ? position === 'top' : cloudHeight + arrowLength < y),
                    x0 = x,
                    y0 = verticalInvert ? y + that._state.offset : y - that._state.offset,
                    x1 = x0 + ARROW_WIDTH,
                    y1 = verticalInvert ? y0 + arrowLength : y0 - arrowLength,
                    x2 = x1 + cloudWidth - ARROW_WIDTH,
                    y2 = y1,
                    x3 = x2,
                    y3 = verticalInvert ? y2 + cloudHeight : y2 - cloudHeight,
                    x4 = x3 - cloudWidth,
                    y4 = y3,
                    x5 = x4,
                    y5 = verticalInvert ? y4 - cloudHeight - arrowLength : y4 + cloudHeight + arrowLength;
                return [x0, y0, x1, y1, x2, y2, x3, y3, x4, y4, x5, y5]
            },
            _setArrowRight: function(cloudWidth, cloudHeight, arrowLength, x, y) {
                var that = this,
                    position = that._options.cloudVerticalPosition,
                    isPosDefined = position !== undefined && position !== null,
                    verticalInvert = !(isPosDefined ? position === 'top' : cloudHeight + arrowLength < y),
                    x0 = x,
                    y0 = verticalInvert ? y + that._state.offset : y - that._state.offset,
                    x1 = x0,
                    y1 = verticalInvert ? y0 + arrowLength + cloudHeight : y0 - arrowLength - cloudHeight,
                    x2 = x1 - cloudWidth,
                    y2 = y1,
                    x3 = x2,
                    y3 = verticalInvert ? y2 - cloudHeight : y2 + cloudHeight,
                    x4 = x3 + cloudWidth - ARROW_WIDTH,
                    y4 = y3,
                    x5 = x4 + ARROW_WIDTH,
                    y5 = verticalInvert ? y4 - arrowLength : y4 + arrowLength;
                return [x0, y0, x1, y1, x2, y2, x3, y3, x4, y4, x5, y5]
            },
            _checkWidthText: function(cloudWidth, cloudHeight) {
                if (this._options._justify)
                    return;
                var x = this._state.x,
                    text = this._state.text,
                    index,
                    paddingLeftRight = this._options.paddingLeftRight,
                    paddingTopBottom = this._options.paddingTopBottom,
                    textLength,
                    maxTooltipWidth,
                    remainLength,
                    newIndex,
                    bbox = this._state.textBBox;
                if (cloudWidth < x || x + cloudWidth < this._canvasWidth || cloudWidth / 2 < x && x + cloudWidth / 2 < this._canvasWidth)
                    return false;
                if (text.indexOf("<br/>") === -1 && text.indexOf(" ") !== -1) {
                    maxTooltipWidth = _max(x, this._canvasWidth - x, 2 * Math.min(x, this._canvasWidth - x));
                    textLength = text.length * maxTooltipWidth / bbox.width;
                    index = text.substr(0, ~~textLength).lastIndexOf(" ");
                    if (index === -1)
                        index = text.substr(0).indexOf(" ");
                    remainLength = text.substr(index + 1).length;
                    this._state.text = text.substr(0, index) + "<br/>";
                    while (textLength <= remainLength) {
                        newIndex = text.substr(index + 1, ~~textLength).lastIndexOf(" ");
                        if (newIndex === -1)
                            newIndex = text.substr(index + 1).indexOf(" ");
                        if (newIndex !== -1) {
                            this._state.text += text.substr(index + 1, newIndex) + "<br/>";
                            remainLength = text.substr(index + 1 + newIndex).length;
                            index += newIndex + 1
                        }
                        else
                            break
                    }
                    this._state.text += text.substr(index + 1);
                    this._text.updateText(this._state.text);
                    bbox = this._text.getBBox();
                    cloudWidth = bbox.width + paddingLeftRight * 2;
                    cloudHeight = bbox.height + paddingTopBottom * 2
                }
                if (cloudWidth > x && x + cloudWidth > this._canvasWidth && (cloudWidth / 2 > x || x + cloudWidth / 2 > this._canvasWidth)) {
                    paddingLeftRight = 5;
                    paddingTopBottom = 5;
                    cloudWidth = bbox.width + 2 * paddingLeftRight;
                    cloudHeight = bbox.height + 2 * paddingTopBottom
                }
                return {
                        bbox: bbox,
                        cloudWidth: cloudWidth,
                        cloudHeight: cloudHeight,
                        paddingTopBottom: paddingTopBottom,
                        paddingLeftRight: paddingLeftRight
                    }
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file legend.js */
    (function(DX, $, undefined) {
        var DEFAULT_MARGIN = 10,
            _math = Math,
            _round = _math.round,
            _ceil = _math.ceil,
            _floor = _math.floor,
            CENTER = 'center',
            RIGHT = 'right',
            LEFT = 'left',
            TOP = 'top',
            BOTTOM = 'bottom',
            HORIZONTAL = 'horizontal',
            VERTICAL = 'vertical',
            INSIDE = 'inside',
            OUTSIDE = 'outside',
            NONE = 'none',
            decreaseGaps = DevExpress.viz.core.utils.decreaseGaps,
            DEFAULT_MARKER_HATCHING_WIDTH = 2,
            DEFAULT_MARKER_HATCHING_STEP = 5;
        function getPatternId(renderer, states, action, color) {
            if (!states)
                return;
            var direction = states[action].hatching.direction,
                hatching,
                colorFromAction = states[action].fill;
            color = colorFromAction === NONE ? color : colorFromAction;
            direction = !direction || direction === NONE ? RIGHT : direction;
            hatching = $.extend({}, states[action].hatching, {
                direction: direction,
                step: DEFAULT_MARKER_HATCHING_STEP,
                width: DEFAULT_MARKER_HATCHING_WIDTH
            });
            return renderer.createPattern(color, hatching).append().id
        }
        DX.viz.core.Legend = DX.Class.inherit({
            ctor: function(data, options, renderer, group) {
                var that = this;
                that._renderer = renderer;
                that._legendGroup = group;
                that._markersPatternsIds = [];
                that._data = data;
                that._init(options)
            },
            _init: function(options) {
                if (!options)
                    return;
                var debug = DX.utils.debug;
                debug.assertParam(options.visible, 'Visibility was not passed');
                debug.assertParam(options.markerSize, 'markerSize was not passed');
                debug.assertParam(options.font.color, 'fontColor was not passed');
                debug.assertParam(options.font.family, 'fontFamily was not passed');
                debug.assertParam(options.font.size, 'fontSize was not passed');
                debug.assertParam(options.paddingLeftRight, 'paddingLeftRight was not passed');
                debug.assertParam(options.paddingTopBottom, 'paddingTopBottom was not passed');
                debug.assertParam(options.columnItemSpacing, 'columnItemSpacing was not passed');
                debug.assertParam(options.rowItemSpacing, 'rowItemSpacing was not passed');
                debug.assertParam(options.equalColumnWidth, 'equalColumnWidth was not passed');
                var that = this,
                    i;
                that._parseMargins(options);
                that._parseAlignments(options);
                options.orientation = (options.orientation || '').toLowerCase();
                if (options.orientation !== VERTICAL && options.orientation !== HORIZONTAL) {
                    if (options.horizontalAlignment === CENTER)
                        options.orientation = HORIZONTAL;
                    if (options.horizontalAlignment === RIGHT || options.horizontalAlignment === LEFT)
                        options.orientation = VERTICAL
                }
                if (!options.itemTextPosition)
                    options.itemTextPosition = options.orientation === HORIZONTAL ? BOTTOM : RIGHT;
                else
                    options.itemTextPosition = options.itemTextPosition;
                options.position = (options.position || '').toLowerCase();
                if (options.position !== OUTSIDE && options.position !== INSIDE)
                    options.position = OUTSIDE;
                options.hoverMode = (options.hoverMode || '').toLowerCase();
                options.customizeText = $.isFunction(options.customizeText) ? options.customizeText : function() {
                    return this.seriesName
                };
                that._options = options;
                !that._trackerGroup && (that._trackerGroup = that._renderer.createGroup({
                    'class': 'dxc-legend-trackers',
                    stroke: NONE,
                    fill: 'grey',
                    opacity: 0.0001
                }));
                that.__initialized = true
            },
            _parseMargins: function(options) {
                if (options.margin >= 0) {
                    options.margin = Number(options.margin);
                    options.margin = {
                        top: options.margin,
                        bottom: options.margin,
                        left: options.margin,
                        right: options.margin
                    }
                }
                else
                    options.margin = {
                        top: options.margin.top >= 0 ? Number(options.margin.top) : DEFAULT_MARGIN,
                        bottom: options.margin.bottom >= 0 ? Number(options.margin.bottom) : DEFAULT_MARGIN,
                        left: options.margin.left >= 0 ? Number(options.margin.left) : DEFAULT_MARGIN,
                        right: options.margin.right >= 0 ? Number(options.margin.right) : DEFAULT_MARGIN
                    }
            },
            _parseAlignments: function(options) {
                options.horizontalAlignment = (options.horizontalAlignment || '').toLowerCase();
                if (options.horizontalAlignment !== CENTER && options.horizontalAlignment !== RIGHT && options.horizontalAlignment !== LEFT)
                    options.horizontalAlignment = RIGHT;
                options.verticalAlignment = (options.verticalAlignment || '').toLowerCase();
                if (options.verticalAlignment !== TOP && options.verticalAlignment !== BOTTOM) {
                    if (options.horizontalAlignment === CENTER)
                        options.verticalAlignment = BOTTOM;
                    if (options.horizontalAlignment === RIGHT || options.horizontalAlignment === LEFT)
                        options.verticalAlignment = TOP
                }
            },
            update: function(data, options) {
                this._data = data;
                this.boundingRect = {
                    width: 0,
                    height: 0,
                    x: 0,
                    y: 0
                };
                this._init(options);
                return this
            },
            setSize: function(size) {
                this._size = size;
                return this
            },
            draw: function() {
                if (!this._options)
                    return this;
                var that = this,
                    renderer = that._renderer,
                    options = that._options,
                    items = that._data,
                    itemsLength = items.length,
                    seriesGroups = [],
                    i,
                    label,
                    marker,
                    singleSeriesGroup,
                    trackers = [],
                    insideLegendGroup,
                    markersId = new Array(itemsLength),
                    passedId,
                    border = options.border,
                    borderVisible = border.visible && border.width && border.color && border.color !== NONE,
                    markers = new Array(itemsLength);
                if (!(options.visible && items && itemsLength)) {
                    that._disposeTrackers();
                    return that
                }
                that._cleanLegendGroups();
                insideLegendGroup = that._insideLegendGroup = renderer.createGroup().append(that._legendGroup);
                that._createBackground(borderVisible);
                for (i = 0; i < itemsLength; i++) {
                    singleSeriesGroup = renderer.createGroup({'class': 'dxc-item'}).append(insideLegendGroup);
                    marker = that._createMarker(items[i], singleSeriesGroup, i);
                    passedId = items[i].id;
                    label = that._createLabel(items[i], passedId, singleSeriesGroup);
                    markersId[i] = passedId;
                    that._locateLabelAndMarker(label, marker);
                    markers[i] = marker;
                    trackers.push({
                        rect: renderer.createRect(0, 0, 0, 0, 0, {
                            stroke: NONE,
                            fill: 'grey',
                            opacity: 0.0001,
                            inh: true
                        }),
                        id: passedId
                    });
                    seriesGroups.push(singleSeriesGroup)
                }
                that._seriesGroups = seriesGroups;
                that._trackers = trackers;
                that.drawTrackers();
                that._markers = markers;
                that._markersId = markersId;
                that._locateElements(options);
                return that
            },
            _locateElements: function(options) {
                var that = this,
                    border = options.border,
                    borderVisible = border.visible && border.width && border.color && border.color !== NONE;
                that._moveInInitialValues();
                that._locateRowsColumns(that._seriesGroups, that._trackers, that._data.length, that._background, options);
                if (that._background)
                    that._adjustBackgroundSettings(that._background, borderVisible, options);
                that._setBoundingRect(options)
            },
            _moveInInitialValues: function() {
                var that = this;
                that._legendGroup && that._legendGroup.move(0, 0);
                that._trackerGroup && that._trackerGroup.move(0, 0);
                that._background && that._background.applySettings({
                    x: 0,
                    y: 0,
                    width: 0,
                    height: 0
                })
            },
            _applyMarkerOptions: function(marker, options) {
                if (marker)
                    marker.applySettings(options)
            },
            applySelected: function(id) {
                var index = this._getIndexById(id);
                if (index !== null)
                    this._applyMarkerOptions(this._markers[index], {fill: this._markersPatternsIds[index].selectedPatternId});
                return this
            },
            applyHover: function(id) {
                var index = this._getIndexById(id);
                if (index !== null)
                    this._applyMarkerOptions(this._markers[index], {fill: this._markersPatternsIds[index].hoverPatternId});
                return this
            },
            resetItem: function(id) {
                var index = this._getIndexById(id);
                if (index !== null)
                    this._applyMarkerOptions(this._markers[index], {fill: this._data[index].color || this._data[index].states.normal.fill})
            },
            _getIndexById: function(id) {
                var i,
                    markersId = this._markersId,
                    markersIdLen = markersId.length;
                for (i = 0; i < markersIdLen; i++)
                    if (markersId[i] === id)
                        return i;
                return null
            },
            drawTrackers: function() {
                if (!this._options.visible || !this._insideLegendGroup)
                    return;
                var that = this,
                    trackerGroup = that._trackerGroup.append(that._legendGroup);
                $.each(that._trackers || [], function(i, tracker) {
                    var trackerRect = tracker.rect;
                    trackerRect.data({
                        itemIndex: tracker.id,
                        mode: that._options.hoverMode
                    });
                    trackerRect.append(trackerGroup)
                });
                return that
            },
            _disposeTrackers: function() {
                var that = this;
                $.each(that._trackers || [], function(_, tracker) {
                    tracker.rect.removeData()
                });
                that._trackers = null
            },
            _createMarker: function(data, group, markerIndex) {
                var that = this,
                    renderer = that._renderer,
                    markersPatternsIds = that._markersPatternsIds,
                    size = that._options.markerSize,
                    states = data.states,
                    marker;
                marker = renderer.createRect(0, 0, size, size, 0, {fill: data.color}).append(group);
                markersPatternsIds[markerIndex] = {
                    hoverPatternId: getPatternId(renderer, states, 'hover', data.color),
                    selectedPatternId: getPatternId(renderer, states, 'selection', data.color)
                };
                return marker
            },
            _createLabel: function(data, index, group) {
                var that = this,
                    options = that._options,
                    position = options.itemTextPosition,
                    align = position === TOP || position === BOTTOM ? CENTER : LEFT,
                    text,
                    label,
                    labelFormatObject = {
                        seriesName: data.text,
                        seriesNumber: index,
                        seriesColor: data.color
                    };
                text = that._formatLabel.call(labelFormatObject, options);
                label = that._renderer.createText(text, 0, 0, {
                    font: options.font,
                    align: align
                }).append(group);
                return label
            },
            _cleanLegendGroups: function() {
                var that = this,
                    trackerGroup = that._trackerGroup,
                    legendGroup = that._legendGroup;
                if (legendGroup) {
                    legendGroup.clear();
                    that._insideLegendGroup && that._insideLegendGroup.dispose();
                    that._insideLegendGroup = null
                }
                if (trackerGroup)
                    trackerGroup.clear()
            },
            _createBackground: function(borderVisible) {
                var that = this,
                    options = that._options,
                    isInside = options.position === INSIDE,
                    color = options.backgroundColor,
                    fill = color || (isInside ? options.containerBackgroundColor : NONE);
                if (isInside || color || borderVisible)
                    that._background = that._renderer.createRect(0, 0, 0, 0, 0, {
                        fill: fill,
                        'class': 'dxc-border'
                    }).append(that._insideLegendGroup)
            },
            _formatLabel: function(options) {
                return options.customizeText.call(this, this)
            },
            _locateLabelAndMarker: function(label, marker) {
                var that = this,
                    defaultXMargin = 7,
                    defaultTopMargin = 4,
                    defaultBottomMargin = 2,
                    labelX = 0,
                    labelY = 0,
                    markerX,
                    markerY,
                    labelBox = label.getBBox(),
                    markerWidth = that._options.markerSize,
                    markerHeight = markerWidth,
                    approximateLabelY = markerHeight / 2 - (labelBox.y + labelBox.height / 2),
                    approximateLabelX = markerWidth / 2 - (labelBox.x + labelBox.width / 2);
                switch (that._options.itemTextPosition) {
                    case LEFT:
                        labelY = _round(approximateLabelY);
                        markerX = labelBox.width + defaultXMargin;
                        break;
                    case RIGHT:
                        labelX = markerWidth + defaultXMargin;
                        labelY = _round(approximateLabelY);
                        break;
                    case TOP:
                        labelX = _round(approximateLabelX);
                        markerY = defaultTopMargin;
                        break;
                    case BOTTOM:
                        labelX = _round(approximateLabelX);
                        labelY = markerHeight + defaultBottomMargin - labelBox.y;
                        break
                }
                label.applySettings({
                    x: labelX,
                    y: labelY
                });
                marker.applySettings({
                    x: markerX,
                    y: markerY
                })
            },
            _locateRowsColumns: function(groups, trackers, count, background, options) {
                var that = this,
                    itemTextPosition = options.itemTextPosition,
                    size = that._size,
                    legendBox,
                    rowsColumns = that._getRowsColumns(count),
                    rows = rowsColumns.rows,
                    columns = rowsColumns.columns,
                    margin = options.margin,
                    paddingLeftRight = background ? options.paddingLeftRight : 0,
                    paddingTopBottom = background ? options.paddingTopBottom : 0,
                    placeholderWidth = size.width - margin.left - margin.right - 2 * paddingLeftRight,
                    placeholderHeight = size.height - margin.top - margin.bottom - 2 * paddingTopBottom,
                    rowsColumnsData,
                    condition;
                var moveRowsColumns = function(rows, columns) {
                        rowsColumnsData = that._getDataRowsColumns(groups, columns, rows);
                        that._moveItems(rowsColumnsData, groups, itemTextPosition, trackers, options)
                    };
                moveRowsColumns(rows, columns);
                legendBox = that._insideLegendGroup ? that._insideLegendGroup.getBBox() : {};
                if (rowsColumns.autoEdit)
                    if (rows === 1) {
                        if (legendBox.width > placeholderWidth && columns > 1) {
                            columns = _floor(columns * placeholderWidth / legendBox.width) || 1;
                            columns = validate(columns, count);
                            rows = _ceil(count / columns);
                            moveRowsColumns(rows, columns)
                        }
                    }
                    else if (columns === 1)
                        if (legendBox.height > placeholderHeight && rows > 1) {
                            rows = _floor(rows * placeholderHeight / legendBox.height) || 1;
                            rows = validate(rows, count);
                            columns = _ceil(count / rows);
                            moveRowsColumns(rows, columns)
                        }
                that._rowsCountDrawed = rows;
                that._columnsCountDrawed = columns;
                function validate(rowsOrColumns, count) {
                    if (rowsOrColumns > count || rowsOrColumns < 1)
                        rowsOrColumns = count;
                    return rowsOrColumns
                }
            },
            _moveItems: function(data, seriesGroups, horizontalTextPosition, trackers, options) {
                var that = this,
                    i,
                    j,
                    rows,
                    cols,
                    number,
                    group,
                    box,
                    xShift = 0,
                    yShift = 0,
                    widthColumn,
                    xPadding = options.columnItemSpacing,
                    yPadding = options.rowItemSpacing,
                    equalColumnWidth = options.equalColumnWidth,
                    renderer = that._renderer,
                    maxWidthPerColumn = [],
                    maxWidthColumn = 0,
                    maxHeightRow = 0;
                rows = data.rows;
                cols = data.cols;
                maxHeightRow = data.maxHeightRow;
                maxWidthColumn = data.maxWidthColumn;
                maxWidthPerColumn = data.maxWidthPerColumn;
                for (i = 0; i < rows; i++) {
                    for (j = 0; j < cols; j++) {
                        if (rows < cols)
                            number = i * cols + j;
                        else
                            number = i + j * rows;
                        group = seriesGroups[number];
                        if (!group)
                            break;
                        box = group.getBBox();
                        widthColumn = !equalColumnWidth ? maxWidthPerColumn[j] : maxWidthColumn;
                        if (horizontalTextPosition === RIGHT) {
                            group.move(xShift - box.x, yShift);
                            trackers[number].rect.applySettings({
                                x: xShift - xPadding / 2,
                                y: yShift + box.y - yPadding / 2,
                                height: maxHeightRow + yPadding,
                                width: widthColumn + xPadding
                            })
                        }
                        else if (horizontalTextPosition === LEFT) {
                            group.move(box.x + widthColumn - box.width + xShift - xPadding / 2, yShift);
                            trackers[number].rect.applySettings({
                                x: box.x + widthColumn - box.width + xShift - xPadding / 2,
                                y: yShift + box.y - yPadding / 2,
                                height: maxHeightRow + yPadding,
                                width: widthColumn + xPadding
                            })
                        }
                        else {
                            group.move(xShift - box.x - box.width / 2 + widthColumn / 2, yShift);
                            trackers[number].rect.applySettings({
                                x: xShift - xPadding / 2,
                                y: yShift + box.y - yPadding / 2,
                                height: maxHeightRow + yPadding,
                                width: widthColumn + xPadding
                            })
                        }
                        xShift = xShift + widthColumn + xPadding
                    }
                    yShift = yShift + maxHeightRow + yPadding;
                    xShift = 0
                }
            },
            _getDataRowsColumns: function(seriesGroups, cols, rows) {
                var that = this,
                    i,
                    j,
                    options = that._options,
                    equalColumnWidth = options.equalColumnWidth,
                    series = that.series || {},
                    maxWidthPerColumn = [],
                    maxWidthColumn = 0,
                    maxHeightRow = 0,
                    group,
                    box;
                for (i = 0; i < cols; i++)
                    maxWidthPerColumn[i] = 0;
                for (i = 0; i < rows; i++)
                    for (j = 0; j < cols; j++) {
                        if (rows < cols)
                            group = seriesGroups[i * cols + j];
                        else
                            group = seriesGroups[i + j * rows];
                        if (!group)
                            break;
                        box = group.getBBox();
                        if (maxHeightRow < box.height)
                            maxHeightRow = box.height;
                        if (!equalColumnWidth) {
                            if (maxWidthPerColumn[j] < box.width)
                                maxWidthPerColumn[j] = box.width
                        }
                        else if (maxWidthColumn < box.width)
                            maxWidthColumn = box.width
                    }
                return {
                        rows: rows,
                        cols: cols,
                        maxWidthPerColumn: maxWidthPerColumn,
                        maxWidthColumn: maxWidthColumn,
                        maxHeightRow: maxHeightRow
                    }
            },
            _getRowsColumns: function(count) {
                var that = this,
                    options = that._options,
                    isHorizontal = options.orientation === HORIZONTAL,
                    rows = options.rowCount,
                    onRows = _ceil(count / rows),
                    columns = options.columnCount,
                    onColumns = _ceil(count / columns),
                    autoEdit;
                if (columns && !rows)
                    rows = onColumns;
                else if (!columns && rows)
                    columns = onRows;
                else if (columns && rows) {
                    if (isHorizontal && columns < onRows)
                        columns = onRows;
                    else if (!isHorizontal && rows < onColumns)
                        rows = onColumns
                }
                else {
                    autoEdit = true;
                    if (isHorizontal) {
                        rows = 1;
                        columns = count
                    }
                    else {
                        columns = 1;
                        rows = count
                    }
                }
                return {
                        rows: rows,
                        columns: columns,
                        autoEdit: autoEdit
                    }
            },
            _adjustBackgroundSettings: function(background, borderVisible, options) {
                var that = this,
                    border = options.border,
                    legendBox = that._insideLegendGroup.getBBox(),
                    backgroundSettings = {
                        x: _round(legendBox.x - options.paddingLeftRight),
                        y: _round(legendBox.y - options.paddingTopBottom),
                        width: _round(legendBox.width) + 2 * options.paddingLeftRight,
                        height: _round(legendBox.height) + 2 * options.paddingTopBottom,
                        opacity: options.backgroundOpacity
                    };
                if (borderVisible) {
                    backgroundSettings.strokeWidth = border.width;
                    backgroundSettings.stroke = border.color;
                    backgroundSettings.strokeOpacity = border.opacity;
                    backgroundSettings.dashStyle = border.dashStyle;
                    backgroundSettings.rx = border.cornerRadius || 0;
                    backgroundSettings.ry = border.cornerRadius || 0
                }
                background.applySettings(backgroundSettings)
            },
            _setBoundingRect: function(options) {
                var that = this,
                    box,
                    margin = options.margin;
                if (!that._insideLegendGroup)
                    return;
                box = that._insideLegendGroup.getBBox();
                box.height += margin.top + margin.bottom;
                box.width += margin.left + margin.right;
                box.x -= margin.left;
                box.y -= margin.top;
                that.boundingRect = box
            },
            changeSize: function(size) {
                var that = this,
                    options = $.extend(true, {}, that._options),
                    margin = options.margin;
                if (size.height >= 0) {
                    size.height = decreaseGaps(margin, ["top", "bottom"], size.height);
                    if (options.border.visible)
                        size.height = 2 * decreaseGaps(options, ["paddingTopBottom"], size.height / 2);
                    if (that._rowsCountDrawed - 1)
                        size.height = (that._rowsCountDrawed - 1) * decreaseGaps(options, ["rowItemSpacing"], size.height / (that._rowsCountDrawed - 1))
                }
                if (size.width >= 0) {
                    size.width = decreaseGaps(margin, ["left", "right"], size.width);
                    if (options.border.visible)
                        size.width = 2 * decreaseGaps(options, ["paddingLeftRight"], size.width / 2);
                    if (that._columnsCountDrawed - 1)
                        size.width = (that._columnsCountDrawed - 1) * decreaseGaps(options, ["columnItemSpacing"], size.width / (that._columnsCountDrawed - 1))
                }
                if (that._insideLegendGroup)
                    if (size.height > 0 || size.width > 0) {
                        that._options._incidentOccured("W2104");
                        that._insideLegendGroup.remove();
                        that._insideLegendGroup = null;
                        that._trackerGroup && that._trackerGroup.clear()
                    }
                    else
                        that._locateElements(options)
            },
            getTrackerGroup: function() {
                return this._trackerGroup
            },
            getActionCallback: function(point) {
                var that = this;
                if (that._options.visible)
                    return function(act) {
                            var pointType = point.type,
                                isSeries = pointType ? true : false,
                                seriesType = pointType || point.series.type;
                            if (isSeries || seriesType === 'pie' || seriesType === 'doughnut' || seriesType === 'donut')
                                that[act] && that[act](point.index)
                        };
                else
                    return $.noop
            },
            getLayoutOptions: function() {
                var options = this._options,
                    boundingRect = this._insideLegendGroup ? this.boundingRect : {
                        width: 0,
                        height: 0,
                        x: 0,
                        y: 0
                    };
                if (options) {
                    boundingRect.verticalAlignment = options.verticalAlignment;
                    boundingRect.horizontalAlignment = options.horizontalAlignment;
                    if (options.orientation === 'horizontal')
                        boundingRect.cutLayoutSide = options.verticalAlignment;
                    else
                        boundingRect.cutLayoutSide = options.horizontalAlignment === 'center' ? options.verticalAlignment : options.horizontalAlignment;
                    return boundingRect
                }
                return null
            },
            shift: function(x, y) {
                var that = this,
                    settings = {},
                    box = that.getLayoutOptions();
                settings.translateX = x - box.x;
                settings.translateY = y - box.y;
                that._insideLegendGroup && that._insideLegendGroup.applySettings(settings);
                that._trackerGroup && that._trackerGroup.applySettings(settings)
            },
            getPosition: function() {
                return this._options.position
            },
            dispose: function() {
                var that = this;
                that._disposeTrackers();
                that._legendGroup = null;
                that._trackerGroup = null;
                that._insideLegendGroup = null;
                that._renderer = null;
                that._options = null;
                that._data = null;
                return that
            }
        })
    })(DevExpress, jQuery);
    /*! Module viz-core, file range.js */
    (function($, DX, undefined) {
        var core = DX.viz.core,
            utils = DX.utils,
            isDefinedUtils = utils.isDefined,
            isDateUtils = utils.isDate,
            getLogUtils = utils.getLog,
            raiseToUtils = utils.raiseTo;
        var NUMBER_EQUALITY_CORRECTION = 1,
            DATETIME_EQUALITY_CORRECTION = 60000;
        var minSelector = 'min',
            maxSelector = 'max',
            minVisibleSelector = 'minVisible',
            maxVisibleSelector = 'maxVisible',
            minValueMarginSelector = 'minValueMargin',
            maxValueMarginSelector = 'maxValueMargin',
            categoriesSelector = 'categories',
            keepValueMarginsSelector = 'keepValueMargins',
            baseSelector = 'base',
            axisTypeSelector = 'axisType';
        var raiseToFlooredLog = function(value, base, correction) {
                return raiseToUtils(Math.floor(getLogUtils(value, base)) + (correction || 0), base)
            };
        var otherLessThan = function(thisValue, otherValue) {
                return otherValue < thisValue
            };
        var otherGreaterThan = function(thisValue, otherValue) {
                return otherValue > thisValue
            };
        var compareAndReplace = function(thisValue, otherValue, setValue, compare) {
                var otherValueDefined = isDefinedUtils(otherValue);
                if (isDefinedUtils(thisValue)) {
                    if (otherValueDefined && compare(thisValue, otherValue))
                        setValue(otherValue)
                }
                else if (otherValueDefined)
                    setValue(otherValue)
            };
        var applyMargin = function(value, margin, rangeLength, coef, isDateTime) {
                value = value.valueOf() + coef * rangeLength * margin;
                return isDateTime ? new Date(value) : value
            };
        DX.viz.core.__NUMBER_EQUALITY_CORRECTION = NUMBER_EQUALITY_CORRECTION;
        DX.viz.core.__DATETIME_EQUALITY_CORRECTION = DATETIME_EQUALITY_CORRECTION;
        core.Range = function(range) {
            range && $.extend(this, range)
        };
        $.extend(core.Range.prototype, {
            dispose: function() {
                this[categoriesSelector] = null
            },
            addRange: function(otherRange) {
                var that = this,
                    categories = that[categoriesSelector],
                    categoriesValues,
                    otherCategories = otherRange[categoriesSelector],
                    i,
                    j,
                    length,
                    found;
                var setIndentByPriority = function(prefix) {
                        var prioritySelector = prefix + 'Priority',
                            priorityRelation = (that[prioritySelector] || 0) - (otherRange[prioritySelector] || 0);
                        if ((that[prefix] || 0) < otherRange[prefix] && priorityRelation === 0 || priorityRelation < 0) {
                            that[prefix] = otherRange[prefix];
                            that[prioritySelector] = otherRange[prioritySelector]
                        }
                    };
                var compareAndReplaceByField = function(field, compare) {
                        compareAndReplace(that[field], otherRange[field], function(value) {
                            that[field] = value
                        }, compare)
                    };
                var controlValuesByVisibleBounds = function(valueField, visibleValueField, compare) {
                        compareAndReplace(that[valueField], that[visibleValueField], function(value) {
                            isDefinedUtils(that[valueField]) && (that[valueField] = value)
                        }, compare)
                    };
                var checkField = function(field) {
                        that[field] = that[field] || otherRange[field]
                    };
                if (utils.isDefined(otherRange["stick"]))
                    that["stick"] = otherRange["stick"];
                checkField('invert');
                checkField(axisTypeSelector);
                checkField('dataType');
                checkField(keepValueMarginsSelector);
                if (that[axisTypeSelector] === 'logarithmic')
                    checkField(baseSelector);
                else
                    that[baseSelector] = undefined;
                compareAndReplaceByField(minSelector, otherLessThan);
                compareAndReplaceByField(maxSelector, otherGreaterThan);
                compareAndReplaceByField(minVisibleSelector, otherLessThan);
                compareAndReplaceByField(maxVisibleSelector, otherGreaterThan);
                compareAndReplaceByField('interval', otherLessThan);
                setIndentByPriority(minValueMarginSelector);
                setIndentByPriority(maxValueMarginSelector);
                controlValuesByVisibleBounds(minSelector, minVisibleSelector, otherLessThan);
                controlValuesByVisibleBounds(minSelector, maxVisibleSelector, otherLessThan);
                controlValuesByVisibleBounds(maxSelector, maxVisibleSelector, otherGreaterThan);
                controlValuesByVisibleBounds(maxSelector, minVisibleSelector, otherGreaterThan);
                if (categories === undefined)
                    that[categoriesSelector] = otherCategories;
                else {
                    length = categories.length;
                    if (otherCategories && otherCategories.length)
                        for (i = 0; i < otherCategories.length; i++) {
                            for (j = 0, found = false; j < length; j++)
                                if (categories[j].valueOf() === otherCategories[i].valueOf()) {
                                    found = true;
                                    break
                                }
                            !found && categories.push(otherCategories[i])
                        }
                }
                return this
            },
            isDefined: function() {
                return isDefinedUtils(this[minSelector]) && isDefinedUtils(this[maxSelector]) || isDefinedUtils(this[categoriesSelector])
            },
            setStubData: function(dataType) {
                var that = this,
                    year = (new Date).getYear() - 1,
                    isDate = dataType === 'datetime',
                    isCategories = that.axisType === 'discrete';
                if (isCategories)
                    that.categories = ['0', '1', '2'];
                else {
                    that[minSelector] = isDate ? new Date(year, 0, 1) : 0;
                    that[maxSelector] = isDate ? new Date(year, 11, 31) : 10
                }
                that.stubData = true;
                return that
            },
            applyValueMargins: function() {
                var that = this,
                    base = that[baseSelector],
                    isDateTime = isDateUtils(that[maxSelector]) || isDateUtils(that[minSelector]);
                var applyMarginWithZeroCorrection = function(min, max, rangeLength) {
                        var minValue = that[min],
                            maxValue = that[max],
                            minMargin = that[minValueMarginSelector],
                            maxMargin = that[maxValueMarginSelector],
                            minCorrected = false,
                            maxCorrected = false;
                        if (rangeLength && !isDateTime && !that[keepValueMarginsSelector]) {
                            if (minValue <= 0 && maxValue <= 0 && maxMargin > maxValue / (minValue - maxValue)) {
                                that[max] = 0;
                                maxCorrected = true
                            }
                            if (minValue >= 0 && maxValue >= 0 && minMargin > minValue / (maxValue - minValue)) {
                                that[min] = 0;
                                minCorrected = true
                            }
                        }
                        if (isDefinedUtils(maxValue) && !maxCorrected && maxMargin)
                            that[max] = applyMargin(maxValue, maxMargin, rangeLength, 1, isDateTime);
                        if (isDefinedUtils(minValue) && !minCorrected && minMargin)
                            that[min] = applyMargin(minValue, minMargin, rangeLength, -1, isDateTime)
                    };
                var correctValueByBoundaries = function(visibleSelector, valueSelector) {
                        that[visibleSelector] = isDefinedUtils(that[visibleSelector]) ? that[visibleSelector] : that[valueSelector]
                    };
                var processLogarithmicMinValue = function(valueField) {
                        if (isDefinedUtils(that[valueField])) {
                            var intermediateValue = raiseToFlooredLog(that[valueField], base);
                            if (getLogUtils(that[valueField] / intermediateValue, base) < 0.1 && that.alwaysCorrectMin && !that[keepValueMarginsSelector])
                                that[valueField] = raiseToFlooredLog(that[valueField], base, -1);
                            else if (that.alwaysCorrectMin && !that[keepValueMarginsSelector])
                                that[valueField] = intermediateValue;
                            else if (getLogUtils(that[valueField] / intermediateValue, base) < 0.5)
                                that[valueField] = intermediateValue
                        }
                    };
                var processLogarithmicMaxValue = function(valueField) {
                        if (isDefinedUtils(that[valueField])) {
                            var intermediateValue = raiseToFlooredLog(that[valueField], base);
                            if (getLogUtils(that[valueField] / intermediateValue, base) > 0.5)
                                that[valueField] = raiseToUtils(Math.ceil(getLogUtils(that[valueField], base)), base)
                        }
                    };
                var correctLogVisibleValue = function(visibleValueSelector, valueSelector, compare) {
                        if (!isDefinedUtils(that[visibleValueSelector]) || compare)
                            that[visibleValueSelector] = that[valueSelector]
                    };
                if (that[axisTypeSelector] === 'logarithmic' && that[minSelector] !== that[maxSelector]) {
                    processLogarithmicMinValue(minSelector);
                    processLogarithmicMaxValue(maxSelector);
                    if (that.isValueRange && that[minVisibleSelector] !== that[maxVisibleSelector]) {
                        processLogarithmicMinValue(minVisibleSelector);
                        processLogarithmicMaxValue(maxVisibleSelector)
                    }
                    correctLogVisibleValue(minVisibleSelector, minSelector, that[minVisibleSelector] < that[minSelector]);
                    correctLogVisibleValue(maxVisibleSelector, maxSelector, that[maxVisibleSelector] > that[maxSelector])
                }
                else {
                    correctValueByBoundaries(minVisibleSelector, minSelector);
                    correctValueByBoundaries(maxVisibleSelector, maxSelector);
                    applyMarginWithZeroCorrection(minSelector, maxSelector, that[maxSelector] - that[minSelector]);
                    applyMarginWithZeroCorrection(minVisibleSelector, maxVisibleSelector, that[maxVisibleSelector] - that[minVisibleSelector])
                }
                return this
            },
            correctValueZeroLevel: function() {
                var that = this;
                if (isDateUtils(that[maxSelector]) || isDateUtils(that[minSelector]))
                    return that;
                function setZeroLevel(min, max) {
                    that[min] < 0 && that[max] < 0 && (that[max] = 0);
                    that[min] > 0 && that[max] > 0 && (that[min] = 0)
                }
                setZeroLevel(minSelector, maxSelector);
                setZeroLevel(minVisibleSelector, maxVisibleSelector);
                return that
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file seriesConsts.js */
    (function(DX) {
        DX.viz.core.series = DX.viz.core.series || {};
        DX.viz.core.series.helpers = DX.viz.core.series.helpers || {};
        DX.viz.core.series.helpers.consts = {
            events: {
                mouseover: "mouseover",
                mouseout: "mouseout",
                mousemove: "mousemove",
                touchstart: "touchstart",
                touchmove: "touchmove",
                touchend: "touchend",
                mousedown: "mousedown",
                mouseup: "mouseup",
                click: "click",
                selectSeries: "selectseries",
                deselectSeries: "deselectseries",
                selectPoint: "selectpoint",
                deselectPoint: "deselectpoint",
                showPointTooltip: "showpointtooltip",
                hidePointTooltip: "hidepointtooltip"
            },
            states: {
                hover: "hover",
                normal: "normal",
                selected: "selected",
                normalMark: 0,
                hoverMark: 1,
                selectedMark: 2
            },
            animations: {
                showDuration: {duration: 400},
                hideGroup: {opacity: 0.0001},
                showGroup: {opacity: 1}
            }
        }
    })(DevExpress);
    /*! Module viz-core, file seriesFamily.js */
    (function($, DX, undefined) {
        var utils = DX.utils,
            _round = Math.round,
            _abs = Math.abs,
            _pow = Math.pow;
        DX.viz.core.series.helpers.SeriesFamily = DX.Class.inherit(function() {
            var ctor = function(options) {
                    var debug = DX.utils.debug;
                    debug.assert(options.type, "type was not passed or empty");
                    var that = this,
                        stubFunction = $.noop;
                    $.extend(that, options);
                    that.type = that.type.toLowerCase();
                    that.series = [];
                    switch (that.type) {
                        case"bar":
                            that.adjustSeriesDimensions = adjustBarSeriesDimensions;
                            that.adjustSeriesValues = stubFunction;
                            that.updateSeriesValues = updateBarSeriesValues;
                            break;
                        case"rangebar":
                            that.adjustSeriesDimensions = adjustBarSeriesDimensions;
                            that.adjustSeriesValues = stubFunction;
                            that.updateSeriesValues = stubFunction;
                            break;
                        case"fullstackedbar":
                            that.fullStacked = true;
                            that.adjustSeriesDimensions = adjustStackedBarSeriesDimensions;
                            that.adjustSeriesValues = adjustStackedSeriesValues;
                            that.updateSeriesValues = updateStackedSeriesValues;
                            break;
                        case"stackedbar":
                            that.adjustSeriesDimensions = adjustStackedBarSeriesDimensions;
                            that.adjustSeriesValues = adjustStackedSeriesValues;
                            that.updateSeriesValues = updateStackedSeriesValues;
                            break;
                        case"fullstackedarea":
                        case"fullstackedline":
                            that.fullStacked = true;
                            that.adjustSeriesDimensions = stubFunction;
                            that.adjustSeriesValues = adjustStackedSeriesValues;
                            that.updateSeriesValues = stubFunction;
                            break;
                        case"stackedarea":
                        case"stackedline":
                            that.adjustSeriesDimensions = stubFunction;
                            that.adjustSeriesValues = adjustStackedSeriesValues;
                            that.updateSeriesValues = stubFunction;
                            break;
                        case"candlestick":
                        case"stock":
                            that.adjustSeriesDimensions = adjustCandlestickSeriesDimensions;
                            that.adjustSeriesValues = stubFunction;
                            that.updateSeriesValues = stubFunction;
                            break;
                        case"bubble":
                            that.adjustSeriesDimensions = adjustBubbleSeriesDimensions;
                            that.adjustSeriesValues = stubFunction;
                            that.updateSeriesValues = stubFunction;
                            break;
                        default:
                            that.adjustSeriesDimensions = stubFunction;
                            that.adjustSeriesValues = stubFunction;
                            that.updateSeriesValues = stubFunction;
                            break
                    }
                };
            var dispose = function() {
                    this.series = null;
                    this.translators = null
                };
            var add = function(series) {
                    var that = this,
                        singleSeries,
                        i;
                    if (!$.isArray(series))
                        series = [series];
                    for (i = 0; i < series.length; i++) {
                        singleSeries = series[i];
                        if (singleSeries.type.toLowerCase() === that.type)
                            that.series.push(singleSeries)
                    }
                };
            var adjustBarSeriesDimensionsCore = function(series, interval, stackCount, equalBarWidth, seriesStackIndexCallback) {
                    var spacing,
                        width,
                        maxWidth,
                        middleIndex,
                        stackIndex,
                        i,
                        point,
                        points,
                        seriesOffset,
                        stackName,
                        argumentsKeeper = {},
                        stackKeepers = {},
                        stacksWithArgument,
                        count;
                    if (equalBarWidth) {
                        width = equalBarWidth.width && equalBarWidth.width < 0 ? 0 : equalBarWidth.width;
                        spacing = equalBarWidth.spacing && equalBarWidth.spacing < 0 ? 0 : equalBarWidth.spacing;
                        if (width && !spacing)
                            if (stackCount > 1) {
                                spacing = _round((interval * 0.7 - width * stackCount) / (stackCount - 1));
                                if (spacing < 1)
                                    spacing = 1
                            }
                            else
                                spacing = 0;
                        else if (spacing && !width) {
                            width = _round((interval * 0.7 - spacing * (stackCount - 1)) / stackCount);
                            if (width < 2)
                                width = 2
                        }
                        else if (!spacing && !width) {
                            if (stackCount > 1) {
                                spacing = _round(interval * 0.7 / stackCount * 0.2);
                                if (spacing < 1)
                                    spacing = 1
                            }
                            else
                                spacing = 0;
                            width = _round((interval * 0.7 - spacing * (stackCount - 1)) / stackCount);
                            if (width < 2)
                                width = 2
                        }
                        if (width * stackCount + spacing * (stackCount - 1) > interval) {
                            spacing = _round((interval * 0.7 - width * stackCount) / (stackCount - 1));
                            if (spacing < 1) {
                                spacing = 1;
                                maxWidth = _round((interval * 0.7 - spacing * (stackCount - 1)) / stackCount)
                            }
                        }
                        middleIndex = stackCount / 2;
                        for (i = 0; i < series.length; i++) {
                            stackIndex = seriesStackIndexCallback(i);
                            points = series[i].getPoints();
                            seriesOffset = (stackIndex - middleIndex + 0.5) * (maxWidth || width) - (middleIndex - stackIndex - 0.5) * spacing;
                            $.each(points, function(_, point) {
                                point.correctCoordinates({
                                    width: width,
                                    offset: seriesOffset
                                })
                            })
                        }
                    }
                    else {
                        $.each(series, function(i, singleSeries) {
                            stackName = singleSeries.getStackName && singleSeries.getStackName();
                            stackName = stackName || i.toString();
                            if (!stackKeepers[stackName])
                                stackKeepers[stackName] = [];
                            stackKeepers[stackName].push(singleSeries)
                        });
                        $.each(series, function(i, singleSeries) {
                            $.each(singleSeries.getPoints(), function(_, point) {
                                var argument = point.argument;
                                if (!argumentsKeeper.hasOwnProperty(argument))
                                    argumentsKeeper[argument.valueOf()] = 1
                            })
                        });
                        for (var argument in argumentsKeeper) {
                            if (!argumentsKeeper.hasOwnProperty(argument))
                                continue;
                            stacksWithArgument = [];
                            $.each(stackKeepers, function(stackName, seriesInStack) {
                                $.each(seriesInStack, function(i, singleSeries) {
                                    point = singleSeries.getPointByArg(argument);
                                    if (point && point.value) {
                                        stacksWithArgument.push(stackName);
                                        return false
                                    }
                                })
                            });
                            count = stacksWithArgument.length;
                            spacing = _round(interval * 0.7 / count * 0.2);
                            if (spacing < 1)
                                spacing = 1;
                            width = _round((interval * 0.7 - spacing * (count - 1)) / count);
                            if (width < 2)
                                width = 2;
                            middleIndex = count / 2;
                            $.each(stackKeepers, function(stackName, seriesInStack) {
                                stackIndex = $.inArray(stackName, stacksWithArgument);
                                if (stackIndex === -1)
                                    return;
                                seriesOffset = (stackIndex - middleIndex + 0.5) * width - (middleIndex - stackIndex - 0.5) * spacing;
                                $.each(seriesInStack, function(i, singleSeries) {
                                    var point = singleSeries.getPointByArg(argument);
                                    if (point && point.value)
                                        point.correctCoordinates({
                                            width: width,
                                            offset: seriesOffset
                                        })
                                })
                            })
                        }
                    }
                };
            var getVisibleSeries = function(that) {
                    return $.map(that.series, function(s) {
                            return s.isVisible() ? s : null
                        })
                };
            var adjustBarSeriesDimensions = function(translators) {
                    var debug = DX.utils.debug;
                    debug.assert(translators, "translator was not passed or empty");
                    var that = this,
                        equalBarWidth = that.equalBarWidth,
                        series = getVisibleSeries(that);
                    adjustBarSeriesDimensionsCore(series, getInterval(that, translators), series.length, equalBarWidth, function(seriesIndex) {
                        return seriesIndex
                    })
                };
            var adjustStackedBarSeriesDimensions = function(translators) {
                    var debug = DX.utils.debug;
                    debug.assert(translators, "translators was not passed or empty");
                    var that = this,
                        interval,
                        series = getVisibleSeries(that),
                        stackIndexes = {},
                        stackCount = 0,
                        equalBarWidth = that.equalBarWidth;
                    $.each(series, function() {
                        var stackName = this.getStackName();
                        if (!stackIndexes.hasOwnProperty(stackName))
                            stackIndexes[stackName] = stackCount++
                    });
                    adjustBarSeriesDimensionsCore(series, getInterval(that, translators), stackCount, equalBarWidth, function(seriesIndex) {
                        return stackIndexes[series[seriesIndex].getStackName()]
                    })
                };
            var adjustStackedSeriesValues = function() {
                    var that = this,
                        series = getVisibleSeries(that),
                        stackKeepers = {
                            positive: {},
                            negative: {}
                        };
                    $.each(series, function(_, singleSeries) {
                        var points = singleSeries.getPoints();
                        $.each(points, function(index, point) {
                            var value = point.initialValue,
                                argument = point.argument,
                                stackName = singleSeries.getStackName ? singleSeries.getStackName() : "default",
                                valueType = value >= 0 ? "positive" : "negative",
                                currentStack;
                            stackKeepers[valueType][stackName] = stackKeepers[valueType][stackName] || {};
                            currentStack = stackKeepers[valueType][stackName];
                            if (currentStack[argument.valueOf()]) {
                                points[index].correctValue(currentStack[argument.valueOf()]);
                                currentStack[argument.valueOf()] += value
                            }
                            else {
                                currentStack[argument.valueOf()] = value;
                                points[index].resetCorrection()
                            }
                        })
                    });
                    setPercentStackedValues(series, stackKeepers, that.fullStacked)
                };
            var setPercentStackedValues = function(series, stackKeepers, fullStacked) {
                    $.each(series, function(_, singleSeries) {
                        var points = singleSeries.getPoints();
                        $.each(points, function(index, point) {
                            var value = point.value,
                                stackName = singleSeries.getStackName ? singleSeries.getStackName() : "default",
                                valueType = value >= 0 ? "positive" : "negative",
                                currentStack;
                            stackKeepers[valueType][stackName] = stackKeepers[valueType][stackName] || {};
                            currentStack = stackKeepers[valueType][stackName];
                            points[index].setPercentValue(currentStack[point.argument.valueOf()], fullStacked)
                        })
                    })
                };
            var getMinShownBusinessValue = function(that, translators, minBarSize) {
                    var rotated = that.rotated,
                        valTranslator = rotated ? translators.x : translators.y,
                        canvas = valTranslator.getCanvasVisibleArea();
                    if (minBarSize)
                        return _abs(valTranslator.untranslate(canvas.min) - valTranslator.untranslate(canvas.min + minBarSize))
                };
            var updateStackedSeriesValues = function(translators) {
                    var that = this,
                        series = getVisibleSeries(that),
                        stackKeepers = {
                            positive: {},
                            negative: {}
                        };
                    $.each(series, function(_, singleSeries) {
                        var points = singleSeries.getPoints(),
                            minBarSize = singleSeries.getOptions().minBarSize;
                        $.each(points, function(index, point) {
                            var value = point.value,
                                minValue = point.minValue,
                                argument = point.argument,
                                updateValue,
                                pointSize,
                                minShownBusinessValue,
                                stackName = singleSeries.getStackName ? singleSeries.getStackName() : "default",
                                valueType = value >= 0 ? "positive" : "negative",
                                currentStack;
                            minShownBusinessValue = getMinShownBusinessValue(that, translators, minBarSize);
                            currentStack = stackKeepers[valueType][stackName] = stackKeepers[valueType][stackName] || {};
                            if (currentStack[argument.valueOf()]) {
                                minValue = utils.isNumber(minValue) ? minValue : 0,
                                pointSize = _abs(minValue - value);
                                if (minShownBusinessValue && pointSize < minShownBusinessValue)
                                    updateValue = minShownBusinessValue;
                                else
                                    updateValue = value - minValue;
                                points[index].minValue = currentStack[argument.valueOf()];
                                points[index].value = currentStack[argument.valueOf()] + updateValue;
                                currentStack[argument.valueOf()] += updateValue
                            }
                            else {
                                pointSize = value;
                                if (minShownBusinessValue && pointSize < minShownBusinessValue)
                                    updateValue = minShownBusinessValue;
                                else
                                    updateValue = value;
                                points[index].value = updateValue;
                                currentStack[argument.valueOf()] = updateValue
                            }
                        })
                    });
                    if (that.fullStacked)
                        updateFullStackedSeriesValues(series, stackKeepers)
                };
            var updateFullStackedSeriesValues = function(series, stackKeepers) {
                    $.each(series, function(_, singleSeries) {
                        var stackName = singleSeries.getStackName ? singleSeries.getStackName() : "default",
                            points = singleSeries.getPoints();
                        $.each(points, function(index, point) {
                            var value = point.value,
                                argument = point.argument,
                                valueType = value >= 0 ? "positive" : "negative",
                                currentStack;
                            stackKeepers[valueType][stackName] = stackKeepers[valueType][stackName] || {};
                            currentStack = stackKeepers[valueType][stackName];
                            points[index].value = points[index].value / currentStack[argument.valueOf()] || 0;
                            if (DX.utils.isNumber(points[index].minValue))
                                points[index].minValue = points[index].minValue / currentStack[argument.valueOf()] || 0
                        })
                    })
                };
            var updateBarSeriesValues = function(translators) {
                    var that = this;
                    $.each(that.series, function(_, singleSeries) {
                        var points = singleSeries.getPoints(),
                            minBarSize = singleSeries.getOptions().minBarSize;
                        $.each(points, function(index, point) {
                            var value = point.value,
                                updateValue,
                                pointSize,
                                minShownBusinessValue;
                            minShownBusinessValue = getMinShownBusinessValue(that, translators, minBarSize);
                            pointSize = _abs(value);
                            if (minShownBusinessValue && pointSize < minShownBusinessValue)
                                updateValue = value >= 0 ? minShownBusinessValue : -minShownBusinessValue;
                            else
                                updateValue = value;
                            points[index].value = updateValue
                        })
                    })
                };
            var adjustCandlestickSeriesDimensions = function(translators) {
                    var debug = DX.utils.debug;
                    debug.assert(translators, "translator was not passed or empty");
                    var that = this,
                        series = getVisibleSeries(that);
                    adjustBarSeriesDimensionsCore(series, getInterval(that, translators), series.length, true, function(seriesIndex) {
                        return seriesIndex
                    })
                };
            var getInterval = function(that, translators) {
                    var argTranslator = !that.rotated ? translators.x : translators.y;
                    return that.interval = argTranslator.getInterval()
                };
            var adjustBubbleSeriesDimensions = function(translators) {
                    var debug = DX.utils.debug;
                    debug.assert(translators, "translator was not passed or empty");
                    var that = this,
                        series = getVisibleSeries(that),
                        points,
                        i,
                        visibleAreaX = translators.x.getCanvasVisibleArea(),
                        visibleAreaY = translators.y.getCanvasVisibleArea(),
                        min = Math.min(visibleAreaX.max - visibleAreaX.min, visibleAreaY.max - visibleAreaY.min),
                        minBubbleArea = _pow(that.minBubbleSize, 2),
                        maxBubbleArea = _pow(min * that.maxBubbleSize, 2),
                        equalBubbleSize = (min * that.maxBubbleSize + that.minBubbleSize) / 2,
                        minPointSize = Infinity,
                        maxPointSize = 0,
                        pointSize,
                        bubbleArea,
                        sizeProportion,
                        sizeDispersion,
                        areaDispersion;
                    for (i = 0; i < series.length; i++) {
                        points = series[i].getPoints();
                        $.each(points, function(_, point) {
                            maxPointSize = maxPointSize > point.size ? maxPointSize : point.size;
                            minPointSize = minPointSize < point.size ? minPointSize : point.size
                        })
                    }
                    sizeDispersion = maxPointSize - minPointSize;
                    areaDispersion = _abs(maxBubbleArea - minBubbleArea);
                    minPointSize = minPointSize < 0 ? 0 : minPointSize;
                    for (i = 0; i < series.length; i++) {
                        points = series[i].getPoints();
                        $.each(points, function(_, point) {
                            if (maxPointSize === minPointSize)
                                pointSize = _round(equalBubbleSize);
                            else {
                                sizeProportion = _abs(point.size - minPointSize) / sizeDispersion;
                                bubbleArea = areaDispersion * sizeProportion + minBubbleArea;
                                pointSize = _round(Math.sqrt(bubbleArea))
                            }
                            point.correctCoordinates(pointSize)
                        })
                    }
                };
            return {
                    ctor: ctor,
                    dispose: dispose,
                    add: add
                }
        }())
    })(jQuery, DevExpress);
    /*! Module viz-core, file baseSeries.js */
    (function($, DX) {
        var viz = DX.viz,
            utils = DX.utils,
            _isDefined = utils.isDefined,
            _each = $.each,
            _extend = $.extend,
            _isEmptyObject = $.isEmptyObject,
            _Event = $.Event,
            _noop = $.noop,
            SELECTED_STATE = 2,
            HOVER_STATE = 1,
            NONE_MODE = "none",
            INCLUDE_POINTS = "includepoints",
            EXLUDE_POINTS = "excludepoints",
            ALL_SERIES_POINTS_MODE = "allseriespoints",
            APPLY_SELECTED = "applySelected",
            APPLY_HOVER = "applyHover",
            getEmptyBusinessRange = function() {
                return {
                        arg: {},
                        val: {}
                    }
            };
        viz.core.series.mixins = {
            chart: {},
            pieChart: {}
        };
        viz.core.series.Series = DX.Class.inherit({
            ctor: function(renderer, options) {
                this.fullState = 0;
                this._renderer = renderer;
                this._group = renderer.createGroup({"class": "dxc-series"});
                this.updateOptions(options)
            },
            update: function(data, options) {
                this.updateOptions(options);
                this.updateData(data)
            },
            _createLegendState: function(styleOptions, defaultColor) {
                return {
                        fill: styleOptions.color || defaultColor,
                        hatching: styleOptions.hatching
                    }
            },
            getLegendStyles: function() {
                return this._styles.legendStyles
            },
            _createStyles: function(options) {
                var mainSeriesColor = options.mainSeriesColor,
                    specialMainColor = this._getSpecialColor(mainSeriesColor);
                this._styles = {
                    normal: this._parseStyle(options, mainSeriesColor, mainSeriesColor),
                    hover: this._parseStyle(options.hoverStyle || {}, specialMainColor, mainSeriesColor),
                    selection: this._parseStyle(options.selectionStyle || {}, specialMainColor, mainSeriesColor),
                    legendStyles: {
                        normal: this._createLegendState(options, mainSeriesColor),
                        hover: this._createLegendState(options.hoverStyle || {}, specialMainColor),
                        selection: this._createLegendState(options.selectionStyle || {}, specialMainColor)
                    }
                }
            },
            setAdjustSeriesLabels: function(adjustSeriesLabels) {
                _each(this._points || [], function(_, point) {
                    point.setAdjustSeriesLabels(adjustSeriesLabels)
                })
            },
            setClippingParams: function(id, forceClipping) {
                this._paneClipRectID = id;
                this._forceClipping = forceClipping
            },
            getTagField: function() {
                return this._options.tagField || "tag"
            },
            getValueFields: _noop,
            getArgumentField: _noop,
            getPoints: function() {
                return this._points
            },
            _createPoint: function(data, pointsArray, index) {
                data.index = index;
                if (this._checkData(data)) {
                    var point = pointsArray[index],
                        options = this._customizePoint(data) || this._getCreatingPointOptions();
                    if (point)
                        point.update(data, options);
                    else {
                        point = viz.core.CoreFactory.createPoint(data, options);
                        pointsArray.push(point)
                    }
                    this.pointsByArgument[point.argument.valueOf()] = this.pointsByArgument[point.argument.valueOf()] || point;
                    return true
                }
            },
            getRangeData: function(zoomArgs, calcIntervalFunction) {
                return this._visible ? _extend(true, {}, this._getRangeData(zoomArgs, calcIntervalFunction)) : getEmptyBusinessRange()
            },
            _deleteElementsGroup: function() {
                if (this._elementsGroup) {
                    this._elementsGroup.detach();
                    this._elementsGroup = null
                }
            },
            _deleteBordersGroup: function() {
                if (this._bordersGroup) {
                    this._bordersGroup.detach();
                    this._bordersGroup = null
                }
            },
            _saveOldAnimationMethods: function() {
                this._oldClearingAnimation = this._clearingAnimation;
                this._oldUpdateElement = this._updateElement
            },
            _deleteOldAnimationMethods: function() {
                this._oldClearingAnimation = null;
                this._oldUpdateElement = null
            },
            updateOptions: function(newOptions) {
                var that = this,
                    widgetType = newOptions.widgetType,
                    oldType = that.type,
                    newType = newOptions.type;
                that.type = newType && newType.toString().toLowerCase();
                if (!that._checkType(widgetType)) {
                    that.dispose();
                    that.isUpdated = false;
                    return
                }
                if (oldType !== that.type) {
                    that._firstDrawing = true;
                    that._saveOldAnimationMethods();
                    that._resetType(oldType, widgetType);
                    that._setType(that.type, widgetType)
                }
                that._options = newOptions;
                that._pointOptions = null;
                that._deletePatterns();
                that._patterns = [];
                that.name = newOptions.name;
                that.pane = newOptions.pane;
                that.axis = newOptions.axis;
                that.tag = newOptions.tag;
                that._createStyles(newOptions);
                that._updateOptions(newOptions);
                that._visible = newOptions.visible;
                that.isUpdated = true
            },
            _disposePoints: function(points) {
                $.each(points || [], function(_, p) {
                    p.dispose()
                })
            },
            _correctPointsLength: function(length, points) {
                this._disposePoints(this._oldPoints);
                this._oldPoints = points.splice(length, points.length)
            },
            _getTicksForAggregation: function(min, max, screenDelta, pointSize) {
                return viz.core.tickProvider.getTicks({
                        min: min,
                        max: max,
                        screenDelta: screenDelta,
                        gridSpacingFactor: pointSize
                    })
            },
            updateDataType: function(settings) {
                var that = this;
                that.argumentType = settings.argumentType;
                that.valueType = settings.valueType;
                that.argumentAxisType = settings.argumentAxisType;
                that.valueAxisType = settings.valueAxisType
            },
            getValueCategories: function() {
                return this._options.valueCategories || []
            },
            getOptions: function() {
                return this._options
            },
            getArgumentCategories: function() {
                return this._options.argumentCategories || []
            },
            updateData: function(data) {
                var that = this,
                    points,
                    lastPointIndex = 0,
                    options = that._options,
                    pointData;
                that.pointsByArgument = {};
                points = that._originalPoints || [];
                that._rangeData = getEmptyBusinessRange();
                if (data && data.length)
                    that._canRenderCompleteHandle = true;
                that._beginUpdateData();
                _each(data, function(index, dataItem) {
                    pointData = that._getPointData(dataItem, options);
                    if (that._createPoint(pointData, points, lastPointIndex)) {
                        that._processRange(points[lastPointIndex], lastPointIndex > 0 ? points[lastPointIndex - 1] : null);
                        lastPointIndex++
                    }
                });
                that._points = that._originalPoints = points;
                that._correctPointsLength(lastPointIndex, points);
                that._endUpdateData()
            },
            getTeamplatedFields: function() {
                var that = this,
                    fields = that.getValueFields(),
                    teampleteFields = [];
                fields.push(that.getTagField());
                _each(fields, function(_, field) {
                    var fieldsObject = {};
                    fieldsObject.teamplateField = field + that.name;
                    fieldsObject.originalField = field;
                    teampleteFields.push(fieldsObject)
                });
                return teampleteFields
            },
            resamplePoints: function(translators, min, max) {
                var that = this,
                    originalPoints = that.getAllPoints(),
                    argTranslator = that._options.rotated ? translators.y : translators.x,
                    minI,
                    maxI,
                    sizePoint,
                    ticks;
                if (originalPoints.length) {
                    _each(originalPoints, function(i, point) {
                        minI = point.argument - min <= 0 ? i : minI;
                        if (!maxI)
                            maxI = point.argument - max > 0 ? i : null
                    });
                    minI = minI ? minI : 1;
                    maxI = utils.isDefined(maxI) ? maxI : originalPoints.length - 1;
                    min = originalPoints[minI - 1].argument;
                    max = originalPoints[maxI].argument;
                    sizePoint = that._getPointSize();
                    if (that.argumentAxisType !== "discrete" && that.valueAxisType !== "discrete")
                        ticks = that._getTicksForAggregation(min, max, argTranslator.canvasLength, sizePoint);
                    else
                        ticks = argTranslator.canvasLength / sizePoint;
                    that._points = that._resample(ticks, ticks.tickInterval)
                }
            },
            _removeOldSegments: function(startIndex) {
                var that = this;
                _each(that._graphics.splice(startIndex, that._graphics.length) || [], function(_, elem) {
                    that._removeElement(elem)
                });
                if (that._trackers)
                    _each(that._trackers.splice(startIndex, that._trackers.length) || [], function(_, elem) {
                        elem.remove()
                    })
            },
            draw: function(translators, animationEnabled, hideLayoutLabels, legendCallback) {
                var that = this;
                if (that._oldClearingAnimation && animationEnabled && that._firstDrawing) {
                    var drawComplete = function() {
                            that._draw(translators, true, hideLayoutLabels)
                        };
                    that._oldClearingAnimation(translators, drawComplete)
                }
                else
                    that._draw(translators, animationEnabled, hideLayoutLabels, legendCallback)
            },
            _clearSeries: function() {
                this._deleteElementsGroup();
                this._deleteBordersGroup();
                this._deleteTrackers();
                this._graphics = [];
                this._trackers = []
            },
            _draw: function(translators, animationEnabled, hideLayoutLabels, legendCallback) {
                var that = this,
                    points = that._points || [],
                    segment = [],
                    segmentCount = 0,
                    firstDrawing = that._firstDrawing,
                    markersGroup,
                    labelsGroup;
                that._graphics = that._graphics || [],
                that._graphics = that._graphics || [];
                that._prepareSeriesToDrawing();
                if (!that._visible) {
                    animationEnabled = false;
                    this._group.detach();
                    return
                }
                else
                    that._group.append(that._options.seriesGroup);
                that.translators = translators;
                that._createGroups(animationEnabled, undefined, firstDrawing);
                that._segments = [];
                markersGroup = that._markersGroup;
                labelsGroup = that._labelsGroup;
                that._drawedPoints = [];
                that._firstDrawing = points.length ? false : true;
                _each(this._patterns || [], function(_, pattern) {
                    pattern.append()
                });
                _each(points, function(i, p) {
                    p.translate(translators);
                    if (p.hasValue()) {
                        that._drawPoint(p, markersGroup, labelsGroup, animationEnabled, firstDrawing);
                        segment.push(p)
                    }
                    else if (segment.length) {
                        that._drawSegment(segment, animationEnabled, segmentCount++);
                        segment = []
                    }
                });
                segment.length && that._drawSegment(segment, animationEnabled, segmentCount++);
                that._removeOldSegments(segmentCount);
                that._defaultSegments = that._generateDefaultSegments();
                that._adjustLabels(firstDrawing);
                hideLayoutLabels && that.hideLabels();
                animationEnabled && that._animate(firstDrawing);
                if (that.isSelected())
                    that._changeStyle(legendCallback, APPLY_SELECTED);
                else if (this.isHovered())
                    this._changeStyle(legendCallback, APPLY_HOVER)
            },
            drawTrackers: function() {
                var that = this,
                    trackers = that._trackers = that._trackers || [],
                    trackersGroup = that._trackersGroup = that._trackersGroup || that._renderer.createGroup(),
                    markerTrackerGroup = that._markerTrackerGroup = that._markerTrackerGroup || that._renderer.createGroup();
                if (!that.isVisible()) {
                    trackersGroup.detach();
                    markerTrackerGroup.detach();
                    return
                }
                else {
                    trackersGroup.append(that._options.trackersGroup);
                    markerTrackerGroup.append(that._options.markerTrackerGroup)
                }
                _each(that._segments || [], function(i, segment) {
                    if (!trackers[i]) {
                        trackers[i] = that._drawTrackerElement(segment).append(trackersGroup);
                        trackers[i].data({series: that})
                    }
                    else
                        that._updateTrackerElement(segment, trackers[i])
                });
                _each(that.getVisiblePoints(), function(_, p) {
                    p.drawTracker(that._renderer, markerTrackerGroup)
                });
                that._applyTrackersClippings()
            },
            _checkType: function(widgetType) {
                return !!viz.core.series.mixins[widgetType][this.type]
            },
            _resetType: function(seriesType, widgetType) {
                var that = this;
                if (seriesType)
                    _each(viz.core.series.mixins[widgetType][seriesType], function(methodName) {
                        delete that[methodName]
                    })
            },
            _setType: function(seriesType, widgetType) {
                var that = this;
                _each(viz.core.series.mixins[widgetType][seriesType], function(methodName, method) {
                    that[methodName] = method
                })
            },
            _setSelectedState: function(state, mode, legendCallback) {
                this.lastSelectionMode = mode = (mode || this._options.selectionMode).toLowerCase();
                if (state && !this.isSelected()) {
                    this.fullState = this.fullState | SELECTED_STATE;
                    this._changeStyle(legendCallback, APPLY_SELECTED)
                }
                else if (!state && this.isSelected()) {
                    this.fullState = this.fullState & ~SELECTED_STATE;
                    if (this.isHovered())
                        this._changeStyle(legendCallback, APPLY_HOVER);
                    else
                        this._changeStyle(legendCallback, "resetItem")
                }
            },
            _setHoverState: function(state, mode, legendCallback) {
                this.lastHoverMode = mode = (mode || this._options.hoverMode).toLowerCase();
                if (state && !this.isHovered()) {
                    this.fullState = this.fullState | HOVER_STATE;
                    !this.isSelected() && this._changeStyle(legendCallback, APPLY_HOVER)
                }
                else if (!state && this.isHovered()) {
                    this.fullState = this.fullState & ~HOVER_STATE;
                    !this.isSelected() && this._changeStyle(legendCallback, "resetItem")
                }
            },
            setHoverState: function(mode, legendCallback) {
                this._setHoverState(true, mode, legendCallback)
            },
            releaseHoverState: function(mode, legendCallback) {
                this._setHoverState(false, mode, legendCallback)
            },
            setSelectedState: function(mode, legendCallback) {
                this._setSelectedState(true, mode, legendCallback)
            },
            releaseSelectedState: function(mode, legendCallback) {
                this._setSelectedState(false, mode, legendCallback)
            },
            isFullStackedSeries: function() {
                return this.type.indexOf("fullstacked") === 0
            },
            isStackedSeries: function() {
                return this.type.indexOf("stacked") === 0
            },
            isFinancialSeries: function() {
                return this.type === "stock" || this.type === "candlestick"
            },
            _changeStyle: function(legendCallBack, legendAction) {
                var style = this._calcStyle(),
                    pointStyle;
                if (style.mode === NONE_MODE)
                    return;
                legendCallBack && legendCallBack(legendAction);
                if (style.mode === INCLUDE_POINTS || style.mode === ALL_SERIES_POINTS_MODE) {
                    pointStyle = style.pointStyle;
                    _each(this._points || [], function(_, p) {
                        !p.isSelected() && p.applyStyle(pointStyle)
                    })
                }
                this._applyStyle(style.series)
            },
            _calcStyle: function() {
                var styles = this._styles,
                    isHoverIncludeModeAndSeriesExcludeMode = false,
                    result;
                switch (this.fullState & 3) {
                    case 0:
                        result = {
                            pointStyle: "normal",
                            mode: INCLUDE_POINTS,
                            series: styles.normal
                        };
                        break;
                    case 1:
                        result = {
                            pointStyle: "hover",
                            mode: this.lastHoverMode,
                            series: styles.hover
                        };
                        break;
                    case 2:
                        result = {
                            pointStyle: "selection",
                            mode: this.lastSelectionMode,
                            series: styles.selection
                        };
                        break;
                    case 3:
                        isHoverIncludeModeAndSeriesExcludeMode = this.lastSelectionMode === EXLUDE_POINTS && (this.lastHoverMode === INCLUDE_POINTS || this.lastHoverMode === ALL_SERIES_POINTS_MODE);
                        result = {
                            pointStyle: isHoverIncludeModeAndSeriesExcludeMode ? "normal" : "selection",
                            mode: isHoverIncludeModeAndSeriesExcludeMode ? INCLUDE_POINTS : this.lastSelectionMode,
                            series: styles.selection
                        }
                }
                return result
            },
            _getMainAxisName: function() {
                return this._options.rotated ? "X" : "Y"
            },
            areLabelsVisible: function() {
                return !_isDefined(this._options.maxLabelCount) || this._points.length <= this._options.maxLabelCount
            },
            getLabelVisibility: function() {
                return this.areLabelsVisible() && this._options.label && this._options.label.visible
            },
            _customizePoint: function(pointData) {
                var that = this,
                    options = that._options,
                    customizePoint = that._options.customizePoint,
                    customizeObject,
                    pointOptions,
                    customLabelOptions,
                    customOptions,
                    customizeLabel = that._options.customizeLabel,
                    useLabelCustomOptions,
                    usePointCustomOptions;
                if (customizeLabel && customizeLabel.call) {
                    customizeObject = _extend({
                        seriesName: that.name,
                        series: that
                    }, pointData);
                    customLabelOptions = customizeLabel.call(customizeObject, customizeObject);
                    useLabelCustomOptions = customLabelOptions && !_isEmptyObject(customLabelOptions);
                    customLabelOptions = useLabelCustomOptions ? _extend(true, {}, options.label, customLabelOptions) : null
                }
                if (customizePoint && customizePoint.call) {
                    customizeObject = customizeObject || _extend({
                        seriesName: that.name,
                        series: that
                    }, pointData);
                    customOptions = customizePoint.call(customizeObject, customizeObject);
                    usePointCustomOptions = customOptions && !_isEmptyObject(customOptions)
                }
                if (useLabelCustomOptions || usePointCustomOptions) {
                    pointOptions = that._parsePointOptions(that._preparePointOptions(customOptions), customLabelOptions || options.label);
                    pointOptions.styles.useLabelCustomOptions = useLabelCustomOptions;
                    pointOptions.styles.usePointCustomOptions = usePointCustomOptions
                }
                return pointOptions
            },
            _getLabelOptions: function(labelOptions, defaultColor) {
                var opt = labelOptions || {},
                    labelFont = opt.font || {},
                    labelBorder = opt.border || {},
                    labelConnector = opt.connector || {},
                    labelAttributes = {
                        align: opt.alignment,
                        font: {
                            color: opt.backgroundColor === "none" && labelFont.color.toLowerCase() === "#ffffff" && opt.position !== "inside" ? defaultColor : labelFont.color,
                            family: labelFont.family,
                            weight: labelFont.weight,
                            size: labelFont.size,
                            opacity: labelFont.opacity
                        },
                        style: opt.style
                    },
                    backgroundAttr = {
                        fill: opt.backgroundColor || defaultColor,
                        strokeWidth: labelBorder.visible ? labelBorder.width || 0 : 0,
                        stroke: labelBorder.visible && labelBorder.width ? labelBorder.color : "none",
                        dashStyle: labelBorder.dashStyle
                    },
                    connectorAttr = {
                        stroke: labelConnector.visible && labelConnector.width ? labelConnector.color || defaultColor : "none",
                        strokeWidth: labelConnector.visible ? labelConnector.width || 0 : 0
                    };
                return {
                        format: opt.format,
                        argumentFormat: opt.argumentFormat,
                        precision: opt.precision,
                        argumentPrecision: opt.argumentPrecision,
                        percentPrecision: opt.percentPrecision,
                        customizeText: $.isFunction(opt.customizeText) ? opt.customizeText : undefined,
                        attributes: labelAttributes,
                        visible: labelFont.size !== 0 ? opt.visible : false,
                        showForZeroValues: opt.showForZeroValues,
                        horizontalOffset: opt.horizontalOffset,
                        verticalOffset: opt.verticalOffset,
                        radialOffset: opt.radialOffset,
                        background: backgroundAttr,
                        position: opt.position,
                        connector: connectorAttr,
                        rotationAngle: opt.rotationAngle,
                        resolveLabelsOverlapping: this._options.resolveLabelsOverlapping
                    }
            },
            show: function() {
                if (!this._visible) {
                    this._visible = true;
                    this.hidePointTooltip();
                    this._options.visibilityChanged()
                }
            },
            hide: function() {
                if (this._visible) {
                    this._visible = false;
                    this.hidePointTooltip();
                    this._options.visibilityChanged()
                }
            },
            hideLabels: function() {
                _each(this._points, function(_, point) {
                    point._label.hide()
                })
            },
            _parsePointOptions: function(pointOptions, labelOptions) {
                var options = this._options,
                    styles = this._createPointStyles(pointOptions),
                    parsedOptions;
                parsedOptions = _extend(true, {}, pointOptions, {
                    type: options.type,
                    tag: this.tag,
                    rotated: options.rotated,
                    series: this,
                    styles: styles,
                    visibilityChanged: options.visibilityChanged
                });
                parsedOptions.label = this._getLabelOptions(labelOptions, styles.normal.fill);
                return parsedOptions
            },
            _resample: function(ticks, ticksInterval) {
                var that = this,
                    fusPoints = [],
                    arrayFusPoints,
                    nowIndexTicks = 0,
                    lastPointIndex = 0,
                    originalPoints = that.getAllPoints();
                if (that.argumentAxisType === "discrete" || that.valueAxisType === "discrete") {
                    ticksInterval = originalPoints.length / ticks;
                    arrayFusPoints = $.map(originalPoints, function(point, index) {
                        if (Math.floor(nowIndexTicks) <= index) {
                            nowIndexTicks += ticksInterval;
                            return point
                        }
                        point.setInvisibility();
                        return null
                    });
                    return arrayFusPoints
                }
                that._aggregatedPoints = that._aggregatedPoints || [];
                _each(originalPoints, function(_, point) {
                    switch (that._isInInterval(point.argument, ticks, nowIndexTicks, ticksInterval)) {
                        case true:
                            fusPoints.push(point);
                            break;
                        case"nextInterval":
                            var pointData = that._fusionPoints(fusPoints, ticks[nowIndexTicks], nowIndexTicks);
                            while (that._isInInterval(point.argument, ticks, nowIndexTicks, ticksInterval) === "nextInterval")
                                nowIndexTicks++;
                            fusPoints = [];
                            that._isInInterval(point.argument, ticks, nowIndexTicks, ticksInterval) === true && fusPoints.push(point);
                            if (that._createPoint(pointData, that._aggregatedPoints, lastPointIndex))
                                lastPointIndex++
                    }
                });
                if (fusPoints.length) {
                    var pointData = that._fusionPoints(fusPoints, ticks[nowIndexTicks], nowIndexTicks);
                    if (that._createPoint(pointData, that._aggregatedPoints, lastPointIndex))
                        lastPointIndex++
                }
                that._correctPointsLength(lastPointIndex, that._aggregatedPoints);
                that._endUpdateData();
                return that._aggregatedPoints
            },
            _isInInterval: function(argument, ticks, nowIndexTicks, ticksInterval) {
                var minTick = ticks[nowIndexTicks],
                    maxTick = ticks[nowIndexTicks + 1],
                    sumMinTickTicksInterval;
                ticksInterval = $.isNumeric(ticksInterval) ? ticksInterval : utils.convertDateTickIntervalToMilliseconds(ticksInterval);
                sumMinTickTicksInterval = utils.isDate(minTick) ? new Date(minTick.getTime() + ticksInterval) : minTick + ticksInterval;
                if (argument >= minTick && argument < sumMinTickTicksInterval)
                    return true;
                if (argument < minTick || maxTick === undefined)
                    return false;
                return "nextInterval"
            },
            canRenderCompleteHandle: function() {
                var result = this._canRenderCompleteHandle;
                delete this._canRenderCompleteHandle;
                return !!result
            },
            isHovered: function() {
                return !!(this.fullState & 1)
            },
            isSelected: function() {
                return !!(this.fullState & 2)
            },
            isVisible: function() {
                return this._visible
            },
            getAllPoints: function() {
                return (this._originalPoints || []).slice()
            },
            getPointByPos: function(pos) {
                return (this._points || [])[pos]
            },
            getVisiblePoints: function() {
                return (this._drawedPoints || []).slice()
            },
            setPointHoverState: function(point, legendCallback) {
                point.fullState = point.fullState | HOVER_STATE;
                if (!(this.isSelected() && (this.lastSelectionMode === ALL_SERIES_POINTS_MODE || this.lastSelectionMode === INCLUDE_POINTS)) && !point.isSelected()) {
                    point.applyStyle("hover");
                    legendCallback && legendCallback("applyHover")
                }
            },
            releasePointHoverState: function(point, legendCallback) {
                point.fullState = point.fullState & ~HOVER_STATE;
                if (!(this.isSelected() && (this.lastSelectionMode === ALL_SERIES_POINTS_MODE || this.lastSelectionMode === INCLUDE_POINTS)) && !point.isSelected())
                    if (!(this.isHovered() && (this.lastHoverMode === ALL_SERIES_POINTS_MODE || this.lastHoverMode === INCLUDE_POINTS))) {
                        point.applyStyle("normal");
                        legendCallback && legendCallback("resetItem")
                    }
            },
            setPointSelectedState: function(point, legendCallback) {
                point.fullState = point.fullState | SELECTED_STATE;
                point.applyStyle("selection");
                legendCallback && legendCallback("applySelected")
            },
            releasePointSelectedState: function(point, legendCallback) {
                point.fullState = point.fullState & ~SELECTED_STATE;
                if (this.isHovered() && (this.lastHoverMode === ALL_SERIES_POINTS_MODE || this.lastHoverMode === INCLUDE_POINTS) || point.isHovered()) {
                    point.applyStyle("hover");
                    legendCallback && legendCallback("applyHover")
                }
                else if (this.isSelected() && (this.lastSelectionMode === ALL_SERIES_POINTS_MODE || this.lastSelectionMode === INCLUDE_POINTS)) {
                    point.applyStyle("selection");
                    legendCallback && legendCallback("applySelected")
                }
                else {
                    point.applyStyle("normal");
                    legendCallback && legendCallback("resetItem")
                }
            },
            selectPoint: function(point) {
                this._options.seriesGroup && this._options.seriesGroup.trigger(new _Event("selectpoint"), point)
            },
            deselectPoint: function(point) {
                this._options.seriesGroup && this._options.seriesGroup.trigger(new _Event("deselectpoint"), point)
            },
            showPointTooltip: function(point) {
                this._options.seriesGroup && this._options.seriesGroup.trigger(new _Event("showpointtooltip"), point)
            },
            hidePointTooltip: function(point) {
                this._options.seriesGroup && this._options.seriesGroup.trigger(new _Event("hidepointtooltip"), point)
            },
            select: function() {
                this._options.seriesGroup && this._options.seriesGroup.trigger(new _Event("selectseries", {target: this}), this._options.selectionMode);
                this._group.toForeground();
                this._trackersGroup && this._trackersGroup.toBackground()
            },
            clearSelection: function clearSelection() {
                this._options.seriesGroup && this._options.seriesGroup.trigger(new _Event("deselectseries", {target: this}), this._options.selectionMode)
            },
            getPointByArg: function(arg) {
                return this.pointsByArgument[arg.valueOf()] || null
            },
            _deletePoints: function() {
                this._disposePoints(this._originalPoints);
                this._disposePoints(this._aggregatedPoints);
                this._disposePoints(this._oldPoints);
                this._points = null;
                this._oldPoints = null;
                this._aggregatedPoints = null;
                this._originalPoints = null;
                this._drawedPoint = null
            },
            _deletePatterns: function() {
                _each(this._patterns || [], function(_, pattern) {
                    pattern && pattern.dispose()
                });
                this._patterns = null
            },
            _deleteTrackers: function() {
                var that = this;
                _each(that._trackers || [], function(_, tracker) {
                    tracker.remove()
                });
                that._trackersGroup && that._trackersGroup.detach();
                that._markerTrackerGroup && that._markerTrackerGroup.detach();
                that._trackers = that._trackersGroup = that._markerTrackerGroup = null
            },
            dispose: function() {
                this._deletePoints();
                this._group.detach();
                this._labelsGroup && this._labelsGroup.detach();
                this._deletePatterns();
                this._deleteTrackers();
                this._group = null;
                this._markersGroup = null;
                this._elementsGroup = null;
                this._bordersGroup = null;
                this._labelsGroup = null;
                this._graphics = null;
                this._rangeData = null;
                this._renderer = null;
                this.translators = null;
                this._styles = null;
                this._options = null;
                this._pointOptions = null;
                this._drawedPoints = null;
                this._aggregatedPoints = null;
                this.pointsByArgument = null;
                this._segments = null
            },
            correctPosition: _noop,
            getColor: function() {
                return this.getLegendStyles().normal.fill
            },
            getStackName: function() {
                return this.type === "stackedbar" || this.type === "fullstackedbar" ? this._stackName : null
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file rangeDataCalculator.js */
    (function($, DX) {
        var _math = Math,
            _abs = _math.abs,
            _min = _math.min,
            _each = $.each,
            _isEmptyObject = $.isEmptyObject,
            _isDefined = DevExpress.utils.isDefined,
            FULLSTACKED_SERIES_VALUE_MARGIN_PRIORITY = 15,
            BAR_ZERO_VALUE_MARGIN_PRIORITY = 20,
            SERIES_VALUE_MARGIN_PRIORITY = 20,
            SERIES_LABEL_VALUE_MARGIN = 0.3,
            CATEGORIES_SELECTOR = "categories",
            INTERVAL_SELECTOR = "interval",
            MIN_VALUE_MARGIN_SELECTOR = "minValueMargin",
            MAX_VALUE_MARGIN_SELECTOR = "maxValueMargin",
            MIN = "min",
            MAX = "max",
            MIN_VISIBLE = "minVisible",
            MAX_VISIBLE = "maxVisible";
        DevExpress.viz.core.series.helpers.rangeDataCalculator = function() {
            var _truncateValue = function(data, minField, maxField, value) {
                    var min = data[minField],
                        max = data[maxField];
                    data[minField] = value < min || !_isDefined(min) ? value : data[minField];
                    data[maxField] = value > max || !_isDefined(max) ? value : data[maxField]
                };
            var _processTwoValues = function(series, point, prevPoint, highValueName, lowValueName) {
                    var val = point[highValueName],
                        minVal = point[lowValueName],
                        arg = point.argument,
                        prevVal = prevPoint && prevPoint[highValueName],
                        prevMinVal = prevPoint && prevPoint[lowValueName],
                        prevArg = prevPoint && prevPoint.argument;
                    point.hasValue() && _processRangeValue(series, val, minVal, prevVal, prevMinVal);
                    _processValue(series, "arg", arg, prevArg)
                };
            var _processValue = function(series, type, value, prevValue, calcInterval) {
                    var axis = type === "arg" ? "argument" : "value",
                        data = series._rangeData[type],
                        minInterval = data[INTERVAL_SELECTOR],
                        interval;
                    if (series[axis + "AxisType"] === "discrete") {
                        data[CATEGORIES_SELECTOR] = data[CATEGORIES_SELECTOR] || [];
                        data[CATEGORIES_SELECTOR].push(value)
                    }
                    else {
                        _truncateValue(data, "min", "max", value);
                        if (type === "arg") {
                            interval = _isDefined(prevValue) ? _abs(calcInterval ? calcInterval(value, prevValue) : value - prevValue) : interval;
                            data[INTERVAL_SELECTOR] = _isDefined(interval) && (interval < minInterval || !_isDefined(minInterval)) ? interval : minInterval
                        }
                    }
                };
            var _addToVisibleRange = function(series, value) {
                    var data = series._rangeData.val,
                        isDiscrete = series.valueAxisType === "discrete";
                    if (isDiscrete) {
                        data.visibleCategories = data.visibleCategories || [];
                        data.visibleCategories.push(value)
                    }
                    else {
                        if (value < data.minVisible || !_isDefined(data.minVisible))
                            data.minVisible = value;
                        if (value > data.maxVisible || !_isDefined(data.maxVisible))
                            data.maxVisible = value
                    }
                };
            var _processRangeValue = function(series, val, minVal, prevVal, prevMinVal) {
                    var data = series._rangeData.val,
                        interval,
                        currentInterval = data[INTERVAL_SELECTOR];
                    if (series.valueAxisType === "discrete") {
                        data.categories = data.categories || [];
                        data.categories.push(val, minVal)
                    }
                    else {
                        _truncateValue(data, MIN, MAX, val);
                        _truncateValue(data, MIN, MAX, minVal)
                    }
                };
            var _unique = function(array) {
                    var values = {};
                    return $.map(array, function(item) {
                            var result = !values[item] ? item : null;
                            values[item] = true;
                            return result
                        })
                };
            var _processZoomArgument = function(series, zoomArgs) {
                    var data = series._rangeData.arg,
                        minArg,
                        maxArg;
                    minArg = zoomArgs.minArg < zoomArgs.maxArg ? zoomArgs.minArg : zoomArgs.maxArg;
                    maxArg = zoomArgs.maxArg > zoomArgs.minArg ? zoomArgs.maxArg : zoomArgs.minArg;
                    data.min = minArg < data.min ? minArg : data.min;
                    data.max = maxArg > data.max ? maxArg : data.max;
                    data.minVisible = minArg;
                    data.maxVisible = maxArg
                };
            var _correctZoomValue = function(series, zoomArgs) {
                    var minVal,
                        maxVal;
                    if (_isDefined(zoomArgs.minVal) && _isDefined(zoomArgs.maxVal)) {
                        minVal = zoomArgs.minVal < zoomArgs.maxVal ? zoomArgs.minVal : zoomArgs.maxVal;
                        maxVal = zoomArgs.maxVal > zoomArgs.minVal ? zoomArgs.maxVal : zoomArgs.minVal
                    }
                    if (_isDefined(zoomArgs.minVal)) {
                        series._rangeData.val.min = minVal < series._rangeData.val.min ? minVal : series._rangeData.val.min;
                        series._rangeData.val.minVisible = minVal
                    }
                    if (_isDefined(zoomArgs.maxVal)) {
                        series._rangeData.val.max = maxVal > series._rangeData.val.max ? maxVal : series._rangeData.val.max;
                        series._rangeData.val.maxVisible = maxVal
                    }
                };
            var _processZoomValue = function(series, zoomArgs) {
                    var adjustOnZoom = zoomArgs.adjustOnZoom,
                        points = series._points || [],
                        lastVisibleIndex,
                        prevPointAdded = false;
                    _each(points, function(index, point) {
                        var arg = point.argument,
                            prevPoint = index > 0 ? points[index - 1] : null;
                        if (adjustOnZoom && arg >= series._rangeData.arg.minVisible && arg <= series._rangeData.arg.maxVisible) {
                            if (!prevPointAdded) {
                                prevPoint && prevPoint.hasValue() && _addToVisibleRange(series, prevPoint.value);
                                prevPointAdded = true
                            }
                            point.hasValue() && _addToVisibleRange(series, point.value);
                            lastVisibleIndex = index
                        }
                    });
                    if (_isDefined(lastVisibleIndex) && lastVisibleIndex < points.length - 1 && points[lastVisibleIndex + 1].hasValue())
                        _addToVisibleRange(series, points[lastVisibleIndex + 1].value);
                    _correctZoomValue(series, zoomArgs)
                };
            var _processZoomRangeValue = function(series, zoomArgs, maxValueName, minValueName) {
                    var adjustOnZoom = zoomArgs.adjustOnZoom,
                        points = series._points || [],
                        lastVisibleIndex,
                        prevPointAdded = false;
                    _each(points, function(index, point) {
                        var arg = point.argument,
                            prevPoint = index > 0 ? points[index - 1] : null;
                        if (adjustOnZoom && arg >= series._rangeData.arg.minVisible && arg <= series._rangeData.arg.maxVisible) {
                            if (!prevPointAdded) {
                                if (prevPoint && prevPoint.hasValue()) {
                                    _addToVisibleRange(series, prevPoint[maxValueName]);
                                    _addToVisibleRange(series, prevPoint[minValueName])
                                }
                                prevPointAdded = true
                            }
                            if (point.hasValue()) {
                                _addToVisibleRange(series, point[maxValueName]);
                                _addToVisibleRange(series, point[minValueName])
                            }
                            lastVisibleIndex = index
                        }
                    });
                    if (_isDefined(lastVisibleIndex) && lastVisibleIndex < points.length - 1 && points[lastVisibleIndex + 1].hasValue())
                        _addToVisibleRange(series, points[lastVisibleIndex + 1].value);
                    _correctZoomValue(series, zoomArgs)
                };
            var _processNewInterval = function(series, calcInterval) {
                    var data = series._rangeData,
                        points = series._points || [],
                        isArgumentAxisDiscrete = series.argumentAxisType === "discrete";
                    delete data.arg.interval;
                    _each(points, function(index, point) {
                        var arg = point.argument,
                            prevPoint = index > 0 ? points[index - 1] : null,
                            prevArg = prevPoint && prevPoint.argument;
                        !isArgumentAxisDiscrete && _processValue(series, "arg", arg, prevArg, calcInterval)
                    })
                };
            var _fillRangeData = function(series) {
                    var data = series._rangeData,
                        mainAxis = series._getMainAxisName(),
                        axis = mainAxis === "X" ? "Y" : "X";
                    data.arg.categories && (data.arg.categories = _unique(data.arg.categories));
                    data.val.categories && (data.val.categories = _unique(data.val.categories));
                    data.arg.visibleCategories && (data.arg.visibleCategories = _unique(data.arg.visibleCategories));
                    data.val.visibleCategories && (data.val.visibleCategories = _unique(data.val.visibleCategories));
                    data.arg.axisType = series.argumentAxisType;
                    data.arg.dataType = series.argumentType;
                    data.val.axisType = series.valueAxisType;
                    data.val.dataType = series.valueType;
                    data.val.isValueRange = true
                };
            var _setPadding = function(range, prefix, val, priority) {
                    range[prefix] = val;
                    range[prefix + "Priority"] = priority
                };
            var _setZeroPadding = function(range, val, prefix) {
                    val === 0 && _setPadding(range, prefix, 0, BAR_ZERO_VALUE_MARGIN_PRIORITY)
                };
            var _calculateRangeMinValue = function(series, zoomArgs) {
                    var data = series._rangeData.val,
                        minVisible = data[MIN_VISIBLE],
                        maxVisible = data[MAX_VISIBLE];
                    zoomArgs = zoomArgs || {};
                    if (data) {
                        data.alwaysCorrectMin = true;
                        series._rangeData.arg.keepValueMargins = true;
                        if (series.valueAxisType !== "logarithmic" && series.valueType !== "datetime" && series.getOptions().showZero !== false) {
                            data[MIN_VISIBLE] = minVisible > (zoomArgs.minVal || 0) ? zoomArgs.minVal || 0 : minVisible;
                            data[MAX_VISIBLE] = maxVisible < (zoomArgs.maxVal || 0) ? zoomArgs.maxVal || 0 : maxVisible;
                            data[MIN] = data[MIN] > 0 ? 0 : data[MIN];
                            _setZeroPadding(data, data[MIN], MIN_VALUE_MARGIN_SELECTOR);
                            data[MAX] = data[MAX] < 0 ? 0 : data[MAX];
                            if (data[MAX] === 0 || data[MAX] > 0 && data[MIN] < 0) {
                                data[MIN_VALUE_MARGIN_SELECTOR] = data[MAX_VALUE_MARGIN_SELECTOR];
                                data[MIN_VALUE_MARGIN_SELECTOR + "Priority"] = data[MAX_VALUE_MARGIN_SELECTOR + "Priority"]
                            }
                            _setZeroPadding(data, data[MAX], MAX_VALUE_MARGIN_SELECTOR)
                        }
                    }
                };
            var _processFullStackedRange = function(series) {
                    var data = series._rangeData.val,
                        isRangeEmpty = _isEmptyObject(data);
                    _setPadding(data, "minValueMargin", 0, FULLSTACKED_SERIES_VALUE_MARGIN_PRIORITY);
                    _setPadding(data, "maxValueMargin", 0, FULLSTACKED_SERIES_VALUE_MARGIN_PRIORITY);
                    !isRangeEmpty && (data.min = 0)
                };
            var _processRange = function(series, point, prevPoint) {
                    var val = point.value,
                        arg = point.argument,
                        prevVal = prevPoint && prevPoint.value,
                        prevArg = prevPoint && prevPoint.argument;
                    point.hasValue() && _processValue(series, "val", val, prevVal);
                    _processValue(series, "arg", arg, prevArg)
                };
            var _addLabelPaddings = function(series) {
                    var labelOptions = series.getOptions().label;
                    if (series.areLabelsVisible() && labelOptions && labelOptions.visible && labelOptions.position !== "inside")
                        _setPadding(series._rangeData.val, "maxValueMargin", SERIES_LABEL_VALUE_MARGIN, SERIES_VALUE_MARGIN_PRIORITY)
                };
            var _addRangeSeriesLabelPaddings = function(series) {
                    var data = series._rangeData.val;
                    if (series.areLabelsVisible() && series._options.label.visible && series._options.label.position !== "inside") {
                        _setPadding(data, "maxValueMargin", SERIES_LABEL_VALUE_MARGIN, SERIES_VALUE_MARGIN_PRIORITY);
                        _setPadding(data, "minValueMargin", SERIES_LABEL_VALUE_MARGIN, SERIES_VALUE_MARGIN_PRIORITY)
                    }
                };
            var _calculateRangeData = function(series, zoomArgs, calcIntervalFunction) {
                    var valueData = series._rangeData.val;
                    if (series.argumentAxisType !== "discrete" && zoomArgs && _isDefined(zoomArgs.minArg) && _isDefined(zoomArgs.maxArg)) {
                        valueData[MIN_VISIBLE] = zoomArgs.minVal;
                        valueData[MAX_VISIBLE] = zoomArgs.maxVal;
                        _processZoomArgument(series, zoomArgs);
                        _processZoomValue(series, zoomArgs)
                    }
                    else if (!zoomArgs && calcIntervalFunction)
                        _processNewInterval(series, calcIntervalFunction);
                    _fillRangeData(series)
                };
            var _calculateTwoValuesRangeData = function(series, zoomArgs, calcIntervalFunction, maxValueName, minValueName) {
                    var valueData = series._rangeData.val;
                    if (series.argumentAxisType !== "discrete" && zoomArgs && _isDefined(zoomArgs.minArg) && _isDefined(zoomArgs.maxArg)) {
                        valueData[MIN_VISIBLE] = zoomArgs.minVal;
                        valueData[MAX_VISIBLE] = zoomArgs.maxVal;
                        _processZoomArgument(series, zoomArgs);
                        _processZoomRangeValue(series, zoomArgs, maxValueName, minValueName)
                    }
                    else if (!zoomArgs && calcIntervalFunction)
                        _processNewInterval(series);
                    _fillRangeData(series)
                };
            return {
                    processRange: _processRange,
                    calculateRangeData: _calculateRangeData,
                    calculateTwoValuesRangeData: _calculateTwoValuesRangeData,
                    addLabelPaddings: _addLabelPaddings,
                    addRangeSeriesLabelPaddings: _addRangeSeriesLabelPaddings,
                    processFullStackedRange: _processFullStackedRange,
                    calculateRangeMinValue: _calculateRangeMinValue,
                    processTwoValues: _processTwoValues
                }
        }
    })(jQuery, DevExpress);
    /*! Module viz-core, file scatterSeries.js */
    (function($, DX) {
        var viz = DX.viz,
            series = viz.core.series,
            rangeCalculator = series.helpers.rangeDataCalculator(),
            _each = $.each,
            _extend = $.extend,
            _map = $.map,
            _noop = $.noop,
            _isDefined = DX.utils.isDefined,
            _floor = Math.floor,
            DEFAULT_SYMBOL_POINT_SIZE = 2,
            DEFAULT_TRACKER_WIDTH = 20,
            DEFAULT_DURATION = 400;
        series.mixins.chart.scatter = {
            _defaultDuration: DEFAULT_DURATION,
            _defaultTrackerWidth: DEFAULT_TRACKER_WIDTH,
            _applyStyle: _noop,
            _adjustLabels: _noop,
            _updateOptions: _noop,
            _parseStyle: _noop,
            _prepareSegment: _noop,
            _drawSegment: _noop,
            _generateDefaultSegments: _noop,
            _prepareSeriesToDrawing: function() {
                this._deleteOldAnimationMethods();
                this._firstDrawing && this._clearSeries();
                this._disposePoints(this._oldPoints);
                this._oldPoints = null
            },
            _applyTrackersClippings: function() {
                this._markerTrackerGroup.applySettings({clipId: this._forceClipping ? this._paneClipRectID : null})
            },
            updateTeamplateFieldNames: function() {
                var that = this,
                    options = that._options;
                options.valueField = that.getValueFields()[0] + that.name;
                options.tagField = that.getTagField() + that.name
            },
            _applyElementsClipRect: function(settings) {
                settings.clipId = this._paneClipRectID
            },
            _applyMarkerClipRect: function(settings) {
                settings.clipId = this._forceClipping ? this._paneClipRectID : null
            },
            _createGroup: function(groupName, parent, target, settings) {
                var group = parent[groupName];
                if (!group)
                    parent[groupName] = group = this._renderer.createGroup(settings);
                else
                    group.applySettings(settings);
                group.append(target)
            },
            _applyClearingSettings: function(settings) {
                settings.opacity = null;
                settings.scale = null;
                if (this._options.rotated)
                    settings.translateX = null;
                else
                    settings.translateY = null
            },
            _createMarkerGroup: function() {
                var settings = this._getPointOptions().styles.normal;
                settings["class"] = "dxc-markers";
                settings.opacity = 1;
                this._applyMarkerClipRect(settings);
                this._createGroup("_markersGroup", this, this._group, settings)
            },
            _createLabelGroup: function() {
                var settings = {
                        "class": "dxc-labels",
                        visibility: this.getLabelVisibility() ? "visible" : "hidden"
                    };
                this._applyElementsClipRect(settings);
                this._applyClearingSettings(settings);
                this._createGroup("_labelsGroup", this, this._options.labelsGroup, settings)
            },
            _createGroups: function(animationEnabled) {
                this._createMarkerGroup();
                this._createLabelGroup();
                animationEnabled && this._labelsGroup && this._labelsGroup.applySettings({opacity: 0.001})
            },
            _getCreatingPointOptions: function() {
                if (!this._predefinedPointOptions) {
                    var defaultPointOptions = this._getPointOptions(),
                        r = defaultPointOptions.styles && defaultPointOptions.styles.normal && defaultPointOptions.styles.normal.r,
                        strokeWidth = defaultPointOptions.styles && defaultPointOptions.styles.normal && defaultPointOptions.styles.normal.strokeWidth,
                        creatingPointOptions = _extend(true, {}, defaultPointOptions);
                    creatingPointOptions.styles = creatingPointOptions.styles || {};
                    creatingPointOptions.styles.normal = {
                        r: r,
                        strokeWidth: strokeWidth
                    };
                    this._predefinedPointOptions = creatingPointOptions
                }
                return this._predefinedPointOptions
            },
            _getSpecialColor: function(mainSeriesColor) {
                return mainSeriesColor
            },
            _getPointOptions: function() {
                return this._pointOptions || (this._pointOptions = this._parsePointOptions(this._preparePointOptions(), this._options.label))
            },
            _preparePointOptions: function(customOptions) {
                return customOptions ? _extend(true, {}, this._options.point, customOptions) : this._options.point
            },
            _parsePointStyle: function(style, defaultColor, defaultBorderColor) {
                var border = style.border || {};
                return {
                        fill: style.color || defaultColor,
                        stroke: border.color || defaultBorderColor,
                        strokeWidth: border.visible ? border.width : 0,
                        r: style.size / 2 + (border.visible && style.size !== 0 ? ~~(border.width / 2) || 0 : 0)
                    }
            },
            _createPointStyles: function(pointOptions) {
                var mainPointColor = pointOptions.color || this._options.mainSeriesColor,
                    containerColor = this._options.containerBackgroundColor,
                    normalStyle = this._parsePointStyle(pointOptions, mainPointColor, mainPointColor);
                normalStyle.visibility = pointOptions.visible ? "visible" : "hidden";
                return {
                        normal: normalStyle,
                        hover: this._parsePointStyle(pointOptions.hoverStyle, containerColor, mainPointColor),
                        selection: this._parsePointStyle(pointOptions.selectionStyle, containerColor, mainPointColor)
                    }
            },
            _checkData: function(data) {
                return _isDefined(data.argument) && data.value !== undefined
            },
            _processRange: function(point, prevPoint) {
                rangeCalculator.processRange(this, point, prevPoint)
            },
            _getRangeData: function(zoomArgs, calcIntervalFunction) {
                rangeCalculator.calculateRangeData(this, zoomArgs, calcIntervalFunction);
                rangeCalculator.addLabelPaddings(this);
                return this._rangeData
            },
            _getPointData: function(data, options) {
                var argumentField = options.argumentField || "arg",
                    valueField = options.valueField || "val",
                    tagField = options.tagField || "tag";
                return {
                        value: data[valueField],
                        argument: data[argumentField],
                        tag: data[tagField]
                    }
            },
            _drawPoint: function(point, markersGroup, labelsGroup, animationEnabled, firstDrawing) {
                if (point.isInVisibleArea()) {
                    point.clearVisibility();
                    point.draw(this._renderer, markersGroup, labelsGroup, animationEnabled, firstDrawing);
                    this._drawedPoints.push(point)
                }
                else
                    point.setInvisibility()
            },
            _clearingAnimation: function(translators, drawComplete) {
                var that = this,
                    params = {opacity: 0.001},
                    options = {
                        duration: that._defaultDuration,
                        partitionDuration: 0.5
                    };
                that._labelsGroup && that._labelsGroup.animate(params, options, function() {
                    that._markersGroup && that._markersGroup.animate(params, options, drawComplete)
                })
            },
            _animate: function(complete) {
                var that = this,
                    lastPointIndex = that._drawedPoints.length - 1,
                    labelAnimFunc = function() {
                        that._labelsGroup && that._labelsGroup.animate({opacity: 1}, {duration: that._defaultDuration})
                    },
                    settings;
                _each(that._drawedPoints || [], function(i, p) {
                    settings = {translate: {
                            x: p.x,
                            y: p.y
                        }};
                    p.animate(i === lastPointIndex ? labelAnimFunc : undefined, settings)
                })
            },
            _getPointSize: function() {
                return this._options.point.visible ? this._options.point.size : DEFAULT_SYMBOL_POINT_SIZE
            },
            _calcMedianValue: function(fusionPoints, valueField) {
                var result,
                    allValue;
                allValue = _map(fusionPoints, function(point) {
                    return point[valueField]
                });
                allValue.sort(function(a, b) {
                    return a - b
                });
                result = allValue[_floor(allValue.length / 2)];
                return _isDefined(result) ? result : null
            },
            _fusionPoints: function(fusionPoints, tick, index) {
                var result = this._calcMedianValue(fusionPoints, "value");
                return {
                        value: result,
                        argument: tick,
                        tag: null,
                        index: index,
                        seriesName: this.name
                    }
            },
            _endUpdateData: function() {
                delete this._predefinedPointOptions
            },
            getArgumentField: function() {
                return this._options.argumentField || "arg"
            },
            getValueFields: function() {
                return [this._options.valueField || "val"]
            },
            _beginUpdateData: $.noop
        }
    })(jQuery, DevExpress);
    /*! Module viz-core, file lineSeries.js */
    (function($, DX) {
        var viz = DX.viz,
            series = viz.core.series.mixins.chart,
            utils = DX.utils,
            scatterSeries = series.scatter,
            _extend = $.extend,
            _map = $.map,
            _each = $.each;
        series.line = _extend({}, scatterSeries, {
            _createElementsGroup: function(elementsStyle) {
                var settings = _extend({"class": "dxc-elements"}, elementsStyle);
                this._applyElementsClipRect(settings);
                this._createGroup("_elementsGroup", this, this._group, settings)
            },
            _createBordersGroup: function(borderStyle) {
                var settings = _extend({"class": "dxc-borders"}, borderStyle);
                this._applyElementsClipRect(settings);
                this._createGroup("_bordersGroup", this, this._group, settings)
            },
            _applyTrackersClippings: function() {
                scatterSeries._applyTrackersClippings.call(this);
                this._trackersGroup.applySettings({clipId: this._paneClipRectID})
            },
            _createGroups: function(animationEnabled, style) {
                var style = style || this._styles.normal;
                this._createElementsGroup(style.elements);
                this._areBordersVisible() && this._createBordersGroup(style.border);
                scatterSeries._createGroups.call(this, animationEnabled, {});
                animationEnabled && this._markersGroup && this._markersGroup.applySettings({opacity: 0.001})
            },
            _areBordersVisible: function() {
                return false
            },
            _getDefaultSegment: function(segment) {
                return {line: _map(segment.line || [], function(pt) {
                            return pt.getDefaultCoords()
                        })}
            },
            _prepareSegment: function(points) {
                return {line: points}
            },
            _parseLineOptions: function(options, defaultColor) {
                return {
                        stroke: options.color || defaultColor,
                        strokeWidth: options.width,
                        dashStyle: options.dashStyle || 'solid'
                    }
            },
            _parseStyle: function(options, defaultColor) {
                return {elements: this._parseLineOptions(options, defaultColor)}
            },
            _applyStyle: function(style) {
                this._elementsGroup && this._elementsGroup.applySettings(style.elements);
                _each(this._graphics || [], function(_, graphic) {
                    graphic.line && graphic.line.applySettings({strokeWidth: style.elements.strokeWidth})
                })
            },
            _drawElement: function(segment, group) {
                return {line: this._createMainElement(segment.line, {strokeWidth: this._styles.normal.elements.strokeWidth}).append(group)}
            },
            _removeElement: function(element) {
                element.line.remove()
            },
            _generateDefaultSegments: function() {
                var that = this,
                    segments = [];
                _each(that._segments || [], function(_, segment) {
                    segments.push(that._getDefaultSegment(segment))
                });
                return segments
            },
            _updateElement: function(element, segment, animate, animateParams, complete) {
                var params = {points: segment.line};
                animate ? element.line.animate(params, animateParams, complete) : element.line.applySettings(params)
            },
            _clearingAnimation: function(translator, drawComplete) {
                var that = this,
                    lastIndex = that._graphics.length - 1,
                    settings = {opacity: 0.001},
                    options = {
                        duration: that._defaultDuration,
                        partitionDuration: 0.5
                    };
                that._labelsGroup && that._labelsGroup.animate(settings, options, function() {
                    that._markersGroup && that._markersGroup.animate(settings, options, function() {
                        _each(that._defaultSegments || [], function(i, segment) {
                            that._oldUpdateElement(that._graphics[i], segment, true, {partitionDuration: 0.5}, i === lastIndex ? drawComplete : undefined)
                        })
                    })
                })
            },
            _animate: function() {
                var that = this,
                    lastIndex = that._graphics.length - 1;
                _each(that._graphics || [], function(i, elem) {
                    that._updateElement(elem, that._segments[i], true, {complete: i === lastIndex ? function() {
                            that._labelsGroup && that._labelsGroup.animate({opacity: 1}, {duration: that._defaultDuration});
                            that._markersGroup && that._markersGroup.animate({opacity: 1}, {duration: that._defaultDuration})
                        } : undefined})
                })
            },
            _drawPoint: function(point, group, labelsGroup) {
                scatterSeries._drawPoint.call(this, point, group, labelsGroup)
            },
            _createMainElement: function(points, settings) {
                return this._renderer.createPath(points, settings)
            },
            _drawSegment: function(points, animationEnabled, segmentCount) {
                var segment = this._prepareSegment(points, this._options.rotated);
                this._segments.push(segment);
                if (!this._graphics[segmentCount])
                    this._graphics[segmentCount] = this._drawElement(animationEnabled ? this._getDefaultSegment(segment) : segment, this._elementsGroup);
                else if (!animationEnabled)
                    this._updateElement(this._graphics[segmentCount], segment)
            },
            _getTrackerSettings: function() {
                return {
                        strokeWidth: this._styles.normal.elements.strokeWidth > this._defaultTrackerWidth ? this._styles.normal.elements.strokeWidth : this._defaultTrackerWidth,
                        fill: "none"
                    }
            },
            _getMainPointsFromSegment: function(segment) {
                return segment.line
            },
            _drawTrackerElement: function(segment) {
                return this._createMainElement(this._getMainPointsFromSegment(segment), this._getTrackerSettings(segment))
            },
            _updateTrackerElement: function(segment, element) {
                var settings = this._getTrackerSettings(segment);
                settings.points = this._getMainPointsFromSegment(segment);
                element.applySettings(settings)
            }
        });
        series.stepline = _extend({}, series.line, {
            _calculateStepLinePoints: function(points) {
                var segment = [];
                _each(points, function(i, pt) {
                    var stepY;
                    if (!i) {
                        segment.push(pt);
                        return
                    }
                    stepY = segment[segment.length - 1].y;
                    if (stepY !== pt.y) {
                        var point = utils.clone(pt);
                        point.y = stepY;
                        segment.push(point)
                    }
                    segment.push(pt)
                });
                return segment
            },
            _prepareSegment: function(points) {
                return series.line._prepareSegment(this._calculateStepLinePoints(points))
            }
        });
        series.stackedline = _extend({}, series.line, {});
        series.fullstackedline = _extend({}, series.line, {_getRangeData: function(zoomArgs, calcIntervalFunction) {
                var rangeCalculator = viz.core.series.helpers.rangeDataCalculator();
                rangeCalculator.calculateRangeData(this, zoomArgs, calcIntervalFunction);
                rangeCalculator.addLabelPaddings(this);
                rangeCalculator.processFullStackedRange(this);
                return this._rangeData
            }});
        series.spline = _extend({}, series.line, {
            _calculateBezierPoints: function(src, rotated) {
                var bezierPoints = [],
                    pointsCopy = src;
                var checkExtr = function(otherPointCoord, pointCoord, controlCoord) {
                        return otherPointCoord > pointCoord && controlCoord > otherPointCoord || otherPointCoord < pointCoord && controlCoord < otherPointCoord ? otherPointCoord : controlCoord
                    };
                var clonePoint = function(point, newX, newY) {
                        var p = utils.clone(point);
                        p.x = newX;
                        p.y = newY;
                        return p
                    };
                if (pointsCopy.length !== 1)
                    _each(pointsCopy, function(i, curPoint) {
                        var leftControlX,
                            leftControlY,
                            rightControlX,
                            rightControlY,
                            prevPoint,
                            nextPoint,
                            xCur,
                            yCur,
                            x1,
                            x2,
                            y1,
                            y2,
                            delta,
                            lambda = 0.5,
                            curIsExtremum,
                            leftPoint,
                            rightPoint,
                            a,
                            b,
                            c,
                            xc,
                            yc,
                            shift;
                        if (!i) {
                            bezierPoints.push(curPoint);
                            bezierPoints.push(curPoint);
                            return
                        }
                        prevPoint = pointsCopy[i - 1];
                        if (i < pointsCopy.length - 1) {
                            nextPoint = pointsCopy[i + 1];
                            xCur = curPoint.x;
                            yCur = curPoint.y;
                            x1 = prevPoint.x;
                            x2 = nextPoint.x;
                            y1 = prevPoint.y;
                            y2 = nextPoint.y;
                            curIsExtremum = !!(!rotated && (yCur <= prevPoint.y && yCur <= nextPoint.y || yCur >= prevPoint.y && yCur >= nextPoint.y) || rotated && (xCur <= prevPoint.x && xCur <= nextPoint.x || xCur >= prevPoint.x && xCur >= nextPoint.x));
                            if (curIsExtremum)
                                if (!rotated) {
                                    rightControlY = leftControlY = yCur;
                                    rightControlX = (xCur + nextPoint.x) / 2;
                                    leftControlX = (xCur + prevPoint.x) / 2
                                }
                                else {
                                    rightControlX = leftControlX = xCur;
                                    rightControlY = (yCur + nextPoint.y) / 2;
                                    leftControlY = (yCur + prevPoint.y) / 2
                                }
                            else {
                                a = y2 - y1;
                                b = x1 - x2;
                                c = y1 * x2 - x1 * y2;
                                if (!rotated) {
                                    xc = xCur;
                                    yc = -1 * (a * xc + c) / b;
                                    shift = yc - yCur || 0;
                                    y1 -= shift;
                                    y2 -= shift
                                }
                                else {
                                    yc = yCur;
                                    xc = -1 * (b * yc + c) / a;
                                    shift = xc - xCur || 0;
                                    x1 -= shift;
                                    x2 -= shift
                                }
                                rightControlX = (xCur + lambda * x2) / (1 + lambda);
                                rightControlY = (yCur + lambda * y2) / (1 + lambda);
                                leftControlX = (xCur + lambda * x1) / (1 + lambda);
                                leftControlY = (yCur + lambda * y1) / (1 + lambda)
                            }
                            if (!rotated) {
                                leftControlY = checkExtr(prevPoint.y, yCur, leftControlY);
                                rightControlY = checkExtr(nextPoint.y, yCur, rightControlY)
                            }
                            else {
                                leftControlX = checkExtr(prevPoint.x, xCur, leftControlX);
                                rightControlX = checkExtr(nextPoint.x, xCur, rightControlX)
                            }
                            leftPoint = clonePoint(curPoint, leftControlX, leftControlY);
                            rightPoint = clonePoint(curPoint, rightControlX, rightControlY);
                            bezierPoints.push(leftPoint, curPoint, rightPoint)
                        }
                        else {
                            bezierPoints.push(curPoint, curPoint);
                            return
                        }
                    });
                else
                    bezierPoints.push(pointsCopy[0]);
                return bezierPoints
            },
            _prepareSegment: function(points, rotated) {
                return series.line._prepareSegment(this._calculateBezierPoints(points, rotated))
            },
            _createMainElement: function(points, settings) {
                return this._renderer.createBezierPath(points, settings)
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file areaSeries.js */
    (function($, DX) {
        var viz = DX.viz,
            utils = DX.utils,
            series = viz.core.series.mixins.chart,
            lineSeries = series.line,
            _map = $.map,
            _extend = $.extend,
            HOVER_COLOR_HIGHLIGHTING = 20;
        series.area = _extend({}, lineSeries, {
            _createBorderElement: lineSeries._createMainElement,
            _processSinglePointsAreaSegment: function(points, rotated) {
                if (points.length == 1) {
                    var p = points[0],
                        p1 = utils.clone(p),
                        coord = rotated ? "y" : "x";
                    p1[coord] += 1;
                    return [p, p1]
                }
                return points
            },
            _prepareSegment: function(points, rotated) {
                var processedPoints = this._processSinglePointsAreaSegment(points, rotated);
                return {
                        line: processedPoints,
                        area: _map(processedPoints, function(pt) {
                            return pt.getCoords()
                        }).concat(_map(processedPoints.slice().reverse(), function(pt) {
                            return pt.getCoords(true)
                        })),
                        singlePointSegment: processedPoints !== points
                    }
            },
            _getSpecialColor: function(color) {
                return this._options._IE8 ? new DX.Color(color).highlight(HOVER_COLOR_HIGHLIGHTING) : color
            },
            _getRangeData: function(zoomArgs, calcIntervalFunction) {
                var rangeCalculator = viz.core.series.helpers.rangeDataCalculator();
                rangeCalculator.calculateRangeData(this, zoomArgs, calcIntervalFunction);
                rangeCalculator.addLabelPaddings(this);
                rangeCalculator.calculateRangeMinValue(this, zoomArgs);
                return this._rangeData
            },
            _getDefaultSegment: function(segment) {
                var defaultSegment = lineSeries._getDefaultSegment(segment);
                defaultSegment.area = defaultSegment.line.concat(defaultSegment.line.slice().reverse());
                return defaultSegment
            },
            _updateElement: function(element, segment, animate, animateParams, complete) {
                var lineParams = {points: segment.line},
                    areaParams = {points: segment.area},
                    borderElement = element.line;
                if (animate) {
                    borderElement && borderElement.animate(lineParams, animateParams);
                    element.area.animate(areaParams, animateParams, complete)
                }
                else {
                    borderElement && borderElement.applySettings(lineParams);
                    element.area.applySettings(areaParams)
                }
            },
            _removeElement: function(element) {
                element.line && element.line.remove();
                element.area.remove()
            },
            _drawElement: function(segment, group) {
                return {
                        line: this._bordersGroup && this._createBorderElement(segment.line, {strokeWidth: this._styles.normal.border.strokeWidth}).append(this._bordersGroup),
                        area: this._createMainElement(segment.area).append(this._elementsGroup)
                    }
            },
            _applyStyle: function(style) {
                this._elementsGroup && this._elementsGroup.applySettings(style.elements);
                this._bordersGroup && this._bordersGroup.applySettings(style.border);
                $.each(this._graphics || [], function(_, graphic) {
                    graphic.line && graphic.line.applySettings({strokeWidth: style.border.strokeWidth})
                })
            },
            _createPattern: function(color, hatching) {
                if (hatching) {
                    var pattern = this._renderer.createPattern(color, hatching);
                    this._patterns.push(pattern);
                    return pattern.id
                }
                return color
            },
            _parseStyle: function(options, defaultColor, defaultBorderColor) {
                var borderOptions = options.border || {},
                    borderStyle = lineSeries._parseLineOptions(borderOptions, defaultBorderColor);
                borderStyle.strokeWidth = borderOptions.visible ? borderStyle.strokeWidth : 0;
                return {
                        border: borderStyle,
                        elements: {
                            stroke: "none",
                            fill: this._createPattern(options.color || defaultColor, options.hatching),
                            opacity: options.opacity
                        }
                    }
            },
            _areBordersVisible: function() {
                var options = this._options;
                return options.border.visible || options.hoverStyle.border.visible || options.selectionStyle.border.visible
            },
            _createMainElement: function(points, settings) {
                return this._renderer.createArea(points, settings)
            },
            _getTrackerSettings: function(segment) {
                return {strokeWidth: segment.singlePointSegment ? this._defaultTrackerWidth : 0}
            },
            _getMainPointsFromSegment: function(segment) {
                return segment.area
            }
        });
        series.steparea = _extend({}, series.area, {_prepareSegment: function(points, rotated) {
                points = series.area._processSinglePointsAreaSegment(points, rotated);
                return series.area._prepareSegment.call(this, series.stepline._calculateStepLinePoints(points))
            }});
        series.splinearea = _extend({}, series.area, {
            _areaPointsToSplineAreaPoints: function(areaPoints) {
                var lastFwPoint = areaPoints[areaPoints.length / 2 - 1],
                    firstBwPoint = areaPoints[areaPoints.length / 2];
                areaPoints.splice(areaPoints.length / 2, 0, {
                    x: lastFwPoint.x,
                    y: lastFwPoint.y
                }, {
                    x: firstBwPoint.x,
                    y: firstBwPoint.y
                });
                if (lastFwPoint.defaultCoords)
                    areaPoints[areaPoints.length / 2].defaultCoords = true;
                if (firstBwPoint.defaultCoords)
                    areaPoints[areaPoints.length / 2 - 1].defaultCoords = true
            },
            _prepareSegment: function(points, rotated) {
                var areaSeries = series.area,
                    processedPoints = areaSeries._processSinglePointsAreaSegment(points, rotated),
                    areaSegment = areaSeries._prepareSegment.call(this, series.spline._calculateBezierPoints(processedPoints, rotated));
                this._areaPointsToSplineAreaPoints(areaSegment.area);
                areaSegment.singlePointSegment = processedPoints !== points;
                return areaSegment
            },
            _getDefaultSegment: function(segment) {
                var areaDefaultSegment = series.area._getDefaultSegment(segment);
                this._areaPointsToSplineAreaPoints(areaDefaultSegment.area);
                return areaDefaultSegment
            },
            _createMainElement: function(points, settings) {
                return this._renderer.createBezierArea(points, settings)
            },
            _createBorderElement: series.spline._createMainElement
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file barSeries.js */
    (function($, DX) {
        var viz = DX.viz,
            series = viz.core.series.mixins.chart,
            scatterSeries = series.scatter,
            areaSeries = series.area,
            _extend = $.extend,
            _each = $.each,
            DEFAULT_BAR_POINT_SIZE = 3;
        series.bar = _extend({}, scatterSeries, {
            _getSpecialColor: areaSeries._getSpecialColor,
            _createPattern: areaSeries._createPattern,
            _updateOptions: function(options) {
                this._stackName = "axis_" + (options.axis || "default") + "_stack_" + (options.stack || "default")
            },
            _getRangeData: function(zoomArgs, calcIntervalFunction) {
                var rangeData = areaSeries._getRangeData.apply(this, arguments);
                rangeData.arg.stick = false;
                return rangeData
            },
            _parsePointStyle: function(style, defaultColor, defaultBorderColor) {
                var color = this._createPattern(style.color || defaultColor, style.hatching),
                    base = scatterSeries._parsePointStyle.call(this, style, color, defaultBorderColor);
                base.fill = color;
                delete base.r;
                return base
            },
            _applyMarkerClipRect: function(settings) {
                settings.clipId = null
            },
            _applyTrackersClippings: function() {
                this._markerTrackerGroup.applySettings({clipId: this._paneClipRectID})
            },
            _clearingAnimation: function(translators, drawComplete) {
                var that = this,
                    settings;
                if (that._options.rotated)
                    settings = {
                        scale: {
                            x: 0.001,
                            y: 1
                        },
                        translate: {x: translators.x.translate("canvas_position_default")}
                    };
                else
                    settings = {
                        scale: {
                            x: 1,
                            y: 0.001
                        },
                        translate: {y: translators.y.translate("canvas_position_default")}
                    };
                that._labelsGroup && that._labelsGroup.animate({opacity: 0.001}, {
                    duration: that._defaultDuration,
                    partitionDuration: 0.5
                }, function() {
                    that._markersGroup.animate(settings, {partitionDuration: 0.5}, function() {
                        that._markersGroup.applySettings({
                            scale: null,
                            translateX: 0,
                            translateY: 0
                        });
                        drawComplete()
                    })
                })
            },
            _createGroups: function(animationEnabled, style, firstDrawing) {
                var settings = {};
                scatterSeries._createGroups.apply(this, arguments);
                if (animationEnabled && firstDrawing)
                    if (!this._options.rotated)
                        settings = {
                            scale: {
                                x: 1,
                                y: 0.001
                            },
                            translateY: this.translators.y.translate("canvas_position_default")
                        };
                    else
                        settings = {
                            scale: {
                                x: 0.001,
                                y: 1
                            },
                            translateX: this.translators.x.translate("canvas_position_default")
                        };
                else if (!animationEnabled)
                    settings = {
                        scale: {
                            x: 1,
                            y: 1
                        },
                        translateX: 0,
                        translateY: 0
                    };
                this._markersGroup.applySettings(settings)
            },
            _drawPoint: function(point, markersGroup, labelsGroup, animationEnabled, firstDrawing) {
                scatterSeries._drawPoint.call(this, point, markersGroup, labelsGroup, animationEnabled && !firstDrawing)
            },
            _getMainColor: function() {
                return this._options.mainSeriesColor
            },
            _createPointStyles: function(pointOptions) {
                var mainColor = pointOptions.color || this._getMainColor(),
                    specialMainColor = this._getSpecialColor(mainColor);
                return {
                        normal: this._parsePointStyle(pointOptions, mainColor, mainColor),
                        hover: this._parsePointStyle(pointOptions.hoverStyle || {}, specialMainColor, mainColor),
                        selection: this._parsePointStyle(pointOptions.selectionStyle || {}, specialMainColor, mainColor)
                    }
            },
            _preparePointOptions: function(customOptions) {
                return customOptions ? _extend(true, {}, this._options, customOptions) : this._options
            },
            _animate: function(firstDrawing) {
                var that = this,
                    labelAnimFunc = function() {
                        that._labelsGroup && that._labelsGroup.animate({opacity: 1}, {duration: that._defaultDuration})
                    },
                    lastPointIndex;
                that._markersGroup.animate({
                    scale: {
                        x: 1,
                        y: 1
                    },
                    translate: {
                        y: 0,
                        x: 0
                    }
                }, undefined, labelAnimFunc);
                if (!firstDrawing) {
                    lastPointIndex = that._drawedPoints.length - 1;
                    _each(this._drawedPoints || [], function(i, point) {
                        point.animate(i === lastPointIndex ? labelAnimFunc : undefined, {
                            x: point.x,
                            y: point.y,
                            height: point.height,
                            width: point.width
                        })
                    })
                }
            },
            _getPointSize: function() {
                return DEFAULT_BAR_POINT_SIZE
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file rangeSeries.js */
    (function($, DX) {
        var viz = DX.viz,
            series = viz.core.series.mixins.chart,
            _extend = $.extend,
            _isDefined = DX.utils.isDefined,
            _map = $.map,
            rangeCalculator = viz.core.series.helpers.rangeDataCalculator(),
            areaSeries = series.area;
        var baseRangeSeries = {
                _checkData: function(data) {
                    return _isDefined(data.argument) && data.value !== undefined && data.minValue !== undefined
                },
                updateTeamplateFieldNames: function() {
                    var that = this,
                        options = that._options,
                        valueFields = that.getValueFields(),
                        name = that.name;
                    options.rangeValue1Field = valueFields[0] + name;
                    options.rangeValue2Field = valueFields[1] + name;
                    options.tagField = that.getTagField() + name
                },
                _processRange: function(point, prevPoint) {
                    rangeCalculator.processTwoValues(this, point, prevPoint, "value", "minValue")
                },
                _getRangeData: function(zoomArgs, calcIntervalFunction) {
                    rangeCalculator.calculateTwoValuesRangeData(this, zoomArgs, calcIntervalFunction, "value", "minValue");
                    rangeCalculator.addRangeSeriesLabelPaddings(this);
                    return this._rangeData
                },
                _getPointData: function(data, options) {
                    var argumentField = options.argumentField || "arg",
                        tagField = options.tagField || "tag",
                        rangeValue1Field = options.rangeValue1Field || "val1",
                        rangeValue2Field = options.rangeValue2Field || "val2";
                    return {
                            tag: data[tagField],
                            minValue: data[rangeValue1Field],
                            value: data[rangeValue2Field],
                            argument: data[argumentField]
                        }
                },
                _fusionPoints: function(fusionPoints, tick, index) {
                    var scatterSeries = series.scatter,
                        value = scatterSeries._calcMedianValue.call(this, fusionPoints, "value"),
                        minValue = scatterSeries._calcMedianValue.call(this, fusionPoints, "minValue"),
                        pointData;
                    if (value === null || minValue === null)
                        value = minValue = null;
                    pointData = {
                        minValue: minValue,
                        value: value,
                        argument: tick,
                        tag: null
                    };
                    return pointData
                },
                getValueFields: function() {
                    return [this._options.rangeValue1Field || "val1", this._options.rangeValue2Field || "val2"]
                }
            };
        series.rangebar = _extend({}, series.bar, baseRangeSeries);
        series.rangearea = _extend({}, areaSeries, {
            _drawPoint: function(point, markersGroup, labelsGroup, animationEnabled) {
                if (point.isInVisibleArea()) {
                    point.clearVisibility();
                    point.draw(this._renderer, markersGroup, labelsGroup, animationEnabled);
                    this._drawedPoints.push(point);
                    if (!point.visibleTopMarker)
                        point.hideMarker("top");
                    if (!point.visibleBottomMarker)
                        point.hideMarker("bottom")
                }
                else
                    point.setInvisibility()
            },
            _prepareSegment: function(points, rotated) {
                var processedPoints = this._processSinglePointsAreaSegment(points, rotated),
                    processedMinPointsCoords = $.map(processedPoints, function(pt) {
                        return pt.getCoords(true)
                    });
                return {
                        line: processedPoints,
                        bottomLine: processedMinPointsCoords,
                        area: $.map(processedPoints, function(pt) {
                            return pt.getCoords()
                        }).concat(processedMinPointsCoords.slice().reverse()),
                        singlePointSegment: processedPoints !== points
                    }
            },
            _getDefaultSegment: function(segment) {
                var defaultSegment = areaSeries._getDefaultSegment.call(this, segment);
                defaultSegment.bottomLine = defaultSegment.line;
                return defaultSegment
            },
            _removeElement: function(element) {
                areaSeries._removeElement.call(this, element);
                element.bottomLine && element.bottomLine.remove()
            },
            _drawElement: function(segment, group) {
                var drawnElement = areaSeries._drawElement.call(this, segment, group);
                drawnElement.bottomLine = this._bordersGroup && this._createBorderElement(segment.bottomLine, {strokeWidth: this._styles.normal.border.strokeWidth}).append(this._bordersGroup);
                return drawnElement
            },
            _applyStyle: function(style) {
                this._elementsGroup && this._elementsGroup.applySettings(style.elements);
                this._bordersGroup && this._bordersGroup.applySettings(style.border);
                $.each(this._graphics || [], function(_, graphic) {
                    graphic.line && graphic.line.applySettings({strokeWidth: style.border.strokeWidth});
                    graphic.bottomLine && graphic.bottomLine.applySettings({strokeWidth: style.border.strokeWidth})
                })
            },
            _updateElement: function(element, segment, animate, animateParams, complete) {
                areaSeries._updateElement.call(this, element, segment, animate, animateParams, complete);
                var bottomLineParams = {points: segment.bottomLine},
                    bottomBorderElement = element.bottomLine;
                if (bottomBorderElement)
                    animate ? bottomBorderElement.animate(bottomLineParams, animateParams) : bottomBorderElement.applySettings(bottomLineParams)
            }
        }, baseRangeSeries)
    })(jQuery, DevExpress);
    /*! Module viz-core, file bubbleSeries.js */
    (function($, DX) {
        var viz = DX.viz,
            series = viz.core.series.mixins.chart,
            scatterSeries = series.scatter,
            barSeries = series.bar,
            _isDefined = DX.utils.isDefined,
            _extend = $.extend,
            _each = $.each;
        series.bubble = _extend({}, scatterSeries, {
            _applyTrackersClippings: barSeries._applyTrackersClippings,
            _getMainColor: barSeries._getMainColor,
            _createPointStyles: barSeries._createPointStyles,
            _createPattern: barSeries._createPattern,
            _preparePointOptions: barSeries._preparePointOptions,
            _getSpecialColor: barSeries._getSpecialColor,
            _applyMarkerClipRect: series.line._applyElementsClipRect,
            _parsePointStyle: function(style, defaultColor, defaultBorderColor) {
                var color = this._createPattern(style.color || defaultColor, style.hatching),
                    base = scatterSeries._parsePointStyle.call(this, style, color, defaultBorderColor);
                base.fill = color;
                base.opacity = style.opacity;
                delete base.r;
                return base
            },
            _createMarkerGroup: function() {
                var markersSettings = this._getPointOptions().styles.normal,
                    groupSettings;
                markersSettings["class"] = "dxc-markers";
                this._applyMarkerClipRect(markersSettings);
                groupSettings = _extend({}, markersSettings);
                delete groupSettings.opacity;
                this._createGroup("_markersGroup", this, this._group, groupSettings)
            },
            _getCreatingPointOptions: function() {
                if (!this._predefinedPointOptions) {
                    var styles,
                        defaultPointOptions = this._getPointOptions(),
                        strokeWidth = defaultPointOptions.styles && defaultPointOptions.styles.normal && defaultPointOptions.styles.normal.strokeWidth,
                        r = defaultPointOptions.styles && defaultPointOptions.styles.normal && defaultPointOptions.styles.normal.r,
                        opacity = defaultPointOptions.styles && defaultPointOptions.styles.normal && defaultPointOptions.styles.normal.opacity,
                        creatingPointOptions = _extend(true, {}, defaultPointOptions);
                    creatingPointOptions.styles = styles = creatingPointOptions.styles || {};
                    styles.normal = {
                        strokeWidth: strokeWidth,
                        r: r,
                        opacity: opacity
                    };
                    this._predefinedPointOptions = creatingPointOptions
                }
                return this._predefinedPointOptions
            },
            _checkData: function(data) {
                return _isDefined(data.argument) && _isDefined(data.size) && data.value !== undefined
            },
            _getPointData: function(data, options) {
                var pointData = scatterSeries._getPointData.call(this, data, options),
                    sizeField = options.sizeField || "size";
                pointData.size = data[sizeField];
                return pointData
            },
            _fusionPoints: function(fusionPoints, tick, index) {
                var medianValue = scatterSeries._calcMedianValue.call(this, fusionPoints, "value"),
                    medianSize = scatterSeries._calcMedianValue.call(this, fusionPoints, "size");
                return {
                        size: medianSize,
                        value: medianValue,
                        argument: tick,
                        tag: null
                    }
            },
            getValueFields: function() {
                return [this._options.valueField || "val", this._options.sizeField || "size"]
            },
            updateTeamplateFieldNames: function() {
                var that = this,
                    options = that._options,
                    valueFields = that.getValueFields(),
                    name = that.name;
                options.valueField = valueFields[0] + name;
                options.sizeField = valueFields[1] + name;
                options.tagField = that.getTagField() + name
            },
            _clearingAnimation: function(translators, drawComplete) {
                var that = this,
                    settings = {r: 0},
                    partitionDuration = 0.5,
                    lastPointIndex = that._drawedPoints.length - 1;
                that._labelsGroup && that._labelsGroup.animate({opacity: 0.001}, {
                    duration: that._defaultDuration,
                    partitionDuration: partitionDuration
                }, function() {
                    _each(that._drawedPoints || [], function(i, p) {
                        p.animate(i === lastPointIndex ? drawComplete : undefined, settings, partitionDuration)
                    })
                })
            },
            _animate: function(firstDrawing) {
                var that = this,
                    lastPointIndex = that._drawedPoints.length - 1,
                    labelAnimFunc = function() {
                        that._labelsGroup && that._labelsGroup.animate({opacity: 1}, {duration: that._defaultDuration})
                    };
                _each(that._drawedPoints || [], function(i, p) {
                    p.animate(i === lastPointIndex ? labelAnimFunc : undefined, {
                        r: p.bubbleSize,
                        translate: {
                            x: p.x,
                            y: p.y
                        }
                    })
                })
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file pieSeries.js */
    (function($, DX) {
        var viz = DX.viz,
            mixins = viz.core.series.mixins,
            pieSeries = mixins.pieChart,
            utils = DX.utils,
            scatterSeries = mixins.chart.scatter,
            barSeries = mixins.chart.bar,
            _extend = $.extend,
            _each = $.each,
            _noop = $.noop,
            _map = $.map,
            _isFinite = isFinite,
            _max = Math.max;
        pieSeries.pie = _extend({}, barSeries, {
            _createLabelGroup: scatterSeries._createLabelGroup,
            _createGroups: scatterSeries._createGroups,
            _drawPoint: function(point) {
                scatterSeries._drawPoint.apply(this, arguments);
                !point.isVisible() && point.setInvisibility()
            },
            _applyTrackersClippings: _noop,
            _processRange: _noop,
            _applyElementsClipRect: _noop,
            _updateDataType: _noop,
            getColor: _noop,
            _prepareSeriesToDrawing: _noop,
            _endUpdateData: scatterSeries._prepareSeriesToDrawing,
            resetLabelEllipsis: function() {
                _each(this._points || [], function(_, point) {
                    point._label.setLabelEllipsis = false
                })
            },
            _adjustLabels: function(firstDrawing) {
                var that = this,
                    options = that._options,
                    canvas = that.canvas,
                    points = that._points || [],
                    labelsBBoxes = [],
                    labelsSpaces = [],
                    labelSpace = 0,
                    maxLabelLength,
                    paneSpaceHeight,
                    paneSpaceWidth,
                    min;
                if (options.label.position !== "inside") {
                    _each(points, function(_, point) {
                        if (point._label.hasText())
                            labelsBBoxes.push(point._label.getTextCoords().width)
                    });
                    if (labelsBBoxes.length) {
                        maxLabelLength = _max.apply(null, labelsBBoxes);
                        _each(points, function(_, point) {
                            if (point._label.hasText()) {
                                point._label.updatePosition(maxLabelLength);
                                point._label.setLabelEllipsis = true
                            }
                        });
                        that._firstDrawing = !that.redraw && firstDrawing;
                        that.redraw = true
                    }
                }
                else
                    _each(points, function(i, point) {
                        if (point._label.hasText()) {
                            point._label.setLabelEllipsis = true;
                            point._label.updatePosition()
                        }
                    })
            },
            _getCreatingPointOptions: function() {
                return this._getPointOptions()
            },
            _updateOptions: function(options) {
                this.labelSpace = 0;
                this.innerRadius = this.type === "pie" ? 0 : options.innerRadius;
                this.redraw = false
            },
            _checkData: function(data) {
                var base = barSeries._checkData(data);
                return this._options.paintNullPoints ? base : base && data.value !== null
            },
            _createMarkerGroup: function() {
                if (!this._markersGroup)
                    this._markersGroup = this._renderer.createGroup({"class": "dxc-markers"}).append(this._group)
            },
            _getMainColor: function() {
                return this._options.mainSeriesColor()
            },
            _getPointOptions: function() {
                return this._parsePointOptions(this._preparePointOptions(), this._options.label)
            },
            _getRangeData: function() {
                return this._rangeData
            },
            _getArrangeTotal: function(points) {
                var total = 0;
                _each(points, function(_, point) {
                    if (point.isVisible())
                        total += point.initialValue
                });
                return total
            },
            _createPointStyles: function(pointOptions) {
                var mainColor = pointOptions.color || this._getMainColor(),
                    specialMainColor = this._getSpecialColor(mainColor);
                return {
                        normal: this._parsePointStyle(pointOptions, mainColor, mainColor),
                        hover: this._parsePointStyle(pointOptions.hoverStyle, specialMainColor, mainColor),
                        selection: this._parsePointStyle(pointOptions.selectionStyle, specialMainColor, mainColor),
                        legendStyles: {
                            normal: this._createLegendState(pointOptions, mainColor),
                            hover: this._createLegendState(pointOptions.hoverStyle, specialMainColor),
                            selection: this._createLegendState(pointOptions.selectionStyle, specialMainColor)
                        }
                    }
            },
            _getArrangeMinShownValue: function(points, total) {
                var minSegmentSize = this._options.minSegmentSize,
                    totalMinSegmentSize = 0,
                    totalNotMinValues = 0;
                total = total || points.length;
                _each(points, function(_, point) {
                    if (point.isVisible())
                        if (point.initialValue < minSegmentSize * total / 360)
                            totalMinSegmentSize += minSegmentSize;
                        else
                            totalNotMinValues += point.initialValue
                });
                return totalMinSegmentSize < 360 ? minSegmentSize * totalNotMinValues / (360 - totalMinSegmentSize) : 0
            },
            _applyArrangeCorrection: function(points, minShownValue, total) {
                var options = this._options,
                    isClockWise = options.segmentsDirection !== "anticlockwise",
                    shiftedAngle = _isFinite(options.startAngle) ? utils.normalizeAngle(options.startAngle) : 0,
                    minSegmentSize = options.minSegmentSize,
                    percent,
                    correction = 0,
                    zeroTotalCorrection = 0;
                if (total === 0) {
                    total = points.length;
                    zeroTotalCorrection = 1
                }
                _each(isClockWise ? points : points.concat([]).reverse(), function(_, point) {
                    var val = point.isVisible() ? zeroTotalCorrection || point.initialValue : 0,
                        updatedZeroValue;
                    if (minSegmentSize && point.isVisible() && val < minShownValue)
                        updatedZeroValue = minShownValue;
                    percent = val / total;
                    point.correctValue(correction, percent, zeroTotalCorrection + (updatedZeroValue || 0));
                    point.shiftedAngle = shiftedAngle;
                    correction = correction + (updatedZeroValue || val)
                });
                this._rangeData = {val: {
                        min: 0,
                        max: correction
                    }}
            },
            arrangePoints: function() {
                var that = this,
                    minSegmentSize = that._options.minSegmentSize,
                    minShownValue,
                    pointIndex = 0,
                    total,
                    points = that._originalPoints = that._points = _map(that._originalPoints || [], function(point) {
                        if (point.value === null || point.value < 0 || point.value === 0 && !minSegmentSize) {
                            point.dispose();
                            return null
                        }
                        else {
                            point.index = pointIndex++;
                            return point
                        }
                    });
                total = that._getArrangeTotal(points);
                if (minSegmentSize)
                    minShownValue = this._getArrangeMinShownValue(points, total);
                that._applyArrangeCorrection(points, minShownValue, total)
            },
            correctPosition: function(correction) {
                var debug = DX.utils.debug;
                debug.assert(correction, "correction was not passed");
                debug.assertParam(correction.centerX, "correction.centerX was not passed");
                debug.assertParam(correction.centerY, "correction.centerY was not passed");
                debug.assertParam(correction.radiusInner, "correction.radiusInner was not passed");
                debug.assertParam(correction.radiusOuter, "correction.radiusOuter was not passed");
                _each(this._points, function(_, point) {
                    point.correctPosition(correction)
                })
            },
            _animate: function(firstDrawing) {
                var that = this,
                    index = 0,
                    timeThreshold = 0.2,
                    points = that._points,
                    pointsCount = points && points.length,
                    duration = 1 / (timeThreshold * (pointsCount - 1) + 1),
                    animateP = function() {
                        points[index] && points[index++].animate(index === pointsCount ? completeFunc : undefined, duration, stepFunc)
                    },
                    stepFunc = function(_, progress) {
                        if (progress >= timeThreshold) {
                            this.step = null;
                            animateP()
                        }
                    },
                    completeFunc = function() {
                        that._labelsGroup && that._labelsGroup.animate({opacity: 1}, {duration: 400})
                    };
                if (firstDrawing)
                    animateP();
                else
                    $.each(points, function(i, p) {
                        p.animate(i == pointsCount - 1 ? completeFunc : undefined)
                    })
            },
            getVisiblePoints: function() {
                return _map(this._points, function(p) {
                        return p.isVisible() ? p : null
                    })
            },
            _beginUpdateData: function() {
                this._deletePatterns();
                this._patterns = []
            }
        });
        pieSeries.doughnut = pieSeries.donut = pieSeries.pie
    })(jQuery, DevExpress);
    /*! Module viz-core, file financialSeries.js */
    (function($, DX) {
        var viz = DX.viz,
            series = viz.core.series.mixins.chart,
            scatterSeries = series.scatter,
            barSeries = series.bar,
            rangeCalculator = viz.core.series.helpers.rangeDataCalculator(),
            _isDefined = DX.utils.isDefined,
            _extend = $.extend,
            _each = $.each,
            _noop = $.noop,
            DEFAULT_FINANCIAL_POINT_SIZE = 10;
        series.stock = _extend({}, scatterSeries, {
            _animate: _noop,
            _applyMarkerClipRect: series.line._applyElementsClipRect,
            _applyTrackersClippings: barSeries._applyTrackersClippings,
            _createPattern: barSeries._createPattern,
            _preparePointOptions: barSeries._preparePointOptions,
            _createMarkerGroup: function() {
                var that = this,
                    markersGroup,
                    styles = that._getPointOptions().styles,
                    defaultStyle = styles.normal,
                    defaultPositiveStyle = styles.positive.normal,
                    reductionStyle = styles.reduction.normal,
                    reductionPositiveStyle = styles.reductionPositive.normal,
                    markerSettings = {"class": "dxc-markers"};
                that._applyMarkerClipRect(markerSettings);
                defaultStyle["class"] = "default-markers";
                defaultPositiveStyle["class"] = "default-positive-markers";
                reductionStyle["class"] = "reduction-markers";
                reductionPositiveStyle["class"] = "reduction-positive-markers";
                that._createGroup("_markersGroup", that, that._group, markerSettings);
                markersGroup = that._markersGroup;
                that._createGroup("defaultMarkersGroup", markersGroup, markersGroup, defaultStyle);
                that._createGroup("reductionMarkersGroup", markersGroup, markersGroup, reductionStyle);
                that._createGroup("defaultPositiveMarkersGroup", markersGroup, markersGroup, defaultPositiveStyle);
                that._createGroup("reductionPositiveMarkersGroup", markersGroup, markersGroup, reductionPositiveStyle)
            },
            _createGroups: function() {
                scatterSeries._createGroups.call(this, false)
            },
            _clearingAnimation: function(translators, drawComplete) {
                drawComplete()
            },
            _getCreatingPointOptions: function() {
                if (!this._predefinedPointOptions) {
                    var styles,
                        defaultPointOptions = this._getPointOptions(),
                        strokeWidth = defaultPointOptions.styles && defaultPointOptions.styles.normal && defaultPointOptions.styles.normal.strokeWidth,
                        creatingPointOptions = _extend(true, {}, defaultPointOptions);
                    creatingPointOptions.styles = styles = creatingPointOptions.styles || {};
                    styles.normal = styles.positive.normal = styles.reduction.normal = styles.reductionPositive.normal = {strokeWidth: strokeWidth};
                    this._predefinedPointOptions = creatingPointOptions
                }
                return this._predefinedPointOptions
            },
            _checkData: function(data) {
                return _isDefined(data.argument) && data.highValue !== undefined && data.lowValue !== undefined && data.openValue !== undefined && data.closeValue !== undefined
            },
            _processRange: function(point, prevPoint) {
                rangeCalculator.processTwoValues(this, point, prevPoint, "highValue", "lowValue")
            },
            _getRangeData: function(zoomArgs, calcIntervalFunction) {
                rangeCalculator.calculateTwoValuesRangeData(this, zoomArgs, calcIntervalFunction, "highValue", "lowValue");
                rangeCalculator.addRangeSeriesLabelPaddings(this);
                return this._rangeData
            },
            _getPointData: function(data, options) {
                var that = this,
                    level,
                    argumentField = options.argumentField || "date",
                    tagField = options.tagField || "tag",
                    openValueField = options.openValueField || "open",
                    closeValueField = options.closeValueField || "close",
                    highValueField = options.highValueField || "high",
                    lowValueField = options.lowValueField || "low",
                    reductionValue;
                that.level = options.reduction.level;
                switch ((that.level || "").toLowerCase()) {
                    case"open":
                        level = openValueField;
                        break;
                    case"high":
                        level = highValueField;
                        break;
                    case"low":
                        level = lowValueField;
                        break;
                    default:
                        level = closeValueField;
                        that.level = "close";
                        break
                }
                reductionValue = data[level];
                return {
                        argument: data[argumentField],
                        highValue: data[highValueField],
                        lowValue: data[lowValueField],
                        closeValue: data[closeValueField],
                        openValue: data[openValueField],
                        reductionValue: reductionValue,
                        tag: data[tagField],
                        isReduction: this._checkReduction(reductionValue)
                    }
            },
            _parsePointStyle: function(style, defaultColor, innerColor) {
                return {
                        stroke: style.color || defaultColor,
                        strokeWidth: style.width,
                        fill: style.color || innerColor
                    }
            },
            updateTeamplateFieldNames: function() {
                var that = this,
                    options = that._options,
                    valueFields = that.getValueFields(),
                    name = that.name;
                options.openValueField = valueFields[0] + name;
                options.highValueField = valueFields[1] + name;
                options.lowValueField = valueFields[2] + name;
                options.closeValueField = valueFields[3] + name;
                options.tagField = that.getTagField() + name
            },
            _getDefaultStyle: function(options) {
                var mainPointColor = options.color || this._options.mainSeriesColor;
                return {
                        normal: this._parsePointStyle(options, mainPointColor, mainPointColor),
                        hover: this._parsePointStyle(options.hoverStyle, mainPointColor, mainPointColor),
                        selection: this._parsePointStyle(options.selectionStyle, mainPointColor, mainPointColor)
                    }
            },
            _getReductionStyle: function(options) {
                var reductionColor = options.reduction.color,
                    normalStyle = this._parsePointStyle({
                        color: reductionColor,
                        width: options.width,
                        hatching: options.hatching
                    }, reductionColor, reductionColor);
                return {
                        normal: normalStyle,
                        hover: this._parsePointStyle(options.hoverStyle, reductionColor, reductionColor),
                        selection: this._parsePointStyle(options.selectionStyle, reductionColor, reductionColor)
                    }
            },
            _createPointStyles: function(pointOptions) {
                var that = this,
                    innerColor = that._options.innerColor,
                    styles = that._getDefaultStyle(pointOptions),
                    positiveStyle,
                    reductionStyle,
                    reductionPositiveStyle;
                positiveStyle = _extend(true, {}, styles);
                reductionStyle = that._getReductionStyle(pointOptions);
                reductionPositiveStyle = _extend(true, {}, reductionStyle);
                positiveStyle.normal.fill = positiveStyle.hover.fill = positiveStyle.selection.fill = innerColor;
                reductionPositiveStyle.normal.fill = reductionPositiveStyle.hover.fill = reductionPositiveStyle.selection.fill = innerColor;
                styles.positive = positiveStyle;
                styles.reduction = reductionStyle;
                styles.reductionPositive = reductionPositiveStyle;
                return styles
            },
            _endUpdateData: function() {
                delete this.prevLevelValue;
                delete this._predefinedPointOptions
            },
            _checkReduction: function(value) {
                var result = false;
                if (value != null) {
                    if (_isDefined(this.prevLevelValue))
                        result = value < this.prevLevelValue;
                    this.prevLevelValue = value
                }
                return result
            },
            _fusionPoints: function(fusionPoints, tick, nowIndexTicks) {
                var fusedPointData = {},
                    reductionLevel,
                    highValue = -Infinity,
                    lowValue = +Infinity,
                    openValue,
                    closeValue;
                if (!fusionPoints.length)
                    return {};
                _each(fusionPoints, function(_, point) {
                    if (!point.hasValue())
                        return;
                    highValue = Math.max(highValue, point.highValue);
                    lowValue = Math.min(lowValue, point.lowValue);
                    openValue = openValue !== undefined ? openValue : point.openValue;
                    closeValue = point.closeValue !== undefined ? point.closeValue : closeValue
                });
                fusedPointData.argument = tick;
                fusedPointData.openValue = openValue;
                fusedPointData.closeValue = closeValue;
                fusedPointData.highValue = highValue;
                fusedPointData.lowValue = lowValue;
                fusedPointData.tag = null;
                switch ((this.level || "").toLowerCase()) {
                    case"open":
                        reductionLevel = openValue;
                        break;
                    case"high":
                        reductionLevel = highValue;
                        break;
                    case"low":
                        reductionLevel = lowValue;
                        break;
                    default:
                        reductionLevel = closeValue;
                        break
                }
                fusedPointData.reductionValue = reductionLevel;
                fusedPointData.isReduction = this._checkReduction(reductionLevel);
                return fusedPointData
            },
            _getPointSize: function() {
                return DEFAULT_FINANCIAL_POINT_SIZE
            },
            getValueFields: function() {
                var options = this._options;
                return [options.openValueField || "open", options.highValueField || "high", options.lowValueField || "low", options.closeValueField || "close"]
            },
            getArgumentField: function() {
                return this._options.argumentField || "date"
            }
        });
        series.candlestick = _extend({}, series.stock, {_parsePointStyle: function(style, defaultColor, innerColor) {
                var color = this._createPattern(style.color || innerColor, style.hatching),
                    base = series.stock._parsePointStyle.call(this, style, defaultColor, color);
                base.fill = color;
                return base
            }})
    })(jQuery, DevExpress);
    /*! Module viz-core, file stackedSeries.js */
    (function($, DX) {
        var viz = DX.viz,
            series = viz.core.series.mixins.chart,
            areaSeries = series.area,
            barSeries = series.bar,
            lineSeries = series.line,
            rangeCalculator = viz.core.series.helpers.rangeDataCalculator(),
            _extend = $.extend,
            _noop = $.noop,
            baseStackedSeries = {
                _processRange: _noop,
                _processStackedRange: function() {
                    var that = this,
                        prevPoint;
                    $.each(that.getAllPoints(), function(i, p) {
                        rangeCalculator.processRange(that, p, prevPoint);
                        prevPoint = p
                    })
                },
                _getRangeData: function() {
                    this._processStackedRange();
                    return areaSeries._getRangeData.apply(this, arguments)
                }
            },
            baseFullStackedSeries = _extend({}, baseStackedSeries, {
                _getRangeData: function(zoomArgs, calcIntervalFunction) {
                    this._processStackedRange();
                    rangeCalculator.calculateRangeData(this, zoomArgs, calcIntervalFunction);
                    rangeCalculator.addLabelPaddings(this);
                    rangeCalculator.processFullStackedRange(this);
                    rangeCalculator.calculateRangeMinValue(this, zoomArgs);
                    return this._rangeData
                },
                isFullStackedSeries: function() {
                    return true
                }
            });
        series.stackedline = _extend({}, lineSeries, baseStackedSeries, {_getRangeData: function() {
                this._processStackedRange();
                return lineSeries._getRangeData.apply(this, arguments)
            }});
        series.fullstackedline = _extend({}, lineSeries, baseFullStackedSeries, {_getRangeData: function(zoomArgs, calcIntervalFunction) {
                this._processStackedRange();
                var rangeCalculator = viz.core.series.helpers.rangeDataCalculator();
                rangeCalculator.calculateRangeData(this, zoomArgs, calcIntervalFunction);
                rangeCalculator.addLabelPaddings(this);
                rangeCalculator.processFullStackedRange(this);
                return this._rangeData
            }});
        series.stackedbar = _extend({}, barSeries, baseStackedSeries, {_getRangeData: function() {
                this._processStackedRange();
                return barSeries._getRangeData.apply(this, arguments)
            }});
        series.fullstackedbar = _extend({}, barSeries, baseFullStackedSeries, {_getRangeData: function(zoomArgs, calcIntervalFunction) {
                var rangeData = baseFullStackedSeries._getRangeData.apply(this, arguments);
                rangeData.arg.stick = false;
                return rangeData
            }});
        series.stackedarea = _extend({}, areaSeries, baseStackedSeries);
        series.fullstackedarea = _extend({}, areaSeries, baseFullStackedSeries)
    })(jQuery, DevExpress);
    /*! Module viz-core, file basePoint.js */
    (function($, DX) {
        var viz = DX.viz,
            statesConsts = viz.core.series.helpers.consts.states,
            _each = $.each,
            _extend = $.extend,
            _isDefined = DX.utils.isDefined,
            _noop = $.noop,
            pointTypes = {
                scatter: "symbolPoint",
                line: "symbolPoint",
                spline: "symbolPoint",
                stepline: "symbolPoint",
                stackedline: "symbolPoint",
                fullstackedline: "symbolPoint",
                area: "symbolPoint",
                splinearea: "symbolPoint",
                steparea: "symbolPoint",
                stackedarea: "symbolPoint",
                fullstackedarea: "symbolPoint",
                rangearea: "rangeSymbolPoint",
                bar: "barPoint",
                stackedbar: "barPoint",
                fullstackedbar: "barPoint",
                rangebar: "rangeBarPoint",
                bubble: "bubblePoint",
                pie: "piePoint",
                doughnut: "piePoint",
                donut: "piePoint",
                stock: "stockPoint",
                candlestick: "candlestickPoint"
            };
        viz.core.series.points = {};
        viz.core.series.points.Point = DX.Class.inherit({
            ctor: function(dataItem, options) {
                this.update(dataItem, options);
                this._emptySettings = {
                    fill: null,
                    stroke: null
                }
            },
            getColor: function() {
                return this._styles.normal.fill || this.series.getColor()
            },
            _getStyle: function() {
                var styles = this._styles,
                    style;
                if (this.isSelected())
                    style = styles.selection;
                else if (this.isHovered())
                    style = styles.hover;
                else {
                    this.fullState = statesConsts.normalMark;
                    style = styles.normal
                }
                return style
            },
            update: function(dataItem, options) {
                this.updateOptions(options);
                this.updateData(dataItem)
            },
            updateData: function(dataItem) {
                this._updateData(dataItem);
                if (!this.hasValue())
                    this.setInvisibility();
                else {
                    this._updateLabelData();
                    this._fillStyle()
                }
            },
            deleteMarker: function() {
                this.graphic && this.graphic.detach();
                this.graphic = null
            },
            deleteTrackerMarker: function() {
                this.trackerGraphic && this.trackerGraphic.remove();
                this.trackerGraphic = null
            },
            draw: function(renderer, markersGroup, labelsGroup, animationEnabled, firstDrawing) {
                if (this._needDeletingOnDraw) {
                    this.deleteMarker();
                    this.deleteTrackerMarker();
                    this._needDeletingOnDraw = false
                }
                if (this._needClearingOnDraw) {
                    this.clearMarker();
                    this._needClearingOnDraw = false
                }
                if (!this._hasGraphic())
                    this._options.visible && this._drawMarker(renderer, markersGroup, animationEnabled, firstDrawing);
                else
                    this._updateMarker(animationEnabled, undefined, markersGroup);
                this._drawLabel(renderer, labelsGroup)
            },
            applyStyle: function(style) {
                if (this.graphic) {
                    if (style === "normal")
                        this.clearMarker();
                    else
                        this.graphic.toForeground();
                    this._updateMarker(true, this._styles[style])
                }
            },
            setHoverState: function() {
                this.series.setPointHoverState(this)
            },
            releaseHoverState: function(callback) {
                this.series.releasePointHoverState(this, callback);
                if (this.graphic)
                    !this.isSelected() && this.graphic.toBackground()
            },
            setSelectedState: function() {
                this.series.setPointSelectedState(this)
            },
            releaseSelectedState: function() {
                this.series.releasePointSelectedState(this)
            },
            select: function() {
                this.series.selectPoint(this)
            },
            clearSelection: function() {
                this.series.deselectPoint(this)
            },
            showTooltip: function() {
                this.series.showPointTooltip(this)
            },
            hideTooltip: function() {
                this.series.hidePointTooltip(this)
            },
            _checkLabelsChanging: function(oldType, newType) {
                if (oldType) {
                    var isNewRange = ~newType.indexOf("range"),
                        isOldRange = ~oldType.indexOf("range");
                    return isOldRange && !isNewRange || !isOldRange && isNewRange
                }
                else
                    return false
            },
            updateOptions: function(newOptions) {
                if (!_isDefined(newOptions))
                    return;
                var that = this,
                    oldOptions = that._options,
                    oldType = oldOptions && oldOptions.type,
                    newType = newOptions.type;
                if (pointTypes[oldType] !== pointTypes[newType]) {
                    that._needDeletingOnDraw = true;
                    that._needClearingOnDraw = false;
                    that._checkLabelsChanging(oldType, newType) && that.deleteLabel();
                    that._resetType(oldType);
                    that._setType(newType)
                }
                else {
                    that._needDeletingOnDraw = that._checkSymbol(oldOptions, newOptions);
                    that._needClearingOnDraw = that._checkCustomize(oldOptions, newOptions)
                }
                that.series = newOptions.series;
                that._options = newOptions;
                that._fillStyle();
                that._updateLabelOptions(pointTypes[newType])
            },
            translate: function(translators) {
                this.translators = translators || this.translators;
                this.translators && this.hasValue() && this._translate(this.translators)
            },
            drawTracker: function(renderer, group) {
                if (!this.trackerGraphic)
                    this._drawTrackerMarker(renderer, group);
                else
                    this._updateTracker()
            },
            _checkCustomize: function(oldOptions, newOptions) {
                return oldOptions.styles.usePointCustomOptions && !newOptions.styles.usePointCustomOptions
            },
            _getCustomLabelVisibility: function() {
                if (this._styles.useLabelCustomOptions)
                    return this._options.label.visible ? "visible" : "hidden"
            },
            _getGraphicSettings: function() {
                return {
                        x: this.graphic.settings.x || 0,
                        y: this.graphic.settings.y || 0,
                        height: this.graphic.settings.height || 0,
                        width: this.graphic.settings.width || 0
                    }
            },
            _resetType: function(type) {
                var that = this;
                if (type)
                    _each(viz.core.series.points.mixins[pointTypes[type]], function(methodName) {
                        delete that[methodName]
                    })
            },
            _setType: function(type) {
                var that = this;
                _each(viz.core.series.points.mixins[pointTypes[type]], function(methodName, method) {
                    that[methodName] = method
                })
            },
            _prepareStatesOptions: function() {
                var that = this;
                if (that._options.states && that._options.states.normal)
                    that._populatePointShape(that._options.states.normal.r)
            },
            isInVisibleArea: function() {
                return this.inVisibleArea
            },
            isSelected: function() {
                return !!(this.fullState & statesConsts.selectedMark)
            },
            isHovered: function() {
                return !!(this.fullState & statesConsts.hoverMark)
            },
            getOptions: function() {
                return this._options
            },
            animate: function(complete, settings, partitionDuration) {
                if (!this.graphic) {
                    complete && complete();
                    return
                }
                this.graphic.animate(settings, {partitionDuration: partitionDuration}, complete)
            },
            getCoords: function(min) {
                if (!min)
                    return {
                            x: this.x,
                            y: this.y
                        };
                if (!this._options.rotated)
                    return {
                            x: this.x,
                            y: this.minY
                        };
                return {
                        x: this.minX,
                        y: this.y
                    }
            },
            getDefaultCoords: function() {
                return !this._options.rotated ? {
                        x: this.x,
                        y: this.defaultY
                    } : {
                        x: this.defaultX,
                        y: this.y
                    }
            },
            _calculateVisibility: function(x, y, width, height) {
                var that = this,
                    visibleAreaX,
                    visibleAreaY,
                    rotated = that._options.rotated;
                if (that.translators) {
                    visibleAreaX = that.translators.x.getCanvasVisibleArea();
                    visibleAreaY = that.translators.y.getCanvasVisibleArea();
                    if (visibleAreaX.min > x + (width || 0) || visibleAreaX.max < x || visibleAreaY.min > y + (height || 0) || visibleAreaY.max < y || rotated && _isDefined(width) && width !== 0 && (visibleAreaX.min === x + width || visibleAreaX.max === x) || !rotated && _isDefined(height) && height !== 0 && (visibleAreaY.min === y + height || visibleAreaY.max === y))
                        that.inVisibleArea = false;
                    else
                        that.inVisibleArea = true
                }
            },
            correctPosition: _noop,
            hasValue: function() {
                return this.value !== null && this.minValue !== null
            },
            _populatePointShape: _noop,
            _checkSymbol: _noop,
            hide: _noop,
            show: _noop,
            hideMarker: _noop,
            setInvisibility: _noop,
            clearVisibility: _noop,
            isVisible: _noop,
            resetCorrection: _noop,
            correctValue: _noop,
            setPercentValue: _noop,
            correctCoordinates: _noop,
            dispose: function() {
                this.deleteMarker();
                this.deleteTrackerMarker();
                this.deleteLabel();
                this._options = null;
                this._styles = null;
                this.series = null;
                this.translators = null
            },
            getTooltipFormatObject: function(tooltip) {
                var that = this,
                    tooltipFormatObject = that._getFormatObject(tooltip),
                    sharedTooltipValuesArray = [],
                    tooltipStackPointsFormatObject = [];
                if (that.stackPoints) {
                    _each(that.stackPoints, function(_, point) {
                        if (!point.isVisible())
                            return;
                        var formatObject = point._getFormatObject(tooltip);
                        tooltipStackPointsFormatObject.push(point._getFormatObject(tooltip));
                        sharedTooltipValuesArray.push(formatObject.seriesName + ": " + formatObject.valueText)
                    });
                    _extend(tooltipFormatObject, {
                        points: tooltipStackPointsFormatObject,
                        valueText: sharedTooltipValuesArray.join("\n"),
                        stackName: that.stackPoints.stackName
                    })
                }
                return tooltipFormatObject
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file label.js */
    (function($, DX) {
        var _isDefined = DX.utils.isDefined,
            _extend = $.extend,
            _math = Math,
            _round = _math.round,
            _floor = _math.floor,
            _isEmptyObject = $.isEmptyObject,
            _noop = $.noop,
            LABEL_BACKGROUND_PADDING_X = 8,
            LABEL_BACKGROUND_PADDING_Y = 4,
            LABEL_OFFSET = 10,
            setPositionFuncCollection = {
                symbolPoint: "_setPositionForSymbol",
                barPoint: "_setPositionForBar",
                bubblePoint: "_setPositionForBubble",
                rangeSymbolPoint: "_setPositionForSymbol",
                rangeBarPoint: "_setPositionForBar",
                stockPoint: "_setPositionForFinancial",
                candlestickPoint: "_setPositionForFinancial"
            },
            checkPositionFuncCollection = {
                symbolPoint: "_checkPositionForSymbol",
                barPoint: "_checkPositionForBar",
                bubblePoint: "_checkPositionForSymbol",
                rangeSymbolPoint: "_checkPositionForSymbol",
                rangeBarPoint: "_checkPositionForBar",
                stockPoint: "_checkPositionForBar",
                candlestickPoint: "_checkPositionForBar"
            };
        DX.viz.core.series.points.Label = DX.Class.inherit({
            ctor: function(data, options) {
                this._offset = LABEL_OFFSET;
                this.updateData(data);
                this.setOptions(options)
            },
            clearVisibility: function() {
                if (this._group && this._group.settings.visibility)
                    this._group.applySettings({visibility: null})
            },
            hide: function() {
                if (this._group && this._group.settings.visibility !== "hidden")
                    this._group.applySettings({visibility: "hidden"})
            },
            updateData: function(data) {
                if (_isDefined(data)) {
                    this._data = data.formatObject;
                    this._initialValue = data.initialValue
                }
            },
            updateOptions: function(options) {
                this.setOptions(_extend(true, {}, {options: this._options}, options))
            },
            setOptions: function(newOptions) {
                var that = this,
                    oldOptions = that._options,
                    newOptions = newOptions || {},
                    pointType = newOptions.type || "symbolPoint";
                that._options = newOptions.options;
                that._rotated = newOptions.rotated;
                that._isFullStacked = newOptions.isFullStacked;
                that._isRange = newOptions.isRange;
                that._setPosition = that[setPositionFuncCollection[pointType]];
                that._checkPosition = that[checkPositionFuncCollection[pointType]];
                if (oldOptions) {
                    that._isBackgroundChanged(oldOptions.background, that._options.background) && that._deleteBackground();
                    that._isConnectorChanged(oldOptions.connector, that._options.connector) && that._deleteConnector()
                }
            },
            setDataField: function(fieldName, fieldValue) {
                this._data = this._data || {};
                this._data[fieldName] = fieldValue
            },
            getData: function() {
                return this._data
            },
            setCoords: function(coords, graphicBbox, location, visibleArea) {
                this._coords = coords;
                this._graphicBbox = graphicBbox;
                this._visibleArea = visibleArea;
                this._location = location
            },
            hasText: function() {
                return !!this._text
            },
            getTextCoords: function() {
                return this._text && this._text.getBBox()
            },
            getCoords: function() {
                var coords = {},
                    insideGroup = this._insideGroup;
                if (insideGroup) {
                    coords = insideGroup.getBBox();
                    coords.x = insideGroup.settings.translateX || 0;
                    coords.y = insideGroup.settings.translateY || 0
                }
                return coords
            },
            updatePosition: function(x, y) {
                var settings = {},
                    insideGroup = this._insideGroup;
                insideGroup.applySettings({
                    translateX: x,
                    translateY: y
                });
                this._connector && this._drawConnector()
            },
            _deleteElements: function() {
                this._deleteConnector();
                this._deleteBackground();
                this._deleteText();
                this._deleteGroups()
            },
            dispose: function() {
                this._data = null;
                this._options = null;
                this._positioningFunction = null;
                this._graphicBbox = null;
                this._visibleArea = null;
                this._coords = null;
                this._deleteElements()
            },
            _deleteText: function() {
                this._text && this._text.detach();
                this._text = null
            },
            _deleteGroups: function() {
                this._insideGroup = null;
                this._group && this._group.detach();
                this._group = null
            },
            _drawGroups: function(renderer, group) {
                if (!this._group)
                    this._group = renderer.createGroup().append(group);
                if (!this._insideGroup)
                    this._insideGroup = renderer.createGroup().append(this._group)
            },
            _drawText: function(renderer, text) {
                if (!this._text)
                    this._text = renderer.createText(text, 0, 0, this._options.attributes).append(this._insideGroup);
                else
                    this._text.applySettings({text: text}, this._options.attributes)
            },
            _drawBackground: function(renderer) {
                var that = this,
                    options = that._options,
                    background = options.background || {},
                    settings;
                if (that._checkBackground(background)) {
                    settings = _extend(that._getBackgroundSettings(), background);
                    if (!that._background)
                        that._background = renderer.createRect(settings.x, settings.y, settings.width, settings.height, 0, background).append(that._insideGroup);
                    else
                        that._background.applySettings(settings);
                    that._background.toBackground()
                }
            },
            _drawConnector: function(renderer) {
                var that = this,
                    connector = that._options.connector || {},
                    points;
                if (that._checkConnector(connector)) {
                    points = that._getConnectorPoints() || [];
                    if (!that._connector)
                        that._connector = renderer.createPath(points, connector).append(that._group);
                    else
                        that._connector.applySettings(_extend({points: points}, connector));
                    that._connector.toBackground()
                }
            },
            _setVisibility: function(visibility) {
                this._group && this._group.applySettings({visibility: visibility})
            },
            draw: function(renderer, group, visibility) {
                var that = this,
                    options = that._options,
                    background = options.background,
                    text = that._format();
                if (_isDefined(text) && text !== "") {
                    that._drawGroups(renderer, group);
                    that._setVisibility(visibility);
                    that._drawText(renderer, text);
                    that._drawBackground(renderer);
                    that._rotateLabel();
                    that._setPosition();
                    that._drawConnector(renderer)
                }
                else
                    that._setVisibility("hidden")
            },
            _deleteBackground: function() {
                this._background && this._background.detach();
                this._background = null
            },
            _isBackgroundChanged: function(oldBackground, newBackground) {
                return this._checkBackground(oldBackground || {}) !== this._checkBackground(newBackground || {})
            },
            _checkBackground: function(background) {
                var hasColor = background.fill && background.fill !== "none",
                    hasBorder = background.strokeWidth && background.stroke && background.stroke !== "none";
                return hasColor || hasBorder
            },
            _getBackgroundSettings: function() {
                var bbox = this._text.getBBox();
                return {
                        x: bbox.x - LABEL_BACKGROUND_PADDING_X,
                        y: bbox.y - LABEL_BACKGROUND_PADDING_Y,
                        width: bbox.width + 2 * LABEL_BACKGROUND_PADDING_X,
                        height: bbox.height + 2 * LABEL_BACKGROUND_PADDING_Y
                    }
            },
            _deleteConnector: function() {
                this._connector && this._connector.detach();
                this._connector = null
            },
            _isConnectorChanged: function(oldConnector, newConnector) {
                return this._checkConnector(oldConnector || {}) !== this._checkConnector(newConnector || {})
            },
            _checkConnector: function(connector) {
                return connector && connector.strokeWidth
            },
            _getConnectorPoints: function() {
                var that = this,
                    rotated = that._rotated,
                    labelBbox = that._insideGroup.getBBox(),
                    graphicBbox = that._graphicBbox,
                    centerLabelY,
                    centerLabelX,
                    rightLabelCoord,
                    bottomLabelCoord,
                    x1,
                    x2,
                    y1,
                    y2;
                labelBbox.x += that._insideGroup.settings.translateX || 0;
                labelBbox.y += that._insideGroup.settings.translateY || 0;
                centerLabelY = that._background ? labelBbox.y + labelBbox.height / 2 : null;
                centerLabelX = that._background ? labelBbox.x + labelBbox.width / 2 : null,
                rightLabelCoord = labelBbox.x + labelBbox.width;
                bottomLabelCoord = labelBbox.y + labelBbox.height;
                x1 = _floor(labelBbox.x + labelBbox.width / 2);
                x2 = _floor(graphicBbox.x + graphicBbox.width / 2);
                if (bottomLabelCoord < graphicBbox.y) {
                    y1 = centerLabelY || bottomLabelCoord;
                    y2 = graphicBbox.y
                }
                else if (labelBbox.y > graphicBbox.y + graphicBbox.height) {
                    y1 = centerLabelY || labelBbox.y;
                    y2 = graphicBbox.y + graphicBbox.height
                }
                else {
                    if (labelBbox.x > graphicBbox.x + graphicBbox.width) {
                        x1 = centerLabelX || labelBbox.x;
                        x2 = graphicBbox.x + graphicBbox.width
                    }
                    else if (rightLabelCoord < graphicBbox.x) {
                        x1 = centerLabelX || rightLabelCoord;
                        x2 = graphicBbox.x
                    }
                    else
                        return;
                    y1 = _floor(labelBbox.y + labelBbox.height / 2);
                    y2 = _floor(graphicBbox.y + graphicBbox.height / 2)
                }
                return [{
                            x: x1,
                            y: y1
                        }, {
                            x: x2,
                            y: y2
                        }]
            },
            _rotateLabel: function() {
                var bbox = this._insideGroup.getBBox();
                this._insideGroup.rotate(this._options.rotationAngle)
            },
            _format: function() {
                var that = this,
                    data = that._data,
                    options = that._options,
                    formatHelper = DX.formatHelper;
                data.valueText = formatHelper.format(data.value, options.format, options.precision);
                data.argumentText = formatHelper.format(data.argument, options.argumentFormat, options.argumentPrecision);
                if (data.percent !== undefined)
                    data.percentText = formatHelper.format(data.percent, "percent", options.percentPrecision);
                if (data.total !== undefined)
                    data.totalText = formatHelper.format(data.total, options.format, options.precision);
                if (data.openValue !== undefined)
                    data.openValueText = formatHelper.format(data.openValue, options.format, options.precision);
                if (data.closeValue !== undefined)
                    data.closeValueText = formatHelper.format(data.closeValue, options.format, options.precision);
                if (data.lowValue !== undefined)
                    data.lowValueText = formatHelper.format(data.lowValue, options.format, options.precision);
                if (data.highValue !== undefined)
                    data.highValueText = formatHelper.format(data.highValue, options.format, options.precision);
                if (data.reductionValue !== undefined)
                    data.reductionValueText = formatHelper.format(data.reductionValue, options.format, options.precision);
                return options.customizeText ? options.customizeText.call(data, data) : data.valueText
            },
            _getOutsideLabelCoords: function(labelBbox) {
                var graphicBbox = this._graphicBbox,
                    x = 0,
                    y = 0,
                    isTop = this._location === "top";
                if (!this._rotated) {
                    x += graphicBbox.width / 2 + graphicBbox.x;
                    if (isTop)
                        y += graphicBbox.y - labelBbox.y - labelBbox.height - this._offset;
                    else
                        y += graphicBbox.y + graphicBbox.height - labelBbox.y + this._offset
                }
                else {
                    y += graphicBbox.y - labelBbox.y - labelBbox.height / 2 + graphicBbox.height / 2;
                    if (isTop)
                        x += graphicBbox.x + graphicBbox.width - labelBbox.x + this._offset;
                    else
                        x += graphicBbox.x - labelBbox.x - labelBbox.width - this._offset
                }
                return {
                        x: x,
                        y: y
                    }
            },
            _getInsideLabelCoords: function(labelBbox) {
                var graphicBbox = this._graphicBbox,
                    isTop = this._location === "top",
                    x = 0,
                    y = 0;
                if (!this._isRange)
                    if (!this._rotated) {
                        x += graphicBbox.width / 2 + graphicBbox.x;
                        y += graphicBbox.y - labelBbox.y - labelBbox.height / 2 + graphicBbox.height / 2
                    }
                    else {
                        y += graphicBbox.y - labelBbox.y - labelBbox.height / 2 + graphicBbox.height / 2;
                        x += graphicBbox.x - labelBbox.x - labelBbox.width / 2 + graphicBbox.width / 2
                    }
                else if (!this._rotated) {
                    x += graphicBbox.width / 2 + graphicBbox.x;
                    if (isTop)
                        y += graphicBbox.y - labelBbox.y - labelBbox.height + this._offset + labelBbox.height;
                    else
                        y += graphicBbox.y + graphicBbox.height - labelBbox.y - this._offset - labelBbox.height
                }
                else {
                    y += graphicBbox.y - labelBbox.y - labelBbox.height / 2 + graphicBbox.height / 2;
                    if (isTop)
                        x += graphicBbox.x + graphicBbox.width - labelBbox.x - labelBbox.width - this._offset;
                    else
                        x += graphicBbox.x - labelBbox.x + this._offset
                }
                return {
                        x: x,
                        y: y
                    }
            },
            _getFullstackedLabelCoords: function(labelBbox) {
                var x = 0,
                    y = 0,
                    graphicBbox = this._graphicBbox;
                if (!this._rotated) {
                    x += graphicBbox.width / 2 + graphicBbox.x;
                    y += this._coords.defaultY - labelBbox.y - labelBbox.height - this._offset
                }
                else {
                    y += graphicBbox.y - labelBbox.y - labelBbox.height / 2 + graphicBbox.height / 2;
                    x += this._coords.defaultX - labelBbox.x + this._offset
                }
                return {
                        x: x,
                        y: y
                    }
            },
            _setPositionForSymbol: function() {
                var that = this,
                    offset = that._offset,
                    labelBBox = that._insideGroup.getBBox(),
                    graphicBbox = that._graphicBbox,
                    isTop = that._location === "top",
                    correctionY = 0,
                    correctionX = 0,
                    x = 0,
                    y = 0;
                if (!that._rotated) {
                    correctionX = that._coords.x;
                    correctionY = isTop ? -labelBBox.y - labelBBox.height - offset : graphicBbox.height - labelBBox.y + offset;
                    correctionY += graphicBbox.y
                }
                else {
                    correctionY = graphicBbox.y - labelBBox.y - labelBBox.height / 2 + graphicBbox.height / 2;
                    correctionX = isTop ? graphicBbox.width - labelBBox.x + offset : -labelBBox.x - labelBBox.width - offset;
                    correctionX += graphicBbox.x
                }
                x += correctionX + that._options.horizontalOffset;
                y += correctionY + that._options.verticalOffset;
                that._checkPosition({
                    x: labelBBox.x + x,
                    y: labelBBox.y + y,
                    height: labelBBox.height,
                    width: labelBBox.width
                }, x, y)
            },
            _checkPositionForSymbol: function(labelBbox, x, y) {
                var that = this,
                    graphicBbox = that._graphicBbox,
                    visibleArea = that._visibleArea,
                    offset = that._offset;
                if (!that._rotated) {
                    if (visibleArea.minX <= graphicBbox.x + graphicBbox.width && visibleArea.maxX >= graphicBbox.x) {
                        if (visibleArea.minX > labelBbox.x && that.adjustSeriesLabels)
                            x += visibleArea.minX - labelBbox.x;
                        if (visibleArea.maxX < labelBbox.x + labelBbox.width && that.adjustSeriesLabels)
                            x -= labelBbox.x + labelBbox.width - visibleArea.maxX;
                        if (visibleArea.minY > labelBbox.y)
                            y += graphicBbox.y + graphicBbox.height - labelBbox.y + offset;
                        if (visibleArea.maxY < labelBbox.y + labelBbox.height)
                            y -= labelBbox.y + labelBbox.height - graphicBbox.y + offset
                    }
                }
                else if (visibleArea.minY <= graphicBbox.y + graphicBbox.height && visibleArea.maxY >= graphicBbox.y) {
                    if (visibleArea.minX > labelBbox.x)
                        x += graphicBbox.x + graphicBbox.width - labelBbox.x + offset;
                    if (visibleArea.maxX < labelBbox.x + labelBbox.width)
                        x -= labelBbox.x + labelBbox.width - graphicBbox.x + offset;
                    if (visibleArea.minY > graphicBbox.y && that.adjustSeriesLabels)
                        y += visibleArea.minY - labelBbox.y;
                    if (visibleArea.maxY < labelBbox.y + labelBbox.height && that.adjustSeriesLabels)
                        y -= labelBbox.y + labelBbox.height - visibleArea.maxY
                }
                that._insideGroup.applySettings({
                    translateX: _round(x),
                    translateY: _round(y)
                })
            },
            _setPositionForBar: function() {
                var that = this,
                    offset = that._offset,
                    labelPosition = that._options.position,
                    labelBbox = that._insideGroup.getBBox(),
                    coords;
                if (that._initialValue === 0 && that._isFullStacked)
                    coords = that._getFullstackedLabelCoords(labelBbox);
                else if (labelPosition === "inside")
                    coords = that._getInsideLabelCoords(labelBbox);
                else
                    coords = that._getOutsideLabelCoords(labelBbox);
                coords.x += that._options.horizontalOffset;
                coords.y += that._options.verticalOffset;
                that._checkPosition({
                    x: labelBbox.x + coords.x,
                    y: labelBbox.y + coords.y,
                    height: labelBbox.height,
                    width: labelBbox.width
                }, coords.x, coords.y)
            },
            _checkPositionForBar: function(labelBbox, x, y) {
                var that = this,
                    graphicBbox = that._graphicBbox,
                    visibleArea = that._visibleArea;
                if (visibleArea.minX <= graphicBbox.x + graphicBbox.width && visibleArea.maxX >= graphicBbox.x && visibleArea.minY <= graphicBbox.y + graphicBbox.height && visibleArea.maxY >= graphicBbox.y)
                    if (!that._rotated) {
                        if (visibleArea.minX > labelBbox.x && that.adjustSeriesLabels)
                            x += visibleArea.minX - labelBbox.x;
                        if (visibleArea.maxX < labelBbox.x + labelBbox.width && that.adjustSeriesLabels)
                            x -= labelBbox.x + labelBbox.width - visibleArea.maxX;
                        if (visibleArea.minY > labelBbox.y)
                            y += visibleArea.minY - labelBbox.y;
                        if (visibleArea.maxY < labelBbox.y + labelBbox.height)
                            y -= labelBbox.y + labelBbox.height - visibleArea.maxY
                    }
                    else {
                        if (visibleArea.minX > labelBbox.x)
                            x += visibleArea.minX - labelBbox.x;
                        if (visibleArea.maxX < labelBbox.x + labelBbox.width)
                            x -= labelBbox.x + labelBbox.width - visibleArea.maxX;
                        if (visibleArea.minY > labelBbox.y && that.adjustSeriesLabels)
                            y += visibleArea.minY - labelBbox.y;
                        if (visibleArea.maxY < labelBbox.y + labelBbox.height && that.adjustSeriesLabels)
                            y -= labelBbox.y + labelBbox.height - visibleArea.maxY
                    }
                if (that._options.resolveLabelsOverlapping && that._options.position === "inside")
                    if (that._rotated) {
                        if (labelBbox.width > graphicBbox.width || labelBbox.height > graphicBbox.height) {
                            that.hide();
                            return
                        }
                    }
                    else if (labelBbox.height > graphicBbox.height || labelBbox.width > graphicBbox.width) {
                        that.hide();
                        return
                    }
                that._insideGroup.applySettings({
                    translateX: _round(x),
                    translateY: _round(y)
                })
            },
            _setPositionForBubble: function() {
                var that = this,
                    labelBbox = that._insideGroup.getBBox(),
                    graphicBbox = that._graphicBbox,
                    x = 0,
                    y = 0;
                if (that._options.position === "inside") {
                    x = that._coords.x;
                    y += graphicBbox.y + graphicBbox.height / 2 - labelBbox.y - labelBbox.height / 2;
                    x += that._options.horizontalOffset;
                    y += that._options.verticalOffset;
                    that._checkPosition({
                        x: labelBbox.x + x,
                        y: labelBbox.y + y,
                        height: labelBbox.height,
                        width: labelBbox.width
                    }, x, y)
                }
                else
                    that._setPositionForSymbol()
            },
            _setPositionForFinancial: function() {
                var that = this,
                    offset = that._offset,
                    labelBbox = that._insideGroup.getBBox(),
                    coords;
                coords = that._getOutsideLabelCoords(labelBbox);
                coords.x += that._options.horizontalOffset;
                coords.y += that._options.verticalOffset;
                that._checkPosition({
                    x: labelBbox.x + coords.x,
                    y: labelBbox.y + coords.y,
                    height: labelBbox.height,
                    width: labelBbox.width
                }, coords.x, coords.y)
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file pieLabel.js */
    (function($, DX) {
        var getCosAndSin = DX.utils.getCosAndSin,
            _extend = $.extend,
            _math = Math,
            _round = _math.round,
            _max = _math.max,
            _inArray = $.inArray,
            _isDefined = DX.utils.isDefined,
            INDENT_FROM_PIE = 30,
            CONNECTOR_LENGTH = 20;
        DX.viz.core.series.points.PieLabel = DX.viz.core.series.points.Label.inherit({
            ctor: function(data, options) {
                this.setLabelEllipsis = false;
                this.indentFromPie = INDENT_FROM_PIE;
                this.updateData(data);
                this.setOptions(options)
            },
            updateData: function(data) {
                if (_isDefined(data))
                    this._data = data.formatObject
            },
            setOptions: function(newOptions) {
                var that = this,
                    oldOptions = that._options,
                    newOptions = newOptions || {};
                that._options = newOptions.options;
                that._borderWidth = newOptions.borderWidth;
                if (oldOptions) {
                    that._isBackgroundChanged(oldOptions.background, that._options.background) && that._deleteBackground();
                    that._isConnectorChanged(oldOptions.connector, that._options.connector) && that._deleteConnector()
                }
            },
            setCoords: function(coords, canvas) {
                this._middleAngle = coords.middleAngle;
                this._radiusOuter = coords.radiusOuter;
                this._centerX = coords.centerX;
                this._centerY = coords.centerY;
                this._canvas = canvas
            },
            updatePosition: function(maxLabelLength) {
                this._setPosition(maxLabelLength);
                this._checkEllipsis();
                this._drawBackground();
                this._rotateLabel();
                this._checkPosition();
                this._connector && this._drawConnector()
            },
            dispose: function() {
                this._data = null;
                this._options = null;
                this._canvas = null;
                this._deleteElements()
            },
            draw: function(renderer, group, visibility) {
                var that = this,
                    options = that._options,
                    background = options.background,
                    text = that._format();
                if (_isDefined(text) && text !== "") {
                    that._drawGroups(renderer, group);
                    that._setVisibility(visibility);
                    that._drawText(renderer, text);
                    that._setPosition();
                    that._checkEllipsis();
                    that._drawBackground(renderer);
                    that._rotateLabel();
                    that._checkPosition();
                    that._drawConnector(renderer)
                }
                else
                    that._setVisibility("hidden")
            },
            _getOutsideGroupLabelPosition: function() {
                this._deleteConnector();
                return this._group && this._group.getBBox()
            },
            _setPosition: function(maxLabelLength) {
                var that = this,
                    bbox = that._text.getBBox(),
                    options = that._options,
                    angleFunctions = getCosAndSin(that._middleAngle),
                    align = "center",
                    rad = that._radiusOuter + options.radialOffset,
                    canvas = that._canvas,
                    rightBorderX = canvas.width - canvas.right,
                    leftBorderX = canvas.left,
                    x,
                    y;
                if (options.position === 'inside') {
                    rad -= that.indentFromPie;
                    x = that._centerX + rad * angleFunctions.cos
                }
                else if (options.position === 'columns') {
                    rad += CONNECTOR_LENGTH;
                    if (angleFunctions.cos >= 0) {
                        align = "right";
                        x = maxLabelLength ? that._centerX + rad + maxLabelLength : rightBorderX;
                        x = x > rightBorderX ? rightBorderX : x
                    }
                    else if (angleFunctions.cos < 0) {
                        align = "left";
                        x = maxLabelLength ? that._centerX - rad - maxLabelLength : leftBorderX;
                        x = x < leftBorderX ? leftBorderX : x
                    }
                }
                else {
                    rad += that.indentFromPie;
                    if (angleFunctions.cos > 0.1)
                        align = "left";
                    else if (angleFunctions.cos < -0.1)
                        align = "right";
                    x = that._centerX + rad * angleFunctions.cos
                }
                y = _round(that._text.settings.y + that._centerY - rad * angleFunctions.sin - bbox.y - bbox.height / 2);
                that._text.applySettings({
                    x: x,
                    y: y,
                    align: align
                })
            },
            _getConnectorPoints: function() {
                var that = this,
                    options = that._options,
                    position = options.position,
                    angleFunctions,
                    rad = that._radiusOuter,
                    box,
                    x,
                    y,
                    points = [];
                if (position !== "inside") {
                    angleFunctions = getCosAndSin(_round(that._middleAngle));
                    points.push({
                        x: _round(that._centerX + (rad - that._borderWidth) * angleFunctions.cos),
                        y: _round(that._centerY - (rad - that._borderWidth) * angleFunctions.sin)
                    });
                    x = _round(that._centerX + (rad + options.radialOffset + CONNECTOR_LENGTH) * angleFunctions.cos);
                    if (position === "columns") {
                        box = that._insideGroup.getBBox();
                        box.x = box.x + (that._insideGroup.settings.translateX || 0);
                        box.y = box.y + (that._insideGroup.settings.translateY || 0);
                        y = box.y + box.height / 2;
                        points.push({
                            x: x,
                            y: y
                        });
                        if (that._background)
                            x = box.x + box.width / 2;
                        else if (angleFunctions.cos < 0)
                            x = box.x + box.width;
                        else if (angleFunctions.cos > 0)
                            x = box.x;
                        points.push({
                            x: x,
                            y: y
                        })
                    }
                    else {
                        y = _round(that._centerY - (rad + options.radialOffset + CONNECTOR_LENGTH) * angleFunctions.sin);
                        points.push({
                            x: x,
                            y: y
                        })
                    }
                }
                return points
            },
            _rotateLabel: function() {
                var that = this,
                    options = that._options,
                    shift = that._radiusOuter + options.radialOffset * 2 + CONNECTOR_LENGTH,
                    angleFunctions = getCosAndSin(that._middleAngle),
                    x,
                    y,
                    box = that._insideGroup.getBBox();
                if (options.position === "inside" || options.position === "columns") {
                    x = box.x + box.width / 2;
                    y = box.y + box.height / 2
                }
                else {
                    x = that._centerX + shift * angleFunctions.cos;
                    y = that._centerY - shift * angleFunctions.sin
                }
                that._insideGroup.applySettings({
                    x: x,
                    y: y,
                    rotate: options.rotationAngle
                })
            },
            _checkEllipsis: function() {
                var that = this,
                    i,
                    LABEL_OFFSET = 10,
                    labelBox,
                    text,
                    textLength = 0,
                    linesLength = [],
                    numLastSpan = [],
                    maxLength,
                    numSpan,
                    index,
                    x,
                    y,
                    width,
                    rotationAngleFunction = getCosAndSin(that._options.rotationAngle),
                    canvas = that._canvas,
                    labelOptions = that._options,
                    angleFunctions = getCosAndSin(that._middleAngle),
                    borderX = that._centerX + (that._radiusOuter + CONNECTOR_LENGTH) * angleFunctions.cos;
                if (!that._text.tspans || !that.setLabelEllipsis)
                    return;
                labelBox = that._text.getBBox();
                x = labelBox.x + labelBox.width < that._centerX ? labelBox.x + labelBox.width : labelBox.x;
                y = labelBox.y + labelBox.height / 2;
                width = labelBox.x + labelBox.width < that._centerX ? -labelBox.width : labelBox.width;
                if (y + width * rotationAngleFunction.sin > canvas.height - canvas.bottom + LABEL_OFFSET || y + width * rotationAngleFunction.sin < canvas.top - LABEL_OFFSET || x + width * rotationAngleFunction.cos < canvas.left - LABEL_OFFSET || x + width * rotationAngleFunction.cos > canvas.width - canvas.right + LABEL_OFFSET || labelOptions.position === "columns" && (angleFunctions.cos < 0 && borderX < x || angleFunctions.cos > 0 && borderX > x))
                    for (i = 0; i < that._text.tspans.length; i++) {
                        textLength += that._text.tspans[i].element.getNumberOfChars();
                        if (!that._text.tspans[i + 1] || that._text.tspans[i + 1].settings.dy > 0) {
                            linesLength.push(textLength);
                            numLastSpan.push(i);
                            textLength = 0
                        }
                    }
                while (y + width * rotationAngleFunction.sin > canvas.height - canvas.bottom + LABEL_OFFSET || y + width * rotationAngleFunction.sin < canvas.top - LABEL_OFFSET || x + width * rotationAngleFunction.cos < canvas.left - LABEL_OFFSET || x + width * rotationAngleFunction.cos > canvas.width - canvas.right + LABEL_OFFSET || labelOptions.position === "columns" && (angleFunctions.cos < 0 && borderX < x || angleFunctions.cos > 0 && borderX > x)) {
                    maxLength = _max.apply(null, linesLength);
                    if (maxLength === 0)
                        break;
                    index = _inArray(maxLength, linesLength);
                    numSpan = numLastSpan[index];
                    if (that._text.tspans[numSpan].element.textContent === "...") {
                        if (that._text.tspans[numSpan].settings.dy > 0 || !that._text.tspans[numSpan - 1])
                            linesLength[index] = 0;
                        else if (that._text.tspans[numSpan - 1]) {
                            that._text.tspans[numSpan].element.textContent = "";
                            numLastSpan[index] -= 1;
                            that._text.tspans[numSpan - 1].element.textContent += "..."
                        }
                    }
                    else {
                        text = that._text.tspans[numSpan].element.textContent;
                        text = text.substr(0, text.length - 1 - 3) + "...";
                        that._text.tspans[numSpan].element.textContent = text;
                        linesLength[index] -= 1
                    }
                    labelBox = that._text.getBBox();
                    x = labelBox.x + labelBox.width < that._centerX ? labelBox.x + labelBox.width : labelBox.x;
                    y = labelBox.y + labelBox.height / 2;
                    width = labelBox.x + labelBox.width < that._centerX ? -labelBox.width : labelBox.width
                }
            },
            _checkPosition: function() {
                var that = this,
                    group = that._insideGroup,
                    box = group.getBBox(),
                    canvas = that._canvas,
                    x = 0,
                    y = 0;
                if (box.y + box.height > canvas.height - canvas.bottom)
                    y = canvas.height - box.y - box.height - canvas.bottom;
                else if (box.y < canvas.top)
                    y = canvas.top - box.y;
                if (box.x + box.width > canvas.width - canvas.right)
                    x = canvas.width - canvas.right - box.x - box.width;
                else if (box.x < canvas.left)
                    x = canvas.left - box.x;
                group.applySettings({
                    translateX: x,
                    translateY: y
                })
            },
            setPosition: function(position) {
                this._options.position = position
            },
            getPosition: function() {
                return this._options.position
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file symbolPoint.js */
    (function($, DX) {
        var viz = DevExpress.viz,
            core = viz.core,
            series = core.series,
            _extend = $.extend,
            _each = $.each,
            _isNumber = DX.utils.isNumber,
            _isDefined = DX.utils.isDefined,
            _isEmptyObject = $.isEmptyObject,
            _math = Math,
            _round = _math.round,
            _floor = _math.floor,
            _ceil = _math.ceil,
            DEFAULT_IMAGE_WIDTH = 20,
            DEFAULT_IMAGE_HEIGHT = 20,
            CANVAS_POSITION_DEFAULT = "canvas_position_default";
        series.points.mixins = series.points.mixins || {};
        series.points.mixins.symbolPoint = {
            deleteLabel: function() {
                this._label.dispose();
                this._label = null
            },
            _hasGraphic: function() {
                return this.graphic
            },
            _clearTrackerVisibility: function() {
                if (this.trackerGraphic && this.trackerGraphic.settings.visibility)
                    this.trackerGraphic.applySettings({visibility: null})
            },
            clearVisibility: function() {
                if (this.graphic && this.graphic.settings.visibility)
                    this.graphic.applySettings({visibility: null});
                this._clearTrackerVisibility();
                this._label.clearVisibility()
            },
            isVisible: function() {
                return this.inVisibleArea && this.series.isVisible()
            },
            _setTrackerInvisibility: function() {
                if (this.trackerGraphic && this.trackerGraphic.settings.visibility !== "hidden")
                    this.trackerGraphic.applySettings({visibility: "hidden"})
            },
            setInvisibility: function() {
                if (this.graphic && this.graphic.settings.visibility !== "hidden")
                    this.graphic.applySettings({visibility: "hidden"});
                this._setTrackerInvisibility();
                this._label.hide()
            },
            clearMarker: function() {
                this.graphic && this.graphic.applySettings(this._emptySettings)
            },
            setAdjustSeriesLabels: function(adjustSeriesLabels) {
                this._label.adjustSeriesLabels = adjustSeriesLabels
            },
            _getLabelOptions: function(type) {
                var options = this._options;
                return {
                        options: options.label,
                        rotated: options.rotated,
                        type: type,
                        isFullStacked: options.series.isFullStackedSeries(),
                        isRange: options.type.indexOf("range") !== -1
                    }
            },
            _createLabel: function() {
                this._label = core.CoreFactory.createLabel(this._options.type)
            },
            _updateLabelData: function() {
                this._label.updateData({
                    formatObject: this._getLabelFormatObject(),
                    initialValue: this.initialValue
                })
            },
            _updateLabelOptions: function(type) {
                !this._label && this._createLabel();
                this._label.setOptions(this._getLabelOptions(type))
            },
            _checkImage: function(image) {
                return _isDefined(image) && (typeof image === "string" || _isDefined(image.url))
            },
            _fillStyle: function() {
                this._styles = this._options.styles
            },
            _checkSymbol: function(oldOptions, newOptions) {
                var oldSymbol = oldOptions.symbol,
                    newSymbol = newOptions.symbol,
                    symbolChanged = oldSymbol === "circle" && newSymbol !== "circle" || oldSymbol !== "circle" && newSymbol === "circle",
                    imageChanged = this._checkImage(oldOptions.image) !== this._checkImage(newOptions.image);
                if (symbolChanged || imageChanged)
                    return true;
                return false
            },
            _getSquareMarkerCoords: function(radius) {
                return [{
                            x: -radius,
                            y: -radius
                        }, {
                            x: radius,
                            y: -radius
                        }, {
                            x: radius,
                            y: radius
                        }, {
                            x: -radius,
                            y: radius
                        }, {
                            x: -radius,
                            y: -radius
                        }]
            },
            _getPolygonMarkerCoords: function(radius) {
                var r = _ceil(radius);
                return [{
                            x: -r,
                            y: 0
                        }, {
                            x: 0,
                            y: -r
                        }, {
                            x: r,
                            y: 0
                        }, {
                            x: 0,
                            y: r
                        }, {
                            x: -r,
                            y: 0
                        }]
            },
            _getTriangleMarkerCoords: function(radius) {
                return [{
                            x: -radius,
                            y: -radius
                        }, {
                            x: radius,
                            y: -radius
                        }, {
                            x: 0,
                            y: radius
                        }, {
                            x: -radius,
                            y: -radius
                        }]
            },
            _getCrossMarkerCoords: function(radius) {
                var r = _ceil(radius),
                    floorHalfRadius = _floor(r / 2),
                    ceilHalfRadius = _ceil(r / 2);
                return [{
                            x: -r,
                            y: -floorHalfRadius
                        }, {
                            x: -floorHalfRadius,
                            y: -r
                        }, {
                            x: 0,
                            y: -ceilHalfRadius
                        }, {
                            x: floorHalfRadius,
                            y: -r
                        }, {
                            x: r,
                            y: -floorHalfRadius
                        }, {
                            x: ceilHalfRadius,
                            y: 0
                        }, {
                            x: r,
                            y: floorHalfRadius
                        }, {
                            x: floorHalfRadius,
                            y: r
                        }, {
                            x: 0,
                            y: ceilHalfRadius
                        }, {
                            x: -floorHalfRadius,
                            y: r
                        }, {
                            x: -r,
                            y: floorHalfRadius
                        }, {
                            x: -ceilHalfRadius,
                            y: 0
                        }]
            },
            _populatePointShape: function(symbol, radius) {
                switch (symbol) {
                    case"square":
                        return this._getSquareMarkerCoords(radius);
                    case"polygon":
                        return this._getPolygonMarkerCoords(radius);
                    case"triangle":
                        return this._getTriangleMarkerCoords(radius);
                    case"cross":
                        return this._getCrossMarkerCoords(radius)
                }
            },
            correctValue: function(correction) {
                if (this.hasValue()) {
                    this.value = this.initialValue + correction;
                    this.minValue = correction;
                    this.translate()
                }
            },
            resetCorrection: function() {
                this.value = this.initialValue;
                this.minValue = CANVAS_POSITION_DEFAULT
            },
            _getTranslates: function(animationEnabled) {
                var translateX,
                    translateY;
                translateX = this.x;
                translateY = this.y;
                if (animationEnabled)
                    if (this._options.rotated)
                        translateX = this.defaultX;
                    else
                        translateY = this.defaultY;
                return {
                        x: translateX,
                        y: translateY
                    }
            },
            _createImageMarker: function(renderer, settings, options) {
                var url,
                    width,
                    height,
                    offsetY,
                    attr = {
                        location: "center",
                        translateX: settings.attr.translateX,
                        translateY: settings.attr.translateY,
                        visibility: settings.attr.visibility
                    };
                url = options.url ? options.url.toString() : options.toString();
                width = options.width || DEFAULT_IMAGE_WIDTH;
                height = options.height || DEFAULT_IMAGE_HEIGHT;
                return renderer.createImage(-_round(width * 0.5), -_round(height * 0.5), width, height, url, attr)
            },
            _createSymbolMarker: function(renderer, pointSettings, animationEnabled) {
                var marker,
                    options = this._options;
                switch (options.symbol) {
                    case"circle":
                        marker = renderer.createCircle(0, 0, pointSettings.r, pointSettings.attr);
                        break;
                    case"square":
                    case"polygon":
                    case"triangle":
                    case"cross":
                        marker = renderer.createArea(pointSettings.points, pointSettings.attr);
                        break
                }
                return marker
            },
            _createMarker: function(renderer, group, image, settings, animationEnabled) {
                var marker = this._checkImage(image) ? this._createImageMarker(renderer, settings, image) : this._createSymbolMarker(renderer, settings, animationEnabled);
                marker && marker.append(group);
                return marker
            },
            _getSymbolBbox: function(x, y, r) {
                var bbox = {};
                bbox.x = x - r;
                bbox.y = y - r;
                bbox.width = bbox.height = r * 2;
                return bbox
            },
            _getImageBbox: function(x, y) {
                var bbox = {},
                    image = this._options.image,
                    width = image.width || DEFAULT_IMAGE_WIDTH,
                    height = image.height || DEFAULT_IMAGE_HEIGHT;
                bbox.x = x - _round(width / 2);
                bbox.y = y - _round(height / 2);
                bbox.width = width;
                bbox.height = height;
                return bbox
            },
            _getGraphicBbox: function() {
                var options = this._options,
                    x = this.x,
                    y = this.y,
                    bbox;
                if (options.visible)
                    bbox = this._checkImage(options.image) ? this._getImageBbox(x, y) : this._getSymbolBbox(x, y, options.styles.normal.r);
                else
                    bbox = {
                        x: x,
                        y: y,
                        width: 0,
                        height: 0
                    };
                return bbox
            },
            _drawLabel: function(renderer, group) {
                var that = this,
                    customVisibility = that._getCustomLabelVisibility(),
                    visibleArea,
                    translators = that.translators;
                if ((that._options.series.getLabelVisibility() || customVisibility) && that.hasValue()) {
                    if (translators)
                        visibleArea = {
                            minX: translators.x.getCanvasVisibleArea().min,
                            maxX: translators.x.getCanvasVisibleArea().max,
                            minY: translators.y.getCanvasVisibleArea().min,
                            maxY: translators.y.getCanvasVisibleArea().max
                        };
                    that._label.setCoords({
                        x: that.x,
                        y: that.y
                    }, that._getGraphicBbox(), that._getLabelPosition(), visibleArea);
                    that._label.draw(renderer, group, customVisibility)
                }
                else
                    that._label.hide()
            },
            _drawMarker: function(renderer, group, animationEnabled) {
                var that = this,
                    options = that._options,
                    translates = that._getTranslates(animationEnabled),
                    marker,
                    style = that._getStyle(),
                    pointAttributes = _extend({
                        translateX: translates.x,
                        translateY: translates.y
                    }, style),
                    pointSettings = {
                        attr: pointAttributes,
                        points: that._populatePointShape(options.symbol, style.r),
                        r: style.r
                    };
                marker = that._createMarker(renderer, group, options.image, pointSettings, animationEnabled);
                that.graphic = marker
            },
            _drawTrackerMarker: function(renderer, group) {
                var radius = this._options.trackerR || this.storeTrackerR();
                this.trackerGraphic = renderer.createCircle(this.x, this.y, radius).append(group);
                this.trackerGraphic.data({point: this})
            },
            getTooltipCoords: function() {
                var coords;
                if (this.graphic)
                    coords = {
                        x: this.x,
                        y: this.y,
                        offset: this.graphic.getBBox().height / 2
                    };
                else
                    coords = {
                        x: this.x,
                        y: this.y,
                        offset: 0
                    };
                return coords
            },
            hasValue: function() {
                return this.value !== null && this.minValue !== null
            },
            setPercentValue: function(total) {
                var valuePercent = this.value / total || 0,
                    percent = valuePercent,
                    minValuePercent = this.minValue / total || 0;
                percent -= _isNumber(this.minValue) ? minValuePercent : 0;
                this._label.setDataField("percent", percent);
                this._label.setDataField("total", total);
                if (this._options.series.isFullStackedSeries() && this.hasValue()) {
                    this.value = valuePercent;
                    this.minValue = !minValuePercent ? this.minValue : minValuePercent;
                    this.translate()
                }
            },
            storeTrackerR: function() {
                var navigator = window.navigator,
                    r = this._options.styles.normal.r,
                    minTrackerSize;
                navigator = this.__debug_navigator || navigator;
                this.__debug_browserNavigator = navigator;
                minTrackerSize = "ontouchstart" in window || navigator.msPointerEnabled && navigator.msMaxTouchPoints || navigator.pointerEnabled && navigator.maxTouchPoints ? 20 : 6;
                this._options.trackerR = r < minTrackerSize ? minTrackerSize : r;
                return this._options.trackerR
            },
            _translate: function(translators) {
                var that = this,
                    mainAxis = that._options.rotated ? "x" : "y",
                    upperMainAxis = mainAxis.toUpperCase(),
                    mainTranslator = translators[mainAxis],
                    axis = that._options.rotated ? "y" : "x";
                that[mainAxis] = mainTranslator.translate(that.value);
                that[axis] = translators[axis].translate(that.argument);
                that["min" + upperMainAxis] = mainTranslator.translate(that.minValue);
                that["default" + upperMainAxis] = mainTranslator.translate(CANVAS_POSITION_DEFAULT);
                that._calculateVisibility(that.x, that.y);
                that._prepareStatesOptions()
            },
            _changeData: function(data) {
                this.value = this.initialValue = this.originalValue = data.value;
                this.argument = this.initialArgument = this.originalArgument = data.argument;
                this.minValue = this.initialMinValue = this.originalMinValue = _isDefined(data.minValue) ? data.minValue : CANVAS_POSITION_DEFAULT;
                this.tag = data.tag;
                this.index = data.index
            },
            _updateData: function(data) {
                if (this.argument !== data.argument || this.value !== data.value || this.tag !== data.tag)
                    this._changeData(data)
            },
            _getImageSettings: function(image) {
                return {
                        href: image.url || image.toString(),
                        width: image.width || DEFAULT_IMAGE_WIDTH,
                        height: image.height || DEFAULT_IMAGE_HEIGHT
                    }
            },
            _updateMarker: function(animationEnabled, style) {
                var that = this,
                    options = that._options,
                    style = style || this._getStyle(),
                    settings,
                    image = options.image;
                if (that._checkImage(image))
                    settings = _extend({}, {visibility: style.visibility}, that._getImageSettings(image));
                else
                    settings = _extend({}, style, {points: that._populatePointShape(options.symbol, style.r)});
                if (!animationEnabled) {
                    settings.translateX = that.x;
                    settings.translateY = that.y
                }
                that.graphic.applySettings(settings)
            },
            _updateTracker: function() {
                this.trackerGraphic.applySettings({
                    cx: this.x,
                    cy: this.y,
                    r: this.storeTrackerR()
                })
            },
            _getLabelFormatObject: function() {
                return {
                        argument: this.initialArgument,
                        value: this.initialValue,
                        originalArgument: this.originalArgument,
                        originalValue: this.originalValue,
                        seriesName: this._options.series.name,
                        point: this
                    }
            },
            _getLabelPosition: function() {
                if (this._options.series.isFullStackedSeries() || this.initialValue > 0)
                    return "top";
                else
                    return "bottom"
            },
            _getFormatObject: function(tooltip) {
                var labelFormatObject = this._label.getData(),
                    valueObject = {
                        argumentText: tooltip.formatValue(this.initialArgument, "argument"),
                        valueText: tooltip.formatValue(this.initialValue)
                    },
                    percentObject = _isDefined(labelFormatObject.percent) ? {percentText: tooltip.formatValue(labelFormatObject.percent, "percent")} : {},
                    totalObject = _isDefined(labelFormatObject.total) ? {totalText: tooltip.formatValue(labelFormatObject.total)} : {};
                return _extend({}, labelFormatObject, valueObject, percentObject, totalObject)
            }
        }
    })(jQuery, DevExpress);
    /*! Module viz-core, file barPoint.js */
    (function($, DX) {
        var viz = DevExpress.viz,
            points = viz.core.series.points.mixins,
            _extend = $.extend,
            _math = Math,
            _round = _math.round,
            _abs = _math.abs,
            _min = _math.min,
            CANVAS_POSITION_DEFAULT = "canvas_position_default",
            DEFAULT_BAR_TRACKER_SIZE = 9,
            CORRECTING_BAR_TRACKER_VALUE = 4;
        points.barPoint = _extend({}, points.symbolPoint, {
            correctCoordinates: function(correctOptions) {
                var correction = correctOptions.offset - _round(correctOptions.width / 2),
                    rotated = this._options.rotated,
                    valueSelector = rotated ? "height" : "width",
                    correctionSelector = (rotated ? "y" : "x") + "Correction";
                this[valueSelector] = correctOptions.width;
                this[correctionSelector] = correction
            },
            _getGraphicBbox: function() {
                var bbox = {};
                bbox.x = this.x;
                bbox.y = this.y;
                bbox.width = this.width;
                bbox.height = this.height;
                return bbox
            },
            _getLabelPosition: function() {
                var position,
                    invertX = this.translators.x.getBusinessRange().invert,
                    invertY = this.translators.y.getBusinessRange().invert,
                    isDiscreteValue = this.series.valueAxisType === "discrete",
                    isFullStacked = this.series.isFullStackedSeries(),
                    notVerticalInverted = !isDiscreteValue && (this.initialValue >= 0 && !invertY || this.initialValue < 0 && invertY) || isDiscreteValue && !invertY || isFullStacked,
                    notHorizontalInverted = !isDiscreteValue && (this.initialValue >= 0 && !invertX || this.initialValue < 0 && invertX) || isDiscreteValue && !invertX || isFullStacked;
                if (!this._options.rotated)
                    position = notVerticalInverted ? "top" : "bottom";
                else
                    position = notHorizontalInverted ? "top" : "bottom";
                return position
            },
            _drawLabel: function(renderer, group) {
                var that = this,
                    options = that._options,
                    coords,
                    commonVisibility = options.series.getLabelVisibility(),
                    customVisibility = that._getCustomLabelVisibility(),
                    visibleArea,
                    translators = that.translators;
                if (that.hasValue() && (options.label.showForZeroValues || that.initialValue) && (customVisibility || commonVisibility)) {
                    coords = {
                        x: that.x,
                        y: that.y,
                        defaultX: that.defaultX,
                        defaultY: that.defaultY
                    };
                    if (translators)
                        visibleArea = {
                            minX: translators.x.getCanvasVisibleArea().min,
                            maxX: translators.x.getCanvasVisibleArea().max,
                            minY: translators.y.getCanvasVisibleArea().min,
                            maxY: translators.y.getCanvasVisibleArea().max
                        };
                    that._label.setCoords(coords, that._getGraphicBbox(), that._getLabelPosition(), visibleArea);
                    that._label.draw(renderer, group, customVisibility)
                }
                else
                    that._label.hide()
            },
            _drawMarker: function(renderer, group, animationEnabled) {
                var style = this._getStyle(),
                    x = this.x,
                    y = this.y,
                    width = this.width,
                    height = this.height;
                if (animationEnabled)
                    if (this._options.rotated) {
                        width = 0;
                        x = this.defaultX
                    }
                    else {
                        height = 0;
                        y = this.defaultY
                    }
                this.graphic = renderer.createRect(x, y, width, height, this._options.cornerRadius, style).append(group)
            },
            _getSettingsForTracker: function() {
                var that = this,
                    y = that.y,
                    height = that.height,
                    x = that.x,
                    width = that.width;
                if (that._options.rotated) {
                    if (width === 1) {
                        width = DEFAULT_BAR_TRACKER_SIZE;
                        x -= CORRECTING_BAR_TRACKER_VALUE
                    }
                }
                else if (height === 1) {
                    height = DEFAULT_BAR_TRACKER_SIZE;
                    y -= CORRECTING_BAR_TRACKER_VALUE
                }
                return {
                        x: x,
                        y: y,
                        width: width,
                        height: height
                    }
            },
            _drawTrackerMarker: function(renderer, group) {
                var settings = this._getSettingsForTracker();
                this.trackerGraphic = renderer.createRect(settings.x, settings.y, settings.width, settings.height, this._options.cornerRadius).append(group);
                this.trackerGraphic.data({point: this})
            },
            getGraphicSettings: function() {
                return {
                        x: this.graphic.settings.x || 0,
                        y: this.graphic.settings.y || 0,
                        height: this.graphic.settings.height || 0,
                        width: this.graphic.settings.width || 0
                    }
            },
            getTooltipCoords: function() {
                var x = this.x + this.width / 2,
                    y = this.y + this.height / 2;
                return {
                        x: x,
                        y: y,
                        offset: 0
                    }
            },
            _truncateCoord: function(coord, minBounce, maxBounce) {
                if (coord < minBounce)
                    return minBounce;
                if (coord > maxBounce)
                    return maxBounce;
                return coord
            },
            _translate: function(translators) {
                var that = this,
                    rotated = that._options.rotated,
                    valAxis = rotated ? "x" : "y",
                    argAxis = rotated ? "y" : "x",
                    valIntervalName = rotated ? "width" : "height",
                    argIntervalName = rotated ? "height" : "width",
                    argTranslator = translators[argAxis],
                    valTranslator = translators[valAxis],
                    argVisibleArea = argTranslator.getCanvasVisibleArea(),
                    valVisibleArea = valTranslator.getCanvasVisibleArea(),
                    arg,
                    minArg,
                    val,
                    minVal;
                arg = minArg = argTranslator.translate(that.argument) + (that[argAxis + "Correction"] || 0);
                val = valTranslator.translate(that.value);
                minVal = valTranslator.translate(that.minValue);
                that[valIntervalName] = _abs(val - minVal);
                that._calculateVisibility(rotated ? _min(val, minVal) : _min(arg, minArg), rotated ? _min(arg, minArg) : _min(val, minVal), that.width, that.height);
                val = that._truncateCoord(val, valVisibleArea.min, valVisibleArea.max);
                minVal = that._truncateCoord(minVal, valVisibleArea.min, valVisibleArea.max);
                that[argAxis] = arg;
                that["min" + argAxis.toUpperCase()] = minArg;
                that[valIntervalName] = _abs(val - minVal);
                that[valAxis] = _min(val, minVal) + (that[valAxis + "Correction"] || 0);
                that["min" + valAxis.toUpperCase()] = minVal + (that[valAxis + "Correction"] || 0);
                that["default" + valAxis.toUpperCase()] = valTranslator.translate(CANVAS_POSITION_DEFAULT);
                if (that.inVisibleArea) {
                    if (that[argAxis] < argVisibleArea.min) {
                        that[argIntervalName] = that[argIntervalName] - (argVisibleArea.min - that[argAxis]);
                        that[argAxis] = argVisibleArea.min;
                        that["min" + argAxis.toUpperCase()] = argVisibleArea.min
                    }
                    if (that[argAxis] + that[argIntervalName] > argVisibleArea.max)
                        that[argIntervalName] = argVisibleArea.max - that[argAxis]
                }
            },
            _updateMarker: function(animationEnabled, style) {
                var style = style || this._getStyle(),
                    attributes = _extend({}, style);
                if (!animationEnabled) {
                    attributes.x = this.x;
                    attributes.y = this.y;
                    attributes.width = this.width;
                    attributes.height = this.height
                }
                else
                    attributes.sharpEdges = false;
                this.graphic.applySettings(attributes)
            },
            _updateTracker: function() {
                this.trackerGraphic.applySettings(this._getSettingsForTracker())
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file bubblePoint.js */
    (function($, DX) {
        var viz = DX.viz,
            points = viz.core.series.points.mixins,
            _extend = $.extend,
            MIN_BUBBLE_HEIGHT = 20;
        points.bubblePoint = _extend({}, points.symbolPoint, {
            correctCoordinates: function(diameter) {
                this.bubbleSize = diameter / 2
            },
            _drawMarker: function(renderer, group, animationEnabled) {
                var style = this._getStyle(),
                    attr = _extend({
                        translateX: this.x,
                        translateY: this.y
                    }, style),
                    marker = renderer.createCircle(0, 0, animationEnabled ? 0 : this.bubbleSize, attr).append(group);
                this.graphic = marker
            },
            _drawTrackerMarker: function(renderer, group) {
                this.trackerGraphic = renderer.createCircle(this.x, this.y, this.bubbleSize).append(group);
                this.trackerGraphic.data({point: this})
            },
            getTooltipCoords: function() {
                if (this.graphic) {
                    var height = this.graphic.getBBox().height;
                    return {
                            x: this.x,
                            y: this.y,
                            offset: height < MIN_BUBBLE_HEIGHT ? height / 2 : 0
                        }
                }
            },
            _getLabelFormatObject: function() {
                var symbolMethods = points.symbolPoint,
                    formatObject = symbolMethods._getLabelFormatObject.call(this);
                formatObject.size = this.initialSize;
                return formatObject
            },
            _updateData: function(data) {
                var symbolMethods = points.symbolPoint;
                if (this.argument !== data.argument || this.value !== data.value || this.size !== data.size || this.tag !== data.tag) {
                    symbolMethods._changeData.call(this, data);
                    this.size = this.initialSize = data.size
                }
            },
            _getGraphicBbox: function() {
                return this._getSymbolBbox(this.x, this.y, this.bubbleSize)
            },
            _updateMarker: function(animationEnabled, style) {
                var style = style || this._getStyle();
                if (!animationEnabled)
                    style = $.extend({
                        r: this.bubbleSize,
                        translateX: this.x,
                        translateY: this.y
                    }, style);
                this.graphic.applySettings(style)
            },
            _updateTracker: function() {
                this.trackerGraphic.applySettings({
                    cx: this.x,
                    cy: this.y,
                    r: this.bubbleSize
                })
            },
            _getFormatObject: function(tooltip) {
                var symbolMethods = points.symbolPoint,
                    formatObject = symbolMethods._getFormatObject.call(this, tooltip);
                formatObject.sizeText = tooltip.formatValue(this.initialSize);
                return formatObject
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file piePoint.js */
    (function($, DX) {
        var viz = DevExpress.viz,
            points = viz.core.series.points.mixins,
            _extend = $.extend,
            _round = Math.round,
            _getCosAndSin = DX.utils.getCosAndSin;
        points.piePoint = _extend({}, points.symbolPoint, {
            _populatePointShape: function(target, deltaRadius) {
                var angleFunctions = _getCosAndSin(point.middleAngle);
                target.x = point.centerX - ~~(deltaRadius * angleFunctions.cos);
                target.y = point.centerY + ~~(deltaRadius * angleFunctions.sin);
                target.outerRadius = this.radiusOuter + deltaRadius;
                target.innerRadius = this.radiusInner;
                target.startAngle = this.toAngle;
                target.endAngle = this.fromAngle
            },
            _changeData: function(data) {
                var that = this;
                that.value = that.initialValue = that.originalValue = data.value;
                that.argument = that.initialArgument = that.originalArgument = data.argument;
                that.minValue = that.initialMinValue = that.originalMinValue = DevExpress.utils.isDefined(data.minValue) ? data.minValue : 0;
                that.tag = data.tag;
                that._visible = true;
                that.index = data.index
            },
            animate: function(complete, duration, step) {
                this.graphic.animate({
                    x: this.centerX,
                    y: this.centerY,
                    outerRadius: this.radiusOuter,
                    innerRadius: this.radiusInner,
                    startAngle: this.toAngle,
                    endAngle: this.fromAngle
                }, {
                    partitionDuration: duration,
                    step: step
                }, complete)
            },
            correctPosition: function(correction) {
                this.radiusInner = correction.radiusInner;
                this.radiusOuter = correction.radiusOuter;
                this.centerX = correction.centerX;
                this.centerY = correction.centerY
            },
            correctValue: function(correction, percent, base) {
                this.value = (base || this.initialValue) + correction;
                this.minValue = correction;
                this.percent = percent;
                this._label.setDataField("percent", percent)
            },
            _getLabelOptions: function() {
                var options = this._options,
                    series = options.series,
                    seriesStyle = this._options.styles.normal,
                    borderWidth = series._options.containerBackgroundColor === seriesStyle.stroke ? _round(seriesStyle.strokeWidth / 2) : _round(-seriesStyle.strokeWidth / 2);
                return {
                        options: options.label,
                        borderWidth: borderWidth
                    }
            },
            _updateLabelData: function() {
                this._label.updateData({formatObject: this._getLabelFormatObject()})
            },
            _updateLabelOptions: function() {
                !this._label && this._createLabel();
                this._label.setOptions(this._getLabelOptions())
            },
            _drawLabel: function(renderer, group) {
                var commonVisibility = this._options.series.getLabelVisibility(),
                    customVisibility = this._getCustomLabelVisibility();
                if ((commonVisibility || customVisibility) && this.hasValue()) {
                    this._label.setCoords({
                        middleAngle: this.middleAngle,
                        radiusOuter: this.radiusOuter,
                        centerX: this.centerX,
                        centerY: this.centerY
                    }, this._options.series.canvas);
                    this._label.draw(renderer, group, customVisibility)
                }
                else
                    this._label.hide()
            },
            _drawMarker: function(renderer, group, animationEnabled, firstDrawing) {
                var styles = this._getStyle(),
                    radiusOuter = this.radiusOuter,
                    radiusInner = this.radiusInner,
                    fromAngle = this.fromAngle,
                    toAngle = this.toAngle;
                if (animationEnabled) {
                    radiusInner = radiusOuter = 0;
                    if (!firstDrawing)
                        fromAngle = toAngle = this.shiftedAngle
                }
                this.graphic = renderer.createArc(this.centerX, this.centerY, radiusOuter, radiusInner, toAngle, fromAngle, styles).append(group)
            },
            _drawTrackerMarker: function(renderer, group) {
                this.trackerGraphic = renderer.createArc(this.centerX, this.centerY, this.radiusOuter, this.radiusInner, this.toAngle, this.fromAngle).append(group);
                this.trackerGraphic.data({point: this})
            },
            getTooltipCoords: function() {
                var angleFunctions = _getCosAndSin(this.middleAngle);
                return {
                        x: this.centerX + (this.radiusInner + (this.radiusOuter - this.radiusInner) / 2) * angleFunctions.cos,
                        y: this.centerY - (this.radiusInner + (this.radiusOuter - this.radiusInner) / 2) * angleFunctions.sin,
                        offset: 0
                    }
            },
            _translate: function(translator) {
                var angle = this.shiftedAngle || 0;
                this.fromAngle = translator.translate(this.minValue) + angle;
                this.toAngle = translator.translate(this.value) + angle;
                this.middleAngle = translator.translate((this.value - this.minValue) / 2 + this.minValue) + angle;
                if (!this.isVisible())
                    this.middleAngle = this.toAngle = this.fromAngle = this.fromAngle || angle
            },
            _updateMarker: function(animationEnabled, style) {
                style = style || this._getStyle();
                if (!animationEnabled)
                    style = _extend({
                        x: this.centerX,
                        y: this.centerY,
                        outerRadius: this.radiusOuter,
                        innerRadius: this.radiusInner,
                        startAngle: this.toAngle,
                        endAngle: this.fromAngle
                    }, style);
                this.graphic.applySettings(style)
            },
            _updateTracker: function() {
                this.trackerGraphic.applySettings({
                    x: this.centerX,
                    y: this.centerY,
                    outerRadius: this.radiusOuter,
                    innerRadius: this.radiusInner,
                    startAngle: this.toAngle,
                    endAngle: this.fromAngle
                })
            },
            getLegendStyles: function() {
                return this._styles.legendStyles
            },
            isInVisibleArea: function() {
                return true
            },
            hide: function() {
                if (this._visible) {
                    this._visible = false;
                    this.hideTooltip();
                    this._options.visibilityChanged(this)
                }
            },
            show: function() {
                if (!this._visible) {
                    this._visible = true;
                    this._options.visibilityChanged(this)
                }
            },
            setInvisibility: function() {
                this._setTrackerInvisibility();
                this._label.hide()
            },
            isVisible: function() {
                return this._visible
            },
            _getFormatObject: function(tooltip) {
                var symbolMethods = points.symbolPoint,
                    formatObject = symbolMethods._getFormatObject.call(this, tooltip);
                formatObject.percent = this.percent;
                formatObject.percentText = tooltip.formatValue(this.percent, "percent");
                return formatObject
            },
            getColor: function() {
                return this._styles.normal.fill
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file rangeSymbolPoint.js */
    (function($, DX) {
        var viz = DevExpress.viz,
            points = viz.core.series.points.mixins,
            _extend = $.extend,
            _isDefined = DX.utils.isDefined,
            _math = Math,
            _abs = _math.abs,
            _min = _math.min,
            _max = _math.max,
            _round = _math.round,
            DEFAULT_IMAGE_WIDTH = 20,
            DEFAULT_IMAGE_HEIGHT = 20;
        points.rangeSymbolPoint = _extend({}, points.symbolPoint, {
            deleteLabel: function() {
                this._topLabel.dispose();
                this._topLabel = null;
                this._bottomLabel.dispose();
                this._bottomLabel = null
            },
            hideMarker: function(type) {
                var marker = this.graphic && this.graphic[type + "Marker"],
                    label = this["_" + type + "Label"];
                if (marker && marker.settings.visibility !== "hidden")
                    marker.applySettings({visibility: "hidden"});
                label.hide()
            },
            setInvisibility: function() {
                this.hideMarker("top");
                this.hideMarker("bottom");
                this._setTrackerInvisibility()
            },
            clearVisibility: function() {
                var graphic = this.graphic;
                if (graphic) {
                    if (graphic.topMarker && graphic.topMarker.settings.visibility)
                        graphic.topMarker.applySettings({visibility: null});
                    if (graphic.bottomMarker && graphic.bottomMarker.settings.visibility)
                        graphic.bottomMarker.applySettings({visibility: null})
                }
                this._clearTrackerVisibility();
                this._topLabel.clearVisibility();
                this._bottomLabel.clearVisibility()
            },
            clearMarker: function() {
                if (this.graphic) {
                    this.graphic.topMarker && this.graphic.topMarker.applySettings(this._emptySettings);
                    this.graphic.bottomMarker && this.graphic.bottomMarker.applySettings(this._emptySettings)
                }
            },
            _getLabelPosition: function(markerType) {
                var position,
                    rotated = this._options.rotated,
                    invertY = this.translators.y.getBusinessRange().invert,
                    invertX = this.translators.x.getBusinessRange().invert,
                    isDiscreteValue = this._options.series._options.valueAxisType === "discrete",
                    notInverted = isDiscreteValue && (!invertY && !rotated || invertX && rotated) || !isDiscreteValue && this.value > this.minValue && (!invertY && !rotated || !invertX && rotated);
                if (markerType === "top")
                    position = notInverted ? "top" : "bottom";
                else
                    position = notInverted ? "bottom" : "top";
                return position
            },
            _getLabelMinFormatObject: function() {
                return {
                        index: 0,
                        argument: this.initialArgument,
                        value: this.initialMinValue,
                        seriesName: this._options.series.name,
                        originalValue: this.originalMinValue,
                        originalArgument: this.originalArgument,
                        point: this
                    }
            },
            _updateLabelData: function() {
                var maxFormatObject = this._getLabelFormatObject();
                maxFormatObject.index = 1;
                this._topLabel.updateData({
                    formatObject: maxFormatObject,
                    initialValue: this.initialValue
                });
                this._bottomLabel.updateData({
                    formatObject: this._getLabelMinFormatObject(),
                    initialValue: this.initialMinValue
                })
            },
            _updateLabelOptions: function(type) {
                var options = this._getLabelOptions(type);
                (!this._topLabel || !this._bottomLabel) && this._createLabel();
                this._topLabel.setOptions(options);
                this._bottomLabel.setOptions(options)
            },
            setAdjustSeriesLabels: function(adjustSeriesLabels) {
                this._topLabel.adjustSeriesLabels = adjustSeriesLabels;
                this._bottomLabel.adjustSeriesLabels = adjustSeriesLabels
            },
            _createLabel: function() {
                this._topLabel = viz.core.CoreFactory.createLabel();
                this._bottomLabel = viz.core.CoreFactory.createLabel()
            },
            _getLabelCoords: function(location) {
                var coords = {},
                    isTop = location === "top";
                if (!this._options.rotated) {
                    coords.x = this.x;
                    coords.y = isTop ? _min(this.y, this.minY) : _max(this.y, this.minY)
                }
                else {
                    coords.x = isTop ? _max(this.x, this.minX) : _min(this.x, this.minX);
                    coords.y = this.y
                }
                return coords
            },
            _getGraphicBbox: function(location) {
                var options = this._options,
                    rotated = options.rotated,
                    isTop = location === "top",
                    images = this._getImage(options.image),
                    image = isTop ? this._checkImage(images.top) : this._checkImage(images.bottom),
                    bbox,
                    x,
                    y;
                if (!rotated) {
                    x = this.x;
                    y = isTop ? _min(this.y, this.minY) : _max(this.y, this.minY)
                }
                else {
                    x = isTop ? _max(this.x, this.minX) : _min(this.x, this.minX);
                    y = this.y
                }
                if (options.visible)
                    bbox = image ? this._getImageBbox(x, y) : this._getSymbolBbox(x, y, options.styles.normal.r);
                else
                    bbox = {
                        x: x,
                        y: y,
                        width: 0,
                        height: 0
                    };
                return bbox
            },
            _checkOverlay: function(bottomCoord, topCoord, topValue) {
                return bottomCoord < topCoord + topValue
            },
            _getOverlayCorrections: function(type, topCoords, bottomCoords) {
                var isVertical = type === "vertical",
                    coordSelector = isVertical ? "y" : "x",
                    valueSelector = isVertical ? "height" : "width",
                    visibleArea = this.translators[coordSelector].getCanvasVisibleArea(),
                    minBound = visibleArea.min,
                    maxBound = visibleArea.max,
                    delta = _round((topCoords[coordSelector] + topCoords[valueSelector] - bottomCoords[coordSelector]) / 2),
                    coord1 = topCoords[coordSelector] - delta,
                    coord2 = bottomCoords[coordSelector] + delta;
                if (coord1 < minBound) {
                    delta = minBound - topCoords[coordSelector];
                    coord1 += delta;
                    coord2 += delta
                }
                else if (coord2 + bottomCoords[valueSelector] > maxBound) {
                    delta = -(bottomCoords[coordSelector] + bottomCoords[valueSelector] - maxBound);
                    coord1 += delta;
                    coord2 += delta
                }
                return {
                        coord1: coord1,
                        coord2: coord2
                    }
            },
            _checkLabelsOverlay: function(topLocation) {
                var that = this,
                    topCoords = that._topLabel.getCoords(),
                    bottomCoords = that._bottomLabel.getCoords(),
                    corrections = {};
                if (!that._options.rotated) {
                    if (topLocation === "top") {
                        if (this._checkOverlay(bottomCoords.y, topCoords.y, topCoords.height)) {
                            corrections = this._getOverlayCorrections("vertical", topCoords, bottomCoords);
                            that._topLabel.updatePosition(undefined, corrections.coord1);
                            that._bottomLabel.updatePosition(undefined, corrections.coord2)
                        }
                    }
                    else if (this._checkOverlay(topCoords.y, bottomCoords.y, bottomCoords.height)) {
                        corrections = this._getOverlayCorrections("vertical", bottomCoords, topCoords);
                        that._topLabel.updatePosition(undefined, corrections.coord2);
                        that._bottomLabel.updatePosition(undefined, corrections.coord1)
                    }
                }
                else if (topLocation === "top") {
                    if (this._checkOverlay(topCoords.x, bottomCoords.x, bottomCoords.width)) {
                        corrections = this._getOverlayCorrections("horizontal", bottomCoords, topCoords);
                        that._topLabel.updatePosition(corrections.coord2);
                        that._bottomLabel.updatePosition(corrections.coord1)
                    }
                }
                else if (this._checkOverlay(bottomCoords.x, topCoords.x, topCoords.width)) {
                    corrections = this._getOverlayCorrections("horizontal", topCoords, bottomCoords);
                    that._topLabel.updatePosition(corrections.coord1);
                    that._bottomLabel.updatePosition(corrections.coord2)
                }
            },
            _drawLabel: function(renderer, group) {
                var topCoords,
                    bottomCoords,
                    rotated = this._options.rotated,
                    topLocation = this._getLabelPosition("top"),
                    bottomLocation = this._getLabelPosition("bottom"),
                    isInside = this._options.label.position === "inside",
                    topPosition = isInside ? bottomLocation : topLocation,
                    bottomPosition = isInside ? topLocation : bottomLocation,
                    translators = this.translators,
                    visibleArea,
                    customVisibility = this._getCustomLabelVisibility(),
                    commonVisibility = this._options.series.getLabelVisibility();
                if ((commonVisibility || customVisibility) && this.hasValue()) {
                    if (translators)
                        visibleArea = {
                            minX: translators.x.getCanvasVisibleArea().min,
                            maxX: translators.x.getCanvasVisibleArea().max,
                            minY: translators.y.getCanvasVisibleArea().min,
                            maxY: translators.y.getCanvasVisibleArea().max
                        };
                    this._topLabel.setCoords(this._getLabelCoords(topLocation), this._getGraphicBbox(topLocation), topPosition, visibleArea);
                    this._bottomLabel.setCoords(this._getLabelCoords(bottomLocation), this._getGraphicBbox(bottomLocation), bottomPosition, visibleArea);
                    this.visibleTopMarker && this._topLabel.draw(renderer, group, customVisibility);
                    this.visibleBottomMarker && this._bottomLabel.draw(renderer, group, customVisibility);
                    this._checkLabelsOverlay(topLocation)
                }
                else {
                    this._topLabel.hide();
                    this._bottomLabel.hide()
                }
            },
            _getImage: function(imageOption) {
                var image = {};
                if (_isDefined(imageOption))
                    if (typeof imageOption === "string")
                        image.top = image.bottom = imageOption;
                    else {
                        image.top = {
                            url: typeof imageOption.url === "string" ? imageOption.url : imageOption.url && imageOption.url.rangeMaxPoint,
                            width: typeof imageOption.width === "number" ? imageOption.width : imageOption.width && imageOption.width.rangeMaxPoint,
                            height: typeof imageOption.height === "number" ? imageOption.height : imageOption.height && imageOption.height.rangeMaxPoint
                        };
                        image.bottom = {
                            url: typeof imageOption.url === "string" ? imageOption.url : imageOption.url && imageOption.url.rangeMinPoint,
                            width: typeof imageOption.width === "number" ? imageOption.width : imageOption.width && imageOption.width.rangeMinPoint,
                            height: typeof imageOption.height === "number" ? imageOption.height : imageOption.height && imageOption.height.rangeMinPoint
                        }
                    }
                return image
            },
            _checkSymbol: function(oldOptions, newOptions) {
                var oldSymbol = oldOptions.symbol,
                    newSymbol = newOptions.symbol,
                    symbolChanged = oldSymbol === "circle" && newSymbol !== "circle" || oldSymbol !== "circle" && newSymbol === "circle",
                    oldImages = this._getImage(oldOptions.image),
                    newImages = this._getImage(newOptions.image),
                    topImageChanged = this._checkImage(oldImages.top) !== this._checkImage(newImages.top),
                    bottomImageChanged = this._checkImage(oldImages.bottom) !== this._checkImage(newImages.bottom);
                return symbolChanged || topImageChanged || bottomImageChanged
            },
            _getSettingsForTwoMarkers: function(style) {
                var that = this,
                    options = that._options,
                    settings = {},
                    x = options.rotated ? _min(that.x, that.minX) : that.x,
                    y = options.rotated ? that.y : _min(that.y, that.minY),
                    radius = style.r,
                    points = that._populatePointShape(options.symbol, radius);
                settings.top = {
                    r: radius,
                    points: points,
                    attr: _extend({
                        translateX: x + that.width,
                        translateY: y
                    }, style)
                };
                settings.bottom = {
                    r: radius,
                    points: points,
                    attr: _extend({
                        translateX: x,
                        translateY: y + that.height
                    }, style)
                };
                return settings
            },
            _hasGraphic: function() {
                return this.graphic && this.graphic.topMarker && this.graphic.bottomMarker
            },
            _drawOneMarker: function(renderer, markerType, imageSettings, settings) {
                if (this.graphic[markerType])
                    this._updateOneMarker(markerType, settings);
                else
                    this.graphic[markerType] = this._createMarker(renderer, this.graphic, imageSettings, settings)
            },
            _drawMarker: function(renderer, group, animationEnabled, style) {
                var that = this,
                    style = style || that._getStyle(),
                    settings = that._getSettingsForTwoMarkers(style),
                    image = that._getImage(that._options.image);
                if (that._checkImage(image.top))
                    settings.top = that._getImageSettings(settings.top, image.top);
                if (that._checkImage(image.bottom))
                    settings.bottom = that._getImageSettings(settings.bottom, image.bottom);
                that.graphic = that.graphic || renderer.createGroup().append(group);
                that.visibleTopMarker && that._drawOneMarker(renderer, 'topMarker', image.top, settings.top);
                that.visibleBottomMarker && that._drawOneMarker(renderer, 'bottomMarker', image.bottom, settings.bottom)
            },
            _getSettingsForTracker: function(radius) {
                var that = this,
                    options = that._options,
                    x = options.rotated ? _min(that.x, that.minX) - radius : that.x - radius,
                    y = options.rotated ? that.y - radius : _min(that.y, that.minY) - radius,
                    width = that.width + 2 * radius,
                    height = that.height + 2 * radius;
                return {
                        translateX: x,
                        translateY: y,
                        width: width,
                        height: height
                    }
            },
            _drawTrackerMarker: function(renderer, group) {
                var that = this,
                    options = that._options,
                    radius = options.trackerR || that.storeTrackerR(),
                    settings = that._getSettingsForTracker(radius),
                    attr = {
                        translateX: settings.translateX,
                        translateY: settings.translateY
                    };
                that.trackerGraphic = renderer.createRect(0, 0, settings.width, settings.height, 0, attr).append(group);
                that.trackerGraphic.data({point: that})
            },
            isInVisibleArea: function() {
                var that = this,
                    minArgument = _min(that.minX, that.x) || that.x,
                    maxArgument = _max(that.minX, that.x) || that.x,
                    maxValue = _max(that.minY, that.y) || that.y,
                    minValue = _min(that.minY, that.y) || that.y,
                    notVisibleBothMarkersRight,
                    notVisibleBothMarkersLeft,
                    notVisibleBothMarkersBottom,
                    notVisibleBothMarkersTop,
                    visibleTopMarker = true,
                    visibleBottomMarker = true,
                    visibleRangeArea = true,
                    visibleAreaX,
                    visibleAreaY;
                if (that.translators) {
                    visibleAreaX = that.translators.x.getCanvasVisibleArea();
                    visibleAreaY = that.translators.y.getCanvasVisibleArea();
                    notVisibleBothMarkersRight = visibleAreaX.max < minArgument && visibleAreaX.max < maxArgument;
                    notVisibleBothMarkersLeft = visibleAreaX.min > minArgument && visibleAreaX.min > maxArgument;
                    notVisibleBothMarkersTop = visibleAreaY.min > minValue && visibleAreaY.min > maxValue;
                    notVisibleBothMarkersBottom = visibleAreaY.max < minValue && visibleAreaY.max < maxValue;
                    if (notVisibleBothMarkersTop || notVisibleBothMarkersBottom || notVisibleBothMarkersRight || notVisibleBothMarkersLeft)
                        visibleTopMarker = visibleBottomMarker = visibleRangeArea = false;
                    else if (!that._options.rotated) {
                        visibleTopMarker = visibleAreaY.min < minValue && visibleAreaY.max > minValue;
                        visibleBottomMarker = visibleAreaY.min < maxValue && visibleAreaY.max > maxValue
                    }
                    else {
                        visibleBottomMarker = visibleAreaX.min < minArgument && visibleAreaX.max > minArgument;
                        visibleTopMarker = visibleAreaX.min < maxArgument && visibleAreaX.max > maxArgument
                    }
                }
                that.visibleTopMarker = visibleTopMarker;
                that.visibleBottomMarker = visibleBottomMarker;
                that.visibleRangeArea = visibleRangeArea;
                return visibleRangeArea
            },
            getTooltipCoords: function() {
                var that = this,
                    x,
                    y,
                    min,
                    max,
                    minValue,
                    visibleAreaX = that.translators.x.getCanvasVisibleArea(),
                    visibleAreaY = that.translators.y.getCanvasVisibleArea();
                if (!that._options.rotated) {
                    minValue = _min(that.y, that.minY);
                    x = that.x;
                    min = visibleAreaY.min > minValue ? visibleAreaY.min : minValue;
                    max = visibleAreaY.max < minValue + that.height ? visibleAreaY.max : minValue + that.height;
                    y = min + (max - min) / 2
                }
                else {
                    minValue = _min(that.x, that.minX);
                    y = that.y;
                    min = visibleAreaX.min > minValue ? visibleAreaX.min : minValue;
                    max = visibleAreaX.max < minValue + that.width ? visibleAreaX.max : minValue + that.width;
                    x = min + (max - min) / 2
                }
                return {
                        x: x,
                        y: y,
                        offset: 0
                    }
            },
            _translate: function(translators) {
                var symbolMethods = points.symbolPoint;
                this.minX = this.minY = translators.y.translate(this.minValue);
                symbolMethods._translate.call(this, translators);
                this.height = this._options.rotated ? 0 : _abs(this.minY - this.y);
                this.width = this._options.rotated ? _abs(this.x - this.minX) : 0
            },
            _updateData: function(data) {
                if (this.argument !== data.argument || this.value !== data.value || this.minValue !== data.minValue || this.tag !== data.tag) {
                    this._changeData(data);
                    this.minValue = this.initialMinValue = this.originalMinValue = data.minValue
                }
            },
            _getImageSettings: function(settings, image) {
                var x = settings.attr.translateX,
                    y = settings.attr.translateY;
                return {
                        href: image.url || image.toString(),
                        width: image.width || DEFAULT_IMAGE_WIDTH,
                        height: image.height || DEFAULT_IMAGE_HEIGHT,
                        attr: {
                            translateX: x,
                            translateY: y
                        }
                    }
            },
            _updateOneMarker: function(markerType, settings) {
                this.graphic && this.graphic[markerType] && this.graphic[markerType].applySettings(_extend({}, settings, settings.attr))
            },
            _updateMarker: function(animationEnabled, style) {
                this._drawMarker(undefined, undefined, undefined, style)
            },
            _updateTracker: function() {
                var radius = this.storeTrackerR(),
                    settings = this._getSettingsForTracker(radius);
                this.trackerGraphic.applySettings(settings)
            },
            _getFormatObject: function(tooltip) {
                var minValue = tooltip.formatValue(this.initialMinValue),
                    value = tooltip.formatValue(this.initialValue);
                return {
                        argument: this.initialArgument,
                        argumentText: tooltip.formatValue(this.initialArgument, "argument"),
                        valueText: minValue + " - " + value,
                        rangeValue1Text: minValue,
                        rangeValue2Text: value,
                        rangeValue1: this.initialMinValue,
                        rangeValue2: this.initialValue,
                        seriesName: this.series.name,
                        point: this,
                        originalMinValue: this.originalMinValue,
                        originalValue: this.originalValue,
                        originalArgument: this.originalArgument
                    }
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file rangeBarPoint.js */
    (function($, DX) {
        var viz = DevExpress.viz,
            points = viz.core.series.points.mixins,
            _isDefined = DX.utils.isDefined,
            rangeSymbolPointMethods = points.rangeSymbolPoint,
            _extend = $.extend;
        points.rangeBarPoint = _extend({}, points.barPoint, {
            deleteLabel: rangeSymbolPointMethods.deleteLabel,
            _getFormatObject: rangeSymbolPointMethods._getFormatObject,
            setAdjustSeriesLabels: rangeSymbolPointMethods.setAdjustSeriesLabels,
            clearVisibility: function() {
                if (this.graphic && this.graphic.settings.visibility)
                    this.graphic.applySettings({visibility: null});
                this._topLabel.clearVisibility();
                this._bottomLabel.clearVisibility()
            },
            setInvisibility: function() {
                if (this.graphic && this.graphic.settings.visibility !== "hidden")
                    this.graphic.applySettings({visibility: "hidden"});
                this._topLabel.hide();
                this._bottomLabel.hide()
            },
            _translate: function(translator) {
                var barMethods = points.barPoint;
                barMethods._translate.call(this, translator);
                if (this._options.rotated)
                    this.width = this.width || 1;
                else
                    this.height = this.height || 1
            },
            _updateData: rangeSymbolPointMethods._updateData,
            _getLabelPosition: rangeSymbolPointMethods._getLabelPosition,
            _getLabelMinFormatObject: rangeSymbolPointMethods._getLabelMinFormatObject,
            _updateLabelData: rangeSymbolPointMethods._updateLabelData,
            _updateLabelOptions: rangeSymbolPointMethods._updateLabelOptions,
            _createLabel: rangeSymbolPointMethods._createLabel,
            _checkOverlay: rangeSymbolPointMethods._checkOverlay,
            _getOverlayCorrections: rangeSymbolPointMethods._getOverlayCorrections,
            _drawLabel: function(renderer, group) {
                var topCoords,
                    bottomCoords,
                    rotated = this._options.rotated,
                    topLocation = this._getLabelPosition("top"),
                    bottomLocation = this._getLabelPosition("bottom"),
                    bbox = this._getGraphicBbox(),
                    coords = {
                        x: this.x,
                        y: this.y,
                        defaultX: this.defaultX,
                        defaultY: this.defaultY
                    },
                    translators = this.translators,
                    visibleArea,
                    customVisibility = this._getCustomLabelVisibility(),
                    commonVisibility = this._options.series.getLabelVisibility();
                if ((customVisibility || commonVisibility) && this.hasValue()) {
                    if (translators)
                        visibleArea = {
                            minX: translators.x.getCanvasVisibleArea().min,
                            maxX: translators.x.getCanvasVisibleArea().max,
                            minY: translators.y.getCanvasVisibleArea().min,
                            maxY: translators.y.getCanvasVisibleArea().max
                        };
                    this._topLabel.setCoords(coords, bbox, topLocation, visibleArea);
                    this._bottomLabel.setCoords(coords, bbox, bottomLocation, visibleArea);
                    this._topLabel.draw(renderer, group, customVisibility);
                    this._bottomLabel.draw(renderer, group, customVisibility);
                    rangeSymbolPointMethods._checkLabelsOverlay.call(this, topLocation)
                }
                else {
                    this._topLabel.hide();
                    this._bottomLabel.hide()
                }
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file candlestickPoint.js */
    (function($, DX) {
        var viz = DevExpress.viz,
            points = viz.core.series.points.mixins,
            _isNumeric = $.isNumeric,
            _extend = $.extend,
            _math = Math,
            _abs = _math.abs,
            _min = _math.min,
            _max = _math.max,
            _round = _math.round,
            DEFAULT_FINANCIAL_TRACKER_MARGIN = 2;
        points.candlestickPoint = _extend({}, points.barPoint, {
            _getContinuousPoints: function(minValueName, maxValueName) {
                var that = this,
                    x = that.x,
                    createPoint = that._options.rotated ? function(x, y) {
                        return {
                                x: y,
                                y: x
                            }
                    } : function(x, y) {
                        return {
                                x: x,
                                y: y
                            }
                    },
                    width = that.width,
                    min = that[minValueName],
                    max = that[maxValueName],
                    points;
                if (min === max)
                    points = [createPoint(x, that.highY), createPoint(x, that.lowY), createPoint(x, that.closeY), createPoint(x - width / 2, that.closeY), createPoint(x + width / 2, that.closeY), createPoint(x, that.closeY)];
                else
                    points = [createPoint(x, that.highY), createPoint(x, max), createPoint(x + width / 2, max), createPoint(x + width / 2, min), createPoint(x, min), createPoint(x, that.lowY), createPoint(x, min), createPoint(x - width / 2, min), createPoint(x - width / 2, max), createPoint(x, max)];
                return points
            },
            _getCategoryPoints: function(y) {
                var that = this,
                    x = that.x,
                    createPoint = that._options.rotated ? function(x, y) {
                        return {
                                x: y,
                                y: x
                            }
                    } : function(x, y) {
                        return {
                                x: x,
                                y: y
                            }
                    };
                return [createPoint(x, that.highY), createPoint(x, that.lowY), createPoint(x, y), createPoint(x - that.width / 2, y), createPoint(x + that.width / 2, y), createPoint(x, y)]
            },
            _getPoints: function() {
                var that = this,
                    points,
                    minValueName,
                    maxValueName,
                    openValue = that.openValue,
                    closeValue = that.closeValue;
                if (_isNumeric(openValue) && _isNumeric(closeValue)) {
                    minValueName = openValue > closeValue ? "closeY" : "openY";
                    maxValueName = openValue > closeValue ? "openY" : "closeY";
                    points = that._getContinuousPoints(minValueName, maxValueName)
                }
                else if (openValue === closeValue)
                    points = [{
                            x: that.x,
                            y: that.highY
                        }, {
                            x: that.x,
                            y: that.lowY
                        }];
                else
                    points = that._getCategoryPoints(_isNumeric(openValue) ? that.openY : that.closeY);
                return points
            },
            getColor: function() {
                return this._isReduction ? this._options.reduction.color : this._styles.normal.stroke || this.series.getColor()
            },
            _drawMarkerInGroup: function(group, attributes, renderer) {
                this.graphic = renderer.createArea(this._getPoints(), attributes).append(group)
            },
            _fillStyle: function() {
                var styles = this._options.styles;
                if (this._isReduction && this._isPositive)
                    this._styles = styles.reductionPositive;
                else if (this._isReduction)
                    this._styles = styles.reduction;
                else if (this._isPositive)
                    this._styles = styles.positive;
                else
                    this._styles = styles
            },
            _getMinTrackerWidth: function() {
                return 1 + 2 * this._styles.normal.strokeWidth
            },
            correctCoordinates: function(correctOptions) {
                var minWidth = this._getMinTrackerWidth(),
                    maxWidth = 10;
                this.width = correctOptions.width < minWidth ? minWidth : correctOptions.width > maxWidth ? maxWidth : correctOptions.width;
                this.xCorrection = correctOptions.offset
            },
            _getMarkerGroup: function(group) {
                var markerGroup;
                if (this._isReduction && this._isPositive)
                    markerGroup = group.reductionPositiveMarkersGroup;
                else if (this._isReduction)
                    markerGroup = group.reductionMarkersGroup;
                else if (this._isPositive)
                    markerGroup = group.defaultPositiveMarkersGroup;
                else
                    markerGroup = group.defaultMarkersGroup;
                return markerGroup
            },
            _drawMarker: function(renderer, group) {
                var attributes = this._getStyle(),
                    rotated = this._options.rotated,
                    pointGroup = this._getMarkerGroup(group);
                this._drawMarkerInGroup(pointGroup, attributes, renderer)
            },
            _getSettingsForTracker: function() {
                var that = this,
                    highY = that.highY,
                    lowY = that.lowY,
                    rotated = that._options.rotated,
                    x,
                    y,
                    width,
                    height;
                if (highY === lowY) {
                    highY = rotated ? highY + DEFAULT_FINANCIAL_TRACKER_MARGIN : highY - DEFAULT_FINANCIAL_TRACKER_MARGIN;
                    lowY = rotated ? lowY - DEFAULT_FINANCIAL_TRACKER_MARGIN : lowY + DEFAULT_FINANCIAL_TRACKER_MARGIN
                }
                if (rotated) {
                    x = _min(lowY, highY);
                    y = that.x - that.width / 2;
                    width = _abs(lowY - highY);
                    height = that.width
                }
                else {
                    x = that.x - that.width / 2;
                    y = _min(lowY, highY);
                    width = that.width;
                    height = _abs(lowY - highY)
                }
                return {
                        x: x,
                        y: y,
                        width: width,
                        height: height
                    }
            },
            _drawTrackerMarker: function(renderer, group) {
                var settings = this._getSettingsForTracker();
                this.trackerGraphic = renderer.createRect(settings.x, settings.y, settings.width, settings.height, 0).append(group);
                this.trackerGraphic.data({point: this})
            },
            _getGraphicBbox: function() {
                var bbox = {},
                    rotated = this._options.rotated;
                bbox.x = !rotated ? this.x - _round(this.width / 2) : this.lowY;
                bbox.y = !rotated ? this.highY : this.x - _round(this.width / 2);
                bbox.width = !rotated ? this.width : this.highY - this.lowY;
                bbox.height = !rotated ? this.lowY - this.highY : this.width;
                return bbox
            },
            _drawLabel: function(renderer, group) {
                var coords,
                    commonVisibility = this._options.series.getLabelVisibility(),
                    customVisibility = this._getCustomLabelVisibility(),
                    visibleArea,
                    reductionColor = this._options.reduction.color,
                    translators = this.translators;
                if (this.hasValue() && (commonVisibility || customVisibility)) {
                    visibleArea = {
                        minX: translators.x.getCanvasVisibleArea().min,
                        maxX: translators.x.getCanvasVisibleArea().max,
                        minY: translators.y.getCanvasVisibleArea().min,
                        maxY: translators.y.getCanvasVisibleArea().max
                    };
                    if (this._isReduction)
                        this._label.updateOptions({options: {
                                background: {fill: reductionColor},
                                connector: {stroke: reductionColor}
                            }});
                    coords = this._options.rotated ? {
                        x: this.highY,
                        y: this.x
                    } : {
                        x: this.x,
                        y: this.highY
                    };
                    this._label.setCoords(coords, this._getGraphicBbox(), "top", visibleArea);
                    this._label.draw(renderer, group, customVisibility)
                }
            },
            getTooltipCoords: function() {
                var that = this;
                if (that.graphic) {
                    var x,
                        y,
                        min,
                        max,
                        minValue = _min(that.lowY, that.highY),
                        maxValue = _max(that.lowY, that.highY),
                        visibleAreaX = that.translators.x.getCanvasVisibleArea(),
                        visibleAreaY = that.translators.y.getCanvasVisibleArea();
                    if (!that._options.rotated) {
                        min = _max(visibleAreaY.min, minValue);
                        max = _min(visibleAreaY.max, maxValue);
                        x = that.x;
                        y = min + (max - min) / 2
                    }
                    else {
                        min = _max(visibleAreaX.min, minValue);
                        max = _min(visibleAreaX.max, maxValue);
                        y = that.x;
                        x = min + (max - min) / 2
                    }
                    return {
                            x: x,
                            y: y,
                            offset: 0
                        }
                }
            },
            hasValue: function() {
                return this.highValue !== null && this.lowValue !== null
            },
            _translate: function(translator) {
                var that = this,
                    rotated = that._options.rotated,
                    argTranslator = rotated ? that.translators.y : that.translators.x,
                    valTranslator = rotated ? that.translators.x : that.translators.y,
                    centerValue,
                    height;
                that.x = argTranslator.translate(that.argument) + (that.xCorrection || 0);
                that.openY = that.openValue !== null ? valTranslator.translate(that.openValue) : null;
                that.highY = valTranslator.translate(that.highValue);
                that.lowY = valTranslator.translate(that.lowValue);
                that.closeY = that.closeValue !== null ? valTranslator.translate(that.closeValue) : null;
                height = _abs(that.lowY - that.highY);
                centerValue = _min(that.lowY, that.highY) + _abs(that.lowY - that.highY) / 2;
                that._calculateVisibility(!rotated ? that.x : centerValue, !rotated ? centerValue : that.x)
            },
            _updateData: function(data) {
                var that = this;
                if (that.argument !== data.argument || that.reductionValue !== data.reductionValue || that.highValue !== data.highValue || that.lowValue !== data.lowValue || that.openValue !== data.openValue || that.closeValue !== data.closeValue || that.tag !== data.tag) {
                    that.value = that.initialValue = data.reductionValue;
                    that.originalValue = data.value;
                    that.argument = that.initialArgument = data.argument;
                    that.originalArgument = data.argument;
                    that.lowValue = data.lowValue;
                    that.originalLowValue = data.lowValue;
                    that.highValue = data.highValue;
                    that.originalHighValue = data.highValue;
                    that.openValue = data.openValue;
                    that.originalOpenValue = data.openValue;
                    that.closeValue = data.closeValue;
                    that.originalCloseValue = data.closeValue;
                    that.tag = data.tag;
                    that.pointClassName = data.pointClassName;
                    that._isPositive = data.openValue < data.closeValue;
                    that._isReduction = data.isReduction
                }
            },
            _updateMarker: function(animationEnabled, style, group) {
                var style = style || this._getStyle();
                this.graphic.applySettings(_extend({points: this._getPoints()}, style));
                group && this.graphic.append(this._getMarkerGroup(group))
            },
            _updateTracker: function() {
                this.trackerGraphic.applySettings(this._getSettingsForTracker())
            },
            _getLabelFormatObject: function() {
                return {
                        openValue: this.openValue,
                        highValue: this.highValue,
                        lowValue: this.lowValue,
                        closeValue: this.closeValue,
                        reductionValue: this.initialValue,
                        argument: this.initialArgument,
                        value: this.initialValue,
                        seriesName: this._options.series.name,
                        originalOpenValue: this.originalOpenValue,
                        originalCloseValue: this.originalCloseValue,
                        originalLowValue: this.originalLowValue,
                        originalHighValue: this.originalHighValue,
                        originalArgument: this.originalArgument,
                        point: this
                    }
            },
            _getFormatObject: function(tooltip) {
                var highValue = tooltip.formatValue(this.highValue),
                    openValue = tooltip.formatValue(this.openValue),
                    closeValue = tooltip.formatValue(this.closeValue),
                    lowValue = tooltip.formatValue(this.lowValue),
                    symbolMethods = points.symbolPoint,
                    formatObject = symbolMethods._getFormatObject.call(this, tooltip);
                return _extend({}, formatObject, {
                        valueText: "h: " + highValue + (openValue !== "" ? " o: " + openValue : "") + (closeValue !== "" ? " c: " + closeValue : "") + " l: " + lowValue,
                        highValueText: highValue,
                        openValueText: openValue,
                        closeValueText: closeValue,
                        lowValueText: lowValue
                    })
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file stockPoint.js */
    (function($, DX) {
        var viz = DevExpress.viz,
            points = viz.core.series.points.mixins,
            _extend = $.extend,
            _isNumeric = $.isNumeric;
        points.stockPoint = _extend({}, points.candlestickPoint, {
            _getPoints: function() {
                var that = this,
                    createPoint = that._options.rotated ? function(x, y) {
                        return {
                                x: y,
                                y: x
                            }
                    } : function(x, y) {
                        return {
                                x: x,
                                y: y
                            }
                    },
                    openYExist = _isNumeric(that.openY),
                    closeYExist = _isNumeric(that.closeY),
                    x = that.x,
                    width = that.width,
                    points = [createPoint(x, that.highY), openYExist && createPoint(x, that.openY), openYExist && createPoint(x - width / 2, that.openY), openYExist && createPoint(x, that.openY), closeYExist && createPoint(x, that.closeY), closeYExist && createPoint(x + width / 2, that.closeY), closeYExist && createPoint(x, that.closeY), createPoint(x, that.lowY)];
                return points
            },
            _drawMarkerInGroup: function(group, attributes, renderer) {
                this.graphic = renderer.createPath(this._getPoints(), attributes).append(group)
            },
            _getMinTrackerWidth: function() {
                return 2 + this._styles.normal.strokeWidth
            }
        })
    })(jQuery, DevExpress);
    /*! Module viz-core, file default.js */
    (function($, DX, undefined) {
        DX.viz.themes = DX.viz.themes || [];
        var fontFamilyDefault = "'Segoe UI', 'Helvetica Neue', 'Trebuchet MS', Verdana",
            fontFamilyLight = "'Segoe UI Light', 'Helvetica Neue Light', 'Segoe UI', 'Helvetica Neue', 'Trebuchet MS', Verdana",
            baseChartTheme = {
                containerBackgroundColor: '#ffffff',
                animation: {
                    enabled: true,
                    duration: 1000,
                    easing: 'easeOutCubic',
                    maxPointCountSupported: 300
                },
                commonSeriesSettings: {
                    border: {
                        visible: false,
                        width: 2
                    },
                    showInLegend: true,
                    visible: true,
                    hoverMode: 'excludePoints',
                    selectionMode: 'includePoints',
                    hoverStyle: {
                        hatching: {
                            direction: 'right',
                            width: 2,
                            step: 6,
                            opacity: 0.75
                        },
                        border: {
                            visible: false,
                            width: 3
                        }
                    },
                    selectionStyle: {
                        hatching: {
                            direction: 'right',
                            width: 2,
                            step: 6,
                            opacity: 0.5
                        },
                        border: {
                            visible: false,
                            width: 3
                        }
                    },
                    label: {
                        visible: false,
                        alignment: 'center',
                        rotationAngle: 0,
                        horizontalOffset: 0,
                        verticalOffset: 0,
                        radialOffset: 0,
                        format: '',
                        argumentFormat: '',
                        precision: 0,
                        argumentPrecision: 0,
                        percentPrecision: 0,
                        showForZeroValues: true,
                        customizeText: undefined,
                        maxLabelCount: undefined,
                        position: 'outside',
                        font: {color: '#ffffff'},
                        border: {
                            visible: false,
                            width: 1,
                            color: '#d3d3d3',
                            dashStyle: 'solid'
                        },
                        connector: {
                            visible: false,
                            width: 1
                        }
                    }
                },
                redrawOnResize: true,
                margin: {
                    left: 0,
                    top: 0,
                    right: 0,
                    bottom: 0
                },
                seriesSelectionMode: 'single',
                pointSelectionMode: 'single',
                legend: {
                    hoverMode: 'includePoints',
                    verticalAlignment: 'top',
                    horizontalAlignment: 'right',
                    position: 'outside',
                    visible: true,
                    customizeText: undefined,
                    itemTextPosition: undefined,
                    margin: 10,
                    equalColumnWidth: false,
                    markerSize: 12,
                    backgroundColor: undefined,
                    backgroundOpacity: undefined,
                    border: {
                        visible: false,
                        width: 1,
                        color: '#d3d3d3',
                        cornerRadius: 0,
                        dashStyle: 'solid'
                    },
                    paddingLeftRight: 20,
                    paddingTopBottom: 15,
                    columnCount: 0,
                    rowCount: 0,
                    columnItemSpacing: 20,
                    rowItemSpacing: 8
                },
                tooltip: {
                    enabled: false,
                    border: {
                        width: 1,
                        color: '#d3d3d3',
                        dashStyle: 'solid',
                        visible: true
                    },
                    font: {
                        family: fontFamilyDefault,
                        weight: 400,
                        size: 12,
                        color: '#232323'
                    },
                    color: '#ffffff',
                    arrowLength: 10,
                    paddingLeftRight: 18,
                    paddingTopBottom: 15,
                    shared: false,
                    format: '',
                    argumentFormat: '',
                    precision: 0,
                    argumentPrecision: 0,
                    percentPrecision: 0,
                    customizeText: undefined,
                    customizeTooltip: undefined,
                    shadow: {
                        opacity: 0.4,
                        offsetX: 0,
                        offsetY: 4,
                        blur: 2,
                        color: '#000000'
                    }
                },
                size: {
                    width: undefined,
                    height: undefined
                },
                loadingIndicator: {
                    font: {},
                    backgroundColor: '#ffffff',
                    text: 'Loading...'
                },
                dataPrepareSettings: {
                    checkTypeForAllData: false,
                    convertToAxisDataType: true,
                    sortingMethod: true
                },
                title: {
                    font: {
                        family: fontFamilyLight,
                        weight: 200,
                        color: '#232323',
                        size: 28
                    },
                    margin: 10
                },
                adaptiveLayout: {
                    width: 80,
                    height: 80,
                    keepLabels: true
                },
                _rtl: {legend: {itemTextPosition: 'left'}}
            },
            baseDarkChartTheme = {
                containerBackgroundColor: '#2b2b2b',
                commonSeriesSettings: {label: {border: {color: '#494949'}}},
                legend: {border: {color: '#494949'}},
                loadingIndicator: {backgroundColor: '#2b2b2b'},
                title: {font: {color: '#929292'}},
                tooltip: {
                    color: '#2b2b2b',
                    border: {color: '#494949'},
                    font: {color: '#929292'}
                }
            };
        DX.viz.themes.push({
            name: 'desktop',
            font: {
                color: '#767676',
                family: fontFamilyDefault,
                weight: 400,
                size: 12,
                cursor: 'default'
            },
            chart: $.extend(true, {}, baseChartTheme, {
                commonSeriesSettings: {
                    type: 'line',
                    stack: 'default',
                    point: {
                        visible: true,
                        symbol: 'circle',
                        size: 12,
                        border: {
                            visible: false,
                            width: 1
                        },
                        hoverMode: 'onlyPoint',
                        selectionMode: 'onlyPoint',
                        hoverStyle: {
                            border: {
                                visible: true,
                                width: 4
                            },
                            size: 12
                        },
                        selectionStyle: {
                            border: {
                                visible: true,
                                width: 4
                            },
                            size: 12
                        }
                    },
                    scatter: {},
                    line: {
                        width: 2,
                        dashStyle: 'solid',
                        hoverStyle: {
                            width: 3,
                            hatching: {direction: 'none'}
                        },
                        selectionStyle: {width: 3}
                    },
                    stackedline: {
                        width: 2,
                        dashStyle: 'solid',
                        hoverStyle: {
                            width: 3,
                            hatching: {direction: 'none'}
                        },
                        selectionStyle: {width: 3}
                    },
                    fullstackedline: {
                        width: 2,
                        dashStyle: 'solid',
                        hoverStyle: {
                            width: 3,
                            hatching: {direction: 'none'}
                        },
                        selectionStyle: {width: 3}
                    },
                    stepline: {
                        width: 2,
                        dashStyle: 'solid',
                        hoverStyle: {
                            width: 3,
                            hatching: {direction: 'none'}
                        },
                        selectionStyle: {width: 3}
                    },
                    area: {
                        point: {visible: false},
                        opacity: 0.5
                    },
                    stackedarea: {
                        point: {visible: false},
                        opacity: 0.5
                    },
                    fullstackedarea: {
                        point: {visible: false},
                        opacity: 0.5
                    },
                    steparea: {
                        border: {
                            visible: true,
                            width: 2
                        },
                        point: {visible: false},
                        hoverStyle: {border: {
                                visible: true,
                                width: 3
                            }},
                        selectionStyle: {border: {
                                visible: true,
                                width: 3
                            }},
                        opacity: 0.5
                    },
                    spline: {
                        width: 2,
                        hoverStyle: {
                            width: 3,
                            hatching: {direction: 'none'}
                        },
                        selectionStyle: {width: 3}
                    },
                    splinearea: {
                        point: {visible: false},
                        opacity: 0.5
                    },
                    bar: {
                        cornerRadius: 0,
                        point: {
                            hoverStyle: {border: {visible: false}},
                            selectionStyle: {border: {visible: false}}
                        }
                    },
                    stackedbar: {
                        cornerRadius: 0,
                        point: {
                            hoverStyle: {border: {visible: false}},
                            selectionStyle: {border: {visible: false}}
                        },
                        label: {position: "inside"}
                    },
                    fullstackedbar: {
                        cornerRadius: 0,
                        point: {
                            hoverStyle: {border: {visible: false}},
                            selectionStyle: {border: {visible: false}}
                        },
                        label: {position: "inside"}
                    },
                    rangebar: {
                        cornerRadius: 0,
                        point: {
                            hoverStyle: {border: {visible: false}},
                            selectionStyle: {border: {visible: false}}
                        }
                    },
                    rangearea: {
                        point: {visible: false},
                        opacity: 0.5
                    },
                    rangesplinearea: {
                        point: {visible: false},
                        opacity: 0.5
                    },
                    bubble: {
                        opacity: 0.5,
                        point: {
                            hoverStyle: {border: {visible: false}},
                            selectionStyle: {border: {visible: false}}
                        }
                    },
                    candlestick: {
                        width: 1,
                        innerColor: '#ffffff',
                        reduction: {color: '#ff0000'},
                        hoverStyle: {
                            width: 3,
                            hatching: {direction: 'none'}
                        },
                        selectionStyle: {width: 3},
                        point: {border: {visible: true}}
                    },
                    stock: {
                        width: 1,
                        reduction: {color: '#ff0000'},
                        hoverStyle: {
                            width: 3,
                            hatching: {direction: 'none'}
                        },
                        selectionStyle: {width: 3},
                        point: {border: {visible: true}}
                    }
                },
                crosshair: {
                    enabled: false,
                    color: '#c6c6c6',
                    width: 1,
                    dashStyle: 'solid',
                    verticalLine: {visible: true},
                    horizontalLine: {visible: true}
                },
                commonAxisSettings: {
                    tickInterval: undefined,
                    setTicksAtUnitBeginning: true,
                    valueMarginsEnabled: true,
                    placeholderSize: null,
                    logarithmBase: 10,
                    discreteAxisDivisionMode: 'betweenLabels',
                    visible: false,
                    color: '#d3d3d3',
                    width: 1,
                    multipleAxesSpacing: 5,
                    label: {
                        visible: true,
                        overlappingBehavior: {
                            mode: 'auto',
                            rotationAngle: 90,
                            staggeringSpacing: 5
                        },
                        precision: 0,
                        format: '',
                        customizeText: undefined,
                        indentFromAxis: 10
                    },
                    grid: {
                        visible: false,
                        color: '#d3d3d3',
                        width: 1
                    },
                    tick: {
                        visible: false,
                        color: '#d3d3d3'
                    },
                    title: {
                        font: {size: 16},
                        margin: 10
                    },
                    stripStyle: {
                        paddingLeftRight: 10,
                        paddingTopBottom: 5
                    },
                    constantLineStyle: {
                        paddingLeftRight: 10,
                        paddingTopBottom: 10,
                        width: 1,
                        color: '#000000',
                        dashStyle: 'solid',
                        label: {
                            visible: true,
                            position: 'inside'
                        }
                    }
                },
                horizontalAxis: {
                    isHorizontal: true,
                    position: 'bottom',
                    axisDivisionFactor: 50,
                    label: {alignment: "center"},
                    stripStyle: {label: {
                            horizontalAlignment: 'center',
                            verticalAlignment: 'top'
                        }},
                    constantLineStyle: {label: {
                            horizontalAlignment: 'right',
                            verticalAlignment: 'top'
                        }},
                    constantLines: {}
                },
                verticalAxis: {
                    isHorizontal: false,
                    position: 'left',
                    axisDivisionFactor: 30,
                    label: {
                        alignment: 'right',
                        overlappingBehavior: {mode: 'enlargeTickInterval'}
                    },
                    stripStyle: {label: {
                            horizontalAlignment: 'left',
                            verticalAlignment: 'center'
                        }},
                    constantLineStyle: {label: {
                            horizontalAlignment: 'left',
                            verticalAlignment: 'top'
                        }},
                    constantLines: {}
                },
                argumentAxis: {},
                valueAxis: {grid: {visible: true}},
                commonPaneSettings: {
                    backgroundColor: 'none',
                    border: {
                        color: '#d3d3d3',
                        width: 1,
                        visible: false,
                        top: true,
                        bottom: true,
                        left: true,
                        right: true,
                        dashStyle: 'solid'
                    }
                },
                useAggregation: false,
                adjustOnZoom: true,
                rotated: false,
                synchronizeMultiAxes: true,
                equalBarWidth: true,
                minBubbleSize: 12,
                maxBubbleSize: 0.2
            }),
            pie: $.extend(true, {}, baseChartTheme, {
                commonSeriesSettings: {
                    type: 'pie',
                    pie: {
                        border: {
                            visible: false,
                            width: 2,
                            color: '#ffffff'
                        },
                        hoverStyle: {
                            hatching: {
                                direction: 'right',
                                width: 4,
                                step: 10,
                                opacity: 0.75
                            },
                            border: {
                                visible: false,
                                width: 2
                            }
                        },
                        selectionStyle: {
                            hatching: {
                                direction: 'right',
                                width: 4,
                                step: 10,
                                opacity: 0.5
                            },
                            border: {
                                visible: false,
                                width: 2
                            }
                        }
                    },
                    doughnut: {
                        innerRadius: 0.5,
                        border: {
                            visible: false,
                            width: 2,
                            color: '#ffffff'
                        },
                        hoverStyle: {
                            hatching: {
                                direction: 'right',
                                width: 4,
                                step: 10,
                                opacity: 0.75
                            },
                            border: {
                                visible: false,
                                width: 2
                            }
                        },
                        selectionStyle: {
                            hatching: {
                                direction: 'right',
                                width: 4,
                                step: 10,
                                opacity: 0.5
                            },
                            border: {
                                visible: false,
                                width: 2
                            }
                        }
                    },
                    donut: {
                        innerRadius: 0.5,
                        border: {
                            visible: false,
                            width: 2,
                            color: '#ffffff'
                        },
                        hoverStyle: {
                            hatching: {
                                direction: 'right',
                                width: 4,
                                step: 10,
                                opacity: 0.75
                            },
                            border: {
                                visible: false,
                                width: 2
                            }
                        },
                        selectionStyle: {
                            hatching: {
                                direction: 'right',
                                width: 4,
                                step: 10,
                                opacity: 0.5
                            },
                            border: {
                                visible: false,
                                width: 2
                            }
                        }
                    }
                },
                legend: {hoverMode: 'markPoint'},
                adaptiveLayout: {keepLabels: false}
            }),
            pieIE8: {commonSeriesSettings: {
                    pie: {
                        hoverStyle: {border: {
                                visible: true,
                                color: '#ffffff'
                            }},
                        selectionStyle: {border: {
                                visible: true,
                                color: '#ffffff'
                            }}
                    },
                    donut: {
                        hoverStyle: {border: {
                                visible: true,
                                color: '#ffffff'
                            }},
                        selectionStyle: {border: {
                                visible: true,
                                color: '#ffffff'
                            }}
                    },
                    doughnut: {
                        hoverStyle: {border: {
                                visible: true,
                                color: '#ffffff'
                            }},
                        selectionStyle: {border: {
                                visible: true,
                                color: '#ffffff'
                            }}
                    }
                }},
            gauge: {
                containerBackgroundColor: '#ffffff',
                scale: {
                    majorTick: {
                        visible: true,
                        length: 5,
                        width: 2,
                        showCalculatedTicks: true,
                        useTicksAutoArrangement: true,
                        color: '#ffffff'
                    },
                    minorTick: {
                        visible: false,
                        length: 3,
                        width: 1,
                        showCalculatedTicks: true,
                        color: '#ffffff'
                    },
                    label: {
                        visible: true,
                        font: {}
                    }
                },
                rangeContainer: {
                    offset: 0,
                    width: 5,
                    backgroundColor: '#808080'
                },
                valueIndicator: {
                    _default: {color: '#c2c2c2'},
                    rangebar: {
                        space: 2,
                        size: 10,
                        color: '#cbc5cf',
                        backgroundColor: 'none',
                        text: {
                            indent: 0,
                            font: {
                                size: 14,
                                color: null
                            }
                        }
                    },
                    twocolorneedle: {secondColor: '#e18e92'},
                    twocolorrectangle: {secondColor: '#e18e92'}
                },
                subvalueIndicator: {
                    _default: {color: '#8798a5'},
                    trianglemarker: {
                        space: 2,
                        length: 14,
                        width: 13,
                        color: '#8798a5'
                    },
                    triangle: {
                        space: 2,
                        length: 14,
                        width: 13,
                        color: '#8798a5'
                    },
                    textcloud: {
                        arrowLength: 5,
                        horizontalOffset: 6,
                        verticalOffset: 3,
                        color: '#679ec5',
                        text: {font: {
                                color: '#ffffff',
                                size: 18
                            }}
                    }
                },
                valueIndicators: {
                    _default: {color: '#c2c2c2'},
                    rangebar: {
                        space: 2,
                        size: 10,
                        color: '#cbc5cf',
                        backgroundColor: 'none',
                        text: {
                            indent: 0,
                            font: {
                                size: 14,
                                color: null
                            }
                        }
                    },
                    twocolorneedle: {secondColor: '#e18e92'},
                    trianglemarker: {
                        space: 2,
                        length: 14,
                        width: 13,
                        color: '#8798a5'
                    },
                    textcloud: {
                        arrowLength: 5,
                        horizontalOffset: 6,
                        verticalOffset: 3,
                        color: '#679ec5',
                        text: {font: {
                                color: '#ffffff',
                                size: 18
                            }}
                    }
                },
                title: {
                    layout: {
                        horizontalAlignment: 'center',
                        verticalAlignment: 'top',
                        overlay: 0
                    },
                    font: {
                        size: 16,
                        color: '#232323',
                        family: fontFamilyDefault,
                        weight: 400
                    }
                },
                subtitle: {font: {
                        size: 14,
                        color: '#232323',
                        family: fontFamilyDefault,
                        weight: 400
                    }},
                indicator: {
                    hasPositiveMeaning: true,
                    layout: {
                        horizontalAlignment: 'center',
                        verticalAlignment: 'bottom',
                        overlay: 0
                    },
                    text: {font: {size: 18}}
                },
                tooltip: {
                    arrowLength: 10,
                    paddingLeftRight: 18,
                    paddingTopBottom: 15,
                    enabled: false,
                    border: {
                        width: 1,
                        color: '#d3d3d3',
                        dashStyle: 'solid',
                        visible: true
                    },
                    color: '#ffffff',
                    font: {
                        color: '#232323',
                        size: 12,
                        family: fontFamilyDefault,
                        weight: 400
                    },
                    shadow: {
                        opacity: 0.4,
                        offsetX: 0,
                        offsetY: 4,
                        blur: 2,
                        color: '#000000'
                    }
                },
                loadingIndicator: {
                    font: {},
                    backgroundColor: '#ffffff',
                    text: 'Loading...'
                },
                _circular: {
                    scale: {
                        orientation: 'outside',
                        label: {indentFromTick: 10}
                    },
                    rangeContainer: {orientation: 'outside'},
                    valueIndicator: {
                        type: 'rectangleneedle',
                        _default: {
                            offset: 20,
                            indentFromCenter: 0,
                            width: 2,
                            spindleSize: 14,
                            spindleGapSize: 10
                        },
                        triangleneedle: {width: 4},
                        triangle: {width: 4},
                        twocolorneedle: {
                            space: 2,
                            secondFraction: 0.4
                        },
                        twocolorrectangle: {
                            space: 2,
                            secondFraction: 0.4
                        },
                        rangebar: {offset: 30}
                    },
                    subvalueIndicator: {
                        type: 'trianglemarker',
                        trianglemarker: {offset: 6},
                        triangle: {offset: 6},
                        textcloud: {offset: -6}
                    },
                    valueIndicators: {
                        _type: 'rectangleneedle',
                        _default: {
                            offset: 20,
                            indentFromCenter: 0,
                            width: 2,
                            spindleSize: 14,
                            spindleGapSize: 10
                        },
                        triangleneedle: {width: 4},
                        twocolorneedle: {
                            space: 2,
                            secondFraction: 0.4
                        },
                        rangebar: {offset: 30},
                        trianglemarker: {offset: 6},
                        textcloud: {offset: -6}
                    }
                },
                _linear: {
                    scale: {
                        horizontalOrientation: 'right',
                        verticalOrientation: 'bottom',
                        label: {indentFromTick: -10}
                    },
                    rangeContainer: {
                        horizontalOrientation: 'right',
                        verticalOrientation: 'bottom'
                    },
                    valueIndicator: {
                        type: 'rangebar',
                        _default: {
                            offset: 2.5,
                            length: 15,
                            width: 15
                        },
                        rectangle: {width: 10},
                        rangebar: {
                            offset: 10,
                            horizontalOrientation: 'right',
                            verticalOrientation: 'bottom'
                        }
                    },
                    subvalueIndicator: {
                        type: 'trianglemarker',
                        _default: {
                            offset: -1,
                            horizontalOrientation: 'left',
                            verticalOrientation: 'top'
                        }
                    },
                    valueIndicators: {
                        _type: 'rectangle',
                        _default: {
                            offset: 2.5,
                            length: 15,
                            width: 15
                        },
                        rectangle: {width: 10},
                        rangebar: {
                            offset: 10,
                            horizontalOrientation: 'right',
                            verticalOrientation: 'bottom'
                        },
                        trianglemarker: {
                            offset: -1,
                            horizontalOrientation: 'left',
                            verticalOrientation: 'top'
                        },
                        textcloud: {
                            offset: -1,
                            horizontalOrientation: 'left',
                            verticalOrientation: 'top'
                        }
                    }
                }
            },
            barGauge: {
                backgroundColor: '#e0e0e0',
                relativeInnerRadius: 0.3,
                barSpacing: 4,
                label: {
                    indent: 20,
                    connectorWidth: 2,
                    font: {size: 16}
                },
                title: {
                    layout: {
                        horizontalAlignment: 'center',
                        verticalAlignment: 'top',
                        overlay: 0
                    },
                    font: {
                        size: 16,
                        color: '#232323',
                        family: fontFamilyDefault,
                        weight: 400
                    }
                },
                subtitle: {font: {
                        size: 14,
                        color: '#232323',
                        family: fontFamilyDefault,
                        weight: 400
                    }},
                indicator: {
                    hasPositiveMeaning: true,
                    layout: {
                        horizontalAlignment: 'center',
                        verticalAlignment: 'bottom',
                        overlay: 0
                    },
                    text: {font: {size: 18}}
                },
                tooltip: {
                    arrowLength: 10,
                    paddingLeftRight: 18,
                    paddingTopBottom: 15,
                    enabled: false,
                    border: {
                        width: 1,
                        color: '#d3d3d3',
                        dashStyle: 'solid',
                        visible: true
                    },
                    color: '#ffffff',
                    font: {
                        size: 12,
                        color: '#232323',
                        family: fontFamilyDefault,
                        weight: 400
                    },
                    shadow: {
                        opacity: 0.4,
                        offsetX: 0,
                        offsetY: 4,
                        blur: 2,
                        color: '#000000'
                    }
                },
                loadingIndicator: {
                    font: {},
                    backgroundColor: '#ffffff',
                    text: 'Loading...'
                }
            },
            rangeSelector: {
                containerBackgroundColor: '#ffffff',
                scale: {
                    label: {
                        topIndent: 7,
                        font: {size: 11}
                    },
                    tick: {
                        width: 1,
                        color: '#000000',
                        opacity: 0.1
                    },
                    marker: {
                        separatorHeight: 33,
                        topIndent: 10,
                        textLeftIndent: 7,
                        textTopIndent: 11
                    }
                },
                loadingIndicator: {
                    font: {},
                    backgroundColor: '#ffffff',
                    text: 'Loading...'
                },
                sliderMarker: {
                    padding: 7,
                    pointerSize: 6,
                    color: '#9b9b9b',
                    invalidRangeColor: '#ff0000',
                    font: {
                        color: '#ffffff',
                        size: 11
                    }
                },
                sliderHandle: {
                    width: 1,
                    color: '#000000',
                    opacity: 0.2
                },
                shutter: {
                    color: undefined,
                    opacity: 0.75
                },
                background: {color: "#c0bae1"},
                chart: {
                    containerBackgroundColor: undefined,
                    commonSeriesSettings: {
                        label: baseChartTheme.commonSeriesSettings.label,
                        border: {
                            visible: false,
                            width: 1
                        },
                        visible: true,
                        type: 'area',
                        hoverMode: 'none',
                        hoverStyle: {border: {}},
                        selectionStyle: {border: {}},
                        point: {
                            visible: false,
                            symbol: 'circle',
                            border: {
                                visible: false,
                                width: 1
                            },
                            size: 12,
                            hoverStyle: {border: {}},
                            selectionStyle: {border: {}}
                        },
                        line: {width: 2},
                        stepline: {width: 2},
                        scatter: {point: {visible: true}},
                        stackedline: {width: 2},
                        fullstackedline: {width: 2},
                        area: {opacity: 0.5},
                        stackedarea: {opacity: 0.5},
                        fullstackedarea: {opacity: 0.5},
                        spline: {width: 2},
                        splinearea: {opacity: 0.5},
                        steparea: {
                            border: {
                                visible: true,
                                width: 2
                            },
                            opacity: 0.5
                        },
                        bubble: {
                            opacity: 0.5,
                            point: {visible: true}
                        },
                        bar: {
                            cornerRadius: 0,
                            point: {visible: true}
                        },
                        stackedbar: {
                            cornerRadius: 0,
                            point: {visible: true}
                        },
                        fullstackedbar: {
                            cornerRadius: 0,
                            point: {visible: true}
                        },
                        rangebar: {
                            cornerRadius: 0,
                            point: {visible: true}
                        },
                        rangearea: {opacity: 0.5},
                        rangesplinearea: {opacity: 0.5},
                        candlestick: {
                            width: 1,
                            innerColor: '#ffffff',
                            reduction: {color: '#ff0000'}
                        },
                        stock: {
                            width: 1,
                            reduction: {color: '#ff0000'}
                        }
                    },
                    dataPrepareSettings: {
                        checkTypeForAllData: false,
                        convertToAxisDataType: true,
                        sortingMethod: true
                    },
                    useAggregation: false,
                    equalBarWidth: true,
                    minBubbleSize: 12,
                    maxBubbleSize: 0.2,
                    topIndent: 0.1,
                    bottomIndent: 0,
                    valueAxis: {
                        min: undefined,
                        max: undefined,
                        inverted: false,
                        logarithmBase: 10
                    }
                }
            },
            map: {
                background: {
                    borderWidth: 1,
                    borderColor: '#cacaca',
                    color: '#ffffff'
                },
                area: {
                    borderWidth: 1,
                    borderColor: '#ffffff',
                    color: '#d2d2d2',
                    hoveredBorderColor: '#303030',
                    selectedBorderWidth: 2,
                    selectedBorderColor: '#303030'
                },
                marker: {
                    font: {
                        color: '#2b2b2b',
                        size: 12
                    },
                    _dot: {
                        borderWidth: 2,
                        borderColor: '#ffffff',
                        color: '#ba4d51',
                        size: 8,
                        selectedStep: 2,
                        backStep: 18,
                        backColor: '#ffffff',
                        backOpacity: 0.32,
                        shadow: true
                    },
                    _bubble: {
                        minSize: 20,
                        maxSize: 50,
                        color: '#ba4d51',
                        hoveredBorderWidth: 1,
                        hoveredBorderColor: '#303030',
                        selectedBorderWidth: 2,
                        selectedBorderColor: '#303030'
                    },
                    _pie: {
                        size: 50,
                        hoveredBorderWidth: 1,
                        hoveredBorderColor: '#303030',
                        selectedBorderWidth: 2,
                        selectedBorderColor: '#303030'
                    },
                    _image: {size: 20}
                },
                controlBar: {
                    borderColor: '#5d5d5d',
                    borderWidth: 3,
                    color: '#ffffff'
                },
                tooltip: {
                    borderWidth: 1,
                    borderColor: '#d7d7d7',
                    arrowLength: 10,
                    paddingLeftRight: 18,
                    paddingTopBottom: 15,
                    border: {
                        width: 1,
                        color: '#d3d3d3',
                        dashStyle: 'solid',
                        visible: true
                    },
                    color: '#ffffff',
                    font: {
                        color: '#232323',
                        size: 12,
                        family: fontFamilyDefault,
                        weight: 400
                    },
                    shadow: {
                        opacity: 0.4,
                        offsetX: 0,
                        offsetY: 4,
                        blur: 2,
                        color: '#000000'
                    }
                },
                legend: {
                    verticalAlignment: 'bottom',
                    horizontalAlignment: 'right',
                    position: 'inside',
                    visible: true,
                    margin: 10,
                    equalColumnWidth: false,
                    markerSize: 12,
                    backgroundColor: '#ffffff',
                    backgroundOpacity: 0.65,
                    border: {
                        visible: true,
                        width: 1,
                        color: '#cacaca',
                        cornerRadius: 0,
                        dashStyle: 'solid'
                    },
                    paddingLeftRight: 16,
                    paddingTopBottom: 12,
                    columnItemSpacing: 20,
                    rowItemSpacing: 8,
                    font: {
                        color: '#2b2b2b',
                        size: 12
                    }
                },
                loadingIndicator: {
                    backgroundColor: '#ffffff',
                    font: {},
                    text: 'Loading...'
                },
                _rtl: {legend: {itemTextPosition: 'left'}}
            },
            sparkline: {
                lineColor: '#666666',
                lineWidth: 2,
                areaOpacity: 0.2,
                minColor: '#e8c267',
                maxColor: '#e55253',
                barPositiveColor: '#a9a9a9',
                barNegativeColor: '#d7d7d7',
                winColor: '#a9a9a9',
                lossColor: '#d7d7d7',
                firstLastColor: '#666666',
                pointSymbol: 'circle',
                pointColor: '#ffffff',
                pointSize: 4,
                tooltip: {
                    enabled: true,
                    allowContainerResizing: true,
                    verticalAlignment: 'top',
                    horizontalAlignment: 'center',
                    format: '',
                    paddingLeftRight: 18,
                    paddingTopBottom: 15,
                    arrowLength: 10,
                    precision: 0,
                    color: '#ffffff',
                    border: {
                        width: 1,
                        color: '#d3d3d3',
                        dashStyle: 'solid',
                        visible: true
                    },
                    font: {
                        color: '#232323',
                        family: fontFamilyDefault,
                        size: 12,
                        weight: 400
                    },
                    shadow: {
                        opacity: 0.4,
                        offsetX: 0,
                        offsetY: 4,
                        blur: 2,
                        color: '#000000'
                    }
                }
            },
            bullet: {
                color: '#e8c267',
                targetColor: '#666666',
                targetWidth: 4,
                showTarget: true,
                showZeroLevel: true,
                tooltip: {
                    enabled: true,
                    allowContainerResizing: true,
                    verticalAlignment: 'top',
                    horizontalAlignment: 'center',
                    format: '',
                    precision: 0,
                    paddingLeftRight: 18,
                    paddingTopBottom: 15,
                    arrowLength: 10,
                    color: '#ffffff',
                    border: {
                        width: 1,
                        color: '#d3d3d3',
                        dashStyle: 'solid',
                        visible: true
                    },
                    font: {
                        color: '#232323',
                        family: fontFamilyDefault,
                        size: 12,
                        weight: 400
                    },
                    shadow: {
                        opacity: 0.4,
                        offsetX: 0,
                        offsetY: 4,
                        blur: 2,
                        color: '#000000'
                    }
                }
            }
        });
        DX.viz.core.registerTheme({
            name: 'desktop-dark',
            font: {color: '#808080'},
            chart: $.extend(true, {}, baseDarkChartTheme, {
                commonSeriesSettings: {candlestick: {innerColor: '#2b2b2b'}},
                crosshair: {color: '#515151'},
                commonAxisSettings: {
                    color: '#494949',
                    grid: {color: '#494949'},
                    tick: {color: '#494949'},
                    constantLineStyle: {color: '#ffffff'}
                },
                commonPaneSettings: {border: {color: '#494949'}}
            }),
            pie: $.extend(true, {}, baseDarkChartTheme),
            pieIE8: {commonSeriesSettings: {
                    pie: {
                        hoverStyle: {border: {color: '#2b2b2b'}},
                        selectionStyle: {border: {color: '#2b2b2b'}}
                    },
                    donut: {
                        hoverStyle: {border: {color: '#2b2b2b'}},
                        selectionStyle: {border: {color: '#2b2b2b'}}
                    },
                    doughnut: {
                        hoverStyle: {border: {color: '#2b2b2b'}},
                        selectionStyle: {border: {color: '#2b2b2b'}}
                    }
                }},
            gauge: {
                containerBackgroundColor: '#2b2b2b',
                scale: {
                    majorTick: {color: '#303030'},
                    minorTick: {color: '#303030'}
                },
                rangeContainer: {backgroundColor: '#b5b5b5'},
                valueIndicator: {
                    _default: {color: '#b5b5b5'},
                    rangebar: {color: '#84788b'},
                    twocolorneedle: {secondColor: '#ba544d'},
                    twocolorrectangle: {secondColor: '#ba544d'}
                },
                subvalueIndicator: {_default: {color: '#b7918f'}},
                valueIndicators: {
                    _default: {color: '#b5b5b5'},
                    rangebar: {color: '#84788b'},
                    twocolorneedle: {secondColor: '#ba544d'},
                    trianglemarker: {color: '#b7918f'},
                    textcloud: {color: '#ba544d'}
                },
                title: {font: {color: '#929292'}},
                subtitle: {font: {color: '#929292'}},
                loadingIndicator: {backgroundColor: '#2b2b2b'},
                tooltip: {
                    color: '#2b2b2b',
                    border: {color: '#494949'},
                    font: {color: '#929292'}
                }
            },
            barGauge: {
                title: {font: {color: '#929292'}},
                subtitle: {font: {color: '#929292'}},
                tooltip: {
                    color: '#2b2b2b',
                    border: {color: '#494949'},
                    font: {color: '#929292'}
                },
                loadingIndicator: {backgroundColor: '#2b2b2b'}
            },
            rangeSelector: {
                containerBackgroundColor: '#2b2b2b',
                scale: {tick: {
                        color: '#ffffff',
                        opacity: 0.05
                    }},
                loadingIndicator: {backgroundColor: '#2b2b2b'},
                sliderMarker: {
                    color: '#b5b5b5',
                    font: {color: '#303030'}
                },
                sliderHandle: {
                    color: '#ffffff',
                    opacity: 0.35
                },
                shutter: {
                    color: '#2b2b2b',
                    opacity: 0.9
                }
            },
            map: {
                background: {
                    borderColor: '#3f3f3f',
                    color: '#303030'
                },
                area: {
                    borderColor: '#303030',
                    color: '#686868',
                    hoveredBorderColor: '#ffffff',
                    selectedBorderColor: '#ffffff'
                },
                marker: {
                    font: {color: '#ffffff'},
                    _bubble: {
                        hoveredBorderColor: '#ffffff',
                        selectedBorderColor: '#ffffff'
                    },
                    _pie: {
                        hoveredBorderColor: '#ffffff',
                        selectedBorderColor: '#ffffff'
                    }
                },
                controlBar: {
                    borderColor: '#c7c7c7',
                    color: '#303030'
                },
                legend: {
                    border: {color: '#3f3f3f'},
                    backgroundColor: '#303030',
                    font: {color: '#ffffff'}
                },
                tooltip: {
                    color: '#2b2b2b',
                    border: {color: '#494949'},
                    font: {color: '#929292'}
                },
                loadingIndicator: {backgroundColor: '#2b2b2b'}
            },
            sparkline: {
                lineColor: '#c7c7c7',
                firstLastColor: '#c7c7c7',
                barPositiveColor: '#b8b8b8',
                barNegativeColor: '#8e8e8e',
                winColor: '#b8b8b8',
                lossColor: '#8e8e8e',
                pointColor: '#303030',
                tooltip: {
                    color: '#2b2b2b',
                    border: {color: '#494949'},
                    font: {color: '#929292'}
                }
            },
            bullet: {
                targetColor: '#8e8e8e',
                tooltip: {
                    color: '#2b2b2b',
                    border: {color: '#494949'},
                    font: {color: '#929292'}
                }
            }
        }, 'desktop')
    })(jQuery, DevExpress);
    /*! Module viz-core, file android.js */
    (function($, DX, undefined) {
        var baseChartTheme = {
                containerBackgroundColor: '#050506',
                title: {font: {color: '#ffffff'}},
                commonSeriesSettings: {label: {border: {color: '#4c4c4c'}}},
                legend: {
                    font: {
                        color: '#ffffff',
                        size: 11
                    },
                    border: {color: '#4c4c4c'}
                },
                loadingIndicator: {backgroundColor: '#050506'},
                tooltip: {
                    color: '#050506',
                    border: {color: '#4c4c4c'},
                    font: {color: '#ffffff'}
                }
            },
            baseLightChartTheme = {
                containerBackgroundColor: '#e8e8e8',
                title: {font: {color: '#808080'}},
                legend: {font: {
                        color: '#000000',
                        size: 11
                    }},
                loadingIndicator: {backgroundColor: '#e8e8e8'},
                tooltip: {
                    color: '#e8e8e8',
                    font: {color: '#808080'}
                }
            };
        DX.viz.core.registerTheme({
            name: 'android',
            chart: $.extend(true, {}, baseChartTheme, {
                commonSeriesSettings: {candlestick: {innerColor: '#050506'}},
                commonAxisSettings: {
                    color: '#4c4c4c',
                    grid: {color: '#4c4c4c'},
                    tick: {color: '#4c4c4c'},
                    title: {font: {color: '#545455'}},
                    label: {font: {
                            color: '#ffffff',
                            size: 11
                        }}
                },
                commonPaneSettings: {border: {color: '#4c4c4c'}}
            }),
            pie: $.extend(true, {}, baseChartTheme),
            pieIE8: {commonSeriesSettings: {
                    pie: {
                        hoverStyle: {border: {color: '#050506'}},
                        selectionStyle: {border: {color: '#050506'}}
                    },
                    donut: {
                        hoverStyle: {border: {color: '#050506'}},
                        selectionStyle: {border: {color: '#050506'}}
                    },
                    doughnut: {
                        hoverStyle: {border: {color: '#050506'}},
                        selectionStyle: {border: {color: '#050506'}}
                    }
                }},
            gauge: {
                containerBackgroundColor: '#050506',
                title: {font: {color: '#ffffff'}},
                subtitle: {font: {color: '#ffffff'}},
                loadingIndicator: {backgroundColor: '#050506'},
                tooltip: {
                    color: '#050506',
                    border: {color: '#4c4c4c'},
                    font: {color: '#ffffff'}
                }
            },
            barGauge: {
                title: {font: {color: '#ffffff'}},
                subtitle: {font: {color: '#ffffff'}},
                tooltip: {
                    color: '#050506',
                    border: {color: '#4c4c4c'},
                    font: {color: '#ffffff'}
                },
                loadingIndicator: {backgroundColor: '#050506'}
            },
            rangeSelector: {
                containerBackgroundColor: '#050506',
                loadingIndicator: {backgroundColor: '#050506'}
            },
            map: {
                loadingIndicator: {backgroundColor: '#050506'},
                tooltip: {
                    color: '#050506',
                    border: {color: '#4c4c4c'},
                    font: {color: '#ffffff'}
                }
            },
            sparkline: {tooltip: {
                    color: '#050506',
                    border: {color: '#4c4c4c'},
                    font: {color: '#ffffff'}
                }},
            bullet: {tooltip: {
                    color: '#050506',
                    border: {color: '#4c4c4c'},
                    font: {color: '#ffffff'}
                }}
        }, 'desktop-dark');
        DX.viz.core.registerTheme({
            name: 'android-holo-light',
            chart: $.extend(true, {}, baseLightChartTheme, {
                commonSeriesSettings: {candlestick: {innerColor: '#e8e8e8'}},
                commonAxisSettings: {
                    title: {font: {color: '#939393'}},
                    label: {font: {
                            color: '#404040',
                            size: 11
                        }}
                }
            }),
            pie: $.extend(true, {}, baseLightChartTheme),
            pieIE8: {commonSeriesSettings: {
                    pie: {
                        hoverStyle: {border: {color: '#e8e8e8'}},
                        selectionStyle: {border: {color: '#e8e8e8'}}
                    },
                    donut: {
                        hoverStyle: {border: {color: '#e8e8e8'}},
                        selectionStyle: {border: {color: '#e8e8e8'}}
                    },
                    doughnut: {
                        hoverStyle: {border: {color: '#e8e8e8'}},
                        selectionStyle: {border: {color: '#e8e8e8'}}
                    }
                }},
            gauge: {
                containerBackgroundColor: '#e8e8e8',
                title: {font: {color: '#808080'}},
                subtitle: {font: {color: '#808080'}},
                loadingIndicator: {backgroundColor: '#e8e8e8'},
                tooltip: {
                    color: '#e8e8e8',
                    font: {color: '#808080'}
                }
            },
            barGauge: {
                title: {font: {color: '#808080'}},
                subtitle: {font: {color: '#808080'}},
                tooltip: {
                    color: '#e8e8e8',
                    font: {color: '#808080'}
                },
                loadingIndicator: {backgroundColor: '#e8e8e8'}
            },
            rangeSelector: {
                containerBackgroundColor: '#e8e8e8',
                loadingIndicator: {backgroundColor: '#e8e8e8'}
            },
            map: {
                loadingIndicator: {backgroundColor: '#e8e8e8'},
                tooltip: {
                    color: '#e8e8e8',
                    font: {color: '#808080'}
                }
            },
            sparkline: {tooltip: {
                    color: '#e8e8e8',
                    font: {color: '#808080'}
                }},
            bullet: {tooltip: {
                    color: '#e8e8e8',
                    font: {color: '#808080'}
                }}
        }, 'desktop')
    })(jQuery, DevExpress);
    /*! Module viz-core, file ios.js */
    (function($, DX, undefined) {
        var baseChartTheme = {
                containerBackgroundColor: '#cbd0da',
                title: {font: {color: '#808080'}},
                commonSeriesSettings: {label: {border: {color: '#b0b3ba'}}},
                legend: {
                    font: {
                        color: '#000000',
                        size: 11
                    },
                    border: {color: '#b0b3ba'}
                },
                loadingIndicator: {backgroundColor: '#cbd0da'},
                tooltip: {font: {color: '#808080'}}
            },
            baseIos7ChartTheme = {
                containerBackgroundColor: '#ffffff',
                title: {font: {color: '#808080'}},
                commonSeriesSettings: {label: {border: {color: '#d3d3d3'}}},
                legend: {
                    font: {
                        color: '#000000',
                        size: 11
                    },
                    border: {color: '#d3d3d3'}
                },
                loadingIndicator: {backgroundColor: '#ffffff'},
                tooltip: {font: {color: '#808080'}}
            };
        DX.viz.core.registerTheme({
            name: 'ios',
            chart: $.extend(true, {}, baseChartTheme, {
                commonSeriesSettings: {candlestick: {innerColor: '#cbd0da'}},
                commonAxisSettings: {
                    color: '#b0b3ba',
                    grid: {color: '#b0b3ba'},
                    tick: {color: '#b0b3ba'},
                    title: {font: {color: '#939393'}},
                    label: {font: {
                            color: '#000000',
                            size: 11
                        }}
                },
                commonPaneSettings: {border: {color: '#b0b3ba'}}
            }),
            pie: $.extend(true, {}, baseChartTheme),
            pieIE8: {commonSeriesSettings: {
                    pie: {
                        hoverStyle: {border: {color: '#cbd0da'}},
                        selectionStyle: {border: {color: '#cbd0da'}}
                    },
                    donut: {
                        hoverStyle: {border: {color: '#cbd0da'}},
                        selectionStyle: {border: {color: '#cbd0da'}}
                    },
                    doughnut: {
                        hoverStyle: {border: {color: '#cbd0da'}},
                        selectionStyle: {border: {color: '#cbd0da'}}
                    }
                }},
            gauge: {
                title: {font: {color: '#808080'}},
                subtitle: {font: {color: '#808080'}},
                tooltip: {font: {color: '#808080'}}
            },
            barGauge: {
                title: {font: {color: '#808080'}},
                subtitle: {font: {color: '#808080'}},
                tooltip: {font: {color: '#808080'}}
            },
            map: {tooltip: {font: {color: '#808080'}}},
            sparkline: {tooltip: {font: {color: '#808080'}}},
            bullet: {tooltip: {font: {color: '#808080'}}}
        }, 'desktop');
        DX.viz.core.registerTheme({
            name: 'ios:7',
            chart: $.extend(true, {}, baseIos7ChartTheme, {
                commonAxisSettings: {
                    color: '#d3d3d3',
                    grid: {color: '#d3d3d3'},
                    tick: {color: '#d3d3d3'},
                    title: {font: {color: '#939393'}},
                    label: {font: {
                            color: '#000000',
                            size: 11
                        }}
                },
                commonPaneSettings: {border: {color: '#d3d3d3'}}
            }),
            pie: $.extend(true, {}, baseIos7ChartTheme),
            gauge: {
                title: {font: {color: '#808080'}},
                subtitle: {font: {color: '#808080'}},
                tooltip: {font: {color: '#808080'}}
            },
            barGauge: {
                title: {font: {color: '#808080'}},
                subtitle: {font: {color: '#808080'}},
                tooltip: {font: {color: '#808080'}}
            },
            map: {tooltip: {font: {color: '#808080'}}},
            sparkline: {tooltip: {font: {color: '#808080'}}},
            bullet: {tooltip: {font: {color: '#808080'}}}
        }, 'desktop')
    })(jQuery, DevExpress);
    /*! Module viz-core, file win8.js */
    (function($, DX) {
        var baseChartTheme = {
                containerBackgroundColor: '#000000',
                title: {font: {color: '#ffffff'}},
                commonSeriesSettings: {label: {border: {color: '#454545'}}},
                legend: {
                    font: {
                        color: '#ffffff',
                        size: 11
                    },
                    border: {color: '#454545'}
                },
                loadingIndicator: {backgroundColor: '#000000'},
                tooltip: {
                    color: '#000000',
                    font: {color: '#ffffff'}
                }
            },
            baseWhiteChartTheme = {
                title: {font: {color: '#808080'}},
                legend: {font: {
                        color: '#000000',
                        size: 11
                    }},
                tooltip: {font: {color: '#808080'}}
            };
        DX.viz.core.registerTheme({
            name: 'win8',
            chart: $.extend(true, {}, baseChartTheme, {
                commonSeriesSettings: {candlestick: {innerColor: '#000000'}},
                commonAxisSettings: {
                    color: '#454545',
                    grid: {color: '#454545'},
                    tick: {color: '#454545'},
                    title: {font: {color: '#535353'}},
                    label: {font: {
                            color: '#ffffff',
                            size: 11
                        }}
                },
                commonPaneSettings: {border: {color: '#454545'}}
            }),
            pie: $.extend(true, {}, baseChartTheme),
            pieIE8: {commonSeriesSettings: {
                    pie: {
                        hoverStyle: {border: {color: '#000000'}},
                        selectionStyle: {border: {color: '#000000'}}
                    },
                    donut: {
                        hoverStyle: {border: {color: '#000000'}},
                        selectionStyle: {border: {color: '#000000'}}
                    },
                    doughnut: {
                        hoverStyle: {border: {color: '#000000'}},
                        selectionStyle: {border: {color: '#000000'}}
                    }
                }},
            gauge: {
                containerBackgroundColor: '#000000',
                title: {font: {color: '#ffffff'}},
                subtitle: {font: {color: '#ffffff'}},
                loadingIndicator: {backgroundColor: '#000000'},
                tooltip: {
                    color: '#000000',
                    font: {color: '#ffffff'}
                }
            },
            barGauge: {
                title: {font: {color: '#ffffff'}},
                subtitle: {font: {color: '#ffffff'}},
                tooltip: {
                    color: '#000000',
                    font: {color: '#ffffff'}
                },
                loadingIndicator: {backgroundColor: '#000000'}
            },
            rangeSelector: {
                containerBackgroundColor: '#000000',
                loadingIndicator: {backgroundColor: '#000000'}
            },
            map: {
                loadingIndicator: {backgroundColor: '#000000'},
                tooltip: {
                    color: '#000000',
                    font: {color: '#ffffff'}
                }
            },
            sparkline: {tooltip: {
                    color: '#000000',
                    font: {color: '#ffffff'}
                }},
            bullet: {tooltip: {
                    color: '#000000',
                    font: {color: '#ffffff'}
                }}
        }, 'desktop-dark');
        DX.viz.core.registerTheme({
            name: 'win8-white',
            chart: $.extend(true, {}, baseWhiteChartTheme, {commonAxisSettings: {
                    title: {font: {color: '#939393'}},
                    label: {font: {
                            color: '#404040',
                            size: 11
                        }}
                }}),
            pie: $.extend(true, {}, baseWhiteChartTheme),
            gauge: {
                title: {font: {color: '#808080'}},
                subtitle: {font: {color: '#808080'}},
                tooltip: {font: {color: '#808080'}}
            },
            barGauge: {
                title: {font: {color: '#808080'}},
                subtitle: {font: {color: '#808080'}},
                tooltip: {font: {color: '#808080'}}
            },
            map: {tooltip: {font: {color: '#808080'}}},
            sparkline: {tooltip: {font: {color: '#808080'}}},
            bullet: {tooltip: {font: {color: '#808080'}}}
        }, 'desktop')
    })(jQuery, DevExpress);
    /*! Module viz-core, file others.js */
    (function($, DX) {
        DX.viz.core.registerTheme({name: 'generic'}, 'desktop');
        DX.viz.core.registerTheme({name: 'generic-dark'}, 'desktop-dark');
        DX.viz.core.registerTheme({name: 'tizen'}, 'desktop');
        DX.viz.core.registerTheme({name: 'tizen-black'}, 'desktop-dark')
    })(jQuery, DevExpress);
    /*! Module viz-core, file namespaces.js */
    (function(DevExpress) {
        DevExpress.viz.renderers = {}
    })(DevExpress);
    /*! Module viz-core, file svgRenderer.js */
    (function($, DX) {
        var renderers = DX.viz.renderers,
            utils = DX.utils,
            Class = DX.Class,
            doc = document,
            MAX_PIXEL_COUNT = 10000000000;
        function createElement(name) {
            return doc.createElementNS('http://www.w3.org/2000/svg', name)
        }
        function getPatternUrl(id, pathModified) {
            return id !== null ? 'url(' + (pathModified ? window.location.href : '') + '#' + id + ')' : null
        }
        var BaseSvgElement = Class.inherit({
                ctor: function baseSvgElementCtor(renderer, name, params) {
                    this.renderer = renderer;
                    this.element = this.createElement(name);
                    this.$element = $(this.element);
                    this.applySettings($.extend({}, this.defaultSettings(), params));
                    this.__passedParams = params
                },
                defaultSettings: $.noop,
                createElemen