# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['nornir_pyxl', 'nornir_pyxl.plugins', 'nornir_pyxl.plugins.tasks']

package_data = \
{'': ['*']}

install_requires = \
['nornir>=3.0.0,<4.0.0', 'openpyxl>=3.0.5,<4.0.0']

setup_kwargs = {
    'name': 'nornir-pyxl',
    'version': '1.0.0',
    'description': 'OpenPyxl plugin for Nornir',
    'long_description': '\n# Open Pyxl Plugin for [Nornir](github.com/nornir-automation/nornir)\n\n## Table of Contents\n\n- [Open Pyxl Plugin for Nornir](#open-pyxl-plugin-for-nornir)\n  - [Table of Contents](#table-of-contents)\n  - [Installation](#installation)\n  - [Plugins -> Tasks](#plugins---tasks)\n  - [Examples](#examples)\n    - [Pyxl Ez Data](#pyxl-ez-data)\n    - [Example - Map Data with Nested Dict Magic Key](#example---map-data-with-nested-dict-magic-key)\n\n## Installation\n\n------------\n\n```bash\npip install nornir_pyxl\n```\n\n## Plugins -> Tasks\n\n------------\n\n- **pyxl_ez_data** - Loads an XLSX file and creates a list of dictionaries. Excel file must be in a specific format.\n- **pyxl_map_data** - Loads an XLSX file and creates a list of dictionaries based on a user provided ENUM map. Allows user to specify row & column start & end. Also allows flexibility on how results are generated by using a magic key, "NESTED_DICT" inside the ENUM map.\n\n## Examples\n\n### Pyxl Ez Data\n\n![Image](docs/images/xlsx-preview.png)\n\nThis task plugin is expecting you to modify the Excel Spreadsheet for best results and it\'s kind of perfect world scenario. The list of dictionaries will be generated starting from row 2, always. Each column header starting from position 0 (A1) will be assigned the KEY. The task will also attempt to standardize the keys by doing the following:\n\n- Trimming White Space\n\n- Replacing dashes with underscores\n\n- Converting all letters to lowercase\n\n- Replacing any whitespace with an underscore\n\n------------\n\n```python\nfrom nornir_pyxl.plugins.tasks import pyxl_ez_data\nfrom nornir_utils.plugins.functions import print_result\nfrom nornir import InitNornir\n\n\nWORKBOOK = "working_example_wb.xlsx"\n\n\nnr = InitNornir("config.yml")\n\n\ndef get_mapped_data(task):\n    """Get a list of dictionaries from Excel Spreadsheet with Nornir Pyxl."""\n    res = task.run(task=pyxl_ez_data,\n        workbook=WORKBOOK,\n        sheetname="IP_DATA",\n    )\n\ndef main():\n    """Execute Tasks."""\n    print_result(nr.run(task=get_mapped_data))\n\n\nif __name__ == "__main__":\n    main()\n```\n\nThe following output is the result of the above tasks:\n\n```bash\n❯ python3 test_pyxl.py\nget_mapped_data*****************************************************************\n* r1-csr ** changed : False ****************************************************\nvvvv get_mapped_data ** changed : False vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv INFO\n---- pyxl_ez_data ** changed : False ------------------------------------------- INFO\n[ { \'clli\': \'PHNZAZ\',\n    \'ntp_server_1_ip\': \'192.168.1.100\',\n    \'ntp_server_2_ip\': \'192.168.1.102/32\',\n    \'ntp_server_3_ip\': \'192.168.100.3\',\n    \'ntp_server_4_ip\': \'time.ntp.com\',\n    \'site_id\': \'Q345501\',\n    \'system_name\': \'PHNZAZ -635696-01\'}]\n^^^^ END get_mapped_data ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n```\n\n### Example - Map Data with Nested Dict Magic Key\n\n![Image](docs/images/xlsx-preview.png)\n\n------------\nThe following example uses the magic ENUM key, "NESTED_DICT". If this key is found in the ENUM mapping, it will take the value represented by the ENUM Mapping result (some index in every row) and create a nested dictionary with the rest of the values from the ENUM mapping inside of it. The actual value of the ENUM, "NESTED_DICT" key will not be duplicated inside the dictionary for each row.\n\nBelow you can see that NESTED_DICT is assigned to \'0\'. This represents the "SITE_ID" column header in our example spreadsheet. Therefore, it will loop through each row and generate a list of nested dictionaries in the following manner:\n\n```json\n[{\'Q345501\': {\'CLLI\': \'PHNZAZ\', \'SYSTEM_NAME\': \'PHNZAZ -635696-01\'}}]\n```\n\nNow, if you were to remove the "NESTED_DICT" magic ENUM mapping, the results for each row would look like this:\n\n```json\n[{\'CLLI\': \'PHNZAZ\', \'SYSTEM_NAME\': \'PHNZAZ -635696-01\'}]\n```\n\nIt\'s important to note, in our example below we tell Open Pyxl to start from "min_row", "2". Otherwise, the list will start generating the list of dictionaries from row 1, which are the headers. This allows you the flexibility to pin point areas of a spread sheet which matter to your workflow.\n\nAvailable options for identifying sections of interest in a spreadsheet:\n\n- min row\n\n- max row\n\n- min col\n\n- max col\n\n```python\nfrom nornir_pyxl.plugins.tasks import pyxl_data_map\nfrom nornir_utils.plugins.functions import print_result\nfrom nornir import InitNornir\nfrom enum import Enum\n\n\nclass DataMap(Enum):\n    """Enum Class Mapping Desired Cells Of Interest by Index within their respective row."""\n\n    NESTED_DICT = 0\n    CLLI = 1\n    SYSTEM_NAME = 2\n\n\nWORKBOOK = "working_example_wb.xlsx"\n\n\nnr = InitNornir("config.yml")\n\n\ndef get_mapped_data(task):\n    """Get a list of dictionaries from Excel Spreadsheet with Nornir Pyxl."""\n    res = task.run(task=pyxl_data_map,\n        workbook=WORKBOOK,\n        sheetname="IP_DATA",\n        mapping=DataMap,\n        min_row=2\n    )\n\ndef main():\n    """Execute Tasks."""\n    print_result(nr.run(task=get_mapped_data))\n\n\nif __name__ == "__main__":\n    main()\n```\n\nThe following output is the result of the above tasks:\n\n```bash\n❯ python3 test_pyxl.py\nget_mapped_data*****************************************************************\n* r1-csr ** changed : False ****************************************************\nvvvv get_mapped_data ** changed : False vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv INFO\n---- pyxl_data_map ** changed : False ------------------------------------------ INFO\n[{\'Q345501\': {\'CLLI\': \'PHNZAZ\', \'SYSTEM_NAME\': \'PHNZAZ -635696-01\'}}]\n^^^^ END get_mapped_data ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n```\n',
    'author': 'Hugo',
    'author_email': 'hugotinoco@icloud.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/h4ndzdatm0ld/nornir_pyxl',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
