"""
Preprocessor for Foliant documentation authoring tool.
Generates image captions using alternative texts for images.
"""

import re
from pathlib import Path

from foliant.preprocessors.base import BasePreprocessor


class Preprocessor(BasePreprocessor):
    defaults = {
        'stylesheet_path': Path('imgcaptions.css'),
        'template': '<p class="image_caption">{caption}</p>',
        'targets': [],
    }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        self.logger = self.logger.getChild('imgcaptions')

        self.logger.debug(f'Preprocessor inited: {self.__dict__}')

        self._stylesheet = self._get_stylesheet(Path(self.options['stylesheet_path']))

    def _get_stylesheet(self, stylesheet_file_path: Path) -> str:
        self.logger.debug(f'Stylesheet file path: {stylesheet_file_path}')

        if stylesheet_file_path.exists():
            self.logger.debug('Using stylesheet from the file')

            with open(stylesheet_file_path, encoding='utf8') as stylesheet_file:
                return stylesheet_file.read()

        else:
            self.logger.debug('Stylesheet file does not exist; using default stylesheet')

            return '''
.image_caption {
    font-size: .8em;
    font-style: italic;
    text-align: right;
    }
'''

    def process_captions(self, content: str) -> str:
        _image_pattern = re.compile(r'!\[(?P<caption>.+)]\((?P<path>.+)\)')
        _front_matter_pattern = re.compile(r"(\A[\s]*---[\s\S]+?---|\A)")
        caption_str = self.options['template'].format(caption=r'\g<caption>')
        content = re.sub(
            _image_pattern,
            r"![\g<caption>](\g<path>)\n\n" +
            caption_str,
            content
        )
        style_insert = f"\\g<1>\\n<style>\n{self._stylesheet}\n</style>\n\n"
        content = re.sub(_front_matter_pattern, style_insert, content)

        self.logger.debug('Content modified')

        return content

    def apply(self):
        self.logger.info('Applying preprocessor')

        self.logger.debug(f'Allowed targets: {self.options["targets"]}')
        self.logger.debug(f'Current target: {self.context["target"]}')

        if not self.options['targets'] or self.context['target'] in self.options['targets']:
            for markdown_file_path in self.working_dir.rglob('*.md'):
                self.logger.debug(f'Processing Markdown file: {markdown_file_path}')

                with open(markdown_file_path, encoding='utf8') as markdown_file:
                    content = markdown_file.read()

                processed_content = self.process_captions(content)

                if processed_content:
                    with open(markdown_file_path, 'w', encoding='utf8') as markdown_file:
                        markdown_file.write(processed_content)

        self.logger.info('Preprocessor applied')
