"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QueueProcessingServiceBase = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * The base class for QueueProcessingEc2Service and QueueProcessingFargateService services.
 */
class QueueProcessingServiceBase extends core_2.Construct {
    /**
     * Constructs a new instance of the QueueProcessingServiceBase class.
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ecs_patterns_QueueProcessingServiceBaseProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        // Create the SQS queue and it's corresponding DLQ if one is not provided
        if (props.queue) {
            this.sqsQueue = props.queue;
        }
        else {
            this.deadLetterQueue = new aws_sqs_1.Queue(this, 'EcsProcessingDeadLetterQueue', {
                retentionPeriod: props.retentionPeriod || core_1.Duration.days(14),
            });
            this.sqsQueue = new aws_sqs_1.Queue(this, 'EcsProcessingQueue', {
                visibilityTimeout: props.visibilityTimeout,
                deadLetterQueue: {
                    queue: this.deadLetterQueue,
                    maxReceiveCount: props.maxReceiveCount || 3,
                },
            });
            new core_1.CfnOutput(this, 'SQSDeadLetterQueue', { value: this.deadLetterQueue.queueName });
            new core_1.CfnOutput(this, 'SQSDeadLetterQueueArn', { value: this.deadLetterQueue.queueArn });
        }
        // Setup autoscaling scaling intervals
        const defaultScalingSteps = [{ upper: 0, change: -1 }, { lower: 100, change: +1 }, { lower: 500, change: +5 }];
        this.scalingSteps = (_b = props.scalingSteps) !== null && _b !== void 0 ? _b : defaultScalingSteps;
        // Create log driver if logging is enabled
        const enableLogging = (_c = props.enableLogging) !== null && _c !== void 0 ? _c : true;
        this.logDriver = (_d = props.logDriver) !== null && _d !== void 0 ? _d : (enableLogging ? this.createAWSLogDriver(this.node.id) : undefined);
        // Add the queue name to environment variables
        this.environment = { ...(props.environment || {}), QUEUE_NAME: this.sqsQueue.queueName };
        this.secrets = props.secrets;
        this.desiredCount = (_e = props.desiredTaskCount) !== null && _e !== void 0 ? _e : 1;
        // Determine the desired task count (minimum) and maximum scaling capacity
        if (!this.node.tryGetContext(cxapi.ECS_REMOVE_DEFAULT_DESIRED_COUNT)) {
            this.minCapacity = (_f = props.minScalingCapacity) !== null && _f !== void 0 ? _f : this.desiredCount;
            this.maxCapacity = props.maxScalingCapacity || (2 * this.desiredCount);
        }
        else {
            if (props.desiredTaskCount != null) {
                this.minCapacity = (_g = props.minScalingCapacity) !== null && _g !== void 0 ? _g : this.desiredCount;
                this.maxCapacity = props.maxScalingCapacity || (2 * this.desiredCount);
            }
            else {
                this.minCapacity = (_h = props.minScalingCapacity) !== null && _h !== void 0 ? _h : 1;
                this.maxCapacity = props.maxScalingCapacity || 2;
            }
        }
        if (!this.desiredCount && !this.maxCapacity) {
            throw new Error('maxScalingCapacity must be set and greater than 0 if desiredCount is 0');
        }
        new core_1.CfnOutput(this, 'SQSQueue', { value: this.sqsQueue.queueName });
        new core_1.CfnOutput(this, 'SQSQueueArn', { value: this.sqsQueue.queueArn });
    }
    /**
     * Configure autoscaling based off of CPU utilization as well as the number of messages visible in the SQS queue
     *
     * @param service the ECS/Fargate service for which to apply the autoscaling rules to
     */
    configureAutoscalingForService(service) {
        const scalingTarget = service.autoScaleTaskCount({ maxCapacity: this.maxCapacity, minCapacity: this.minCapacity });
        scalingTarget.scaleOnCpuUtilization('CpuScaling', {
            targetUtilizationPercent: 50,
        });
        scalingTarget.scaleOnMetric('QueueMessagesVisibleScaling', {
            metric: this.sqsQueue.metricApproximateNumberOfMessagesVisible(),
            scalingSteps: this.scalingSteps,
        });
    }
    /**
     * Grant SQS permissions to an ECS service.
     * @param service the ECS/Fargate service to which to grant SQS permissions
     */
    grantPermissionsToService(service) {
        this.sqsQueue.grantConsumeMessages(service.taskDefinition.taskRole);
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Create an AWS Log Driver with the provided streamPrefix
     *
     * @param prefix the Cloudwatch logging prefix
     */
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.QueueProcessingServiceBase = QueueProcessingServiceBase;
_a = JSII_RTTI_SYMBOL_1;
QueueProcessingServiceBase[_a] = { fqn: "@aws-cdk/aws-ecs-patterns.QueueProcessingServiceBase", version: "1.157.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicXVldWUtcHJvY2Vzc2luZy1zZXJ2aWNlLWJhc2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJxdWV1ZS1wcm9jZXNzaW5nLXNlcnZpY2UtYmFzZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFFQSw4Q0FHMEI7QUFDMUIsOENBQWlEO0FBQ2pELHdDQUEyRDtBQUMzRCx5Q0FBeUM7QUFHekMsZ0hBQWdIO0FBQ2hILDJCQUEyQjtBQUMzQix3Q0FBMkQ7QUE4TTNEOztHQUVHO0FBQ0gsTUFBc0IsMEJBQTJCLFNBQVEsZ0JBQWE7SUFzRHBFOztPQUVHO0lBQ0gsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFzQzs7UUFDOUUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQzs7Ozs7Ozs7OztRQUVqQixJQUFJLEtBQUssQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDLEdBQUcsRUFBRTtZQUM5QixNQUFNLElBQUksS0FBSyxDQUFDLHFGQUFxRixDQUFDLENBQUM7U0FDeEc7UUFDRCxJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLElBQUksSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFeEUseUVBQXlFO1FBQ3pFLElBQUksS0FBSyxDQUFDLEtBQUssRUFBRTtZQUNmLElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQztTQUM3QjthQUFNO1lBQ0wsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLGVBQUssQ0FBQyxJQUFJLEVBQUUsOEJBQThCLEVBQUU7Z0JBQ3JFLGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZSxJQUFJLGVBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO2FBQzVELENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxlQUFLLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFO2dCQUNwRCxpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCO2dCQUMxQyxlQUFlLEVBQUU7b0JBQ2YsS0FBSyxFQUFFLElBQUksQ0FBQyxlQUFlO29CQUMzQixlQUFlLEVBQUUsS0FBSyxDQUFDLGVBQWUsSUFBSSxDQUFDO2lCQUM1QzthQUNGLENBQUMsQ0FBQztZQUVILElBQUksZ0JBQVMsQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxTQUFTLEVBQUUsQ0FBQyxDQUFDO1lBQ3JGLElBQUksZ0JBQVMsQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1NBQ3hGO1FBRUQsc0NBQXNDO1FBQ3RDLE1BQU0sbUJBQW1CLEdBQUcsQ0FBQyxFQUFFLEtBQUssRUFBRSxDQUFDLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsR0FBRyxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLEdBQUcsRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQy9HLElBQUksQ0FBQyxZQUFZLFNBQUcsS0FBSyxDQUFDLFlBQVksbUNBQUksbUJBQW1CLENBQUM7UUFFOUQsMENBQTBDO1FBQzFDLE1BQU0sYUFBYSxTQUFHLEtBQUssQ0FBQyxhQUFhLG1DQUFJLElBQUksQ0FBQztRQUNsRCxJQUFJLENBQUMsU0FBUyxTQUFHLEtBQUssQ0FBQyxTQUFTLG1DQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFeEcsOENBQThDO1FBQzlDLElBQUksQ0FBQyxXQUFXLEdBQUcsRUFBRSxHQUFHLENBQUMsS0FBSyxDQUFDLFdBQVcsSUFBSSxFQUFFLENBQUMsRUFBRSxVQUFVLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsQ0FBQztRQUN6RixJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFFN0IsSUFBSSxDQUFDLFlBQVksU0FBRyxLQUFLLENBQUMsZ0JBQWdCLG1DQUFJLENBQUMsQ0FBQztRQUVoRCwwRUFBMEU7UUFDMUUsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxnQ0FBZ0MsQ0FBQyxFQUFFO1lBQ3BFLElBQUksQ0FBQyxXQUFXLFNBQUcsS0FBSyxDQUFDLGtCQUFrQixtQ0FBSSxJQUFJLENBQUMsWUFBWSxDQUFDO1lBQ2pFLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLENBQUMsQ0FBQyxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUN4RTthQUFNO1lBQ0wsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLElBQUksSUFBSSxFQUFFO2dCQUNsQyxJQUFJLENBQUMsV0FBVyxTQUFHLEtBQUssQ0FBQyxrQkFBa0IsbUNBQUksSUFBSSxDQUFDLFlBQVksQ0FBQztnQkFDakUsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsa0JBQWtCLElBQUksQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO2FBQ3hFO2lCQUFNO2dCQUNMLElBQUksQ0FBQyxXQUFXLFNBQUcsS0FBSyxDQUFDLGtCQUFrQixtQ0FBSSxDQUFDLENBQUM7Z0JBQ2pELElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLENBQUMsQ0FBQzthQUNsRDtTQUNGO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsd0VBQXdFLENBQUMsQ0FBQztTQUMzRjtRQUVELElBQUksZ0JBQVMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUNwRSxJQUFJLGdCQUFTLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRSxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7S0FDdkU7SUFFRDs7OztPQUlHO0lBQ08sOEJBQThCLENBQUMsT0FBb0I7UUFDM0QsTUFBTSxhQUFhLEdBQUcsT0FBTyxDQUFDLGtCQUFrQixDQUFDLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO1FBQ25ILGFBQWEsQ0FBQyxxQkFBcUIsQ0FBQyxZQUFZLEVBQUU7WUFDaEQsd0JBQXdCLEVBQUUsRUFBRTtTQUM3QixDQUFDLENBQUM7UUFDSCxhQUFhLENBQUMsYUFBYSxDQUFDLDZCQUE2QixFQUFFO1lBQ3pELE1BQU0sRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLHdDQUF3QyxFQUFFO1lBQ2hFLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtTQUNoQyxDQUFDLENBQUM7S0FDSjtJQUVEOzs7T0FHRztJQUNPLHlCQUF5QixDQUFDLE9BQW9CO1FBQ3RELElBQUksQ0FBQyxRQUFRLENBQUMsb0JBQW9CLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQztLQUNyRTtJQUVEOztPQUVHO0lBQ08saUJBQWlCLENBQUMsS0FBZ0IsRUFBRSxHQUFVO1FBQ3RELCtEQUErRDtRQUMvRCxNQUFNLGtCQUFrQixHQUFHLDZCQUE2QixHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQztRQUNqRixNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLE9BQU8sS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsa0JBQWtCLENBQVksSUFBSSxJQUFJLGlCQUFPLENBQUMsS0FBSyxFQUFFLGtCQUFrQixFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQztLQUNsSDtJQUVEOzs7O09BSUc7SUFDSyxrQkFBa0IsQ0FBQyxNQUFjO1FBQ3ZDLE9BQU8sSUFBSSxzQkFBWSxDQUFDLEVBQUUsWUFBWSxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUM7S0FDbkQ7O0FBaktILGdFQWtLQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IFNjYWxpbmdJbnRlcnZhbCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1hcHBsaWNhdGlvbmF1dG9zY2FsaW5nJztcbmltcG9ydCB7IElWcGMgfSBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCB7XG4gIEF3c0xvZ0RyaXZlciwgQmFzZVNlcnZpY2UsIENhcGFjaXR5UHJvdmlkZXJTdHJhdGVneSwgQ2x1c3RlciwgQ29udGFpbmVySW1hZ2UsIERlcGxveW1lbnRDb250cm9sbGVyLCBEZXBsb3ltZW50Q2lyY3VpdEJyZWFrZXIsXG4gIElDbHVzdGVyLCBMb2dEcml2ZXIsIFByb3BhZ2F0ZWRUYWdTb3VyY2UsIFNlY3JldCxcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjcyc7XG5pbXBvcnQgeyBJUXVldWUsIFF1ZXVlIH0gZnJvbSAnQGF3cy1jZGsvYXdzLXNxcyc7XG5pbXBvcnQgeyBDZm5PdXRwdXQsIER1cmF0aW9uLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgY3hhcGkgZnJvbSAnQGF3cy1jZGsvY3gtYXBpJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG4vLyB2MiAtIGtlZXAgdGhpcyBpbXBvcnQgYXMgYSBzZXBhcmF0ZSBzZWN0aW9uIHRvIHJlZHVjZSBtZXJnZSBjb25mbGljdCB3aGVuIGZvcndhcmQgbWVyZ2luZyB3aXRoIHRoZSB2MiBicmFuY2guXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmVcbmltcG9ydCB7IENvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbi8qKlxuICogVGhlIHByb3BlcnRpZXMgZm9yIHRoZSBiYXNlIFF1ZXVlUHJvY2Vzc2luZ0VjMlNlcnZpY2Ugb3IgUXVldWVQcm9jZXNzaW5nRmFyZ2F0ZVNlcnZpY2Ugc2VydmljZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBRdWV1ZVByb2Nlc3NpbmdTZXJ2aWNlQmFzZVByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBzZXJ2aWNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIENsb3VkRm9ybWF0aW9uLWdlbmVyYXRlZCBuYW1lLlxuICAgKi9cbiAgcmVhZG9ubHkgc2VydmljZU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBjbHVzdGVyIHRoYXQgaG9zdHMgdGhlIHNlcnZpY2UuXG4gICAqXG4gICAqIElmIGEgY2x1c3RlciBpcyBzcGVjaWZpZWQsIHRoZSB2cGMgY29uc3RydWN0IHNob3VsZCBiZSBvbWl0dGVkLiBBbHRlcm5hdGl2ZWx5LCB5b3UgY2FuIG9taXQgYm90aCBjbHVzdGVyIGFuZCB2cGMuXG4gICAqIEBkZWZhdWx0IC0gY3JlYXRlIGEgbmV3IGNsdXN0ZXI7IGlmIGJvdGggY2x1c3RlciBhbmQgdnBjIGFyZSBvbWl0dGVkLCBhIG5ldyBWUEMgd2lsbCBiZSBjcmVhdGVkIGZvciB5b3UuXG4gICAqL1xuICByZWFkb25seSBjbHVzdGVyPzogSUNsdXN0ZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgd2hlcmUgdGhlIGNvbnRhaW5lciBpbnN0YW5jZXMgd2lsbCBiZSBsYXVuY2hlZCBvciB0aGUgZWxhc3RpYyBuZXR3b3JrIGludGVyZmFjZXMgKEVOSXMpIHdpbGwgYmUgZGVwbG95ZWQuXG4gICAqXG4gICAqIElmIGEgdnBjIGlzIHNwZWNpZmllZCwgdGhlIGNsdXN0ZXIgY29uc3RydWN0IHNob3VsZCBiZSBvbWl0dGVkLiBBbHRlcm5hdGl2ZWx5LCB5b3UgY2FuIG9taXQgYm90aCB2cGMgYW5kIGNsdXN0ZXIuXG4gICAqIEBkZWZhdWx0IC0gdXNlcyB0aGUgVlBDIGRlZmluZWQgaW4gdGhlIGNsdXN0ZXIgb3IgY3JlYXRlcyBhIG5ldyBWUEMuXG4gICAqL1xuICByZWFkb25seSB2cGM/OiBJVnBjO1xuXG4gIC8qKlxuICAgKiBUaGUgaW1hZ2UgdXNlZCB0byBzdGFydCBhIGNvbnRhaW5lci5cbiAgICovXG4gIHJlYWRvbmx5IGltYWdlOiBDb250YWluZXJJbWFnZTtcblxuICAvKipcbiAgICogVGhlIGNvbW1hbmQgdGhhdCBpcyBwYXNzZWQgdG8gdGhlIGNvbnRhaW5lci5cbiAgICpcbiAgICogSWYgeW91IHByb3ZpZGUgYSBzaGVsbCBjb21tYW5kIGFzIGEgc2luZ2xlIHN0cmluZywgeW91IGhhdmUgdG8gcXVvdGUgY29tbWFuZC1saW5lIGFyZ3VtZW50cy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBDTUQgdmFsdWUgYnVpbHQgaW50byBjb250YWluZXIgaW1hZ2UuXG4gICAqL1xuICByZWFkb25seSBjb21tYW5kPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFRoZSBkZXNpcmVkIG51bWJlciBvZiBpbnN0YW50aWF0aW9ucyBvZiB0aGUgdGFzayBkZWZpbml0aW9uIHRvIGtlZXAgcnVubmluZyBvbiB0aGUgc2VydmljZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBJZiB0aGUgZmVhdHVyZSBmbGFnLCBFQ1NfUkVNT1ZFX0RFRkFVTFRfREVTSVJFRF9DT1VOVCBpcyBmYWxzZSwgdGhlIGRlZmF1bHQgaXMgMTtcbiAgICogaWYgdHJ1ZSwgdGhlIG1pblNjYWxpbmdDYXBhY2l0eSBpcyAxIGZvciBhbGwgbmV3IHNlcnZpY2VzIGFuZCB1c2VzIHRoZSBleGlzdGluZyBzZXJ2aWNlcyBkZXNpcmVkIGNvdW50XG4gICAqIHdoZW4gdXBkYXRpbmcgYW4gZXhpc3Rpbmcgc2VydmljZS5cbiAgICogQGRlcHJlY2F0ZWQgLSBVc2UgYG1pblNjYWxpbmdDYXBhY2l0eWAgb3IgYSBsaXRlcmFsIG9iamVjdCBpbnN0ZWFkLlxuICAgKi9cbiAgcmVhZG9ubHkgZGVzaXJlZFRhc2tDb3VudD86IG51bWJlcjtcblxuICAvKipcbiAgICogRmxhZyB0byBpbmRpY2F0ZSB3aGV0aGVyIHRvIGVuYWJsZSBsb2dnaW5nLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBlbmFibGVMb2dnaW5nPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIGVudmlyb25tZW50IHZhcmlhYmxlcyB0byBwYXNzIHRvIHRoZSBjb250YWluZXIuXG4gICAqXG4gICAqIFRoZSB2YXJpYWJsZSBgUVVFVUVfTkFNRWAgd2l0aCB2YWx1ZSBgcXVldWUucXVldWVOYW1lYCB3aWxsXG4gICAqIGFsd2F5cyBiZSBwYXNzZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0ICdRVUVVRV9OQU1FOiBxdWV1ZS5xdWV1ZU5hbWUnXG4gICAqL1xuICByZWFkb25seSBlbnZpcm9ubWVudD86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgLyoqXG4gICAqIFRoZSBzZWNyZXQgdG8gZXhwb3NlIHRvIHRoZSBjb250YWluZXIgYXMgYW4gZW52aXJvbm1lbnQgdmFyaWFibGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gc2VjcmV0IGVudmlyb25tZW50IHZhcmlhYmxlcy5cbiAgICovXG4gIHJlYWRvbmx5IHNlY3JldHM/OiB7IFtrZXk6IHN0cmluZ106IFNlY3JldCB9O1xuXG4gIC8qKlxuICAgKiBBIHF1ZXVlIGZvciB3aGljaCB0byBwcm9jZXNzIGl0ZW1zIGZyb20uXG4gICAqXG4gICAqIElmIHNwZWNpZmllZCBhbmQgdGhpcyBpcyBhIEZJRk8gcXVldWUsIHRoZSBxdWV1ZSBuYW1lIG11c3QgZW5kIGluIHRoZSBzdHJpbmcgJy5maWZvJy4gU2VlXG4gICAqIFtDcmVhdGVRdWV1ZV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU1NpbXBsZVF1ZXVlU2VydmljZS9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVRdWV1ZS5odG1sKVxuICAgKlxuICAgKiBAZGVmYXVsdCAnU1FTUXVldWUgd2l0aCBDbG91ZEZvcm1hdGlvbi1nZW5lcmF0ZWQgbmFtZSdcbiAgICovXG4gIHJlYWRvbmx5IHF1ZXVlPzogSVF1ZXVlO1xuXG4gIC8qKlxuICAgKiBUaGUgbWF4aW11bSBudW1iZXIgb2YgdGltZXMgdGhhdCBhIG1lc3NhZ2UgY2FuIGJlIHJlY2VpdmVkIGJ5IGNvbnN1bWVycy5cbiAgICogV2hlbiB0aGlzIHZhbHVlIGlzIGV4Y2VlZGVkIGZvciBhIG1lc3NhZ2UgdGhlIG1lc3NhZ2Ugd2lsbCBiZSBhdXRvbWF0aWNhbGx5IHNlbnQgdG8gdGhlIERlYWQgTGV0dGVyIFF1ZXVlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAzXG4gICAqL1xuICByZWFkb25seSBtYXhSZWNlaXZlQ291bnQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRpbWVvdXQgb2YgcHJvY2Vzc2luZyBhIHNpbmdsZSBtZXNzYWdlLiBBZnRlciBkZXF1ZXVpbmcsIHRoZSBwcm9jZXNzb3IgaGFzIHRoaXMgbXVjaCB0aW1lIHRvIGhhbmRsZSB0aGUgbWVzc2FnZSBhbmQgZGVsZXRlIGl0IGZyb20gdGhlIHF1ZXVlXG4gICAqIGJlZm9yZSBpdCBiZWNvbWVzIHZpc2libGUgYWdhaW4gZm9yIGRlcXVldWVpbmcgYnkgYW5vdGhlciBwcm9jZXNzb3IuIFZhbHVlcyBtdXN0IGJlIGJldHdlZW4gMCBhbmQgKDEyIGhvdXJzKS5cbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24uc2Vjb25kcygzMClcbiAgICovXG4gIHJlYWRvbmx5IHZpc2liaWxpdHlUaW1lb3V0PzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2Ygc2Vjb25kcyB0aGF0IERlYWQgTGV0dGVyIFF1ZXVlIHJldGFpbnMgYSBtZXNzYWdlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5kYXlzKDE0KVxuICAgKi9cbiAgcmVhZG9ubHkgcmV0ZW50aW9uUGVyaW9kPzogRHVyYXRpb247XG5cbiAgLyoqXG4gICAqIE1heGltdW0gY2FwYWNpdHkgdG8gc2NhbGUgdG8uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSWYgdGhlIGZlYXR1cmUgZmxhZywgRUNTX1JFTU9WRV9ERUZBVUxUX0RFU0lSRURfQ09VTlQgaXMgZmFsc2UsIHRoZSBkZWZhdWx0IGlzIChkZXNpcmVkVGFza0NvdW50ICogMik7IGlmIHRydWUsIHRoZSBkZWZhdWx0IGlzIDIuXG4gICAqL1xuICByZWFkb25seSBtYXhTY2FsaW5nQ2FwYWNpdHk/OiBudW1iZXJcblxuICAvKipcbiAgICogTWluaW11bSBjYXBhY2l0eSB0byBzY2FsZSB0by5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBJZiB0aGUgZmVhdHVyZSBmbGFnLCBFQ1NfUkVNT1ZFX0RFRkFVTFRfREVTSVJFRF9DT1VOVCBpcyBmYWxzZSwgdGhlIGRlZmF1bHQgaXMgdGhlIGRlc2lyZWRUYXNrQ291bnQ7IGlmIHRydWUsIHRoZSBkZWZhdWx0IGlzIDEuXG4gICAqL1xuICByZWFkb25seSBtaW5TY2FsaW5nQ2FwYWNpdHk/OiBudW1iZXJcblxuICAvKipcbiAgICogVGhlIGludGVydmFscyBmb3Igc2NhbGluZyBiYXNlZCBvbiB0aGUgU1FTIHF1ZXVlJ3MgQXBwcm94aW1hdGVOdW1iZXJPZk1lc3NhZ2VzVmlzaWJsZSBtZXRyaWMuXG4gICAqXG4gICAqIE1hcHMgYSByYW5nZSBvZiBtZXRyaWMgdmFsdWVzIHRvIGEgcGFydGljdWxhciBzY2FsaW5nIGJlaGF2aW9yLiBTZWVcbiAgICogW1NpbXBsZSBhbmQgU3RlcCBTY2FsaW5nIFBvbGljaWVzIGZvciBBbWF6b24gRUMyIEF1dG8gU2NhbGluZ10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2F1dG9zY2FsaW5nL2VjMi91c2VyZ3VpZGUvYXMtc2NhbGluZy1zaW1wbGUtc3RlcC5odG1sKVxuICAgKlxuICAgKiBAZGVmYXVsdCBbeyB1cHBlcjogMCwgY2hhbmdlOiAtMSB9LHsgbG93ZXI6IDEwMCwgY2hhbmdlOiArMSB9LHsgbG93ZXI6IDUwMCwgY2hhbmdlOiArNSB9XVxuICAgKi9cbiAgcmVhZG9ubHkgc2NhbGluZ1N0ZXBzPzogU2NhbGluZ0ludGVydmFsW107XG5cbiAgLyoqXG4gICAqIFRoZSBsb2cgZHJpdmVyIHRvIHVzZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBd3NMb2dEcml2ZXIgaWYgZW5hYmxlTG9nZ2luZyBpcyB0cnVlXG4gICAqL1xuICByZWFkb25seSBsb2dEcml2ZXI/OiBMb2dEcml2ZXI7XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB3aGV0aGVyIHRvIHByb3BhZ2F0ZSB0aGUgdGFncyBmcm9tIHRoZSB0YXNrIGRlZmluaXRpb24gb3IgdGhlIHNlcnZpY2UgdG8gdGhlIHRhc2tzIGluIHRoZSBzZXJ2aWNlLlxuICAgKiBUYWdzIGNhbiBvbmx5IGJlIHByb3BhZ2F0ZWQgdG8gdGhlIHRhc2tzIHdpdGhpbiB0aGUgc2VydmljZSBkdXJpbmcgc2VydmljZSBjcmVhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBub25lXG4gICAqL1xuICByZWFkb25seSBwcm9wYWdhdGVUYWdzPzogUHJvcGFnYXRlZFRhZ1NvdXJjZTtcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHdoZXRoZXIgdG8gZW5hYmxlIEFtYXpvbiBFQ1MgbWFuYWdlZCB0YWdzIGZvciB0aGUgdGFza3Mgd2l0aGluIHRoZSBzZXJ2aWNlLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlXG4gICAqIFtUYWdnaW5nIFlvdXIgQW1hem9uIEVDUyBSZXNvdXJjZXNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2Vjcy11c2luZy10YWdzLmh0bWwpXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBlbmFibGVFQ1NNYW5hZ2VkVGFncz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIGEgZmFtaWx5IHRoYXQgdGhlIHRhc2sgZGVmaW5pdGlvbiBpcyByZWdpc3RlcmVkIHRvLiBBIGZhbWlseSBncm91cHMgbXVsdGlwbGUgdmVyc2lvbnMgb2YgYSB0YXNrIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQXV0b21hdGljYWxseSBnZW5lcmF0ZWQgbmFtZS5cbiAgICovXG4gIHJlYWRvbmx5IGZhbWlseT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gbnVtYmVyIG9mIHRhc2tzLCBzcGVjaWZpZWQgYXMgYSBwZXJjZW50YWdlIG9mIHRoZSBBbWF6b24gRUNTXG4gICAqIHNlcnZpY2UncyBEZXNpcmVkQ291bnQgdmFsdWUsIHRoYXQgY2FuIHJ1biBpbiBhIHNlcnZpY2UgZHVyaW5nIGFcbiAgICogZGVwbG95bWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBkZWZhdWx0IGZyb20gdW5kZXJseWluZyBzZXJ2aWNlLlxuICAgKi9cbiAgcmVhZG9ubHkgbWF4SGVhbHRoeVBlcmNlbnQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtaW5pbXVtIG51bWJlciBvZiB0YXNrcywgc3BlY2lmaWVkIGFzIGEgcGVyY2VudGFnZSBvZlxuICAgKiB0aGUgQW1hem9uIEVDUyBzZXJ2aWNlJ3MgRGVzaXJlZENvdW50IHZhbHVlLCB0aGF0IG11c3RcbiAgICogY29udGludWUgdG8gcnVuIGFuZCByZW1haW4gaGVhbHRoeSBkdXJpbmcgYSBkZXBsb3ltZW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGRlZmF1bHQgZnJvbSB1bmRlcmx5aW5nIHNlcnZpY2UuXG4gICAqL1xuICByZWFkb25seSBtaW5IZWFsdGh5UGVyY2VudD86IG51bWJlcjtcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHdoaWNoIGRlcGxveW1lbnQgY29udHJvbGxlciB0byB1c2UgZm9yIHRoZSBzZXJ2aWNlLiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlXG4gICAqIFtBbWF6b24gRUNTIERlcGxveW1lbnQgVHlwZXNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25FQ1MvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2RlcGxveW1lbnQtdHlwZXMuaHRtbClcbiAgICpcbiAgICogQGRlZmF1bHQgLSBSb2xsaW5nIHVwZGF0ZSAoRUNTKVxuICAgKi9cbiAgcmVhZG9ubHkgZGVwbG95bWVudENvbnRyb2xsZXI/OiBEZXBsb3ltZW50Q29udHJvbGxlcjtcblxuICAvKipcbiAgICogV2hldGhlciB0byBlbmFibGUgdGhlIGRlcGxveW1lbnQgY2lyY3VpdCBicmVha2VyLiBJZiB0aGlzIHByb3BlcnR5IGlzIGRlZmluZWQsIGNpcmN1aXQgYnJlYWtlciB3aWxsIGJlIGltcGxpY2l0bHlcbiAgICogZW5hYmxlZC5cbiAgICogQGRlZmF1bHQgLSBkaXNhYmxlZFxuICAgKi9cbiAgcmVhZG9ubHkgY2lyY3VpdEJyZWFrZXI/OiBEZXBsb3ltZW50Q2lyY3VpdEJyZWFrZXI7XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBDYXBhY2l0eSBQcm92aWRlciBzdHJhdGVnaWVzIHVzZWQgdG8gcGxhY2UgYSBzZXJ2aWNlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHVuZGVmaW5lZFxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgY2FwYWNpdHlQcm92aWRlclN0cmF0ZWdpZXM/OiBDYXBhY2l0eVByb3ZpZGVyU3RyYXRlZ3lbXTtcbn1cblxuLyoqXG4gKiBUaGUgYmFzZSBjbGFzcyBmb3IgUXVldWVQcm9jZXNzaW5nRWMyU2VydmljZSBhbmQgUXVldWVQcm9jZXNzaW5nRmFyZ2F0ZVNlcnZpY2Ugc2VydmljZXMuXG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBRdWV1ZVByb2Nlc3NpbmdTZXJ2aWNlQmFzZSBleHRlbmRzIENvcmVDb25zdHJ1Y3Qge1xuICAvKipcbiAgICogVGhlIFNRUyBxdWV1ZSB0aGF0IHRoZSBzZXJ2aWNlIHdpbGwgcHJvY2VzcyBmcm9tXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3FzUXVldWU6IElRdWV1ZTtcblxuICAvKipcbiAgICogVGhlIGRlYWQgbGV0dGVyIHF1ZXVlIGZvciB0aGUgcHJpbWFyeSBTUVMgcXVldWVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZWFkTGV0dGVyUXVldWU/OiBJUXVldWU7XG5cbiAgLyoqXG4gICAqIFRoZSBjbHVzdGVyIHdoZXJlIHlvdXIgc2VydmljZSB3aWxsIGJlIGRlcGxveWVkXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY2x1c3RlcjogSUNsdXN0ZXI7XG5cbiAgLy8gUHJvcGVydGllcyB0aGF0IGhhdmUgZGVmYXVsdHMgZGVmaW5lZC4gVGhlIFF1ZXVlIFByb2Nlc3NpbmcgU2VydmljZSB3aWxsIGhhbmRsZSBhc3NpZ25pbmcgdW5kZWZpbmVkIHByb3BlcnRpZXMgd2l0aCBkZWZhdWx0XG4gIC8vIHZhbHVlcyBzbyB0aGF0IGRlcml2ZWQgY2xhc3NlcyBkbyBub3QgbmVlZCB0byBtYWludGFpbiB0aGUgc2FtZSBsb2dpYy5cblxuICAvKipcbiAgICogRW52aXJvbm1lbnQgdmFyaWFibGVzIHRoYXQgd2lsbCBpbmNsdWRlIHRoZSBxdWV1ZSBuYW1lXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZW52aXJvbm1lbnQ6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgLyoqXG4gICAqIFRoZSBzZWNyZXQgZW52aXJvbm1lbnQgdmFyaWFibGVzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHNlY3JldHM/OiB7IFtrZXk6IHN0cmluZ106IFNlY3JldCB9O1xuXG4gIC8qKlxuICAgKiBUaGUgbWluaW11bSBudW1iZXIgb2YgdGFza3MgdG8gcnVuLlxuICAgKiBAZGVwcmVjYXRlZCAtIFVzZSBgbWluQ2FwYWNpdHlgIGluc3RlYWQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGVzaXJlZENvdW50OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIG51bWJlciBvZiBpbnN0YW5jZXMgZm9yIGF1dG9zY2FsaW5nIHRvIHNjYWxlIHVwIHRvLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG1heENhcGFjaXR5OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtaW5pbXVtIG51bWJlciBvZiBpbnN0YW5jZXMgZm9yIGF1dG9zY2FsaW5nIHRvIHNjYWxlIGRvd24gdG8uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbWluQ2FwYWNpdHk6IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIHNjYWxpbmcgaW50ZXJ2YWwgZm9yIGF1dG9zY2FsaW5nIGJhc2VkIG9mZiBhbiBTUVMgUXVldWUgc2l6ZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzY2FsaW5nU3RlcHM6IFNjYWxpbmdJbnRlcnZhbFtdO1xuICAvKipcbiAgICogVGhlIEF3c0xvZ0RyaXZlciB0byB1c2UgZm9yIGxvZ2dpbmcgaWYgbG9nZ2luZyBpcyBlbmFibGVkLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGxvZ0RyaXZlcj86IExvZ0RyaXZlcjtcblxuICAvKipcbiAgICogQ29uc3RydWN0cyBhIG5ldyBpbnN0YW5jZSBvZiB0aGUgUXVldWVQcm9jZXNzaW5nU2VydmljZUJhc2UgY2xhc3MuXG4gICAqL1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUXVldWVQcm9jZXNzaW5nU2VydmljZUJhc2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBpZiAocHJvcHMuY2x1c3RlciAmJiBwcm9wcy52cGMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignWW91IGNhbiBvbmx5IHNwZWNpZnkgZWl0aGVyIHZwYyBvciBjbHVzdGVyLiBBbHRlcm5hdGl2ZWx5LCB5b3UgY2FuIGxlYXZlIGJvdGggYmxhbmsnKTtcbiAgICB9XG4gICAgdGhpcy5jbHVzdGVyID0gcHJvcHMuY2x1c3RlciB8fCB0aGlzLmdldERlZmF1bHRDbHVzdGVyKHRoaXMsIHByb3BzLnZwYyk7XG5cbiAgICAvLyBDcmVhdGUgdGhlIFNRUyBxdWV1ZSBhbmQgaXQncyBjb3JyZXNwb25kaW5nIERMUSBpZiBvbmUgaXMgbm90IHByb3ZpZGVkXG4gICAgaWYgKHByb3BzLnF1ZXVlKSB7XG4gICAgICB0aGlzLnNxc1F1ZXVlID0gcHJvcHMucXVldWU7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuZGVhZExldHRlclF1ZXVlID0gbmV3IFF1ZXVlKHRoaXMsICdFY3NQcm9jZXNzaW5nRGVhZExldHRlclF1ZXVlJywge1xuICAgICAgICByZXRlbnRpb25QZXJpb2Q6IHByb3BzLnJldGVudGlvblBlcmlvZCB8fCBEdXJhdGlvbi5kYXlzKDE0KSxcbiAgICAgIH0pO1xuICAgICAgdGhpcy5zcXNRdWV1ZSA9IG5ldyBRdWV1ZSh0aGlzLCAnRWNzUHJvY2Vzc2luZ1F1ZXVlJywge1xuICAgICAgICB2aXNpYmlsaXR5VGltZW91dDogcHJvcHMudmlzaWJpbGl0eVRpbWVvdXQsXG4gICAgICAgIGRlYWRMZXR0ZXJRdWV1ZToge1xuICAgICAgICAgIHF1ZXVlOiB0aGlzLmRlYWRMZXR0ZXJRdWV1ZSxcbiAgICAgICAgICBtYXhSZWNlaXZlQ291bnQ6IHByb3BzLm1heFJlY2VpdmVDb3VudCB8fCAzLFxuICAgICAgICB9LFxuICAgICAgfSk7XG5cbiAgICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ1NRU0RlYWRMZXR0ZXJRdWV1ZScsIHsgdmFsdWU6IHRoaXMuZGVhZExldHRlclF1ZXVlLnF1ZXVlTmFtZSB9KTtcbiAgICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ1NRU0RlYWRMZXR0ZXJRdWV1ZUFybicsIHsgdmFsdWU6IHRoaXMuZGVhZExldHRlclF1ZXVlLnF1ZXVlQXJuIH0pO1xuICAgIH1cblxuICAgIC8vIFNldHVwIGF1dG9zY2FsaW5nIHNjYWxpbmcgaW50ZXJ2YWxzXG4gICAgY29uc3QgZGVmYXVsdFNjYWxpbmdTdGVwcyA9IFt7IHVwcGVyOiAwLCBjaGFuZ2U6IC0xIH0sIHsgbG93ZXI6IDEwMCwgY2hhbmdlOiArMSB9LCB7IGxvd2VyOiA1MDAsIGNoYW5nZTogKzUgfV07XG4gICAgdGhpcy5zY2FsaW5nU3RlcHMgPSBwcm9wcy5zY2FsaW5nU3RlcHMgPz8gZGVmYXVsdFNjYWxpbmdTdGVwcztcblxuICAgIC8vIENyZWF0ZSBsb2cgZHJpdmVyIGlmIGxvZ2dpbmcgaXMgZW5hYmxlZFxuICAgIGNvbnN0IGVuYWJsZUxvZ2dpbmcgPSBwcm9wcy5lbmFibGVMb2dnaW5nID8/IHRydWU7XG4gICAgdGhpcy5sb2dEcml2ZXIgPSBwcm9wcy5sb2dEcml2ZXIgPz8gKGVuYWJsZUxvZ2dpbmcgPyB0aGlzLmNyZWF0ZUFXU0xvZ0RyaXZlcih0aGlzLm5vZGUuaWQpIDogdW5kZWZpbmVkKTtcblxuICAgIC8vIEFkZCB0aGUgcXVldWUgbmFtZSB0byBlbnZpcm9ubWVudCB2YXJpYWJsZXNcbiAgICB0aGlzLmVudmlyb25tZW50ID0geyAuLi4ocHJvcHMuZW52aXJvbm1lbnQgfHwge30pLCBRVUVVRV9OQU1FOiB0aGlzLnNxc1F1ZXVlLnF1ZXVlTmFtZSB9O1xuICAgIHRoaXMuc2VjcmV0cyA9IHByb3BzLnNlY3JldHM7XG5cbiAgICB0aGlzLmRlc2lyZWRDb3VudCA9IHByb3BzLmRlc2lyZWRUYXNrQ291bnQgPz8gMTtcblxuICAgIC8vIERldGVybWluZSB0aGUgZGVzaXJlZCB0YXNrIGNvdW50IChtaW5pbXVtKSBhbmQgbWF4aW11bSBzY2FsaW5nIGNhcGFjaXR5XG4gICAgaWYgKCF0aGlzLm5vZGUudHJ5R2V0Q29udGV4dChjeGFwaS5FQ1NfUkVNT1ZFX0RFRkFVTFRfREVTSVJFRF9DT1VOVCkpIHtcbiAgICAgIHRoaXMubWluQ2FwYWNpdHkgPSBwcm9wcy5taW5TY2FsaW5nQ2FwYWNpdHkgPz8gdGhpcy5kZXNpcmVkQ291bnQ7XG4gICAgICB0aGlzLm1heENhcGFjaXR5ID0gcHJvcHMubWF4U2NhbGluZ0NhcGFjaXR5IHx8ICgyICogdGhpcy5kZXNpcmVkQ291bnQpO1xuICAgIH0gZWxzZSB7XG4gICAgICBpZiAocHJvcHMuZGVzaXJlZFRhc2tDb3VudCAhPSBudWxsKSB7XG4gICAgICAgIHRoaXMubWluQ2FwYWNpdHkgPSBwcm9wcy5taW5TY2FsaW5nQ2FwYWNpdHkgPz8gdGhpcy5kZXNpcmVkQ291bnQ7XG4gICAgICAgIHRoaXMubWF4Q2FwYWNpdHkgPSBwcm9wcy5tYXhTY2FsaW5nQ2FwYWNpdHkgfHwgKDIgKiB0aGlzLmRlc2lyZWRDb3VudCk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aGlzLm1pbkNhcGFjaXR5ID0gcHJvcHMubWluU2NhbGluZ0NhcGFjaXR5ID8/IDE7XG4gICAgICAgIHRoaXMubWF4Q2FwYWNpdHkgPSBwcm9wcy5tYXhTY2FsaW5nQ2FwYWNpdHkgfHwgMjtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBpZiAoIXRoaXMuZGVzaXJlZENvdW50ICYmICF0aGlzLm1heENhcGFjaXR5KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ21heFNjYWxpbmdDYXBhY2l0eSBtdXN0IGJlIHNldCBhbmQgZ3JlYXRlciB0aGFuIDAgaWYgZGVzaXJlZENvdW50IGlzIDAnKTtcbiAgICB9XG5cbiAgICBuZXcgQ2ZuT3V0cHV0KHRoaXMsICdTUVNRdWV1ZScsIHsgdmFsdWU6IHRoaXMuc3FzUXVldWUucXVldWVOYW1lIH0pO1xuICAgIG5ldyBDZm5PdXRwdXQodGhpcywgJ1NRU1F1ZXVlQXJuJywgeyB2YWx1ZTogdGhpcy5zcXNRdWV1ZS5xdWV1ZUFybiB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDb25maWd1cmUgYXV0b3NjYWxpbmcgYmFzZWQgb2ZmIG9mIENQVSB1dGlsaXphdGlvbiBhcyB3ZWxsIGFzIHRoZSBudW1iZXIgb2YgbWVzc2FnZXMgdmlzaWJsZSBpbiB0aGUgU1FTIHF1ZXVlXG4gICAqXG4gICAqIEBwYXJhbSBzZXJ2aWNlIHRoZSBFQ1MvRmFyZ2F0ZSBzZXJ2aWNlIGZvciB3aGljaCB0byBhcHBseSB0aGUgYXV0b3NjYWxpbmcgcnVsZXMgdG9cbiAgICovXG4gIHByb3RlY3RlZCBjb25maWd1cmVBdXRvc2NhbGluZ0ZvclNlcnZpY2Uoc2VydmljZTogQmFzZVNlcnZpY2UpIHtcbiAgICBjb25zdCBzY2FsaW5nVGFyZ2V0ID0gc2VydmljZS5hdXRvU2NhbGVUYXNrQ291bnQoeyBtYXhDYXBhY2l0eTogdGhpcy5tYXhDYXBhY2l0eSwgbWluQ2FwYWNpdHk6IHRoaXMubWluQ2FwYWNpdHkgfSk7XG4gICAgc2NhbGluZ1RhcmdldC5zY2FsZU9uQ3B1VXRpbGl6YXRpb24oJ0NwdVNjYWxpbmcnLCB7XG4gICAgICB0YXJnZXRVdGlsaXphdGlvblBlcmNlbnQ6IDUwLFxuICAgIH0pO1xuICAgIHNjYWxpbmdUYXJnZXQuc2NhbGVPbk1ldHJpYygnUXVldWVNZXNzYWdlc1Zpc2libGVTY2FsaW5nJywge1xuICAgICAgbWV0cmljOiB0aGlzLnNxc1F1ZXVlLm1ldHJpY0FwcHJveGltYXRlTnVtYmVyT2ZNZXNzYWdlc1Zpc2libGUoKSxcbiAgICAgIHNjYWxpbmdTdGVwczogdGhpcy5zY2FsaW5nU3RlcHMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnQgU1FTIHBlcm1pc3Npb25zIHRvIGFuIEVDUyBzZXJ2aWNlLlxuICAgKiBAcGFyYW0gc2VydmljZSB0aGUgRUNTL0ZhcmdhdGUgc2VydmljZSB0byB3aGljaCB0byBncmFudCBTUVMgcGVybWlzc2lvbnNcbiAgICovXG4gIHByb3RlY3RlZCBncmFudFBlcm1pc3Npb25zVG9TZXJ2aWNlKHNlcnZpY2U6IEJhc2VTZXJ2aWNlKSB7XG4gICAgdGhpcy5zcXNRdWV1ZS5ncmFudENvbnN1bWVNZXNzYWdlcyhzZXJ2aWNlLnRhc2tEZWZpbml0aW9uLnRhc2tSb2xlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBkZWZhdWx0IGNsdXN0ZXIuXG4gICAqL1xuICBwcm90ZWN0ZWQgZ2V0RGVmYXVsdENsdXN0ZXIoc2NvcGU6IENvbnN0cnVjdCwgdnBjPzogSVZwYyk6IENsdXN0ZXIge1xuICAgIC8vIG1hZ2ljIHN0cmluZyB0byBhdm9pZCBjb2xsaXNpb24gd2l0aCB1c2VyLWRlZmluZWQgY29uc3RydWN0c1xuICAgIGNvbnN0IERFRkFVTFRfQ0xVU1RFUl9JRCA9IGBFY3NEZWZhdWx0Q2x1c3Rlck1uTDNtTk5ZTiR7dnBjID8gdnBjLm5vZGUuaWQgOiAnJ31gO1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2Yoc2NvcGUpO1xuICAgIHJldHVybiBzdGFjay5ub2RlLnRyeUZpbmRDaGlsZChERUZBVUxUX0NMVVNURVJfSUQpIGFzIENsdXN0ZXIgfHwgbmV3IENsdXN0ZXIoc3RhY2ssIERFRkFVTFRfQ0xVU1RFUl9JRCwgeyB2cGMgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGFuIEFXUyBMb2cgRHJpdmVyIHdpdGggdGhlIHByb3ZpZGVkIHN0cmVhbVByZWZpeFxuICAgKlxuICAgKiBAcGFyYW0gcHJlZml4IHRoZSBDbG91ZHdhdGNoIGxvZ2dpbmcgcHJlZml4XG4gICAqL1xuICBwcml2YXRlIGNyZWF0ZUFXU0xvZ0RyaXZlcihwcmVmaXg6IHN0cmluZyk6IEF3c0xvZ0RyaXZlciB7XG4gICAgcmV0dXJuIG5ldyBBd3NMb2dEcml2ZXIoeyBzdHJlYW1QcmVmaXg6IHByZWZpeCB9KTtcbiAgfVxufVxuIl19