from __future__ import annotations

from itertools import product
from pathlib import Path
from typing import Tuple, Dict, Union, Optional

import numpy as np
import rdflib
from tqdm import tqdm

from cfpq_data.src.graphs.rdf_graph import RDF
from cfpq_data.src.utils.rdf_helper import write_to_rdf, add_rdf_edge
from cfpq_data.src.utils.utils import add_graph_dir

SCALE_FREE_GRAPH_TO_GEN = list(product(
    [100, 500, 2500, 10000],
    [1, 3, 5, 10]
))


class ScaleFree(RDF):
    """
    ScaleFree — graphs generated by using the Barab'asi-Albert model of scale-free networks

    - graphs: already built graphs
    """

    graphs: Dict[Tuple[str, str], Path] = dict()

    @classmethod
    def build(cls,
              *args: Union[Path, str, int],
              source_file_format: str = 'rdf',
              config: Optional[Dict[str, str]] = None) -> ScaleFree:
        """
        Builds ScaleFree graph instance by number of vertices and degree of vertex

        - args[0] - number of vertices in the graph or path
        - args[1] - degree of the vertex in the graph (if args[0] is not a path)

        :param args: arguments
        :type args: int
        :param source_file_format: graph format ('txt'/'rdf')
        :type source_file_format: str
        :param config: edge configuration
        :type config: Optional[Dict[str, str]]
        :return: ScaleFree graph instance
        :rtype: ScaleFree
        """

        if len(args) > 1:
            vertices_number = args[0]
            vertices_degree = args[1]

            path_to_graph = gen_scale_free_graph(add_graph_dir('ScaleFree'),
                                                 vertices_number,
                                                 vertices_degree)

            graph = ScaleFree.load_from_rdf(path_to_graph)
        else:
            source = args[0]
            if source_file_format == 'txt':
                graph = cls.load_from_txt(source, config)
            else:
                graph = cls.load_from_rdf(source)

        graph.save_metadata()

        cls.graphs[(graph.basename, graph.file_extension)] = graph.path

        return graph


def gen_scale_free_graph(destination_folder: Path,
                         vertices_number: int,
                         vertices_degree: int,
                         labels: Tuple[str, ...] = ('A', 'B', 'C', 'D')
                         ) -> Path:
    """
    Generates scale free graph

    :param destination_folder: directory to save the graph
    :type destination_folder: Path
    :param vertices_number: number of vertices in the graph
    :type vertices_number: int
    :param vertices_degree: degree of a vertex in the graph
    :type vertices_degree: int
    :param labels: edge labels in the graph
    :type labels: Tuple[str, ...]
    :return: path to generated graph
    :rtype: Path
    """

    g = {
        i: [(j, np.random.choice(labels)) for j in range(vertices_degree)]
        for i in range(vertices_degree)
    }

    degree = [3] * vertices_degree

    for i in range(vertices_degree, vertices_number):
        to_vertices = np.random.choice(
            range(i),
            size=vertices_degree,
            replace=False,
            p=np.array(degree) / sum(degree)
        )

        g[i] = []
        degree.append(0)
        for to in to_vertices:
            label = np.random.choice(labels)
            g[i].append((to, label))
            degree[to] += 1
            degree[i] += 1

    output_graph = rdflib.Graph()

    edges = list()

    for v in g:
        for to in g[v]:
            edges.append((v, to[1], to[0]))

    for subj, pred, obj in tqdm(
            edges,
            desc=f'scale_free_graph_{vertices_number}_{vertices_degree} generation'
    ):
        add_rdf_edge(subj, pred, obj, output_graph)

    target = destination_folder / f'scale_free_graph_{vertices_number}_{vertices_degree}.xml'

    write_to_rdf(target, output_graph)

    return target
