"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const vpc_1 = require("./vpc");
/**
 * Turn an arbitrary string into one that can be used as a CloudFormation identifier by stripping special characters
 *
 * (At the moment, no efforts are taken to prevent collissions, but we can add that later when it becomes necessary).
 */
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
exports.slugify = slugify;
/**
 * The default names for every subnet type
 */
function defaultSubnetName(type) {
    switch (type) {
        case vpc_1.SubnetType.PUBLIC: return 'Public';
        case vpc_1.SubnetType.PRIVATE: return 'Private';
        case vpc_1.SubnetType.ISOLATED: return 'Isolated';
    }
}
exports.defaultSubnetName = defaultSubnetName;
/**
 * Return a subnet name from its construct ID
 *
 * All subnet names look like NAME <> "Subnet" <> INDEX
 */
function subnetGroupNameFromConstructId(subnet) {
    return subnet.node.id.replace(/Subnet\d+$/, '');
}
exports.subnetGroupNameFromConstructId = subnetGroupNameFromConstructId;
/**
 * Make the subnet construct ID from a name and number
 */
function subnetId(name, i) {
    return `${name}Subnet${i + 1}`;
}
exports.subnetId = subnetId;
class ImportSubnetGroup {
    constructor(subnetIds, names, routeTableIds, type, availabilityZones, idField, nameField, routeTableIdField) {
        this.availabilityZones = availabilityZones;
        this.subnetIds = subnetIds || [];
        this.routeTableIds = routeTableIds || [];
        this.groups = this.subnetIds.length / this.availabilityZones.length;
        if (Math.floor(this.groups) !== this.groups) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Number of ${idField} (${this.subnetIds.length}) must be a multiple of availability zones (${this.availabilityZones.length}).`);
        }
        if (this.routeTableIds.length !== this.subnetIds.length && routeTableIds != null) {
            // We don't err if no routeTableIds were provided to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            // tslint:disable-next-line: max-line-length
            throw new Error(`Number of ${routeTableIdField} (${this.routeTableIds.length}) must be equal to the amount of ${idField} (${this.subnetIds.length}).`);
        }
        this.names = this.normalizeNames(names, defaultSubnetName(type), nameField);
    }
    import(scope) {
        return range(this.subnetIds.length).map(i => {
            const k = Math.floor(i / this.availabilityZones.length);
            return vpc_1.Subnet.fromSubnetAttributes(scope, subnetId(this.names[k], i), {
                availabilityZone: this.pickAZ(i),
                subnetId: this.subnetIds[i],
                routeTableId: this.routeTableIds[i],
            });
        });
    }
    /**
     * Return a list with a name for every subnet
     */
    normalizeNames(names, defaultName, fieldName) {
        // If not given, return default
        if (names === undefined || names.length === 0) {
            return [defaultName];
        }
        // If given, must match given subnets
        if (names.length !== this.groups) {
            throw new Error(`${fieldName} must have an entry for every corresponding subnet group, got: ${JSON.stringify(names)}`);
        }
        return names;
    }
    /**
     * Return the i'th AZ
     */
    pickAZ(i) {
        return this.availabilityZones[i % this.availabilityZones.length];
    }
}
exports.ImportSubnetGroup = ImportSubnetGroup;
/**
 * Generate the list of numbers of [0..n)
 */
function range(n) {
    const ret = [];
    for (let i = 0; i < n; i++) {
        ret.push(i);
    }
    return ret;
}
exports.range = range;
/**
 * Return the union of table IDs from all selected subnets
 */
function allRouteTableIds(...ssns) {
    const ret = new Set();
    for (const ssn of ssns) {
        for (const subnet of ssn.subnets) {
            if (subnet.routeTable && subnet.routeTable.routeTableId) {
                ret.add(subnet.routeTable.routeTableId);
            }
        }
    }
    return Array.from(ret);
}
exports.allRouteTableIds = allRouteTableIds;
//# sourceMappingURL=data:application/json;base64,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