import subprocess
from pathlib import Path
from fujin.config import Config, HostConfig, Webserver, ProcessConfig, InstallationMode
from fujin.commands.deploy import Deploy
from unittest.mock import patch


def test_full_deployment_flow(vps_container, ssh_key_setup, tmp_path):
    # 1. Setup Config pointing to Docker container
    config = Config(
        app_name="integration-test",
        version="0.1.0",
        build_command="echo 'building'",
        distfile="dist/app-{version}.whl",
        installation_mode=InstallationMode.BINARY,  # Using binary for simplicity in this test
        host=HostConfig(
            ip=vps_container["ip"],
            user=vps_container["user"],
            ssh_port=vps_container["port"],
            _key_filename=ssh_key_setup,
            domain_name="localhost",
        ),
        webserver=Webserver(upstream="localhost:8000", enabled=True),
        processes={"web": ProcessConfig(command="integration-test")},
    )

    # Mock distfile
    dist_dir = tmp_path / "dist"
    dist_dir.mkdir()
    distfile = dist_dir / "app-0.1.0.whl"
    # Create a dummy executable script
    script_content = "#!/bin/sh\nwhile true; do sleep 1; done"
    distfile.write_text(script_content)
    distfile.chmod(0o755)

    # We need to patch the config to use our tmp_path as root or adjust paths
    # But Config uses relative paths. Let's just mock the distfile existence check if needed
    # Or better, run the test from tmp_path?
    # For now, let's assume we run pytest from root and we create the file relative to root?
    # Actually, Config.get_distfile_path returns a Path object.
    # Let's monkeypatch the distfile path resolution or just create the file in the current working dir?
    # Creating in CWD is risky.

    # Let's use a simpler approach: Mock the build command to create the file?
    # Or just create it in the test.

    # Since we are running in a real environment (the test runner), we need to create the file locally.
    Path("dist").mkdir(exist_ok=True)
    # Copy the executable script to the location expected by Config
    Path("dist/app-0.1.0.whl").write_text(script_content)
    Path("dist/app-0.1.0.whl").chmod(0o755)

    try:
        # 2. Run Deploy
        deploy_cmd = Deploy()
        deploy_cmd.config = config
        # We need to capture stdout to avoid cluttering test output
        # Patch handle_stdin to avoid "Inappropriate ioctl for device" or "reading from stdin" errors
        # when running fabric with pty=True in a test environment.
        with patch("invoke.runners.Runner.handle_stdin"):
            deploy_cmd()

        # 3. Verify inside the container
        # Check if service is active
        # The service name is generated by Config.get_service_name("web") -> "integration-test.service"
        service_name = "integration-test.service"

        check_cmd = [
            "docker",
            "exec",
            vps_container["name"],
            "systemctl",
            "is-active",
            service_name,
        ]
        result = subprocess.run(check_cmd, capture_output=True, text=True)

        if result.stdout.strip() != "active":
            subprocess.run(
                [
                    "docker",
                    "exec",
                    vps_container["name"],
                    "systemctl",
                    "status",
                    service_name,
                ],
                check=False,
            )
            subprocess.run(
                [
                    "docker",
                    "exec",
                    vps_container["name"],
                    "journalctl",
                    "-u",
                    service_name,
                    "--no-pager",
                ],
                check=False,
            )

        assert result.stdout.strip() == "active"

    finally:
        # Cleanup local artifact
        if Path("dist/app-0.1.0.whl").exists():
            Path("dist/app-0.1.0.whl").unlink()
        if Path("dist").exists() and not any(Path("dist").iterdir()):
            Path("dist").rmdir()
