#
# Copyright (c) European Synchrotron Radiation Facility (ESRF)
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to
# use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
# the Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
# COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
# IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

__authors__ = ["O. Svensson"]
__license__ = "MIT"
__date__ = "28/04/2020"

# Corresponding EDNA code:
# https://github.com/olofsvensson/edna-mx
# kernel/src/UtilsSymmetry.py


def getMinimumSymmetrySpaceGroupFromBravaisLattice(bravaisLattice):
    """
    This static method returns a string containing the name of the space
    group with minimum symmetry that corresponds to a given two-letter Bravais
    Lattice.

    Note that the "hR" Bravais Lattice is translated into the "H3" and not the
    "R3" space group since the "H3" space group is the one used in MX in general
    and in particular for structures deposited in the PDB.

    If the Bravais Lattice is not recognised a "None" object is returned.
    """
    spaceGroup = None
    if bravaisLattice == "aP":
        spaceGroup = "P1"

    elif bravaisLattice == "mP":
        spaceGroup = "P2"

    elif bravaisLattice == "mC" or bravaisLattice == "mI":
        spaceGroup = "C2"

    elif bravaisLattice == "oP":
        spaceGroup = "P222"

    elif (
        bravaisLattice == "oA"
        or bravaisLattice == "oB"
        or bravaisLattice == "oC"
        or bravaisLattice == "oS"
    ):
        spaceGroup = "C222"

    elif bravaisLattice == "oF":
        spaceGroup = "F222"

    elif bravaisLattice == "oI":
        spaceGroup = "I222"

    elif bravaisLattice == "tP" or bravaisLattice == "tC":
        spaceGroup = "P4"

    elif bravaisLattice == "tI" or bravaisLattice == "tF":
        spaceGroup = "I4"

    elif bravaisLattice == "hP":
        spaceGroup = "P3"

    elif bravaisLattice == "hR":
        spaceGroup = "H3"

    elif bravaisLattice == "cP":
        spaceGroup = "P23"

    elif bravaisLattice == "cF":
        spaceGroup = "F23"

    elif bravaisLattice == "cI":
        spaceGroup = "I23"

    return spaceGroup


def getNumberOfSymmetryOperatorsFromSpaceGroupITNumber(spaceGroupId):
    return getNumberOfSymmetryOperators(spaceGroupId, 0)


def getNumberOfSymmetryOperatorsFromSpaceGroupName(spaceGroupName):
    return getNumberOfSymmetryOperators(spaceGroupName, 3)


def getNumberOfSymmetryOperators(spaceGroupIdOrName, index):
    for strLine in getSymOpLib().splitlines():
        listItems = strLine.split()
        if len(listItems) > 3 and listItems[index] == str(spaceGroupIdOrName):
            numOperators = int(listItems[1])
            break
    return numOperators


def getITNumberFromSpaceGroupName(spaceGroupName):
    iITNumber = None
    for strLine in getSymOpLib().splitlines():
        listItems = strLine.split(" ")
        if len(listItems) > 3:
            if spaceGroupName == listItems[3]:
                iITNumber = int(listItems[0])
    return iITNumber


def getSpaceGroupNameFromITNumber(itNumber):
    strSpaceGroupName = None
    for strLine in getSymOpLib().splitlines():
        listItems = strLine.split(" ")
        if len(listItems) > 3:
            if itNumber == int(listItems[0]):
                strSpaceGroupName = listItems[3]
    return strSpaceGroupName


def getSymOpLib():
    """
    ccp4/lib/data/symop.lib file as of 20170831
    """
    return """1 1 1 P1 PG1 TRICLINIC 'P 1'
 X,Y,Z
2 2 2 P-1 PG1bar TRICLINIC 'P -1'
 X,Y,Z
 -X,-Y,-Z
3 2 2 P2 PG2 MONOCLINIC 'P 1 2 1'
 X,Y,Z
 -X,Y,-Z
4 2 2 P21 PG2 MONOCLINIC 'P 1 21 1'
 X,Y,Z
 -X,Y+1/2,-Z
5 4 2 C2 PG2 MONOCLINIC 'C 1 2 1'
 X,Y,Z
 -X,Y,-Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2+Y,-Z
6 2 2 Pm PGm MONOCLINIC 'P 1 m 1'
 X,Y,Z
 X,-Y,Z
7 2 2 Pc PGm MONOCLINIC 'P 1 c 1'
 X,Y,Z
 X,-Y,1/2+Z
8 4 2 Cm PGm MONOCLINIC 'C 1 m 1'
 X,Y,Z
 X,-Y,Z
 1/2+X,1/2+Y,Z
 1/2+X,1/2-Y,Z
9 4 2 Cc PGm MONOCLINIC 'C 1 c 1'
 X,Y,Z
 X,-Y,1/2+Z
 1/2+X,1/2+Y,Z
 1/2+X,1/2-Y,1/2+Z
10 4 4 P2/m PG2/m MONOCLINIC 'P 1 2/m 1'
 X,Y,Z
 X,-Y,Z
 -X,Y,-Z
 -X,-Y,-Z
11 4 4 P21/m PG2/m MONOCLINIC 'P 1 21/m 1'
 X,Y,Z
 -X,1/2+Y,-Z
 -X,-Y,-Z
 X,1/2-Y,Z
12 8 4 C2/m PG2/m MONOCLINIC 'C 1 2/m 1'
 X,Y,Z
 X,-Y,Z
 -X,Y,-Z
 -X,-Y,-Z
 1/2+X,1/2+Y,Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2-X,1/2-Y,-Z
13 4 4 P2/c PG2/m MONOCLINIC 'P 1 2/c 1'
 X,Y,Z
 -X,Y,1/2-Z
 -X,-Y,-Z
 X,-Y,1/2+Z
14 4 4 P21/c PG2/m MONOCLINIC 'P 1 21/c 1'
 X,Y,Z
 -X,-Y,-Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2+Z
15 8 4 C2/c PG2/m MONOCLINIC 'C 1 2/c 1'
 X,Y,Z
 -X,Y,1/2-Z
 -X,-Y,-Z
 X,-Y,1/2+Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2+Y,1/2-Z
 1/2-X,1/2-Y,-Z
 1/2+X,1/2-Y,1/2+Z
16 4 4 P222 PG222 ORTHORHOMBIC 'P 2 2 2'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
17 4 4 P2221 PG222 ORTHORHOMBIC 'P 2 2 21'
 X,Y,Z
 -X,-Y,1/2+Z
 -X,Y,1/2-Z
 X,-Y,-Z
18 4 4 P21212 PG222 ORTHORHOMBIC 'P 21 21 2'
 X,Y,Z
 -X,-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
19 4 4 P212121 PG222 ORTHORHOMBIC 'P 21 21 21'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
20 8 4 C2221 PG222 ORTHORHOMBIC 'C 2 2 21'
 X,Y,Z
 -X,-Y,1/2+Z
 -X,Y,1/2-Z
 X,-Y,-Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
21 8 4 C222 PG222 ORTHORHOMBIC 'C 2 2 2'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
22 16 4 F222 PG222 ORTHORHOMBIC 'F 2 2 2'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
23 8 4 I222 PG222 ORTHORHOMBIC 'I 2 2 2'
 X,Y,Z
 -X,-Y,Z
 X,-Y,-Z
 -X,Y,-Z
 X+1/2,Y+1/2,Z+1/2
 -X+1/2,-Y+1/2,Z+1/2
 X+1/2,-Y+1/2,-Z+1/2
 -X+1/2,Y+1/2,-Z+1/2
24 8 4 I212121 PG222 ORTHORHOMBIC 'I 21 21 21'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 1/2+X,1/2+Y,1/2+Z
 -X,1/2-Y,Z
 1/2-X,Y,-Z
 X,-Y,1/2-Z
25 4 4 Pmm2 PGmm2 ORTHORHOMBIC 'P m m 2'
 X,Y,Z
 -X,-Y,Z
 X,-Y,Z
 -X,Y,Z
26 4 4 Pmc21 PGmm2 ORTHORHOMBIC 'P m c 21'
 X,Y,Z
 -X,-Y,1/2+Z
 X,-Y,1/2+Z
 -X,Y,Z
27 4 4 Pcc2 PGmm2 ORTHORHOMBIC 'P c c 2'
 X,Y,Z
 -X,-Y,Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
28 4 4 Pma2 PGmm2 ORTHORHOMBIC 'P m a 2'
 X,Y,Z
 -X,-Y,Z
 1/2+X,-Y,Z
 1/2-X,Y,Z
29 4 4 Pca21 PGmm2 ORTHORHOMBIC 'P c a 21'
 X,Y,Z
 -X,-Y,1/2+Z
 1/2+X,-Y,Z
 1/2-X,Y,1/2+Z
30 4 4 Pnc2 PGmm2 ORTHORHOMBIC 'P n c 2'
 X,Y,Z
 -X,-Y,Z
 X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2+Z
31 4 4 Pmn21 PGmm2 ORTHORHOMBIC 'P m n 21'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 1/2+X,-Y,1/2+Z
 -X,Y,Z
32 4 4 Pba2 PGmm2 ORTHORHOMBIC 'P b a 2'
 X,Y,Z
 -X,-Y,Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
33 4 4 Pna21 PGmm2 ORTHORHOMBIC 'P n a 21'
 X,Y,Z
 -X,-Y,1/2+Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,1/2+Z
34 4 4 Pnn2 PGmm2 ORTHORHOMBIC 'P n n 2'
 X,Y,Z
 -X,-Y,Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
35 8 4 Cmm2 PGmm2 ORTHORHOMBIC 'C m m 2'
 X,Y,Z
 -X,-Y,Z
 X,-Y,Z
 -X,Y,Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
36 8 4 Cmc21 PGmm2 ORTHORHOMBIC 'C m c 21'
 X,Y,Z
 -X,-Y,1/2+Z
 X,-Y,1/2+Z
 -X,Y,Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,1/2+Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,Z
37 8 4 Ccc2 PGmm2 ORTHORHOMBIC 'C c c 2'
 X,Y,Z
 -X,-Y,Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
38 8 4 Amm2 PGmm2 ORTHORHOMBIC 'A m m 2'
 X,Y,Z
 -X,-Y,Z
 X,-Y,Z
 -X,Y,Z
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2+Z
39 8 4 Abm2 PGmm2 ORTHORHOMBIC 'A b m 2'
 X,Y,Z
 -X,-Y,Z
 X,1/2-Y,Z
 -X,1/2+Y,Z
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
40 8 4 Ama2 PGmm2 ORTHORHOMBIC 'A m a 2'
 X,Y,Z
 -X,-Y,Z
 1/2+X,-Y,Z
 1/2-X,Y,Z
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
41 8 4 Aba2 PGmm2 ORTHORHOMBIC 'A b a 2'
 X,Y,Z
 -X,-Y,Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 1/2+X,-Y,1/2+Z
 1/2-X,Y,1/2+Z
42 16 4 Fmm2 PGmm2 ORTHORHOMBIC 'F m m 2'
 X,Y,Z
 -X,-Y,Z
 X,-Y,Z
 -X,Y,Z
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2+Z
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2+X,-Y,1/2+Z
 1/2-X,Y,1/2+Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
43 16 4 Fdd2 PGmm2 ORTHORHOMBIC 'F d d 2'
 X,Y,Z
 -X,-Y,Z
 1/4+X,1/4-Y,1/4+Z
 1/4-X,1/4+Y,1/4+Z
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 1/4+X,3/4-Y,3/4+Z
 1/4-X,3/4+Y,3/4+Z
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 3/4+X,1/4-Y,3/4+Z
 3/4-X,1/4+Y,3/4+Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 3/4+X,3/4-Y,1/4+Z
 3/4-X,3/4+Y,1/4+Z
44 8 4 Imm2 PGmm2 ORTHORHOMBIC 'I m m 2'
 X,Y,Z
 -X,-Y,Z
 X,-Y,Z
 -X,Y,Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
45 8 4 Iba2 PGmm2 ORTHORHOMBIC 'I b a 2'
 X,Y,Z
 -X,-Y,Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
46 8 4 Ima2 PGmm2 ORTHORHOMBIC 'I m a 2'
 X,Y,Z
 -X,-Y,Z
 1/2+X,-Y,Z
 1/2-X,Y,Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2+Z
47 8 8 Pmmm PGmmm ORTHORHOMBIC 'P 2/m 2/m 2/m' 'P m m m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
48 8 8 Pnnn PGmmm ORTHORHOMBIC 'P 2/n 2/n 2/n'  'P n n n'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
49 8 8 Pccm PGmmm ORTHORHOMBIC 'P 2/c 2/c 2/m' 'P c c m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,1/2-Z
 X,-Y,1/2-Z
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
50 8 8 Pban PGmmm ORTHORHOMBIC 'P 2/b 2/a 2/n' 'P b a n'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
51 8 8 Pmma PGmmm ORTHORHOMBIC 'P 21/m 2/m 2/a'  'P m m a'
 X,Y,Z
 1/2-X,-Y,Z
 -X,Y,-Z
 1/2+X,-Y,-Z
 -X,-Y,-Z
 1/2+X,Y,-Z
 X,-Y,Z
 1/2-X,Y,Z
52 8 8 Pnna PGmmm ORTHORHOMBIC 'P 2/n 21/n 2/a'  'P n n a'
 X,Y,Z
 1/2-X,-Y,Z
 1/2-X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 -X,-Y,-Z
 1/2+X,Y,-Z
 1/2+X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2+Z
53 8 8 Pmna PGmmm ORTHORHOMBIC 'P 2/m 2/n 21/a' 'P m n a'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 X,-Y,-Z
 -X,-Y,-Z
 1/2+X,Y,1/2-Z
 1/2+X,-Y,1/2+Z
 -X,Y,Z
54 8 8 Pcca PGmmm ORTHORHOMBIC 'P 21/c 2/c 2/a' 'P c c a'
 X,Y,Z
 1/2-X,-Y,Z
 -X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 -X,-Y,-Z
 1/2+X,Y,-Z
 X,-Y,1/2+Z
 1/2-X,Y,1/2+Z
55 8 8 Pbam PGmmm ORTHORHOMBIC 'P 21/b 21/a 2/m' 'P b a m'
 X,Y,Z
 -X,-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 -X,-Y,-Z
 X,Y,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
56 8 8 Pccn PGmmm ORTHORHOMBIC 'P 21/c 21/c 2/n' 'P c c n'
 X,Y,Z
 1/2-X,1/2-Y,Z
 -X,1/2+Y,1/2-Z
 1/2+X,-Y,1/2-Z
 -X,-Y,-Z
 1/2+X,1/2+Y,-Z
 X,1/2-Y,1/2+Z
 1/2-X,Y,1/2+Z
57 8 8 Pbcm PGmmm ORTHORHOMBIC 'P 2/b 21/c 21/m' 'P b c m'
 X,Y,Z
 -X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,-Z
 -X,-Y,-Z
 X,Y,1/2-Z
 X,1/2-Y,1/2+Z
 -X,1/2+Y,Z
58 8 8 Pnnm PGmmm ORTHORHOMBIC 'P 21/n 21/n 2/m' 'P n n m'
 X,Y,Z
 -X,-Y,Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 -X,-Y,-Z
 X,Y,-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
59 8 8 Pmmn PGmmm ORTHORHOMBIC 'P 21/m 21/m 2/n' 'P m m n'
 X,Y,Z
 -X,-Y,Z
 1/2-X,Y+1/2,-Z
 X+1/2,1/2-Y,-Z
 1/2-X,1/2-Y,-Z
 X+1/2,Y+1/2,-Z
 X,-Y,Z
 -X,Y,Z
60 8 8 Pbcn PGmmm ORTHORHOMBIC 'P 21/b 2/c 21/n' 'P b c n'
 X,Y,Z
 1/2-X,1/2-Y,1/2+Z
 -X,Y,1/2-Z
 1/2+X,1/2-Y,-Z
 -X,-Y,-Z
 1/2+X,1/2+Y,1/2-Z
 X,-Y,1/2+Z
 1/2-X,1/2+Y,Z
61 8 8 Pbca PGmmm ORTHORHOMBIC 'P 21/b 21/c 21/a' 'P b c a'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 -X,-Y,-Z
 1/2+X,Y,1/2-Z
 X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,Z
62 8 8 Pnma PGmmm ORTHORHOMBIC 'P 21/n 21/m 21/a' 'P n m a'
 X,Y,Z
 -X+1/2,-Y,Z+1/2
 -X,Y+1/2,-Z
 X+1/2,-Y+1/2,-Z+1/2
 -X,-Y,-Z
 X+1/2,Y,-Z+1/2
 X,-Y+1/2,Z
 -X+1/2,Y+1/2,Z+1/2
63 16 8 Cmcm PGmmm ORTHORHOMBIC 'C 2/m 2/c 21/m' 'C m c m'
 X,Y,Z
 -X,-Y,1/2+Z
 -X,Y,1/2-Z
 X,-Y,-Z
 -X,-Y,-Z
 X,Y,1/2-Z
 X,-Y,1/2+Z
 -X,Y,Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,Z
64 16 8 Cmca PGmmm ORTHORHOMBIC 'C 2/m 2/c 21/a' 'C m c a'
 X,Y,Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,-Y,-Z
 -X,-Y,-Z
 X,1/2+Y,1/2-Z
 X,1/2-Y,1/2+Z
 -X,Y,Z
 1/2+X,1/2+Y,Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,1/2-Y,-Z
 1/2-X,1/2-Y,-Z
 1/2+X,Y,1/2-Z
 1/2+X,-Y,1/2+Z
 1/2-X,1/2+Y,Z
65 16 8 Cmmm PGmmm ORTHORHOMBIC 'C 2/m 2/m 2/m'  'C m m m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
66 16 8 Cccm PGmmm ORTHORHOMBIC 'C 2/c 2/c 2/m'  'C c c m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,1/2-Z
 X,-Y,1/2-Z
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
67 16 8 Cmma PGmmm ORTHORHOMBIC 'C 2/m 2/m 2/a' 'C m m a'
 X,Y,Z
 -X,1/2-Y,Z
 -X,1/2+Y,-Z
 X,-Y,-Z
 -X,-Y,-Z
 X,1/2+Y,-Z
 X,1/2-Y,Z
 -X,Y,Z
 1/2+X,1/2+Y,Z
 1/2-X,-Y,Z
 1/2-X,Y,-Z
 1/2+X,1/2-Y,-Z
 1/2-X,1/2-Y,-Z
 1/2+X,Y,-Z
 1/2+X,-Y,Z
 1/2-X,1/2+Y,Z
68 16 8 Ccca PGmmm ORTHORHOMBIC 'C 2/c 2/c 2/a' 'C c c a'
 X,Y,Z
 1/2-X,1/2-Y,Z
 -X,Y,-Z
 1/2+X,1/2-Y,-Z
 -X,1/2-Y,1/2-Z
 1/2+X,Y,1/2-Z
 X,1/2-Y,1/2+Z
 1/2-X,Y,1/2+Z
 1/2+X,1/2+Y,Z
 -X,-Y,Z
 1/2-X,1/2+Y,-Z
 X,-Y,-Z
 1/2-X,-Y,1/2-Z
 X,1/2+Y,1/2-Z
 1/2+X,-Y,1/2+Z
 -X,1/2+Y,1/2+Z
69 32 8 Fmmm PGmmm ORTHORHOMBIC 'F 2/m 2/m 2/m' 'F m m m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 -X,1/2-Y,1/2-Z
 X,1/2+Y,1/2-Z
 X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2+Z
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 1/2-X,-Y,1/2-Z
 1/2+X,Y,1/2-Z
 1/2+X,-Y,1/2+Z
 1/2-X,Y,1/2+Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
70 32 8 Fddd PGmmm ORTHORHOMBIC 'F 2/d 2/d 2/d' 'F d d d'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 1/4-X,1/4-Y,1/4-Z
 1/4+X,1/4+Y,1/4-Z
 1/4+X,1/4-Y,1/4+Z
 1/4-X,1/4+Y,1/4+Z
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 1/4-X,3/4-Y,3/4-Z
 1/4+X,3/4+Y,3/4-Z
 1/4+X,3/4-Y,3/4+Z
 1/4-X,3/4+Y,3/4+Z
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 3/4-X,1/4-Y,3/4-Z
 3/4+X,1/4+Y,3/4-Z
 3/4+X,1/4-Y,3/4+Z
 3/4-X,1/4+Y,3/4+Z
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 3/4-X,3/4-Y,1/4-Z
 3/4+X,3/4+Y,1/4-Z
 3/4+X,3/4-Y,1/4+Z
 3/4-X,3/4+Y,1/4+Z
71 16 8 Immm PGmmm ORTHORHOMBIC 'I 2/m 2/m 2/m' 'I m m m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
72 16 8 Ibam PGmmm ORTHORHOMBIC 'I 2/b 2/a 2/m' 'I b a m'
 X,Y,Z
 -X,-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 -X,-Y,-Z
 X,Y,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 -X,Y,1/2-Z
 X,-Y,1/2-Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
73 16 8 Ibca PGmmm ORTHORHOMBIC 'I 21/b 21/c 21/a' 'I b c a'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 -X,-Y,-Z
 1/2+X,Y,1/2-Z
 X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,Z
 1/2+X,1/2+Y,1/2+Z
 -X,1/2-Y,Z
 1/2-X,Y,-Z
 X,-Y,1/2-Z
 1/2-X,1/2-Y,1/2-Z
 X,1/2+Y,-Z
 1/2+X,-Y,Z
 -X,Y,1/2+Z
74 16 8 Imma PGmmm ORTHORHOMBIC 'I 21/m 21/m 21/a' 'I m m a'
 X,Y,Z
 -X,1/2-Y,Z
 -X,1/2+Y,-Z
 X,-Y,-Z
 -X,-Y,-Z
 X,1/2+Y,-Z
 X,1/2-Y,Z
 -X,Y,Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,Y,1/2-Z
 1/2+X,-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
75 4 4 P4 PG4 TETRAGONAL 'P 4'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
76 4 4 P41 PG4 TETRAGONAL 'P 41'
 X,Y,Z
 -X,-Y,1/2+Z
 -Y,X,1/4+Z
 Y,-X,3/4+Z
77 4 4 P42 PG4 TETRAGONAL 'P 42'
 X,Y,Z
 -X,-Y,Z
 -Y,X,1/2+Z
 Y,-X,1/2+Z
78 4 4 P43 PG4 TETRAGONAL 'P 43'
 X,Y,Z
 -X,-Y,1/2+Z
 -Y,X,3/4+Z
 Y,-X,1/4+Z
79 8 4 I4 PG4 TETRAGONAL 'I 4'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
80 8 4 I41 PG4 TETRAGONAL 'I 41'
 X,Y,Z
 1/2-X,1/2-Y,1/2+Z
 -Y,1/2+X,1/4+Z
 1/2+Y,-X,3/4+Z
 1/2+X,1/2+Y,1/2+Z
 -X,-Y,Z
 1/2-Y,X,3/4+Z
 Y,1/2-X,1/4+Z
81 4 4 P-4 PG4bar TETRAGONAL 'P -4'
 X,Y,Z
 -X,-Y,Z
 Y,-X,-Z
 -Y,X,-Z
82 8 4 I-4 PG4bar TETRAGONAL 'I -4'
 X,Y,Z
 -X,-Y,Z
 Y,-X,-Z
 -Y,X,-Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2+Y,1/2-X,1/2-Z
 1/2-Y,1/2+X,1/2-Z
83 8 8 P4/m PG4/m TETRAGONAL 'P 4/m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,-Z
 -Y,X,-Z
84 8 8 P42/m PG4/m TETRAGONAL 'P 42/m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,1/2+Z
 Y,-X,1/2+Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,1/2-Z
 -Y,X,1/2-Z
85 8 8 P4/n PG4/m TETRAGONAL 'P 4/n'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,1/2+X,Z
 1/2+Y,1/2-X,Z
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,-Z
 Y,-X,-Z
 -Y,X,-Z
86 8 8 P42/n PG4/m TETRAGONAL 'P 42/n'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 Y,-X,-Z
 -Y,X,-Z
87 16 8 I4/m PG4/m TETRAGONAL 'I 4/m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,-Z
 -Y,X,-Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 1/2-Y,1/2+X,1/2-Z
88 16 8 I41/a PG4/m TETRAGONAL 'I 41/a'
 X,Y,Z
 1/2-X,1/2-Y,1/2+Z
 -Y,1/2+X,1/4+Z
 1/2+Y,-X,3/4+Z
 -X,1/2-Y,1/4-Z
 1/2+X,Y,3/4-Z
 Y,-X,-Z
 1/2-Y,1/2+X,1/2-Z
 1/2+X,1/2+Y,1/2+Z
 -X,-Y,Z
 1/2-Y,X,3/4+Z
 Y,1/2-X,1/4+Z
 1/2-X,-Y,3/4-Z
 X,1/2+Y,1/4-Z
 1/2+Y,1/2-X,1/2-Z
 -Y,X,-Z
89 8 8 P422 PG422 TETRAGONAL 'P 4 2 2'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 -X,Y,-Z
 X,-Y,-Z
 Y,X,-Z
 -Y,-X,-Z
90 8 8 P4212 PG422 TETRAGONAL 'P 4 21 2'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,1/2+X,Z
 1/2+Y,1/2-X,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 Y,X,-Z
 -Y,-X,-Z
91 8 8 P4122 PG422 TETRAGONAL 'P 41 2 2'
 X,Y,Z
 -X,-Y,1/2+Z
 -Y,X,1/4+Z
 Y,-X,3/4+Z
 -X,Y,-Z
 X,-Y,1/2-Z
 Y,X,3/4-Z
 -Y,-X,1/4-Z
92 8 8 P41212 PG422 TETRAGONAL 'P 41 21 2'
 X,Y,Z
 -X,-Y,1/2+Z
 1/2-Y,1/2+X,1/4+Z
 1/2+Y,1/2-X,3/4+Z
 1/2-X,1/2+Y,1/4-Z
 1/2+X,1/2-Y,3/4-Z
 Y,X,-Z
 -Y,-X,1/2-Z
93 8 8 P4222 PG422 TETRAGONAL 'P 42 2 2'
 X,Y,Z
 -X,-Y,Z
 -Y,X,1/2+Z
 Y,-X,1/2+Z
 -X,Y,-Z
 X,-Y,-Z
 Y,X,1/2-Z
 -Y,-X,1/2-Z
94 8 8 P42212 PG422 TETRAGONAL 'P 42 21 2'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 Y,X,-Z
 -Y,-X,-Z
95 8 8 P4322 PG422 TETRAGONAL 'P 43 2 2'
 X,Y,Z
 -X,-Y,1/2+Z
 -Y,X,3/4+Z
 Y,-X,1/4+Z
 -X,Y,-Z
 X,-Y,1/2-Z
 Y,X,1/4-Z
 -Y,-X,3/4-Z
96 8 8 P43212 PG422 TETRAGONAL 'P 43 21 2'
 X,Y,Z
 -X,-Y,1/2+Z
 1/2-Y,1/2+X,3/4+Z
 1/2+Y,1/2-X,1/4+Z
 1/2-X,1/2+Y,3/4-Z
 1/2+X,1/2-Y,1/4-Z
 Y,X,-Z
 -Y,-X,1/2-Z
97 16 8 I422 PG422 TETRAGONAL 'I 4 2 2'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 -X,Y,-Z
 X,-Y,-Z
 Y,X,-Z
 -Y,-X,-Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
98 16 8 I4122 PG422 TETRAGONAL 'I 41 2 2'
 X,Y,Z
 1/2-X,1/2-Y,1/2+Z
 -Y,1/2+X,1/4+Z
 1/2+Y,-X,3/4+Z
 1/2-X,Y,3/4-Z
 X,1/2-Y,1/4-Z
 1/2+Y,1/2+X,1/2-Z
 -Y,-X,-Z
 1/2+X,1/2+Y,1/2+Z
 -X,-Y,Z
 1/2-Y,X,3/4+Z
 Y,1/2-X,1/4+Z
 -X,1/2+Y,1/4-Z
 1/2+X,-Y,3/4-Z
 Y,X,-Z
 1/2-Y,1/2-X,1/2-Z
99 8 8 P4mm PG4mm TETRAGONAL 'P 4 m m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 X,-Y,Z
 -X,Y,Z
 -Y,-X,Z
 Y,X,Z
100 8 8 P4bm PG4mm TETRAGONAL 'P 4 b m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2-Y,1/2-X,Z
 1/2+Y,1/2+X,Z
101 8 8 P42cm PG4mm TETRAGONAL 'P 42 c m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,1/2+Z
 Y,-X,1/2+Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 -Y,-X,Z
 Y,X,Z
102 8 8 P42nm PG4mm TETRAGONAL 'P 42 n m'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 -Y,-X,Z
 Y,X,Z
103 8 8 P4cc PG4mm TETRAGONAL 'P 4 c c'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 -Y,-X,1/2+Z
 Y,X,1/2+Z
104 8 8 P4nc PG4mm TETRAGONAL 'P 4 n c'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
105 8 8 P42mc PG4mm TETRAGONAL 'P 42 m c'
 X,Y,Z
 -X,-Y,Z
 -Y,X,1/2+Z
 Y,-X,1/2+Z
 X,-Y,Z
 -X,Y,Z
 -Y,-X,1/2+Z
 Y,X,1/2+Z
106 8 8 P42bc PG4mm TETRAGONAL 'P 42 b c'
 X,Y,Z
 -X,-Y,Z
 -Y,X,1/2+Z
 Y,-X,1/2+Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
107 16 8 I4mm PG4mm TETRAGONAL 'I 4 m m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 X,-Y,Z
 -X,Y,Z
 -Y,-X,Z
 Y,X,Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
108 16 8 I4cm PG4mm TETRAGONAL 'I 4 c m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 -Y,-X,1/2+Z
 Y,X,1/2+Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2-Y,1/2-X,Z
 1/2+Y,1/2+X,Z
109 16 8 I41md PG4mm TETRAGONAL 'I 41 m d'
 X,Y,Z
 1/2-X,1/2-Y,1/2+Z
 -Y,1/2+X,1/4+Z
 1/2+Y,-X,3/4+Z
 X,-Y,Z
 1/2-X,1/2+Y,1/2+Z
 -Y,1/2-X,1/4+Z
 1/2+Y,X,3/4+Z
 1/2+X,1/2+Y,1/2+Z
 -X,-Y,Z
 1/2-Y,X,3/4+Z
 Y,1/2-X,1/4+Z
 1/2+X,1/2-Y,1/2+Z
 -X,Y,Z
 1/2-Y,-X,3/4+Z
 Y,1/2+X,1/4+Z
110 16 8 I41cd PG4mm TETRAGONAL 'I 41 c d'
 X,Y,Z
 1/2-X,1/2-Y,1/2+Z
 -Y,1/2+X,1/4+Z
 1/2+Y,-X,3/4+Z
 X,-Y,1/2+Z
 1/2-X,1/2+Y,Z
 -Y,1/2-X,3/4+Z
 1/2+Y,X,1/4+Z
 1/2+X,1/2+Y,1/2+Z
 -X,-Y,Z
 1/2-Y,X,3/4+Z
 Y,1/2-X,1/4+Z
 1/2+X,1/2-Y,Z
 -X,Y,1/2+Z
 1/2-Y,-X,1/4+Z
 Y,1/2+X,3/4+Z
111 8 8 P-42m PG4bar2m TETRAGONAL 'P -4 2 m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,-Z
 Y,-X,-Z
 -X,Y,-Z
 X,-Y,-Z
 -Y,-X,Z
 Y,X,Z
112 8 8 P-42c PG4bar2m TETRAGONAL 'P -4 2 c'
 X,Y,Z
 -X,-Y,Z
 -Y,X,-Z
 Y,-X,-Z
 -X,Y,1/2-Z
 X,-Y,1/2-Z
 -Y,-X,1/2+Z
 Y,X,1/2+Z
113 8 8 P-421m PG4bar2m TETRAGONAL 'P -4 21 m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,-Z
 Y,-X,-Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2-Y,1/2-X,Z
 1/2+Y,1/2+X,Z
114 8 8 P-421c PG4bar2m TETRAGONAL 'P -4 21 c'
 X,Y,Z
 -X,-Y,Z
 -Y,X,-Z
 Y,-X,-Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
115 8 8 P-4m2 PG4barm2 TETRAGONAL 'P -4 m 2'
 X,Y,Z
 -X,-Y,Z
 Y,-X,-Z
 -Y,X,-Z
 X,-Y,Z
 -X,Y,Z
 Y,X,-Z
 -Y,-X,-Z
116 8 8 P-4c2 PG4barm2 TETRAGONAL 'P -4 c 2'
 X,Y,Z
 -X,-Y,Z
 -Y,X,-Z
 Y,-X,-Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 Y,X,1/2-Z
 -Y,-X,1/2-Z
117 8 8 P-4b2 PG4barm2 TETRAGONAL 'P -4 b 2'
 X,Y,Z
 -X,-Y,Z
 -Y,X,-Z
 Y,-X,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2+Y,1/2+X,-Z
 1/2-Y,1/2-X,-Z
118 8 8 P-4n2 PG4barm2 TETRAGONAL 'P -4 n 2'
 X,Y,Z
 -X,-Y,Z
 -Y,X,-Z
 Y,-X,-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
119 16 8 I-4m2 PG4barm2 TETRAGONAL 'I -4 m 2'
 X,Y,Z
 -X,-Y,Z
 -Y,X,-Z
 Y,-X,-Z
 X,-Y,Z
 -X,Y,Z
 Y,X,-Z
 -Y,-X,-Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-Y,1/2+X,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
120 16 8 I-4c2 PG4barm2 TETRAGONAL 'I -4 c 2'
 X,Y,Z
 -X,-Y,Z
 -Y,X,-Z
 Y,-X,-Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 Y,X,1/2-Z
 -Y,-X,1/2-Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-Y,1/2+X,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2+Y,1/2+X,-Z
 1/2-Y,1/2-X,-Z
121 16 8 I-42m PG4bar2m TETRAGONAL 'I -4 2 m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,-Z
 Y,-X,-Z
 -X,Y,-Z
 X,-Y,-Z
 -Y,-X,Z
 Y,X,Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-Y,1/2+X,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
122 16 8 I-42d PG4bar2m TETRAGONAL 'I -4 2 d'
 X,Y,Z
 -X,-Y,Z
 -Y,X,-Z
 Y,-X,-Z
 1/2-X,Y,3/4-Z
 1/2+X,-Y,3/4-Z
 1/2-Y,-X,3/4+Z
 1/2+Y,X,3/4+Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-Y,1/2+X,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 -X,1/2+Y,1/4-Z
 X,1/2-Y,1/4-Z
 -Y,1/2-X,1/4+Z
 Y,1/2+X,1/4+Z
123 16 16 P4/mmm PG4/mmm TETRAGONAL 'P 4/m 2/m 2/m' 'P4/m m m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 -X,Y,-Z
 X,-Y,-Z
 Y,X,-Z
 -Y,-X,-Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,-Z
 -Y,X,-Z
 X,-Y,Z
 -X,Y,Z
 -Y,-X,Z
 Y,X,Z
124 16 16 P4/mcc PG4/mmm TETRAGONAL 'P 4/m 2/c 2/c' 'P4/m c c'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 -X,Y,1/2-Z
 X,-Y,1/2-Z
 Y,X,1/2-Z
 -Y,-X,1/2-Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,-Z
 -Y,X,-Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 -Y,-X,1/2+Z
 Y,X,1/2+Z
125 16 16 P4/nbm PG4/mmm TETRAGONAL 'P 4/n 2/b 2/m' 'P4/n b m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 -X,Y,-Z
 X,-Y,-Z
 Y,X,-Z
 -Y,-X,-Z
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,-Z
 1/2+Y,1/2-X,-Z
 1/2-Y,1/2+X,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2-Y,1/2-X,Z
 1/2+Y,1/2+X,Z
126 16 16 P4/nnc PG4/mmm TETRAGONAL 'P 4/n 2/n 2/c' 'P4/n n c'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 -X,Y,-Z
 X,-Y,-Z
 Y,X,-Z
 -Y,-X,-Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 1/2-Y,1/2+X,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
127 16 16 P4/mbm PG4/mmm TETRAGONAL 'P 4/m 21/b 2/m' 'P4/m b m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2+Y,1/2+X,-Z
 1/2-Y,1/2-X,-Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,-Z
 -Y,X,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2-Y,1/2-X,Z
 1/2+Y,1/2+X,Z
128 16 16 P4/mnc PG4/mmm TETRAGONAL 'P 4/m 21/n 2/c' 'P4/m n c'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,-Z
 -Y,X,-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
129 16 16 P4/nmm PG4/mmm TETRAGONAL 'P 4/n 21/m 2/m' 'P4/n m m'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,1/2+X,Z
 1/2+Y,1/2-X,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 Y,X,-Z
 -Y,-X,-Z
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,-Z
 Y,-X,-Z
 -Y,X,-Z
 X,-Y,Z
 -X,Y,Z
 1/2-Y,1/2-X,Z
 1/2+Y,1/2+X,Z
130 16 16 P4/ncc PG4/mmm TETRAGONAL 'P 4/n 2/c 2/c' 'P4/n c c'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,1/2+X,Z
 1/2+Y,1/2-X,Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 Y,X,1/2-Z
 -Y,-X,1/2-Z
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,-Z
 Y,-X,-Z
 -Y,X,-Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
131 16 16 P42/mmc PG4/mmm TETRAGONAL 'P 42/m 2/m 2/c' 'P42/m m c'
 X,Y,Z
 -X,-Y,Z
 -Y,X,1/2+Z
 Y,-X,1/2+Z
 -X,Y,-Z
 X,-Y,-Z
 Y,X,1/2-Z
 -Y,-X,1/2-Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,1/2-Z
 -Y,X,1/2-Z
 X,-Y,Z
 -X,Y,Z
 -Y,-X,1/2+Z
 Y,X,1/2+Z
132 16 16 P42/mcm PG4/mmm TETRAGONAL 'P 42/m 2/c 2/m' 'P42/m c m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,1/2+Z
 Y,-X,1/2+Z
 -X,Y,1/2-Z
 X,-Y,1/2-Z
 Y,X,-Z
 -Y,-X,-Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,1/2-Z
 -Y,X,1/2-Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 -Y,-X,Z
 Y,X,Z
133 16 16 P42/nbc PG4/mmm TETRAGONAL 'P 42/n 2/b 2/c' 'P42/n b c'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 -X,Y,1/2-Z
 X,-Y,1/2-Z
 1/2+Y,1/2+X,-Z
 1/2-Y,1/2-X,-Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 Y,-X,-Z
 -Y,X,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 -Y,-X,1/2+Z
 Y,X,1/2+Z
134 16 16 P42/nnm PG4/mmm TETRAGONAL 'P 42/n 2/n 2/m' 'P42/n n m'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 -X,Y,-Z
 X,-Y,-Z
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 Y,-X,-Z
 -Y,X,-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 -Y,-X,Z
 Y,X,Z
135 16 16 P42/mbc PG4/mmm TETRAGONAL 'P 42/m 21/b 2/c' 'P42/m b c'
 X,Y,Z
 -X,-Y,Z
 -Y,X,1/2+Z
 Y,-X,1/2+Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,1/2-Z
 -Y,X,1/2-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
136 16 16 P42/mnm PG4/mmm TETRAGONAL 'P 42/m 21/n 2/m' 'P42/m n m'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,X+1/2,Z+1/2
 Y+1/2,1/2-X,Z+1/2
 1/2-X,Y+1/2,1/2-Z
 X+1/2,1/2-Y,1/2-Z
 Y,X,-Z
 -Y,-X,-Z
 -X,-Y,-Z
 X,Y,-Z
 Y+1/2,1/2-X,1/2-Z
 1/2-Y,X+1/2,1/2-Z
 X+1/2,1/2-Y,Z+1/2
 1/2-X,Y+1/2,Z+1/2
 -Y,-X,Z
 Y,X,Z
137 16 16 P42/nmc PG4/mmm TETRAGONAL 'P 42/n 21/m 2/c' 'P42/n m c'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 Y,X,-Z
 -Y,-X,-Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 Y,-X,-Z
 -Y,X,-Z
 X,-Y,Z
 -X,Y,Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
138 16 16 P42/ncm PG4/mmm TETRAGONAL 'P 42/n 21/c 2/m' 'P42/n c m'
 X,Y,Z
 -X,-Y,Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 Y,X,1/2-Z
 -Y,-X,1/2-Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 Y,-X,-Z
 -Y,X,-Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 1/2-Y,1/2-X,Z
 1/2+Y,1/2+X,Z
139 32 16 I4/mmm PG4/mmm TETRAGONAL 'I 4/m 2/m 2/m' 'I4/m m m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 -X,Y,-Z
 X,-Y,-Z
 Y,X,-Z
 -Y,-X,-Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,-Z
 -Y,X,-Z
 X,-Y,Z
 -X,Y,Z
 -Y,-X,Z
 Y,X,Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 1/2-Y,1/2+X,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
140 32 16 I4/mcm PG4/mmm TETRAGONAL 'I 4/m 2/c 2/m' 'I4/m c m'
 X,Y,Z
 -X,-Y,Z
 -Y,X,Z
 Y,-X,Z
 -X,Y,1/2-Z
 X,-Y,1/2-Z
 Y,X,1/2-Z
 -Y,-X,1/2-Z
 -X,-Y,-Z
 X,Y,-Z
 Y,-X,-Z
 -Y,X,-Z
 X,-Y,1/2+Z
 -X,Y,1/2+Z
 -Y,-X,1/2+Z
 Y,X,1/2+Z
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+Y,1/2-X,1/2+Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2+Y,1/2+X,-Z
 1/2-Y,1/2-X,-Z
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 1/2-Y,1/2+X,1/2-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2-Y,1/2-X,Z
 1/2+Y,1/2+X,Z
141 32 16 I41/amd PG4/mmm TETRAGONAL 'I 41/a 2/m 2/d' 'I41/a m d'
 X,Y,Z
 1/2-X,1/2-Y,1/2+Z
 -Y,1/2+X,1/4+Z
 1/2+Y,-X,3/4+Z
 1/2-X,Y,3/4-Z
 X,1/2-Y,1/4-Z
 1/2+Y,1/2+X,1/2-Z
 -Y,-X,-Z
 -X,1/2-Y,1/4-Z
 1/2+X,Y,3/4-Z
 Y,-X,-Z
 1/2-Y,1/2+X,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 -X,Y,Z
 1/2-Y,-X,3/4+Z
 Y,1/2+X,1/4+Z
 1/2+X,1/2+Y,1/2+Z
 -X,-Y,Z
 1/2-Y,X,3/4+Z
 Y,1/2-X,1/4+Z
 -X,1/2+Y,1/4-Z
 1/2+X,-Y,3/4-Z
 Y,X,-Z
 1/2-Y,1/2-X,1/2-Z
 1/2-X,-Y,3/4-Z
 X,1/2+Y,1/4-Z
 1/2+Y,1/2-X,1/2-Z
 -Y,X,-Z
 X,-Y,Z
 1/2-X,1/2+Y,1/2+Z
 -Y,1/2-X,1/4+Z
 1/2+Y,X,3/4+Z
142 32 16 I41/acd PG4/mmm TETRAGONAL 'I 41/a 2/c 2/d' 'I41/a c d'
 X,Y,Z
 1/2-X,1/2-Y,1/2+Z
 -Y,1/2+X,1/4+Z
 1/2+Y,-X,3/4+Z
 1/2-X,Y,1/4-Z
 X,1/2-Y,3/4-Z
 1/2+Y,1/2+X,-Z
 -Y,-X,1/2-Z
 -X,1/2-Y,1/4-Z
 1/2+X,Y,3/4-Z
 Y,-X,-Z
 1/2-Y,1/2+X,1/2-Z
 1/2+X,1/2-Y,Z
 -X,Y,1/2+Z
 1/2-Y,-X,1/4+Z
 Y,1/2+X,3/4+Z
 1/2+X,1/2+Y,1/2+Z
 -X,-Y,Z
 1/2-Y,X,3/4+Z
 Y,1/2-X,1/4+Z
 -X,1/2+Y,3/4-Z
 1/2+X,-Y,1/4-Z
 Y,X,1/2-Z
 1/2-Y,1/2-X,-Z
 1/2-X,-Y,3/4-Z
 X,1/2+Y,1/4-Z
 1/2+Y,1/2-X,1/2-Z
 -Y,X,-Z
 X,-Y,1/2+Z
 1/2-X,1/2+Y,Z
 -Y,1/2-X,3/4+Z
 1/2+Y,X,1/4+Z
143 3 3 P3 PG3 TRIGONAL 'P 3'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
144 3 3 P31 PG3 TRIGONAL 'P 31'
 X,Y,Z
 -Y,X-Y,Z+1/3
 Y-X,-X,Z+2/3
145 3 3 P32 PG3 TRIGONAL 'P 32'
 X,Y,Z
 -Y,X-Y,Z+2/3
 Y-X,-X,Z+1/3
146 9 3 H3 PG3 TRIGONAL 'H 3'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 X+2/3,Y+1/3,Z+1/3
 -Y+2/3,X-Y+1/3,Z+1/3
 Y-X+2/3,-X+1/3,Z+1/3
 X+1/3,Y+2/3,Z+2/3
 -Y+1/3,X-Y+2/3,Z+2/3
 Y-X+1/3,-X+2/3,Z+2/3
1146 3 3 R3 PG3 TRIGONAL 'R 3'
 X,Y,Z
 Z,X,Y
 Y,Z,X
147 6 6 P-3 PG3bar TRIGONAL 'P -3'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
148 18 6 H-3 PG3bar TRIGONAL 'H -3'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 2/3+X,1/3+Y,1/3+Z
 2/3-Y,1/3+X-Y,1/3+Z
 2/3+Y-X,1/3-X,1/3+Z
 2/3-X,1/3-Y,1/3-Z
 2/3+Y,1/3+Y-X,1/3-Z
 2/3+X-Y,1/3+X,1/3-Z
 1/3+X,2/3+Y,2/3+Z
 1/3-Y,2/3+X-Y,2/3+Z
 1/3+Y-X,2/3-X,2/3+Z
 1/3-X,2/3-Y,2/3-Z
 1/3+Y,2/3+Y-X,2/3-Z
 1/3+X-Y,2/3+X,2/3-Z
1148 6 6 R-3 PG3bar TRIGONAL 'R -3'
 X,Y,Z
 Z,X,Y
 Y,Z,X
 -X,-Y,-Z
 -Z,-X,-Y
 -Y,-Z,-X
149 6 6 P312 PG312 TRIGONAL 'P 3 1 2'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -Y,-X,-Z
 Y-X,Y,-Z
 X,X-Y,-Z
150 6 6 P321 PG321 TRIGONAL 'P 3 2 1'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 Y,X,-Z
 X-Y,-Y,-Z
 -X,Y-X,-Z
151 6 6 P3112 PG312 TRIGONAL 'P 31 1 2'
 X,Y,Z
 -Y,X-Y,1/3+Z
 Y-X,-X,2/3+Z
 -Y,-X,2/3-Z
 Y-X,Y,1/3-Z
 X,X-Y,-Z
152 6 6 P3121 PG321 TRIGONAL 'P 31 2 1'
 X,Y,Z
 -Y,X-Y,Z+1/3
 Y-X,-X,Z+2/3
 Y,X,-Z
 X-Y,-Y,2/3-Z
 -X,Y-X,1/3-Z
153 6 6 P3212 PG312 TRIGONAL 'P 32 1 2'
 X,Y,Z
 -Y,X-Y,2/3+Z
 Y-X,-X,1/3+Z
 -Y,-X,1/3-Z
 Y-X,Y,2/3-Z
 X,X-Y,-Z
154 6 6 P3221 PG321 TRIGONAL 'P 32 2 1'
 X,Y,Z
 -Y,X-Y,Z+2/3
 Y-X,-X,Z+1/3
 Y,X,-Z
 X-Y,-Y,1/3-Z
 -X,Y-X,2/3-Z
155 18 6 H32 PG321 TRIGONAL 'H 3 2'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 Y,X,-Z
 X-Y,-Y,-Z
 -X,Y-X,-Z
 2/3+X,1/3+Y,1/3+Z
 2/3-Y,1/3+X-Y,1/3+Z
 2/3+Y-X,1/3-X,1/3+Z
 2/3+Y,1/3+X,1/3-Z
 2/3+X-Y,1/3-Y,1/3-Z
 2/3-X,1/3+Y-X,1/3-Z
 1/3+X,2/3+Y,2/3+Z
 1/3-Y,2/3+X-Y,2/3+Z
 1/3+Y-X,2/3-X,2/3+Z
 1/3+Y,2/3+X,2/3-Z
 1/3+X-Y,2/3-Y,2/3-Z
 1/3-X,2/3+Y-X,2/3-Z
1155 6 6 R32 PG32 TRIGONAL 'R 3 2'
 X,Y,Z
 Z,X,Y
 Y,Z,X
 -Y,-X,-Z
 -X,-Z,-Y
 -Z,-Y,-X
156 6 6 P3m1 PG3m1 TRIGONAL 'P 3 m 1'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -Y,-X,Z
 Y-X,Y,Z
 X,X-Y,Z
157 6 6 P31m PG31m TRIGONAL 'P 3 1 m'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 Y,X,Z
 X-Y,-Y,Z
 -X,Y-X,Z
158 6 6 P3c1 PG3m1 TRIGONAL 'P 3 c 1'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -Y,-X,1/2+Z
 Y-X,Y,1/2+Z
 X,X-Y,1/2+Z
159 6 6 P31c PG31m TRIGONAL 'P 3 1 c'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 Y,X,1/2+Z
 X-Y,-Y,1/2+Z
 -X,Y-X,1/2+Z
160 18 6 H3m PG3m TRIGONAL 'H 3 m'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -Y,-X,Z
 Y-X,Y,Z
 X,X-Y,Z
 2/3+X,1/3+Y,1/3+Z
 2/3-Y,1/3+X-Y,1/3+Z
 2/3+Y-X,1/3-X,1/3+Z
 2/3-Y,1/3-X,1/3+Z
 2/3+Y-X,1/3+Y,1/3+Z
 2/3+X,1/3+X-Y,1/3+Z
 1/3+X,2/3+Y,2/3+Z
 1/3-Y,2/3+X-Y,2/3+Z
 1/3+Y-X,2/3-X,2/3+Z
 1/3-Y,2/3-X,2/3+Z
 1/3+Y-X,2/3+Y,2/3+Z
 1/3+X,2/3+X-Y,2/3+Z
1160 6 6 R3m PG3m TRIGONAL 'R 3 m'
 X,Y,Z
 Z,X,Y
 Y,Z,X
 Y,X,Z
 X,Z,Y
 Z,Y,X
161 18 6 H3c PG3m TRIGONAL 'H 3 c'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -Y,-X,1/2+Z
 Y-X,Y,1/2+Z
 X,X-Y,1/2+Z
 2/3+X,1/3+Y,1/3+Z
 2/3-Y,1/3+X-Y,1/3+Z
 2/3+Y-X,1/3-X,1/3+Z
 2/3-Y,1/3-X,5/6+Z
 2/3+Y-X,1/3+Y,5/6+Z
 2/3+X,1/3+X-Y,5/6+Z
 1/3+X,2/3+Y,2/3+Z
 1/3-Y,2/3+X-Y,2/3+Z
 1/3+Y-X,2/3-X,2/3+Z
 1/3-Y,2/3-X,1/6+Z
 1/3+Y-X,2/3+Y,1/6+Z
 1/3+X,2/3+X-Y,1/6+Z
1161 6 6 R3c PG3m TRIGONAL 'R 3 c'
 X,Y,Z
 Z,X,Y
 Y,Z,X
 Y+1/2,X+1/2,Z+1/2
 X+1/2,Z+1/2,Y+1/2
 Z+1/2,Y+1/2,X+1/2
162 12 12 P-31m PG3bar1m TRIGONAL 'P -3 1 2/m' 'P -3 1 m'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -Y,-X,-Z
 Y-X,Y,-Z
 X,X-Y,-Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 Y,X,Z
 X-Y,-Y,Z
 -X,Y-X,Z
163 12 12 P-31c PG3bar1m TRIGONAL 'P -3 1 2/c' 'P -3 1 c'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -Y,-X,1/2-Z
 Y-X,Y,1/2-Z
 X,X-Y,1/2-Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 Y,X,1/2+Z
 X-Y,-Y,1/2+Z
 -X,Y-X,1/2+Z
164 12 12 P-3m1 PG3barm1 TRIGONAL 'P -3 2/m 1' 'P -3 m 1'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 Y,X,-Z
 X-Y,-Y,-Z
 -X,Y-X,-Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 -Y,-X,Z
 Y-X,Y,Z
 X,X-Y,Z
165 12 12 P-3c1 PG3barm1 TRIGONAL 'P -3 2/c 1' 'P -3 c 1'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 Y,X,1/2-Z
 X-Y,-Y,1/2-Z
 -X,Y-X,1/2-Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 -Y,-X,1/2+Z
 Y-X,Y,1/2+Z
 X,X-Y,1/2+Z
166 36 12 H-3m PG3barm TRIGONAL 'H -3 2/m' 'H -3 m'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 Y,X,-Z
 X-Y,-Y,-Z
 -X,Y-X,-Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 -Y,-X,Z
 Y-X,Y,Z
 X,X-Y,Z
 2/3+X,1/3+Y,1/3+Z
 2/3-Y,1/3+X-Y,1/3+Z
 2/3+Y-X,1/3-X,1/3+Z
 2/3+Y,1/3+X,1/3-Z
 2/3+X-Y,1/3-Y,1/3-Z
 2/3-X,1/3+Y-X,1/3-Z
 2/3-X,1/3-Y,1/3-Z
 2/3+Y,1/3+Y-X,1/3-Z
 2/3+X-Y,1/3+X,1/3-Z
 2/3-Y,1/3-X,1/3+Z
 2/3+Y-X,1/3+Y,1/3+Z
 2/3+X,1/3+X-Y,1/3+Z
 1/3+X,2/3+Y,2/3+Z
 1/3-Y,2/3+X-Y,2/3+Z
 1/3+Y-X,2/3-X,2/3+Z
 1/3+Y,2/3+X,2/3-Z
 1/3+X-Y,2/3-Y,2/3-Z
 1/3-X,2/3+Y-X,2/3-Z
 1/3-X,2/3-Y,2/3-Z
 1/3+Y,2/3+Y-X,2/3-Z
 1/3+X-Y,2/3+X,2/3-Z
 1/3-Y,2/3-X,2/3+Z
 1/3+Y-X,2/3+Y,2/3+Z
 1/3+X,2/3+X-Y,2/3+Z
1166 12 12 R-3m PG3barm TRIGONAL 'R -3 2/m' 'R -3 m'
 X,Y,Z
 Z,X,Y
 Y,Z,X
 -Y,-X,-Z
 -X,-Z,-Y
 -Z,-Y,-X
 -X,-Y,-Z
 -Z,-X,-Y
 -Y,-Z,-X
 Y,X,Z
 X,Z,Y
 Z,Y,X
167 36 12 H-3c PG3barm TRIGONAL 'H -3 2/c' 'H -3 c'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 Y,X,1/2-Z
 X-Y,-Y,1/2-Z
 -X,Y-X,1/2-Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 -Y,-X,1/2+Z
 Y-X,Y,1/2+Z
 X,X-Y,1/2+Z
 2/3+X,1/3+Y,1/3+Z
 2/3-Y,1/3+X-Y,1/3+Z
 2/3+Y-X,1/3-X,1/3+Z
 2/3+Y,1/3+X,5/6-Z
 2/3+X-Y,1/3-Y,5/6-Z
 2/3-X,1/3+Y-X,5/6-Z
 2/3-X,1/3-Y,1/3-Z
 2/3+Y,1/3+Y-X,1/3-Z
 2/3+X-Y,1/3+X,1/3-Z
 2/3-Y,1/3-X,5/6+Z
 2/3+Y-X,1/3+Y,5/6+Z
 2/3+X,1/3+X-Y,5/6+Z
 1/3+X,2/3+Y,2/3+Z
 1/3-Y,2/3+X-Y,2/3+Z
 1/3+Y-X,2/3-X,2/3+Z
 1/3+Y,2/3+X,1/6-Z
 1/3+X-Y,2/3-Y,1/6-Z
 1/3-X,2/3+Y-X,1/6-Z
 1/3-X,2/3-Y,2/3-Z
 1/3+Y,2/3+Y-X,2/3-Z
 1/3+X-Y,2/3+X,2/3-Z
 1/3-Y,2/3-X,1/6+Z
 1/3+Y-X,2/3+Y,1/6+Z
 1/3+X,2/3+X-Y,1/6+Z
1167 12 12 R-3c PG3barm TRIGONAL 'R -3 2/c' 'R -3 c'
 X,Y,Z
 Z,X,Y
 Y,Z,X
 -Y+1/2,-X+1/2,-Z+1/2
 -X+1/2,-Z+1/2,-Y+1/2
 -Z+1/2,-Y+1/2,-X+1/2
 -X,-Y,-Z
 -Z,-X,-Y
 -Y,-Z,-X
 Y+1/2,X+1/2,Z+1/2
 X+1/2,Z+1/2,Y+1/2
 Z+1/2,Y+1/2,X+1/2
168 6 6 P6 PG6 HEXAGONAL 'P 6'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,Z
 Y,Y-X,Z
 X-Y,X,Z
169 6 6 P61 PG6 HEXAGONAL 'P 61'
 X,Y,Z
 -Y,X-Y,Z+1/3
 Y-X,-X,Z+2/3
 -X,-Y,Z+1/2
 Y,Y-X,Z+5/6
 X-Y,X,Z+1/6
170 6 6 P65 PG6 HEXAGONAL 'P 65'
 X,Y,Z
 -Y,X-Y,Z+2/3
 Y-X,-X,Z+1/3
 -X,-Y,Z+1/2
 Y,Y-X,Z+1/6
 X-Y,X,Z+5/6
171 6 6 P62 PG6 HEXAGONAL 'P 62'
 X,Y,Z
 -Y,X-Y,2/3+Z
 Y-X,-X,1/3+Z
 -X,-Y,Z
 Y,Y-X,2/3+Z
 X-Y,X,1/3+Z
172 6 6 P64 PG6 HEXAGONAL 'P 64'
 X,Y,Z
 -Y,X-Y,1/3+Z
 Y-X,-X,2/3+Z
 -X,-Y,Z
 Y,Y-X,1/3+Z
 X-Y,X,2/3+Z
173 6 6 P63 PG6 HEXAGONAL 'P 63'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,1/2+Z
 Y,Y-X,1/2+Z
 X-Y,X,1/2+Z
174 6 6 P-6 PG6bar HEXAGONAL 'P -6'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 X,Y,-Z
 -Y,X-Y,-Z
 Y-X,-X,-Z
175 12 12 P6/m PG6/m HEXAGONAL 'P 6/m'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,Z
 Y,Y-X,Z
 X-Y,X,Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 X,Y,-Z
 -Y,X-Y,-Z
 Y-X,-X,-Z
176 12 12 P63/m PG6/m HEXAGONAL 'P 63/m'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,1/2+Z
 Y,Y-X,1/2+Z
 X-Y,X,1/2+Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 X,Y,1/2-Z
 -Y,X-Y,1/2-Z
 Y-X,-X,1/2-Z
177 12 12 P622 PG622 HEXAGONAL 'P 6 2 2'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,Z
 Y,Y-X,Z
 X-Y,X,Z
 Y,X,-Z
 X-Y,-Y,-Z
 -X,Y-X,-Z
 -Y,-X,-Z
 Y-X,Y,-Z
 X,X-Y,-Z
178 12 12 P6122 PG622 HEXAGONAL 'P 61 2 2'
 X,Y,Z
 -Y,X-Y,1/3+Z
 Y-X,-X,2/3+Z
 -X,-Y,1/2+Z
 Y,Y-X,5/6+Z
 X-Y,X,1/6+Z
 Y,X,1/3-Z
 X-Y,-Y,-Z
 -X,Y-X,2/3-Z
 -Y,-X,5/6-Z
 Y-X,Y,1/2-Z
 X,X-Y,1/6-Z
179 12 12 P6522 PG622 HEXAGONAL 'P 65 2 2'
 X,Y,Z
 -Y,X-Y,2/3+Z
 Y-X,-X,1/3+Z
 -X,-Y,1/2+Z
 Y,Y-X,1/6+Z
 X-Y,X,5/6+Z
 Y,X,2/3-Z
 X-Y,-Y,-Z
 -X,Y-X,1/3-Z
 -Y,-X,1/6-Z
 Y-X,Y,1/2-Z
 X,X-Y,5/6-Z
180 12 12 P6222 PG622 HEXAGONAL 'P 62 2 2'
 X,Y,Z
 -Y,X-Y,2/3+Z
 Y-X,-X,1/3+Z
 -X,-Y,Z
 Y,Y-X,2/3+Z
 X-Y,X,1/3+Z
 Y,X,2/3-Z
 X-Y,-Y,-Z
 -X,Y-X,1/3-Z
 -Y,-X,2/3-Z
 Y-X,Y,-Z
 X,X-Y,1/3-Z
181 12 12 P6422 PG622 HEXAGONAL 'P 64 2 2'
 X,Y,Z
 -Y,X-Y,1/3+Z
 Y-X,-X,2/3+Z
 -X,-Y,Z
 Y,Y-X,1/3+Z
 X-Y,X,2/3+Z
 Y,X,1/3-Z
 X-Y,-Y,-Z
 -X,Y-X,2/3-Z
 -Y,-X,1/3-Z
 Y-X,Y,-Z
 X,X-Y,2/3-Z
182 12 12 P6322 PG622 HEXAGONAL 'P 63 2 2'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,1/2+Z
 Y,Y-X,1/2+Z
 X-Y,X,1/2+Z
 Y,X,-Z
 X-Y,-Y,-Z
 -X,Y-X,-Z
 -Y,-X,1/2-Z
 Y-X,Y,1/2-Z
 X,X-Y,1/2-Z
183 12 12 P6mm PG6mm HEXAGONAL 'P 6 m m'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,Z
 Y,Y-X,Z
 X-Y,X,Z
 -Y,-X,Z
 Y-X,Y,Z
 X,X-Y,Z
 Y,X,Z
 X-Y,-Y,Z
 -X,Y-X,Z
184 12 12 P6cc PG6mm HEXAGONAL 'P 6 c c'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,Z
 Y,Y-X,Z
 X-Y,X,Z
 -Y,-X,1/2+Z
 Y-X,Y,1/2+Z
 X,X-Y,1/2+Z
 Y,X,1/2+Z
 X-Y,-Y,1/2+Z
 -X,Y-X,1/2+Z
185 12 12 P63cm PG6mm HEXAGONAL 'P 63 c m'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,1/2+Z
 Y,Y-X,1/2+Z
 X-Y,X,1/2+Z
 -Y,-X,1/2+Z
 Y-X,Y,1/2+Z
 X,X-Y,1/2+Z
 Y,X,Z
 X-Y,-Y,Z
 -X,Y-X,Z
186 12 12 P63mc PG6mm HEXAGONAL 'P 63 m c'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,1/2+Z
 Y,Y-X,1/2+Z
 X-Y,X,1/2+Z
 -Y,-X,Z
 Y-X,Y,Z
 X,X-Y,Z
 Y,X,1/2+Z
 X-Y,-Y,1/2+Z
 -X,Y-X,1/2+Z
187 12 12 P-6m2 PG6barm2 HEXAGONAL 'P -6 m 2'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 X,Y,-Z
 -Y,X-Y,-Z
 Y-X,-X,-Z
 -Y,-X,Z
 Y-X,Y,Z
 X,X-Y,Z
 -Y,-X,-Z
 Y-X,Y,-Z
 X,X-Y,-Z
188 12 12 P-6c2 PG6barm2 HEXAGONAL 'P -6 c 2'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 X,Y,1/2-Z
 -Y,X-Y,1/2-Z
 Y-X,-X,1/2-Z
 -Y,-X,1/2+Z
 Y-X,Y,1/2+Z
 X,X-Y,1/2+Z
 -Y,-X,-Z
 Y-X,Y,-Z
 X,X-Y,-Z
189 12 12 P-62m PG6bar2m HEXAGONAL 'P -6 2 m'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 X,Y,-Z
 -Y,X-Y,-Z
 Y-X,-X,-Z
 Y,X,-Z
 X-Y,-Y,-Z
 -X,Y-X,-Z
 Y,X,Z
 X-Y,-Y,Z
 -X,Y-X,Z
190 12 12 P-62c PG6bar2m HEXAGONAL 'P -6 2 c'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 X,Y,1/2-Z
 -Y,X-Y,1/2-Z
 Y-X,-X,1/2-Z
 Y,X,-Z
 X-Y,-Y,-Z
 -X,Y-X,-Z
 Y,X,1/2+Z
 X-Y,-Y,1/2+Z
 -X,Y-X,1/2+Z
191 24 24 P6/mmm PG6/mmm HEXAGONAL 'P 6/m 2/m 2/m' 'P 6/m m m'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,Z
 Y,Y-X,Z
 X-Y,X,Z
 Y,X,-Z
 X-Y,-Y,-Z
 -X,Y-X,-Z
 -Y,-X,-Z
 Y-X,Y,-Z
 X,X-Y,-Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 X,Y,-Z
 Y-X,-X,-Z
 -Y,X-Y,-Z
 -Y,-X,Z
 Y-X,Y,Z
 X,X-Y,Z
 Y,X,Z
 X-Y,-Y,Z
 -X,Y-X,Z
192 24 24 P6/mcc PG6/mmm HEXAGONAL 'P 6/m 2/c 2/c' 'P 6/m c c'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,Z
 Y,Y-X,Z
 X-Y,X,Z
 Y,X,1/2-Z
 X-Y,-Y,1/2-Z
 -X,Y-X,1/2-Z
 -Y,-X,1/2-Z
 Y-X,Y,1/2-Z
 X,X-Y,1/2-Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 X,Y,-Z
 Y-X,-X,-Z
 -Y,X-Y,-Z
 -Y,-X,1/2+Z
 Y-X,Y,1/2+Z
 X,X-Y,1/2+Z
 Y,X,1/2+Z
 X-Y,-Y,1/2+Z
 -X,Y-X,1/2+Z
193 24 24 P63/mcm PG6/mmm HEXAGONAL 'P 63/m 2/c 2/m' 'P 63/m c m'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,1/2+Z
 Y,Y-X,1/2+Z
 X-Y,X,1/2+Z
 Y,X,1/2-Z
 X-Y,-Y,1/2-Z
 -X,Y-X,1/2-Z
 -Y,-X,-Z
 Y-X,Y,-Z
 X,X-Y,-Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 X,Y,1/2-Z
 Y-X,-X,1/2-Z
 -Y,X-Y,1/2-Z
 -Y,-X,1/2+Z
 Y-X,Y,1/2+Z
 X,X-Y,1/2+Z
 Y,X,Z
 X-Y,-Y,Z
 -X,Y-X,Z
194 24 24 P63/mmc PG6/mmm HEXAGONAL 'P 63/m 2/m 2/c' 'P 63/m m c'
 X,Y,Z
 -Y,X-Y,Z
 Y-X,-X,Z
 -X,-Y,1/2+Z
 Y,Y-X,1/2+Z
 X-Y,X,1/2+Z
 Y,X,-Z
 X-Y,-Y,-Z
 -X,Y-X,-Z
 -Y,-X,1/2-Z
 Y-X,Y,1/2-Z
 X,X-Y,1/2-Z
 -X,-Y,-Z
 Y,Y-X,-Z
 X-Y,X,-Z
 X,Y,1/2-Z
 Y-X,-X,1/2-Z
 -Y,X-Y,1/2-Z
 -Y,-X,Z
 Y-X,Y,Z
 X,X-Y,Z
 Y,X,1/2+Z
 X-Y,-Y,1/2+Z
 -X,Y-X,1/2+Z
195 12 12 P23 PG23 CUBIC 'P 2 3'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
196 48 12 F23 PG23 CUBIC 'F 2 3'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 Z,1/2+X,1/2+Y
 Z,1/2-X,1/2-Y
 -Z,1/2-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,1/2+Z,1/2+X
 -Y,1/2+Z,1/2-X
 Y,1/2-Z,1/2-X
 -Y,1/2-Z,1/2+X
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 1/2+Z,X,1/2+Y
 1/2+Z,-X,1/2-Y
 1/2-Z,-X,1/2+Y
 1/2-Z,X,1/2-Y
 1/2+Y,Z,1/2+X
 1/2-Y,Z,1/2-X
 1/2+Y,-Z,1/2-X
 1/2-Y,-Z,1/2+X
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2+Z,1/2+X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,1/2-X,Y
 1/2-Z,1/2+X,-Y
 1/2+Y,1/2+Z,X
 1/2-Y,1/2+Z,-X
 1/2+Y,1/2-Z,-X
 1/2-Y,1/2-Z,X
197 24 12 I23 PG23 CUBIC 'I 2 3'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2+Z,1/2+X,1/2+Y
 1/2+Z,1/2-X,1/2-Y
 1/2-Z,1/2-X,1/2+Y
 1/2-Z,1/2+X,1/2-Y
 1/2+Y,1/2+Z,1/2+X
 1/2-Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,1/2-X
 1/2-Y,1/2-Z,1/2+X
198 12 12 P213 PG23 CUBIC 'P 21 3'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,Z,X
 -Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,-X
 1/2-Y,-Z,1/2+X
199 24 12 I213 PG23 CUBIC 'I 21 3'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,Z,X
 -Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,-X
 1/2-Y,-Z,1/2+X
 1/2+X,1/2+Y,1/2+Z
 -X,1/2-Y,Z
 1/2-X,Y,-Z
 X,-Y,1/2-Z
 1/2+Z,1/2+X,1/2+Y
 Z,-X,1/2-Y
 -Z,1/2-X,Y
 1/2-Z,X,-Y
 1/2+Y,1/2+Z,1/2+X
 1/2-Y,Z,-X
 Y,-Z,1/2-X
 -Y,1/2-Z,X
200 24 24 Pm-3 PGm3bar CUBIC 'P 2/m -3' 'P m -3'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
 -Z,-X,-Y
 -Z,X,Y
 Z,X,-Y
 Z,-X,Y
 -Y,-Z,-X
 Y,-Z,X
 -Y,Z,X
 Y,Z,-X
201 24 24 Pn-3 PGm3bar CUBIC 'P 2/n -3' 'P n -3'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2-Z,1/2-X,1/2-Y
 1/2-Z,1/2+X,1/2+Y
 1/2+Z,1/2+X,1/2-Y
 1/2+Z,1/2-X,1/2+Y
 1/2-Y,1/2-Z,1/2-X
 1/2+Y,1/2-Z,1/2+X
 1/2-Y,1/2+Z,1/2+X
 1/2+Y,1/2+Z,1/2-X
202 96 24 Fm-3 PGm3bar CUBIC 'F 2/m -3' 'F m -3'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
 -Z,-X,-Y
 -Z,X,Y
 Z,X,-Y
 Z,-X,Y
 -Y,-Z,-X
 Y,-Z,X
 -Y,Z,X
 Y,Z,-X
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 Z,1/2+X,1/2+Y
 Z,1/2-X,1/2-Y
 -Z,1/2-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,1/2+Z,1/2+X
 -Y,1/2+Z,1/2-X
 Y,1/2-Z,1/2-X
 -Y,1/2-Z,1/2+X
 -X,1/2-Y,1/2-Z
 X,1/2+Y,1/2-Z
 X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2+Z
 -Z,1/2-X,1/2-Y
 -Z,1/2+X,1/2+Y
 Z,1/2+X,1/2-Y
 Z,1/2-X,1/2+Y
 -Y,1/2-Z,1/2-X
 Y,1/2-Z,1/2+X
 -Y,1/2+Z,1/2+X
 Y,1/2+Z,1/2-X
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 1/2+Z,X,1/2+Y
 1/2+Z,-X,1/2-Y
 1/2-Z,-X,1/2+Y
 1/2-Z,X,1/2-Y
 1/2+Y,Z,1/2+X
 1/2-Y,Z,1/2-X
 1/2+Y,-Z,1/2-X
 1/2-Y,-Z,1/2+X
 1/2-X,-Y,1/2-Z
 1/2+X,Y,1/2-Z
 1/2+X,-Y,1/2+Z
 1/2-X,Y,1/2+Z
 1/2-Z,-X,1/2-Y
 1/2-Z,X,1/2+Y
 1/2+Z,X,1/2-Y
 1/2+Z,-X,1/2+Y
 1/2-Y,-Z,1/2-X
 1/2+Y,-Z,1/2+X
 1/2-Y,Z,1/2+X
 1/2+Y,Z,1/2-X
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2+Z,1/2+X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,1/2-X,Y
 1/2-Z,1/2+X,-Y
 1/2+Y,1/2+Z,X
 1/2-Y,1/2+Z,-X
 1/2+Y,1/2-Z,-X
 1/2-Y,1/2-Z,X
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2-Z,1/2-X,-Y
 1/2-Z,1/2+X,Y
 1/2+Z,1/2+X,-Y
 1/2+Z,1/2-X,Y
 1/2-Y,1/2-Z,-X
 1/2+Y,1/2-Z,X
 1/2-Y,1/2+Z,X
 1/2+Y,1/2+Z,-X
203 96 24 Fd-3 PGm3bar CUBIC 'F 2/d -3' 'F d -3'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 1/4-X,1/4-Y,1/4-Z
 1/4+X,1/4+Y,1/4-Z
 1/4+X,1/4-Y,1/4+Z
 1/4-X,1/4+Y,1/4+Z
 1/4-Z,1/4-X,1/4-Y
 1/4-Z,1/4+X,1/4+Y
 1/4+Z,1/4+X,1/4-Y
 1/4+Z,1/4-X,1/4+Y
 1/4-Y,1/4-Z,1/4-X
 1/4+Y,1/4-Z,1/4+X
 1/4-Y,1/4+Z,1/4+X
 1/4+Y,1/4+Z,1/4-X
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 Z,1/2+X,1/2+Y
 Z,1/2-X,1/2-Y
 -Z,1/2-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,1/2+Z,1/2+X
 -Y,1/2+Z,1/2-X
 Y,1/2-Z,1/2-X
 -Y,1/2-Z,1/2+X
 1/4-X,3/4-Y,3/4-Z
 1/4+X,3/4+Y,3/4-Z
 1/4+X,3/4-Y,3/4+Z
 1/4-X,3/4+Y,3/4+Z
 1/4-Z,3/4-X,3/4-Y
 1/4-Z,3/4+X,3/4+Y
 1/4+Z,3/4+X,3/4-Y
 1/4+Z,3/4-X,3/4+Y
 1/4-Y,3/4-Z,3/4-X
 1/4+Y,3/4-Z,3/4+X
 1/4-Y,3/4+Z,3/4+X
 1/4+Y,3/4+Z,3/4-X
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 1/2+Z,X,1/2+Y
 1/2+Z,-X,1/2-Y
 1/2-Z,-X,1/2+Y
 1/2-Z,X,1/2-Y
 1/2+Y,Z,1/2+X
 1/2-Y,Z,1/2-X
 1/2+Y,-Z,1/2-X
 1/2-Y,-Z,1/2+X
 3/4-X,1/4-Y,3/4-Z
 3/4+X,1/4+Y,3/4-Z
 3/4+X,1/4-Y,3/4+Z
 3/4-X,1/4+Y,3/4+Z
 3/4-Z,1/4-X,3/4-Y
 3/4-Z,1/4+X,3/4+Y
 3/4+Z,1/4+X,3/4-Y
 3/4+Z,1/4-X,3/4+Y
 3/4-Y,1/4-Z,3/4-X
 3/4+Y,1/4-Z,3/4+X
 3/4-Y,1/4+Z,3/4+X
 3/4+Y,1/4+Z,3/4-X
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2+Z,1/2+X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,1/2-X,Y
 1/2-Z,1/2+X,-Y
 1/2+Y,1/2+Z,X
 1/2-Y,1/2+Z,-X
 1/2+Y,1/2-Z,-X
 1/2-Y,1/2-Z,X
 3/4-X,3/4-Y,1/4-Z
 3/4+X,3/4+Y,1/4-Z
 3/4+X,3/4-Y,Z+1/4
 3/4-X,3/4+Y,Z+1/4
 3/4-Z,3/4-X,1/4-Y
 3/4-Z,3/4+X,1/4+Y
 3/4+Z,3/4+X,1/4-Y
 3/4+Z,3/4-X,1/4+Y
 3/4-Y,3/4-Z,1/4-X
 3/4+Y,3/4-Z,1/4+X
 3/4-Y,3/4+Z,1/4+X
 3/4+Y,3/4+Z,1/4-X
204 48 24 Im-3 PGm3bar CUBIC 'I 2/m -3' 'I m -3'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
 -Z,-X,-Y
 -Z,X,Y
 Z,X,-Y
 Z,-X,Y
 -Y,-Z,-X
 Y,-Z,X
 -Y,Z,X
 Y,Z,-X
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2+Z,1/2+X,1/2+Y
 1/2+Z,1/2-X,1/2-Y
 1/2-Z,1/2-X,1/2+Y
 1/2-Z,1/2+X,1/2-Y
 1/2+Y,1/2+Z,1/2+X
 1/2-Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,1/2-X
 1/2-Y,1/2-Z,1/2+X
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2-Z,1/2-X,1/2-Y
 1/2-Z,1/2+X,1/2+Y
 1/2+Z,1/2+X,1/2-Y
 1/2+Z,1/2-X,1/2+Y
 1/2-Y,1/2-Z,1/2-X
 1/2+Y,1/2-Z,1/2+X
 1/2-Y,1/2+Z,1/2+X
 1/2+Y,1/2+Z,1/2-X
205 24 24 Pa-3 PGm3bar CUBIC 'P 21/a -3' 'P a -3'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,Z,X
 -Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,-X
 1/2-Y,-Z,1/2+X
 -X,-Y,-Z
 1/2+X,Y,1/2-Z
 X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,Z
 -Z,-X,-Y
 1/2-Z,1/2+X,Y
 1/2+Z,X,1/2-Y
 Z,1/2-X,1/2+Y
 -Y,-Z,-X
 Y,1/2-Z,1/2+X
 1/2-Y,1/2+Z,X
 1/2+Y,Z,1/2-X
206 48 24 Ia-3 PGm3bar CUBIC 'I 21/a -3' 'I a -3'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,Z,X
 -Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,-X
 1/2-Y,-Z,1/2+X
 -X,-Y,-Z
 1/2+X,Y,1/2-Z
 X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,Z
 -Z,-X,-Y
 1/2-Z,1/2+X,Y
 1/2+Z,X,1/2-Y
 Z,1/2-X,1/2+Y
 -Y,-Z,-X
 Y,1/2-Z,1/2+X
 1/2-Y,1/2+Z,X
 1/2+Y,Z,1/2-X
 1/2+X,1/2+Y,1/2+Z
 -X,1/2-Y,Z
 1/2-X,+Y,-Z
 X,-Y,1/2-Z
 1/2+Z,1/2+X,1/2+Y
 Z,-X,1/2-Y
 -Z,1/2-X,Y
 1/2-Z,X,-Y
 1/2+Y,1/2+Z,1/2+X
 1/2-Y,Z,-X
 Y,-Z,1/2-X
 -Y,1/2-Z,X
 1/2-X,1/2-Y,1/2-Z
 X,1/2+Y,-Z
 1/2+X,-Y,Z
 -X,Y,1/2+Z
 1/2-Z,1/2-X,1/2-Y
 -Z,X,1/2+Y
 Z,1/2+X,-Y
 1/2+Z,-X,Y
 1/2-Y,1/2-Z,1/2-X
 1/2+Y,-Z,X
 -Y,Z,1/2+X
 Y,1/2+Z,-X
207 24 24 P432 PG432 CUBIC 'P 4 3 2'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 Y,X,-Z
 -Y,-X,-Z
 Y,-X,Z
 -Y,X,Z
 X,Z,-Y
 -X,Z,Y
 -X,-Z,-Y
 X,-Z,Y
 Z,Y,-X
 Z,-Y,X
 -Z,Y,X
 -Z,-Y,-X
208 24 24 P4232 PG432 CUBIC 'P 42 3 2'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
 1/2+Y,1/2-X,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+X,1/2+Z,1/2-Y
 1/2-X,1/2+Z,1/2+Y
 1/2-X,1/2-Z,1/2-Y
 1/2+X,1/2-Z,1/2+Y
 1/2+Z,1/2+Y,1/2-X
 1/2+Z,1/2-Y,1/2+X
 1/2-Z,1/2+Y,1/2+X
 1/2-Z,1/2-Y,1/2-X
209 96 24 F432 PG432 CUBIC 'F 4 3 2'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 Y,X,-Z
 -Y,-X,-Z
 Y,-X,Z
 -Y,X,Z
 X,Z,-Y
 -X,Z,Y
 -X,-Z,-Y
 X,-Z,Y
 Z,Y,-X
 Z,-Y,X
 -Z,Y,X
 -Z,-Y,-X
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 Z,1/2+X,1/2+Y
 Z,1/2-X,1/2-Y
 -Z,1/2-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,1/2+Z,1/2+X
 -Y,1/2+Z,1/2-X
 Y,1/2-Z,1/2-X
 -Y,1/2-Z,1/2+X
 Y,1/2+X,1/2-Z
 -Y,1/2-X,1/2-Z
 Y,1/2-X,1/2+Z
 -Y,1/2+X,1/2+Z
 X,1/2+Z,1/2-Y
 -X,1/2+Z,1/2+Y
 -X,1/2-Z,1/2-Y
 X,1/2-Z,1/2+Y
 Z,1/2+Y,1/2-X
 Z,1/2-Y,1/2+X
 -Z,1/2+Y,1/2+X
 -Z,1/2-Y,1/2-X
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 1/2+Z,X,1/2+Y
 1/2+Z,-X,1/2-Y
 1/2-Z,-X,1/2+Y
 1/2-Z,X,1/2-Y
 1/2+Y,Z,1/2+X
 1/2-Y,Z,1/2-X
 1/2+Y,-Z,1/2-X
 1/2-Y,-Z,1/2+X
 1/2+Y,X,1/2-Z
 1/2-Y,-X,1/2-Z
 1/2+Y,-X,1/2+Z
 1/2-Y,X,1/2+Z
 1/2+X,Z,1/2-Y
 1/2-X,Z,1/2+Y
 1/2-X,-Z,1/2-Y
 1/2+X,-Z,1/2+Y
 1/2+Z,Y,1/2-X
 1/2+Z,-Y,1/2+X
 1/2-Z,Y,1/2+X
 1/2-Z,-Y,1/2-X
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2+Z,1/2+X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,1/2-X,Y
 1/2-Z,1/2+X,-Y
 1/2+Y,1/2+Z,X
 1/2-Y,1/2+Z,-X
 1/2+Y,1/2-Z,-X
 1/2-Y,1/2-Z,X
 1/2+Y,1/2+X,-Z
 1/2-Y,1/2-X,-Z
 1/2+Y,1/2-X,Z
 1/2-Y,1/2+X,Z
 1/2+X,1/2+Z,-Y
 1/2-X,1/2+Z,Y
 1/2-X,1/2-Z,-Y
 1/2+X,1/2-Z,Y
 1/2+Z,1/2+Y,-X
 1/2+Z,1/2-Y,X
 1/2-Z,1/2+Y,X
 1/2-Z,1/2-Y,-X
210 96 24 F4132 PG432 CUBIC 'F 41 3 2'
 X,Y,Z
 -X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,-Z
 1/2+X,-Y,1/2-Z
 Z,X,Y
 1/2+Z,-X,1/2-Y
 -Z,1/2-X,1/2+Y
 1/2-Z,1/2+X,-Y
 Y,Z,X
 1/2-Y,1/2+Z,-X
 1/2+Y,-Z,1/2-X
 -Y,1/2-Z,1/2+X
 3/4+Y,1/4+X,3/4-Z
 1/4-Y,1/4-X,1/4-Z
 1/4+Y,3/4-X,3/4+Z
 3/4-Y,3/4+X,1/4+Z
 3/4+X,1/4+Z,3/4-Y
 3/4-X,3/4+Z,1/4+Y
 1/4-X,1/4-Z,1/4-Y
 1/4+X,3/4-Z,3/4+Y
 3/4+Z,1/4+Y,3/4-X
 1/4+Z,3/4-Y,3/4+X
 3/4-Z,3/4+Y,1/4+X
 1/4-Z,1/4-Y,1/4-X
 X,1/2+Y,1/2+Z
 -X,-Y,Z
 1/2-X,Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,1/2+X,1/2+Y
 1/2+Z,1/2-X,-Y
 -Z,-X,Y
 1/2-Z,X,1/2-Y
 Y,1/2+Z,1/2+X
 1/2-Y,Z,1/2-X
 1/2+Y,1/2-Z,-X
 -Y,-Z,X
 3/4+Y,3/4+X,1/4-Z
 1/4-Y,3/4-X,3/4-Z
 1/4+Y,1/4-X,1/4+Z
 3/4-Y,1/4+X,3/4+Z
 3/4+X,3/4+Z,1/4-Y
 3/4-X,1/4+Z,3/4+Y
 1/4-X,3/4-Z,3/4-Y
 1/4+X,1/4-Z,1/4+Y
 3/4+Z,3/4+Y,1/4-X
 1/4+Z,1/4-Y,1/4+X
 3/4-Z,1/4+Y,3/4+X
 1/4-Z,3/4-Y,3/4-X
 1/2+X,Y,1/2+Z
 1/2-X,1/2-Y,Z
 -X,1/2+Y,1/2-Z
 X,-Y,-Z
 1/2+Z,X,1/2+Y
 Z,-X,-Y
 1/2-Z,1/2-X,Y
 -Z,1/2+X,1/2-Y
 1/2+Y,Z,1/2+X
 -Y,1/2+Z,1/2-X
 Y,-Z,-X
 1/2-Y,1/2-Z,X
 1/4+Y,1/4+X,1/4-Z
 3/4-Y,1/4-X,3/4-Z
 3/4+Y,3/4-X,1/4+Z
 1/4-Y,3/4+X,3/4+Z
 1/4+X,1/4+Z,1/4-Y
 1/4-X,3/4+Z,3/4+Y
 3/4-X,1/4-Z,3/4-Y
 3/4+X,3/4-Z,1/4+Y
 1/4+Z,1/4+Y,1/4-X
 3/4+Z,3/4-Y,1/4+X
 1/4-Z,3/4+Y,3/4+X
 3/4-Z,1/4-Y,3/4-X
 1/2+X,1/2+Y,Z
 1/2-X,-Y,1/2+Z
 -X,Y,-Z
 X,1/2-Y,1/2-Z
 1/2+Z,1/2+X,Y
 Z,1/2-X,1/2-Y
 1/2-Z,-X,1/2+Y
 -Z,X,-Y
 1/2+Y,1/2+Z,X
 -Y,Z,-X
 Y,1/2-Z,1/2-X
 1/2-Y,-Z,1/2+X
 1/4+Y,3/4+X,3/4-Z
 3/4-Y,3/4-X,1/4-Z
 3/4+Y,1/4-X,3/4+Z
 1/4-Y,1/4+X,1/4+Z
 1/4+X,3/4+Z,3/4-Y
 1/4-X,1/4+Z,1/4+Y
 3/4-X,3/4-Z,1/4-Y
 3/4+X,1/4-Z,3/4+Y
 1/4+Z,3/4+Y,3/4-X
 3/4+Z,1/4-Y,3/4+X
 1/4-Z,1/4+Y,1/4+X
 3/4-Z,3/4-Y,1/4-X
211 48 24 I432 PG432 CUBIC 'I 4 3 2'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 Y,X,-Z
 -Y,-X,-Z
 Y,-X,Z
 -Y,X,Z
 X,Z,-Y
 -X,Z,Y
 -X,-Z,-Y
 X,-Z,Y
 Z,Y,-X
 Z,-Y,X
 -Z,Y,X
 -Z,-Y,-X
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2+Z,1/2+X,1/2+Y
 1/2+Z,1/2-X,1/2-Y
 1/2-Z,1/2-X,1/2+Y
 1/2-Z,1/2+X,1/2-Y
 1/2+Y,1/2+Z,1/2+X
 1/2-Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,1/2-X
 1/2-Y,1/2-Z,1/2+X
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
 1/2+Y,1/2-X,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+X,1/2+Z,1/2-Y
 1/2-X,1/2+Z,1/2+Y
 1/2-X,1/2-Z,1/2-Y
 1/2+X,1/2-Z,1/2+Y
 1/2+Z,1/2+Y,1/2-X
 1/2+Z,1/2-Y,1/2+X
 1/2-Z,1/2+Y,1/2+X
 1/2-Z,1/2-Y,1/2-X
212 24 24 P4332 PG432 CUBIC 'P 43 3 2'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,Z,X
 -Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,-X
 1/2-Y,-Z,1/2+X
 1/4+Y,3/4+X,3/4-Z
 1/4-Y,1/4-X,1/4-Z
 3/4+Y,3/4-X,1/4+Z
 3/4-Y,1/4+X,3/4+Z
 1/4+X,3/4+Z,3/4-Y
 3/4-X,1/4+Z,3/4+Y
 1/4-X,1/4-Z,1/4-Y
 3/4+X,3/4-Z,1/4+Y
 1/4+Z,3/4+Y,3/4-X
 3/4+Z,3/4-Y,1/4+X
 3/4-Z,1/4+Y,3/4+X
 1/4-Z,1/4-Y,1/4-X
213 24 24 P4132 PG432 CUBIC 'P 41 3 2'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,Z,X
 -Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,-X
 1/2-Y,-Z,1/2+X
 3/4+Y,1/4+X,1/4-Z
 3/4-Y,3/4-X,3/4-Z
 1/4+Y,1/4-X,3/4+Z
 1/4-Y,3/4+X,1/4+Z
 3/4+X,1/4+Z,1/4-Y
 1/4-X,3/4+Z,1/4+Y
 3/4-X,3/4-Z,3/4-Y
 1/4+X,1/4-Z,3/4+Y
 3/4+Z,1/4+Y,1/4-X
 1/4+Z,1/4-Y,3/4+X
 1/4-Z,3/4+Y,1/4+X
 3/4-Z,3/4-Y,3/4-X
214 48 24 I4132 PG432 CUBIC 'I 41 3 2'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,Z,X
 -Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,-X
 1/2-Y,-Z,1/2+X
 3/4+Y,1/4+X,1/4-Z
 3/4-Y,3/4-X,3/4-Z
 1/4+Y,1/4-X,3/4+Z
 1/4-Y,3/4+X,1/4+Z
 3/4+X,1/4+Z,1/4-Y
 1/4-X,3/4+Z,1/4+Y
 3/4-X,3/4-Z,3/4-Y
 1/4+X,1/4-Z,3/4+Y
 3/4+Z,1/4+Y,1/4-X
 1/4+Z,1/4-Y,3/4+X
 1/4-Z,3/4+Y,1/4+X
 3/4-Z,3/4-Y,3/4-X
 1/2+X,1/2+Y,1/2+Z
 -X,1/2-Y,Z
 1/2-X,Y,-Z
 X,-Y,1/2-Z
 1/2+Z,1/2+X,1/2+Y
 Z,-X,1/2-Y
 -Z,1/2-X,Y
 1/2-Z,X,-Y
 1/2+Y,1/2+Z,1/2+X
 1/2-Y,Z,-X
 Y,-Z,1/2-X
 -Y,1/2-Z,X
 1/4+Y,3/4+X,3/4-Z
 1/4-Y,1/4-X,1/4-Z
 3/4+Y,3/4-X,1/4+Z
 3/4-Y,1/4+X,3/4+Z
 1/4+X,3/4+Z,3/4-Y
 3/4-X,1/4+Z,3/4+Y
 1/4-X,1/4-Z,1/4-Y
 3/4+X,3/4-Z,1/4+Y
 1/4+Z,3/4+Y,3/4-X
 3/4+Z,3/4-Y,1/4+X
 3/4-Z,1/4+Y,3/4+X
 1/4-Z,1/4-Y,1/4-X
215 24 24 P-43m PG4bar3m CUBIC 'P -4 3 m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 Y,X,Z
 -Y,-X,Z
 Y,-X,-Z
 -Y,X,-Z
 X,Z,Y
 -X,Z,-Y
 -X,-Z,Y
 X,-Z,-Y
 Z,Y,X
 Z,-Y,-X
 -Z,Y,-X
 -Z,-Y,X
216 96 24 F-43m PG4bar3m CUBIC 'F -4 3 m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 Y,X,Z
 -Y,-X,Z
 Y,-X,-Z
 -Y,X,-Z
 X,Z,Y
 -X,Z,-Y
 -X,-Z,Y
 X,-Z,-Y
 Z,Y,X
 Z,-Y,-X
 -Z,Y,-X
 -Z,-Y,X
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 Z,1/2+X,1/2+Y
 Z,1/2-X,1/2-Y
 -Z,1/2-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,1/2+Z,1/2+X
 -Y,1/2+Z,1/2-X
 Y,1/2-Z,1/2-X
 -Y,1/2-Z,1/2+X
 Y,1/2+X,1/2+Z
 -Y,1/2-X,1/2+Z
 Y,1/2-X,1/2-Z
 -Y,1/2+X,1/2-Z
 X,1/2+Z,1/2+Y
 -X,1/2+Z,1/2-Y
 -X,1/2-Z,1/2+Y
 X,1/2-Z,1/2-Y
 Z,1/2+Y,1/2+X
 Z,1/2-Y,1/2-X
 -Z,1/2+Y,1/2-X
 -Z,1/2-Y,1/2+X
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 1/2+Z,X,1/2+Y
 1/2+Z,-X,1/2-Y
 1/2-Z,-X,1/2+Y
 1/2-Z,X,1/2-Y
 1/2+Y,Z,1/2+X
 1/2-Y,Z,1/2-X
 1/2+Y,-Z,1/2-X
 1/2-Y,-Z,1/2+X
 1/2+Y,X,1/2+Z
 1/2-Y,-X,1/2+Z
 1/2+Y,-X,1/2-Z
 1/2-Y,X,1/2-Z
 1/2+X,Z,1/2+Y
 1/2-X,Z,1/2-Y
 1/2-X,-Z,1/2+Y
 1/2+X,-Z,1/2-Y
 1/2+Z,Y,1/2+X
 1/2+Z,-Y,1/2-X
 1/2-Z,Y,1/2-X
 1/2-Z,-Y,1/2+X
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2+Z,1/2+X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,1/2-X,Y
 1/2-Z,1/2+X,-Y
 1/2+Y,1/2+Z,X
 1/2-Y,1/2+Z,-X
 1/2+Y,1/2-Z,-X
 1/2-Y,1/2-Z,X
 1/2+Y,1/2+X,Z
 1/2-Y,1/2-X,Z
 1/2+Y,1/2-X,-Z
 1/2-Y,1/2+X,-Z
 1/2+X,1/2+Z,Y
 1/2-X,1/2+Z,-Y
 1/2-X,1/2-Z,Y
 1/2+X,1/2-Z,-Y
 1/2+Z,1/2+Y,X
 1/2+Z,1/2-Y,-X
 1/2-Z,1/2+Y,-X
 1/2-Z,1/2-Y,X
217 48 24 I-43m PG4bar3m CUBIC 'I -4 3 m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 Y,X,Z
 -Y,-X,Z
 Y,-X,-Z
 -Y,X,-Z
 X,Z,Y
 -X,Z,-Y
 -X,-Z,Y
 X,-Z,-Y
 Z,Y,X
 Z,-Y,-X
 -Z,Y,-X
 -Z,-Y,X
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2+Z,1/2+X,1/2+Y
 1/2+Z,1/2-X,1/2-Y
 1/2-Z,1/2-X,1/2+Y
 1/2-Z,1/2+X,1/2-Y
 1/2+Y,1/2+Z,1/2+X
 1/2-Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,1/2-X
 1/2-Y,1/2-Z,1/2+X
 1/2+Y,1/2+X,1/2+Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2-X,1/2-Z
 1/2-Y,1/2+X,1/2-Z
 1/2+X,1/2+Z,1/2+Y
 1/2-X,1/2+Z,1/2-Y
 1/2-X,1/2-Z,1/2+Y
 1/2+X,1/2-Z,1/2-Y
 1/2+Z,1/2+Y,1/2+X
 1/2+Z,1/2-Y,1/2-X
 1/2-Z,1/2+Y,1/2-X
 1/2-Z,1/2-Y,1/2+X
218 24 24 P-43n PG4bar3m CUBIC 'P -4 3 n'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 1/2+Y,1/2+X,1/2+Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2-X,1/2-Z
 1/2-Y,1/2+X,1/2-Z
 1/2+X,1/2+Z,1/2+Y
 1/2-X,1/2+Z,1/2-Y
 1/2-X,1/2-Z,1/2+Y
 1/2+X,1/2-Z,1/2-Y
 1/2+Z,1/2+Y,1/2+X
 1/2+Z,1/2-Y,1/2-X
 1/2-Z,1/2+Y,1/2-X
 1/2-Z,1/2-Y,1/2+X
219 96 24 F-43c PG4bar3m CUBIC 'F -4 3 c'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 1/2+Y,1/2+X,1/2+Z
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2-X,1/2-Z
 1/2-Y,1/2+X,1/2-Z
 1/2+X,1/2+Z,1/2+Y
 1/2-X,1/2+Z,1/2-Y
 1/2-X,1/2-Z,1/2+Y
 1/2+X,1/2-Z,1/2-Y
 1/2+Z,1/2+Y,1/2+X
 1/2+Z,1/2-Y,1/2-X
 1/2-Z,1/2+Y,1/2-X
 1/2-Z,1/2-Y,1/2+X
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 Z,1/2+X,1/2+Y
 Z,1/2-X,1/2-Y
 -Z,1/2-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,1/2+Z,1/2+X
 -Y,1/2+Z,1/2-X
 Y,1/2-Z,1/2-X
 -Y,1/2-Z,1/2+X
 1/2+Y,X,Z
 1/2-Y,-X,Z
 1/2+Y,-X,-Z
 1/2-Y,X,-Z
 1/2+X,Z,Y
 1/2-X,Z,-Y
 1/2-X,-Z,Y
 1/2+X,-Z,-Y
 1/2+Z,Y,X
 1/2+Z,-Y,-X
 1/2-Z,Y,-X
 1/2-Z,-Y,X
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 1/2+Z,X,1/2+Y
 1/2+Z,-X,1/2-Y
 1/2-Z,-X,1/2+Y
 1/2-Z,X,1/2-Y
 1/2+Y,Z,1/2+X
 1/2-Y,Z,1/2-X
 1/2+Y,-Z,1/2-X
 1/2-Y,-Z,1/2+X
 Y,1/2+X,Z
 -Y,1/2-X,Z
 Y,1/2-X,-Z
 -Y,1/2+X,-Z
 X,1/2+Z,Y
 -X,1/2+Z,-Y
 -X,1/2-Z,Y
 X,1/2-Z,-Y
 Z,1/2+Y,X
 Z,1/2-Y,-X
 -Z,1/2+Y,-X
 -Z,1/2-Y,X
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2+Z,1/2+X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,1/2-X,Y
 1/2-Z,1/2+X,-Y
 1/2+Y,1/2+Z,X
 1/2-Y,1/2+Z,-X
 1/2+Y,1/2-Z,-X
 1/2-Y,1/2-Z,X
 Y,X,1/2+Z
 -Y,-X,1/2+Z
 Y,-X,1/2-Z
 -Y,X,1/2-Z
 X,Z,1/2+Y
 -X,Z,1/2-Y
 -X,-Z,1/2+Y
 X,-Z,1/2-Y
 Z,Y,1/2+X
 Z,-Y,1/2-X
 -Z,Y,1/2-X
 -Z,-Y,1/2+X
220 48 24 I-43d PG4bar3m CUBIC 'I -4 3 d'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,Z,X
 -Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,-X
 1/2-Y,-Z,1/2+X
 1/4+Y,1/4+X,1/4+Z
 1/4-Y,3/4-X,3/4+Z
 3/4+Y,1/4-X,3/4-Z
 3/4-Y,3/4+X,1/4-Z
 1/4+X,1/4+Z,1/4+Y
 3/4-X,3/4+Z,1/4-Y
 1/4-X,3/4-Z,3/4+Y
 3/4+X,1/4-Z,3/4-Y
 1/4+Z,1/4+Y,1/4+X
 3/4+Z,1/4-Y,3/4-X
 3/4-Z,3/4+Y,1/4-X
 1/4-Z,3/4-Y,3/4+X
 1/2+X,1/2+Y,1/2+Z
 -X,1/2-Y,Z
 1/2-X,Y,-Z
 X,-Y,1/2-Z
 1/2+Z,1/2+X,1/2+Y
 Z,-X,1/2-Y
 -Z,1/2-X,Y
 1/2-Z,X,-Y
 1/2+Y,1/2+Z,1/2+X
 1/2-Y,Z,-X
 Y,-Z,1/2-X
 -Y,1/2-Z,X
 3/4+Y,3/4+X,3/4+Z
 3/4-Y,1/4-X,1/4+Z
 1/4+Y,3/4-X,1/4-Z
 1/4-Y,1/4+X,3/4-Z
 3/4+X,3/4+Z,3/4+Y
 1/4-X,1/4+Z,3/4-Y
 3/4-X,1/4-Z,1/4+Y
 1/4+X,3/4-Z,1/4-Y
 3/4+Z,3/4+Y,3/4+X
 1/4+Z,3/4-Y,1/4-X
 1/4-Z,1/4+Y,3/4-X
 3/4-Z,1/4-Y,1/4+X
221 48 48 Pm-3m PGm3barm CUBIC 'P 4/m -3 2/m' 'P m -3 m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 Y,X,-Z
 -Y,-X,-Z
 Y,-X,Z
 -Y,X,Z
 X,Z,-Y
 -X,Z,Y
 -X,-Z,-Y
 X,-Z,Y
 Z,Y,-X
 Z,-Y,X
 -Z,Y,X
 -Z,-Y,-X
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
 -Z,-X,-Y
 -Z,X,Y
 Z,X,-Y
 Z,-X,Y
 -Y,-Z,-X
 Y,-Z,X
 -Y,Z,X
 Y,Z,-X
 -Y,-X,Z
 Y,X,Z
 -Y,X,-Z
 Y,-X,-Z
 -X,-Z,Y
 X,-Z,-Y
 X,Z,Y
 -X,Z,-Y
 -Z,-Y,X
 -Z,Y,-X
 Z,-Y,-X
 Z,Y,X
222 48 48 Pn-3n PGm3barm CUBIC 'P 4/n -3 2/n' P n -3 n'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 Y,X,-Z
 -Y,-X,-Z
 Y,-X,Z
 -Y,X,Z
 X,Z,-Y
 -X,Z,Y
 -X,-Z,-Y
 X,-Z,Y
 Z,Y,-X
 Z,-Y,X
 -Z,Y,X
 -Z,-Y,-X
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2-Z,1/2-X,1/2-Y
 1/2-Z,1/2+X,1/2+Y
 1/2+Z,1/2+X,1/2-Y
 1/2+Z,1/2-X,1/2+Y
 1/2-Y,1/2-Z,1/2-X
 1/2+Y,1/2-Z,1/2+X
 1/2-Y,1/2+Z,1/2+X
 1/2+Y,1/2+Z,1/2-X
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
 1/2-Y,1/2+X,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 1/2-X,1/2-Z,1/2+Y
 1/2+X,1/2-Z,1/2-Y
 1/2+X,1/2+Z,1/2+Y
 1/2-X,1/2+Z,1/2-Y
 1/2-Z,1/2-Y,1/2+X
 1/2-Z,1/2+Y,1/2-X
 1/2+Z,1/2-Y,1/2-X
 1/2+Z,1/2+Y,1/2+X
223 48 48 Pm-3n PGm3barm CUBIC 'P 42/m -3 2/n' 'P m -3 n'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
 1/2+Y,1/2-X,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+X,1/2+Z,1/2-Y
 1/2-X,1/2+Z,1/2+Y
 1/2-X,1/2-Z,1/2-Y
 1/2+X,1/2-Z,1/2+Y
 1/2+Z,1/2+Y,1/2-X
 1/2+Z,1/2-Y,1/2+X
 1/2-Z,1/2+Y,1/2+X
 1/2-Z,1/2-Y,1/2-X
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
 -Z,-X,-Y
 -Z,X,Y
 Z,X,-Y
 Z,-X,Y
 -Y,-Z,-X
 Y,-Z,X
 -Y,Z,X
 Y,Z,-X
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
 1/2-Y,1/2+X,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 1/2-X,1/2-Z,1/2+Y
 1/2+X,1/2-Z,1/2-Y
 1/2+X,1/2+Z,1/2+Y
 1/2-X,1/2+Z,1/2-Y
 1/2-Z,1/2-Y,1/2+X
 1/2-Z,1/2+Y,1/2-X
 1/2+Z,1/2-Y,1/2-X
 1/2+Z,1/2+Y,1/2+X
224 48 48 Pn-3m PGm3barm CUBIC 'P 42/n -3 2/m' 'P n -3 m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
 1/2+Y,1/2-X,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+X,1/2+Z,1/2-Y
 1/2-X,1/2+Z,1/2+Y
 1/2-X,1/2-Z,1/2-Y
 1/2+X,1/2-Z,1/2+Y
 1/2+Z,1/2+Y,1/2-X
 1/2+Z,1/2-Y,1/2+X
 1/2-Z,1/2+Y,1/2+X
 1/2-Z,1/2-Y,1/2-X
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2-Z,1/2-X,1/2-Y
 1/2-Z,1/2+X,1/2+Y
 1/2+Z,1/2+X,1/2-Y
 1/2+Z,1/2-X,1/2+Y
 1/2-Y,1/2-Z,1/2-X
 1/2+Y,1/2-Z,1/2+X
 1/2-Y,1/2+Z,1/2+X
 1/2+Y,1/2+Z,1/2-X
 -Y,-X,Z
 Y,X,Z
 -Y,X,-Z
 Y,-X,-Z
 -X,-Z,Y
 X,-Z,-Y
 X,Z,Y
 -X,Z,-Y
 -Z,-Y,X
 -Z,Y,-X
 Z,-Y,-X
 Z,Y,X
225 192 48 Fm-3m PGm3barm CUBIC 'F 4/m -3 2/m' 'F m -3 m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 Y,X,-Z
 -Y,-X,-Z
 Y,-X,Z
 -Y,X,Z
 X,Z,-Y
 -X,Z,Y
 -X,-Z,-Y
 X,-Z,Y
 Z,Y,-X
 Z,-Y,X
 -Z,Y,X
 -Z,-Y,-X
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
 -Z,-X,-Y
 -Z,X,Y
 Z,X,-Y
 Z,-X,Y
 -Y,-Z,-X
 Y,-Z,X
 -Y,Z,X
 Y,Z,-X
 -Y,-X,Z
 Y,X,Z
 -Y,X,-Z
 Y,-X,-Z
 -X,-Z,Y
 X,-Z,-Y
 X,Z,Y
 -X,Z,-Y
 -Z,-Y,X
 -Z,Y,-X
 Z,-Y,-X
 Z,Y,X
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 Z,1/2+X,1/2+Y
 Z,1/2-X,1/2-Y
 -Z,1/2-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,1/2+Z,1/2+X
 -Y,1/2+Z,1/2-X
 Y,1/2-Z,1/2-X
 -Y,1/2-Z,1/2+X
 Y,1/2+X,1/2-Z
 -Y,1/2-X,1/2-Z
 Y,1/2-X,1/2+Z
 -Y,1/2+X,1/2+Z
 X,1/2+Z,1/2-Y
 -X,1/2+Z,1/2+Y
 -X,1/2-Z,1/2-Y
 X,1/2-Z,1/2+Y
 Z,1/2+Y,1/2-X
 Z,1/2-Y,1/2+X
 -Z,1/2+Y,1/2+X
 -Z,1/2-Y,1/2-X
 -X,1/2-Y,1/2-Z
 X,1/2+Y,1/2-Z
 X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2+Z
 -Z,1/2-X,1/2-Y
 -Z,1/2+X,1/2+Y
 Z,1/2+X,1/2-Y
 Z,1/2-X,1/2+Y
 -Y,1/2-Z,1/2-X
 Y,1/2-Z,1/2+X
 -Y,1/2+Z,1/2+X
 Y,1/2+Z,1/2-X
 -Y,1/2-X,1/2+Z
 Y,1/2+X,1/2+Z
 -Y,1/2+X,1/2-Z
 Y,1/2-X,1/2-Z
 -X,1/2-Z,1/2+Y
 X,1/2-Z,1/2-Y
 X,1/2+Z,1/2+Y
 -X,1/2+Z,1/2-Y
 -Z,1/2-Y,1/2+X
 -Z,1/2+Y,1/2-X
 Z,1/2-Y,1/2-X
 Z,1/2+Y,1/2+X
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 1/2+Z,X,1/2+Y
 1/2+Z,-X,1/2-Y
 1/2-Z,-X,1/2+Y
 1/2-Z,X,1/2-Y
 1/2+Y,Z,1/2+X
 1/2-Y,Z,1/2-X
 1/2+Y,-Z,1/2-X
 1/2-Y,-Z,1/2+X
 1/2+Y,X,1/2-Z
 1/2-Y,-X,1/2-Z
 1/2+Y,-X,1/2+Z
 1/2-Y,X,1/2+Z
 1/2+X,Z,1/2-Y
 1/2-X,Z,1/2+Y
 1/2-X,-Z,1/2-Y
 1/2+X,-Z,1/2+Y
 1/2+Z,Y,1/2-X
 1/2+Z,-Y,1/2+X
 1/2-Z,Y,1/2+X
 1/2-Z,-Y,1/2-X
 1/2-X,-Y,1/2-Z
 1/2+X,Y,1/2-Z
 1/2+X,-Y,1/2+Z
 1/2-X,Y,1/2+Z
 1/2-Z,-X,1/2-Y
 1/2-Z,X,1/2+Y
 1/2+Z,X,1/2-Y
 1/2+Z,-X,1/2+Y
 1/2-Y,-Z,1/2-X
 1/2+Y,-Z,1/2+X
 1/2-Y,Z,1/2+X
 1/2+Y,Z,1/2-X
 1/2-Y,-X,1/2+Z
 1/2+Y,X,1/2+Z
 1/2-Y,X,1/2-Z
 1/2+Y,-X,1/2-Z
 1/2-X,-Z,1/2+Y
 1/2+X,-Z,1/2-Y
 1/2+X,Z,1/2+Y
 1/2-X,Z,1/2-Y
 1/2-Z,-Y,1/2+X
 1/2-Z,Y,1/2-X
 1/2+Z,-Y,1/2-X
 1/2+Z,Y,1/2+X
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2+Z,1/2+X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,1/2-X,Y
 1/2-Z,1/2+X,-Y
 1/2+Y,1/2+Z,X
 1/2-Y,1/2+Z,-X
 1/2+Y,1/2-Z,-X
 1/2-Y,1/2-Z,X
 1/2+Y,1/2+X,-Z
 1/2-Y,1/2-X,-Z
 1/2+Y,1/2-X,Z
 1/2-Y,1/2+X,Z
 1/2+X,1/2+Z,-Y
 1/2-X,1/2+Z,Y
 1/2-X,1/2-Z,-Y
 1/2+X,1/2-Z,Y
 1/2+Z,1/2+Y,-X
 1/2+Z,1/2-Y,X
 1/2-Z,1/2+Y,X
 1/2-Z,1/2-Y,-X
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2-Z,1/2-X,-Y
 1/2-Z,1/2+X,Y
 1/2+Z,1/2+X,-Y
 1/2+Z,1/2-X,Y
 1/2-Y,1/2-Z,-X
 1/2+Y,1/2-Z,X
 1/2-Y,1/2+Z,X
 1/2+Y,1/2+Z,-X
 1/2-Y,1/2-X,Z
 1/2+Y,1/2+X,Z
 1/2-Y,1/2+X,-Z
 1/2+Y,1/2-X,-Z
 1/2-X,1/2-Z,Y
 1/2+X,1/2-Z,-Y
 1/2+X,1/2+Z,Y
 1/2-X,1/2+Z,-Y
 1/2-Z,1/2-Y,X
 1/2-Z,1/2+Y,-X
 1/2+Z,1/2-Y,-X
 1/2+Z,1/2+Y,X
226 192 48 Fm-3c PGm3barm CUBIC 'F 4/m -3 2/c' 'F m -3 c'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
 1/2+Y,1/2-X,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+X,1/2+Z,1/2-Y
 1/2-X,1/2+Z,1/2+Y
 1/2-X,1/2-Z,1/2-Y
 1/2+X,1/2-Z,1/2+Y
 1/2+Z,1/2+Y,1/2-X
 1/2+Z,1/2-Y,1/2+X
 1/2-Z,1/2+Y,1/2+X
 1/2-Z,1/2-Y,1/2-X
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
 -Z,-X,-Y
 -Z,X,Y
 Z,X,-Y
 Z,-X,Y
 -Y,-Z,-X
 Y,-Z,X
 -Y,Z,X
 Y,Z,-X
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
 1/2-Y,1/2+X,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 1/2-X,1/2-Z,1/2+Y
 1/2+X,1/2-Z,1/2-Y
 1/2+X,1/2+Z,1/2+Y
 1/2-X,1/2+Z,1/2-Y
 1/2-Z,1/2-Y,1/2+X
 1/2-Z,1/2+Y,1/2-X
 1/2+Z,1/2-Y,1/2-X
 1/2+Z,1/2+Y,1/2+X
 X,1/2+Y,1/2+Z
 -X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 X,1/2-Y,1/2-Z
 Z,1/2+X,1/2+Y
 Z,1/2-X,1/2-Y
 -Z,1/2-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,1/2+Z,1/2+X
 -Y,1/2+Z,1/2-X
 Y,1/2-Z,1/2-X
 -Y,1/2-Z,1/2+X
 1/2+Y,X,-Z
 1/2-Y,-X,-Z
 1/2+Y,-X,Z
 1/2-Y,X,Z
 1/2+X,Z,-Y
 1/2-X,Z,Y
 1/2-X,-Z,-Y
 1/2+X,-Z,Y
 1/2+Z,Y,-X
 1/2+Z,-Y,X
 1/2-Z,Y,X
 1/2-Z,-Y,-X
 -X,1/2-Y,1/2-Z
 X,1/2+Y,1/2-Z
 X,1/2-Y,1/2+Z
 -X,1/2+Y,1/2+Z
 -Z,1/2-X,1/2-Y
 -Z,1/2+X,1/2+Y
 Z,1/2+X,1/2-Y
 Z,1/2-X,1/2+Y
 -Y,1/2-Z,1/2-X
 Y,1/2-Z,1/2+X
 -Y,1/2+Z,1/2+X
 Y,1/2+Z,1/2-X
 1/2-Y,-X,Z
 1/2+Y,X,Z
 1/2-Y,X,-Z
 1/2+Y,-X,-Z
 1/2-X,-Z,Y
 1/2+X,-Z,-Y
 1/2+X,Z,Y
 1/2-X,Z,-Y
 1/2-Z,-Y,X
 1/2-Z,Y,-X
 1/2+Z,-Y,-X
 1/2+Z,Y,X
 1/2+X,Y,1/2+Z
 1/2-X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 1/2+X,-Y,1/2-Z
 1/2+Z,X,1/2+Y
 1/2+Z,-X,1/2-Y
 1/2-Z,-X,1/2+Y
 1/2-Z,X,1/2-Y
 1/2+Y,Z,1/2+X
 1/2-Y,Z,1/2-X
 1/2+Y,-Z,1/2-X
 1/2-Y,-Z,1/2+X
 Y,1/2+X,-Z
 -Y,1/2-X,-Z
 Y,1/2-X,Z
 -Y,1/2+X,Z
 X,1/2+Z,-Y
 -X,1/2+Z,Y
 -X,1/2-Z,-Y
 X,1/2-Z,Y
 Z,1/2+Y,-X
 Z,1/2-Y,X
 -Z,1/2+Y,X
 -Z,1/2-Y,-X
 1/2-X,-Y,1/2-Z
 1/2+X,Y,1/2-Z
 1/2+X,-Y,1/2+Z
 1/2-X,Y,1/2+Z
 1/2-Z,-X,1/2-Y
 1/2-Z,X,1/2+Y
 1/2+Z,X,1/2-Y
 1/2+Z,-X,1/2+Y
 1/2-Y,-Z,1/2-X
 1/2+Y,-Z,1/2+X
 1/2-Y,Z,1/2+X
 1/2+Y,Z,1/2-X
 -Y,1/2-X,Z
 Y,1/2+X,Z
 -Y,1/2+X,-Z
 Y,1/2-X,-Z
 -X,1/2-Z,Y
 X,1/2-Z,-Y
 X,1/2+Z,Y
 -X,1/2+Z,-Y
 -Z,1/2-Y,X
 -Z,1/2+Y,-X
 Z,1/2-Y,-X
 Z,1/2+Y,X
 1/2+X,1/2+Y,Z
 1/2-X,1/2-Y,Z
 1/2-X,1/2+Y,-Z
 1/2+X,1/2-Y,-Z
 1/2+Z,1/2+X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,1/2-X,Y
 1/2-Z,1/2+X,-Y
 1/2+Y,1/2+Z,X
 1/2-Y,1/2+Z,-X
 1/2+Y,1/2-Z,-X
 1/2-Y,1/2-Z,X
 Y,X,1/2-Z
 -Y,-X,1/2-Z
 Y,-X,1/2+Z
 -Y,X,1/2+Z
 X,Z,1/2-Y
 -X,Z,1/2+Y
 -X,-Z,1/2-Y
 X,-Z,1/2+Y
 Z,Y,1/2-X
 Z,-Y,1/2+X
 -Z,Y,1/2+X
 -Z,-Y,1/2-X
 1/2-X,1/2-Y,-Z
 1/2+X,1/2+Y,-Z
 1/2+X,1/2-Y,Z
 1/2-X,1/2+Y,Z
 1/2-Z,1/2-X,-Y
 1/2-Z,1/2+X,Y
 1/2+Z,1/2+X,-Y
 1/2+Z,1/2-X,Y
 1/2-Y,1/2-Z,-X
 1/2+Y,1/2-Z,X
 1/2-Y,1/2+Z,X
 1/2+Y,1/2+Z,-X
 -Y,-X,1/2+Z
 Y,X,1/2+Z
 -Y,X,1/2-Z
 Y,-X,1/2-Z
 -X,-Z,1/2+Y
 X,-Z,1/2-Y
 X,Z,1/2+Y
 -X,Z,1/2-Y
 -Z,-Y,1/2+X
 -Z,Y,1/2-X
 Z,-Y,1/2-X
 Z,Y,1/2+X
227 192 48 Fd-3m PGm3barm CUBIC 'F 41/d -3 2/m' 'F d -3 m'
 X,Y,Z
 -X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,-Z
 1/2+X,-Y,1/2-Z
 Z,X,Y
 1/2+Z,-X,1/2-Y
 -Z,1/2-X,1/2+Y
 1/2-Z,1/2+X,-Y
 Y,Z,X
 1/2-Y,1/2+Z,-X
 1/2+Y,-Z,1/2-X
 -Y,1/2-Z,1/2+X
 3/4+Y,1/4+X,3/4-Z
 1/4-Y,1/4-X,1/4-Z
 1/4+Y,3/4-X,3/4+Z
 3/4-Y,3/4+X,1/4+Z
 3/4+X,1/4+Z,3/4-Y
 3/4-X,3/4+Z,1/4+Y
 1/4-X,1/4-Z,1/4-Y
 1/4+X,3/4-Z,3/4+Y
 3/4+Z,1/4+Y,3/4-X
 1/4+Z,3/4-Y,3/4+X
 3/4-Z,3/4+Y,1/4+X
 1/4-Z,1/4-Y,1/4-X
 1/4-X,1/4-Y,1/4-Z
 1/4+X,3/4+Y,3/4-Z
 3/4+X,3/4-Y,1/4+Z
 3/4-X,1/4+Y,3/4+Z
 1/4-Z,1/4-X,1/4-Y
 3/4-Z,1/4+X,3/4+Y
 1/4+Z,3/4+X,3/4-Y
 3/4+Z,3/4-X,1/4+Y
 1/4-Y,1/4-Z,1/4-X
 3/4+Y,3/4-Z,1/4+X
 3/4-Y,1/4+Z,3/4+X
 1/4+Y,3/4+Z,3/4-X
 1/2-Y,-X,1/2+Z
 Y,X,Z
 -Y,1/2+X,1/2-Z
 1/2+Y,1/2-X,-Z
 1/2-X,-Z,1/2+Y
 1/2+X,1/2-Z,-Y
 X,Z,Y
 -X,1/2+Z,1/2-Y
 1/2-Z,-Y,1/2+X
 -Z,1/2+Y,1/2-X
 1/2+Z,1/2-Y,-X
 Z,Y,X
 X,1/2+Y,1/2+Z
 -X,-Y,Z
 1/2-X,Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,1/2+X,1/2+Y
 1/2+Z,1/2-X,-Y
 -Z,-X,Y
 1/2-Z,X,1/2-Y
 Y,1/2+Z,1/2+X
 1/2-Y,Z,1/2-X
 1/2+Y,1/2-Z,-X
 -Y,-Z,X
 3/4+Y,3/4+X,1/4-Z
 1/4-Y,3/4-X,3/4-Z
 1/4+Y,1/4-X,1/4+Z
 3/4-Y,1/4+X,3/4+Z
 3/4+X,3/4+Z,1/4-Y
 3/4-X,1/4+Z,3/4+Y
 1/4-X,3/4-Z,3/4-Y
 1/4+X,1/4-Z,1/4+Y
 3/4+Z,3/4+Y,1/4-X
 1/4+Z,1/4-Y,1/4+X
 3/4-Z,1/4+Y,3/4+X
 1/4-Z,3/4-Y,3/4-X
 1/4-X,3/4-Y,3/4-Z
 1/4+X,1/4+Y,1/4-Z
 3/4+X,1/4-Y,3/4+Z
 3/4-X,3/4+Y,1/4+Z
 1/4-Z,3/4-X,3/4-Y
 3/4-Z,3/4+X,1/4+Y
 1/4+Z,1/4+X,1/4-Y
 3/4+Z,1/4-X,3/4+Y
 1/4-Y,3/4-Z,3/4-X
 3/4+Y,1/4-Z,3/4+X
 3/4-Y,3/4+Z,1/4+X
 1/4+Y,1/4+Z,1/4-X
 1/2-Y,1/2-X,Z
 Y,1/2+X,1/2+Z
 -Y,X,-Z
 1/2+Y,-X,1/2-Z
 1/2-X,1/2-Z,Y
 1/2+X,-Z,1/2-Y
 X,1/2+Z,1/2+Y
 -X,Z,-Y
 1/2-Z,1/2-Y,X
 -Z,Y,-X
 1/2+Z,-Y,1/2-X
 Z,1/2+Y,1/2+X
 1/2+X,Y,1/2+Z
 1/2-X,1/2-Y,Z
 -X,1/2+Y,1/2-Z
 X,-Y,-Z
 1/2+Z,X,1/2+Y
 Z,-X,-Y
 1/2-Z,1/2-X,Y
 -Z,1/2+X,1/2-Y
 1/2+Y,Z,1/2+X
 -Y,1/2+Z,1/2-X
 Y,-Z,-X
 1/2-Y,1/2-Z,X
 1/4+Y,1/4+X,1/4-Z
 3/4-Y,1/4-X,3/4-Z
 3/4+Y,3/4-X,1/4+Z
 1/4-Y,3/4+X,3/4+Z
 1/4+X,1/4+Z,1/4-Y
 1/4-X,3/4+Z,3/4+Y
 3/4-X,1/4-Z,3/4-Y
 3/4+X,3/4-Z,1/4+Y
 1/4+Z,1/4+Y,1/4-X
 3/4+Z,3/4-Y,1/4+X
 1/4-Z,3/4+Y,3/4+X
 3/4-Z,1/4-Y,3/4-X
 3/4-X,1/4-Y,3/4-Z
 3/4+X,3/4+Y,1/4-Z
 1/4+X,3/4-Y,3/4+Z
 1/4-X,1/4+Y,1/4+Z
 3/4-Z,1/4-X,3/4-Y
 1/4-Z,1/4+X,1/4+Y
 3/4+Z,3/4+X,1/4-Y
 1/4+Z,3/4-X,3/4+Y
 3/4-Y,1/4-Z,3/4-X
 1/4+Y,3/4-Z,3/4+X
 1/4-Y,1/4+Z,1/4+X
 3/4+Y,3/4+Z,1/4-X
 -Y,-X,Z
 1/2+Y,X,1/2+Z
 1/2-Y,1/2+X,-Z
 Y,1/2-X,1/2-Z
 -X,-Z,Y
 X,1/2-Z,1/2-Y
 1/2+X,Z,1/2+Y
 1/2-X,1/2+Z,-Y
 -Z,-Y,X
 1/2-Z,1/2+Y,-X
 Z,1/2-Y,1/2-X
 1/2+Z,Y,1/2+X
 1/2+X,1/2+Y,Z
 1/2-X,-Y,1/2+Z
 -X,Y,-Z
 X,1/2-Y,1/2-Z
 1/2+Z,1/2+X,Y
 Z,1/2-X,1/2-Y
 1/2-Z,-X,1/2+Y
 -Z,X,-Y
 1/2+Y,1/2+Z,X
 -Y,Z,-X
 Y,1/2-Z,1/2-X
 1/2-Y,-Z,1/2+X
 1/4+Y,3/4+X,3/4-Z
 3/4-Y,3/4-X,1/4-Z
 3/4+Y,1/4-X,3/4+Z
 1/4-Y,1/4+X,1/4+Z
 1/4+X,3/4+Z,3/4-Y
 1/4-X,1/4+Z,1/4+Y
 3/4-X,3/4-Z,1/4-Y
 3/4+X,1/4-Z,3/4+Y
 1/4+Z,3/4+Y,3/4-X
 3/4+Z,1/4-Y,3/4+X
 1/4-Z,1/4+Y,1/4+X
 3/4-Z,3/4-Y,1/4-X
 3/4-X,3/4-Y,1/4-Z
 3/4+X,1/4+Y,3/4-Z
 1/4+X,1/4-Y,1/4+Z
 1/4-X,3/4+Y,3/4+Z
 3/4-Z,3/4-X,1/4-Y
 1/4-Z,3/4+X,3/4+Y
 3/4+Z,1/4+X,3/4-Y
 1/4+Z,1/4-X,1/4+Y
 3/4-Y,3/4-Z,1/4-X
 1/4+Y,1/4-Z,1/4+X
 1/4-Y,3/4+Z,3/4+X
 3/4+Y,1/4+Z,3/4-X
 -Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,Z
 1/2-Y,X,1/2-Z
 Y,-X,-Z
 -X,1/2-Z,1/2+Y
 X,-Z,-Y
 1/2+X,1/2+Z,Y
 1/2-X,Z,1/2-Y
 -Z,1/2-Y,1/2+X
 1/2-Z,Y,1/2-X
 Z,-Y,-X
 1/2+Z,1/2+Y,X
228 192 48 Fd-3c PGm3barm CUBIC 'F 41/d -3 2/c' 'F d -3 c'
 X,Y,Z
 -X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,-Z
 1/2+X,-Y,1/2-Z
 Z,X,Y
 1/2+Z,-X,1/2-Y
 -Z,1/2-X,1/2+Y
 1/2-Z,1/2+X,-Y
 Y,Z,X
 1/2-Y,1/2+Z,-X
 1/2+Y,-Z,1/2-X
 -Y,1/2-Z,1/2+X
 3/4+Y,1/4+X,3/4-Z
 1/4-Y,1/4-X,1/4-Z
 1/4+Y,3/4-X,3/4+Z
 3/4-Y,3/4+X,1/4+Z
 3/4+X,1/4+Z,3/4-Y
 3/4-X,3/4+Z,1/4+Y
 1/4-X,1/4-Z,1/4-Y
 1/4+X,3/4-Z,3/4+Y
 3/4+Z,1/4+Y,3/4-X
 1/4+Z,3/4-Y,3/4+X
 3/4-Z,3/4+Y,1/4+X
 1/4-Z,1/4-Y,1/4-X
 3/4-X,3/4-Y,3/4-Z
 3/4+X,1/4+Y,1/4-Z
 1/4+X,1/4-Y,3/4+Z
 1/4-X,3/4+Y,1/4+Z
 3/4-Z,3/4-X,3/4-Y
 1/4-Z,3/4+X,1/4+Y
 3/4+Z,1/4+X,1/4-Y
 1/4+Z,1/4-X,3/4+Y
 3/4-Y,3/4-Z,3/4-X
 1/4+Y,1/4-Z,3/4+X
 1/4-Y,3/4+Z,1/4+X
 3/4+Y,1/4+Z,1/4-X
 -Y,1/2-X,Z
 1/2+Y,1/2+X,1/2+Z
 1/2-Y,X,-Z
 Y,-X,1/2-Z
 -X,1/2-Z,Y
 X,-Z,1/2-Y
 1/2+X,1/2+Z,1/2+Y
 1/2-X,Z,-Y
 -Z,1/2-Y,X
 1/2-Z,Y,-X
 Z,-Y,1/2-X
 1/2+Z,1/2+Y,1/2+X
 X,1/2+Y,1/2+Z
 -X,-Y,Z
 1/2-X,Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,1/2+X,1/2+Y
 1/2+Z,1/2-X,-Y
 -Z,-X,Y
 1/2-Z,X,1/2-Y
 Y,1/2+Z,1/2+X
 1/2-Y,Z,1/2-X
 1/2+Y,1/2-Z,-X
 -Y,-Z,X
 3/4+Y,3/4+X,1/4-Z
 1/4-Y,3/4-X,3/4-Z
 1/4+Y,1/4-X,1/4+Z
 3/4-Y,1/4+X,3/4+Z
 3/4+X,3/4+Z,1/4-Y
 3/4-X,1/4+Z,3/4+Y
 1/4-X,3/4-Z,3/4-Y
 1/4+X,1/4-Z,1/4+Y
 3/4+Z,3/4+Y,1/4-X
 1/4+Z,1/4-Y,1/4+X
 3/4-Z,1/4+Y,3/4+X
 1/4-Z,3/4-Y,3/4-X
 3/4-X,1/4-Y,1/4-Z
 3/4+X,3/4+Y,3/4-Z
 1/4+X,3/4-Y,1/4+Z
 1/4-X,1/4+Y,3/4+Z
 3/4-Z,1/4-X,1/4-Y
 1/4-Z,1/4+X,3/4+Y
 3/4+Z,3/4+X,3/4-Y
 1/4+Z,3/4-X,1/4+Y
 3/4-Y,1/4-Z,1/4-X
 1/4+Y,3/4-Z,1/4+X
 1/4-Y,1/4+Z,3/4+X
 3/4+Y,3/4+Z,3/4-X
 -Y,-X,1/2+Z
 1/2+Y,X,Z
 1/2-Y,1/2+X,1/2-Z
 Y,1/2-X,-Z
 -X,-Z,1/2+Y
 X,1/2-Z,-Y
 1/2+X,Z,Y
 1/2-X,1/2+Z,1/2-Y
 -Z,-Y,1/2+X
 1/2-Z,1/2+Y,1/2-X
 Z,1/2-Y,-X
 1/2+Z,Y,X
 1/2+X,Y,1/2+Z
 1/2-X,1/2-Y,Z
 -X,1/2+Y,1/2-Z
 X,-Y,-Z
 1/2+Z,X,1/2+Y
 Z,-X,-Y
 1/2-Z,1/2-X,Y
 -Z,1/2+X,1/2-Y
 1/2+Y,Z,1/2+X
 -Y,1/2+Z,1/2-X
 Y,-Z,-X
 1/2-Y,1/2-Z,X
 1/4+Y,1/4+X,1/4-Z
 3/4-Y,1/4-X,3/4-Z
 3/4+Y,3/4-X,1/4+Z
 1/4-Y,3/4+X,3/4+Z
 1/4+X,1/4+Z,1/4-Y
 1/4-X,3/4+Z,3/4+Y
 3/4-X,1/4-Z,3/4-Y
 3/4+X,3/4-Z,1/4+Y
 1/4+Z,1/4+Y,1/4-X
 3/4+Z,3/4-Y,1/4+X
 1/4-Z,3/4+Y,3/4+X
 3/4-Z,1/4-Y,3/4-X
 1/4-X,3/4-Y,1/4-Z
 1/4+X,1/4+Y,3/4-Z
 3/4+X,1/4-Y,1/4+Z
 3/4-X,3/4+Y,3/4+Z
 1/4-Z,3/4-X,1/4-Y
 3/4-Z,3/4+X,3/4+Y
 1/4+Z,1/4+X,3/4-Y
 3/4+Z,1/4-X,1/4+Y
 1/4-Y,3/4-Z,1/4-X
 3/4+Y,1/4-Z,1/4+X
 3/4-Y,3/4+Z,3/4+X
 1/4+Y,1/4+Z,3/4-X
 1/2-Y,1/2-X,1/2+Z
 Y,1/2+X,Z
 -Y,X,1/2-Z
 1/2+Y,-X,-Z
 1/2-X,1/2-Z,1/2+Y
 1/2+X,-Z,-Y
 X,1/2+Z,Y
 -X,Z,1/2-Y
 1/2-Z,1/2-Y,1/2+X
 -Z,Y,1/2-X
 1/2+Z,-Y,-X
 Z,1/2+Y,X
 1/2+X,1/2+Y,Z
 1/2-X,-Y,1/2+Z
 -X,Y,-Z
 X,1/2-Y,1/2-Z
 1/2+Z,1/2+X,Y
 Z,1/2-X,1/2-Y
 1/2-Z,-X,1/2+Y
 -Z,X,-Y
 1/2+Y,1/2+Z,X
 -Y,Z,-X
 Y,1/2-Z,1/2-X
 1/2-Y,-Z,1/2+X
 1/4+Y,3/4+X,3/4-Z
 3/4-Y,3/4-X,1/4-Z
 3/4+Y,1/4-X,3/4+Z
 1/4-Y,1/4+X,1/4+Z
 1/4+X,3/4+Z,3/4-Y
 1/4-X,1/4+Z,1/4+Y
 3/4-X,3/4-Z,1/4-Y
 3/4+X,1/4-Z,3/4+Y
 1/4+Z,3/4+Y,3/4-X
 3/4+Z,1/4-Y,3/4+X
 1/4-Z,1/4+Y,1/4+X
 3/4-Z,3/4-Y,1/4-X
 1/4-X,1/4-Y,3/4-Z
 1/4+X,3/4+Y,1/4-Z
 3/4+X,3/4-Y,3/4+Z
 3/4-X,1/4+Y,1/4+Z
 1/4-Z,1/4-X,3/4-Y
 3/4-Z,1/4+X,1/4+Y
 1/4+Z,3/4+X,1/4-Y
 3/4+Z,3/4-X,3/4+Y
 1/4-Y,1/4-Z,3/4-X
 3/4+Y,3/4-Z,3/4+X
 3/4-Y,1/4+Z,1/4+X
 1/4+Y,3/4+Z,1/4-X
 1/2-Y,-X,Z
 Y,X,1/2+Z
 -Y,1/2+X,-Z
 1/2+Y,1/2-X,1/2-Z
 1/2-X,-Z,Y
 1/2+X,1/2-Z,1/2-Y
 X,Z,1/2+Y
 -X,1/2+Z,-Y
 1/2-Z,-Y,X
 -Z,1/2+Y,-X
 1/2+Z,1/2-Y,1/2-X
 Z,Y,1/2+X
229 96 48 Im-3m PGm3barm CUBIC 'I 4/m -3 2/m' 'I m -3 m'
 X,Y,Z
 -X,-Y,Z
 -X,Y,-Z
 X,-Y,-Z
 Z,X,Y
 Z,-X,-Y
 -Z,-X,Y
 -Z,X,-Y
 Y,Z,X
 -Y,Z,-X
 Y,-Z,-X
 -Y,-Z,X
 Y,X,-Z
 -Y,-X,-Z
 Y,-X,Z
 -Y,X,Z
 X,Z,-Y
 -X,Z,Y
 -X,-Z,-Y
 X,-Z,Y
 Z,Y,-X
 Z,-Y,X
 -Z,Y,X
 -Z,-Y,-X
 -X,-Y,-Z
 X,Y,-Z
 X,-Y,Z
 -X,Y,Z
 -Z,-X,-Y
 -Z,X,Y
 Z,X,-Y
 Z,-X,Y
 -Y,-Z,-X
 Y,-Z,X
 -Y,Z,X
 Y,Z,-X
 -Y,-X,Z
 Y,X,Z
 -Y,X,-Z
 Y,-X,-Z
 -X,-Z,Y
 X,-Z,-Y
 X,Z,Y
 -X,Z,-Y
 -Z,-Y,X
 -Z,Y,-X
 Z,-Y,-X
 Z,Y,X
 1/2+X,1/2+Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2-Z
 1/2+Z,1/2+X,1/2+Y
 1/2+Z,1/2-X,1/2-Y
 1/2-Z,1/2-X,1/2+Y
 1/2-Z,1/2+X,1/2-Y
 1/2+Y,1/2+Z,1/2+X
 1/2-Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,1/2-X
 1/2-Y,1/2-Z,1/2+X
 1/2+Y,1/2+X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
 1/2+Y,1/2-X,1/2+Z
 1/2-Y,1/2+X,1/2+Z
 1/2+X,1/2+Z,1/2-Y
 1/2-X,1/2+Z,1/2+Y
 1/2-X,1/2-Z,1/2-Y
 1/2+X,1/2-Z,1/2+Y
 1/2+Z,1/2+Y,1/2-X
 1/2+Z,1/2-Y,1/2+X
 1/2-Z,1/2+Y,1/2+X
 1/2-Z,1/2-Y,1/2-X
 1/2-X,1/2-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,1/2+Z
 1/2-Z,1/2-X,1/2-Y
 1/2-Z,1/2+X,1/2+Y
 1/2+Z,1/2+X,1/2-Y
 1/2+Z,1/2-X,1/2+Y
 1/2-Y,1/2-Z,1/2-X
 1/2+Y,1/2-Z,1/2+X
 1/2-Y,1/2+Z,1/2+X
 1/2+Y,1/2+Z,1/2-X
 1/2-Y,1/2-X,1/2+Z
 1/2+Y,1/2+X,1/2+Z
 1/2-Y,1/2+X,1/2-Z
 1/2+Y,1/2-X,1/2-Z
 1/2-X,1/2-Z,1/2+Y
 1/2+X,1/2-Z,1/2-Y
 1/2+X,1/2+Z,1/2+Y
 1/2-X,1/2+Z,1/2-Y
 1/2-Z,1/2-Y,1/2+X
 1/2-Z,1/2+Y,1/2-X
 1/2+Z,1/2-Y,1/2-X
 1/2+Z,1/2+Y,1/2+X
230 96 48 Ia-3d PGm3barm CUBIC 'I 41/a -3 2/d' 'I a -3 d'
 X,Y,Z
 1/2-X,-Y,1/2+Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2-Y,-Z
 Z,X,Y
 1/2+Z,1/2-X,-Y
 1/2-Z,-X,1/2+Y
 -Z,1/2+X,1/2-Y
 Y,Z,X
 -Y,1/2+Z,1/2-X
 1/2+Y,1/2-Z,-X
 1/2-Y,-Z,1/2+X
 3/4+Y,1/4+X,1/4-Z
 3/4-Y,3/4-X,3/4-Z
 1/4+Y,1/4-X,3/4+Z
 1/4-Y,3/4+X,1/4+Z
 3/4+X,1/4+Z,1/4-Y
 1/4-X,3/4+Z,1/4+Y
 3/4-X,3/4-Z,3/4-Y
 1/4+X,1/4-Z,3/4+Y
 3/4+Z,1/4+Y,1/4-X
 1/4+Z,1/4-Y,3/4+X
 1/4-Z,3/4+Y,1/4+X
 3/4-Z,3/4-Y,3/4-X
 -X,-Y,-Z
 1/2+X,Y,1/2-Z
 X,1/2-Y,1/2+Z
 1/2-X,1/2+Y,Z
 -Z,-X,-Y
 1/2-Z,1/2+X,Y
 1/2+Z,X,1/2-Y
 Z,1/2-X,1/2+Y
 -Y,-Z,-X
 Y,1/2-Z,1/2+X
 1/2-Y,1/2+Z,X
 1/2+Y,Z,1/2-X
 1/4-Y,3/4-X,3/4+Z
 1/4+Y,1/4+X,1/4+Z
 3/4-Y,3/4+X,1/4-Z
 3/4+Y,1/4-X,3/4-Z
 1/4-X,3/4-Z,3/4+Y
 3/4+X,1/4-Z,3/4-Y
 1/4+X,1/4+Z,1/4+Y
 3/4-X,3/4+Z,1/4-Y
 1/4-Z,3/4-Y,3/4+X
 3/4-Z,3/4+Y,1/4-X
 3/4+Z,1/4-Y,3/4-X
 1/4+Z,1/4+Y,1/4+X
 1/2+X,1/2+Y,1/2+Z
 -X,1/2-Y,Z
 1/2-X,Y,-Z
 X,-Y,1/2-Z
 1/2+Z,1/2+X,1/2+Y
 Z,-X,1/2-Y
 -Z,1/2-X,Y
 1/2-Z,X,-Y
 1/2+Y,1/2+Z,1/2+X
 1/2-Y,Z,-X
 Y,-Z,1/2-X
 -Y,1/2-Z,X
 1/4+Y,3/4+X,3/4-Z
 1/4-Y,1/4-X,1/4-Z
 3/4+Y,3/4-X,1/4+Z
 3/4-Y,1/4+X,3/4+Z
 1/4+X,3/4+Z,3/4-Y
 3/4-X,1/4+Z,3/4+Y
 1/4-X,1/4-Z,1/4-Y
 3/4+X,3/4-Z,1/4+Y
 1/4+Z,3/4+Y,3/4-X
 3/4+Z,3/4-Y,1/4+X
 3/4-Z,1/4+Y,3/4+X
 1/4-Z,1/4-Y,1/4-X
 1/2-X,1/2-Y,1/2-Z
 X,1/2+Y,-Z
 1/2+X,-Y,Z
 -X,Y,1/2+Z
 1/2-Z,1/2-X,1/2-Y
 -Z,X,1/2+Y
 Z,1/2+X,-Y
 1/2+Z,-X,Y
 1/2-Y,1/2-Z,1/2-X
 1/2+Y,-Z,X
 -Y,Z,1/2+X
 Y,1/2+Z,-X
 3/4-Y,1/4-X,1/4+Z
 3/4+Y,3/4+X,3/4+Z
 1/4-Y,1/4+X,3/4-Z
 1/4+Y,3/4-X,1/4-Z
 3/4-X,1/4-Z,1/4+Y
 1/4+X,3/4-Z,1/4-Y
 3/4+X,3/4+Z,3/4+Y
 1/4-X,1/4+Z,3/4-Y
 3/4-Z,1/4-Y,1/4+X
 1/4-Z,1/4+Y,3/4-X
 1/4+Z,3/4-Y,1/4-X
 3/4+Z,3/4+Y,3/4+X
1003 2 2 P112 PG2C MONOCLINIC 'P 1 1 2'  !(dyad along z)
 X,Y,Z
 -X,-Y,Z
1004 2 2 P1121 PG2C MONOCLINIC 'P 1 1 21'  !(unique axis c)
 X,Y,Z
 -X,-Y,1/2+Z
1005 4 2 B2 PG2C MONOCLINIC 'B 1 1 2' 'B 2'
 X,Y,Z
 -X,-Y,Z
 1/2+X,+Y,1/2+Z
 1/2-X,-Y,1/2+Z
2005 4 2 A2 PG2 MONOCLINIC 'A 1 2 1'
 X,Y,Z
 -X,Y,-Z
 X,1/2+Y,1/2+Z
 -X,1/2+Y,1/2-Z
3005 4 2 C21 PG2 MONOCLINIC 'C 1 21 1' ! (Origin on screw at 1/4X)
 X,Y,Z
 -X,1/2+Y,-Z
 1/2+X,1/2+Y,Z
 1/2-X,Y,-Z
4005 4 2 I2 PG2 MONOCLINIC 'I 1 2 1' 'I 2' !!! GJK @ 2003-06-02
 X,Y,Z
 -X,Y,-Z
 X+1/2,Y+1/2,Z+1/2
 -X+1/2,Y+1/2,-Z+1/2
5005 4 2 I21 PG2 MONOCLINIC  'I 1 21 1'
 X,Y,Z
 -X,1/2+Y,-Z
 X+1/2,Y+1/2,Z+1/2
 -X+1/2,Y,1/2-Z
1006 2 2 P11m       PGmC MONOCLINIC 'P 1 1 m'
 X,Y,Z
 X,Y,-Z
1007 2 2 P11b       PGmC MONOCLINIC 'P 1 1 b'
 X,Y,Z
 X,1/2+Y,-Z
1008 4 2 B11m       PGmC MONOCLINIC 'B 1 1 m'
 X,Y,Z
 X,Y,-Z
 1/2+X,Y,1/2+Z
 1/2+X,Y,1/2-Z
1009 4 2 B11b       PGmC MONOCLINIC 'B 1 1 b'
 X,Y,Z
 X,1/2+Y,-Z
 1/2+X,Y,1/2+Z
 1/2+X,1/2+Y,1/2-Z
1010 4 4 P112/m     PG2/mC MONOCLINIC 'P 1 1 2/m'
 X,Y,Z
 X,Y,-Z
 -X,-Y,Z
 -X,-Y,-Z
1011 4 4 P1121/m    PG2/mC MONOCLINIC 'P 1 1 21/m'
 X,Y,Z
 -X,-Y,1/2+Z
 -X,-Y,-Z
 X,Y,1/2-Z
1012 8 4 B112/m     PG2/mC MONOCLINIC 'B 1 1 2/m'
 X,Y,Z
 X,Y,-Z
 -X,-Y,Z
 -X,-Y,-Z
 1/2+X,Y,1/2+Z
 1/2+X,Y,1/2-Z
 1/2-X,-Y,1/2+Z
 1/2-X,-Y,1/2-Z
1013 4 4 P112/b     PG2/mC MONOCLINIC 'P 1 1 2/b'
 X,Y,Z
 -X,1/2-Y,Z
 -X,-Y,-Z
 X,1/2+Y,-Z
1014 4 4 P1121/b    PG2/mC MONOCLINIC 'P 1 1 21/b'
 X,Y,Z
 -X,-Y,-Z
 -X,1/2-Y,1/2+Z
 X,1/2+Y,1/2-Z
1015 8 4 B112/b     PG2/mC MONOCLINIC 'B 1 1 2/b'
 X,Y,Z
 -X,1/2-Y,Z
 -X,-Y,-Z
 X,1/2+Y,-Z
 1/2+X,Y,1/2+Z
 1/2-X,1/2-Y,1/2+Z
 1/2-X,-Y,1/2-Z
 1/2+X,1/2+Y,1/2-Z
1017 4 4 P2122 PG222 ORTHORHOMBIC 'P 21 2 2' !(unique axis a)
 X,Y,Z
 -X,Y,-Z
 1/2+X,-Y,-Z
 1/2-X,-Y,Z
2017 4 4 P2212 PG222 ORTHORHOMBIC 'P 2 21 2' !(unique axis b)
 X,Y,Z
 X,1/2-Y,-Z
 -X,1/2+Y,-Z
 -X,-Y,Z
1018 4 4 P21212a PG222 ORTHORHOMBIC 'P 21 21 2 (a)' ! origin on 21 21, shift (1/4,1/4,0)
 X,Y,Z
 1/2-X,1/2-Y,Z
 X+1/2,-Y,-Z
 -X,Y+1/2,-Z
2018 4 4 P21221 PG222 ORTHORHOMBIC 'P 21 2 21'  !(unique axis b)
 X,Y,Z
 -X,Y,-Z
 1/2+X,-Y,1/2-Z
 1/2-X,-Y,1/2+Z
3018 4 4 P22121 PG222 ORTHORHOMBIC 'P 2 21 21'  !(unique axis a)
 X,Y,Z
 X,-Y,-Z
 -X,1/2+Y,1/2-Z
 -X,1/2-Y,1/2+Z
1020 8 4 C2221a PG222 ORTHORHOMBIC 'C 2 2 21a)' ! P212121 with C centring, shift(1/4,0,0)
 X,Y,Z
 1/2-X,-Y,1/2+Z
 1/2+X,1/2-Y,-Z
 -X,1/2+Y,1/2-Z
 1/2+X,1/2+Y,Z
 -X,1/2-Y,1/2+Z
 X,-Y,-Z
 1/2-X,Y,1/2-Z
1021 8 4 C222a PG222 ORTHORHOMBIC 'C 2 2 2a'  ! C21212a origin on 21 21
 X,Y,Z
 1/2-X,1/2-Y,Z
 X+1/2,-Y,-Z
 -X,Y+1/2,-Z
 1/2+ X,1/2+Y,Z
 -X,-Y,Z
 X,1/2-Y,-Z
 1/2-X,Y,-Z
1022 16 4 F222a PG222 ORTHORHOMBIC 'F 2 2 2a' ! same as 1018 with face centring  shift (1/4,0,0)
 X,Y,Z
 1/2-X,1/2-Y,Z
 X+1/2,-Y,-Z
 -X,Y+1/2,-Z
 X,Y+1/2,Z+1/2
 1/2-X,-Y,Z+1/2
 X+1/2,-Y+1/2,-Z+1/2
 -X,Y,-Z+1/2
 X+1/2,Y,Z+1/2
 -X,1/2-Y,Z+1/2
 X,-Y,-Z+1/2
 -X+1/2,Y+1/2,-Z+1/2
 X+1/2,Y+1/2,Z
 -X,-Y,Z
 X,-Y+1/2,-Z
 -X+1/2,Y,-Z
1023 8 4 I222a PG222 ORTHORHOMBIC 'I 2 2 2a'  ! as 1018 with origin shift (1/4,1/4,1/4)
 X,Y,Z
 1/2-X,1/2-Y,Z
 X+1/2,-Y,-Z
 -X,Y+1/2,-Z
 1/2+X,1/2+Y,1/2+Z
 -X,-Y,1/2+Z
 1/2-X,Y,1/2-Z
 X,1/2-Y,1/2-Z
1059 8 8 Pmmn2 PGmmm ORTHORHOMBIC 'P 21/m 21/m 2/n a'
 X,Y,Z
 1/2-X,1/2-Y,Z
 -X,1/2+Y,-Z
 1/2+X,-Y,-Z
 -X,-Y,-Z
 X+1/2,Y+1/2,-Z
 X,1/2-Y,Z
 1/2-X,Y,Z
1094 8 8 P42212a PG422 TETRAGONAL 'P 42 21 2a' ! (as P21212a) origin on 21 21 ie Shift 1/4,1/4,1/4
 X,Y,Z
 1/2-X,1/2-Y,Z
 -Y,X+1/2,1/2+Z
 Y+1/2,-X,1/2+Z
 -X,Y+1/2,-Z
 X+1/2,-Y,-Z
 Y,X,1/2-Z
 1/2-Y,1/2-X,1/2-Z
1197 24 12 I23a PG23 CUBIC 'I 2 3a' ! Expansion of 1023 which is an expansion of 1018
 X,Y,Z
 1/2-X,1/2-Y,Z
 X+1/2,-Y,-Z
 -X,Y+1/2,-Z
 Y,Z,X
 1/2-Y,1/2-Z,X
 Y+1/2,-Z,-X
 -Y,Z+1/2,-X
 Z,X,Y
 1/2-Z,1/2-X,Y
 Z+1/2,-X,-Y
 -Z,X+1/2,-Y
 1/2+X,1/2+Y,1/2+Z
 -X,-Y,1/2+Z
 X,1/2-Y,1/2-Z
 1/2-X,Y,1/2-Z
 1/2+Y,1/2+Z,1/2+X
 -Y,-Z,1/2+X
 Y,1/2-Z,1/2-X
 1/2-Y,Z,1/2-X
 1/2+Z,1/2+X,1/2+Y
 -Z,-X,1/2+Y
 Z,1/2-X,1/2-Y
 1/2-Z,X,1/2-Y
2014 4 4 P21/n PG2/m MONOCLINIC 'P 1 21/n 1'
 X,Y,Z
 -X,-Y,-Z
 1/2-X,1/2+Y,1/2-Z
 X+1/2,1/2-Y,1/2+Z"""
