"""Exceptions for Broadlink devices."""
import struct


class BroadlinkException(Exception):
    """Common base class for all Broadlink exceptions."""

    def __init__(self, *args, **kwargs):
        """Initialize the exception."""
        super().__init__(*args, **kwargs)
        if len(args) >= 3:
            self.errno = args[0]
            self.strerror = "%s: %s" % (args[1], args[2])
        elif len(args) == 2:
            self.errno = args[0]
            self.strerror = str(args[1])
        elif len(args) == 1:
            self.errno = None
            self.strerror = str(args[0])
        else:
            self.errno = None
            self.strerror = ""

    def __str__(self):
        """Return the error message."""
        if self.errno is not None:
            return "[Errno %s] %s" % (self.errno, self.strerror)
        return self.strerror


class FirmwareException(BroadlinkException):
    """Common base class for all firmware exceptions."""


class AuthenticationError(FirmwareException):
    """Authentication error."""


class AuthorizationError(FirmwareException):
    """Authorization error."""


class CommandNotSupportedError(FirmwareException):
    """Command not supported error."""


class ConnectionClosedError(FirmwareException):
    """Connection closed error."""


class DataValidationError(FirmwareException):
    """Data validation error."""


class DeviceOfflineError(FirmwareException):
    """Device offline error."""


class ReadError(FirmwareException):
    """Read error."""


class SendError(FirmwareException):
    """Send error."""


class SSIDNotFoundError(FirmwareException):
    """SSID not found error."""


class StorageError(FirmwareException):
    """Storage error."""


class WriteError(FirmwareException):
    """Write error."""


class SDKException(BroadlinkException):
    """Common base class for all SDK exceptions."""


class DeviceInformationError(SDKException):
    """Device information is not intact."""


class ChecksumError(SDKException):
    """Received data packet check error."""


class LengthError(SDKException):
    """Received data packet length error."""


class NetworkTimeoutError(SDKException):
    """Network timeout error."""


class UnknownError(BroadlinkException):
    """Unknown error."""


BROADLINK_EXCEPTIONS = {
    # Firmware-related errors are generated by the device.
    -1: (AuthenticationError, "Authentication failed"),
    -2: (ConnectionClosedError, "You have been logged out"),
    -3: (DeviceOfflineError, "The device is offline"),
    -4: (CommandNotSupportedError, "Command not supported"),
    -5: (StorageError, "The device storage is full"),
    -6: (DataValidationError, "Structure is abnormal"),
    -7: (AuthorizationError, "Control key is expired"),
    -8: (SendError, "Send error"),
    -9: (WriteError, "Write error"),
    -10: (ReadError, "Read error"),
    -11: (SSIDNotFoundError, "SSID could not be found in AP configuration"),
    # SDK related errors are generated by this module.
    -2040: (DeviceInformationError, "Device information is not intact"),
    -4000: (NetworkTimeoutError, "Network timeout"),
    -4007: (LengthError, "Received data packet length error"),
    -4008: (ChecksumError, "Received data packet check error"),
}


def exception(error_code):
    """Return exception corresponding to an error code."""
    try:
        exc, msg = BROADLINK_EXCEPTIONS[error_code]
        return exc(error_code, msg)
    except KeyError:
        return UnknownError(error_code, "Unknown error")


def check_error(error):
    """Raise exception if an error occurred."""
    error_code = struct.unpack("h", error)[0]
    if error_code:
        raise exception(error_code)
