#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# qats documentation build configuration file, created by
# sphinx-quickstart on Sat Dec 16 19:53:09 2017.
#
# changes:
#   02.01.2023/enlo     Updated to use sphinx_rtd_theme.
#
#
import os
import sys
import qats

# which theme to use
# options: 'sphinx_rtd_theme' or 'furo'
THEME = 'furo'  # 'sphinx_rtd_theme'

# print("Generating doc for QATS version {version} installed in {path}"
#       .format(version=qats.__version__, path=qats.__path__))
print("Generating doc for QATS version {version} installed in {path},\n"
      "... using Sphinx theme '{theme}'"
      .format(version=qats.__version__, path=qats.__path__, theme=THEME))

# If extensions (or modules to document with autodoc) are in another directory,
# add these directories to sys.path here. If the directory is relative to the
# documentation root, use os.path.abspath to make it absolute, like shown here.
sys.path.append(os.path.abspath(os.path.join('..', '..')))

# ----------------------------------------------------------------------------------------------------------------------
# General configuration
# ----------------------------------------------------------------------------------------------------------------------
# Add any Sphinx extension module names here, as strings. They can be extensions
# coming with Sphinx (named 'sphinx.ext.*') or your custom ones.
extensions = [
    'sphinx.ext.autodoc',
    'sphinx.ext.autosummary',
    'sphinx.ext.napoleon',      # makes sphinx understand docstrings in numpy and google format
    'sphinx.ext.intersphinx',
    'sphinx.ext.todo',
    # 'sphinx.ext.viewcode',      # remove this one to disable code view
    # 'sphinx.ext.doctest',
    # 'sphinx.ext.imgmath',
    # 'sphinx.ext.ifconfig',
    # 'sphinx.ext.coverage',
    #
    # 23.12.2022/enlo: added myst-parser to enable rendering of CHANGELOG.md as part of Sphinx docs.
    'myst_parser',
]

# Add any paths that contain templates here, relative to this directory.
#
# templates_path = ['_templates']

# The suffix(es) of source filenames.
# You can specify multiple suffix as a list of string:
#
source_suffix = ['.rst', '.md']  # source_suffix = '.rst'

# The master toctree document.
master_doc = 'index'

# General information about the project.
project = 'QATS'
copyright = 'DNV'
author = 'Per Voie, Erling Lone'

# The version info for the project you're documenting, acts as replacement for
# |version| and |release|, also used in various other places throughout the
# built documents.
#
# version: the short X.Y version.
# release: the full version, including alpha/beta/rc tags.
# version = ''
# release = ''
version = qats.__version__
release = version

# The language for content autogenerated by Sphinx. Refer to documentation
# for a list of supported languages.
#
language = 'en'

# List of patterns, relative to source directory, that match files and
# directories to ignore when looking for source files.
# This patterns also effect to html_static_path and html_extra_path
exclude_patterns = []


# If true, `todo` and `todoList` produce output, else they produce nothing.
todo_include_todos = True


# ----------------------------------------------------------------------------------------------------------------------
# Napoleon configuration
#   ref.: https://www.sphinx-doc.org/en/master/usage/extensions/napoleon.html
# ----------------------------------------------------------------------------------------------------------------------
napoleon_google_docstring = True
napoleon_numpy_docstring = True
napoleon_include_private_with_doc = False
napoleon_include_special_with_doc = False
napoleon_use_admonition_for_examples = True     # default: False
napoleon_use_admonition_for_notes = True        # default: False
napoleon_use_admonition_for_references = False
napoleon_use_ivar = True   # default: False
napoleon_use_param = True   # default: True
napoleon_use_rtype = True   # default: True
napoleon_use_keyword = True
napoleon_preprocess_types = False    # default: False
napoleon_attr_annotations = True


# ----------------------------------------------------------------------------------------------------------------------
# HTML output configuration
# ----------------------------------------------------------------------------------------------------------------------
# The theme to use for HTML and HTML Help pages.  See the documentation for
# a list of builtin themes.
#
html_theme = THEME  # 'sphinx_rtd_theme'
html_logo = 'img/qats.ico'

# Add any paths that contain custom static files (such as style sheets) here,
# relative to this directory. They are copied after the builtin static files,
# so a file named "default.css" will overwrite the builtin "default.css".
html_static_path = ['_static']

# theme-specific variables
if THEME == 'sphinx_rtd_theme':
    import sphinx_rtd_theme
    html_theme_path = [sphinx_rtd_theme.get_html_theme_path()]

    # custom css file (theme-specific)
    # these paths are either relative to html_static_path or fully qualified paths (eg. https://...)
    html_css_files = [
        'css/custom-sphinx-rtd-theme.css',
    ]

    # theme options: sphinx_rtd_theme
    # ref. https://sphinx-rtd-theme.readthedocs.io/en/stable/configuring.html#theme-options
    html_theme_options = {
        # 'analytics_id': '',  #  Provided by Google in your dashboard
        # 'analytics_anonymize_ip': False,
        'logo_only': False,
        'display_version': True,
        'prev_next_buttons_location': 'bottom',
        'style_external_links': True,  # default: False
        'vcs_pageview_mode': '',
        'style_nav_header_background': '#efefef',  # 'whitesmoke' is '#f5f5f5'  # 'lightgrey',  # 'white',
        #
        # TOC options
        #
        'collapse_navigation': True,
        'sticky_navigation': True,
        'navigation_depth': 4,
        'includehidden': True,
        'titles_only': False,
    }

    # The name of the Pygments (syntax highlighting) style to use.
    pygments_style = 'default'  # 'sphinx'

elif THEME == 'furo':
    try:
        import furo
    except ImportError:
        raise ImportError("Could not import 'furo' -- install it through `python -m pip install furo` (or similar)")

    # html_theme_path = []

    # custom css file (theme-specific)
    # these paths are either relative to html_static_path or fully qualified paths (eg. https://...)
    html_css_files = [
        'css/custom-furo.css',
    ]

    # theme options: furo
    # ref. https://pradyunsg.me/furo/customisation/
    html_theme_options = {
        "sidebar_hide_name": False,  # True  (if True, show only logo and hide project name)
        "navigation_with_keys": True,
        # "top_of_page_button": "edit",
        # "announcement": "<em>Note: Backward incompatible changes introduced in version 5.0.0</em>",
    }

    # syntax highlighting, incl. for dark mode
    # ref. https://pradyunsg.me/furo/customisation/colors/#code-block-styling
    pygments_style = 'default'  # 'sphinx'
    pygments_dark_style = 'monokai'

    # sidebar title
    # ref. https://pradyunsg.me/furo/customisation/sidebar-title/#changing-sidebar-title
    html_title = f"{project} {release}"

    # sidebar
    # ref. https://pradyunsg.me/furo/customisation/sidebar/#using-html-sidebars
    html_sidebars = {
        "**": [
            "sidebar/brand.html",
            "sidebar/search.html",
            "sidebar/scroll-start.html",
            "sidebar/navigation.html",
            # "sidebar/ethical-ads.html",
            "sidebar/scroll-end.html",
            "sidebar/variant-selector.html",
        ]
    }

else:
    raise ValueError(f"THEME must be set to 'sphinx_rtd_theme' or 'furo', got '{THEME}'")


# Theme options are theme-specific and customize the look and feel of a theme
# further.  For a list of options available for each theme, see the
# documentation.
#
# html_theme_options = {}

# Include sidebar
# html_sidebars = {
#     '**': ['localtoc.html', ],  # 'relations.html', ],  #'globaltoc.html', 'sourcelink.html', 'searchbox.html']}
# }


# ----------------------------------------------------------------------------------------------------------------------
# Other configuration options
# ----------------------------------------------------------------------------------------------------------------------

# Intersphinx mapping
intersphinx_mapping = {
    'python': ('https://docs.python.org/3', None),
    'numpy': ('https://numpy.org/doc/stable/', None),
    'scipy': ('https://docs.scipy.org/doc/scipy/', None),
}


# ----------------------------------------------------------------------------------------------------------------------
# Autodoc options
# ----------------------------------------------------------------------------------------------------------------------
# autosummary_generate = True  # what difference does this variable do, actually??
autodoc_default_options = {
    'members': True,
    'member-order': 'bysource',  # 'alphabetical', 'groupwise', 'bysource'
    # 'special-members': '__init__',
}
autoclass_content = "both"  # include both class' and __init__ method's docstrings (concatenated)
add_module_names = False

# ----------------------------------------------------------------------------------------------------------------------
# Autosummary
# ... based on code and inspiration provided by:
#   https://stackoverflow.com/questions/20569011/python-sphinx-autosummary-automated-listing-of-member-functions
#   https://github.com/markovmodel/PyEMMA/blob/devel/doc/source/conf.py#L285
# ----------------------------------------------------------------------------------------------------------------------


# try to exclude deprecated
def skip_deprecated(app, what, name, obj, skip, options):
    if hasattr(obj, "func_dict") and "__deprecated__" in obj.func_dict:
        print("skipping " + name)
        return True
    return skip or False


def setup(app):
    app.connect('autodoc-skip-member', skip_deprecated)
    app.add_css_file("custom-todo-style.css")  # also can be a full URL

    try:
        import inspect
        from sphinx.ext.autosummary import Autosummary
        from docutils.parsers.rst import directives
        # import sphinx.ext.autodoc
        # from sphinx.ext.autosummary import get_documenter
        # from sphinx.util.inspect import safe_getattr
        # import re

        class AutoAutoSummary(Autosummary):

            option_spec = {
                'modules': directives.unchanged,
                'functions': directives.unchanged,
                'classes': directives.unchanged,
                'methods': directives.unchanged,
                'properties': directives.unchanged,
                'toctree': directives.unchanged,
            }

            required_arguments = 1

            @staticmethod
            def get_class_members(obj, typ, include_public=None):
                """
                Parameters
                ----------
                obj : object
                    Object (class) to inspect.
                typ : str
                    Type of members requested. Valid options: 'method', 'property'
                include_public : list, optional
                    Methods/attributes to include in public list

                Returns
                -------
                public : list
                    List of public members of requested type.
                items : list
                    List of all members of requested type.
                """
                if include_public is None:
                    include_public = []
                if typ == "method":
                    # enlo, 22.08.2019: it appears not all methods are captured by ismethod() any more;
                    # we therefore need to extract members that predicate True for isfunction() as well.
                    items = [m[0] for m in
                             inspect.getmembers(obj, predicate=lambda x: inspect.ismethod(x) or inspect.isfunction(x))]
                elif typ == "property":
                    items = [m[0] for m in inspect.getmembers(obj, predicate=inspect.isdatadescriptor)]
                else:
                    raise ValueError(f"Invalid value for parameter 'typ': {typ}")
                public = [x for x in items if x in include_public or not x.startswith('_')]
                return public, items

            @staticmethod
            def get_module_members(module, typ=('class', 'function')):
                """
                Parameters
                ----------
                module : module
                    Module to inspect.
                typ : str or list, optional
                    Type of members requested. Valid options: 'class', 'function', 'module'

                Returns
                -------
                members : list
                    List of module members
                """
                if isinstance(typ, str):
                    typ = typ,  # make tuple

                def check(mod, member, mtyp):
                    """
                    Function that returns True if `member` should be included in returned list.
                    The following criteria are used:
                        - classes are included if 'classes' is specified
                        - functions are included if 'functions' is specified
                        - modules are included if 'modules' is specified
                        - classes and functions are only included if they are defined in the module inspected
                        - modules are only included if they are not imported
                    """
                    if (inspect.isclass(member) and 'class' in mtyp) and member.__module__ == mod.__name__:
                        return True
                    if (inspect.isfunction(member) and 'function' in mtyp) and member.__module__ == mod.__name__:
                        return True
                    if (inspect.ismodule(member) and 'module' in mtyp) and member.__package__ in member.__name__:
                        return True
                    return False

                # members = [check(module, m[0], typ) for m in inspect.getmembers(module)]
                members = [m[0] for m in inspect.getmembers(module, predicate=lambda x: check(module, x, typ))]
                return members

            def run(self):
                # check that options doesn't contain invalid combinations
                if ('functions' in self.options or 'classes' in self.options) \
                        and ('methods' in self.options or 'properties' in self.options):
                    raise Exception("invalid option combination: %s" % self.options)

                # import requested module, or module of requested class
                fullname = self.arguments[0]
                root, name = fullname.rsplit('.', 1)
                r = __import__(root, globals(), locals(), [name])   # root module
                cm = getattr(r, name)                               # class or module
                if inspect.isclass(cm):
                    cm_typ = "class"
                elif inspect.ismodule(cm):
                    cm_typ = "module"
                else:
                    raise TypeError(f"Invalid type for {fullname}: {cm}")

                # debug
                '''
                with open("_abc.txt", "a") as f:
                    f.write(f"run : {fullname}, {cm_typ}\n")
                '''

                if cm_typ == "class":
                    try:
                        r = __import__(root, globals(), locals(), [name])
                        cm = getattr(r, name)

                        self.content = []

                        if 'properties' in self.options:
                            _, props = self.get_class_members(cm, 'property')
                            self.content += ["~%s.%s" % (fullname, prop) for prop in props if not prop.startswith('_')]

                        if 'methods' in self.options:
                            _, methods = self.get_class_members(cm, 'method', ['__init__'])

                            self.content += ["~%s.%s" % (fullname, method) for method in methods if
                                             not method.startswith('_')]
                    finally:
                        return super(AutoAutoSummary, self).run()

                elif cm_typ == "module":
                    try:
                        typ = []
                        if 'classes' in self.options:
                            typ.append('class')
                        if 'functions' in self.options:
                            typ.append('function')
                        if 'modules' in self.options:
                            typ.append('modules')

                        members = self.get_module_members(cm, typ=typ)
                        self.content = ["~%s.%s" % (fullname, member) for member in members
                                        if not member.startswith('_')]

                    finally:
                        return super(AutoAutoSummary, self).run()

        app.add_directive('autoautosummary', AutoAutoSummary)
    except BaseException as e:
        raise e


# ----------------------------------------------------------------------------------------------------------------------
# Modify Napoleon NumpyDocstring class: for class documentation, rename 'Variables' to 'Attributes'
#   - Relevant only when `napoleon_use_ivar=True`
#
# ... based on code and inspiration provided by:
#   https://michaelgoerz.net/notes/extending-sphinx-napoleon-docstring-sections.html
# ----------------------------------------------------------------------------------------------------------------------
from sphinx.ext.napoleon.docstring import NumpyDocstring


def parse_attributes_section(self, section):
    return self._format_fields('Attributes', self._consume_fields())


NumpyDocstring._parse_attributes_section = parse_attributes_section
