#!/usr/bin/env python
# coding: utf-8

# # SageMath and LaTeX

# In this document, we explore the connection between SageMath and $\LaTeX$.  In SageMath, most objects have a 
# `latex()` function that returns the code to write that object in $\LaTeX$.  For example, we could define the 
# following objects, which are matrices over the field $\mathbb{Z}_{11}$, of size $5\times 3$, and then print 
# out both the object as well as the $\LaTeX$ code for that object.

# In[ ]:


space = MatrixSpace(GF(11), 5, 3)

m = space([1,6,4,
           5,3,3,
           5,7,6,
           2,10,4,
           2,8,8])

print(m)

print()

print(latex(m))


# We should see the matrix $m$ print out, then the $\LaTeX$ code that begins an array, lists all the elements 
# of that array, and encloses the array with a set of parenthesis.  Here we have another example:

# In[ ]:


# we have not discussed this in the Project, but for plotting, we need to tell SageMath which variables we
# are going to be using, and we do so with the var() command.
x = var('x')

# plot sin(x) from 0 to 2pi
y = plot(sin(x),0,2*pi)

# print out the plot, in the next cell we will see what happens 
show(y)


# In[ ]:


print(latex(y))


# This is alot.  Although it does give the suggestion to save this in a separate file with the extension `.pgf`,
# and then to include the figure in your $\LaTeX$ document with the command `\input{<filename>.pgf}`.  The code 
# generated by SageMath can easily become very complicated, but is often still a tool worthy of knowing about. 
# 
# We can also use our knowledge of formatting strings in Python to our advantage here, say we have have the group 
# $S_4$, and we are interested in the subgroup generated by $\{(123),(24)\}$.  To calculate the subgroup useing 
# SageMath we would type the following:

# In[6]:


G = SymmetricGroup(4)

# create the subgroup generated by two cycles
H = G.subgroup(["(1,2,3)","(2,4)"])

# print out the subgroup
H.list()


# If we want to include this in our $\LaTeX$ file, with four elements of the subgroup per line, we could
# write the following code:

# In[11]:


count_per_line = 0
print("\\begin{align*}")
for element in H.list():
    count_per_line = count_per_line + 1
    if count_per_line % 4 == 0:
        print(f"{element}")
    else:
        print(f"{element} & ")
    if count_per_line % 4 == 0:
        print("\\\\")
print("\\end{align*}")


# This produces an align environment that contains the entirety of the subgroup $H\leq G$.  

# Throughout this project, experiment with the `latex()` command to see which objects lend themselves nicely
# to $\LaTeX$.
