"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseQuery = void 0;
const path = require("path");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib");
const customresources = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const cluster_1 = require("../cluster");
class DatabaseQuery extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const adminUser = this.getAdminUser(props);
        const handler = new lambda.SingletonFunction(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'database-query-provider'), {
                exclude: ['*.ts'],
            }),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            timeout: cdk.Duration.minutes(1),
            uuid: '3de5bea7-27da-4796-8662-5efb56431b5f',
            lambdaPurpose: 'Query Redshift Database',
        });
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['redshift-data:DescribeStatement', 'redshift-data:ExecuteStatement'],
            resources: ['*'],
        }));
        adminUser.grantRead(handler);
        const provider = new customresources.Provider(this, 'Provider', {
            onEventHandler: handler,
        });
        const queryHandlerProps = {
            handler: props.handler,
            clusterName: props.cluster.clusterName,
            adminUserArn: adminUser.secretArn,
            databaseName: props.databaseName,
            ...props.properties,
        };
        this.resource = new cdk.CustomResource(this, 'Resource', {
            resourceType: 'Custom::RedshiftDatabaseQuery',
            serviceToken: provider.serviceToken,
            removalPolicy: props.removalPolicy,
            properties: queryHandlerProps,
        });
        this.grantPrincipal = handler.grantPrincipal;
        this.ref = this.resource.ref;
    }
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
    getAttString(attributeName) {
        return this.resource.getAttString(attributeName);
    }
    getAdminUser(props) {
        const cluster = props.cluster;
        let adminUser = props.adminUser;
        if (!adminUser) {
            if (cluster instanceof cluster_1.Cluster) {
                if (cluster.secret) {
                    adminUser = cluster.secret;
                }
                else {
                    throw new Error('Administrative access to the Redshift cluster is required but an admin user secret was not provided and the cluster did not generate admin user credentials (they were provided explicitly)');
                }
            }
            else {
                throw new Error('Administrative access to the Redshift cluster is required but an admin user secret was not provided and the cluster was imported');
            }
        }
        return adminUser;
    }
}
exports.DatabaseQuery = DatabaseQuery;
//# sourceMappingURL=data:application/json;base64,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