#ifndef API_H
#define API_H

#include <stddef.h>
#include <stdint.h>

#define CRYPTO_SECRETKEYBYTES   1281
#define CRYPTO_PUBLICKEYBYTES   897
#define CRYPTO_BYTES            690

#define CRYPTO_ALGNAME          "Falcon-512"

/*
 * Generate a new key pair. Public key goes into pk[], private key in sk[].
 * Key sizes are exact (in bytes):
 *   public (pk): CRYPTO_PUBLICKEYBYTES
 *   private (sk): CRYPTO_SECRETKEYBYTES
 *
 * Return value: 0 on success, -1 on error.
 */
int crypto_sign_keypair(
    uint8_t *pk, uint8_t *sk, uint8_t *myseed);

/*
 * Compute a signature on a provided message (m, mlen), with a given
 * private key (sk). Signature is written in sig[], with length written
 * into *siglen. Signature length is variable; maximum signature length
 * (in bytes) is CRYPTO_BYTES.
 *
 * sig[], m[] and sk[] may overlap each other arbitrarily.
 *
 * Return value: 0 on success, -1 on error.
 */
int crypto_sign_signature(
    uint8_t *sig, size_t *siglen,
    const uint8_t *m, size_t mlen, const uint8_t *sk);

/*
 * Verify a signature (sig, siglen) on a message (m, mlen) with a given
 * public key (pk).
 *
 * sig[], m[] and pk[] may overlap each other arbitrarily.
 *
 * Return value: 0 on success, -1 on error.
 */
int crypto_sign_verify(
    const uint8_t *sig, size_t siglen,
    const uint8_t *m, size_t mlen, const uint8_t *pk);

/*
 * Compute a signature on a message and pack the signature and message
 * into a single object, written into sm[]. The length of that output is
 * written in *smlen; that length may be larger than the message length
 * (mlen) by up to CRYPTO_BYTES.
 *
 * sm[] and m[] may overlap each other arbitrarily; however, sm[] shall
 * not overlap with sk[].
 *
 * Return value: 0 on success, -1 on error.
 */
int crypto_sign(
    uint8_t *sm, size_t *smlen,
    const uint8_t *m, size_t mlen, const uint8_t *sk);

/*
 * Open a signed message object (sm, smlen) and verify the signature;
 * on success, the message itself is written into m[] and its length
 * into *mlen. The message is shorter than the signed message object,
 * but the size difference depends on the signature value; the difference
 * may range up to CRYPTO_BYTES.
 *
 * m[], sm[] and pk[] may overlap each other arbitrarily.
 *
 * Return value: 0 on success, -1 on error.
 */
int crypto_sign_open(
    uint8_t *m, size_t *mlen,
    const uint8_t *sm, size_t smlen, const uint8_t *pk);

#endif
