/*
 * Copyright 2018-2021 Elyra Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { elyraIcon } from '@elyra/ui-components';
import { Launcher as JupyterlabLauncher } from '@jupyterlab/launcher';
import { each } from '@lumino/algorithm';
import * as React from 'react';
/**
 * The known categories of launcher items and their default ordering.
 */
const ELYRA_CATEGORY = 'Elyra';
const KNOWN_CATEGORIES = ['Notebook', 'Console', ELYRA_CATEGORY, 'Other'];
export class Launcher extends JupyterlabLauncher {
    /**
     * Construct a new launcher widget.
     */
    constructor(options) {
        super(options);
    }
    replaceCategoryIcon(category, icon) {
        const children = React.Children.map(category.props.children, child => {
            if (child.props.className === 'jp-Launcher-sectionHeader') {
                const grandchildren = React.Children.map(child.props.children, grandchild => {
                    if (grandchild.props.className !== 'jp-Launcher-sectionTitle') {
                        return React.createElement(icon.react, { stylesheet: "launcherSection" });
                    }
                    else {
                        return grandchild;
                    }
                });
                return React.cloneElement(child, child.props, grandchildren);
            }
            else {
                return child;
            }
        });
        return React.cloneElement(category, category.props, children);
    }
    /**
     * Render the launcher to virtual DOM nodes.
     */
    render() {
        // Bail if there is no model.
        if (!this.model) {
            return null;
        }
        // get the rendering from JupyterLab Launcher
        // and resort the categories
        const launcherBody = super.render();
        const launcherContent = launcherBody.props.children;
        const launcherCategories = launcherContent.props.children;
        const categories = [];
        // Assemble the final ordered list of categories
        // based on KNOWN_CATEGORIES.
        each(KNOWN_CATEGORIES, (category, index) => {
            React.Children.forEach(launcherCategories, cat => {
                if (cat.key === category) {
                    if (cat.key === ELYRA_CATEGORY) {
                        cat = this.replaceCategoryIcon(cat, elyraIcon);
                    }
                    categories.push(cat);
                }
            });
        });
        // Wrap the sections in body and content divs.
        return (React.createElement("div", { className: "jp-Launcher-body" },
            React.createElement("div", { className: "jp-Launcher-content" },
                React.createElement("div", { className: "jp-Launcher-cwd" },
                    React.createElement("h3", null, this.cwd)),
                categories)));
    }
}
//# sourceMappingURL=launcher.js.map