"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const appmesh = require("../lib");
module.exports = {
    'When adding route to existing VirtualRouter': {
        'should create route resource'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const mesh = new appmesh.Mesh(stack, 'mesh', {
                meshName: 'test-mesh',
            });
            const router = mesh.addVirtualRouter('router');
            const service1 = new appmesh.VirtualService(stack, 'service-1', {
                virtualServiceName: 'service1.domain.local',
                mesh,
            });
            const node = mesh.addVirtualNode('test-node', {
                dnsHostName: 'test',
                listener: {
                    portMapping: {
                        port: 8080,
                        protocol: appmesh.Protocol.HTTP,
                    },
                },
                backends: [service1],
            });
            router.addRoute('route-1', {
                routeTargets: [
                    {
                        virtualNode: node,
                        weight: 50,
                    },
                ],
                prefix: '/',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::Route', {
                RouteName: 'route-1',
                Spec: {
                    HttpRoute: {
                        Action: {
                            WeightedTargets: [
                                {
                                    VirtualNode: {
                                        'Fn::GetAtt': ['meshtestnodeF93946D4', 'VirtualNodeName'],
                                    },
                                    Weight: 50,
                                },
                            ],
                        },
                        Match: {
                            Prefix: '/',
                        },
                    },
                },
                VirtualRouterName: {
                    'Fn::GetAtt': ['meshrouter81B8087E', 'VirtualRouterName'],
                },
            }));
            test.done();
        },
    },
    'When adding routes to a VirtualRouter with existing routes': {
        'should add routes and not overwrite'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const mesh = new appmesh.Mesh(stack, 'mesh', {
                meshName: 'test-mesh',
            });
            const router = mesh.addVirtualRouter('router');
            const service1 = new appmesh.VirtualService(stack, 'service-1', {
                virtualServiceName: 'service1.domain.local',
                mesh,
            });
            const service2 = new appmesh.VirtualService(stack, 'service-2', {
                virtualServiceName: 'service2.domain.local',
                mesh,
            });
            const node = mesh.addVirtualNode('test-node', {
                dnsHostName: 'test',
                listener: {
                    portMapping: {
                        port: 8080,
                        protocol: appmesh.Protocol.HTTP,
                    },
                },
                backends: [
                    service1,
                ],
            });
            const node2 = mesh.addVirtualNode('test-node2', {
                dnsHostName: 'test',
                listener: {
                    portMapping: {
                        port: 8080,
                        protocol: appmesh.Protocol.HTTP,
                    },
                },
                backends: [
                    service2,
                ],
            });
            const node3 = mesh.addVirtualNode('test-node3', {
                dnsHostName: 'test',
                listener: {
                    portMapping: {
                        port: 8080,
                        protocol: appmesh.Protocol.HTTP,
                    },
                },
                backends: [
                    service1,
                ],
            });
            router.addRoute('route-1', {
                routeTargets: [
                    {
                        virtualNode: node,
                        weight: 50,
                    },
                ],
                prefix: '/',
            });
            router.addRoute('route-2', {
                routeTargets: [
                    {
                        virtualNode: node2,
                        weight: 30,
                    },
                ],
                prefix: '/path2',
            });
            router.addRoute('route-3', {
                routeTargets: [
                    {
                        virtualNode: node3,
                        weight: 20,
                    },
                ],
                prefix: '/path3',
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::Route', {
                RouteName: 'route-1',
                Spec: {
                    HttpRoute: {
                        Action: {
                            WeightedTargets: [
                                {
                                    VirtualNode: {
                                        'Fn::GetAtt': ['meshtestnodeF93946D4', 'VirtualNodeName'],
                                    },
                                    Weight: 50,
                                },
                            ],
                        },
                        Match: {
                            Prefix: '/',
                        },
                    },
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::Route', {
                RouteName: 'route-2',
                Spec: {
                    HttpRoute: {
                        Action: {
                            WeightedTargets: [
                                {
                                    VirtualNode: {
                                        'Fn::GetAtt': ['meshtestnode20C58B1B2', 'VirtualNodeName'],
                                    },
                                    Weight: 30,
                                },
                            ],
                        },
                        Match: {
                            Prefix: '/path2',
                        },
                    },
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::Route', {
                RouteName: 'route-3',
                Spec: {
                    HttpRoute: {
                        Action: {
                            WeightedTargets: [
                                {
                                    VirtualNode: {
                                        'Fn::GetAtt': ['meshtestnode316EEA2D7', 'VirtualNodeName'],
                                    },
                                    Weight: 20,
                                },
                            ],
                        },
                        Match: {
                            Prefix: '/path3',
                        },
                    },
                },
            }));
            test.done();
        },
    },
    'When adding a TCP route to existing VirtualRouter': {
        'should create route resource'(test) {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const mesh = new appmesh.Mesh(stack, 'mesh', {
                meshName: 'test-mesh',
            });
            const router = mesh.addVirtualRouter('router');
            const service1 = new appmesh.VirtualService(stack, 'service-1', {
                virtualServiceName: 'service1.domain.local',
                mesh,
            });
            const node = mesh.addVirtualNode('test-node', {
                dnsHostName: 'test',
                listener: {
                    portMapping: {
                        port: 8080,
                        protocol: appmesh.Protocol.HTTP,
                    },
                },
                backends: [
                    service1,
                ],
            });
            router.addRoute('route-tcp-1', {
                routeTargets: [
                    {
                        virtualNode: node,
                        weight: 50,
                    },
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AppMesh::Route', {
                RouteName: 'route-tcp-1',
                Spec: {
                    TcpRoute: {
                        Action: {
                            WeightedTargets: [
                                {
                                    VirtualNode: {
                                        'Fn::GetAtt': ['meshtestnodeF93946D4', 'VirtualNodeName'],
                                    },
                                    Weight: 50,
                                },
                            ],
                        },
                    },
                },
                VirtualRouterName: {
                    'Fn::GetAtt': ['meshrouter81B8087E', 'VirtualRouterName'],
                },
            }));
            test.done();
        },
    },
    'can import a virtual router'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const vr = appmesh.VirtualRouter.fromVirtualRouterName(stack, 'Router', 'MyMesh', 'MyRouter');
        // THEN
        test.ok(vr.mesh !== undefined);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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